/*
   DGBLAS.C
   BLAS routines used by LAPACK matrix solvers.

   $Id$
 */

#include "dg.h"


/*-----prototypes of functions defined here-----*/
extern long  idamax(long n,double dx[],long incx);
extern void dtrsv ( char uplo, char trans, char diag, long n,
		   double a[], long lda, double x[], long incx );
extern void dtrsm ( char side, char uplo, char transa, char diag,
		   long m, long n, double alpha, double a[], long lda,
		   double b[], long ldb );
extern void dtrmv ( char uplo, char trans, char diag, long n,
		   double a[], long lda, double x[], long incx );
extern void dtrmm ( char side, char uplo, char transa, char diag,
		   long m, long n, double alpha, double a[], long lda,
		   double b[], long ldb );
extern void  dswap (long n,double dx[],long incx,double dy[],long incy);
extern void  dscal(long n,double da,double dx[],long incx);
extern void  drot (long n,double dx[],long incx,double dy[],long incy,
		   double c,double s);
extern double   dnrm2 ( long n, double dx[], long incx);
extern void dger  ( long m, long n, double alpha, double x[], long incx,
		   double y[], long incy, double a[], long lda );
extern void dgemv ( char trans, long m, long n, double alpha,
		   double a[], long lda, double x[], long incx,
		   double beta, double y[], long incy );
extern void dgemm ( char transa, char transb, long m, long n, long k,
		   double alpha, double a[], long lda, double b[], long ldb,
		   double beta, double c[], long ldc );
extern double   ddot(long n,double dx[],long incx,double dy[],long incy);
extern void  dcopy(long n,double dx[],long incx,double dy[],long incy);
extern void daxpy(long n,double da,double dx[],long incx,
		  double dy[],long incy);
extern double   dasum(long n,double dx[],long incx);
/*-----end of prototypes-----*/



/*---converted nutty string switches to single characters (lower case)---*/
#define lsame(x,y) ((x)==(y))



/*-----Fortran intrinsics converted-----*/
#define abs(x) ((x)>=0?(x):-(x))
#define dabs(x) ((double)((x)>=0?(x):-(x)))
extern double sqrt(double);
#define mod(x,y) ((x)%(y))
#define max(x,y) ((x)>(y)?(x):(y))
/*-----end of Fortran intrinsics-----*/



double   dasum(long n,double dx[],long incx)
{
  /*c
    c     takes the sum of the absolute values.
    c     jack dongarra, linpack, 3/11/78.
    c     modified 3/93 to return if incx .le. 0.
    c*/
  double dasum_R;
  double  dtemp;
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,m,mp1,nincx;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  dasum_R = 0.0e0;
  dtemp = 0.0e0;
  if( n<=0 || incx<=0 )return dasum_R;
  if(incx==1)goto L_20;
  /*c
    c        code for increment not equal to 1
    c*/
  nincx = n*incx;
  for (i=1 ; incx>0?i<=nincx:i>=nincx ; i+=incx) {
    dtemp = dtemp + dabs(dx_1(i));
  }
  dasum_R = dtemp;
  return dasum_R;
  /*c
    c        code for increment equal to 1
    c*/
  /*c
    c        clean-up loop
    c*/
 L_20: m = mod(n,6);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dtemp = dtemp + dabs(dx_1(i));
  }
  if( n < 6 ) goto L_60;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=6) {
    dtemp = dtemp + dabs(dx_1(i)) + dabs(dx_1(i + 1)) + dabs(dx_1(i + 2))
      + dabs(dx_1(i + 3)) + dabs(dx_1(i + 4)) + dabs(dx_1(i + 5));
  }
 L_60: dasum_R = dtemp;
  return dasum_R;
}



void daxpy(long n,double da,double dx[],long incx,double dy[],long incy)
{
  /*c
    c     constant times a vector plus a vector.
    c     uses unrolled loops for increments equal to one.
    c     jack dongarra, linpack, 3/11/78.
    c*/
#undef dy_1
#define dy_1(a1) dy[a1-1]
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix,iy,m,mp1;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if(n<=0)return;
  if (da == 0.0e0) return;
  if(incx==1&&incy==1)goto L_20;
  /*c
    c        code for unequal increments or equal increments
    c          not equal to 1
    c*/
  ix = 1;
  iy = 1;
  if(incx<0)ix = (-n+1)*incx + 1;
  if(incy<0)iy = (-n+1)*incy + 1;
  for (i=1 ; i<=n ; i+=1) {
    dy_1(iy) = dy_1(iy) + da*dx_1(ix);
    ix = ix + incx;
    iy = iy + incy;
  }
  return;
  /*c
    c        code for both increments equal to 1
    c*/
  /*c
    c        clean-up loop
    c*/
 L_20: m = mod(n,4);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dy_1(i) = dy_1(i) + da*dx_1(i);
  }
  if( n < 4 ) return;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=4) {
    dy_1(i) = dy_1(i) + da*dx_1(i);
    dy_1(i + 1) = dy_1(i + 1) + da*dx_1(i + 1);
    dy_1(i + 2) = dy_1(i + 2) + da*dx_1(i + 2);
    dy_1(i + 3) = dy_1(i + 3) + da*dx_1(i + 3);
  }
  return;
}



void  dcopy(long n,double dx[],long incx,double dy[],long incy)
{
  /*c
    c     copies a vector, x, to a vector, y.
    c     uses unrolled loops for increments equal to one.
    c     jack dongarra, linpack, 3/11/78.
    c*/
#undef dy_1
#define dy_1(a1) dy[a1-1]
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix,iy,m,mp1;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if(n<=0)return;
  if(incx==1&&incy==1)goto L_20;
  /*c
    c        code for unequal increments or equal increments
    c          not equal to 1
    c*/
  ix = 1;
  iy = 1;
  if(incx<0)ix = (-n+1)*incx + 1;
  if(incy<0)iy = (-n+1)*incy + 1;
  for (i=1 ; i<=n ; i+=1) {
    dy_1(iy) = dx_1(ix);
    ix = ix + incx;
    iy = iy + incy;
  }
  return;
  /*c
    c        code for both increments equal to 1
    c*/
  /*c
    c        clean-up loop
    c*/
 L_20: m = mod(n,7);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dy_1(i) = dx_1(i);
  }
  if( n < 7 ) return;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=7) {
    dy_1(i) = dx_1(i);
    dy_1(i + 1) = dx_1(i + 1);
    dy_1(i + 2) = dx_1(i + 2);
    dy_1(i + 3) = dx_1(i + 3);
    dy_1(i + 4) = dx_1(i + 4);
    dy_1(i + 5) = dx_1(i + 5);
    dy_1(i + 6) = dx_1(i + 6);
  }
  return;
}



double   ddot(long n,double dx[],long incx,double dy[],long incy)
{
  /*c
    c     forms the dot product of two vectors.
    c     uses unrolled loops for increments equal to one.
    c     jack dongarra, linpack, 3/11/78.
    c*/
  double ddot_R;
  double  dtemp;
#undef dy_1
#define dy_1(a1) dy[a1-1]
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix,iy,m,mp1;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  ddot_R = 0.0e0;
  dtemp = 0.0e0;
  if(n<=0)return ddot_R;
  if(incx==1&&incy==1)goto L_20;
  /*c
    c        code for unequal increments or equal increments
    c          not equal to 1
    c*/
  ix = 1;
  iy = 1;
  if(incx<0)ix = (-n+1)*incx + 1;
  if(incy<0)iy = (-n+1)*incy + 1;
  for (i=1 ; i<=n ; i+=1) {
    dtemp = dtemp + dx_1(ix)*dy_1(iy);
    ix = ix + incx;
    iy = iy + incy;
  }
  ddot_R = dtemp;
  return ddot_R;
  /*c
    c        code for both increments equal to 1
    c*/
  /*c
    c        clean-up loop
    c*/
 L_20: m = mod(n,5);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dtemp = dtemp + dx_1(i)*dy_1(i);
  }
  if( n < 5 ) goto L_60;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=5) {
    dtemp = dtemp + dx_1(i)*dy_1(i) + dx_1(i + 1)*dy_1(i + 1) +
      dx_1(i + 2)*dy_1(i + 2) + dx_1(i + 3)*dy_1(i + 3) + dx_1(i + 4)*dy_1(i + 4);
  }
 L_60: ddot_R = dtemp;
  return ddot_R;
}



void dgemm ( char transa, char transb, long m, long n, long k,
	    double alpha, double a[], long lda, double b[], long ldb,
	    double beta, double c[], long ldc )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef c_2
#define c_2(a1,a2) c[a1-1+ldc*(a2-1)]
#undef b_2
#define b_2(a1,a2) b[a1-1+ldb*(a2-1)]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DGEMM  performs one of the matrix-matrix operations
   *
   *     C := alpha*op( A )*op( B ) + beta*C,
   *
   *  where  op( X ) is one of
   *
   *     op( X ) = X   or   op( X ) = X',
   *
   *  alpha and beta are scalars, and A, B and C are matrices, with op( A )
   *  an m by k matrix,  op( B )  a  k by n matrix and  C an m by n matrix.
   *
   *  Parameters
   *  ==========
   *
   *  TRANSA - CHARACTER*1.
   *           On entry, TRANSA specifies the form of op( A ) to be used in
   *           the matrix multiplication as follows:
   *
   *              TRANSA = 'N' or 'n',  op( A ) = A.
   *
   *              TRANSA = 'T' or 't',  op( A ) = A'.
   *
   *              TRANSA = 'C' or 'c',  op( A ) = A'.
   *
   *           Unchanged on exit.
   *
   *  TRANSB - CHARACTER*1.
   *           On entry, TRANSB specifies the form of op( B ) to be used in
   *           the matrix multiplication as follows:
   *
   *              TRANSB = 'N' or 'n',  op( B ) = B.
   *
   *              TRANSB = 'T' or 't',  op( B ) = B'.
   *
   *              TRANSB = 'C' or 'c',  op( B ) = B'.
   *
   *           Unchanged on exit.
   *
   *  M      - INTEGER.
   *           On entry,  M  specifies  the number  of rows  of the  matrix
   *           op( A )  and of the  matrix  C.  M  must  be at least  zero.
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry,  N  specifies the number  of columns of the matrix
   *           op( B ) and the number of columns of the matrix C. N must be
   *           at least zero.
   *           Unchanged on exit.
   *
   *  K      - INTEGER.
   *           On entry,  K  specifies  the number of columns of the matrix
   *           op( A ) and the number of rows of the matrix op( B ). K must
   *           be at least  zero.
   *           Unchanged on exit.
   *
   *  ALPHA  - DOUBLE PRECISION.
   *           On entry, ALPHA specifies the scalar alpha.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
   *           k  when  TRANSA = 'N' or 'n',  and is  m  otherwise.
   *           Before entry with  TRANSA = 'N' or 'n',  the leading  m by k
   *           part of the array  A  must contain the matrix  A,  otherwise
   *           the leading  k by m  part of the array  A  must contain  the
   *           matrix A.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program. When  TRANSA = 'N' or 'n' then
   *           LDA must be at least  max( 1, m ), otherwise  LDA must be at
   *           least  max( 1, k ).
   *           Unchanged on exit.
   *
   *  B      - DOUBLE PRECISION array of DIMENSION ( LDB, kb ), where kb is
   *           n  when  TRANSB = 'N' or 'n',  and is  k  otherwise.
   *           Before entry with  TRANSB = 'N' or 'n',  the leading  k by n
   *           part of the array  B  must contain the matrix  B,  otherwise
   *           the leading  n by k  part of the array  B  must contain  the
   *           matrix B.
   *           Unchanged on exit.
   *
   *  LDB    - INTEGER.
   *           On entry, LDB specifies the first dimension of B as declared
   *           in the calling (sub) program. When  TRANSB = 'N' or 'n' then
   *           LDB must be at least  max( 1, k ), otherwise  LDB must be at
   *           least  max( 1, n ).
   *           Unchanged on exit.
   *
   *  BETA   - DOUBLE PRECISION.
   *           On entry,  BETA  specifies the scalar  beta.  When  BETA  is
   *           supplied as zero then C need not be set on input.
   *           Unchanged on exit.
   *
   *  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
   *           Before entry, the leading  m by n  part of the array  C must
   *           contain the matrix  C,  except when  beta  is zero, in which
   *           case C need not be set on entry.
   *           On exit, the array  C  is overwritten by the  m by n  matrix
   *           ( alpha*op( A )*op( B ) + beta*C ).
   *
   *  LDC    - INTEGER.
   *           On entry, LDC specifies the first dimension of C as declared
   *           in  the  calling  (sub)  program.   LDC  must  be  at  least
   *           max( 1, m ).
   *           Unchanged on exit.
   **/
  /**
   *  Level 3 Blas routine.
   *
   *  -- Written on 8-February-1989.
   *     Jack Dongarra, Argonne National Laboratory.
   *     Iain Duff, AERE Harwell.
   *     Jeremy Du Croz, Numerical Algorithms Group Ltd.
   *     Sven Hammarling, Numerical Algorithms Group Ltd.
   **/
  /**
   *     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     .. Local Scalars ..*/
  int            nota, notb;
  long            i, info, j, l, ncola, nrowa, nrowb;
  double    temp;
  /**     .. Parameters ..*/
#undef one
#define one 1.0e+0
#undef zero
#define zero 0.0e+0
  /**     ..
   *     .. Executable Statements ..
   *
   *     Set  NOTA  and  NOTB  as  true if  A  and  B  respectively are not
   *     transposed and set  NROWA, NCOLA and  NROWB  as the number of rows
   *     and  columns of  A  and the  number of  rows  of  B  respectively.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  nota  = lsame( transa, 'n' );
  notb  = lsame( transb, 'n' );
  if( nota ){
    nrowa = m;
    ncola = k;
  } else {
    nrowa = k;
    ncola = m;
  }
  if( notb ){
    nrowb = k;
  } else {
    nrowb = n;
  }
  /**
   *     Test the input parameters.
   **/
  info = 0;
  if(      ( !nota                 )&&
     ( !lsame( transa, 'c' ) )&&
     ( !lsame( transa, 't' ) )      ){
    info = 1;
  } else if( ( !notb                 )&&
	    ( !lsame( transb, 'c' ) )&&
	    ( !lsame( transb, 't' ) )      ){
    info = 2;
  } else if( m  <0               ){
    info = 3;
  } else if( n  <0               ){
    info = 4;
  } else if( k  <0               ){
    info = 5;
  } else if( lda<max( 1, nrowa ) ){
    info = 8;
  } else if( ldb<max( 1, nrowb ) ){
    info = 10;
  } else if( ldc<max( 1, m     ) ){
    info = 13;
  }
  if( info!=0 ){
    xerbla( "dgemm ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( ( m==0 )||( n==0 )||
     ( ( ( alpha==zero )||( k==0 ) )&&( beta==one ) ) )
    return;
  /**
   *     And if  alpha.eq.zero.
   **/
  if( alpha==zero ){
    if( beta==zero ){
      for (j=1 ; j<=n ; j+=1) {
	for (i=1 ; i<=m ; i+=1) {
	  c_2( i, j ) = zero;
	}
      }
    } else {
      for (j=1 ; j<=n ; j+=1) {
	for (i=1 ; i<=m ; i+=1) {
	  c_2( i, j ) = beta*c_2( i, j );
	}
      }
    }
    return;
  }
  /**
   *     Start the operations.
   **/
  if( notb ){
    if( nota ){
      /**
       *           Form  C := alpha*A*B + beta*C.
       **/
      for (j=1 ; j<=n ; j+=1) {
	if( beta==zero ){
	  for (i=1 ; i<=m ; i+=1) {
	    c_2( i, j ) = zero;
	  }
	} else if( beta!=one ){
	  for (i=1 ; i<=m ; i+=1) {
	    c_2( i, j ) = beta*c_2( i, j );
	  }
	}
	for (l=1 ; l<=k ; l+=1) {
	  if( b_2( l, j )!=zero ){
	    temp = alpha*b_2( l, j );
	    for (i=1 ; i<=m ; i+=1) {
	      c_2( i, j ) = c_2( i, j ) + temp*a_2( i, l );
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  C := alpha*A'*B + beta*C
       **/
      for (j=1 ; j<=n ; j+=1) {
	for (i=1 ; i<=m ; i+=1) {
	  temp = zero;
	  for (l=1 ; l<=k ; l+=1) {
	    temp = temp + a_2( l, i )*b_2( l, j );
	  }
	  if( beta==zero ){
	    c_2( i, j ) = alpha*temp;
	  } else {
	    c_2( i, j ) = alpha*temp + beta*c_2( i, j );
	  }
	}
      }
    }
  } else {
    if( nota ){
      /**
       *           Form  C := alpha*A*B' + beta*C
       **/
      for (j=1 ; j<=n ; j+=1) {
	if( beta==zero ){
	  for (i=1 ; i<=m ; i+=1) {
	    c_2( i, j ) = zero;
	  }
	} else if( beta!=one ){
	  for (i=1 ; i<=m ; i+=1) {
	    c_2( i, j ) = beta*c_2( i, j );
	  }
	}
	for (l=1 ; l<=k ; l+=1) {
	  if( b_2( j, l )!=zero ){
	    temp = alpha*b_2( j, l );
	    for (i=1 ; i<=m ; i+=1) {
	      c_2( i, j ) = c_2( i, j ) + temp*a_2( i, l );
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  C := alpha*A'*B' + beta*C
       **/
      for (j=1 ; j<=n ; j+=1) {
	for (i=1 ; i<=m ; i+=1) {
	  temp = zero;
	  for (l=1 ; l<=k ; l+=1) {
	    temp = temp + a_2( l, i )*b_2( j, l );
	  }
	  if( beta==zero ){
	    c_2( i, j ) = alpha*temp;
	  } else {
	    c_2( i, j ) = alpha*temp + beta*c_2( i, j );
	  }
	}
      }
    }
  }

  return;
  /**
   *     End of DGEMM .
   **/
}



void dgemv ( char trans, long m, long n, double alpha,
	    double a[], long lda, double x[], long incx,
	    double beta, double y[], long incy )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef y_1
#define y_1(a1) y[a1-1]
#undef x_1
#define x_1(a1) x[a1-1]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DGEMV  performs one of the matrix-vector operations
   *
   *     y := alpha*A*x + beta*y,   or   y := alpha*A'*x + beta*y,
   *
   *  where alpha and beta are scalars, x and y are vectors and A is an
   *  m by n matrix.
   *
   *  Parameters
   *  ==========
   *
   *  TRANS  - CHARACTER*1.
   *           On entry, TRANS specifies the operation to be performed as
   *           follows:
   *
   *              TRANS = 'N' or 'n'   y := alpha*A*x + beta*y.
   *
   *              TRANS = 'T' or 't'   y := alpha*A'*x + beta*y.
   *
   *              TRANS = 'C' or 'c'   y := alpha*A'*x + beta*y.
   *
   *           Unchanged on exit.
   *
   *  M      - INTEGER.
   *           On entry, M specifies the number of rows of the matrix A.
   *           M must be at least zero.
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the number of columns of the matrix A.
   *           N must be at least zero.
   *           Unchanged on exit.
   *
   *  ALPHA  - DOUBLE PRECISION.
   *           On entry, ALPHA specifies the scalar alpha.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, n ).
   *           Before entry, the leading m by n part of the array A must
   *           contain the matrix of coefficients.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program. LDA must be at least
   *           max( 1, m ).
   *           Unchanged on exit.
   *
   *  X      - DOUBLE PRECISION array of DIMENSION at least
   *           ( 1 + ( n - 1 )*abs( INCX ) ) when TRANS = 'N' or 'n'
   *           and at least
   *           ( 1 + ( m - 1 )*abs( INCX ) ) otherwise.
   *           Before entry, the incremented array X must contain the
   *           vector x.
   *           Unchanged on exit.
   *
   *  INCX   - INTEGER.
   *           On entry, INCX specifies the increment for the elements of
   *           X. INCX must not be zero.
   *           Unchanged on exit.
   *
   *  BETA   - DOUBLE PRECISION.
   *           On entry, BETA specifies the scalar beta. When BETA is
   *           supplied as zero then Y need not be set on input.
   *           Unchanged on exit.
   *
   *  Y      - DOUBLE PRECISION array of DIMENSION at least
   *           ( 1 + ( m - 1 )*abs( INCY ) ) when TRANS = 'N' or 'n'
   *           and at least
   *           ( 1 + ( n - 1 )*abs( INCY ) ) otherwise.
   *           Before entry with BETA non-zero, the incremented array Y
   *           must contain the vector y. On exit, Y is overwritten by the
   *           updated vector y.
   *
   *  INCY   - INTEGER.
   *           On entry, INCY specifies the increment for the elements of
   *           Y. INCY must not be zero.
   *           Unchanged on exit.
   **/
  /**
   *  Level 2 Blas routine.
   *
   *  -- Written on 22-October-1986.
   *     Jack Dongarra, Argonne National Lab.
   *     Jeremy Du Croz, Nag Central Office.
   *     Sven Hammarling, Nag Central Office.
   *     Richard Hanson, Sandia National Labs.
   **/
  /**
   *     .. Parameters ..*/
#undef one
#define one 1.0e+0
#undef zero
#define zero 0.0e+0
  /**     .. Local Scalars ..*/
  double    temp;
  long            i, info, ix, iy, j, jx, jy, kx, ky, lenx, leny;
  /**     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  info = 0;
  if     ( !lsame( trans, 'n' )&&
	  !lsame( trans, 't' )&&
	  !lsame( trans, 'c' )      ){
    info = 1;
  } else if( m<0 ){
    info = 2;
  } else if( n<0 ){
    info = 3;
  } else if( lda<max( 1, m ) ){
    info = 6;
  } else if( incx==0 ){
    info = 8;
  } else if( incy==0 ){
    info = 11;
  }
  if( info!=0 ){
    xerbla( "dgemv ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( ( m==0 )||( n==0 )||
     ( ( alpha==zero )&&( beta==one ) ) )
    return;
  /**
   *     Set  LENX  and  LENY, the lengths of the vectors x and y, and set
   *     up the start points in  X  and  Y.
   **/
  if( lsame( trans, 'n' ) ){
    lenx = n;
    leny = m;
  } else {
    lenx = m;
    leny = n;
  }
  if( incx>0 ){
    kx = 1;
  } else {
    kx = 1 - ( lenx - 1 )*incx;
  }
  if( incy>0 ){
    ky = 1;
  } else {
    ky = 1 - ( leny - 1 )*incy;
  }
  /**
   *     Start the operations. In this version the elements of A are
   *     accessed sequentially with one pass through A.
   *
   *     First form  y := beta*y.
   **/
  if( beta!=one ){
    if( incy==1 ){
      if( beta==zero ){
	for (i=1 ; i<=leny ; i+=1) {
	  y_1( i ) = zero;
	}
      } else {
	for (i=1 ; i<=leny ; i+=1) {
	  y_1( i ) = beta*y_1( i );
	}
      }
    } else {
      iy = ky;
      if( beta==zero ){
	for (i=1 ; i<=leny ; i+=1) {
	  y_1( iy ) = zero;
	  iy      = iy   + incy;
	}
      } else {
	for (i=1 ; i<=leny ; i+=1) {
	  y_1( iy ) = beta*y_1( iy );
	  iy      = iy           + incy;
	}
      }
    }
  }
  if( alpha==zero )
    return;
  if( lsame( trans, 'n' ) ){
    /**
     *        Form  y := alpha*A*x + y.
     **/
    jx = kx;
    if( incy==1 ){
      for (j=1 ; j<=n ; j+=1) {
	if( x_1( jx )!=zero ){
	  temp = alpha*x_1( jx );
	  for (i=1 ; i<=m ; i+=1) {
	    y_1( i ) = y_1( i ) + temp*a_2( i, j );
	  }
	}
	jx = jx + incx;
      }
    } else {
      for (j=1 ; j<=n ; j+=1) {
	if( x_1( jx )!=zero ){
	  temp = alpha*x_1( jx );
	  iy   = ky;
	  for (i=1 ; i<=m ; i+=1) {
	    y_1( iy ) = y_1( iy ) + temp*a_2( i, j );
	    iy      = iy      + incy;
	  }
	}
	jx = jx + incx;
      }
    }
  } else {
    /**
     *        Form  y := alpha*A'*x + y.
     **/
    jy = ky;
    if( incx==1 ){
      for (j=1 ; j<=n ; j+=1) {
	temp = zero;
	for (i=1 ; i<=m ; i+=1) {
	  temp = temp + a_2( i, j )*x_1( i );
	}
	y_1( jy ) = y_1( jy ) + alpha*temp;
	jy      = jy      + incy;
      }
    } else {
      for (j=1 ; j<=n ; j+=1) {
	temp = zero;
	ix   = kx;
	for (i=1 ; i<=m ; i+=1) {
	  temp = temp + a_2( i, j )*x_1( ix );
	  ix   = ix   + incx;
	}
	y_1( jy ) = y_1( jy ) + alpha*temp;
	jy      = jy      + incy;
      }
    }
  }

  return;
  /**
   *     End of DGEMV .
   **/
}



void dger  ( long m, long n, double alpha, double x[], long incx,
	    double y[], long incy, double a[], long lda )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef y_1
#define y_1(a1) y[a1-1]
#undef x_1
#define x_1(a1) x[a1-1]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DGER   performs the rank 1 operation
   *
   *     A := alpha*x*y' + A,
   *
   *  where alpha is a scalar, x is an m element vector, y is an n element
   *  vector and A is an m by n matrix.
   *
   *  Parameters
   *  ==========
   *
   *  M      - INTEGER.
   *           On entry, M specifies the number of rows of the matrix A.
   *           M must be at least zero.
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the number of columns of the matrix A.
   *           N must be at least zero.
   *           Unchanged on exit.
   *
   *  ALPHA  - DOUBLE PRECISION.
   *           On entry, ALPHA specifies the scalar alpha.
   *           Unchanged on exit.
   *
   *  X      - DOUBLE PRECISION array of dimension at least
   *           ( 1 + ( m - 1 )*abs( INCX ) ).
   *           Before entry, the incremented array X must contain the m
   *           element vector x.
   *           Unchanged on exit.
   *
   *  INCX   - INTEGER.
   *           On entry, INCX specifies the increment for the elements of
   *           X. INCX must not be zero.
   *           Unchanged on exit.
   *
   *  Y      - DOUBLE PRECISION array of dimension at least
   *           ( 1 + ( n - 1 )*abs( INCY ) ).
   *           Before entry, the incremented array Y must contain the n
   *           element vector y.
   *           Unchanged on exit.
   *
   *  INCY   - INTEGER.
   *           On entry, INCY specifies the increment for the elements of
   *           Y. INCY must not be zero.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, n ).
   *           Before entry, the leading m by n part of the array A must
   *           contain the matrix of coefficients. On exit, A is
   *           overwritten by the updated matrix.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program. LDA must be at least
   *           max( 1, m ).
   *           Unchanged on exit.
   **/
  /**
   *  Level 2 Blas routine.
   *
   *  -- Written on 22-October-1986.
   *     Jack Dongarra, Argonne National Lab.
   *     Jeremy Du Croz, Nag Central Office.
   *     Sven Hammarling, Nag Central Office.
   *     Richard Hanson, Sandia National Labs.
   **/
  /**
   *     .. Parameters ..*/
#undef zero
#define zero 0.0e+0
  /**     .. Local Scalars ..*/
  double    temp;
  long            i, info, ix, j, jy, kx;
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  info = 0;
  if     ( m<0 ){
    info = 1;
  } else if( n<0 ){
    info = 2;
  } else if( incx==0 ){
    info = 5;
  } else if( incy==0 ){
    info = 7;
  } else if( lda<max( 1, m ) ){
    info = 9;
  }
  if( info!=0 ){
    xerbla( "dger  ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( ( m==0 )||( n==0 )||( alpha==zero ) )
    return;
  /**
   *     Start the operations. In this version the elements of A are
   *     accessed sequentially with one pass through A.
   **/
  if( incy>0 ){
    jy = 1;
  } else {
    jy = 1 - ( n - 1 )*incy;
  }
  if( incx==1 ){
    for (j=1 ; j<=n ; j+=1) {
      if( y_1( jy )!=zero ){
	temp = alpha*y_1( jy );
	for (i=1 ; i<=m ; i+=1) {
	  a_2( i, j ) = a_2( i, j ) + x_1( i )*temp;
	}
      }
      jy = jy + incy;
    }
  } else {
    if( incx>0 ){
      kx = 1;
    } else {
      kx = 1 - ( m - 1 )*incx;
    }
    for (j=1 ; j<=n ; j+=1) {
      if( y_1( jy )!=zero ){
	temp = alpha*y_1( jy );
	ix   = kx;
	for (i=1 ; i<=m ; i+=1) {
	  a_2( i, j ) = a_2( i, j ) + x_1( ix )*temp;
	  ix        = ix        + incx;
	}
      }
      jy = jy + incy;
    }
  }

  return;
  /**
   *     End of DGER  .
   **/
}



double   dnrm2 ( long n, double dx[], long incx)
{
  double dnrm2_R;
  long i, ix, j, next;
  double /*cuthi,*/ hitest, sum, xmax=0.0;
#undef dx_1
#define dx_1(a1) dx[a1-1]
#undef zero
#define zero 0.0e+0
#undef one
#define one 1.0e+0
  /*c
    c     euclidean norm of the n-vector stored in dx_1() with storage
    c     increment incx .
    c     if    n .le. 0 return with result = 0.
    c     if n .ge. 1 then incx must be .ge. 1
    c
    c           c.l.lawson, 1978 jan 08
    c     modified to correct failure to update ix, 1/25/92.
    c     modified 3/93 to return if incx .le. 0.
    c
    c     four phase method     using two built-in constants that are
    c     hopefully applicable to all machines.
    c         cutlo = maximum of  sqrt(u/eps)  over all known machines.
    c         cuthi = minimum of  sqrt(v)      over all known machines.
    c     where
    c         eps = smallest no. such that eps + 1. .gt. 1.
    c         u   = smallest positive no.   (underflow limit)
    c         v   = largest  no.            (overflow  limit)
    c
    c     brief outline of algorithm..
    c
    c     phase 1    scans zero components.
    c     move to phase 2 when a component is nonzero and .le. cutlo
    c     move to phase 3 when a component is .gt. cutlo
    c     move to phase 4 when a component is .ge. cuthi/m
    c     where m = n for x() real and m = 2*n for complex.
    c
    c     values for cutlo and cuthi..
    c     from the environmental parameters listed in the imsl converter
    c     document the limiting values are as follows..
    c     cutlo, s.p.   u/eps = 2**(-102) for  honeywell.  close seconds are
    c                   univac and dec at 2**(-103)
    c                   thus cutlo = 2**(-51) = 4.44089e-16
    c     cuthi, s.p.   v = 2**127 for univac, honeywell, and dec.
    c                   thus cuthi = 2**(63.5) = 1.30438e19
    c     cutlo, d.p.   u/eps = 2**(-67) for honeywell and dec.
    c                   thus cutlo = 2**(-33.5) = 8.23181d-11
    c     cuthi, d.p.   same as s.p.  cuthi = 1.30438d19
    c     data cutlo, cuthi / 8.232d-11,  1.304d19 /
    c     data cutlo, cuthi / 4.441e-16,  1.304e19 /  */
#undef cutlo
#define cutlo 8.232e-11
#undef cuthi
#define cuthi 1.304e19

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if(n > 0 && incx>0) goto L_10;
  dnrm2_R  = zero;
  goto L_300;

 L_10: next= 1;
  sum = zero;
  i = 1;
  ix = 1;
  /*c                                                 begin main loop*/
 L_20:
  switch (next) {
  case 1: goto L_30;
  case 2: goto L_50;
  case 3: goto L_70;
  case 4: goto L_110;
  }
 L_30: if( dabs(dx_1(i)) > cutlo) goto L_85;
  next= 2;
  xmax = zero;
  /*c
    c                        phase 1.  sum is zero
    c*/
 L_50: if( dx_1(i) == zero) goto L_200;
  if( dabs(dx_1(i)) > cutlo) goto L_85;
  /*c
    c                                prepare for phase 2.*/
  next= 3;
  goto L_105;
  /*c
    c                                prepare for phase 4.
    c*/
 L_100:
  ix = j;
  next= 4;
  sum = (sum / dx_1(i)) / dx_1(i);
 L_105: xmax = dabs(dx_1(i));
  goto L_115;
  /*c
    c                   phase 2.  sum is small.
    c                             scale to avoid destructive underflow.
    c*/
 L_70: if( dabs(dx_1(i)) > cutlo ) goto L_75;
  /*c
    c                     common code for phases 2 and 4.
    c                     in phase 4 sum is large.  scale to avoid overflow.
    c*/
 L_110: if( dabs(dx_1(i)) <= xmax ) goto L_115;
  sum = one + sum * (xmax/dx_1(i))*(xmax/dx_1(i));
  xmax = dabs(dx_1(i));
  goto L_200;

 L_115: sum = sum + (dx_1(i)/xmax)*(dx_1(i)/xmax);
  goto L_200;

  /*c
    c                  prepare for phase 3.
    c*/
 L_75: sum = (sum * xmax) * xmax;

  /*c
    c     for real or d.p. set hitest = cuthi/n
    c     for complex      set hitest = cuthi/(2*n)
    c*/
 L_85: hitest = cuthi/n;
  /*c
    c                   phase 3.  sum is mid-range.  no scaling.
    c*/
  for (j=ix ; j<=n ; j+=1) {
    if(dabs(dx_1(i)) >= hitest) goto L_100;
    sum = sum + dx_1(i)*dx_1(i);
    i = i + incx;
  }
  dnrm2_R = sqrt( sum );
  goto L_300;

 L_200:
  ix = ix + 1;
  i = i + incx;
  if( ix <= n ) goto L_20;
  /*c
    c              end of main loop.
    c
    c              compute square root and adjust for scaling.
    c*/
  dnrm2_R = xmax * sqrt(sum);
 L_300:
  return dnrm2_R;
}



void  drot (long n,double dx[],long incx,double dy[],long incy,
	    double c,double s)
{
  /*c
    c     applies a plane rotation.
    c     jack dongarra, linpack, 3/11/78.
    c*/
  double  dtemp;
#undef dy_1
#define dy_1(a1) dy[a1-1]
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix,iy;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if(n<=0)return;
  if(incx==1&&incy==1)goto L_20;
  /*c
    c       code for unequal increments or equal increments not equal
    c         to 1
    c*/
  ix = 1;
  iy = 1;
  if(incx<0)ix = (-n+1)*incx + 1;
  if(incy<0)iy = (-n+1)*incy + 1;
  for (i=1 ; i<=n ; i+=1) {
    dtemp = c*dx_1(ix) + s*dy_1(iy);
    dy_1(iy) = c*dy_1(iy) - s*dx_1(ix);
    dx_1(ix) = dtemp;
    ix = ix + incx;
    iy = iy + incy;
  }
  return;
  /*c
    c       code for both increments equal to 1
    c*/
 L_20: for (i=1 ; i<=n ; i+=1) {
   dtemp = c*dx_1(i) + s*dy_1(i);
   dy_1(i) = c*dy_1(i) - s*dx_1(i);
   dx_1(i) = dtemp;
 }
  return;
}



void  dscal(long n,double da,double dx[],long incx)
{
  /*c
    c     scales a vector by a constant.
    c     uses unrolled loops for increment equal to one.
    c     jack dongarra, linpack, 3/11/78.
    c     modified 3/93 to return if incx .le. 0.
    c*/
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,m,mp1,nincx;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if( n<=0 || incx<=0 )return;
  if(incx==1)goto L_20;
  /*c
    c        code for increment not equal to 1
    c*/
  nincx = n*incx;
  for (i=1 ; incx>0?i<=nincx:i>=nincx ; i+=incx) {
    dx_1(i) = da*dx_1(i);
  }
  return;
  /*c
    c        code for increment equal to 1
    c*/
  /*c
    c        clean-up loop
    c*/
 L_20: m = mod(n,5);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dx_1(i) = da*dx_1(i);
  }
  if( n < 5 ) return;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=5) {
    dx_1(i) = da*dx_1(i);
    dx_1(i + 1) = da*dx_1(i + 1);
    dx_1(i + 2) = da*dx_1(i + 2);
    dx_1(i + 3) = da*dx_1(i + 3);
    dx_1(i + 4) = da*dx_1(i + 4);
  }
  return;
}



void  dswap (long n,double dx[],long incx,double dy[],long incy)
{
  /*c
    c     interchanges two vectors.
    c     uses unrolled loops for increments equal one.
    c     jack dongarra, linpack, 3/11/78.
    c*/
  double  dtemp;
#undef dy_1
#define dy_1(a1) dy[a1-1]
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix,iy,m,mp1;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  if(n<=0)return;
  if(incx==1&&incy==1)goto L_20;
  /*c
    c       code for unequal increments or equal increments not equal
    c         to 1
    c*/
  ix = 1;
  iy = 1;
  if(incx<0)ix = (-n+1)*incx + 1;
  if(incy<0)iy = (-n+1)*incy + 1;
  for (i=1 ; i<=n ; i+=1) {
    dtemp = dx_1(ix);
    dx_1(ix) = dy_1(iy);
    dy_1(iy) = dtemp;
    ix = ix + incx;
    iy = iy + incy;
  }
  return;
  /*c
    c       code for both increments equal to 1
    c*/
  /*c
    c       clean-up loop
    c*/
 L_20: m = mod(n,3);
  if( m == 0 ) goto L_40;
  for (i=1 ; i<=m ; i+=1) {
    dtemp = dx_1(i);
    dx_1(i) = dy_1(i);
    dy_1(i) = dtemp;
  }
  if( n < 3 ) return;
 L_40: mp1 = m + 1;
  for (i=mp1 ; i<=n ; i+=3) {
    dtemp = dx_1(i);
    dx_1(i) = dy_1(i);
    dy_1(i) = dtemp;
    dtemp = dx_1(i + 1);
    dx_1(i + 1) = dy_1(i + 1);
    dy_1(i + 1) = dtemp;
    dtemp = dx_1(i + 2);
    dx_1(i + 2) = dy_1(i + 2);
    dy_1(i + 2) = dtemp;
  }
  return;
}



void dtrmm ( char side, char uplo, char transa, char diag,
	    long m, long n, double alpha, double a[], long lda,
	    double b[], long ldb )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef b_2
#define b_2(a1,a2) b[a1-1+ldb*(a2-1)]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DTRMM  performs one of the matrix-matrix operations
   *
   *     B := alpha*op( A )*B,   or   B := alpha*B*op( A ),
   *
   *  where  alpha  is a scalar,  B  is an m by n matrix,  A  is a unit, or
   *  non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
   *
   *     op( A ) = A   or   op( A ) = A'.
   *
   *  Parameters
   *  ==========
   *
   *  SIDE   - CHARACTER*1.
   *           On entry,  SIDE specifies whether  op( A ) multiplies B from
   *           the left or right as follows:
   *
   *              SIDE = 'L' or 'l'   B := alpha*op( A )*B.
   *
   *              SIDE = 'R' or 'r'   B := alpha*B*op( A ).
   *
   *           Unchanged on exit.
   *
   *  UPLO   - CHARACTER*1.
   *           On entry, UPLO specifies whether the matrix A is an upper or
   *           lower triangular matrix as follows:
   *
   *              UPLO = 'U' or 'u'   A is an upper triangular matrix.
   *
   *              UPLO = 'L' or 'l'   A is a lower triangular matrix.
   *
   *           Unchanged on exit.
   *
   *  TRANSA - CHARACTER*1.
   *           On entry, TRANSA specifies the form of op( A ) to be used in
   *           the matrix multiplication as follows:
   *
   *              TRANSA = 'N' or 'n'   op( A ) = A.
   *
   *              TRANSA = 'T' or 't'   op( A ) = A'.
   *
   *              TRANSA = 'C' or 'c'   op( A ) = A'.
   *
   *           Unchanged on exit.
   *
   *  DIAG   - CHARACTER*1.
   *           On entry, DIAG specifies whether or not A is unit triangular
   *           as follows:
   *
   *              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
   *
   *              DIAG = 'N' or 'n'   A is not assumed to be unit
   *                                  triangular.
   *
   *           Unchanged on exit.
   *
   *  M      - INTEGER.
   *           On entry, M specifies the number of rows of B. M must be at
   *           least zero.
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the number of columns of B.  N must be
   *           at least zero.
   *           Unchanged on exit.
   *
   *  ALPHA  - DOUBLE PRECISION.
   *           On entry,  ALPHA specifies the scalar  alpha. When  alpha is
   *           zero then  A is not referenced and  B need not be set before
   *           entry.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, k ), where k is m
   *           when  SIDE = 'L' or 'l'  and is  n  when  SIDE = 'R' or 'r'.
   *           Before entry  with  UPLO = 'U' or 'u',  the  leading  k by k
   *           upper triangular part of the array  A must contain the upper
   *           triangular matrix  and the strictly lower triangular part of
   *           A is not referenced.
   *           Before entry  with  UPLO = 'L' or 'l',  the  leading  k by k
   *           lower triangular part of the array  A must contain the lower
   *           triangular matrix  and the strictly upper triangular part of
   *           A is not referenced.
   *           Note that when  DIAG = 'U' or 'u',  the diagonal elements of
   *           A  are not referenced either,  but are assumed to be  unity.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
   *           LDA  must be at least  max( 1, m ),  when  SIDE = 'R' or 'r'
   *           then LDA must be at least max( 1, n ).
   *           Unchanged on exit.
   *
   *  B      - DOUBLE PRECISION array of DIMENSION ( LDB, n ).
   *           Before entry,  the leading  m by n part of the array  B must
   *           contain the matrix  B,  and  on exit  is overwritten  by the
   *           transformed matrix.
   *
   *  LDB    - INTEGER.
   *           On entry, LDB specifies the first dimension of B as declared
   *           in  the  calling  (sub)  program.   LDB  must  be  at  least
   *           max( 1, m ).
   *           Unchanged on exit.
   **/
  /**
   *  Level 3 Blas routine.
   *
   *  -- Written on 8-February-1989.
   *     Jack Dongarra, Argonne National Laboratory.
   *     Iain Duff, AERE Harwell.
   *     Jeremy Du Croz, Numerical Algorithms Group Ltd.
   *     Sven Hammarling, Numerical Algorithms Group Ltd.
   **/
  /**
   *     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     .. Local Scalars ..*/
  int            lside, nounit, upper;
  long            i, info, j, k, nrowa;
  double    temp;
  /**     .. Parameters ..*/
#undef one
#define one 1.0e+0
#undef zero
#define zero 0.0e+0
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  lside  = lsame( side  , 'l' );
  if( lside ){
    nrowa = m;
  } else {
    nrowa = n;
  }
  nounit = lsame( diag  , 'n' );
  upper  = lsame( uplo  , 'u' );

  info   = 0;
  if(      ( !lside                )&&
     ( !lsame( side  , 'r' ) )      ){
    info = 1;
  } else if( ( !upper                )&&
	    ( !lsame( uplo  , 'l' ) )      ){
    info = 2;
  } else if( ( !lsame( transa, 'n' ) )&&
	    ( !lsame( transa, 't' ) )&&
	    ( !lsame( transa, 'c' ) )      ){
    info = 3;
  } else if( ( !lsame( diag  , 'u' ) )&&
	    ( !lsame( diag  , 'n' ) )      ){
    info = 4;
  } else if( m  <0               ){
    info = 5;
  } else if( n  <0               ){
    info = 6;
  } else if( lda<max( 1, nrowa ) ){
    info = 9;
  } else if( ldb<max( 1, m     ) ){
    info = 11;
  }
  if( info!=0 ){
    xerbla( "dtrmm ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( n==0 )
    return;
  /**
   *     And when  alpha.eq.zero.
   **/
  if( alpha==zero ){
    for (j=1 ; j<=n ; j+=1) {
      for (i=1 ; i<=m ; i+=1) {
	b_2( i, j ) = zero;
      }
    }
    return;
  }
  /**
   *     Start the operations.
   **/
  if( lside ){
    if( lsame( transa, 'n' ) ){
      /**
       *           Form  B := alpha*A*B.
       **/
      if( upper ){
	for (j=1 ; j<=n ; j+=1) {
	  for (k=1 ; k<=m ; k+=1) {
	    if( b_2( k, j )!=zero ){
	      temp = alpha*b_2( k, j );
	      for (i=1 ; i<=k - 1 ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*a_2( i, k );
	      }
	      if( nounit )
		temp = temp*a_2( k, k );
	      b_2( k, j ) = temp;
	    }
	  }
	}
      } else {
	for (j=1 ; j<=n ; j+=1) {
	  for (k=m ; k>=1 ; k+=-1) {
	    if( b_2( k, j )!=zero ){
	      temp      = alpha*b_2( k, j );
	      b_2( k, j ) = temp;
	      if( nounit )
		b_2( k, j ) = b_2( k, j )*a_2( k, k );
	      for (i=k + 1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*a_2( i, k );
	      }
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  B := alpha*B*A'.
       **/
      if( upper ){
	for (j=1 ; j<=n ; j+=1) {
	  for (i=m ; i>=1 ; i+=-1) {
	    temp = b_2( i, j );
	    if( nounit )
	      temp = temp*a_2( i, i );
	    for (k=1 ; k<=i - 1 ; k+=1) {
	      temp = temp + a_2( k, i )*b_2( k, j );
	    }
	    b_2( i, j ) = alpha*temp;
	  }
	}
      } else {
	for (j=1 ; j<=n ; j+=1) {
	  for (i=1 ; i<=m ; i+=1) {
	    temp = b_2( i, j );
	    if( nounit )
	      temp = temp*a_2( i, i );
	    for (k=i + 1 ; k<=m ; k+=1) {
	      temp = temp + a_2( k, i )*b_2( k, j );
	    }
	    b_2( i, j ) = alpha*temp;
	  }
	}
      }
    }
  } else {
    if( lsame( transa, 'n' ) ){
      /**
       *           Form  B := alpha*B*A.
       **/
      if( upper ){
	for (j=n ; j>=1 ; j+=-1) {
	  temp = alpha;
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=1 ; i<=m ; i+=1) {
	    b_2( i, j ) = temp*b_2( i, j );
	  }
	  for (k=1 ; k<=j - 1 ; k+=1) {
	    if( a_2( k, j )!=zero ){
	      temp = alpha*a_2( k, j );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*b_2( i, k );
	      }
	    }
	  }
	}
      } else {
	for (j=1 ; j<=n ; j+=1) {
	  temp = alpha;
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=1 ; i<=m ; i+=1) {
	    b_2( i, j ) = temp*b_2( i, j );
	  }
	  for (k=j + 1 ; k<=n ; k+=1) {
	    if( a_2( k, j )!=zero ){
	      temp = alpha*a_2( k, j );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*b_2( i, k );
	      }
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  B := alpha*B*A'.
       **/
      if( upper ){
	for (k=1 ; k<=n ; k+=1) {
	  for (j=1 ; j<=k - 1 ; j+=1) {
	    if( a_2( j, k )!=zero ){
	      temp = alpha*a_2( j, k );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*b_2( i, k );
	      }
	    }
	  }
	  temp = alpha;
	  if( nounit )
	    temp = temp*a_2( k, k );
	  if( temp!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = temp*b_2( i, k );
	    }
	  }
	}
      } else {
	for (k=n ; k>=1 ; k+=-1) {
	  for (j=k + 1 ; j<=n ; j+=1) {
	    if( a_2( j, k )!=zero ){
	      temp = alpha*a_2( j, k );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) + temp*b_2( i, k );
	      }
	    }
	  }
	  temp = alpha;
	  if( nounit )
	    temp = temp*a_2( k, k );
	  if( temp!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = temp*b_2( i, k );
	    }
	  }
	}
      }
    }
  }

  return;
  /**
   *     End of DTRMM .
   **/
}



void dtrmv ( char uplo, char trans, char diag, long n,
	    double a[], long lda, double x[], long incx )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef x_1
#define x_1(a1) x[a1-1]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DTRMV  performs one of the matrix-vector operations
   *
   *     x := A*x,   or   x := A'*x,
   *
   *  where x is an n element vector and  A is an n by n unit, or non-unit,
   *  upper or lower triangular matrix.
   *
   *  Parameters
   *  ==========
   *
   *  UPLO   - CHARACTER*1.
   *           On entry, UPLO specifies whether the matrix is an upper or
   *           lower triangular matrix as follows:
   *
   *              UPLO = 'U' or 'u'   A is an upper triangular matrix.
   *
   *              UPLO = 'L' or 'l'   A is a lower triangular matrix.
   *
   *           Unchanged on exit.
   *
   *  TRANS  - CHARACTER*1.
   *           On entry, TRANS specifies the operation to be performed as
   *           follows:
   *
   *              TRANS = 'N' or 'n'   x := A*x.
   *
   *              TRANS = 'T' or 't'   x := A'*x.
   *
   *              TRANS = 'C' or 'c'   x := A'*x.
   *
   *           Unchanged on exit.
   *
   *  DIAG   - CHARACTER*1.
   *           On entry, DIAG specifies whether or not A is unit
   *           triangular as follows:
   *
   *              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
   *
   *              DIAG = 'N' or 'n'   A is not assumed to be unit
   *                                  triangular.
   *
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the order of the matrix A.
   *           N must be at least zero.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, n ).
   *           Before entry with  UPLO = 'U' or 'u', the leading n by n
   *           upper triangular part of the array A must contain the upper
   *           triangular matrix and the strictly lower triangular part of
   *           A is not referenced.
   *           Before entry with UPLO = 'L' or 'l', the leading n by n
   *           lower triangular part of the array A must contain the lower
   *           triangular matrix and the strictly upper triangular part of
   *           A is not referenced.
   *           Note that when  DIAG = 'U' or 'u', the diagonal elements of
   *           A are not referenced either, but are assumed to be unity.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program. LDA must be at least
   *           max( 1, n ).
   *           Unchanged on exit.
   *
   *  X      - DOUBLE PRECISION array of dimension at least
   *           ( 1 + ( n - 1 )*abs( INCX ) ).
   *           Before entry, the incremented array X must contain the n
   *           element vector x. On exit, X is overwritten with the
   *           tranformed vector x.
   *
   *  INCX   - INTEGER.
   *           On entry, INCX specifies the increment for the elements of
   *           X. INCX must not be zero.
   *           Unchanged on exit.
   **/
  /**
   *  Level 2 Blas routine.
   *
   *  -- Written on 22-October-1986.
   *     Jack Dongarra, Argonne National Lab.
   *     Jeremy Du Croz, Nag Central Office.
   *     Sven Hammarling, Nag Central Office.
   *     Richard Hanson, Sandia National Labs.
   **/
  /**
   *     .. Parameters ..*/
#undef zero
#define zero 0.0e+0
  /**     .. Local Scalars ..*/
  double    temp;
  long            i, info, ix, j, jx, kx=0;
  int            nounit;
  /**     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  info = 0;
  if     ( !lsame( uplo , 'u' )&&
	  !lsame( uplo , 'l' )      ){
    info = 1;
  } else if( !lsame( trans, 'n' )&&
	    !lsame( trans, 't' )&&
	    !lsame( trans, 'c' )      ){
    info = 2;
  } else if( !lsame( diag , 'u' )&&
	    !lsame( diag , 'n' )      ){
    info = 3;
  } else if( n<0 ){
    info = 4;
  } else if( lda<max( 1, n ) ){
    info = 6;
  } else if( incx==0 ){
    info = 8;
  }
  if( info!=0 ){
    xerbla( "dtrmv ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( n==0 )
    return;

  nounit = lsame( diag, 'n' );
  /**
   *     Set up the start point in X if the increment is not unity. This
   *     will be  ( N - 1 )*INCX  too small for descending loops.
   **/
  if( incx<=0 ){
    kx = 1 - ( n - 1 )*incx;
  } else if( incx!=1 ){
    kx = 1;
  }
  /**
   *     Start the operations. In this version the elements of A are
   *     accessed sequentially with one pass through A.
   **/
  if( lsame( trans, 'n' ) ){
    /**
     *        Form  x := A*x.
     **/
    if( lsame( uplo, 'u' ) ){
      if( incx==1 ){
	for (j=1 ; j<=n ; j+=1) {
	  if( x_1( j )!=zero ){
	    temp = x_1( j );
	    for (i=1 ; i<=j - 1 ; i+=1) {
	      x_1( i ) = x_1( i ) + temp*a_2( i, j );
	    }
	    if( nounit )
	      x_1( j ) = x_1( j )*a_2( j, j );
	  }
	}
      } else {
	jx = kx;
	for (j=1 ; j<=n ; j+=1) {
	  if( x_1( jx )!=zero ){
	    temp = x_1( jx );
	    ix   = kx;
	    for (i=1 ; i<=j - 1 ; i+=1) {
	      x_1( ix ) = x_1( ix ) + temp*a_2( i, j );
	      ix      = ix      + incx;
	    }
	    if( nounit )
	      x_1( jx ) = x_1( jx )*a_2( j, j );
	  }
	  jx = jx + incx;
	}
      }
    } else {
      if( incx==1 ){
	for (j=n ; j>=1 ; j+=-1) {
	  if( x_1( j )!=zero ){
	    temp = x_1( j );
	    for (i=n ; i>=j + 1 ; i+=-1) {
	      x_1( i ) = x_1( i ) + temp*a_2( i, j );
	    }
	    if( nounit )
	      x_1( j ) = x_1( j )*a_2( j, j );
	  }
	}
      } else {
	kx = kx + ( n - 1 )*incx;
	jx = kx;
	for (j=n ; j>=1 ; j+=-1) {
	  if( x_1( jx )!=zero ){
	    temp = x_1( jx );
	    ix   = kx;
	    for (i=n ; i>=j + 1 ; i+=-1) {
	      x_1( ix ) = x_1( ix ) + temp*a_2( i, j );
	      ix      = ix      - incx;
	    }
	    if( nounit )
	      x_1( jx ) = x_1( jx )*a_2( j, j );
	  }
	  jx = jx - incx;
	}
      }
    }
  } else {
    /**
     *        Form  x := A'*x.
     **/
    if( lsame( uplo, 'u' ) ){
      if( incx==1 ){
	for (j=n ; j>=1 ; j+=-1) {
	  temp = x_1( j );
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=j - 1 ; i>=1 ; i+=-1) {
	    temp = temp + a_2( i, j )*x_1( i );
	  }
	  x_1( j ) = temp;
	}
      } else {
	jx = kx + ( n - 1 )*incx;
	for (j=n ; j>=1 ; j+=-1) {
	  temp = x_1( jx );
	  ix   = jx;
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=j - 1 ; i>=1 ; i+=-1) {
	    ix   = ix   - incx;
	    temp = temp + a_2( i, j )*x_1( ix );
	  }
	  x_1( jx ) = temp;
	  jx      = jx   - incx;
	}
      }
    } else {
      if( incx==1 ){
	for (j=1 ; j<=n ; j+=1) {
	  temp = x_1( j );
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=j + 1 ; i<=n ; i+=1) {
	    temp = temp + a_2( i, j )*x_1( i );
	  }
	  x_1( j ) = temp;
	}
      } else {
	jx = kx;
	for (j=1 ; j<=n ; j+=1) {
	  temp = x_1( jx );
	  ix   = jx;
	  if( nounit )
	    temp = temp*a_2( j, j );
	  for (i=j + 1 ; i<=n ; i+=1) {
	    ix   = ix   + incx;
	    temp = temp + a_2( i, j )*x_1( ix );
	  }
	  x_1( jx ) = temp;
	  jx      = jx   + incx;
	}
      }
    }
  }

  return;
  /**
   *     End of DTRMV .
   **/
}



void dtrsm ( char side, char uplo, char transa, char diag,
	    long m, long n, double alpha, double a[], long lda,
	    double b[], long ldb )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef b_2
#define b_2(a1,a2) b[a1-1+ldb*(a2-1)]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DTRSM  solves one of the matrix equations
   *
   *     op( A )*X = alpha*B,   or   X*op( A ) = alpha*B,
   *
   *  where alpha is a scalar, X and B are m by n matrices, A is a unit, or
   *  non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
   *
   *     op( A ) = A   or   op( A ) = A'.
   *
   *  The matrix X is overwritten on B.
   *
   *  Parameters
   *  ==========
   *
   *  SIDE   - CHARACTER*1.
   *           On entry, SIDE specifies whether op( A ) appears on the left
   *           or right of X as follows:
   *
   *              SIDE = 'L' or 'l'   op( A )*X = alpha*B.
   *
   *              SIDE = 'R' or 'r'   X*op( A ) = alpha*B.
   *
   *           Unchanged on exit.
   *
   *  UPLO   - CHARACTER*1.
   *           On entry, UPLO specifies whether the matrix A is an upper or
   *           lower triangular matrix as follows:
   *
   *              UPLO = 'U' or 'u'   A is an upper triangular matrix.
   *
   *              UPLO = 'L' or 'l'   A is a lower triangular matrix.
   *
   *           Unchanged on exit.
   *
   *  TRANSA - CHARACTER*1.
   *           On entry, TRANSA specifies the form of op( A ) to be used in
   *           the matrix multiplication as follows:
   *
   *              TRANSA = 'N' or 'n'   op( A ) = A.
   *
   *              TRANSA = 'T' or 't'   op( A ) = A'.
   *
   *              TRANSA = 'C' or 'c'   op( A ) = A'.
   *
   *           Unchanged on exit.
   *
   *  DIAG   - CHARACTER*1.
   *           On entry, DIAG specifies whether or not A is unit triangular
   *           as follows:
   *
   *              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
   *
   *              DIAG = 'N' or 'n'   A is not assumed to be unit
   *                                  triangular.
   *
   *           Unchanged on exit.
   *
   *  M      - INTEGER.
   *           On entry, M specifies the number of rows of B. M must be at
   *           least zero.
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the number of columns of B.  N must be
   *           at least zero.
   *           Unchanged on exit.
   *
   *  ALPHA  - DOUBLE PRECISION.
   *           On entry,  ALPHA specifies the scalar  alpha. When  alpha is
   *           zero then  A is not referenced and  B need not be set before
   *           entry.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, k ), where k is m
   *           when  SIDE = 'L' or 'l'  and is  n  when  SIDE = 'R' or 'r'.
   *           Before entry  with  UPLO = 'U' or 'u',  the  leading  k by k
   *           upper triangular part of the array  A must contain the upper
   *           triangular matrix  and the strictly lower triangular part of
   *           A is not referenced.
   *           Before entry  with  UPLO = 'L' or 'l',  the  leading  k by k
   *           lower triangular part of the array  A must contain the lower
   *           triangular matrix  and the strictly upper triangular part of
   *           A is not referenced.
   *           Note that when  DIAG = 'U' or 'u',  the diagonal elements of
   *           A  are not referenced either,  but are assumed to be  unity.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
   *           LDA  must be at least  max( 1, m ),  when  SIDE = 'R' or 'r'
   *           then LDA must be at least max( 1, n ).
   *           Unchanged on exit.
   *
   *  B      - DOUBLE PRECISION array of DIMENSION ( LDB, n ).
   *           Before entry,  the leading  m by n part of the array  B must
   *           contain  the  right-hand  side  matrix  B,  and  on exit  is
   *           overwritten by the solution matrix  X.
   *
   *  LDB    - INTEGER.
   *           On entry, LDB specifies the first dimension of B as declared
   *           in  the  calling  (sub)  program.   LDB  must  be  at  least
   *           max( 1, m ).
   *           Unchanged on exit.
   **/
  /**
   *  Level 3 Blas routine.
   **/
  /**
   *  -- Written on 8-February-1989.
   *     Jack Dongarra, Argonne National Laboratory.
   *     Iain Duff, AERE Harwell.
   *     Jeremy Du Croz, Numerical Algorithms Group Ltd.
   *     Sven Hammarling, Numerical Algorithms Group Ltd.
   **/
  /**
   *     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     .. Local Scalars ..*/
  int            lside, nounit, upper;
  long            i, info, j, k, nrowa;
  double    temp;
  /**     .. Parameters ..*/
#undef one
#define one 1.0e+0
#undef zero
#define zero 0.0e+0
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  lside  = lsame( side  , 'l' );
  if( lside ){
    nrowa = m;
  } else {
    nrowa = n;
  }
  nounit = lsame( diag  , 'n' );
  upper  = lsame( uplo  , 'u' );

  info   = 0;
  if(      ( !lside                )&&
     ( !lsame( side  , 'r' ) )      ){
    info = 1;
  } else if( ( !upper                )&&
	    ( !lsame( uplo  , 'l' ) )      ){
    info = 2;
  } else if( ( !lsame( transa, 'n' ) )&&
	    ( !lsame( transa, 't' ) )&&
	    ( !lsame( transa, 'c' ) )      ){
    info = 3;
  } else if( ( !lsame( diag  , 'u' ) )&&
	    ( !lsame( diag  , 'n' ) )      ){
    info = 4;
  } else if( m  <0               ){
    info = 5;
  } else if( n  <0               ){
    info = 6;
  } else if( lda<max( 1, nrowa ) ){
    info = 9;
  } else if( ldb<max( 1, m     ) ){
    info = 11;
  }
  if( info!=0 ){
    xerbla( "dtrsm ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( n==0 )
    return;
  /**
   *     And when  alpha.eq.zero.
   **/
  if( alpha==zero ){
    for (j=1 ; j<=n ; j+=1) {
      for (i=1 ; i<=m ; i+=1) {
	b_2( i, j ) = zero;
      }
    }
    return;
  }
  /**
   *     Start the operations.
   **/
  if( lside ){
    if( lsame( transa, 'n' ) ){
      /**
       *           Form  B := alpha*inv( A )*B.
       **/
      if( upper ){
	for (j=1 ; j<=n ; j+=1) {
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = alpha*b_2( i, j );
	    }
	  }
	  for (k=m ; k>=1 ; k+=-1) {
	    if( b_2( k, j )!=zero ){
	      if( nounit )
		b_2( k, j ) = b_2( k, j )/a_2( k, k );
	      for (i=1 ; i<=k - 1 ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - b_2( k, j )*a_2( i, k );
	      }
	    }
	  }
	}
      } else {
	for (j=1 ; j<=n ; j+=1) {
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = alpha*b_2( i, j );
	    }
	  }
	  for (k=1 ; k<=m ; k+=1) {
	    if( b_2( k, j )!=zero ){
	      if( nounit )
		b_2( k, j ) = b_2( k, j )/a_2( k, k );
	      for (i=k + 1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - b_2( k, j )*a_2( i, k );
	      }
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  B := alpha*inv( A' )*B.
       **/
      if( upper ){
	for (j=1 ; j<=n ; j+=1) {
	  for (i=1 ; i<=m ; i+=1) {
	    temp = alpha*b_2( i, j );
	    for (k=1 ; k<=i - 1 ; k+=1) {
	      temp = temp - a_2( k, i )*b_2( k, j );
	    }
	    if( nounit )
	      temp = temp/a_2( i, i );
	    b_2( i, j ) = temp;
	  }
	}
      } else {
	for (j=1 ; j<=n ; j+=1) {
	  for (i=m ; i>=1 ; i+=-1) {
	    temp = alpha*b_2( i, j );
	    for (k=i + 1 ; k<=m ; k+=1) {
	      temp = temp - a_2( k, i )*b_2( k, j );
	    }
	    if( nounit )
	      temp = temp/a_2( i, i );
	    b_2( i, j ) = temp;
	  }
	}
      }
    }
  } else {
    if( lsame( transa, 'n' ) ){
      /**
       *           Form  B := alpha*B*inv( A ).
       **/
      if( upper ){
	for (j=1 ; j<=n ; j+=1) {
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = alpha*b_2( i, j );
	    }
	  }
	  for (k=1 ; k<=j - 1 ; k+=1) {
	    if( a_2( k, j )!=zero ){
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - a_2( k, j )*b_2( i, k );
	      }
	    }
	  }
	  if( nounit ){
	    temp = one/a_2( j, j );
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = temp*b_2( i, j );
	    }
	  }
	}
      } else {
	for (j=n ; j>=1 ; j+=-1) {
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = alpha*b_2( i, j );
	    }
	  }
	  for (k=j + 1 ; k<=n ; k+=1) {
	    if( a_2( k, j )!=zero ){
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - a_2( k, j )*b_2( i, k );
	      }
	    }
	  }
	  if( nounit ){
	    temp = one/a_2( j, j );
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, j ) = temp*b_2( i, j );
	    }
	  }
	}
      }
    } else {
      /**
       *           Form  B := alpha*B*inv( A' ).
       **/
      if( upper ){
	for (k=n ; k>=1 ; k+=-1) {
	  if( nounit ){
	    temp = one/a_2( k, k );
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = temp*b_2( i, k );
	    }
	  }
	  for (j=1 ; j<=k - 1 ; j+=1) {
	    if( a_2( j, k )!=zero ){
	      temp = a_2( j, k );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - temp*b_2( i, k );
	      }
	    }
	  }
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = alpha*b_2( i, k );
	    }
	  }
	}
      } else {
	for (k=1 ; k<=n ; k+=1) {
	  if( nounit ){
	    temp = one/a_2( k, k );
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = temp*b_2( i, k );
	    }
	  }
	  for (j=k + 1 ; j<=n ; j+=1) {
	    if( a_2( j, k )!=zero ){
	      temp = a_2( j, k );
	      for (i=1 ; i<=m ; i+=1) {
		b_2( i, j ) = b_2( i, j ) - temp*b_2( i, k );
	      }
	    }
	  }
	  if( alpha!=one ){
	    for (i=1 ; i<=m ; i+=1) {
	      b_2( i, k ) = alpha*b_2( i, k );
	    }
	  }
	}
      }
    }
  }

  return;
  /**
   *     End of DTRSM .
   **/
}



void dtrsv ( char uplo, char trans, char diag, long n,
	    double a[], long lda, double x[], long incx )
{
  /**     .. Scalar Arguments ..*/
  /**     .. Array Arguments ..*/
#undef x_1
#define x_1(a1) x[a1-1]
#undef a_2
#define a_2(a1,a2) a[a1-1+lda*(a2-1)]
  /**     ..
   *
   *  Purpose
   *  =======
   *
   *  DTRSV  solves one of the systems of equations
   *
   *     A*x = b,   or   A'*x = b,
   *
   *  where b and x are n element vectors and A is an n by n unit, or
   *  non-unit, upper or lower triangular matrix.
   *
   *  No test for singularity or near-singularity is included in this
   *  routine. Such tests must be performed before calling this routine.
   *
   *  Parameters
   *  ==========
   *
   *  UPLO   - CHARACTER*1.
   *           On entry, UPLO specifies whether the matrix is an upper or
   *           lower triangular matrix as follows:
   *
   *              UPLO = 'U' or 'u'   A is an upper triangular matrix.
   *
   *              UPLO = 'L' or 'l'   A is a lower triangular matrix.
   *
   *           Unchanged on exit.
   *
   *  TRANS  - CHARACTER*1.
   *           On entry, TRANS specifies the equations to be solved as
   *           follows:
   *
   *              TRANS = 'N' or 'n'   A*x = b.
   *
   *              TRANS = 'T' or 't'   A'*x = b.
   *
   *              TRANS = 'C' or 'c'   A'*x = b.
   *
   *           Unchanged on exit.
   *
   *  DIAG   - CHARACTER*1.
   *           On entry, DIAG specifies whether or not A is unit
   *           triangular as follows:
   *
   *              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
   *
   *              DIAG = 'N' or 'n'   A is not assumed to be unit
   *                                  triangular.
   *
   *           Unchanged on exit.
   *
   *  N      - INTEGER.
   *           On entry, N specifies the order of the matrix A.
   *           N must be at least zero.
   *           Unchanged on exit.
   *
   *  A      - DOUBLE PRECISION array of DIMENSION ( LDA, n ).
   *           Before entry with  UPLO = 'U' or 'u', the leading n by n
   *           upper triangular part of the array A must contain the upper
   *           triangular matrix and the strictly lower triangular part of
   *           A is not referenced.
   *           Before entry with UPLO = 'L' or 'l', the leading n by n
   *           lower triangular part of the array A must contain the lower
   *           triangular matrix and the strictly upper triangular part of
   *           A is not referenced.
   *           Note that when  DIAG = 'U' or 'u', the diagonal elements of
   *           A are not referenced either, but are assumed to be unity.
   *           Unchanged on exit.
   *
   *  LDA    - INTEGER.
   *           On entry, LDA specifies the first dimension of A as declared
   *           in the calling (sub) program. LDA must be at least
   *           max( 1, n ).
   *           Unchanged on exit.
   *
   *  X      - DOUBLE PRECISION array of dimension at least
   *           ( 1 + ( n - 1 )*abs( INCX ) ).
   *           Before entry, the incremented array X must contain the n
   *           element right-hand side vector b. On exit, X is overwritten
   *           with the solution vector x.
   *
   *  INCX   - INTEGER.
   *           On entry, INCX specifies the increment for the elements of
   *           X. INCX must not be zero.
   *           Unchanged on exit.
   **/
  /**
   *  Level 2 Blas routine.
   *
   *  -- Written on 22-October-1986.
   *     Jack Dongarra, Argonne National Lab.
   *     Jeremy Du Croz, Nag Central Office.
   *     Sven Hammarling, Nag Central Office.
   *     Richard Hanson, Sandia National Labs.
   **/
  /**
   *     .. Parameters ..*/
#undef zero
#define zero 0.0e+0
  /**     .. Local Scalars ..*/
  double    temp;
  long            i, info, ix, j, jx, kx=0;
  int            nounit;
  /**     .. External Functions ..*/
  /*extern           lsame();*/
  /**     .. External Subroutines ..*/
  extern void xerbla(char *,long);
  /**     .. Intrinsic Functions ..*/
  /*      intrinsic          max;*/
  /**     ..
   *     .. Executable Statements ..
   *
   *     Test the input parameters.
   **/
  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  info = 0;
  if     ( !lsame( uplo , 'u' )&&
	  !lsame( uplo , 'l' )      ){
    info = 1;
  } else if( !lsame( trans, 'n' )&&
	    !lsame( trans, 't' )&&
	    !lsame( trans, 'c' )      ){
    info = 2;
  } else if( !lsame( diag , 'u' )&&
	    !lsame( diag , 'n' )      ){
    info = 3;
  } else if( n<0 ){
    info = 4;
  } else if( lda<max( 1, n ) ){
    info = 6;
  } else if( incx==0 ){
    info = 8;
  }
  if( info!=0 ){
    xerbla( "dtrsv ", info );
    return;
  }
  /**
   *     Quick return if possible.
   **/
  if( n==0 )
    return;

  nounit = lsame( diag, 'n' );
  /**
   *     Set up the start point in X if the increment is not unity. This
   *     will be  ( N - 1 )*INCX  too small for descending loops.
   **/
  if( incx<=0 ){
    kx = 1 - ( n - 1 )*incx;
  } else if( incx!=1 ){
    kx = 1;
  }
  /**
   *     Start the operations. In this version the elements of A are
   *     accessed sequentially with one pass through A.
   **/
  if( lsame( trans, 'n' ) ){
    /**
     *        Form  x := inv( A )*x.
     **/
    if( lsame( uplo, 'u' ) ){
      if( incx==1 ){
	for (j=n ; j>=1 ; j+=-1) {
	  if( x_1( j )!=zero ){
	    if( nounit )
	      x_1( j ) = x_1( j )/a_2( j, j );
	    temp = x_1( j );
	    for (i=j - 1 ; i>=1 ; i+=-1) {
	      x_1( i ) = x_1( i ) - temp*a_2( i, j );
	    }
	  }
	}
      } else {
	jx = kx + ( n - 1 )*incx;
	for (j=n ; j>=1 ; j+=-1) {
	  if( x_1( jx )!=zero ){
	    if( nounit )
	      x_1( jx ) = x_1( jx )/a_2( j, j );
	    temp = x_1( jx );
	    ix   = jx;
	    for (i=j - 1 ; i>=1 ; i+=-1) {
	      ix      = ix      - incx;
	      x_1( ix ) = x_1( ix ) - temp*a_2( i, j );
	    }
	  }
	  jx = jx - incx;
	}
      }
    } else {
      if( incx==1 ){
	for (j=1 ; j<=n ; j+=1) {
	  if( x_1( j )!=zero ){
	    if( nounit )
	      x_1( j ) = x_1( j )/a_2( j, j );
	    temp = x_1( j );
	    for (i=j + 1 ; i<=n ; i+=1) {
	      x_1( i ) = x_1( i ) - temp*a_2( i, j );
	    }
	  }
	}
      } else {
	jx = kx;
	for (j=1 ; j<=n ; j+=1) {
	  if( x_1( jx )!=zero ){
	    if( nounit )
	      x_1( jx ) = x_1( jx )/a_2( j, j );
	    temp = x_1( jx );
	    ix   = jx;
	    for (i=j + 1 ; i<=n ; i+=1) {
	      ix      = ix      + incx;
	      x_1( ix ) = x_1( ix ) - temp*a_2( i, j );
	    }
	  }
	  jx = jx + incx;
	}
      }
    }
  } else {
    /**
     *        Form  x := inv( A' )*x.
     **/
    if( lsame( uplo, 'u' ) ){
      if( incx==1 ){
	for (j=1 ; j<=n ; j+=1) {
	  temp = x_1( j );
	  for (i=1 ; i<=j - 1 ; i+=1) {
	    temp = temp - a_2( i, j )*x_1( i );
	  }
	  if( nounit )
	    temp = temp/a_2( j, j );
	  x_1( j ) = temp;
	}
      } else {
	jx = kx;
	for (j=1 ; j<=n ; j+=1) {
	  temp = x_1( jx );
	  ix   = kx;
	  for (i=1 ; i<=j - 1 ; i+=1) {
	    temp = temp - a_2( i, j )*x_1( ix );
	    ix   = ix   + incx;
	  }
	  if( nounit )
	    temp = temp/a_2( j, j );
	  x_1( jx ) = temp;
	  jx      = jx   + incx;
	}
      }
    } else {
      if( incx==1 ){
	for (j=n ; j>=1 ; j+=-1) {
	  temp = x_1( j );
	  for (i=n ; i>=j + 1 ; i+=-1) {
	    temp = temp - a_2( i, j )*x_1( i );
	  }
	  if( nounit )
	    temp = temp/a_2( j, j );
	  x_1( j ) = temp;
	}
      } else {
	kx = kx + ( n - 1 )*incx;
	jx = kx;
	for (j=n ; j>=1 ; j+=-1) {
	  temp = x_1( jx );
	  ix   = kx;
	  for (i=n ; i>=j + 1 ; i+=-1) {
	    temp = temp - a_2( i, j )*x_1( ix );
	    ix   = ix   - incx;
	  }
	  if( nounit )
	    temp = temp/a_2( j, j );
	  x_1( jx ) = temp;
	  jx      = jx   - incx;
	}
      }
    }
  }

  return;
  /**
   *     End of DTRSV .
   **/
}



long  idamax(long n,double dx[],long incx)
{
  /*c
    c     finds the index of element having max. absolute value.
    c     jack dongarra, linpack, 3/11/78.
    c     modified 3/93 to return if incx .le. 0.
    c*/
  long idamax_R;
  double  dmax;
#undef dx_1
#define dx_1(a1) dx[a1-1]
  long i,ix;

  /*-----implicit-declarations-----*/
  /*-----end-of-declarations-----*/
  idamax_R = 0;
  if( n<1 || incx<=0 ) return idamax_R;
  idamax_R = 1;
  if(n==1)return idamax_R;
  if(incx==1)goto L_20;
  /*c
    c        code for increment not equal to 1
    c*/
  ix = 1;
  dmax = dabs(dx_1(1));
  ix = ix + incx;
  for (i=2 ; i<=n ; i+=1) {
    if(dabs(dx_1(ix))<=dmax) goto L_5;
    idamax_R = i;
    dmax = dabs(dx_1(ix));
  L_5:    ix = ix + incx;
  }
  return idamax_R;
  /*c
    c        code for increment equal to 1
    c*/
 L_20: dmax = dabs(dx_1(1));
  for (i=2 ; i<=n ; i+=1) {
    if(dabs(dx_1(i))<=dmax) continue;
    idamax_R = i;
    dmax = dabs(dx_1(i));
  }
  return idamax_R;
}
