/*
 *  Routines for sampling from Pro-Audio Spectrum 16bit soundcards
 *  These routines require the libraries from the MediaVision SDK.
 *  This SDK is available from ftp.uwp.edu:/pub/msdos/proaudio/passdk30.zip
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "freq.h"

#ifdef SC_PAS16

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <graphics.h>
#include <pcmio.h>                   // Standard PCM record functions
#include <state.h>                   // Need this for Mixer code
#include <mixers.h>                  // Mixer setting functions
#include <binary.h>                  // Hardware values
#include "extern.h"

/* Use default DMA and IRQ values, 4k DMA buffer divided into 4 parts */
#define DMA         -1
#define IRQ         -1
#define DMASize      4
#define DMADivisions 4
#define stereo       0
#define Compression  0
#define SampleSize  16
/* Variables for saving/restoring mixer values */
int LeftChannel;
int RightChannel;
int FilterVal;

/* Turn off stack checking for this routine */
#pragma option -N-
void far callback1()
{
   /*
    *  Callback function.  This function is called every time a buffer has
    *  been filled.  It sets a flag so the main loop recognises and processes
    *  the buffer.
    */
   flag[record_buffer]=1;
   if(++record_buffer>=BUFFERS)
      record_buffer=0;
}
/* Restore stack checking to the command-line specified state */
#pragma option -N.

void init_pas16(void)
{
   /*
    *  Initialize link to Mixer control routines
    *  Then save current mixer settings and turn off the PCM output
    *  mixer to stop feedback and clicking noises.
    */
   DOUT("PAS: Initializing mixer connection");
   MVInitMixerCode(0);

   DOUT("PAS: Getting mixer values");
   mic_level=cMVGetMixerFunction(BI_INPUTMIXER,BI_R_MIC)&0xff;
   ext_level=cMVGetMixerFunction(BI_INPUTMIXER,BI_R_EXT)&0xff;
   int_level=cMVGetMixerFunction(BI_INPUTMIXER,BI_R_INT)&0xff;
   LeftChannel=cMVGetMixerFunction(BI_OUTPUTMIXER,BI_L_PCM)&0xff;
   RightChannel=cMVGetMixerFunction(BI_OUTPUTMIXER,BI_R_PCM)&0xff;
   FilterVal=cMVGetFilterFunction();

   DOUT("PAS: Setting PCM output mixer levels");
   cMVSetMixerFunction(0,BI_OUTPUTMIXER,BI_L_PCM);
   cMVSetMixerFunction(0,BI_OUTPUTMIXER,BI_R_PCM);

   reset_soundcard=reset_pas16;
   halt_soundcard=halt_pas16;
   cleanup_soundcard=cleanup_pas16;
   recordblock=recordblock_pas16;
   set_mixer=set_mixer_pas16;
   sample_size=16;
   mixers=1;
}

void reset_pas16(void)
{
   int i;
   DOUT("PAS: Setting filter function");
   cMVSetFilterFunction(SampleRate/441);

   DOUT("PAS: Enabling PCM buffering");
   if(OpenPCMBuffering(DMA,IRQ,DMASize,DMADivisions)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error trying to open PCM buffering.");
      exit(1);
   }
   DOUT("PAS: Setting PCM state");
   if(PCMState(SampleRate,stereo,Compression,SampleSize)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error setting sample rate.");
      exit(1);
   }
   DOUT("PAS: Setting up the buffers");
   // Reset the buffer pointers
   queue_buffer=0;     // Pointer to next buffer to be queued
   record_buffer=0;    // Pointer to next buffer to be filled
   process_buffer=0;   // Pointer to next buffer to be FFTed
   for(i=0;i<BUFFERS;i++)
      flag[i]=0;
   /*
    *  Queue up all but the last buffer.  The last is queued
    *  up below, with the RecordThisBlock call.
    *  DMA transfer does not start until the Record Block call is made.
    */
   for(i=0;i<BUFFERS-1;i++)
   {
      if(QueueThisBlock((char far *)buffer[queue_buffer],fftlen*2*(stereo+1),callback1)!=0)
      {
         cleanup_pas16();
         closegraph();
         puts("Error queueing block.");
         exit(1);
      }
      if(++queue_buffer>=BUFFERS)
         queue_buffer=0;
   }
   /*
    *  This function starts the DMA process.
    */
   //recordblock_pas16((char far *)buffer[queue_buffer]);
   DOUT("Starting the recording process");
   if(RecordThisBlock((char far *)buffer[queue_buffer],fftlen*2*(stereo+1),callback1)==0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error recording block.");
      exit(1);
   }
   if(++queue_buffer>=BUFFERS)
      queue_buffer=0;
}

void halt_pas16(void)
{
   /*
    *  Shut down the DMA system.
    */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();
}

void cleanup_pas16(void)
{
   /*
    *  Shut down the DMA system.
    */
   DOUT("PAS: Closing the PCM buffering");
   ClosePCMBuffering();

   /*
    *  Restore mixers to their original value.
    */
   DOUT("PAS: Restoring mixer levels");
   cMVSetFilterFunction(FilterVal);
   cMVSetMixerFunction(LeftChannel,BI_OUTPUTMIXER,BI_L_PCM);
   cMVSetMixerFunction(RightChannel,BI_OUTPUTMIXER,BI_R_PCM);
}

void recordblock_pas16(void far *buffer)
{
   /*
    *  Place a buffer on the queue to be recorded,
    *  and start the DMA process if not currently active
    */
   if(QueueThisBlock((char far *)buffer,fftlen*2*(stereo+1),callback1)!=0)
   {
      cleanup_pas16();
      closegraph();
      puts("Error queueing block.");
      exit(1);
   }
}

void set_mixer_pas16(int mix,int level)
{
   /*
    *  Set a mixer level on the PAS16 card
    */
   if(mix==MIXER_EXT)
   {
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_L_EXT);
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_R_EXT);
   }
   else if(mix==MIXER_INT)
   {
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_L_INT);
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_R_INT);
   }
   else if(mix==MIXER_MIC)
   {
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_L_MIC);
      cMVSetMixerFunction(level,BI_INPUTMIXER,BI_R_MIC);
   }
}

#endif
