// track.cc,v 2.2 1995/06/19 19:04:47 andreas Exp

/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "track.h"
#include "util.h"
#include "player.h"
#include "jazz.h"

#include <stdlib.h>

#include <assert.h>

int tParam::Write(tWriteBase &io) {
	return( Msb.Write( io ) + Lsb.Write( io ) + Data.Write( io ) );
}

void tParam::SetCha( uchar cha ) { 
	Msb.Channel = cha;
	Lsb.Channel = cha;
	Data.Channel = cha;
}

uchar sys_moddepth[7] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x20, 0x04 };
const uchar sys_reverbtype[7] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x30 };
const uchar sys_chorustype[7] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x38 };
const uchar sys_mastervol[7] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x04 };
const uchar sys_masterpan[7] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x06 };

uchar *SysExDT1( uchar aa, uchar bb, uchar cc, uchar dd ) {

	 uchar *mess = new uchar[10];
	 mess[0] = 0x41;
	 mess[1] = 0x10;
	 mess[2] = 0x42;
	 mess[3] = 0x12;
	 mess[4] = aa;
   	 mess[5] = bb;
	 mess[6] = cc;
   	 mess[7] = dd;
   	 mess[8] = 0x80 - ((mess[4] + mess[5] + mess[6] + mess[7]) & 0x7f);
	 mess[9] = 0xf7;
	 return( mess );
}

tEventArray::tEventArray()
{
  nEvents = 0;

  MaxEvents = 0;
  Events = 0;
  Channel = 0;
  ForceChannel = 0;

  Clear();
}


void tEventArray::Clear()
{
  for (int i = 0; i < nEvents; i++)
    delete Events[i];
  nEvents = 0;

  Name   = 0;
  Patch  = 0;
  Volume = 0;
  Pan    = 0;
  Reverb = 0;
  Chorus = 0;
  Bank = 0;
  Speed = 0;
  Channel = 1;

  ModDepth = 0;
  ReverbType = 0;
  ChorusType = 0;
  MasterVol = 0;
  MasterPan = 0;

  VibRate = 0;
  VibDepth = 0;
  VibDelay = 0;
  Cutoff = 0;
  Resonance = 0;
  EnvAttack = 0;
  EnvDecay = 0;
  EnvRelease = 0;
  PitchBendSens = 0;

  if (Events)
    delete [] Events;
  Events = 0;
  MaxEvents = 0;
}


tEventArray::~tEventArray()
{
  Clear();
}

void tEventArray::Resize()
{
  long i;
  MaxEvents += 20;
  tEvent **tmp = new tEvent * [MaxEvents];
  for (i = 0; i < nEvents; i++)
    tmp[i] = Events[i];
  for (; i < MaxEvents; i++)
    tmp[i] = 0;
  delete [] Events;
  Events = tmp;

  assert(Events);	// crash!!
}


void tEventArray::Put(tEvent *e)
{
  if (nEvents >= MaxEvents)
    Resize();
  Events[nEvents++] = e;
}


static int compare(const void *p1, const void *p2)
{
  tEvent *e1 = *(tEvent **)p1;
  tEvent *e2 = *(tEvent **)p2;
  return e1->Compare(*e2);
}


void tEventArray::Sort()
{
  qsort(Events, nEvents, sizeof(tEvent *), compare);
}



void tEventArray::Cleanup(tUndoBuffer *UndoBuffer)
{
  tEvent *e;
  tControl *c;
  tSysEx *s;

  Sort();	// Alle Killed-Events sind am Ende des Arrays

  Name = 0;
  Speed = 0;
  Volume = 0;
  Pan = 0;
  Reverb = 0;
  Chorus = 0;

  ModDepth = 0;
  ReverbType = 0;
  ChorusType = 0;
  MasterVol = 0;
  MasterPan = 0;

  for (int i = 0; i < nEvents; i++)
  {
    if ((e = Events[i])->IsKilled())
    {
      if (UndoBuffer)
      {
	for (int j = i; j < nEvents; j++)
	  UndoBuffer->Put(Events[j]);
      }
      else
      {
	for (int j = i; j < nEvents; j++)
	  delete Events[j];
      }
      nEvents = i;
      break;
    }

    if (!Name) 
      Name = e->IsTrackName();
    if (!Speed)
      Speed = e->IsSetTempo();
    if ((c = e->IsControl()) != 0)
    {
      switch (c->Control)
      {
        case 0x07: if (!Volume) Volume = c; break;
        case 0x0a: if (!Pan)    Pan    = c; break;
        case 0x5b: if (!Reverb) Reverb = c; break;
        case 0x5d: if (!Chorus) Chorus = c; break;
      }
    }
    if ((s = e->IsSysEx()) != 0) {
	sys_moddepth[5] = 0x20 | Channel;
	if ( !ModDepth && !memcmp( sys_moddepth, s->Data, 7 ) ) {
		ModDepth = s;
	}
	else if ( !ReverbType && !memcmp( sys_reverbtype, s->Data, 7 ) ) {
		ReverbType = s;
	}
	else if ( !ChorusType && !memcmp( sys_chorustype, s->Data, 7 ) ) {
		ChorusType = s;
	}
	else if ( !MasterVol && !memcmp( sys_mastervol, s->Data, 7 ) ) {
		MasterVol = s;
	}
	else if ( !MasterPan && !memcmp( sys_masterpan, s->Data, 7 ) ) {
		MasterPan = s;
	}
    }
  }
}


void tEventArray::Length2Keyoff()
{
  int n = nEvents;
  for (int i = 0; i < n; i++)
  {
    tKeyOn *on;
    if ((on = Events[i]->IsKeyOn()) != 0 && on->Length != 0)
    {
      tEvent *of = new tKeyOff(on->Clock + on->Length, on->Channel, on->Key);
      on->Length = 0;
      Put(of);
    }
  }
  Sort();
}



void tEventArray::Keyoff2Length()
{
  for (int i = 1; i < nEvents; i++)
  {
    tKeyOff *of;
    if ((of = Events[i]->IsKeyOff()) != 0)
    {
      tEvent **e = &Events[i - 1];
      while (e >= Events)
      {
        tKeyOn *on = (*e)->IsKeyOn();
        if (on && on->Key == of->Key && on->Channel == of->Channel && on->Length == 0)
        {
          on->Length = of->Clock - on->Clock;
          if (on->Length <= 0L)
            on->Length = 1;
          of->Kill();
	  break;
	}
	-- e;
      }
    }
  }
  Cleanup(0);
}

void tEventArray::Write(tWriteBase &io)
{
  tEvent *e;
  int WrittenBefore;

  Length2Keyoff();
  io.NextTrack();

  // Rpn / Nrpn:
  // All these must be written in order (three tControl's in a row)
  if (VibRate) VibRate->Write(io);
  if (VibDepth) VibDepth->Write(io);
  if (VibDelay) VibDelay->Write(io);
  if (Cutoff) Cutoff->Write(io);
  if (Resonance) Resonance->Write(io);
  if (EnvAttack) EnvAttack->Write(io);
  if (EnvDecay) EnvDecay->Write(io);
  if (EnvRelease) EnvRelease->Write(io);
  if (PitchBendSens) PitchBendSens->Write(io);

  // Bank: Must be sure bank is written before program:
  if (Bank) Bank->Write(io);
  if (Patch) Patch->Write(io);

  for (int i = 0; i < nEvents; i++) {
	e = Events[i];
	WrittenBefore = 0;
	if (e->IsControl()) {
		switch (e->IsControl()->Control) {
			// Don't write these again if present as events
			// (should not happen)
			case 0x65: // Rpn Msb
			case 0x64: // Rpn Lsb
			case 0x63: // Nrpn Msb
			case 0x62: // Nrpn Lsb
			case 0x06: // Rpn/Nrpn Data
			case 0x00: // Bank
				WrittenBefore = 1;
				break;
			default:
				WrittenBefore = 0;
		}
	}
	else if (e->IsProgram()) {
		// Should not happen
		WrittenBefore = 1;
	}
	if (!WrittenBefore) {
	    	e->Write(io);
	}
  }
  Keyoff2Length();
}



void tEventArray::Read(tReadBase &io)
{
  tEvent *e;
  Channel = 0;
  uchar Msb, Lsb, Data;
  int SpecialEvent;

  Msb = Lsb = Data = 0xff;
  int cha;

  io.NextTrack();
  while ((e = io.Read()) != 0)
  {

    SpecialEvent = 0;
    if (e->IsControl()) {
	switch (e->IsControl()->Control) {
		// Grab Rpn/Nrpn/Bank from file and save them, don't put
		// them into event-array
		case 0x63:
		case 0x65:
			Msb = e->IsControl()->Value; // Rpn/Nrpn Msb
			SpecialEvent = 1;
			break;
		case 0x62:
		case 0x64:
			Lsb = e->IsControl()->Value; // Rpn/Nrpn Lsb
			SpecialEvent = 1;
			break;
		case 0x06:
			Data = e->IsControl()->Value; // Rpn/Nrpn Data
			SpecialEvent = 1;
			cha = e->IsControl()->Channel;
			switch (Msb) {
				case 0x01: // Nrpn
					switch (Lsb) {
						case 0x08:
							VibRate = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x09:
							VibDepth = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x0a:
							VibDelay = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x20:
							Cutoff = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x21:
							Resonance = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x63:
							EnvAttack = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x64:
							EnvDecay = new tNrpn( cha, Msb, Lsb, Data );
							break;
						case 0x66:
							EnvRelease = new tNrpn( cha, Msb, Lsb, Data );
							break;
						default:
							break;
					}
					break;
				case 0x00: // Rpn
					if (Lsb == 0x00) { // Pitch Bend Sensivity
						PitchBendSens = new tRpn( cha, Msb, Lsb, Data );
					}
					break;
				default:
					break;
			}
  			Msb = Lsb = Data = 0xff;
			break;
		case 0x00:
			SpecialEvent = 1;
			Bank = e->IsControl(); // Bank
			break;
		default:
			SpecialEvent = 0; // Other control
			break;
	}
    }
    else if (e->IsProgram()) {
		Patch = e->IsProgram();
		SpecialEvent = 1;
    }
    if (!SpecialEvent) {
	    Put(e);
	    if (!Channel && e->IsChannelEvent())
	      Channel = e->IsChannelEvent()->Channel + 1;
    }
  } // while read

  if (!Channel)
    Channel = 1;

  Keyoff2Length();
}


long tEventArray::GetLastClock()
{
  if (!nEvents)
    return 0;
  return Events[nEvents-1]->Clock;
}

int tEventArray::IsEmpty() 
{ 
  return nEvents == 0; 
}

int tEventArray::GetFirstClock() 
{ 
  if (nEvents) 
    return Events[0]->Clock; 
  return LastClock; 
}

// ***********************************************************************
// Dialog
// ***********************************************************************

class tTrackDlg : public wxForm
{
  tTrack *trk;
  char *TrackName;
  tNamedChoice PatchChoice;
  long PatchNr;
  long BankNr;
  int ClearTrack;

 public:
  tTrackDlg::tTrackDlg(tTrack *t);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
};


tTrackDlg::tTrackDlg(tTrack *t)
  : PatchChoice("Patch", VoiceNames, &PatchNr)
{
  trk = t;
}


void tTrackDlg::OnCancel()
{
  trk->DialogBox = 0;
  wxForm::OnCancel();
}


void tTrackDlg::OnOk()
{
  trk->DialogBox = 0;
  if (ClearTrack) {
	trk->Clear();
  	delete TrackName;
  	wxForm::OnOk();
	return;
  }
  trk->SetName(TrackName);
  delete TrackName;
  PatchChoice.GetValue();
  BankNr = (PatchNr & 0x0000ff00) >> 8;
  PatchNr = PatchNr & 0x000000ff;
  trk->SetBank(BankNr);
  trk->SetPatch(PatchNr);
  if (trk->ForceChannel) {
	tChannelEvent *c;
	tSysEx *s;
	tEventIterator Iterator(trk);
	trk->Sort();
	tEvent *e = Iterator.Range(0, (long unsigned) trk->GetLastClock() + 1);
	while (e) {
	    	if ((c = e->IsChannelEvent()) != 0) {
		  	c = (tChannelEvent *)e->Copy();
		      	c->Channel = trk->Channel - 1;
			trk->Kill(e);
			trk->Put(c);
		}
		else if ((s = e->IsSysEx()) != 0) {
			// Check for ModDepth sysex (contains channel number)
			if ( 	!memcmp( sys_moddepth, s->Data, 5 ) && 
				(s->Data[6] == 0x04) &&
				(s->Data[5] & 0x20)) {
			    s = (tSysEx *) e->Copy();
			    s->Data[5] = 0x20 | trk->Channel;
			    trk->Kill(e);
			    trk->Put(s);
			}
		}
		e = Iterator.Next();
	} // while e
	if (trk->VibRate) trk->VibRate->SetCha( trk->Channel - 1 );
	if (trk->VibDepth) trk->VibDepth->SetCha( trk->Channel - 1 );
	if (trk->VibDelay) trk->VibDelay->SetCha( trk->Channel - 1 );
	if (trk->Cutoff) trk->Cutoff->SetCha( trk->Channel - 1 );
	if (trk->Resonance) trk->Resonance->SetCha( trk->Channel - 1 );
	if (trk->EnvAttack) trk->EnvAttack->SetCha( trk->Channel - 1 );
	if (trk->EnvDecay) trk->EnvDecay->SetCha( trk->Channel - 1 );
	if (trk->EnvRelease) trk->EnvRelease->SetCha( trk->Channel - 1 );
	if (trk->PitchBendSens) trk->PitchBendSens->SetCha( trk->Channel - 1 );
	if (trk->Bank) trk->Bank->Channel = trk->Channel - 1;
	if (trk->Patch) trk->Patch->Channel = trk->Channel - 1;
	trk->Cleanup();
  }
  wxForm::OnOk();
}



void tTrackDlg::EditForm(wxPanel *panel)
{
  PatchNr   = trk->GetPatch() + (trk->GetBank() << 8);
  TrackName = copystring(trk->GetName());
  Add(wxMakeFormString("Trackname:", &TrackName, wxFORM_DEFAULT )); //, 0,0,0, 100, -1));

  Add(wxMakeFormNewLine());
  Add(PatchChoice.mkFormItem(300, 300));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Channel", &trk->Channel, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(1.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Force channel number onto all events on track", &trk->ForceChannel));
  ClearTrack = 0;
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Clear track (NB! erase all events, name etc...)", &ClearTrack));

  AssociatePanel(panel);
}


void tTrack::Dialog(wxFrame *parent)
{
  tTrackDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(parent, "Track Settings", FALSE);
  dlg = new tTrackDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

// ***********************************************************************
// tUndoBuffer
// ***********************************************************************

tUndoBuffer::tUndoBuffer()
{
}

void tUndoBuffer::Clear()
{
  Killed.Clear();
  Copies.nEvents = 0;
  Copies.Clear();
}

void tUndoBuffer::Put(tEvent *e)
{
  if (e->IsKilled())
    Killed.Put(e);
  else
    Copies.Put(e);
}

void tUndoBuffer::Debug()
{
  if (Killed.nEvents || Copies.nEvents)
  {
    int i;

    printf("tUndoBuffer.Killed:\n");
    for (i = 0; i < Killed.nEvents; i++)
    {
      tEvent *e = Killed.Events[i];
      printf("%p clk %08lx sta %02x\n", e, e->Clock, (unsigned char)e->Stat);
    }

    printf("tUndoBuffer.Copies:\n");
    for (i = 0; i < Copies.nEvents; i++)
    {
      tEvent *e = Copies.Events[i];
      printf("%p clk %08lx sta %02x\n", e, e->Clock, (unsigned char)e->Stat);
    }
    fflush(stdout);
  }
}


// ***********************************************************************
// tTrack
// ***********************************************************************


tTrack::tTrack()
  : tEventArray()
{
  iUndo = 0;
  DialogBox = 0;
  PatchNames = 0;
  State = tsPlay;
  ForceChannel = 1;
}


void tTrack::Merge(tEventArray *t)
{
  for (int i = 0; i < t->nEvents; i++)
    Put(t->Events[i]);
  t->nEvents = 0;
}


void tTrack::MergeRange(tEventArray *other, long FromClock, long ToClock, int Replace)
{
  // Erase destin
  if (Replace)
  {
    tEventIterator Erase(this);
    tEvent *e = Erase.Range(FromClock, ToClock);
    while (e)
    {
      Kill(e);
      e = Erase.Next();
    }
  }

  // Merge Recorded Events
  tEventIterator Copy(other);
  tEvent *e = Copy.Range(FromClock, ToClock);
  while (e)
  {

    #if 0
    {
      tGetMidiBytes mb;
      e->Write(mb);
      printf("ReplaceRange: Clock %ld, Stat %02x, Bytes %d: ", e->Clock, e->Stat, mb.nBytes);
      for (int i = 0; i < mb.nBytes; i++)
	printf("%02x ", mb.Buffer[i]);
      tKeyOn *on = e->IsKeyOn();
      if (on)
	printf("KeyOn, Length = %d", on->Length);
      tKeyOff *of = e->IsKeyOff();
      if (of)
	printf("KeyOFF********");

      putchar('\n');
      fflush(stdout);
    }
    #endif

    Put(e->Copy());
    e = Copy.Next();
  }
  Cleanup();
}


void tTrack::Cleanup()
{
  tEventArray::Cleanup(&UndoBuffers[iUndo]);
}



void tTrack::Undo()
{
  int i;
  tEventArray *Killed, *Copies;
  tUndoBuffer *u;

  u = &UndoBuffers[iUndo];
  Killed = &u->Killed;
  Copies = &u->Copies;

  for (i = 0; i < Killed->nEvents; i++)
  {
    tEvent *e = Killed->Events[i];
    e->UnKill();
    tEventArray::Put(e);
  }
  Killed->nEvents = 0;

  for (i = 0; i < Copies->nEvents; i++)
  {
    tEvent *e = Copies->Events[i];
    e->Kill();
  }
  Copies->nEvents = 0;

  iUndo = (iUndo - 1 + MaxUndo) % MaxUndo;
  tEventArray::Cleanup(0); 
}


void tTrack::NewUndoBuffer()
{
  iUndo = (iUndo + 1) % MaxUndo;
  UndoBuffers[iUndo].Clear();
};


void tTrack::Clear()
{
  tEventArray::Clear();
  for (int i = 0; i < MaxUndo; i++)
    UndoBuffers[i].Clear();
  State  = tsPlay;
}

// ----------------------- Name ------------------------------------

char *tTrack::GetName()
{
  if (Name)
    return (char *)Name->Data;
  return "";
}



void tTrack::SetName(char *str)
{
  if (Name)
    Kill(Name);
  if (strlen(str))
    Put(new tTrackName(0, (uchar *)str, strlen(str)));
  Cleanup();
}

// ------------------------  Volume ------------------------------

int tTrack::GetVolume()
{
  if (Volume)
    return Volume->Value + 1;
  return 0;
}

void tTrack::SetVolume(int Value)
{
  if (Volume)
    Kill(Volume);
  if (Value > 0)
  {
    tEvent *e = new tControl(0, Channel - 1, 0x07, Value - 1);
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Pan ------------------------------

int tTrack::GetPan()
{
  if (Pan)
    return Pan->Value + 1;
  return 0;
}

void tTrack::SetPan(int Value)
{
  if (Pan)
    Kill(Pan);
  if (Value > 0)
  {
    tEvent *e = new tControl(0, Channel - 1, 0x0a, Value - 1);
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Reverb ------------------------------

int tTrack::GetReverb()
{
  if (Reverb)
    return Reverb->Value + 1;
  return 0;
}

void tTrack::SetReverb(int Value)
{
  if (Reverb)
    Kill(Reverb);
  if (Value > 0)
  {
    tEvent *e = new tControl(0, Channel - 1, 0x5B, Value - 1);
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Chorus ------------------------------

int tTrack::GetChorus()
{
  if (Chorus)
    return Chorus->Value + 1;
  return 0;
}

void tTrack::SetChorus(int Value)
{
  if (Chorus)
    Kill(Chorus);
  if (Value > 0)
  {
    tEvent *e = new tControl(0, Channel - 1, 0x5D, Value - 1);
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Bank ------------------------------

int tTrack::GetBank()
{
  if (Bank)
    return Bank->Value;
  return 0;
}

void tTrack::SetBank(int Value)
{
  if (Bank) {
	delete Bank;
	Bank = 0;
  }
  if (Value >= 0)
  {
    Bank = new tControl(0, Channel - 1, 0x00, Value);
    Midi->OutNow(Bank);
  }
}

// ------------------------  Patch ------------------------------

int tTrack::GetPatch()
{
  if (Patch)
    return Patch->Program + 1;
  return 0;
}

void tTrack::SetPatch(int PatchNr)
{
  if (Patch) {
    delete Patch;
    Patch = 0;
  }
  if (PatchNr > 0)
  {
    Patch = new tProgram(0, Channel - 1, PatchNr - 1);
    Midi->OutNow(Patch);
  }
}

// ------------------------  VibRate ------------------------------

int tTrack::GetVibRate()
{
  if (VibRate)
    return VibRate->GetVal() + 1;
  return 0;
}

void tTrack::SetVibRate(int Value)
{
  if (VibRate) {
    delete VibRate;
    VibRate = 0;
  }
  if (Value > 0)
  {
    VibRate = new tNrpn( Channel - 1, 0x01, 0x08, Value - 1 );
    Midi->OutNow( VibRate );
  }
}

// ------------------------  VibDepth ------------------------------

int tTrack::GetVibDepth()
{
  if (VibDepth)
    return VibDepth->GetVal() + 1;
  return 0;
}

void tTrack::SetVibDepth(int Value)
{
  if (VibDepth) {
    delete VibDepth;
    VibDepth = 0;
  }
  if (Value > 0)
  {
	VibDepth = new tNrpn( Channel - 1, 0x01, 0x09, Value - 1 );
	Midi->OutNow( VibDepth );
  }
}

// ------------------------  VibDelay ------------------------------

int tTrack::GetVibDelay()
{
  if (VibDelay)
    return VibDelay->GetVal() + 1;
  return 0;
}

void tTrack::SetVibDelay(int Value)
{
  if (VibDelay) {
    delete VibDelay;
    VibDelay = 0;
  }
  if (Value > 0)
  {
	VibDelay = new tNrpn( Channel - 1, 0x01, 0x0a, Value - 1 );
	Midi->OutNow( VibDelay );
  }
}

// ------------------------  Cutoff ------------------------------

int tTrack::GetCutoff()
{
  if (Cutoff)
    return Cutoff->GetVal() + 1;
  return 0;
}

void tTrack::SetCutoff(int Value)
{
  if (Cutoff) {
    delete Cutoff;
    Cutoff = 0;
  }
  if (Value > 0)
  {
	Cutoff = new tNrpn( Channel - 1, 0x01, 0x20, Value - 1 );
	Midi->OutNow( Cutoff );
  }
}

// ------------------------  Resonance ------------------------------

int tTrack::GetResonance()
{
  if (Resonance)
    return Resonance->GetVal() + 1;
  return 0;
}

void tTrack::SetResonance(int Value)
{
  if (Resonance) {
	delete Resonance;
	Resonance = 0;
  }
  if (Value > 0)
  {
	Resonance = new tNrpn( Channel - 1, 0x01, 0x21, Value - 1 );
	Midi->OutNow( Resonance );
  }
}

// ------------------------  EnvAttack ------------------------------

int tTrack::GetEnvAttack()
{
  if (EnvAttack)
    return EnvAttack->GetVal() + 1;
  return 0;
}

void tTrack::SetEnvAttack(int Value)
{
  if (EnvAttack) {
	delete EnvAttack;
	EnvAttack = 0;
  }
  if (Value > 0)
  {
	EnvAttack = new tNrpn( Channel - 1, 0x01, 0x63, Value - 1 );
	Midi->OutNow( EnvAttack );
  }
}

// ------------------------  EnvDecay ------------------------------

int tTrack::GetEnvDecay()
{
  if (EnvDecay)
    return EnvDecay->GetVal() + 1;
  return 0;
}

void tTrack::SetEnvDecay(int Value)
{
  if (EnvDecay) {
	delete EnvDecay;
	EnvDecay = 0;
  }
  if (Value > 0)
  {
	EnvDecay = new tNrpn( Channel - 1, 0x01, 0x64, Value - 1 );
	Midi->OutNow( EnvDecay );
  }
}

// ------------------------  EnvRelease ------------------------------

int tTrack::GetEnvRelease()
{
  if (EnvRelease)
    return EnvRelease->GetVal() + 1;
  return 0;
}

void tTrack::SetEnvRelease(int Value)
{
  if (EnvRelease) {
	delete EnvRelease;
	EnvRelease = 0;
  }
  if (Value > 0)
  {
	EnvRelease = new tNrpn( Channel - 1, 0x01, 0x66, Value - 1 );
	Midi->OutNow( EnvRelease );
  }
}

// ------------------------  PitchBendSens ------------------------------

int tTrack::GetPitchBendSens()
{
  if (PitchBendSens)
    return PitchBendSens->GetVal() + 1;
  return 0;
}

void tTrack::SetPitchBendSens(int Value)
{
  if (PitchBendSens) {
	delete PitchBendSens;
	PitchBendSens = 0;
  }
  if (Value > 0)
  {
	PitchBendSens = new tRpn( Channel - 1, 0x00, 0x00, Value - 1 );
	Midi->OutNow( PitchBendSens );
  }
}

// ------------------------  Modulation Depth ------------------------------

int tTrack::GetModDepth()
{
  if (ModDepth)
    return ModDepth->Data[7] + 1;
  return 0;
}

void tTrack::SetModDepth(int Value)
{
  if (ModDepth)
    Kill(ModDepth);
  if (Value > 0)
  {
    uchar *mess = SysExDT1( 0x40, 0x20 | Channel, 0x04, Value-1);
    tEvent *e = new tSysEx(0, mess, 10);
    delete mess;
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Reverb Type ------------------------------

int tTrack::GetReverbType()
{
  if (ReverbType)
    return ReverbType->Data[7] + 1;
  return 0;
}

void tTrack::SetReverbType(int Value)
{
  if (ReverbType)
    Kill(ReverbType);
  if (Value > 0)
  {
    uchar *mess = SysExDT1( 0x40, 0x01, 0x30, Value-1);
    tEvent *e = new tSysEx(0, mess, 10);
    delete mess;
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}

// ------------------------  Chorus Type ------------------------------

int tTrack::GetChorusType()
{
  if (ChorusType)
    return ChorusType->Data[7] + 1;
  return 0;
}

void tTrack::SetChorusType(int Value)
{
  if (ChorusType)
    Kill(ChorusType);
  if (Value > 0)
  {
    uchar *mess = SysExDT1( 0x40, 0x01, 0x38, Value-1);
    tEvent *e = new tSysEx(0, mess, 10);
    delete mess;
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}


// ------------------------  Master Volume ------------------------------

int tTrack::GetMasterVol()
{
  if (MasterVol)
    return MasterVol->Data[7] + 1;
  return 0;
}

void tTrack::SetMasterVol(int Value)
{
  if (MasterVol)
    Kill(MasterVol);
  if (Value > 0)
  {
    uchar *mess = SysExDT1( 0x40, 0x00, 0x04, Value-1);
    tEvent *e = new tSysEx(0, mess, 10);
    delete mess;
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}


// ------------------------  Master Pan ------------------------------

int tTrack::GetMasterPan()
{
  if (MasterPan)
    return MasterPan->Data[7] + 1;
  return 0;
}

void tTrack::SetMasterPan(int Value)
{
  if (MasterPan)
    Kill(MasterPan);
  if (Value > 0)
  {
    uchar *mess = SysExDT1( 0x40, 0x00, 0x06, Value-1);
    tEvent *e = new tSysEx(0, mess, 10);
    delete mess;
    Put(e);
    Midi->OutNow(e);
  }
  Cleanup();
}


// ------------------------  Speed ------------------------------

int tTrack::GetSpeed()
{
  if (Speed)
    return Speed->GetBPM();
  return 100;
}


void tTrack::SetSpeed(int bpm)
{
  tEvent *e = new tSetTempo(0, bpm);
  if (Speed)
    Kill(Speed);
  Put(e);
  Midi->OutNow(e);
  Cleanup();
}


// ------------------------- State ----------------------------------


char *tTrack::GetStateChar()
{
  switch (State)
  {
    case tsPlay: return "P";
    case tsMute: return "M";
    case tsSolo: return "S";
  }
  return "?";
}

void tTrack::SetState(int NewState)
{
  State = NewState % 3;
}

void tTrack::ToggleState(int Direction)
{
  State = (State + Direction + 3) % 3;
}

// ------------------------- Channel ---------------------------

void tTrack::SetChannel(int NewChannel)
{
  Channel = NewChannel;
}


