/*
 * Slide.c - slide widget.
 *
 * (c) 1995 by Martin Denn <mdenn@unix-ag.uni-kl.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include <stdio.h>

#include "SlideP.h"

#define MIN(x,y) (x < y ? x : y)
#define MAX(x,y) (x > y ? x : y)

#define MAXVALUE 100

#define offset(field) XtOffsetOf(SlideRec, field)

static XtResource resources[] = {
     {
	XtNslideColor, 
	XtCSlideColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(slide.slide_color), 
	XtRString, 
	XtDefaultBackground
     },
     {
	XtNcommon, 
	XtCCommon,
	XtRBoolean, 
	sizeof(Boolean),
	offset(slide.common), 
	XtRImmediate, 
	(XtPointer) True
     },
     {
	XtNcallback, 
	XtCCallback, 
	XtRCallback, 
	sizeof(XtPointer),
	offset(slide.callback), 
	XtRCallback, 
	NULL
     },
     {
        XtNleftValue,
        XtCLeftValue,
        XtRInt,
        sizeof(int),
        offset(slide.left_value),
        XtRImmediate,
        (XtPointer) 0
     },
     {
        XtNrightValue,
        XtCRightValue,
        XtRInt,
        sizeof(int),
        offset(slide.right_value),
        XtRImmediate,
        (XtPointer) 0
     },
     {
        XtNnumberOfLEDs,
        XtCNumberOfLEDs,
        XtRInt,
        sizeof(int),
        offset(slide.number_of_LEDs),
        XtRImmediate,
        (XtPointer) 16
     },
     {
        XtNpixmap,
        XtCPixmap,
        XtRPixmap,
        sizeof(Pixmap),
        offset(slide.pixmap),
        XtRImmediate,
        (XtPointer) XtUnspecifiedPixmap
     },
     {
	XtNslideBackground, 
	XtCSlideBackground, 
	XtRPixel, 
	sizeof(Pixel),
	offset(slide.slide_background), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNrBExists, 
	XtCRBExists, 
	XtRBoolean, 
	sizeof(Boolean),
	offset(slide.rb_exists), 
	XtRImmediate, 
	(XtPointer) False
     },
     {
	XtNrBOn, 
	XtCRBOn, 
	XtRBoolean, 
	sizeof(Boolean),
	offset(slide.rb_on), 
	XtRImmediate, 
	(XtPointer) False
     },
     {
	XtNrBColor, 
	XtCRBColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(slide.rb_color), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNrBBackColor, 
	XtCRBBackColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(slide.rb_back_color), 
	XtRString, 
	XtDefaultBackground
     },
     {
	XtNrBCallback, 
	XtCRBCallback, 
	XtRCallback, 
	sizeof(XtPointer),
	offset(slide.rb_callback), 
	XtRCallback, 
	NULL
     },
};

/* Declaration of methods */

static void Initialize();
static void Redisplay();
static void Destroy();
static void Resize();
static Boolean SetValues();

/* the following are private functions unique to Slide */
static void UpdateSlide(), UpdateRadioButton();

/* the following are actions of Slide */
static void MoveSlide(); 

static char defaultTranslations[] =
	"<Btn1Down>:   MoveSlide()              \n\
	<Btn1Motion>:  MoveSlide()              \n\
	<Btn3Down>:    MoveSlide()              \n\
	<Btn3Motion>:  MoveSlide()";

static XtActionsRec actions[] = {
        {"MoveSlide", MoveSlide},
};

/* definition in Slide.h */
static SlideInfo info;

/* only use internally */
static int SlideHeight;

SlideClassRec slideClassRec = {
    {
    /* core_class fields */
    /* superclass	  	 */ (WidgetClass) &coreClassRec,
    /* class_name	  	 */ "Slide",
    /* widget_size	  	 */ sizeof(SlideRec),
    /* class_initialize   	 */ NULL,
    /* class_part_initialize	 */ NULL,
    /* class_inited       	 */ FALSE,
    /* initialize	  	 */ Initialize,
    /* initialize_hook		 */ NULL,
    /* realize		  	 */ XtInheritRealize,
    /* actions		  	 */ actions,
    /* num_actions	  	 */ XtNumber(actions),
    /* resources	  	 */ resources,
    /* num_resources	  	 */ XtNumber(resources),
    /* xrm_class	  	 */ NULLQUARK,
    /* compress_motion	  	 */ TRUE,
    /* compress_exposure  	 */ XtExposeCompressMultiple,
    /* compress_enterleave	 */ TRUE,
    /* visible_interest	  	 */ FALSE,
    /* destroy		  	 */ Destroy,
    /* resize		  	 */ Resize,
    /* expose		  	 */ Redisplay,
    /* set_values	  	 */ SetValues,
    /* set_values_hook		 */ NULL,
    /* set_values_almost	 */ XtInheritSetValuesAlmost,
    /* get_values_hook		 */ NULL,
    /* accept_focus	 	 */ NULL,
    /* version			 */ XtVersion,
    /* callback_private   	 */ NULL,
    /* tm_table		   	 */ defaultTranslations,
    /* query_geometry		 */ NULL, 
    /* display_accelerator       */ XtInheritDisplayAccelerator,
    /* extension                 */ NULL
    },
    {
    /* dummy_field               */ 0,
    },
};

WidgetClass slideWidgetClass = (WidgetClass) & slideClassRec;


/* ARGSUSED */
static void
Initialize(Widget treq, Widget tnew, ArgList args, Cardinal *num_args)
{
    SlideWidget w = (SlideWidget) tnew;
    XtGCMask value, dynamic, unused;
    Window dummywin;
    int dummyint, pixw, pixh;

    
    /* 
     *  Check instance values set by resources that may be invalid. 
     */

    if (w->slide.number_of_LEDs <= 0)
    {
        w->slide.number_of_LEDs = 100;
        XtWarning("Slide: numberOfLEDs must be greater than zero.\n");
    }
    if (w->slide.left_value > MAXVALUE)
    {
        w->slide.left_value = MAXVALUE;
        XtWarning("Slide: leftValue must not exceed 100.\n");
    }
    if (w->slide.left_value < 0)
    {
        w->slide.left_value = 0;
        XtWarning("Slide: leftValue must not fall below zero.\n");
    }
    if (w->slide.right_value > MAXVALUE)
    {
        w->slide.right_value = MAXVALUE;
        XtWarning("Slide: rightValue must not exceed maxValue.\n");
    }
    if (w->slide.left_value < 0)
    {
        w->slide.right_value = 0;
        XtWarning("Slide: rightValue must not fall below zero.\n");
    }

    /* initialize height of the slide */ 

    if (w->slide.pixmap == XtUnspecifiedPixmap) SlideHeight = w->core.height;
    else
    {
        XGetGeometry(XtDisplay(w), w->slide.pixmap, &dummywin, &dummyint,
                     &dummyint, &pixw, &pixh, &dummyint, &dummyint);
        SlideHeight = w->core.height - pixh - (w->core.width/2);
    }


    /*
     *  Get GC (and set overall values)
     */
    value = (XtGCMask) 0;
    dynamic = ( GCForeground );
    unused = (XtGCMask) 0;
    w->slide.gc = XtAllocateGC((Widget) w, 0, value, NULL, dynamic, unused);
}

/* ARGSUSED */
static void Redisplay(Widget cw, XExposeEvent *event)
{
    SlideWidget w = (SlideWidget) cw;
    Window dummywin;
    double spaceh, blockh;
    int i,x,z, pixw, pixh, dummyint;

    if (w->slide.pixmap == XtUnspecifiedPixmap) SlideHeight = w->core.height;
    else
    {
        XGetGeometry(XtDisplay(w), w->slide.pixmap, &dummywin, &dummyint,
                     &dummyint, &pixw, &pixh, &dummyint, &dummyint);
        SlideHeight = w->core.height - pixh - (w->core.width*2/3);
        XCopyArea(XtDisplay(w), w->slide.pixmap, XtWindow(w), w->slide.gc, 
                  0, 0, pixw, pixh,
                  (w->core.width-pixw)/2, /* horiz. in the middle */
                  w->core.height-pixh);   /* vert. on the bottom */
        if (w->slide.rb_exists) 
        {
            XSetForeground(XtDisplay(w), w->slide.gc, w->slide.rb_back_color);
            XFillArc(XtDisplay(w), XtWindow(w), w->slide.gc, 
                    (w->core.width)/4+1, SlideHeight+1, 
                    (w->core.width-2)/2, (w->core.width-2)/2 , 0, 360*64);
            if (w->slide.rb_on)
            {
                XSetForeground(XtDisplay(w), w->slide.gc, w->slide.rb_color);
                XFillArc(XtDisplay(w), XtWindow(w), w->slide.gc, 
                        (w->core.width)/4+2, SlideHeight+2,
                        (w->core.width-4)/2, (w->core.width-4)/2, 0, 360*64);
            }
        }
    }
    

    spaceh = SlideHeight/75.0;
    blockh =  (SlideHeight-(w->slide.number_of_LEDs+1)*spaceh)
                /w->slide.number_of_LEDs;
    x = (int) w->core.width/12;
    z = (int) (w->core.width/2) - 2*(w->core.width/12);
       
    XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_background);
    XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, 
                       0, 0, w->core.width, SlideHeight);
    XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_color);
    for (i=0; i< w->slide.number_of_LEDs; i++)
    {   
        if (w->slide.left_value*w->slide.number_of_LEDs/MAXVALUE > i)
        {
            XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, x,
                (int) SlideHeight-4*spaceh-i*(blockh+spaceh),
                z, (int) blockh);
        }
        if (w->slide.right_value*w->slide.number_of_LEDs/MAXVALUE > i)
        {
            XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, w->core.width/2+x,
                (int) SlideHeight-4*spaceh-i*(blockh+spaceh),
                z, (int) blockh);
        }
    }


}

/* ARGSUSED */
static Boolean SetValues(Widget current, Widget request, Widget new, 
                                     ArgList args, Cardinal *num_args)
{
    SlideWidget w = (SlideWidget) new;
    SlideWidget wo= (SlideWidget) current;
    Boolean needs_redraw = False;
    int newvalue1, newvalue2;
    
    if (w->slide.number_of_LEDs != wo->slide.number_of_LEDs)
    {
        w->slide.number_of_LEDs = wo->slide.number_of_LEDs;
        XtWarning("Slide: numberOfLEDs cannot be set by XtSetValues.\n");
    }
    if (w->slide.left_value != wo->slide.left_value)
    {
        if (w->slide.left_value > MAXVALUE)
        {
            w->slide.left_value = MAXVALUE;
            XtWarning("Slide: leftValue must not exceed 100.\n");
        }
        if (w->slide.left_value < 0)
        {
            w->slide.left_value = 0;
            XtWarning("Slide: leftValue must not fall below zero.\n");
        }
    }
    if (w->slide.right_value != wo->slide.right_value)
    {  
        if (w->slide.right_value > MAXVALUE)
        {
            w->slide.right_value = MAXVALUE;
            XtWarning("Slide: rightValue must not exceed 100.\n");
        }
        if (w->slide.left_value < 0)
        {
            w->slide.right_value = 0;
            XtWarning("Slide: rightValue must not fall below zero.\n");
        }
    }
    if ((w->slide.left_value != wo->slide.left_value)
       || (w->slide.right_value != wo->slide.right_value))
    {
        if XtIsRealized((Widget) w)
        {
            newvalue1=w->slide.left_value;
            w->slide.left_value = wo->slide.left_value;
            newvalue2=w->slide.right_value;
            w->slide.right_value = wo->slide.right_value;
            UpdateSlide((Widget) w, newvalue1, newvalue2);
        }
    }
    if (w->slide.rb_on != wo->slide.rb_on)
    {
        if XtIsRealized((Widget) w)
        {
            UpdateRadioButton((Widget) w);
        }
    }
    if (( w->slide.slide_color != wo->slide.slide_color )
        || ( w->slide.slide_background != wo->slide.slide_background )
        || ( w->slide.rb_color != wo->slide.rb_color )
        || ( w->slide.rb_back_color !=wo->slide.rb_back_color ))
        needs_redraw = True;

    return needs_redraw;
}

/* ARGSUSED */
static void Destroy(Widget cw)
{
    /* SlideWidget w = (SlideWidget) cw; */

    /* nothing allocated - nothing to be done */
}

/* ARGSUSED */
static void Resize(Widget cw)
{
    /* SlideWidget w = (SlideWidget) cw; */
    
    /* nothing to be done */
}

static void UpdateRadioButton(Widget cw)
{
    SlideWidget w = (SlideWidget) cw;

    if (w->slide.rb_exists) 
    {
        if (!w->slide.rb_on)
        {
            XSetForeground(XtDisplay(w), w->slide.gc, w->slide.rb_back_color);
            XFillArc(XtDisplay(w), XtWindow(w), w->slide.gc, 
                    (w->core.width)/4+1, SlideHeight+1, 
                    (w->core.width-2)/2, (w->core.width-2)/2 , 0, 360*64);
        }
        if (w->slide.rb_on)
        {
            XSetForeground(XtDisplay(w), w->slide.gc, w->slide.rb_color);
            XFillArc(XtDisplay(w), XtWindow(w), w->slide.gc, 
                    (w->core.width)/4+2, SlideHeight+2,
                    (w->core.width-4)/2, (w->core.width-4)/2, 0, 360*64);
            /* Call Callback */
            XtCallCallbacks((Widget) w, XtNrBCallback, NULL);
        }
    }
}

static void UpdateSlide(Widget cw, int value1, int value2)
{
    SlideWidget w = (SlideWidget) cw;
    double spaceh, blockh;
    int j, x, z, oldposl, oldposr, newposl, newposr;
    
    spaceh =  SlideHeight/75.0;
    blockh =  (SlideHeight-(w->slide.number_of_LEDs+1)*spaceh)
                /w->slide.number_of_LEDs;
                    
    x = (int) w->core.width/12;
    z = (int) (w->core.width/2) - 2*(w->core.width/12);
    oldposl = w->slide.left_value*w->slide.number_of_LEDs/MAXVALUE;
    oldposr = w->slide.right_value*w->slide.number_of_LEDs/MAXVALUE;
    w->slide.left_value = value1;
    w->slide.right_value = value2;
    newposl = w->slide.left_value*w->slide.number_of_LEDs/MAXVALUE;
    newposr = w->slide.right_value*w->slide.number_of_LEDs/MAXVALUE;
    if (oldposl < newposl)
    {
        XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_color);
        for (j=oldposl; j<newposl; j++)
        {
           XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, x,
                (int) SlideHeight-4*spaceh-j*(blockh+spaceh),
                z, (int) blockh);
        }
    }
    if (oldposr < newposr) 
    {
        XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_color);
        for (j=oldposr; j<newposr; j++)
        {
            XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, 
                w->core.width/2+x,
                (int) SlideHeight-4*spaceh-j*(blockh+spaceh),
                z, (int) blockh);
        }
    }
    if (oldposl > newposl)
    {
        XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_background);
        for (j=oldposl; j>=newposl; j--)
        {
            XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, x,
                (int) SlideHeight-4*spaceh-j*(blockh+spaceh),
                z, (int) blockh);
        }
    }
    if (oldposr > newposr)
    {
        XSetForeground(XtDisplay(w), w->slide.gc, w->slide.slide_background);
        for (j=oldposr; j>=newposr; j--)
        {
            XFillRectangle(XtDisplay(w), XtWindow(w), w->slide.gc, 
                w->core.width/2+x,
                (int) SlideHeight-4*spaceh-j*(blockh+spaceh),
                z, (int) blockh);
        }
    }
    /* now call the callback functions */
    info.lvalue = value1;
    info.rvalue = value2;
    if ((oldposl != newposl) || (oldposr != newposr))
        XtCallCallbacks( (Widget) w, XtNcallback, &info);  

}

static void MoveSlide(Widget cw, XEvent *event)
{
    SlideWidget w = (SlideWidget) cw;
    int newvalue,xm,ym;
    static int left, too_deep, both;

    if (event->type==ButtonPress)
    {
        xm=event->xbutton.x;
        ym=event->xbutton.y;
        too_deep=(ym>SlideHeight);
        both=(event->xbutton.button==1);
        left=(xm<(int) (w->core.width/2));
    }
    else
    {
        xm=event->xmotion.x;
        ym=event->xmotion.y;
    }
    if (!too_deep)
    {
        newvalue = (int) MAXVALUE-(MAXVALUE*ym/SlideHeight);
        if (newvalue > MAXVALUE) newvalue = MAXVALUE;
        if (newvalue < 0) newvalue = 0;

        if ((both) || (w->slide.common))
            UpdateSlide((Widget) w, newvalue, newvalue);
        else if (left) UpdateSlide((Widget) w, newvalue, w->slide.right_value);
        else UpdateSlide((Widget) w, w->slide.left_value, newvalue);
    }
    else if (!w->slide.rb_on)
    {
        w->slide.rb_on = True;
        UpdateRadioButton((Widget) w);
    }
}
