/*
 * ---------------------------------------------------------------------------
 * sfxserver/sample.c
 *
 * Copyright by Terry Evans 1994
 * tevans@cs.utah.edu, tevans@slc.unisys.com
 * ---------------------------------------------------------------------------
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ---------------------------------------------------------------------------
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>


#include "global.h"
#include "types.h"
#include "error.h"
#include "io.h"
#include "sample.h"


/* Current sample number */
static unsigned int sample_number;


void S_Init(sample_t **sample)
{
  int loop;

  IO_WriteStdout("        Setting up sample data structures\n");

  for(loop = 0; loop < MAX_SAMPLES; loop++)
    sample[loop] = NULL;

  /* Initialize the sample number */
  sample_number = 0;
}


void S_DeInit(sample_t **sample)
{
  int loop;

  for(loop = 0; loop < sample_number; loop++)
  {
    if(sample[loop] != NULL)
    {
      /* Free that data for the sound */
      if(sample[loop]->data != NULL)
	free(sample[loop]->data);

      /* Free the sound structure */
      free(sample[loop]);
      sample[loop] = NULL;
    }
  }

  sample_number = 0;
}


int S_LoadRawSample(const char *file_name, sample_t **sample)
{
  FILE *fp;

  /* Open the file for the sound */
  fp = fopen(file_name, "r");
   
  if(fp == NULL)
  {
    E_ErrnoNonFatalError("S_LoadRawSample", "Could not load sound file %s", file_name);
    return(-1);
  }
   
  /* See if we need to allocate for the sample number */
  if(sample[sample_number] == NULL)
  {
    if((sample[sample_number] = (sample_t *)malloc(sizeof(sample_t))) == NULL)
    {
      E_NonFatalError("S_LoadRawSample", "Could not malloc %ul bytes", sizeof(sample_t));
      return(-1);
    }
  }

  else
  {
    E_NonFatalError("S_LoadRawSample", "Sample slot %s already in use", sample_number);
    return(-1);
  }

  sample[sample_number]->data = NULL;
  sample[sample_number]->length  = 0;
   
  /* Get the length of the file */
  sample[sample_number]->length = lseek(fileno(fp), 0, SEEK_END);

  /* go back to beginning of file */
  lseek(fileno(fp), 0, SEEK_SET);

  /* alloc memory for sample */
  sample[sample_number]->data = 
    (unsigned char *)malloc(sample[sample_number]->length);
   
  if(sample[sample_number]->data == NULL)
  {
    fclose(fp);
    free(sample[sample_number]);
    E_NonFatalError("S_LoadRawSample", "No sound data found in %s", file_name);
    return(-1);
  }
   
  fread(sample[sample_number]->data, 1, sample[sample_number]->length, fp);
  fclose(fp);

  return(sample_number++);
}


int S_VerifySample(int number)
{
  if(number < sample_number)
    return(TRUE);
  else
    return(FALSE);
}
      
