/* rplay.c - A sample RPLAY and RPTP client.  */

/* 
 * Copyright (C) 1993-95 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <sys/signal.h>
#include <sys/param.h>
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <unistd.h>
#include "rplay.h"
#include "getopt.h"

#define OPTIONS "+spcN:P:h:rv:n:R:fb:i:"

static struct option longopts[] =
{
    {"continue", no_argument, NULL, 'c'},
    {"count", required_argument, NULL, 'n'},
    {"flow", no_argument, NULL, 'f'},
    {"help", no_argument, NULL, 3},
    {"host", required_argument, NULL, 'h'},
    {"info", required_argument, NULL, 'i'},
    {"info-amd", no_argument, NULL, 8},
    {"info-dbri", no_argument, NULL, 9},
    {"info-cs4231", no_argument, NULL, 10},
    {"buffer-size", required_argument, NULL, 'b'},
    {"list-count", required_argument, NULL, 'N'},
    {"list-name", required_argument, NULL, 7},
    {"no-flow", no_argument, NULL, 11},
    {"pause", no_argument, NULL, 'p'},
    {"port", required_argument, NULL, 4},
    {"priority", required_argument, NULL, 'P'},
    {"random", no_argument, NULL, 'r'},
    {"reset", no_argument, NULL, 5},
    {"sample-rate", required_argument, NULL, 'R'},
    {"stop", no_argument, NULL, 's'},
    {"version", no_argument, NULL, 2},
    {"volume", required_argument, NULL, 'v'},
    {NULL, 0, NULL, 0}
};

static void usage (), flow ();
static void flow_interrupt ();

static int rplay_fd = -1;
static RPLAY *rp;
static int default_buffer_size = 8192;

#ifdef __STDC__
main (int argc, char **argv)
#else
main (argc, argv)
    int argc;
    char **argv;
#endif
{
    int c, command, volume, val;
    int list_count, count, priority, do_random, use_flows = 0;
    unsigned long sample_rate;
    char *hosts, *p, *q, *name, *list_name;
    char *sound_info = "";
    extern char *optarg;
    extern int optind, opterr;
    int optind_val = optind;
    int port;
    int no_flows = 0;
    char buf[RPTP_MAX_LINE];

    if (argc < 2)
    {
	usage ();
    }

    signal (SIGINT, flow_interrupt);
    
    command = RPLAY_PLAY;
    list_count = RPLAY_DEFAULT_LIST_COUNT;
    list_name = NULL;
    priority = RPLAY_DEFAULT_PRIORITY;
    do_random = 0;
    hosts = rplay_default_host ();
    port = -1;

    name = NULL;
    sample_rate = RPLAY_DEFAULT_SAMPLE_RATE;
    volume = RPLAY_DEFAULT_VOLUME;
    count = RPLAY_DEFAULT_COUNT;

    /*
     * Scan the args to see what the command is.
     */
    opterr = 0; /* disable getopt errors */
    while ((c = getopt_long (argc, argv, OPTIONS, longopts, 0)) != -1)
    {
	switch (c)
	{
	case 2:		/* --version */
	    printf ("rplay %s\n", RPLAY_VERSION);
	    exit (0);

	case 3:		/* --help */
	    usage ();

	case 5:		/* --reset */
	    if (command != RPLAY_PLAY)
	    {
		usage ();
	    }
	    command = RPLAY_RESET;
	    break;

	case 's':
	    if (command != RPLAY_PLAY)
	    {
		usage ();
	    }
	    command = RPLAY_STOP;
	    break;

	case 'p':
	    if (command != RPLAY_PLAY)
	    {
		usage ();
	    }
	    command = RPLAY_PAUSE;
	    break;

	case 'c':
	    if (command != RPLAY_PLAY)
	    {
		usage ();
	    }
	    command = RPLAY_CONTINUE;
	    break;
	}
    }
    opterr = 1; /* enable getopt errors */

#ifdef 0
    printf ("command=%s\n", command == RPLAY_PLAY ? "play" : command == RPLAY_STOP ? "stop" :
	command == RPLAY_PAUSE ? "pause" : command == RPLAY_CONTINUE ? "continue" : "???");
#endif

    optind = optind_val;	/* reset optind */

    rp = rplay_create (command);
    if (rp == NULL)
    {
	rplay_perror ("rplay_create");
	exit (1);
    }

    while (argc > 1)
    {
	while ((c = getopt_long (argc, argv, OPTIONS, longopts, 0)) != -1)
	{
	    switch (c)
	    {
	    case 0:
		/* getopt has processed a long-named option -- do nothing */
		break;

	    case 1:
		break;
		
	    case 2:		/* --version */
		printf ("rplay %s\n", RPLAY_VERSION);
		exit (0);

	    case 3:		/* --help */
		usage ();

	    case 4:		/* --port */
		port = atoi (optarg);
		break;

	    case 5:		/* --reset */
	    case 's':
	    case 'p':
	    case 'c':
		break;

	    case 7:		/* --list-name */
		list_name = optarg;
		break;

	    case 8:		/* --info-amd */
		/* Sun's amd audio device. */
		sound_info = "8000,ulaw,8,1,big-endian,0";
		break;

	    case 9:		/* --info-dbri */
		/* Sun's dbri audio device. */
		sound_info = "11025,linear16,16,1,big-endian,0";
		break;

	    case 10:		/* --info-cs4231 */
		/* Sun's CS4231 audio device. */
		sound_info = "11025,linear16,16,1,big-endian,0";
		break;

	    case 11:		/* --no-flow */
		no_flows++;
		break;
		
	    case 'N':
		list_count = atoi (optarg);
		break;

	    case 'P':
		priority = atoi (optarg);
		break;

	    case 'r':
		do_random++;
		break;

	    case 'h':
		hosts = optarg;
		break;

	    case 'v':
		volume = atoi (optarg);
		break;

	    case 'n':
		count = atoi (optarg);
		break;

	    case 'R':
		sample_rate = atoi (optarg);
		break;

	    case 'f':
		use_flows++;
		break;

	    case 'i':
		sound_info = optarg;
		break;

	    case 'b':
		default_buffer_size = atoi (optarg);
		break;
		
	    default:
		fprintf (stderr, "Try `rplay --help' for more information.\n");
		exit (1);
	    }
	}

	if (argc == optind)
	{
	    if (command == RPLAY_PLAY)
	    {
		usage ();
	    }
	    name = "#0";
	}
	else if ((*(argv[optind]) != '/') && (strchr (argv[optind], '/')))
	{
	    char pathname[MAXPATHLEN];

	    if (getcwd (pathname, sizeof (pathname)) != NULL)
	    {
		name = (char *) malloc (strlen (pathname) + strlen (argv[optind]) + 2);
		strcpy (name, pathname);
		strcat (name, "/");
		strcat (name, argv[optind]);
	    }
	    else
	    {
		name = argv[optind];
	    }
	}
	else
	{
	    name = argv[optind];
	}

	argv += optind;
	argc -= optind;
	optind = optind_val;

#if 0
	printf ("sound=%s, volume=%d, count=%d, rate=%d\n",
	    name, volume, count, sample_rate);
	printf ("optind = %d, argc = %d\n", optind, argc);
#endif

	if (rplay_set (rp, RPLAY_LIST_COUNT, list_count, NULL) < 0)
	{
	    rplay_perror ("rplay_set");
	    exit (1);
	}

	if (rplay_set (rp, RPLAY_PRIORITY, priority, NULL) < 0)
	{
	    rplay_perror ("rplay_set");
	    exit (1);
	}

	if (list_name && rplay_set (rp, RPLAY_LIST_NAME, list_name, NULL) < 0)
	{
	    rplay_perror ("rplay_set");
	    exit (1);
	}
	
	val = rplay_set (rp, RPLAY_APPEND,
			 RPLAY_SOUND, name,
			 RPLAY_VOLUME, volume,
			 RPLAY_COUNT, count,
			 RPLAY_SAMPLE_RATE, sample_rate,
			 RPLAY_CLIENT_DATA, sound_info,
			 NULL);
	if (val < 0)
	{
	    rplay_perror ("rplay_set");
	    exit (1);
	}

	if (!no_flows
	    && (strcmp (name, "-") == 0 || access (name, R_OK) == 0))
	{
	    use_flows++;
	}
    }

    /* Pick the random sound. */
    if (do_random)
    {
	val = rplay_set (rp, RPLAY_RANDOM_SOUND, NULL);
	if (val < 0)
	{
	    rplay_perror ("rplay_set");
	    exit (1);
	}
    }

    /* Set the correct port if `--port' wasn't specfied. */
    if (port == -1)
    {
	port = (!no_flows && use_flows) ? RPTP_PORT : RPLAY_PORT;
    }

    q = hosts;
    do
    {
	p = q;
	q = strchr (p, ':');
	if (q != NULL)
	{
	    *q++ = '\0';
	}
	
	if (!no_flows && use_flows)
	{
	    rplay_fd = rptp_open (p, port, buf, sizeof (buf));
	    if (rplay_fd < 0)
	    {
		rptp_perror (p);
		exit (1);
	    }
	    flow (rplay_fd, rp);
	    rptp_close (rplay_fd);
	}
	else
	{
	    rplay_fd = rplay_open_port (p, port);
	    if (rplay_fd < 0)
	    {
		rplay_perror (p);
		exit (1);
	    }
	    if (rplay (rplay_fd, rp) < 0)
	    {
		rplay_perror (p);
		exit (1);
	    }
	    rplay_close (rplay_fd);
	}
    }
    while (q != NULL);

    exit (0);
}

static int flow_fd = -1, flow_id, flow_size, flow_nleft;
static int flow_buffer_nread, flow_interrupted, flow_index;
static char *flow_buffer;
static int flow_buffer_size;

static int flow_play ();
static void flow_put (), flow_done (), flow_next (), flow_stop ();

static int
flow_play ()
{
    int fd;
    struct stat st;
    char *client_data, *sound_name, *sound_input;
    char *sound_info, info_buf[1024];
    
    flow_id = 0;
    flow_buffer_nread = 0;
    flow_interrupted = 0;

    if (flow_buffer)
    {
	free ((char *) flow_buffer);
    }
    flow_buffer = NULL;
    flow_buffer_size = default_buffer_size;
    
    if (strcmp ((char *) rplay_get (rp, RPLAY_SOUND, flow_index), "-") == 0)
    {
	flow_fd = 0;
	flow_size = -1;
	flow_nleft = -1;
	client_data = "flow";
	sound_name = "stdin";
	sound_input = "flow";
    }
    else
    {
	sound_name = (char *) rplay_get (rp, RPLAY_SOUND, flow_index);

	if (access (sound_name, R_OK) == 0)
	{
	    fd = open (sound_name, O_RDONLY, 0);
	    if (fd < 0)
	    {
		fprintf (stderr, "rplay: can't open sound `%s'\n",
			 sound_name);
		return -1;
	    }

	    if (fstat (fd, &st) < 0)
	    {
		fprintf (stderr, "rplay: can't fstat sound `%s'\n",
			 sound_name);
		close (fd);
		return -1;
	    }

	    flow_fd = fd;
	    flow_size = (int) st.st_size;
	    flow_nleft = flow_size;
	    client_data = "flow";
	    sound_input = "flow";
	}
	else
	{
	    flow_fd = -1;
	    flow_size = 0;
	    flow_nleft = 0;
	    client_data = "play";
	    sound_input = "file";
	}
    }

    info_buf[0] = '\0';
    sound_info = (char *) rplay_get (rp, RPLAY_CLIENT_DATA, flow_index);
    if (sound_info && *sound_info)
    {
	char buf[1024], *p;

	/* 8000,ulaw,8,1,big-endian,offset */
	strcpy (buf, sound_info);
	p = strtok (buf, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-sample-rate=%s ", p);
	p = strtok (NULL, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-format=%s ", p);
	p = strtok (NULL, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-bits=%s ", p);
	p = strtok (NULL, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-channels=%s ", p);
	p = strtok (NULL, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-byte-order=%s ", p);
	p = strtok (NULL, ", ");
	if (p) sprintf (info_buf + strlen(info_buf), "input-offset=%s ", p);
    }
    
    /* Send the `play' command to grab a spool entry. */
    rptp_async_putline (rplay_fd, NULL, "\
play input=%s %s client-data=%s priority=%d sample-rate=%d volume=%d sound=\"%s\"",
			sound_input,
			info_buf,
			client_data,
			(int) rplay_get (rp, RPLAY_PRIORITY, flow_index),
			(int) rplay_get (rp, RPLAY_SAMPLE_RATE, flow_index),
			(int) rplay_get (rp, RPLAY_VOLUME, flow_index),
			sound_name);

    flow_buffer = (char *) malloc (flow_buffer_size);
    if (!flow_buffer)
    {
	fprintf (stderr, "rplay: out of memory.\n");
	exit (1);
    }

    return 0;
}

static void
flow_put ()
{
    int n, nread;

    if (flow_interrupted)
    {
	flow_interrupted = 0;
	flow_stop ();
	return;
    }
    
    flow_buffer_nread = 0;
    
    if (flow_size == -1)
    {
	n = flow_buffer_size;
    }
    else
    {
	n = MIN (flow_buffer_size, flow_nleft);
    }
    
    nread = read (flow_fd, flow_buffer, n);
    if (nread <= 0)
    {
	flow_done ();
    }
    else
    {
	rptp_async_putline (rplay_fd, NULL, "put client-data=put id=#%d size=%d", flow_id, nread);
	flow_buffer_nread = nread;
	flow_nleft -= nread;
    }
}

static void
flow_stop ()
{
    if (flow_id > 0)
    {
	rptp_async_putline (rplay_fd, NULL, "stop client-data=stop id=#%d",
			    flow_id);
    }
    flow_done ();
}

static void
flow_done ()
{
    if (flow_id > 0)
    {
	rptp_async_putline (rplay_fd, NULL, "done client-data=done id=#%d",
			    flow_id);
    }
    
    close (flow_fd);
    flow_fd = -1;
}

static void
flow_next ()
{
    flow_index++;
    if (flow_index >= (int) rplay_get (rp, RPLAY_NSOUNDS))
    {
	exit (0);
    }
    else
    {
	flow_play ();
    }
}

static void
flow_interrupt ()
{
    /* Not flowing a sound. */
    if (flow_id <= 0)
    {
	exit (0);
    }
    /* A sound is being flowed. */
    else if (flow_nleft == -1 || flow_nleft > 0)
    {
	flow_interrupted++;	/* flow_put will deal with this. */
    }
    /* A sound was being flowed. */
    else if (flow_size == -1 || flow_size > 0)
    {
	flow_stop ();
    }
    else
    {
	flow_done ();
    }
}

#ifdef _STDC__
static void
callback (int fd, int event, char *string)
#else
static void
callback (fd, event, string)
    int fd;
    int event;
    char *string;
#endif    
{
    char *id, *data, *error;

    id = rptp_parse (string, "id");
    data = rptp_parse (0, "client-data");
    
    switch (event)
    {
    case RPTP_EVENT_OK:
	if (data && strcmp (data, "flow") == 0)
	{
	    /* Save the spool id and start flowing the sound. */
	    flow_id = atoi (id+1);
	    flow_put ();
	}
	else if (data && strcmp (data, "play") == 0)
	{
	    flow_id = atoi (id+1);
	}
	else if (data && strcmp (data, "put") == 0)
	{
	    rptp_async_write (rplay_fd, flow_put, flow_buffer, flow_buffer_nread);
	}
	break;

    case RPTP_EVENT_ERROR:
	error = rptp_parse (0, "error");
	if (error && *error)
	{
	    fprintf (stderr, "rplay: %s\n", error);
	}
	
	if (data && (strcmp (data, "play") == 0
		     || strcmp (data, "put") == 0))
	{
	    flow_done ();
	    flow_next ();
	}
	else
	{
	    flow_next ();
	}
	break;

    case RPTP_EVENT_DONE:
	if (flow_id && atoi (id+1) == flow_id)
	{
	    flow_next ();
	}
	break;
	
    case RPTP_EVENT_CLOSE:
	fprintf (stderr, "rplay: server connection closed.\n");
	exit (0);

    default:
	break;
    }
}

#ifdef __STDC__
static void
flow (int rplay_fd, RPLAY *rp)
#else
static void
flow (rplay_fd, rp)
    int rplay_fd;
    RPLAY *rp;
#endif	
{
    rptp_async_notify (rplay_fd,
		       RPTP_EVENT_OK | RPTP_EVENT_ERROR | RPTP_EVENT_DONE | RPTP_EVENT_CLOSE,
		       callback);
    flow_index = 0;
    flow_play ();
    rptp_main_loop ();
}

static void
usage ()
{
    printf ("\nrplay %s\n\n", RPLAY_VERSION);
    printf ("usage: rplay [options] [sound ... ]\n\n");

    printf ("-b BYTES, --buffer-size=BYTES\n");
    printf ("\tUse of a buffer size of BYTES when playing sounds using RPTP flows.\n");
    printf ("\tThe default is 8K.\n");
    printf ("\n");
    
    printf ("-c, --continue\n");
    printf ("\tContinue sounds.\n");
    printf ("\n");

    printf ("-f, --flow\n");
    printf ("\tUse RPTP flows to play sounds.  This will force sounds\n");
    printf ("\tto be sent over the network to be played.  A sound will be\n");
    printf ("\tread from standard input if its name is `-'.\n");
    printf ("\tIn most cases this option is not necessary since rplay will\n");
    printf ("\twill automatically check if a flow can be used.\n");
    printf ("\n");

    printf ("-n N, --count=N\n");
    printf ("\tNumber of times to play the sound, default = %d.\n",
	    RPLAY_DEFAULT_COUNT);
    printf ("\n");

    printf ("-N N, --list-count=N\n");
    printf ("\tNumber of times to play all the sounds, default = %d.\n",
	    RPLAY_DEFAULT_LIST_COUNT);
    printf ("\n");

    printf ("--list-name=NAME\n");
    printf ("\tName this list NAME.  rplayd appends sounds with the same\n");
    printf ("\tNAME into the same sound list -- it plays them sequentially.\n");
    printf ("\n");

    printf ("--help\n");
    printf ("\tDisplay helpful information.\n");
    printf ("\n");

    printf ("-h HOST, --host=HOST\n");
    printf ("\tSpecify the rplay host, default = %s.\n", rplay_default_host ());
    printf ("\n");

    printf ("-i INFO, --info=INFO\n");
    printf ("\tAudio information for a sound file.  This option is intended\n");
    printf ("\tto be used when sounds are read from standard input.\n");
    printf ("\tINFO must be of the form:\n");
    printf ("\t    `sample-rate,format,bits,channels,byte-order,offset'\n");
    printf ("\tAn example would be: 8000,ulaw,8,1,big-endian,0\n");
    printf ("\tShorthand info is provided for Sun's audio devices using the\n");
    printf ("\tfollowing options: --info-amd, --info-dbri, --info-cs4231.\n");
    printf ("\n");

    printf ("--no-flow\n");
    printf ("\tNever use RPTP flows even if --flow is specified.\n");
    printf ("\n");

    printf ("-p, --pause\n");
    printf ("\tPause sounds.\n");
    printf ("\n");

    printf ("--port=PORT\n");
    printf ("\tUse PORT instead of the default RPLAY/UDP or RPTP/TCP port.\n");
    printf ("\n");

    printf ("-P N, --priority=N\n");
    printf ("\tPlay sounds at priority N (%d <= N <= %d), default = %d.\n",
	    RPLAY_MIN_PRIORITY,
	    RPLAY_MAX_PRIORITY,
	    RPLAY_DEFAULT_PRIORITY);
    printf ("\n");

    printf ("-r, --random\n");
    printf ("\tRandomly choose one of the given sounds.\n");
    printf ("\n");

    printf ("--reset\n");
    printf ("\tTell the server to reset itself.\n");
    printf ("\n");

    printf ("-R N, --sample-rate=N\n");
    printf ("\tPlay sounds at sample rate N, default = %d.\n",
	    RPLAY_DEFAULT_SAMPLE_RATE);
    printf ("\n");

    printf ("-s, --stop\n");
    printf ("\tStop sounds.\n");
    printf ("\n");

    printf ("--version\n");
    printf ("\tPrint the rplay version and exit.\n");
    printf ("\n");

    printf ("-v N, --volume=N\n");
    printf ("\tPlay sounds at volume N (%d <= N <= %d), default = %d.\n",
	    RPLAY_MIN_VOLUME,
	    RPLAY_MAX_VOLUME,
	    RPLAY_DEFAULT_VOLUME);

    exit (1);
}
