/* 
 * Copyright (C) 1995 Andrew Scherpbier <Andrew@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "io.h"
#include "state.h"
#include "connect.h"

#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <rplay.h>
#include <memory.h>
#include <stdarg.h>

typedef struct BufferS
{
	struct BufferS	*next;
	char			*data;
	char			*current;
	int				length;
	Callback		callback;
} Buffer;

typedef struct
{
	Buffer			*ilist;
	Buffer			*itail;
	Callback		read_callback;
	Callback		process_callback;
	char			*line;
	int				line_length;

	Buffer			*olist;
	Buffer			*otail;
	Callback		write_callback;
} Group;

static Group	group[FD_SETSIZE];

static void process_rptp_input(int fd);


/********************************************************************
 * Arrange for a function to be called when data is available on
 * a file descriptor.
 */
void register_read(int fd, Callback callback)
{
	group[fd].read_callback = callback;
}


/********************************************************************
 * Arrange for a function to be called when a full line of text is available.
 */
void register_process(int fd, Callback callback)
{
	group[fd].process_callback = callback;
}


/********************************************************************
 * Arrange for a function to be called when data can be written to
 * a file descriptor.
 */
void register_write(int fd, Callback callback)
{
	group[fd].write_callback = callback;
}


/********************************************************************
 * Send a line to rplayd.  We will append <cr><lf> to the line to terminate it
 */
void io_write_line(int fd, char *fmt, ...)
{
	va_list		args;
	char		buf[RPTP_MAX_LINE];

	va_start(args, fmt);

	vsprintf(buf, fmt, args);
	va_end(args);
	strcat(buf, "\r\n");

	io_write(fd, buf, strlen(buf), NULL);
}


/********************************************************************
 * Send raw data to rplayd.
 * The callback will be called when the data was completely written.
 */
void io_write(int fd, char *data, int length, Callback callback)
{
	Buffer	*new;

	new = (Buffer *) malloc(sizeof(Buffer));
	new->next = NULL;
	new->data = malloc(length);
	memcpy(new->data, data, length);
	new->current = new->data;
	new->length = length;
	new->callback = callback;

	/*
	 * Append the Buffer to the list for this descriptor
	 */
	if (group[fd].otail)
	{
		group[fd].otail->next = new;
		group[fd].otail = new;
	}
	else
	{
		group[fd].otail = group[fd].olist = new;
	}

	/*
	 * If the output system is already writing we don't need to do anything else,
	 * but if not, we need to register our output function
	 */
	if (!group[fd].write_callback)
	{
		register_write(fd, write_proc);
	}
}


/********************************************************************
 * The main I/O loop.  We will sit in a select waiting for events
 * and calling callbacks whenever things happen.
 */
void io_loop()
{
	int			i;
	fd_set		read_fds;
	fd_set		write_fds;

	while (1)
	{
		FD_ZERO(&read_fds);
		FD_ZERO(&write_fds);

		for (i = 0; i < FD_SETSIZE; i++)
		{
			if (group[i].read_callback)
				FD_SET(i, &read_fds);
			if (group[i].write_callback)
				FD_SET(i, &write_fds);
		}

		if (select(FD_SETSIZE, &read_fds, &write_fds, NULL, NULL) < 0)
		{
			if (errno == EINTR)
				continue;
			fprintf(stderr, "%s: select() gave unexpected error %d\n", state.program_name, errno);
			perror(state.program_name);
			exit(1);
		}

		for (i = 0; i < FD_SETSIZE; i++)
		{
			if (group[i].read_callback && FD_ISSET(i, &read_fds))
				(*group[i].read_callback)(i);
			if (group[i].write_callback && FD_ISSET(i, &write_fds))
				(*group[i].write_callback)(i);
		}
	}
}


/********************************************************************
 * Data is available from a rptp connection.  Read it and deal with it.
 */
void read_proc(int fd)
{
	char		buffer[RPTP_MAX_LINE];
	int			n;
	Buffer		*new;

	n = read(fd, buffer, sizeof(buffer));
	if (state.debug > 2)
		printf("%s: read_proc: n = %d, fd = %d, errno = %d\n", state.program_name, n, fd, errno);
	if (n <= 0)
	{
		/*
		 * Something happened to the connection
		 */
		tcp_close(state.command);
		tcp_close(state.data);
		state.command_state = Not_connected;
		state.data_state = Not_connected;
		display_message("Lost connection with %s", state.rptp_host);
		if (state.debug)
			printf("%s: lost connection on %d\n", state.program_name, fd);
		start_connect_machine();
		return;
	}

	new = (Buffer *) malloc(sizeof(Buffer));
	new->next = NULL;
	new->data = malloc(n);
	new->current = new->data;
	memcpy(new->data, buffer, n);
	new->length = n;
	
	if (group[fd].itail)
	{
		group[fd].itail->next = new;
		group[fd].itail = new;
	}
	else
	{
		group[fd].itail = group[fd].ilist = new;
	}

	process_rptp_input(fd);
}


/********************************************************************
 * Try to takes full lines from the input queue and pass them on to the
 * registered callback.
 */
static void process_rptp_input(int fd)
{
	char		*p;
	Buffer		*ptr;

	/*
	 * Only actually process data it we can actually deal with it.
	 */
	if (!group[fd].process_callback)
		return;

	/*
	 * Make sure we have a valid line buffer.  This is the buffer which
	 * will accumulate a full line from the list of input buffers.
	 */
	if (!group[fd].line)
	{
		group[fd].line = malloc(RPTP_MAX_LINE * 2);
		group[fd].line_length = 0;
		group[fd].line[0] = '\0';
	}

	while ((ptr = group[fd].ilist))
	{
		if (p = memchr(ptr->current, '\r', ptr->length))
		{
			int		size = p - ptr->current;
			memcpy(group[fd].line + group[fd].line_length, ptr->current, size);
			group[fd].line_length += size;
			group[fd].line[group[fd].line_length] = '\0';
			if (group[fd].process_callback)
				(*group[fd].process_callback)(fd, group[fd].line);
			group[fd].line_length = 0;

			p += 2;
			if (p < ptr->current + ptr->length)
			{
				/*
				 * There is still unused data in this block.
				 */
				ptr->length -= p - ptr->current;
				ptr->current = p;
			}
			else
			{
				free(ptr->data);
				if (ptr == group[fd].itail)
				{
					group[fd].ilist = group[fd].itail = NULL;
				}
				else
				{
					group[fd].ilist = ptr->next;
				}
				free(ptr);
			}
		}
		else
		{
			/*
			 * The current block of data doesn't contain any carriage
			 * returns.  We'll just add it to our line and get rid of it.
			 */
			memcpy(group[fd].line + group[fd].line_length, ptr->current, ptr->length);
			group[fd].line_length += ptr->length;
			free(ptr->data);
			if (ptr == group[fd].itail)
			{
				group[fd].ilist = group[fd].itail = NULL;
			}
			else
			{
				group[fd].ilist = ptr->next;
			}
			free(ptr);
		}
	}
}


/********************************************************************
 * We can write data to a rptp connection.
 */
void write_proc(int fd)
{
	Buffer	*current;
	int		n;

	if (!group[fd].olist)
	{
		register_write(fd, NULL);
	}

	current = group[fd].olist;

	n = write(fd, current->current, current->length);
	if (state.debug > 2)
		printf("%s: write_proc: wrote %d bytes.  current_id = %s\n", state.program_name, n, state.current_id);
	if (n < 0)
	{
		return;
	}
	current->length -= n;
	current->current += n;
	if (current->length <= 0)
	{
		if (current->callback)
		{
			(*current->callback)(fd);
		}
		free(current->data);
		group[fd].olist = current->next;
		if (group[fd].otail == current)
		{
			group[fd].otail = NULL;
		}
		free(current);

		if (!group[fd].olist)
		{
			register_write(fd, NULL);
		}
	}
}
