/* 
 * Copyright (C) 1995 Andrew Scherpbier <Andrew@sdsu.edu>
 *
 * This file is part of rplay.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "state.h"
#include "tcp.h"

#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>

/********************************************************************
 * Create a new connection object.  This will initialize the members
 * of the structure.
 */
Connection *tcp_new()
{
	Connection	*c = (Connection *) malloc(sizeof(Connection));
	if (!c)
		return NULL;

	c->state = C_Not_connected;
	c->fd = -1;
	c->port = 23;
	strcpy(c->hostname, "localhost");
	if (state.debug)
		printf("%s: new connection create. host = %s\n", state.program_name, c->hostname);
	return c;
}


/********************************************************************
 * Start the connection establishment.  This assumes that the hostname
 * and port values of the Connection object are correct.
 */
int tcp_initiate_connect(Connection *c)
{
	struct sockaddr_in		server_address;
	struct hostent			*hp;
	unsigned long			address;
	int						flags;

	c->state = Not_connected;

	/*
	 * Assign the server name and port.
	 */
	server_address.sin_family = AF_INET;
	address = inet_addr(c->hostname);
	if (address == ~0)
	{
		hp = gethostbyname(c->hostname);
		if (hp == NULL)
		{
			return -1;
		}
		memcpy((char *) &server_address.sin_addr, (char *) hp->h_addr, hp->h_length);
	}
	else
	{
		memcpy((char *) &server_address.sin_addr, (char *) &address, sizeof(address));
	}
	server_address.sin_port = htons(c->port);

	/*
	 * Keep a copy of the address for later...
	 */
	memcpy(c->addr, (char *) &server_address.sin_addr, sizeof(address));

	/*
	 * Setup for the connect
	 */
	c->fd = socket(AF_INET, SOCK_STREAM, 0);
	if (c->fd < 0)
		return -1;

	/*
	 * Make the socket non-blocking
	 */
	flags = fcntl(c->fd, F_GETFL, 0);
	if (flags < 0)
		return -1;
#ifdef linux
	flags |= O_NONBLOCK;
#else
	flags |= FNDELAY;
#endif linux
	if (fcntl(c->fd, F_SETFL, flags) < 0)
		return -1;

	/*
	 * Now everthing should be ready to build the connection...
	 */
	if (connect(c->fd, (struct sockaddr *) &server_address, sizeof(server_address)) < 0)
	{
		if (state.debug)
		{
			perror(state.program_name);
		}
		if (errno == EINPROGRESS)
		{
			/*
			 * The connect process has been started.  Set the connection state to this.
			 */
			c->state = C_Building_connection;
		}
		else if (errno == ECONNREFUSED)
		{
			c->state = C_Not_connected;
			close(c->fd);
		}
	}
	else
	{
		c->state = C_Connected;
		state.command_state = Connected;
	}
	return 0;
}


/********************************************************************
 * Attempt to make another non-blocking connect.
 */
int tcp_connect(Connection *c)
{
	struct sockaddr_in		server_address;
	int						n;

	server_address.sin_family = AF_INET;
	memcpy((char *) &server_address.sin_addr, c->addr, 4);
	server_address.sin_port = htons(c->port);

	n = connect(c->fd, (struct sockaddr *) &server_address, sizeof(server_address));
	if (n < 0 && errno == EISCONN)
		c->state = C_Connected;
	return n;
}


/********************************************************************
 * Close an already connected connection.
 */
int tcp_close(Connection *c)
{
	if (c && c->fd >= 0)
	{
		close(c->fd);
		c->state = C_Not_connected;
		c->fd = -1;
	}
}


