/*-
 * Copyright (c) 1995 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * BSDI 2.0 specific driver entry points
 */

#include "midi-bsdi2.0.h"
#include "midifn.h"

static void midiforceintr __P((struct midi_softc *));

struct cfdriver midicd = {
	NULL, "midi", midiprobe, midiattach, DV_DULL,
	sizeof(struct bsdi20_midi_softc)
};

struct devsw midisw = {
	&midicd,
	midiopen, midiclose, midiread, midiwrite, midiioctl, midiselect,
	nommap, nostrat, nodump, nopsize, 0, nostop
};

int
midiprobe(parent, cf, aux)
	struct device *parent;
	struct cfdata *cf;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	struct midi_softc softc;

	/*
	 * We're faking a temp softc here.  All it really needs is
	 * the addr field, but we supply it so midiforceintr can
	 * call midi_reset
	 */
	softc.addr = ia->ia_iobase;

	/* should be able to do a reset if it exists */
	if (ia->ia_irq == IRQUNK) {
		ia->ia_irq = isa_discoverintr(midiforceintr, &softc);
		if (ffs(ia->ia_irq) - 1 == 0)
			return (0);
	}
	if (!midi_reset(&softc))
		return (0);

	/* We only use addr and addr + 1 */
	ia->ia_iosize = 2;
	ia->ia_msize = 0;

	return (1);
}

void
midiforceintr(softc)
	struct midi_softc *softc;
{

	/*
	 * If the first reset fails, it's because we are in UART
	 * mode.  UART mode does not generate interrupts or
	 * acks for commands.  Thus the first reset takes the
	 * board out of UART mode and the second will cause
	 * an ack and an interrupt.
	 */
	if (!midi_reset(softc))
		midi_reset(softc);
}

void
midiattach(parent, self, aux)
	struct device *parent, *self;
	void *aux;
{
	register struct isa_attach_args *ia = (struct isa_attach_args *)aux;
	struct bsdi20_midi_softc *bsdi_softc;
	struct midi_softc *softc;
	u_char rev;

	bsdi_softc = (struct bsdi20_midi_softc *)self;
	softc = &bsdi_softc->s;

	rev = midi_init_dev(softc, ia->ia_iobase);
	if (rev & 0x04)
		printf(" SMPTE Equipped");
	printf("\n");

	/* allocate memory for event queues */
	if ((softc->rqueue = malloc(sizeof(struct event_queue), M_DEVBUF,
	    M_NOWAIT)) == NULL) {
		log(LOG_ERR, "No memory for rqueue\n");
		softc->status |= MIDI_UNAVAILABLE;
		return;
	}
	if ((softc->wqueue = malloc(sizeof(struct event_queue), M_DEVBUF,
	    M_NOWAIT)) == NULL) {
		log(LOG_ERR, "No memory for wqueue\n");
		softc->status |= MIDI_UNAVAILABLE;
		free(softc->rqueue, M_DEVBUF);
		return;
	}
	/* zero read/write queue to clear stynamic structures */
	bzero(softc->rqueue, sizeof(struct event_queue));
	bzero(softc->wqueue, sizeof(struct event_queue));
	stynamic_init(&softc->rpartial);
	stynamic_init(&softc->wpartial);
	softc->wpartialpos = 0;

	isa_establish(&bsdi_softc->sc_id, &bsdi_softc->sc_dev);
        bsdi_softc->sc_ih.ih_fun = midiintr;
        bsdi_softc->sc_ih.ih_arg = softc;
	intr_establish(ia->ia_irq, &bsdi_softc->sc_ih, DV_TTY);
}

int
midiopen(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
	int f;

	f = 0;
	if (flag & FREAD)
		f |= MIDI_OREAD;
	if (flag & FWRITE)
		f |= MIDI_OWRITE;
	return (gen_midiopen((void *)dev, f, p->p_pid, &midicd));
}

int
midiclose(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
	int unit;
	struct bsdi20_midi_softc *sc;

	unit = minor(dev) & 0x7f;
	sc = midicd.cd_devs[unit];
	return (gen_midiclose(&sc->s));
}

int
midiread(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	int f, unit;
	struct bsdi20_midi_softc *sc;

	f = 0;
	if (flag & IO_NDELAY)
		f |= MIDI_NDELAY;

	unit = minor(dev) & 0x7f;
	sc = midicd.cd_devs[unit];
	return (gen_midiread(&sc->s, uio, f));
}

int
midiwrite(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	int f, unit;
	struct bsdi20_midi_softc *sc;

	f = 0;
	if (flag & IO_NDELAY)
		f |= MIDI_NDELAY;

	unit = minor(dev) & 0x7f;
	sc = midicd.cd_devs[unit];
	return (gen_midiwrite(&sc->s, uio, f));
}

int
midiintr(softc)
	struct bsdi20_midi_softc *softc;
{

	return(gen_midiintr(&softc->s));
}

int
midiioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	int unit;
	struct bsdi20_midi_softc *sc;

	unit = minor(dev) & 0x7f;
	sc = midicd.cd_devs[unit];
	return (gen_midiioctl(&sc->s, cmd, data));
}

/*
 * gen_midiselect
 * I don't have a good way of generalizing select yet, so it is done
 * on a per machine basis.
 */
int
midiselect(dev, which, p)
	dev_t dev;
	int which;
	struct proc *p;
{
	struct bsdi20_midi_softc *sc;
	struct midi_softc *softc;
	int ret, s, unit;

	unit = minor(dev) & 0x7f;
	sc = midicd.cd_devs[unit];
	softc = &sc->s;

	s = spltty();
	switch (which) {
	case FREAD:
		if (!(softc->status & MIDI_RD_BLOCK))
			ret = 1;
		else {
			ret = 0;
			softc->status |= MIDI_RSEL;
			selrecord(p, &softc->rsel);
		}
		break;
	case FWRITE:
		if (!(softc->status & MIDI_WR_BLOCK))
			ret = 1;
		else {
			ret = 0;
			softc->status |= MIDI_WSEL;
			selrecord(p, &softc->wsel);
		}
		break;
	default:
		ret = 0;
		softc->status |= MIDI_ESEL;
		selrecord(p, &softc->esel);
		break;
	}
	splx(s);
	return (ret);
}

struct midi_softc *
UNIT_TO_SOFTC(unit, client)
	int unit;
	void *client;
{
	struct cfdriver *midicd = (struct cfdriver *)client;
	struct bsdi20_midi_softc *sc;
	struct midi_softc *softc;

	if (unit >= midicd->cd_ndevs || (sc = midicd->cd_devs[unit])
	    == NULL)
		softc = NULL;
	softc = &sc->s;
	return (softc);
}

void
SIGNAL_PG(pgid, sig)
	int pgid, sig;
{
	struct proc *p;

	if (pgid < 0)
		gsignal(-pgid, sig);
	else if ((p = pfind(pgid)) != 0)
		psignal(p, sig);
}

u_long
GET_KERN_TIME(softc)
	struct midi_softc *softc;
{

	return(time.tv_sec * softc->hz + time.tv_usec / (1000000 / softc->hz));
}

void
TIMEOUT(id, fn, arg, t)
	TIMEOUT_ID *id;
	TIMEOUT_FN fn;
	void *arg;
	int t;
{

	timeout(fn, arg, t);
	*id = fn;
}
