// files.c : interactively opening files
//           wolf 12/94
#include "window.h"
#include "files.h"
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#define CONFIRM_UNDEF 0
#define CONFIRM_YES 1
#define CONFIRM_NO 2

// simply set a variable of type T to value 
template <class T>
class variable_button : public button {
  T value;
  T* variable;
public:
  variable_button(window &parent, char *Name, T *variable, T value, 
		  int w, int h, int x, int y) :
    button(parent,Name,w,h,x,y), value(value), variable(variable) { }
  void BRelease_CB(XButtonEvent) { *variable = value; }
};

// popup window with two buttons, special loop handler,
// can be used as stack object    
class confirm_box : public main_window {
char **text;
int status;
public:  
  confirm_box(char *Name, int w, int h, char **text ) :
    main_window(Name,w,h), text(text) {
    status = CONFIRM_UNDEF;
    int y = h - 25, x0 = 10, x1 = w/2 + 10, wb = w/2 - 20; 
    button *bbx[] = {
      new variable_button <int> (*this,"Yes",&status, CONFIRM_YES,wb,20,x0,y),
      new variable_button <int> (*this,"No",&status, CONFIRM_NO, wb, 20,x1,y)
    };    
  }
  void Expose_CB(XExposeEvent ev) {
    int ln = 0, y = 10;
    while (text[ln]) { y+= 15; PlaceText(text[ln++],0,y); }
  }  
  
  int local_loop() {
    RealizeChildren(); 
    do { 
      XEvent event;   
      XNextEvent(display, &event);
      // only handle events for myself and my children windows
      // events for others are flushed !
      unsigned wid = event.xany.window; 
      if ( wid == Win ) { handle_event(event); continue; }
      win_list *ch = children;
      while (ch) {
        if (wid == ch->child->Win) { handle_event(event); break; }
	ch = ch->next;
      }
    } while (status == CONFIRM_UNDEF);
    Unmap();
    return(status);
  }
};

// for opening output files (type = "w") with overwrite test
// if it already exists opens confirmation box "yes" , "no" 
// in case "no" returns NULL, else FILE-pointer to opened file
FILE *fopen_test(char *fina) {
  FILE *ff = fopen(fina,"r");
  if (ff) { // file exists
    fclose(ff);
    char *text[] = {"File exists :",fina,"","Overwrite it ?",0};
    // there exists only one instance cbox per application (ressources) !
    static confirm_box *cbox = new confirm_box("Existing File",150,120,text); 
    // klappt nicht : XMoveWindow(display,cbox.Win,400,400);
    int stat = cbox->local_loop();
    if (stat == CONFIRM_NO) return (NULL);
  }
  return fopen(fina,"w");
}

class file_window;

// class for entering filenames
class fina_edit : public edit_window {
  void (file_window::*meth)();
  file_window *inst;
public: 
  fina_edit(window &parent, char *defname, void (file_window::*meth)(), 
            file_window *inst, int w, int h, int x, int y) : 
    edit_window(parent,defname,w,h,x,y), meth(meth), inst(inst) {}
  virtual void enter() {
    (inst->*meth)(); // also calls ok-method like the button
   // else format_error();
  } 
};

// opens a popup window to enter a filename (with defname)
// ok-button tries to open it, depending on mode
// MODE_WRITE : if file already exists then popup a confirmation box 
// MODE_READ : no more used -> replaced by file_selection_box
class file_window : public main_window {
public: 
  FILE *fptr;
  char *fname;
  int mode;
private:
  fina_edit *name_ed;
  void ok() { 
    switch (mode) {
    case MODE_WRITE: fptr = fopen_test(fname); break;
    case MODE_READ: fptr = fopen(fname,"r"); break;
    case MODE_OVWRT: fptr = fopen(fname,"w"); break;
    }
    if (fptr) exit_flag = True; else printf("error opening file %s\n",fname);
  }

public:
  file_window() : 
    main_window("open file",200,100) {
    name_ed = new fina_edit(*this,"",&file_window::ok,this,180,20,10,10);
    fname = name_ed->value; 
    button *ok = new instance_button <file_window> 
                 (*this,"Ok",&file_window::ok,this,60,20,20,50);     
    button *esc = // simply exits main_loop since sets exit_flag
       new variable_button <int> (*this,"Abort",&exit_flag,True,60,20,120,50);
  }  
 // calls main_loop, returns name and opened file
 FILE* file_input_loop(char *defname, int amode, char *name_ret) {
    fptr = NULL; // initial value -> for Abort
    mode = amode; strcpy(name_ed->value,defname);
    main_loop();
    strcpy(name_ret,fname);
    return fptr;
  }
};

typedef void (*VPCP)(void *, char *);

class file_display : public plate {
public:
  char *val;
  file_display(window &parent, int w, int h, int x, int y) :
    plate (parent, w,h,x,y, down3d) { val = NULL; }
  virtual void redraw() { plate::redraw(); PlaceText(val); }
  void set_val(char *x) { val = x; redraw(); }
};

// class for entering filename masks calls enter_cb 
class mask_edit : public edit_window {
  VPCP cb;
  void *wptr;
public: 
  mask_edit(window &parent, VPCP enter_cb, void *inst, 
	    int w,int h, int x, int y) : 
    edit_window(parent,"",w,h,x,y) { cb = enter_cb; wptr = inst;}
  virtual void enter() { cb(wptr,value); } 
};

// class for item in selector :
// 2 callbacks : enter_cb on enter, and act_cb on bpress
class select_item : public plate {
  VPCP ecb, acb;
  void *wptr;

public: 
  char * value;
  select_item(window &parent, VPCP enter_cb, VPCP act_cb, void *wp,
	      int w, int h, int x, int y) :
    plate(parent,w,h,x,y,flat3d) { 
    char empty = 0; value = &empty;
    ecb = enter_cb; acb = act_cb, wptr = wp;
    selection_mask &= ~(PointerMotionMask);
  }

  virtual void redraw() { if (!hidden) { plate::redraw(); PlaceText(value); } }
  virtual void Enter_CB(XCrossingEvent) {  frame3d(up3d); ecb(wptr,value); }
  
  virtual void Leave_CB(XCrossingEvent) { default_frame(); }
  virtual void BPress_CB(XButtonEvent) { frame3d(down3d); acb(wptr,value); }
 
  virtual void BRelease_CB(XButtonEvent) { default_frame(); }
  void enable(char *x) { value = x; hidden = False; redraw(); }
  void disable() { hidden = True; }
};

// makes a plate with (max) nits select_items -> selection box
// not all are neseccarily drawn
// this should be completed with a scrollbar
class selector : public plate {
select_item **pits;
char **itptr;
int itdsp; // number of actually displayed items
int itmax; // total number of items
int ishift; // index of first displayed item
  window *interior; // the inner part w/o boundary
public:
  selector(window &parent, int nits, VPCP show_cb, VPCP act_cb,
           int w, int h, int x, int y, int hi) :
    plate (parent, w,h,x,y,down3d) {
    selection_mask |= KeyPressMask; // catch key events
    itdsp = nits; 
    interior = new window(*this,w-6,h-6,3,3,0); // 3 pixels for boundary
    int yp = 0; int i;
    pits = new (select_item* [nits]);
    for (i=0; i < nits; i++) {
      pits[i] = new select_item(*interior,show_cb,act_cb,&parent,w-6,hi,0,yp);
      yp += hi;
    }
  }
  void set_items(char **items, int nits) { 
    int i; itptr = items; itmax = nits; ishift = 0;
    for (i=0; i < itdsp; i++) { 
      if (i < nits) pits[i]->enable(items[i]); else pits[i]->disable();
     }
  }

  ~selector() { delete[] pits; }
   
  void resize(int w, int h) { 
    if (width == w && height == h) return; // only once
    width = w; height = h; 
    XResizeWindow(display,Win,w,h);  
    if (h > 6) {  // else no items
      XResizeWindow(display,interior->Win,w-6,h-6); 
      RealizeChildren(); 
    }
  }
  void KeyPress_CB(XKeyEvent ev) {
    KeySym ks = XLookupKeysym(&ev,ev.state);
    int i, ishift_prev = ishift; 
    switch (ks) { 
      case XK_Up: 	ishift--; break;	   
      case XK_Down: 	ishift++; break;	 
      case XK_Prior:	ishift -= itdsp; break;	 
      case XK_Next: 	ishift += itdsp; break;
    }
    if (ishift + itdsp > itmax) ishift = itmax-itdsp; 
    if (ishift < 0) ishift = 0; 
    if (ishift != ishift_prev )
      for (i=0;i < itdsp; i++) pits[i]->enable(itptr[i+ishift]); 
  }
 
};

// recursively parses filename against mask, recons only *, but yet no ?
// see also "test-compare.c"
int filter(const char *mask, const char *fina) {
  // printf("%s %s\n",mask,fina);
  if (*mask == 0) return (*fina == 0); else
    if (*mask == '*') { 
      int i; 
      // compare all tail-strings of fina with mask part following "*"
      for (i = strlen(fina); i >= 0; i--) // end 0 is included !
        if (filter(mask+1,fina+i)) return (1);
      return (0); // no match found
    } else 
      if (*mask == *fina) return (filter(mask+1,fina+1)); else return (0);
}

#define ITMAX 200  // max # of items in each selector
#define BHEIGHT 18 // button height
#define ITDISP 20  // max of displayed items in selectors

// compare function for qsort
int qcomp(char **e1, char **e2) { 
  return (strcmp(*e1,*e2));  //  compare strings pointed to, not the pointers
}

typedef int (*IVPVP)(const void *,const void *); // the correct type for qsort

// interactively open files for reading (scanning directory)
// after completion fptr is either the opened FILE* (Ok) or NULL (Abort)
// functions: 
//   selection mask = edit_window (*.c), enter-key (Mask)
//   selectors - BPress (Ok), Enter (display)
//   buttons : Ok, Mask, Abort  
class file_selection_box : public main_window {
public: 
  FILE *fptr;
  char *fname;
  char strings[4000]; // all names in one field
  char *fitems[ITMAX],*ditems[ITMAX]; // max 200 entries into this field
  char pwd[100]; // current directory path
  int fits, dits;
private:
  edit_window *mask_ed;
  file_display *fdsp;
  selector *fsel,*dsel;
  button *okb, *abortb, *chb;

  // searches pwd for matching files and displays them in fsel
  void make_select(char *mask) {
    DIR *dirp = opendir(".");
    struct dirent *dp; 
    fits = dits = 0; int nstrp = 0; 
    for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
      struct stat st; 
      stat(dp->d_name,&st);
      Bool regfi = (st.st_mode & S_IFREG), dirfi = (st.st_mode & S_IFDIR);
      if ((dirfi && (dits < ITMAX)) || 
          (regfi && filter(mask,dp->d_name) && (fits < ITMAX))) {	 
	int nl = strlen(dp->d_name) + 1; 
	if ((nl + nstrp) >= 4000) error("too long name strings");
        char *strent = strings + nstrp; // entry point in strings
	strcpy(strent,dp->d_name); 
      
        if (regfi) fitems[fits++] = strent; else ditems[dits++] = strent;
        nstrp += nl;
      }
    }   
    qsort(fitems,fits,4,(IVPVP) qcomp); // sorting the pointers 
    qsort(ditems,dits,4,(IVPVP) qcomp); // not the strings !
    fsel->set_items(fitems, fits);
    dsel->set_items(ditems, dits);
    int ffits = MIN(fits,ITDISP), ddits =  MIN(dits,ITDISP);   
    int wh = MAX(ffits, ddits) * BHEIGHT + 6; 
    XResizeWindow(display,Win,220,wh+110);
    dsel->resize(80,ddits*BHEIGHT + 6);
    fsel->resize(110,ffits*BHEIGHT + 6); 

    // printf("window %d %d\n",Win,wh+110);
  }
   
  void ok() { 
    fptr = fopen(fname,"r");
    if (fptr) exit_flag = True; else printf("error opening file %s\n",fname);
  }

  void fsel_cb(char *val) { fname = val; fdsp->set_val(val);  }
  // callback for dsel enter 
  void dsel_cb(char *val) { 
    //  getcwd(pwd,100); 
    int lpwd = strlen(pwd);
    sprintf(pwd+lpwd,"/%s",val); // append val to pwd
    fdsp->set_val(pwd);  
    pwd[lpwd] = 0; // reset pwd
  }

  void chmask() { // callback for change mask
    make_select(mask_ed->value);
  }
  void cd_cb(char *val) { 
    chdir(val); 
    getcwd(pwd,100); fdsp->set_val(pwd);  
    make_select(mask_ed->value); 
 }
public:
  file_selection_box() : 
    main_window("file selection box",220,250) {
    
    char empty = 0; fname = &empty; 
    fdsp = new file_display(*this,200,20,10,10);
    window *tw = new text_win(*this,"selection mask",200,20,10,30);
    mask_ed = new mask_edit(*this, (VPCP) &make_select, this,200,20,10,50);
    // directory selector 
    dsel = new selector(*this, ITDISP, (VPCP) &dsel_cb, (VPCP) &cd_cb,
		       80,0,10,80,BHEIGHT);
    // file selector
    fsel = new selector(*this, ITDISP, (VPCP) &fsel_cb, (VPCP) &ok,
		       110,100,100,80,BHEIGHT);
    okb = new instance_button <file_selection_box> 
                 (*this,"Ok",&file_selection_box::ok,this,50,20,0,0);
    chb = new instance_button <file_selection_box> 
                 (*this,"Mask",&file_selection_box::chmask,this,50,20,0,0);
    abortb =
       new variable_button <int> (*this,"Abort",&exit_flag,True,50,20,0,0);

  }   
 
  void resize(int w, int h) { 
    if (width == w && height == h) return; // only once
    // printf("resize %x ->  %d %d\n",Win,w,h); 
    width = w; height = h;
    XMoveWindow(display,okb->Win,10,height-25);  
    XMoveWindow(display,chb->Win,75,height-25);
    XMoveWindow(display,abortb->Win,140,height-25);
  } 
  ~file_selection_box() { }

  // calls main_loop, returns selected filename and opened file
  FILE* file_input_loop(char *defname, char *name_ret) {
    fptr = NULL; // initial value -> for Abort
    strcpy(mask_ed->value, defname);
    getcwd(pwd,100); fdsp->val = pwd; 
    fname = NULL;
    make_select(mask_ed->value);
    main_loop();
    if (fptr) strcpy(name_ret,fname);
    return fptr;
  }
};

// simple interface function
// creates popup window and calls its main_loop 
// the opened FILE* is returned and its filename (with strcpy)
FILE *open_file_interactive(char *defname, char *name_ret, int mode) {
  // extern int debug_create; debug_create = True;
  if (mode == MODE_READ) {
    // there should only exist one instance of this fsb per application !
    static file_selection_box fsb;
    return (fsb.file_input_loop(defname, name_ret));
  } else {
    static file_window ofw;   
    return (ofw.file_input_loop(defname, mode, name_ret));
  }
}
