#ifndef _LOGRECS_H_
#define _LOGRECS_H_
#ifndef __LOGRECS_H__
#define __LOGRECS_H__
/*
 *   $RCSfile: logrecs.h,v $  
 *   $Revision: 2.21 $  
 *   $Date: 1993/04/21 17:21:23 $      
 */ 

/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/

/*
 *	define the maximum number of images that can be passed
 *	in the write log call.  Also the max size of a log record must
 *	fit in a max page since these are used to buffer log records
 *	which span multiple pages.  A sanity check is performed in
 *	openLogFile().
 */
#define MAX_LOGREC_IMAGES		3
#define LOGREC_BUF_PAGE2SIZE	( MAX_PAGE2SIZE  )
#define MAX_LOGREC_LEN			( MAX_PAGESIZE  )


/*
 *	Magic numbers for log records are only two bytes to save space.
 */
typedef UTWO MAGICLOG;

/*
 *	define the structure of a log record
 *	note that a four must be the last member for alignment purposes
 */
typedef struct	{

	MAGICLOG	magic;				/* magic number of this record	*/
	UTWO		length;				/* size of the log record		*/
	UONE		type;				/* type of record entry			*/
	UONE		imageCount;			/* count of record images		*/
	UONE		action;				/* client specific action		*/
	UONE		flags;
	TID			tid;				/* tid of this record			*/
	LSNOFFSET	previousLSN;		/* previous LSN for transaction	*/
	LSNOFFSET	nextUndoLSN;		/* for compensation log records	*/
	LSN			recordLSN;			/* lsn of this record			*/
	LRC			actionLRC;			/* lrc of this action on page 	*/
	PID			actionPid;			/* pid the action applies to	*/
	/*FID			actionFid;	*/		/* file page is a member of	*/
	UTWO		imageOffset[MAX_LOGREC_IMAGES];	/* offset of the images			*/
	UTWO		imageSize[MAX_LOGREC_IMAGES];	/* size of the images			*/
	/* UFOUR		delimiter;	*/		/* for debugging ease		*/

} LOGRECORDHDR;


/*
 *	define the magic numbers of a log record
 */
#define LOGRECORD_MAGIC		0xbbbb
#define LOGRECORD_DELIMIT	0xcccccccc

#define MAX_LOGIMAGE_LEN	(MAX_LOGREC_LEN - sizeof(LOGRECORDHDR))

/*
 *	define the flags for the log records
 */
#define REDO_ONLY		0x1
#define REDO_ALWAYS		0x2
#define UNDO_ONLY		0x4

/*
 *	define the structure for the log page header
 */
typedef struct	{

	MAGIC		magic;
	SHORTPID	pageNumber;
	UFOUR		wrapCount;
	SHORTPID	lastRecord;

} LOGPAGEHDR;


/*
 *	define the magic number of a log page
 */
#define LOGPAGE_MAGIC		0xaaaaaaaa


/*
 *	define the initial wrap count of the log
 */
#define NULL_LOG_WRAP_COUNT		0
#define INIT_LOG_WRAP_COUNT		1


/*
 *	define the null lastRecord pointer
 */
#define NULL_LAST_RECORD		0

#define VOID void


/*
 *	define a record type that is used to pass
 *	information into a write log call for the server
 */
typedef struct	{

	UONE	action;
	UONE	type;
	UONE	flags;
	UONE	imageCount;
	UTWO	imageSize[MAX_LOGREC_IMAGES];
	VOID	*imageData[MAX_LOGREC_IMAGES];
	UFOUR	nextUndoLSN;
	PID		*actionPid;
	LRC		*actionLRC;
	/*FID		*actionFid;*/

} LOGRECORDINFO;


/*
 *	define a record type that is used to pass
 *	information into a write log call for the client
 */
typedef struct	{

	UONE	action;
	UONE	type;
	UONE	flags;
	UONE	imageCount;
	UTWO	imageSize[MAX_LOGREC_IMAGES];
	VOID	*imageData[MAX_LOGREC_IMAGES];
	UFOUR	nextUndoLSN;
	PID		actionPid;
	LRC		actionLRC;

} CLIENTLOGRECORDINFO;


/*
 *	define flags
 */
#define LOG_FORCE	0x1


/*
 *	define the types of log records
 */
#define LOG_REC_TYPE_NULL				0
#define LOG_REC_TYPE_USER				1
#define LOG_REC_TYPE_ABORT				2
#define LOG_REC_TYPE_COMMIT				3
#define LOG_REC_TYPE_CHECKPOINT			4
#define LOG_REC_TYPE_COMPENSATION		5
#define LOG_REC_TYPE_DIRTYPAGELIST		6
#define LOG_REC_TYPE_MOUNT_VOLUME		7
#define LOG_REC_TYPE_DISMOUNT_VOLUME	8
#define LOG_REC_TYPE_USER_COMPENSATION  9
#define LOG_REC_TYPE_SERVER_PREPARE	   10	/* for distr trans */
#define LOG_REC_TYPE_COORD_PREPARE	   11	/* for distr trans */
#define LOG_REC_TYPE_END			   12	/* for distr trans */


/*
 *	fixed length (image[0]) data for write/insert/delete/append
 */
typedef struct  {
	UNIQUE		unique;	/* unique field of oid 			*/
	SLOTINDEX	slot;	/* slot field of oid (2 bytes)	*/
	UTWO		start;  /* start location of operation	*/
} WRITEINFO;

/*
 *	fixed length (image[0]) data for write/insert/delete/append
 *	This is for those operations where the data is not logged
 */
typedef struct  {
	WRITEINFO	common;
	UTWO		size;  	/* size of operation			*/
} FULLWRITEINFO;

/*
 *	fixed length (image[0]) data for write of large objects
 */
typedef struct  {
	UTWO	start;
	UTWO	size;
	TWO		rootSlot;
	UTWO	prevSize;
} LGWRITEINFO;

/*
 *	fixed length (image[0]) data for update of large obj slots
 */
typedef struct  {
	UTWO	start;
	TWO		rootSlot;
	FOUR	size;
} LGSLOTUPDATEINFO;


/*
 *	Page allocation/deallocation info (image[0] for logging (de)alloc)
 */
typedef struct {
	FOUR		numPages;
	PAGE2SIZE	page2size;
	SHORTPID	firstBitmapPage;
	SHORTPID	volumeHdrPage;
	LRC			headerLRC; 			/* volume header LRC 	*/
} PAGEALLOCINFO;

/*
 *	Information for root entry setting log record
 */
typedef struct {
	ONE		oldFlags;
	UTWO	nameLen;
	UONE	name[MAX_ROOTNAME_SIZE];
} ROOTENTRYLOGINFO;

/*
 *	define the body of a checkpoint record
 */
typedef struct	{

	UTWO		numActiveTrans;
	UTWO		mountedVolCount;
	UFOUR		numDirtyPages;
	UFOUR		wrapCount;

} CHECKPOINTINFO;

/*
 *	define the body of a checkpoint dirty page table record
 *	The checkpoint DPT is sent as a series of log records with
 *	this struct as the first image and a list of DPT entries as
 *	the second image.  Since is may take multiple checkpointDPT
 *	records to record the entire DPT, the recordNum field orders
 *	these records.
 */
typedef struct	{

	UTWO		numDirtyPages;	/* # of pages in this record 	*/
	UTWO		recordNum;		/* sequence # of this record	*/

} CHECKPOINTDPTINFO;


/*
 *	define the state info for a transaction in checkpoint
 */
typedef struct {

	ONE			state;
	TID			tid;
	LSN			firstLSN;
	LSNOFFSET	lastLSN;
	LSNOFFSET	nextUndoLSN;
	/*
	 *	for distr trans
	 */
	UFOUR		numServers;
	LSN			prepareLSN;

} CHECKTRANS;


/*
 *	define the pointer to the checkpoint record
 */
typedef struct	{

	MAGIC		magic;
	UFOUR		wrapCount;
	LSN			checkRecordLSN;
	UFOUR		oldestDirtyLSN;

} CHECKPOINTMASTER;


#define CHECKPOINTMASTER_MAGIC	0x04c6745a


/*
 *	define the magic checking macros
 */
#define CHECK_LOGRECORD_MAGIC(_record)			\
	((_record)->magic != LOGRECORD_MAGIC)


#define CHECK_LOGPAGE_MAGIC(_header)			\
	((_header)->magic != LOGPAGE_MAGIC)


#define REMAIN_BYTES(_openLog, _offset)	(_openLog->pagesize - (_offset + sizeof(FOUR)))


#ifdef DEBUG
	/*
	 *	define a macro to get at the ith element of the log record
	 */
#	define GET_LOG_IMAGE(_recordHeader, index)					\
																\
		( ( (index < (_recordHeader)->imageCount) && (index >= 0) && (index < MAX_LOGREC_IMAGES)) ? \
				(((char *) (_recordHeader)) + (_recordHeader)->imageOffset[index] + sizeof(LOGRECORDHDR))	: \
				(SM_ERROR(TYPE_FATAL, esmINTERNAL), (char*) NULL))

#else

	/*
	 *	define a macro to get at the ith element of the log record
	 */
#	define GET_LOG_IMAGE(_recordHeader, index)					\
																\
		(((char *) _recordHeader) + (_recordHeader)->imageOffset[index] + sizeof(LOGRECORDHDR))
#endif


#ifdef DEBUG
	/*
	 *	define a macro getting the size of the ith element of the 
	 *	log record
	 */
#	define GET_LOG_IMAGE_SIZE(_recordHeader, index)					\
																\
		( ( (index < (_recordHeader)->imageCount) && (index >= 0) ) ? \
				( (_recordHeader)->imageSize[index])	: \
				(SM_ERROR(TYPE_FATAL, esmINTERNAL), -1))

#else

	/*
	 *	define a macro getting the size of the ith element of the 
	 *	log record
	 */
#	define GET_LOG_IMAGE_SIZE(_recordHeader, index)					\
																\
		( (_recordHeader)->imageSize[index])
#endif


/*
 *	define LSN manipulation macros
 */
#define LSN_TO_BLOCK(_lsn, _openLog)		\
	((_openLog)->logFileAddr + ( ((_lsn) >> (_openLog)->page2size) * (_openLog)->blocksPerPage))


#define LOG_PAGE_TO_BLOCK(_page, _openLog)		\
	((_openLog)->logFileAddr + ((_page) * (_openLog)->blocksPerPage))


#define LSN_TO_LOG_PAGE(_lsn, _openLog)				\
		((_lsn) >> (_openLog)->page2size)


#define LOG_PAGE_TO_LSN(_page, _openLog)				\
		((_page) << (_openLog)->page2size)


#define LOG_PAGE_OFFSET(_lsn, _openLog)				\
		((_lsn) & ((_openLog)->pageMask))

/*
 *	offset of first lsn on a page
 */
#define FIRST_LSN (sizeof(LOGPAGEHDR))

/*
 *	Given a zero lsn return the first valid lsn on a page, otherwise
 *	just return the lsn
 */
#define FIRST_LSN_ON_PAGE(_lsn, _openLog)			\
		( LOG_PAGE_OFFSET((_lsn), (_openLog)) == 0 ? (_lsn) + FIRST_LSN : (_lsn) )

#endif __LOGRECS_H__
#endif /* _LOGRECS_H_ */
