#ifndef _OPTION_H_
#define _OPTION_H_
#ifndef __OPTION_H__
#define __OPTION_H__
/*
 *   $RCSfile: option.h,v $  
 *   $Revision: 1.7 $  
 *   $Date: 1993/04/21 17:21:28 $      
 */ 


/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/

class OptionInfo; 	/* forward decl for cfront */

#include "option_macros.h"

const MaxOptLength  = 	MAXOPTLENGTH;
const MaxOptClassification = 	MAXOPTCLASSIFICATION;
const MaxOptClassLength = MAXOPTCLASSLENGTH;


//
// OptionInfo Class:
//
// Description:
//   Information about an option is stored in a OptionInfo object.
//   The possibleValueM field is used to print example possible
//   values for usage.  The defaultValueM field hold a default value
//   if requiredM is FALSE.  It is also used to initialize valueM.
//   DescriptionM is printed for long usage information.  RequiredM
//   indicates that the option must be set.  SetM is set to TRUE
//   by setValue().  ValueM holds the latest value from setValue().
//   If the option is not required and defaultValueM is not NULL then
//   valueM will be set to defaultValueM if setValue() is not called.
//
class OptionInfo {

protected:

	int		id;
	const char*	nameM;				// name of the option
	const char*	possibleValuesM;	// example possible values
	const char*	defaultValueM;		// default value
	const char*	descriptionM;		// description string
	BOOL	requiredM;				// must option be set
	BOOL	setM;					// option has been set
	char	valueM[MaxOptLength+1];	// value for the option

	/*
	 *      call-back function to call when option value is set
	 *
	 *      Call back functions should return 0 on success or a on-zero
	 *      error code on failure.  This error code should not confict
	 *      with the option error codes above.
	 */


	typedef int (*OptionCallBackFunc)(OptionInfo*, const char * value);
	OptionCallBackFunc callBackM;

public:

			OptionInfo();

			// initialize an OptionInfo object
	int		init(int newId, const char* name, const char* newPoss,
				 const char* defaultValue, const char* description,
				 BOOL required, OptionCallBackFunc callBack);

			// returns true if the option name matches matchName
	BOOL	match(const char* matchName);

			// set the value of an option if it is not already set
			// or if overRide is true.  Return 1 if option is set
			// 0 if not set, and -1 on error.
	int		setValue(const char* value, BOOL overRide);

// inline functions 
	void		unsetValue()	{ setM = FALSE; }
	BOOL		set()			{ return setM; }
	BOOL		required()		{ return requiredM; }
	int			ID()			{ return id; }
	const char*	name()			{ return nameM; }
	const char*	possibleValues(){ return possibleValuesM; }	
	const char*	defaultValue()	{ return defaultValueM; }	
	const char*	description()	{ return descriptionM; }	
	char*		value()			{ return valueM; }	
};


//
// OptionList Class
//
// Description:
//   An OptionList manages a set of options.  These options have 
//   a classification hierarchy associated with them.  Each level
//   of the hierarchy is given a string name.  Levels are added
//   with addClassLevel().  The level hierarch is printed in the
//   form: "level1.level2.level3."  A complete option name is
//   specified by "level1.level2.level3.optionName:".  
//
class OptionList {
protected:
	OptionInfo*		options;
	int				optCount;
	int				maxOptions;
	int				scanLocation;
	const char*		levelNameM[MaxOptClassification];
	int				numLevelsM;
	char*			classStringName;
	int				classStringLen;

public:
				OptionList(int length);
				~OptionList();
	OptionInfo*	addOpt();
	int			addClassLevel(const char* name);
	int			setClassLevel(const char* name, int level);
	int			lookup(const char* name, OptionInfo*&);
	int			lookupByClass(const char* optClassName, OptionInfo*&);
	void		printUsage(BOOL longForm, const char* command, FILE* f);
	void		printValues(BOOL longForm, FILE* f);
	void		initScan();		
	OptionInfo*	getNext();		

// inlined
	int			numClassLevels()	{ return numLevelsM; }	
	const char*	className()	{ return classStringName; }	
	
};


class OptionFileScan {
protected:
	OptionList*	optList;
	const char*	fileName;
	FILE*		f;
	int			lineNum;
	char*		line;
	int			maxLineLen;
	char*		classOption;
	
public:
			OptionFileScan(const char* optFile, OptionList* optList);
			~OptionFileScan();
	int 	scan(BOOL overRide, char*& errorString);
};


//
// C interface functions
//

BEGIN_EXTERNC

extern int	processOptionFile(char*, OptionList*, BOOL, char**);
extern BOOL checkRequiredOptions(OptionList*, FILE* );

END_EXTERNC

#endif __OPTION_H__
#endif /* _OPTION_H_ */
