
/*
 * $RCSfile: command.c,v $
 * $Revision: 1.22 $
 * $Date: 1993/05/03 14:19:28 $
 */

/* 
 *  DEVELOPERS, NB: define BASELINE when you are
 *  compiling ET for use with the baseline.
 *  Use #ifndef BASELINE, etc. to hack in changes that
 *  are required to compile ET with your experimental versions
 *  of the SM.  When you check your changes into the baseline,
 *  remove the code that is specific to the old baseline.
 *  ALWAYS leave BASELINE defined in the checked-in version of et (et.h).
 */

#include "et.h"
#include "etError.h"
#include "func.h"
#include "command_funcs.h"

#ifdef BASELINE
#define WHICHVOL
#else
#define WHICHVOL flags, volid
#endif

int
no_command(int cmd)
{
	(void) next_commandarg(Whitespace); /* to get rid of the '\0' stuck in by strtok */
	INFO
		"No such command: %s", cmd2word(cmd)
	ENDINFO
	return esmFAILURE;
}

int
abort_command(ELIPSES)
{
	ignore_restofline();
	/* In this case, we wouldn't want to redo this
	 * transaction, because it's a user-initiated abort.
	 * We just directly call abort_tx(), and do not go
	 * through checkStatus().  checkStatus() also
	 * lets us scan ahead to the next begin, but
	 * we don't want that either, in this case, so
	 * abort_tx() is exactly what we want .
	 */
	(void) abort_tx(TRUE);
	return esmNOERROR;
}

int
commit_command(ELIPSES)
{
	ignore_restofline();
	return commit_tx();
}

int
crashserver_command(ELIPSES)
{
	/* crash [[o# o# ELIPSES or v# v# ...]  | all | used | trans ] */
	FLAGS   flags = VOL_USED_SINCE_INIT;
		/* NB: use VOL_USED_SINCE_INIT so that commit followed by
		 * crash actually does something, yet doesn't crash servers
		 * that we're not even using
		 * The  problem here is the implementation may contact a 
		 * server twice.
		 */
	int 	e;
	VOLID	*volidlist;

	volidlist = get_serverlist (&flags);

	checkNumBufs(__LINE__, __FILE__, TRUE);
	if (flags == VOL_BY_VOLID) {

		if(volidlist == NULL) {
			return esmFAILURE;
		}
		do {
			sm_errno=esmNOERROR;
			e = sm_ShutdownServer(VOL_BY_VOLID, *volidlist, 0);
			if ((e == esmFAILURE) && (sm_errno != esmNOTCONNECTED)
				&& (sm_errno != esmSERVERDIED)) {
				HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_SHUTDOWNSERVER_ERR);
			}
		} while ( *(++volidlist) != 0 );

		checkNumBufs(__LINE__, __FILE__, FALSE);
	} else if ( multiportHack &&
		((flags == VOL_ALL) ||  (flags == VOL_USED_SINCE_INIT))) {

		{
			register int j; 
			for(j=0; j<Nvolumes; j++)  {
				if(Volids[j].connected) {
					sm_errno=esmNOERROR;
					e = sm_ShutdownServer(VOL_BY_VOLID, Volids[j].volid, 0);
					if ((e == esmFAILURE) && (sm_errno != esmNOTCONNECTED)
						&& (sm_errno != esmSERVERDIED)) {
						HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_SHUTDOWNSERVER_ERR);
					}
				}
			}
		}
		
	} else {
		sm_errno=esmNOERROR;
		e = sm_ShutdownServer(flags, *volidlist, 0);
		checkNumBufs(__LINE__, __FILE__, FALSE);
		if ((e == esmFAILURE) && (sm_errno != esmNOTCONNECTED)
			&& (sm_errno != esmSERVERDIED)) {
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_SHUTDOWNSERVER_ERR);
		}
	}

	/* we abort iff not prepared */
	(void) abort_tx(FALSE);
	/* We cannot seek back and redo this tx (because
	 * we would be in an infinite loop of doing
	 * the (partial) tx, and crashing the server)
	 * so we do NOT do a checkStatus()!
	 */
	restart_starts_here(); /* to be safe */
	return esmNOERROR;
}

/* 
 * TODO: remove??????
 */
int
connect_command(ELIPSES)
{
	char *arg = next_commandarg(Whitespace);
	int number;

	if(arg) {
		if(getnum(arg, &number) == esmFAILURE) {
			return esmFAILURE;
		}
	} else 
		number = Numretries;

	ignore_restofline();

	return volumeStatus(FALSE, number, 4 /* sleep time */);
}

int
configfile_command(ELIPSES)
{
	int		e;
	char	*errorMsg;
	int 	argumentCount;
    char	command[COMMANDSIZE];
    char	opStr1[STRINGSIZE];
    char	opStr2[STRINGSIZE];

	argumentCount = sscanf(InputLine, "%s%d%d", command, opStr1, opStr2);

	if (argumentCount < 2) {
		if(interactive) {
			HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
			INFO "Try \"configfile filename [progname]\".\n" ENDINFO
		} else {
			HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, SYNTAXERROR);
		}
		return esmFAILURE;
	} else if (argumentCount == 2)
		e = sm_ReadConfigFile(opStr1, argv0, &errorMsg);
	else if (argumentCount >= 3)
		e = sm_ReadConfigFile(opStr1, opStr2, &errorMsg);

	if (e != esmNOERROR) {
		INFO "%s\n", errorMsg 	ENDINFO
		HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, NOINFO);
	}
	return e;
}

static VOLID *volids = NULL;

void
free_serverlist ()
{
	if(volids != NULL) {
		free((char *)volids);
		volids = NULL;
	}
}
int
exit_command(ELIPSES)
{
	int e, i;

	ignore_restofline();
	(void) _abort_tx(FALSE, FALSE); /* don't try reconnect, don't begin anew */
	close_script();
	freeVar();
	freePatternBuf();
	fclose(OVFPtr);

	closebg();
	dismount();

	shut_sm();
	free_serverlist();

	checkMalloc("exit", TRUE);

	checkNumBufs(__LINE__, __FILE__, TRUE);
	sm_PrintMessageStats(stdout, &MsgStats);
	checkNumBufs(__LINE__, __FILE__, FALSE);

	if (InvalidCount == 0) {
		INFO "TEST COMPLETED\n" ENDINFO
	} else {
		INFO "%d VALIDATION ERRORS\n", InvalidCount ENDINFO
		exit(EXIT_VALIDATIONERR);
	}
	if (reportableProblems != 0) {
		INFO "%d NON-validation ERRORS\n", reportableProblems ENDINFO
		exit(EXIT_INTERNAL);
	}
	exit(EXIT_SUCCESS);
	/* NOTREACHED */
	return esmNOERROR;
}

int
flush_command(ELIPSES)
{
	ignore_restofline();
	/* TODO */
	printf("Flushing buffers\n");
	checkNumBufs(__LINE__, __FILE__, TRUE);
	sm_FlushBuffers(FALSE);
	checkNumBufs(__LINE__, __FILE__, FALSE);
	return esmNOERROR;
}
int
intarg_command(int cmd)
{
#ifndef BASELINE
	FLAGS flags = VOL_USED_IN_TRANSACTION;
#endif
	char *arg = next_commandarg(Whitespace);
	int	 number, e = esmNOERROR;


	if(getnum(arg, &number) == esmFAILURE) {
		return esmFAILURE;
	}
	ignore_restofline();

	switch(cmd) {
	case WORD_SLEEP:
		sleep(number);
		break;

	case WORD_CHECKPOINT:
		checkNumBufs(__LINE__, __FILE__, TRUE);
		if ((e = sm_TakeCheckpoint(flags, 0, number)) != esmNOERROR)	{
			HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_CHECK_POINT_ERR);
		}
		checkNumBufs(__LINE__, __FILE__, FALSE);
		break;

	case WORD_FREQUENCY:
		printf("new checkpoint freq is %d records\n", number);
		checkNumBufs(__LINE__, __FILE__, TRUE);
		if ((e = sm_ChangeCheckpointFrequency(flags, 0, number)) != esmNOERROR) {
			HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_CHANGE_CKPT_FREQ_ERR);
		}
		checkNumBufs(__LINE__, __FILE__, FALSE);
		break;
	}
	return e;
}

int
generic_command(int cmd, unsigned int mask)
{
	/* objNum  location  numOfBytes [howMany] */

    USERDESC	*userDesc;
	int	objNum=0, location=0, amount=0, howMany=1, e=esmNOERROR, err=0, i;
	ObjInfo 	*objinfo, *parent=NULL, *child=NULL;
    OBJHDR		objHdr;
	char 		*arg, *data;
	int			toBeCreated;
	int			objectsAffected=0, saveHowMany;
	
	if(mask & NEED_OBJNUM) {
		arg = next_commandarg(Whitespace);
		if(getnum(arg, &objNum) == esmFAILURE) {
bad:
			sm_UnsetAbortIfError();
			HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
			return esmFAILURE;
		}
	} 
	if(mask & NEED_LOCATION) {
		arg = next_commandarg(Whitespace);
		if(getnum(arg, &location) == esmFAILURE) {
			goto bad;
		}
	}
	if(mask & NEED_AMOUNT) {
		arg = next_commandarg(Whitespace);
		if(getnum(arg, &amount) == esmFAILURE) {
			goto bad;
		}
	}
	if(mask & OPT_HOWMANY) {
		arg = next_commandarg(Whitespace);
		if(arg) {
			if(getnum(arg, &howMany) == esmFAILURE) {
				goto bad;
			}
		} else {
			howMany = 1;
		}
	}
	ignore_restofline();

	if(cmd == WORD_CREATE) {
		objNum = objUACount;
		toBeCreated = howMany;
	} else if (cmd == WORD_VERSION) {
		toBeCreated = howMany;
	} else {
		toBeCreated = 0;
	}
DEBUGINFO
"generic: cmd %d objNum %d howMany %d location %d amount %d objUACount %d, toBeCreated %d\n",
	cmd, objNum, howMany, location, amount, objUACount, toBeCreated
ENDINFO

	/*
	 *	put the objNum into variable 'current',
	 *	this is done for all object-related operations.
	 */
	sprintf(currentVar->itstring, "%d", objNum);

	if(objNum >= objUACount + toBeCreated) {
		HANDLE_ERROR_AND_CHECK(CAUSE_USER, TREAT_NONFATAL, BAD_OBJ_NUM);
		sm_UnsetAbortIfError();
		return esmFAILURE;
	}
	/* howMany represents the number left to do */
	for (saveHowMany = howMany, i=objNum; i < objUACount + toBeCreated; i++) {
		if( howMany-- <= 0)
			break;

		objinfo = &UA[i];

		if(cmd != WORD_CREATE) {
			if (isDestroyed(objinfo, CAUSE_NONE)) {
				continue;
			}
		}
		objectsAffected++;

		switch(cmd) {
		case WORD_APPEND:
			data = formPattern("A", i, amount);
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_AppendToObject(bufGroup, &objinfo->objID, amount, data);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_APPEND_OBJ_ERR;
			break;

		case WORD_CREATE: {
			FID *fid;
			fid = get_fid(); /* may be first reference to this file */
			if(fid == NULL) {
				sm_UnsetAbortIfError();
				return esmFAILURE;
			}

			if(interactive) {
				INFO
				"creating obj #%d\n", i
				ENDINFO
			}
			data = formPattern("C", i, amount);
			objHdr.tag = (TWO)i;
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_CreateObject(bufGroup, fid,
					NEAR_LAST_PHYSICAL,
					NULL, &objHdr, amount, data, &(UA[i].objID));
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_CREATE_OBJ_ERROR;
			}
			break;

		case WORD_DESTROY:
			if(interactive) {
				INFO
				"destroying obj #%d\n", i
				ENDINFO
			}
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_DestroyObject(bufGroup, &objinfo->objID );
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_DESTROY_OBJ_ERR;
			break;

		case WORD_READ:
			/* fall through: gets done below */
			err = esmNOERROR;
			break;

		case WORD_VERSION: 

			sprintf(currentVar->itstring, "%d", objUACount);

			if(interactive) {
				INFO
				"freezing obj #%d\n", i
				ENDINFO
			}
			parent = objinfo;
			child = &UA[objUACount];

			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_FreezeVersion(bufGroup, &parent->objID);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			if( e != esmNOERROR) {
				err = SM_FREEZE_VERSION_ERR;
				break;
			}
			{
				/*
				 * the parent obj should be marked 'frozen' and 'version' in 
				 * its UA entry
				 * Read into temp objHdr.
				 */
				checkNumBufs(__LINE__, __FILE__, TRUE);
				e = sm_ReadObjectHeader(bufGroup, &parent->objID, &objHdr);
				checkNumBufs(__LINE__, __FILE__, FALSE);

				if(e != esmNOERROR) {
					err = SM_READ_OBJ_HEADER_ERR;
					break;
				}
				if(parent->objHdr.properties & (P_VERSIONED | P_FROZEN) 
					!= (P_VERSIONED | P_FROZEN)) {
					HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
				/*
				 * ok- update the objinfo 
				 */
				parent->objHdr.properties |=  (P_VERSIONED | P_FROZEN);
			}
			if(interactive) {
				INFO
					"creating version of obj #%d\n", i
				ENDINFO
			}
				
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_CreateVersion(bufGroup, NEAR_OBJ, &parent->objID,
					/* near hint */&parent->objID, &child->objID);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_CREATE_VERSION_ERR;

			break;

		case WORD_INSERT:
			if(interactive) {
				INFO
				"inserting %d bytes in obj #%d\n", amount, i
				ENDINFO
			}
			data = formPattern("I", i, amount);
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_InsertInObject(bufGroup, &objinfo->objID, location, amount, data);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_INSERT_OBJ_ERR;
			break;
		case WORD_DELETE:
			if(interactive) {
				INFO
				"deleting %d bytes from obj #%d\n", amount, i
				ENDINFO
			}
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_DeleteFromObject(bufGroup, &objinfo->objID, location, amount);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_DELETE_FROM_OBJ_ERR;
			break;
		case WORD_WRITE:
			if(interactive) {
				INFO
				"writing %d bytes at location %d in  obj #%d\n", amount, 
					location, i
				ENDINFO
			}
			e = sm_ReadObject(bufGroup, &objinfo->objID, 0, 
				location + amount, &userDesc);
			if (e  != esmNOERROR) {
				err = SM_READ_OBJ_ERR;
				break;
			}

			data = formPattern( "W", i, amount);
			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_WriteObject(bufGroup, location, amount, data, 
				userDesc, FALSE); /* don't free */
			checkNumBufs(__LINE__, __FILE__, FALSE);
			err = SM_WRITE_OBJ_ERR;

			/* now free the user descr */
			if(sm_ReleaseObject(userDesc)) {
				err = SM_RELEASE_OBJ_ERR;
				break;
			}

			break;
		}

		if(e != esmNOERROR) {
			HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, err);
			break;
		}
		mark_used(objinfo->objID.diskAddr.volid);

		switch(cmd) {
		case WORD_VERSION: 
			/*
			 * enter a new entry for the versioned object in UA
			 */
			fillUA(UA, objUACount, parent->checkSum, parent->index);

			objUACount++; /* we have to do this */
			toBeCreated--; /* once to cancel objUACount's growth */
			toBeCreated--; /* once again to account for the # left to do */

			{
				/* set the child's tag */
				objHdr.tag = child->index;

				checkNumBufs(__LINE__, __FILE__, TRUE);
				e = sm_SetObjectHeader(bufGroup, &child->objID, &objHdr);
				checkNumBufs(__LINE__, __FILE__, FALSE);
				err = SM_SET_OBJHEADER_ERR;
			}
			{
				if(interactive) {
					DEBUGINFO
					"checking child's header...\n"
					ENDINFO
					INFO
					"Parent object #%d, child #%d\n", parent->index, child->index
					ENDINFO
				}
				/*
				 * read back child's header.
				 * the child should be marked 'version'
				 * its size should match parent's size
				 * its tag should be its o# 
				 */
				checkNumBufs(__LINE__, __FILE__, TRUE);
				e = sm_ReadObjectHeader(bufGroup, &child->objID, &child->objHdr);
				checkNumBufs(__LINE__, __FILE__, FALSE);

				if(e != esmNOERROR) {
					err = SM_READ_OBJ_HEADER_ERR;
					break;
				}
				if(child->objHdr.properties & (P_VERSIONED | P_FROZEN) != P_VERSIONED) {
					HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
				if( child->objHdr.size != parent->objHdr.size ) {
					HANDLE_ERROR_AND_CHECK(CAUSE_VALIDATION, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
				if( child->objHdr.tag != child->index) {
					HANDLE_ERROR_AND_CHECK(CAUSE_VALIDATION, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
			}
			break;

		case WORD_CREATE: {
			fillUA(UA, i, checkSum(data, amount), -1);

			/* read back the header; check the tag, properties, and size */
				checkNumBufs(__LINE__, __FILE__, TRUE);
				e = sm_ReadObjectHeader(bufGroup, &objinfo->objID, 
					&objinfo->objHdr);
				checkNumBufs(__LINE__, __FILE__, FALSE);
				if (e != esmNOERROR) {
					HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_READ_OBJ_HEADER_ERR);
					break;
				}
				if(objinfo->objHdr.tag != objinfo->index) {
					HANDLE_ERROR_AND_CHECK(CAUSE_VALIDATION, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
				if(objinfo->objHdr.size != amount) {
					HANDLE_ERROR_AND_CHECK(CAUSE_VALIDATION, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
				if((objinfo->objHdr.properties & (P_VERSIONED | P_FROZEN)) != 0) {
					HANDLE_ERROR_AND_CHECK(CAUSE_VALIDATION, TREAT_FATAL, 
						SM_READ_OBJ_HEADER_ERR);
				}
			}
			objUACount++;
			toBeCreated--;
			break;
		case WORD_DESTROY:
			objinfo->destroyed = TRUE;
			break;
		default:
			checkNumBufs(__LINE__, __FILE__, TRUE);
			if ((e = sm_ReadObject(bufGroup, &objinfo->objID, 0, -1, 
				&userDesc)) != esmNOERROR) {
				HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_READ_OBJ_ERR);
				break;
			}
			if(cmd == WORD_READ) {
				printf("OBJ-%d %1.*s, versioned:%d frozen:%d prop:%x size:%d\n",
					i, userDesc->objectSize, userDesc->basePtr,
				  	userDesc->userFlags & P_VERSIONED,
				 	userDesc->userFlags & P_FROZEN,
				 	userDesc->userFlags, userDesc->objectSize);
			} else {
				objinfo->checkSum = checkSum(userDesc->basePtr, (int)(userDesc->objectSize));
			}
			e = sm_ReleaseObject(userDesc);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			if (e  != esmNOERROR) {
				HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_RELEASE_OBJ_ERR);
				break;
			}

			checkNumBufs(__LINE__, __FILE__, TRUE);
			e = sm_ReadObjectHeader(bufGroup, 
				&objinfo->objID, &objinfo->objHdr);
			checkNumBufs(__LINE__, __FILE__, FALSE);
			if (e != esmNOERROR) {
				HANDLE_ERROR_AND_CHECK(CAUSE_SM, TREAT_NONFATAL, SM_READ_OBJ_HEADER_ERR);
			}
			break;
		}/* switch */
		DEBUGINFO
			"end for loop: i %d howMany %d objUACount %d toBeCreated %d",
				i, howMany, objUACount, toBeCreated
		ENDINFO
	} /* for */
	if(objectsAffected < saveHowMany) {
		INFO
			"WARNING: %d objects affected\n", objectsAffected
		ENDINFO
	}
	sm_UnsetAbortIfError();
	return e;
}

int
read_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | OPT_HOWMANY);
}
int
destroy_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | OPT_HOWMANY);
}
int
create_command(int cmd)
{
	return generic_command(cmd, NEED_AMOUNT | OPT_HOWMANY);
}
int
append_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | NEED_AMOUNT | OPT_HOWMANY);
}
int
write_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | NEED_LOCATION | NEED_AMOUNT | OPT_HOWMANY);
}
int
insert_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | NEED_LOCATION | NEED_AMOUNT | OPT_HOWMANY);
}
int
delete_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | NEED_LOCATION | NEED_AMOUNT | OPT_HOWMANY);
}
int
list_command(ELIPSES)
{
	listVar();
	return esmNOERROR;
}
int
print_command(ELIPSES)
{
	char *whichset = next_commandarg(Whitespace);

	if(whichset == NULL) {
		return esmNOERROR;
	}

	switch( word2cmd(whichset) ) {
	case WORD_OBJSIZELIMIT:
		print_objsizelimit();
		break;

	case WORD_RANDOP:
		print_randop();
		break;

	case WORD_SEED:
		print_seed();
		break;

	case WORD_VOLID:
		print_volid();
		break;

	case WORD_TRANSACTION:
		if(tid != 0) {
			INFO "In transaction %d.\n", tid ENDINFO
		} else {
			INFO "No active transaction.\n" ENDINFO
		}
		if(sm_errno != esmNOERROR) {
			INFO "sm_errno is %s\n", sm_ErrorId(sm_errno)
			ENDINFO
		}
		break;
	case WORD_UA:
		if(objUACount == 0) {
			INFO "No uncommitted objects.\n" ENDINFO
			break;
		}
		INFO "\nUNCOMMITTED OBJECTS:\n" ENDINFO
		printArray(UA, objUACount);
		break;
	case WORD_CA:
		if(objCACount == 0) {
			INFO "No committed objects.\n" ENDINFO
			break;
		}
		INFO "\nCOMMITTED OBJECTS:\n" ENDINFO
		printArray(CA, objCACount);
		break;

	case WORD_NONE:
		{
			char buffer[INPUTSIZE + 1];		
			if (getString(whichset, buffer, TRUE)) {
				INFO
					"%s = %s", whichset, buffer
				ENDINFO
			} else  {
				return esmFAILURE;
			}
		}
	}
	return esmNOERROR;
}

int
scan_command(ELIPSES)
{
	FID *fid;
	ignore_restofline();

	fid = get_fid();
	if(fid == NULL)
		return esmFAILURE;
	if (scanFile(fid, FALSE, TRUE, FALSE, UA, objUACount) == -1)
		checkStatus(TRUE);
	return esmNOERROR;
}
int
set_command(ELIPSES)
{
	BOOL	blank;
	int e;
	char *rest = next_commandarg("\0");
	char *var, *val;

	if(!rest) {
bad:
		INFO
			"Try: set variable value\n"
		ENDINFO
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}

	var = next_word(rest, Whitespace);

	if(!var) {
		INFO
			"Missing variable name. "
		ENDINFO
		goto bad;
	}
	/* set val to point to the rest of the line */
	val = var + (strlen(var)+1);

	while (isspace(*val)) val++;
	if(*val == '=') {
		val++;
		while (isspace(*val)) val++;
	}

	if(strlen(val) < 1)  {
		/* set to blank */
		blank = TRUE;
	} else blank = FALSE;

	switch( word2cmd(var) ) {
	case WORD_LOGLEVEL:
		if(blank) goto bad;
		return set_loglevel(val);
		break;

	case WORD_OBJSIZELIMIT:
		if(blank) goto bad;
		e = set_objsizelimit(val);
		break;

	case WORD_VOLID:
		if(blank) goto bad;
		e = set_volid(val);
		break;
		
	case WORD_SEED:
		if(blank) goto bad;
		e = set_seed(val);
		break;
	case WORD_RANDOP:
		if(blank) goto bad;
		e = set_randop(val);
		break;
	case WORD_ECHO:
		if(blank) goto bad;
		e = open_echofile(val);
		break;

	case WORD_NONE:
		setVar(var, val);
		e = esmNOERROR;
		break;

	default:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		e = esmFAILURE;
		break;
	} 
	return e;
}
static void
tryoption(char *x, void *y, SMDATATYPE t)
{
	int e;
	e = sm_GetClientOption(x, y);
	if (e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return ;
	}
	switch(t) {
	case SM_char:
		DEBUGINFO
			"option: %20.20s=0x%x\n", x, *(char *)y 
		ENDINFO
		break;

	case SM_string:
		DEBUGINFO
			"option: %20.20s=%s\n", x, *(char **)y 
		ENDINFO
		break;

	case SM_float:
		DEBUGINFO
			"option: %20.20s=%f\n", x, *(float *)y 
		ENDINFO
		break;
	case SM_double:
		DEBUGINFO
			"option: %20.20s=%f\n", x, *(double *)y 
		ENDINFO
		break;

	case SM_long:
		DEBUGINFO
			"option: %20.20s=%d (0x%x)\n", x, *(long *)y , *(long *)y
		ENDINFO
		break;
	case SM_short:
		DEBUGINFO
			"option: %20.20s=%d (0x%x)\n", x, *(short *)y , *(short *)y
		ENDINFO
		break;
	case SM_int:
		DEBUGINFO
			"option: %20.20s=%d (0x%x)\n", x, *(int *)y , *(int *)y
		ENDINFO
		break;
	default:
		HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, esmINTERNAL);
	}
}
int
clientoption_command(ELIPSES)
{
	/* clientoption_command optionname optionvalue */
	char 	*optionname =  next_commandarg(Whitespace);
	char 	*optionvalue;
	int 	e;

	if(optionname == NULL) {
bad:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		if(interactive) {
			INFO 
			"Try \"clientoption_command optionName optionValue\".\n" 
			ENDINFO
		}
		return esmFAILURE;
	}
	optionvalue	=  next_commandarg(Whitespace);
	if(optionvalue == NULL) {
		goto bad;
	}
	e = sm_SetClientOption(optionname, optionvalue, SM_string);
	if (e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}
	{
		int	integer;
		long longint;
		short shortint;
		float floatingpt;
		double doublefloat;
		char *strptr = NULL;
		char character;
		BOOL Boolean;

		tryoption("bufpages", &integer, SM_int);
		tryoption("locktimeout", &integer, SM_int);
		tryoption("userdesc", &integer, SM_int);
		tryoption("groups", &integer, SM_int);
		tryoption("traceflags", &integer, SM_int);

		tryoption("deallocpages", &Boolean, SM_Boolean);
		tryoption("lognewpages", &Boolean, SM_Boolean);

		tryoption("filelock", &strptr, SM_string);
		tryoption("pagelock", &strptr, SM_string);

		tryoption("mount", &strptr, SM_string);
	}
	return esmNOERROR;
}


int
__serverstats_command(BOOL reset COMMA ELIPSES)
{
	FLAGS flags = VOL_USED_IN_TRANSACTION;
    SERVERSTATS	*stats;  
	int			numservers = 0;
	char 		*arg =  next_commandarg(Whitespace);
	VOLID		*volidlist;
	int 		e = esmNOERROR;

	volidlist = get_serverlist(&flags);
	if(!volidlist) {
		INFO
			"No volumes!\n"
		ENDINFO
		return esmFAILURE;
	}
	ignore_restofline();

	checkNumBufs(__LINE__, __FILE__, TRUE);
	if(flags == VOL_BY_VOLID) {
		while( *volidlist != 0) {
			stats = NULL;
			e = esmNOERROR;
			e = sm_ServerStatistics(flags, *volidlist, &numservers, &stats, reset);
			if(e != esmNOERROR) break;
			printStats(reset, stats);
			free((char *)(stats));
			volidlist++;
			stats = NULL;
			numservers = 0; /* so we don't print the last one  a 2nd time */
		}
	} else {
		stats = NULL;
		e = sm_ServerStatistics(flags, 0, &numservers, &stats, reset);
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	if(e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_SERVERSTAT_ERR);
		if(stats)
			free((char *)(stats));
		return esmFAILURE;
	}
	for( e = numservers-1; e >= 0; e--) {
		printStats(reset, stats + e);
	}
	if(numservers > 0) {
		if(stats ==NULL) {
			/* should not happen */
			HANDLE_ERROR(CAUSE_VALIDATION, TREAT_NONFATAL, SM_SERVERSTAT_ERR);
		} else {
			free((char *)(stats));
		}
	} else {
		if(stats !=NULL) {
			/* should not happen */
			HANDLE_ERROR(CAUSE_VALIDATION, TREAT_NONFATAL, SM_SERVERSTAT_ERR);
			/* possible memory leak but don't free 'cause we don't know */
		} 
	}
	return esmNOERROR;
}
int
serverstats_command(ELIPSES)
{
	return __serverstats_command(FALSE);
}
int
reset_command(ELIPSES) /* reset [serverstats or rusage] args */
{
	char 		*arg =  next_commandarg(Whitespace);

	if(arg) switch( word2cmd(arg) ) {
	case WORD_SERVER:
		__serverstats_command(TRUE);
		break;
	/* rusage isn't implemented yet */
	case WORD_NONE:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	} else{
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}
	return esmNOERROR;
}

int
validate_command(ELIPSES)
{
	/* validate ua|ca [objNum] [howMany] */
	char 	*arg = next_commandarg(Whitespace);
	ObjInfo *which;
	int     count, e, howMany, objNum;

	switch(word2cmd(arg)) {
		case WORD_UA:
			count = objUACount;
			which = UA;
			break;
		case WORD_CA:
			count = objCACount;
			which = CA;
			break;
		default:
			INFO
				"Try: \"validate ua\" or \"validate ca\"\n"
			ENDINFO
			HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
			return esmFAILURE;
	}
	arg = next_commandarg(Whitespace);
	if(arg) {
		if(getnum(arg, &objNum) == esmFAILURE) {
			return esmFAILURE;
		}

		if (objNum >= count) {
			HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, INVALID_OBJ_NUM);
			return esmFAILURE;
		}

		/* now get # objects */
		arg = next_commandarg(Whitespace);
		if(arg) {
			if(getnum(arg, &howMany) == esmFAILURE) {
				return esmFAILURE;
			}
			if( objNum + howMany > count)
				return esmFAILURE;
		} else {
			/* objNum, 1 */
			howMany = 1;
		}
	} else  {
		/* no objNum ==> all */
		objNum = 0; 
		howMany = count;
	}

	sprintf(currentVar->itstring, "%d", objNum);

	e = validate(bufGroup, howMany, objNum, which);
	return e;
}

int
version_command(int cmd)
{
	return generic_command(cmd, NEED_OBJNUM | OPT_HOWMANY );
}
static void
copy_volids( FLAGS flags, VOLID *list )
{
	register int i, j;

	for(j = 0, i=0; i<Nvolumes; i++) {
		if( (flags == VOL_ALL)  ||
			(Volids[i].lastused == tid) ) {
			list[j++] = Volids[i].volid;
		}
	}
}



VOLID *
get_serverlist (FLAGS *flags)
{
	/* parses [[o# o# ELIPSES v# v#] | all | used | trans ] 
	 * returns flags & list of volids apropos to the input
	 * default is whatever is passed in through flags
	 */ 
	static int numvolids = 0;
	register int i;
	char 	*arg;

	if(numvolids < objUACount)
		numvolids = objUACount; /* simplistic but oh well */
	if(numvolids < Nvolumes)
		numvolids = Nvolumes;

	if(volids != NULL)
		free((char *)volids);
	volids = (VOLID *)calloc(numvolids+1, sizeof(VOLID));

	i = 0;
	while( (arg = next_commandarg(Whitespace)) != NULL) {
		switch( word2cmd(arg) ) {
		case WORD_USED:
			*flags = VOL_USED_SINCE_INIT;
			/*
			copy_volids( *flags, volids );
			*/
			goto done;

		case WORD_ALL:
			*flags = VOL_ALL;
			/*
			copy_volids( *flags, volids );
			*/
			goto done;

		case WORD_TRANSACTION:
			*flags = VOL_USED_IN_TRANSACTION;
			goto done;

		default: {
				int number;
				char *numstr;
				BOOL by_volume = FALSE;

				if( *arg == 'o' ) {
					numstr = arg+1;
					by_volume = FALSE;
				} else if(*arg == 'v') {
					numstr = arg+1;
					by_volume = TRUE;
				} else {
					numstr = arg;
					by_volume = TRUE;
				}
				if(getnum(numstr, &number)==esmFAILURE)
					goto bad;
				*flags = VOL_BY_VOLID;

				if(by_volume) {
					if (find_volid_index(number) == -1) {
						goto bad;
					}
					volids[i++] = number;
				} else {
					if (number >= objCACount) {
						goto bad;
					}
					volids[i++] = UA[number].objID.diskAddr.volid;
				}
			} /* default */
			break;
		} /* switch */
	}/* while */
done:
	DEBUGINFO
		"i=%d\n", i
	ENDINFO
	return volids;
bad:
	HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
	return NULL;
}

int 
servervar_command(ELIPSES)
{
	/* tell a server or servers which way to vote by default 
	 * (of course, server can override this
	 */
	/* syntax: servervar var value [obj# | all | used | transaction ]  */
	/* syntax: var: vote | crashloc
	 *  	value:  commit|abort|yes|no| # 
	 */
	extern int sm_DebugSetVar(FLAGS, VOLID, int, int);
	char 	*arg = next_commandarg(Whitespace);
	FLAGS	flags = 0;
	VOLID	*volidlist;
	int		var, val, e;

	if(arg == NULL) {
bad:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}
	if(strcmp(arg, "vote") == 0) {
		var = 1;
	} else if(strcmp(arg,"crashloc") == 0) {
		var = 2;
	} else {
		if(getnum(arg, &var) == esmFAILURE) {
			goto bad;
		}
	}
	arg = next_commandarg(Whitespace);
	if(arg == NULL) {
		goto bad;
	}
	if(strcmp(arg, "abort") == 0) {
		val = NOVOTE;
	} else if(strcmp(arg,"commit") == 0) {
		val = YESVOTE;
	} else if(strcmp(arg,"yes") == 0) {
		val = YESVOTE;
	} else if(strcmp(arg,"no") == 0) {
		val = NOVOTE;
	} else {
		if(getnum(arg, &val) == esmFAILURE) {
			goto bad;
		}
	}
	volidlist = get_serverlist(&flags);
	if(!volidlist) {
		INFO
			"No volumes!\n"
		ENDINFO
		return esmFAILURE;
	}
	checkNumBufs(__LINE__, __FILE__, TRUE);
	if(flags == VOL_BY_VOLID) {
		while( *volidlist != 0) {
			e = sm_DebugSetVar(flags, *volidlist, var, val);
			if(e != esmNOERROR) {
				fprintf(stderr, 
				"servervar  %d=%d for volume %d, flags 0x%x NOT IMPLEMENTED\n",
				var, val, *volidlist, flags);
			}
			volidlist++;

			if(e != esmNOERROR)
				break;
		}
	} else {
		e = sm_DebugSetVar(flags, 0, var, val);
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	return e;
}

int 
prepare_command(ELIPSES)
{
	ignore_restofline();
	return prepare_tx();
}

int enter2PC_command(ELIPSES)
{
	char *arg = next_commandarg(Whitespace);
	ignore_restofline();
	return enter2pc(arg);
}

int recover2PC_command(ELIPSES)
{
	return command2pc(WORD_RECOVER);
}

int continue2PC_command(ELIPSES)
{
	return command2pc(WORD_CONTINUE);
}

int
sync_command()
{
	int e;
	checkNumBufs(__LINE__, __FILE__, TRUE);
	if( e= sm_SyncTransaction(tid) ) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, NOINFO);
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	return e;
}

int
addvol_command()
{
	/* syntax: addvolume option formatinfo 
		[obj# | all | used | transaction ]  
		formatinfo must have no whitespace in it
	*/

	char 	*opt = next_commandarg(Whitespace);
	char 	*fmt;
	FLAGS	flags = VOL_USED_SINCE_INIT;
	VOLID	*volidlist;
	int		e;

	if(opt == NULL) {
bad:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}
	fmt = next_commandarg(Whitespace);
	if(fmt == NULL) {
		goto bad;
	}
	volidlist = get_serverlist(&flags);
	if(!volidlist) {
		INFO
			"No volumes!\n"
		ENDINFO
		return esmFAILURE;
	}
	if(flags == VOL_BY_VOLID) {
		while( *volidlist != 0) {
			e = sm_AddServerVolume(flags, *volidlist, opt, fmt);
			if(e != esmNOERROR) {
				break;/*while*/
			}
			volidlist++;
		} /* while */
	} else {
		e = sm_AddServerVolume(flags, 0, opt, fmt);
	}
	if(e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, NOINFO);
	}
	return e;
}

int
rmvol_command()
{
	/* syntax: rmvolume volid [obj# | all | used | transaction ]  */
	char 	*vol = next_commandarg(Whitespace);
	FLAGS	flags = VOL_USED_SINCE_INIT;
	VOLID	*volidlist;
	int		e;

	if(vol == NULL) {
bad:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}
	volidlist = get_serverlist(&flags);
	if(!volidlist) {
		INFO
			"No volumes!\n"
		ENDINFO
		return esmFAILURE;
	}
	if(flags == VOL_BY_VOLID) {
		while( *volidlist != 0) {
			e = sm_RemoveServerVolume(flags, *volidlist, atoi(vol));
			if(e != esmNOERROR) {
				break;/* while loop */
			}
			volidlist++;
		} /* while */
	} else {
		e = sm_RemoveServerVolume(flags, 0, atoi(vol));
	}
	if(e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, NOINFO);
	}
	return e;
}
