/*
 * $RCSfile: session.c,v $
 * $Revision: 1.30 $
 * $Date: 1993/05/03 14:19:16 $
 */

#include "et.h"
#include "etError.h"
#include "func.h"
#include "command_funcs.h"

static 	char	ovfName[STRINGSIZE + 5];

static int debug_vote = 0;
static int debug_crashloc = 0;

BOOL tx_is_prepared;
#define UNKNOWN_TID ((TID)(-1)) /* prepared & must recover */
static	TID	 prepared_tx = NULL_TID; /* none */

static COORD_HANDLE handle = {
	-1,
	"initialvaluehost",
	"initialvalueserver"
};
static GTID			gtid;
static FILE			*logfile = NULL;

static int
delete_log(char *filename)
{
	if(filename==NULL)
		filename = "et.log";
	if(logfile != NULL) {
		DEBUGINFO
			"Remove log file \"%s\"\n", filename
		ENDINFO
		if(unlink(filename)<0) {
			INFO
				"Could not remove ET log file \"%s\"; errno %d\n", 
				filename, errno
			ENDINFO
			return esmFAILURE;
		}
		logfile = NULL;
	}
	return esmNOERROR;
}

static int
open_log(char *filename, char *which)
{
	if(filename==NULL)
		filename = "et.log";
	if(logfile != NULL) {
		fsync(fileno(logfile));
		fclose(logfile);
		logfile = NULL;
	}
	errno = 0;
	if(logfile == NULL) {
		DEBUGINFO
			"Opening log file\"%s\" for %s\n", filename,which
		ENDINFO
		logfile = fopen(filename, which);
		if(logfile==NULL) {
			return esmFAILURE;
		}
		setbuf(logfile, 0);
	}
	return esmNOERROR;
}


BOOL
connection_error( int e)
{
	switch(e) {
	case	esmLOGDISABLED:
	case	esmTRANSDISABLED:
	case	esmSERVERDIED:
	case	ENETDOWN:
	case	ENETUNREACH:
	case	ENETRESET:
	case	ECONNABORTED:
	case	ECONNRESET:
	case	EISCONN:
	case	ENOTCONN:
	case	ESHUTDOWN:
	case	ETIMEDOUT:
	case	ECONNREFUSED:
		return TRUE;
	default:
		return FALSE;
	}
}


void
clr_debugvars()
{
	debug_vote = 0;
	debug_crashloc = 0;
}

void
get_debugvars()
{
	char *c;
	c = getenv("ECRASHLOC");
	if(c) {
		debug_crashloc = atoi(c);
	}
	c = getenv("EVOTE");
	if(c) {
		debug_vote = atoi(c);
	}
}

void
set_debugvars(VOLID v)
{
	int e;

	checkNumBufs(__LINE__, __FILE__, TRUE);
	if(debug_vote != 0) {
		DEBUGINFO
			"Setting vote var to %d\n", debug_vote
		ENDINFO
		e = sm_DebugSetVar(VOL_BY_VOLID, v, 1, debug_vote);
		if((e == esmNOERROR) && interactive) {
			INFO
				"Set vote var to %d\n", debug_vote
			ENDINFO
		}
		DEBUGINFO
			"result: %d", e
		ENDINFO
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	checkNumBufs(__LINE__, __FILE__, TRUE);
	if(debug_crashloc != 0) {
		DEBUGINFO
			"Setting crashloc var to %d\n", debug_crashloc
		ENDINFO
		e = sm_DebugSetVar(VOL_BY_VOLID, v, 2, debug_crashloc);
		if((e == esmNOERROR) && interactive) {
			INFO
				"Set crashloc var to %d\n", debug_crashloc
			ENDINFO
		}
		DEBUGINFO
			"result: %d", e
		ENDINFO
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	clr_debugvars();
}
void
closebg()
{
	int e;

	sm_errno = 0;
	if(bufGroup != NO_BUFGROUP) {
		checkNumBufs(__LINE__, __FILE__, TRUE);
		e = sm_CloseBufferGroup(bufGroup);
		checkNumBufs(__LINE__, __FILE__, FALSE);

		if(e != esmNOERROR) {
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, CANTCLOSEBG);
		}
		bufGroup = NO_BUFGROUP;
	}
}
static void
newBufGroup( int bufGroupPages )
{
	extern int bufGroup;
	int 	e;

	closebg();
	if(bufGroupPages == 0) {
		/* 
		 * get #buf group pages bufpages from config file
		 */
		if( sm_GetClientOption("bufpages", &bufGroupPages) != esmNOERROR) {
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_MOUNTVOL_ERR);
		}
		bufGroupPages /= 2; /* use half */
	}
	if(bufGroupPages <= MINBUFGROUPPAGES) {
		INFO 
			"\nET will not open a buffer group with fewer than %d pages\n",
				MINBUFGROUPPAGES
		ENDINFO
		HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, CANTOPENBG);
	}
	checkNumBufs(__LINE__, __FILE__, TRUE);
	if (sm_OpenBufferGroup(bufGroupPages, BF_LRU, &bufGroup, NOFLAGS)) {
		HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, CANTOPENBG);
	}
	checkNumBufs(__LINE__, __FILE__, FALSE);
	INFO 
		"\nOpened bufgroup %d with %d pages\n", bufGroup, bufGroupPages
	ENDINFO
}

BOOL
istemp(VOLID volid)
{
	int index;
	index = find_volid_index(volid);
	if(index >= 0) {
		return Volids[index].temp;
	}
	HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, NOINFO);
	return FALSE; /* not reached */
}
void
gettemp(VOLID volid, int index, int retries)
{
	int props, e;
	BOOL result;

	set_retry();
	checkNumBufs(__LINE__, __FILE__, TRUE);
	e = sm_VolumeProperties(volid, &props);
	checkNumBufs(__LINE__, __FILE__, FALSE);

	if(e  == esmNOERROR) {
		clr_retry();
		if(props & VOLPROP_TEMP) {
			result = TRUE;
		} else {
			result = FALSE;
		}
	} else {
		/* could have crashed after mount */
		HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_FATAL, SM_CONNTO_SERVER_ERR, volid, retries);
	}
	if(index >= 0) {
		if(Volids[index].volid != volid) {
			HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, NOINFO);
		}
	} else {
		index = find_volid_index(volid);
	}
	Volids[index].temp = result;
}
BOOL multiportHack = FALSE;

static void
newport(VOLID volid) 
{
	static char optionvalue[200];
	static BOOL first=TRUE;
	static int 	firstport, nports, curport;
	static char	*host = "localhost";


	if(first) {
		char *ports = getenv("EPORTS"); /* first port - # ports */
		char *h;

		multiportHack = TRUE;

		h = getenv("EHOSTNAME");
		if(h != NULL) {
			host = h;
		}
		
		if(ports) {
			if(Nvolumes > 1) {
				fprintf(stderr, 
			"Can't use EPORTS/EHOSTNAME with multiple volumes; ignored\n");
				first = FALSE;
				return;
			}
			char *f;
			f = strtok(ports, " ;:,");
			if(f) {
				firstport = atoi(f);
				if(firstport <= 0) {
					firstport = -1;
				}
				f = strtok(NULL, " ;:,");
				if(f)  {
					nports = atoi(f);
				} else {
					nports = 1;
				}
			} else {
				firstport = -1;
				/* use what's in config file only */
			}
		} else {
			firstport = -1;
		}
		curport = firstport;
		first = FALSE;
	}
	if(curport > 0) {
		/* create a string: volid host@port */
		sprintf(optionvalue, "%d %d@%s", volid, curport, host);

		if(sm_SetClientOption("mount",optionvalue, SM_string)!=esmNOERROR) {
			HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, NOVOLUME);
		}
		if(interactive) {
			INFO
				"Trying port %s\n", optionvalue
			ENDINFO
		} else {
			DEBUGINFO
				"Trying port %s\n", optionvalue
			ENDINFO
		}
		if(++curport > (firstport + nports -1)) 
			curport = firstport;
	} else {
		sleep(6);
	}
}

BOOL
connect2server(VOLID volid, int tries)
{
	BOOL connected;
	int e;

	for(connected = FALSE; (tries > 0) && !connected; tries--) {
		sm_errno = 0;
		checkNumBufs(__LINE__, __FILE__, TRUE);
		e = sm_MountVolume(volid);
		checkNumBufs(__LINE__, __FILE__, FALSE);

		if(e == esmNOERROR) {
			connected = TRUE;
		} else {
			switch(sm_errno) {
			case esmNOERROR:
				break;

			case esmBADVOLID:
			case esmDUPVOLID:
			case esmMOUNTTABLEFULL:
			case esmBADVOLHEADER:
			case esmNOSUCHVOLUME:
				/* problem mounting - quit trying */
				tries = 0;
				HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_CONNTO_SERVER_ERR);
				break;
				
			case esmDISKMOUNTED:
				/* already mounted - no problem at all */
				connected = TRUE;
				break;

			default:
				HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_CONNTO_SERVER_ERR);
				break;

			}
			if(tries>0) {
				newport(volid);
				DEBUGINFO
					"tries left=%d\n",tries
				ENDINFO
			}
		}
	}
	if(connected) {
		gettemp(volid, -1, 1);
		clear_connect_error();
	} else {
		INFO
			"Cannot connect to server for volume %d\n", volid
		ENDINFO
	}

	return connected;
}

int
volumeStatus(BOOL destroyData, int tries, int seconds)
{   
	register int i, e, errtotal = 0;
	int	nconnected;

	(void) begin_tx(FALSE,FALSE);
	for(i=0; i<Nvolumes; i++) {
		Volids[i].connected = FALSE;
	}
	for (nconnected = 0; (tries > 0) && (nconnected<Nvolumes); tries--) {
		for(nconnected = 0, i=0; i<Nvolumes; i++) {
			if(Volids[i].connected) {
				nconnected++;
			} else {
				sm_errno=esmNOERROR;
				e = esmNOERROR;
				if(connect2server(Volids[i].volid, 3))  {
					Volids[i].connected = TRUE;
					nconnected++;
					if(interactive) {
						INFO
							"Found server for volume %d %s\n",
							Volids[i].volid , Volids[i].temp?"(temporary volume)":""
						ENDINFO
					}
					set_debugvars(Volids[i].volid);
					(void) get_fid_of(Volids[i].volid);
				} else {
					if(sm_errno == esmTRANSABORTED) {
						/* TODO: had better have one outstanding */
						(void) abort_tx(FALSE);
					}
					if(!connection_error(sm_errno)) {
						errtotal++;
						HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, SM_INIT_ERR);
					}
				}
			}
		}
		sleep(seconds);
	}
	if(errtotal > 0) {
		return esmFAILURE;
	} else {
		return esmNOERROR;
	}
}
/****************************************************************
 *
 *	processes the session command line 
 *	It searches for flags
 *	(-r -v, -f -F, -b) in it, and assigns the given values to cor-
 *	responding global variables: volName, rootEntryName and
 *	bufGroupPages. The "rootEntryName" uniquely identifies
 *	a datafile, and rootEntryName.ovf will be the ovf name.
 *  -F is like -f but it also means : blow away the existing data file.
 *
 ***************************************************************/
static	BOOL	sm_initialized = FALSE;
void
shut_sm()
{
	if(sm_initialized) {
		closebg();
		checkMalloc("before sm_ShutDown", FALSE);
		if(sm_ShutDown() != esmNOERROR) {
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_INIT_ERR);
		}
		checkMalloc("after sm_ShutDown", TRUE);
	}
	bufGroup = NO_BUFGROUP; 
	sm_initialized = FALSE;
}
void 
initializeSM(BOOL reinit)
{
	/* if we have initialized the SM code before, we don't have
	 * to do it again unless the user tells us to do so with the
	 * '-r' flag
	 */
	static int resbufs = 0;

	if(sm_initialized && (reinit == FALSE))
		return;

	/* 
	 * initialize or reinitialize the sm
	 */
    if(sm_initialized) {
		char *errorMsg;

		compareNumBufs(__LINE__,__FILE__, resbufs, "shut_sm");
		shut_sm();

		/* reread the default config file(s) */
		/* to get the mount info */
		if(sm_ReadConfigFile(NULL, argv0, &errorMsg) != esmNOERROR) {
			INFO
				"Missing or bad config file: %s\n", errorMsg
			ENDINFO
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_INIT_ERR);
		}
		sm_initialized = FALSE;
	}

	checkMalloc("before sm_Initialize", FALSE);
	if( sm_Initialize()!= esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_INIT_ERR);
	}
	sm_initialized = TRUE;
	resbufs = getNumBufs();
	checkMalloc("after sm_Initialize", TRUE);

	if(tid != NULL_TID) {
		if(interactive) {
			INFO
				"Warning: A transaction is %s",
				tx_is_prepared?"prepared":"active, will be aborted"
			ENDINFO
		}
	}
}

int
session_command(ELIPSES)
{
	BOOL	destroyDataFiles = FALSE, continueSession = FALSE;
	BOOL	reinit = FALSE; 
	char 	*arg;
	int 	bufGroupPages = 0;
	int		SaveNvolumes = Nvolumes;

	if( (!tx_is_prepared) && (tid != NULL_TID)) {
		abort_tx( FALSE ); /* so that temp files are gone */
	}
	/* 
	 * New session: start over with volume list.
	 */
	Nvolumes = 0;

    /*
     *	if no rootEntryName is given, "default" is used 
     */
	strcpy(rootEntryName, "default");

	get_debugvars();

    while( (arg = next_commandarg(Whitespace)) != NULL ) {
		if( (*arg == '+') || (*arg == '-') ) {
			arg++;
			switch (*arg) {

			case 'c': /* continue with old ovf file */
				continueSession = TRUE;
				if(reinit) {
					if(interactive) {
						INFO
							"Cannot use -c and -r\n"
						ENDINFO
					}
					HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
					return esmFAILURE;
				}
				break;

			case 'r': /* reinitialize */
				if(continueSession) {
					if(interactive) {
						INFO
							"Cannot use -c and -r\n"
						ENDINFO
					}
					HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
					return esmFAILURE;
				}
				reinit = TRUE;
				break;

			case 'F':
				destroyDataFiles = TRUE;
			case 'f':
				arg = next_commandarg(Whitespace);
				if(arg==NULL) {
					HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, SYNTAXERROR);
				}
				strcpy(rootEntryName, arg);
				break;

			case 'v':
				arg = next_commandarg(Whitespace);
				if(arg==NULL) {
					HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, SYNTAXERROR);
				}
				(void) add_volid(arg);
				break;

			case 'b':
				if(continueSession) {
					if(interactive) {
						INFO
							"Cannot use -c and -b\n"
						ENDINFO
					}
					HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
					return esmFAILURE;
				}
				arg = next_commandarg(Whitespace);
				if(arg==NULL) {
					INFO
						"missing value for -b; ignored"
					ENDINFO
					HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
					break;
				}
				if(getnum(arg, &bufGroupPages) == esmFAILURE) {
					INFO
						"bad value for -b; ignored"
					ENDINFO
					HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
					break;
				}
				INFO
					"%d buffer group pages\n", bufGroupPages
				ENDINFO
				break;

			default:
				HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
				break;
			} /* switch on flag */
		}
    }
	if(tx_is_prepared && !reinit){
		INFO
			"Transaction is prepared. You must use session -r if you wish to test recovery.\n"
		ENDINFO
		INFO
			"If not, you must commit or abort.\n"
		ENDINFO
		return esmFAILURE;
	}
	if(Nvolumes == 0) {
		use_evolids();
	}
	initializeSM(reinit);

    strcpy(ovfName, rootEntryName);
    strcat(ovfName, ".ovf");
	if(open_ovf(ovfName, continueSession) == esmFAILURE) {
		if(interactive) {
			INFO
				"Session not continued. Must start a new session.\n"
			ENDINFO
			return esmFAILURE;
		} else {
			HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, SM_INIT_ERR);
		}
	}

	newBufGroup(bufGroupPages);

	if( tx_is_prepared && !continueSession ) {
		if(interactive) {
			INFO
				"Transaction is prepared.\n"
			ENDINFO
			INFO
				"You must recover, commit or abort, then start a new session.\n"
			ENDINFO
		}
		return esmNOERROR;
	}
	if(volumeStatus(destroyDataFiles, Numretries, 4) != esmNOERROR) {
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SM_INIT_ERR);
		return esmFAILURE;
	}
	/* Make sure CA (or prepared UA) matches OVF:
	 * if continueSession : validate
	 * if new session : create new ovf from existing data.
	 */
	if(continueSession || (objUACount > 0) || (objCACount > 0) ) {
		int e;

		/* debugging assert */
		if(objUACount > 0) {
			if((prepared_tx != UNKNOWN_TID) || (tx_is_prepared == FALSE)) {
				/* internal error */
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, OVF_ERR);
			}
		} else {
			if( (prepared_tx == UNKNOWN_TID) || tx_is_prepared) {
				/* internal error */
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, OVF_ERR);
			}
		}

		if(tx_is_prepared) {
			e = validate(bufGroup, objUACount, /*starting with obj# */0, UA);
		} else {
			if( (e = validate(bufGroup, objCACount, /*starting with obj# */0, CA)) == esmNOERROR) {
				objUACount = copyArrays(UA, CA, objCACount);
			}
		}
		if(e != esmNOERROR) {
			if(interactive) {
				INFO
					"Data do not match OVF.\n"
				ENDINFO
			}
			HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_INIT_ERR);
		}
	} else {
		/* nothing in ovf, not continuing */
		register int i;
		i = count_objs_in_files( TRUE, NULL, 0 );
		objCACount = copyArrays(CA, UA, objUACount);
		if(i != objCACount) {
			/* internal */
			HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, esmINTERNAL);
		}
	/*debugging sanity check */
	check_no_temp_objects(CA, objCACount);
		writeOVF(WORD_COMMIT, CA, objCACount, OVFPtr);
		INFO
			"New OVF has %d objects.\n", objCACount
		ENDINFO
	}

	if(interactive) {
		INFO
			"session ready\n"
		ENDINFO
	}
	return esmNOERROR;
}

int
begin_tx(BOOL barf, BOOL errIfAlreadyActive)
{
	BOOL inProgress;
	if(tid != NULL_TID) {
		inProgress = TRUE;
		DEBUGINFO
			"DEBUG: Transaction already in progress: tid=%d\n", tid
		ENDINFO
	} else inProgress = FALSE;

	DEBUGINFO
		"STARTING NEW TRANSACTION... (tid=%d)\n", tid
	ENDINFO
	sm_errno=esmNOERROR;
	if (sm_BeginTransaction(&tid)) {
		if(barf) {
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_BEGINTRANS_ERR);
		} else if (sm_errno == esmTRANSINPROGRESS) {
			if(inProgress == FALSE) {
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, SM_BEGINTRANS_ERR);
			}
			if(errIfAlreadyActive) {
				HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_BEGINTRANS_ERR);
			}
			/* don't change restart location */
			return esmNOERROR;
		} else {
			if(inProgress ) {
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, SM_BEGINTRANS_ERR);
			} else {
				HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_BEGINTRANS_ERR);
			}
			INFO
			"Transaction not started.\n"
			ENDINFO
			/* don't change restart location */
			return esmFAILURE;
		}
	} 
	if(interactive) {
		INFO "Transaction started. tid = %d\n", tid ENDINFO
	}
	/*
	 *  if reading from a script file, remember the current position, 
	 *  the beginning of the first transaction, may need to return to
	 *  here if server dies or abort this transaction.
	 */
	restart_starts_here();
	begin_tx_bufs(tid);
	return esmNOERROR;
}

void
lost_tx(BOOL in_recover)
{
	HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_2PC_ERR);
	if(sm_errno == esmBADTRANSID) {
		/* server crashed and lost this tx (aborted
		* during recovery
		*/
		if(interactive) {
			INFO
			"No such transaction for server to %s\n",
				in_recover?"recover":"continue"
			ENDINFO
			if( ! in_recover) {
				/* we just don't have a tid to print in the recover case */
				INFO
					"Aborted tid %d.\n", tid
				ENDINFO
			}
		}
		tid = NULL_TID;
		if(in_recover == FALSE) {
			/* go back to the committed or starting data */
			objUACount = copyArrays(UA, CA, objCACount);
			check_no_temp_objects(UA, objUACount);
			mark_temp_volumes_empty();
			if(begin_tx(TRUE, TRUE) != esmNOERROR) {
				HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, NOINFO);
			}
		}
	}
	delete_log(NULL); /* default */
}

BOOL
_abort_tx( BOOL tryReconnect, BOOL startanew )
{
	BOOL canRedo;
	BOOL fatal = FALSE;
	int  e;

	if(tid == NULL_TID) {
notx:
		if(interactive) {
			INFO
				"No active transaction to abort (tid=%d).\n", tid
			ENDINFO
		}
		if(fatal) {
			HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, esmINTERNAL);
		} else {
			return FALSE;
		}
	}
	if( sm_errno == esmNOACTIVETRANS) {
		fatal = TRUE;
		goto notx;
	}
	if(interactive) {
		INFO
			"Aborting transaction %d.\n", tid
		ENDINFO
	}

	sm_errno=esmNOERROR;
    e = sm_AbortTransaction(tid);
	end_tx_bufs(tid);

    if( e == esmNOERROR ) {
		canRedo = TRUE;
worked:
		tid = NULL_TID;
		delete_log(NULL); /* default */

		if(OVFPtr == NULL) {
			if(!tx_is_prepared) {
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, OVF_ERR);
			}
			/* else ok to have no OVF at this point */
		} else {
			/* blow away prepared stuff in OVF file */
			writeOVF(WORD_PREPARE, UA, /* no objects */ 0, OVFPtr);
		}
		prepared_tx = NULL_TID;
		tx_is_prepared = FALSE;

		/* go back to the committed or starting data */
		objUACount = copyArrays(UA, CA, objCACount);

		check_no_temp_objects(UA, objUACount);
		mark_temp_volumes_empty();
		if(startanew) {
			if(begin_tx(TRUE, TRUE) != esmNOERROR) {
				HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, NOINFO);
			}
		}
	} else {
		if (sm_errno == esmTRANSABORTED) {
			canRedo = FALSE; /* server aborted this tx - 
								better not try to redo it */
			goto worked;
		}
		if ( connection_error(sm_errno) ) {
			HANDLE_ERROR(CAUSE_SM, TREAT_INFO, NOINFO);
			if( !tx_is_prepared) {
				canRedo = TRUE;
				goto worked;
			}
			/* no dice; tx is still running */
		}
		HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_ABORT_TRANS_ERR);
		if(interactive) {
			INFO
			"Abort failed. Transaction %d still active.\n", tid
			ENDINFO
		}
		canRedo = FALSE; /* NOTREACHED, actually */
	}
	return canRedo;
}

BOOL
abort_tx( BOOL tryReconnect )
{
	return _abort_tx(tryReconnect, TRUE);
}

static void
put_handle(char *filename)
{
	short	port;
	if(open_log(filename, "w") == esmFAILURE) {
		INFO
			"Cannot open ET log file\"%s\" for writing\n", filename
		ENDINFO
		HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, esmINTERNAL);
	}
	port = atoi(handle.serverName);

	fprintf(logfile, "%d\n%s\n%s\n%s\n", 
		handle.coordTid, handle.hostName, handle.serverName,
		ovfName);
	fsync(fileno(logfile));
	fclose(logfile);
}


static int
get_handle(char *filename)
{
	int n;
	errno = 0;
	if(open_log(filename, "r") == esmFAILURE) {
		INFO
			"Cannot open ET log file\"%s\"\n", filename
		ENDINFO
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, 0);
		return esmFAILURE;
	}

	n = fscanf(logfile, "%d%s%s", &(handle.coordTid),handle.hostName,
		handle.serverName);
	if(n < 3) {
		INFO
			"\"%s\" is not an ET log file.\n", filename
		ENDINFO
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, 0);
		return esmFAILURE;
	}
	return esmNOERROR;
}

BOOL
probe_handle()
{
	int n;

	/* all we can check for is default log file name */
	if(open_log(NULL, "r") == esmFAILURE) {
		return FALSE;
	}
	n = fscanf(logfile, "%d%s%s%s", &(handle.coordTid),handle.hostName,
		handle.serverName, ovfName);
	INFO
		"There is a prepared transaction to recover: coord %s, %s, tid %d\n",
		handle.hostName, handle.serverName, handle.coordTid
	ENDINFO

	tx_is_prepared = TRUE;
	prepared_tx = UNKNOWN_TID;
	fclose(logfile);

	INFO
		"OVF name is %s\n", ovfName
	ENDINFO
	if(open_ovf(ovfName, TRUE) == esmFAILURE) {
		INFO
			"Cannot validate results of recovered transaction.\n"
		ENDINFO
	}
	logfile = NULL;
	return TRUE;
}

static char *
gtid2a(GTID *gtid) 
{
	static char buf[MAXOPAQUELEN + 1];
	char *c = &gtid->opaque[gtid->length-1], *b = &buf[gtid->length];
	register int	 i = gtid->length;
	BOOL	allnull = TRUE;

	*b-- = '\0';
	while(i-- > 0) {
		if ( allnull && (*c == '\0') ) {
			*b-- =  '\0';
			continue;
		} 
		allnull = FALSE;
		if( !isprint(*c) ) { /* this includes nulls */
			*b-- = '*';
			c--;
			continue;
		} 
		*b-- = *c--;
	}
	return buf;
}

int
_continue2pc(BOOL block)
{
	int e;

	if(interactive) {
		INFO
			"Re-entering 2-phase commit for tid %d...\n", tid
		ENDINFO
	}
again:
	sm_errno = esmNOERROR;
    e = sm_Continue2PC(tid, block);
	if(e != esmNOERROR) {
		if(connection_error(sm_errno)) {
			goto again;
		}
		lost_tx(FALSE);
		return esmFAILURE;
	}
	return esmNOERROR;
}
int
continue2pc(char *arg)
{
	int e;
	BOOL block = TRUE;

	if(arg==NULL) {
		block = TRUE;
	} else {
		if(strcasecmp(arg, "false")==0) {
			block = FALSE;
		}
	}
	return _continue2pc(block);
}

int
enter2pc(char *filename)
{
	int e;
	gtid.length = 1;

	if(interactive) {
		INFO
			"Entering 2-phase commit, gtid %s\n", gtid2a(&gtid)
		ENDINFO
	}
	checkNumBufs(__LINE__, __FILE__, TRUE);
    e = sm_Enter2PC(tid, gtid, &handle);
	checkNumBufs(__LINE__, __FILE__, FALSE);
	if(e != esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_2PC_ERR);
		return esmFAILURE;
	}
	if(interactive) {
		INFO
			"2PC Coordinator is %s.%s (coord tid=%d)\n", 
				handle.hostName, handle.serverName, handle.coordTid
		ENDINFO
	}
	put_handle(filename);
	return esmNOERROR;
}

int
recover2pc(char *filename)
{
	int e;
	gtid.length = 1;

	initializeSM(FALSE /* don't force it */);
	if(get_handle(filename) == esmFAILURE) {
		return esmFAILURE;
	}

	if(interactive) {
		INFO
			"Recovering transaction\n\tfrom coordinator %s.%s (coord tid=%d)\n", 
				handle.hostName, handle.serverName, handle.coordTid
		ENDINFO
	}

	checkNumBufs(__LINE__, __FILE__, TRUE);
    e = sm_Recover2PC(handle, gtid, &tid, TRUE);
	checkNumBufs(__LINE__, __FILE__, FALSE);
	if(e != esmNOERROR) {
		lost_tx(TRUE);
		return esmFAILURE;
	}
	return esmNOERROR;
}

int 
command2pc(int which)
{
	BOOL in_recover = (which == WORD_RECOVER);
	int vote;
	char *votestr = next_commandarg(Whitespace);
	char *arg;
	int e;

	if(!votestr) {
bad:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		INFO
			"%s2pc commit-or-abort %s\n", 
				in_recover ? "recover":"continue",
				in_recover ? "[log file name]":"[true|false]"
		ENDINFO
		return esmFAILURE;
	}
	arg = next_commandarg(Whitespace);
	ignore_restofline();

	switch( vote = word2cmd(votestr) ) {
		case WORD_COMMIT:
		case WORD_ABORT:
			break;
		default:
			goto bad;
	}
	switch(which) {
		case WORD_RECOVER:
			e =  recover2pc(arg);
			break;
		case WORD_CONTINUE:
			e =  continue2pc(arg);
			break;
		default:
			HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, NOINFO); /* internal error */
	}
	if( e == esmFAILURE) {
		return esmFAILURE;
	}
	prepared_tx = NULL_TID;
	tx_is_prepared = TRUE;
	if(interactive) {
		INFO
			"Tid %d is prepared.\n", tid
		ENDINFO
	}
	if(vote == WORD_COMMIT) {
		if(interactive) {
			INFO
				"Committing transaction %d, gtid  %s\n", tid, gtid2a(&gtid)
			ENDINFO
		}
		return commit_tx();
	} else {
		if(interactive) {
			INFO
				"Aborting transaction %d, gtid  %s\n", tid, gtid2a(&gtid)
			ENDINFO
		}
		return abort_tx(FALSE/* don't have to reconnect */);
	}
}

int
prepare_tx()
{
	int e;
	VOTE vote;

	if(interactive) {
		INFO
			"Preparing transaction %d\n", tid
		ENDINFO
	}
	checkNumBufs(__LINE__, __FILE__, TRUE);
	e = sm_PrepareTransaction(tid, &vote);
	checkNumBufs(__LINE__, __FILE__, FALSE);

	if (e != esmNOERROR) {
		if(interactive) {
			INFO
				"Prepare failed.\n"
			ENDINFO
		}
		if(connection_error(sm_errno)) {
			/* don't check status because we want to 
			 * be able to continue this 2PC
			 */
			prepared_tx = tid;
			tx_is_prepared = TRUE;
			if(interactive) {
				INFO
					"Transaction %d still active (you need to continue or abort).\n", tid
				ENDINFO
			}
		}
		/* handle-error tries to continue if it was a cx err and is prepared */
		/* use TREAT_INFO because we don't want to report them yet... */
		HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_2PC_ERR);
		if((sm_errno == esmTRANSABORTED) || (sm_errno==esmBADTRANSID) ||
			(sm_errno == esmTRANSNOTIN2PC)) {
			INFO
				"Transaction %d was aborted.\n", tid
			ENDINFO
			(void) abort_tx(FALSE /* try reconnect */);
			return esmFAILURE;
		}
		/* report these */
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_2PC_ERR);
		return esmFAILURE;
	}
	prepared_tx = tid;
	tx_is_prepared = TRUE;
	switch(vote) {
	case YESVOTE:
		mark_temp_objects_destroyed(UA, objUACount);
		writeOVF(WORD_PREPARE, UA, objUACount, OVFPtr);
		if(interactive) {
			INFO
			"Transaction %d prepared. (you need to commit or abort)\n", tid
			ENDINFO
		}
		break;

	case READVOTE:
		if(interactive) {
			INFO
			"Transaction %d is read-only (already committed).\n", tid
			ENDINFO
		}
		tid = NULL_TID;
		return begin_tx(FALSE, TRUE);

	case NOVOTE:
		if(interactive) {
			INFO
			"Transaction %d was aborted.\n", tid
			ENDINFO
		}
		(void) abort_tx(TRUE /* try reconnect */);
		break;
	}
	return esmNOERROR;
}

int
commit_tx()
{
	int err = sm_CommitTransaction(tid);

	end_tx_bufs(tid);

	if (err != esmNOERROR) {
		HANDLE_ERROR(CAUSE_SM, TREAT_NONFATAL, SM_COMMIT_TRANS_ERR);
		if(interactive) {
			INFO
				"Commit failed. Transaction %d still active.\n", tid
			ENDINFO
		}
		checkStatus(FALSE); /* don't seek ahead */
		return esmFAILURE;
	} else {
		if(interactive) {
			INFO
			"Transaction %d committed.\n", tid
			ENDINFO
		}
		delete_log(NULL); /* default */
		mark_temp_volumes_empty();
		mark_temp_objects_destroyed(UA, objUACount);
		check_no_temp_objects(CA, objCACount);

		/* commit the data */
		objCACount = copyArrays(CA, UA, objUACount);
		/* copy committed back to uncommitted - this is so that
		* temp volume stuff works right
		*/
		objUACount = copyArrays(UA, CA, objCACount);
		check_no_temp_objects(CA, objCACount);

		if(OVFPtr == NULL) {
			if(!tx_is_prepared) {
				HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, OVF_ERR);
			}
			/* else ok to have no OVF at this point */
		} else {
			writeOVF(WORD_COMMIT, CA, objCACount, OVFPtr);
		}
		tid = NULL_TID;
		prepared_tx = NULL_TID;
		tx_is_prepared = FALSE;
		return begin_tx(FALSE, TRUE);
	}
}
int
open_ovf( char *ovfName, BOOL ovfMustExist )
{
	/*
	 * Find out if the OVF exists.
	 */
	if(OVFPtr != NULL) {
		writeOVF(WORD_COMMIT, CA, objCACount, OVFPtr); 
		if(tx_is_prepared) {
			writeOVF(WORD_PREPARE, UA, objUACount, OVFPtr); 
		}
		fflush(OVFPtr);
		fclose(OVFPtr);
		OVFPtr = NULL;
	}
	DEBUGINFO
		"OPENING OVF %s, mustexist=%x\n", ovfName, ovfMustExist
	ENDINFO
	/*
	 *  Open the ovf file for reading and writing
	 */
	if ((OVFPtr = fopen(ovfName, "r+")) == NULL) {
		if( errno != ENOENT ) {
			/* No such file or directory */
			HANDLE_ERROR(CAUSE_UNIX, TREAT_FATAL, OVF_ERR);
		}
		if( ovfMustExist ) {
			if(interactive) {
				INFO
					"OVF is missing!\n"
				ENDINFO
			}
			return esmFAILURE;
		}
		/* try again to open it */
		if ((OVFPtr = fopen(ovfName, "w+")) == NULL) {
			HANDLE_ERROR(CAUSE_UNIX, TREAT_FATAL, OVF_ERR);
		}
		writeOVF(WORD_COMMIT, CA, 0, OVFPtr); /* zero out the data */
		objCACount = objUACount = 0;
	} else {
		/* it was there already - read it into CA */
		readOVF(WORD_COMMIT, CA, &objCACount, OVFPtr);
		readOVF(WORD_PREPARE, UA, &objUACount, OVFPtr);
	/*debugging sanity check */
	check_no_temp_objects(CA, objCACount);
	}
	DEBUGINFO
		"OPENED OVF %s: %d, %d objects in CA,UA\n", ovfName, objCACount, objUACount
	ENDINFO
	return esmNOERROR;
}
