#include "sysdefs.h"
#include "trie.h"

#define ABNORMAL_EXIT 99

#if !(defined(hpux) || defined(linux))
BEGIN_EXTERNC
	extern int strcasecmp(char *, char *);
	extern int strncasecmp(char *const, char *const, const int);
END_EXTERNC
#endif /* !hpux */

typedef char **WORDLIST;

struct Trie {
	struct anod *trie;
	WORDLIST	wordlist;
	int		nwords;
	int		nstates;
};


typedef struct node {
	char 	letter;
	int 	depth;
	BOOL	match;
	int		matchindex;
	struct node	*child[26];
	int		nchildren;
} NODE;


#define ANOD_STOP -1 /* for state */

typedef struct anod {
	int state;
	int match; 
} ANOD;

static FILE *debugfile = NULL;
static int arraybytesmalloced = 0;
static int nodebytesmalloced = 0;

/* funcs */
static char printableLetter(char);
static int	letterEquiv(char);
static void insert_tree( NODE 	*, NODE *, WORDLIST , int , int	);
static int 	newstate(TRIE);
static void anod_insert(TRIE , NODE *, int );
static void free_tree( NODE *);

static char
printableLetter(char z)
{
	return z + 'a';
}

static int 
letterEquiv(char z)
{
	if(isalpha(z)) {
		if(isupper(z)) {
			return (int)(z - 'A');
		} else if(islower(z)) {
			return (int)(z - 'a');
		} else {
			fprintf(stderr, "BAD LETTER %c/0x%x\n",z,z);
			exit(ABNORMAL_EXIT);
		}
	}
	return(1);
}

static void
free_tree(NODE *root) 
{
	register int i;
	if(root->nchildren > 0) {
		for(i=0; i<26; i++) if(root->child[i] != NULL)
			free_tree(root->child[i]);
	}
	free(root);
}


static void
insert_tree(
	NODE 	*root, 
	NODE *Root,
	WORDLIST wordlist,
	int wordindex,
	int	calldepth
)
{
	NODE 	*next;
	NODE 	*ptr;
	int		letter, depth;
	int 	wordlen;
	char	*word;


	word = wordlist[wordindex];
	wordlen = strlen(word);

	if(debugfile)
		fprintf(debugfile, "wordlen %d, calldepth %d\n", wordlen, calldepth);

	if(calldepth >= wordlen) {
		fprintf(stderr, "(%d/%d/%d/%s) ASSERTION FAILED AT LINE %d, file %s\n",
			calldepth, wordlen, wordindex, word,
			__LINE__,__FILE__);
		exit(ABNORMAL_EXIT);
	}
	/* 
	 * go down the tree until we hit the end 
	 * of either the tree or the word
	 */
	for( ptr = root, depth = calldepth; 
		depth < wordlen;
		ptr = next, depth++) {

		letter = letterEquiv( word[depth] );
		next = ptr->child[ letter ];
		if(next == NULL)
			break;
		if(debugfile)
			fprintf(debugfile, "found letter %c\n", printableLetter(letter));
	}

	if(depth == wordlen) {
		depth--;

		if(next != NULL) {
			/* possible match or re-match */
			ptr->match =  TRUE;
			if(ptr->matchindex > 0) {
				if(strcasecmp(wordlist[ptr->matchindex], wordlist[wordindex])) {
					fprintf(stderr, " CONFLICT! word (%d,%d) %s\n", 
						ptr->matchindex, wordindex,
						wordlist[wordindex]);
					exit(ABNORMAL_EXIT);
				}
				fprintf(stderr, " DUPLICATE word (%d,%d) %s\n", 
					ptr->matchindex, wordindex,
					wordlist[wordindex]);
				ptr->matchindex = wordindex;
			} else {
				if(debugfile)
					fprintf(debugfile, 
					" MATCH %s NO insert \n", wordlist[wordindex]);
			}
			return;
		}
	}
	/* 
	 * next is null, ptr is non-null
	 * letter is the word we're inspecting at this depth
	 */
	next = (NODE *)calloc(1, sizeof(NODE));
	nodebytesmalloced += sizeof(NODE);

	next->depth = depth;
	next->letter = letter;
	ptr->nchildren++;
	ptr->child[letter] = next;
	if(debugfile)
		fprintf(debugfile, 
		"INSERTING node: letter %c @ depth %d, #%d child of parent(%c)",
		printableLetter(letter), 
		depth, 
		ptr->nchildren,
		(ptr==Root)?'*': printableLetter(ptr->letter)
		);

	if(wordlen == depth+1) { /*depth starts at 0 */

		if(debugfile)
			fprintf(debugfile, " MATCH %s\n", wordlist[wordindex]);
		next->match =  TRUE;
		next->matchindex = wordindex;
	} else {
		if(debugfile)
			fprintf(debugfile, " NOMATCH\n");
		next->match = FALSE;
		next->matchindex = NOMATCH;
		/* keep on inserting nodes */
		insert_tree(next, Root, wordlist, wordindex, depth+1);
	}
}

static int newstate(TRIE trie)
{
	return trie->nstates++;
}

#define INDEX(a, l,s) a[(s*26)+l]

static void
anod_insert(TRIE trie, NODE *p, int state)
{
	register int i;
	ANOD 		*a;

	a = & INDEX(trie->trie,(p->letter),state);

	if(p->match == TRUE) {
		a->match = p->matchindex;
	} else {
		a->match = NOMATCH;
	}
	if(p->nchildren == 0) {
		a->state = ANOD_STOP;
	} else {
		a->state = newstate(trie);

		for(i=0; i<26; i++) {
			if(p->child[i] != NULL) {
				anod_insert(trie, p->child[i], a->state);
			}
		}
	}
}

static void
convert_tree_to_array(NODE *root, TRIE trie)
{
	ANOD 	*a;
	register int i,j;

	trie->trie = (ANOD *)calloc(26*trie->nstates, sizeof(ANOD));

	arraybytesmalloced = 26*(trie->nstates*sizeof(ANOD));

	/* initialize them all */
	for(i=0; i<26; i++) {
		for(j = 0; j<trie->nstates; j++) {
			a = & INDEX(trie->trie,i,j);
			a->state = ANOD_STOP;
			a->match = NOMATCH;
		}
	}
	j = trie -> nstates;
	trie->nstates = -1; /* start over again;  we will count up again */
	anod_insert(trie, root, newstate(trie));
	if(trie->nstates != j) {
		fprintf(stderr, "(%d/%d)ASSERTION FAILED AT LINE %d, file %s\n",
			trie->nstates, j,
			__LINE__,__FILE__);
		exit(ABNORMAL_EXIT);
	}
}

static int
maxdepth(NODE *p)
{
	static int maxd=0;
	register int i;

	if(p->depth > maxd)
		maxd = p->depth;
	if(p->nchildren > 0) {
		for(i=0; i<26; i++) if(p->child[i] != NULL) 
			(void) maxdepth(p->child[i]);
	}

	return maxd;
}

static void
prune(NODE *parent, NODE *p, WORDLIST wordlist)
{
	register int i;

	if(debugfile)
		fprintf(debugfile, 
		"%*.*s> %c (node 0x%x) at depth %d, nchildren %d %s %s \n",
		p->depth, p->depth, "-------------------",
		printableLetter(p->letter), 
		p,
		p->depth, p->nchildren,
		p->match?"match":"nomatch",
		(p->matchindex== NOMATCH)?"":wordlist[p->matchindex]);

	if((p->nchildren) >= 1) {
		for(i=0; i<26; i++) {
			if(p->child[i] != NULL)
				prune(p, p->child[i], wordlist);
		}
	}

	if(p->nchildren == 0) {
		if(p->match && (parent != NULL)) {
			/* I am a leaf and I match a word, and my parent doesn't have
			 * such a match, and I have no siblings, there's no need to
			 * keep me around.
			 */
			if((parent->nchildren == 1) && (parent->match==FALSE)) {
				if(debugfile)
					fprintf(debugfile, "      PRUNING node 0x%x\n",p);
				/* prune the tree */
				parent->match = TRUE;
				parent->matchindex = p->matchindex;
				parent->child[p->letter] = NULL;
				parent->nchildren = 0;
				free(p);
				nodebytesmalloced -= sizeof(NODE);
				return;
			}
		}
	} 
}

static int
states(NODE *p, WORDLIST wordlist)
{
	register int i;
	int n = 0;

	if(debugfile)
		fprintf(debugfile, 
		"%*.*s> %c (node 0x%x) at depth %d, nchildren %d %s %s \n",
		p->depth, p->depth, "-------------------",
		printableLetter(p->letter), 
		p,
		p->depth, p->nchildren,
		p->match?"match":"nomatch",
		(p->matchindex== NOMATCH)?"":wordlist[p->matchindex]);

	if(p->nchildren == 0) {
		return 0;
	} 

	/* i have children */
	n = 1;
	for(i=0; i<26; i++) {
		if(p->child[i] != NULL)
			n += states(p->child[i], wordlist);
	}
	return n;
}

static void
dump_array(TRIE trie)
{
	ANOD	*array = trie->trie;
	int 	across = trie->nstates;
	int		down = 26;
	register int l, s;
	static char *alphabet = "abcdefghijklmnopqrstuvwxyz";
	ANOD *anod;

	if(!debugfile) return;

	fprintf(debugfile, "%3.3s ","L\\S");
	for(s=0; s<across; s++) {
		fprintf(debugfile, "%3.3d ",s);
	}
	fprintf(debugfile, "\n");
	fprintf(debugfile, "%3.3s ","___");
	for(s=0; s<across; s++) {
		fprintf(debugfile, "%3.3s ","___");
	}
	fprintf(debugfile, "\n");

	for(l=0; l<down; l++) {
		fprintf(debugfile, "  %c ",alphabet[l]);


		for(s=0; s<across; s++) {
			anod = & INDEX(array,l,s);
			if(anod->state != ANOD_STOP) 
				fprintf(debugfile, "%3.3d ",anod->state);
			else {
				if(anod->match != NOMATCH)
					fprintf(debugfile, "W%2.2d ",anod->match);
				else
					fprintf(debugfile, " %2.2s ","   ");
			}
		}
		fprintf(debugfile, "\n");
	}
	fprintf(debugfile, "\n");
}

static void
printMatches(TRIE trie, FILE *output, int state)
{
	/* print every match in this state, and go on through the states 
	 * we can reach from here
	 */
	ANOD			*a;
	register int	i;


	/* sanity */
	if(state == ANOD_STOP) return;

	for(i=0; i<26 ; i++) {
		a = &INDEX(trie->trie, i, state);

		if(a->match != NOMATCH) {
			fprintf(output, "\t%s\n", trie->wordlist[a->match]);
		}
	}

	for(i=0; i<26 ; i++) {
		a = &INDEX(trie->trie, i, state);
		if(a->state == ANOD_STOP) 
			continue;
		printMatches(trie, output, a->state);
	}
}


/* returns command number */
int
trie_search(TRIE trie, char *word, FILE *output, BOOL abbrev)
	/* abbrev == TRUE means we are accepting abbreviations, and
	* if there's an ambiguity, it's an error.
	*/
{
	ANOD		*a = NULL;
	int			state, letter;
	int			wordlen;
	register int i;

	if(word)
		wordlen = strlen(word);
	else
		return NOMATCH;

	for(i=0, state = 0; 
		state != ANOD_STOP; 
		i++, state = a->state) {

		if(i >= wordlen) break;

		letter = letterEquiv(word[i]);
		a = &INDEX(trie->trie, letter, state);

		if(a->match == NOMATCH) 
			continue;

		/* else it matches something at this node */

		if(a->state == ANOD_STOP) {
			/* perfect match */

			if( strncasecmp(trie->wordlist[a->match], word, wordlen) == 0 ) {
				if(output!=NULL) {
					/*
					fprintf(output, "MATCH: %s\n", &(trie->wordlist[a->match][i+1]));
					*/
					/*
					fprintf(output, "MATCHES %d: %s\n", 
						a->match, trie->wordlist[a->match]);
					*/
				}
				return a->match;
			} else 
				goto bad;
		} else {
			if(abbrev)
				goto ambiguous;
			else  
				return a->match;
		}
	}
	/* did not match anything given: word was too short or
	 * we hit stop or not even the first letter matched anything. 
	 */

	if((state == ANOD_STOP) || (state ==0) )
		goto bad;
	if(a==NULL)
		goto bad;

	if(!abbrev)
		goto bad;

ambiguous:
	if(output) {
		fprintf(output, "Ambiguous command; choices are:\n");
		if(a->match != NOMATCH) {
			fprintf(output, "%s\n", trie->wordlist[a->match]);
		}
		printMatches(trie, output, a->state);
		fprintf(output, "Type enough letters to be unambiguous.\n");
	}
	return NOMATCH;
bad:
	return NOMATCH;
}

TRIE 
trie_make(WORDLIST wordlist)
{
	extern 	BOOL debugon;
	NODE 	*root = NULL;
	TRIE	result;
	int		maxd;
	int	 	nwords;
	register int i;


	result = (TRIE) calloc(1, sizeof (struct  Trie));
	if(result==NULL) {
		fprintf(stderr, "Cannot malloc %d bytes\n", sizeof(struct Trie));
		exit(ABNORMAL_EXIT);
	}
	result->wordlist = wordlist;
	result->nstates = -1; /* starting value */

	for(nwords=0; ; nwords++) {
		if(wordlist[nwords] == 0) break;
	}
	result->nwords = nwords;

	if(debugfile) fprintf(debugfile, "%d words\n",nwords);

	root = (NODE *)calloc(1, sizeof(NODE));
	nodebytesmalloced += sizeof(NODE);

	root->depth = -1;
	root->match = FALSE;

	for(i=0; i<nwords; i++) {
		insert_tree(root, root, result->wordlist, i, 0);
	}

	prune(NULL, root, result->wordlist);

	result->nstates = states(root, result->wordlist);

	maxd = maxdepth(root);

	convert_tree_to_array(root, result);

	dump_array(result);
	free_tree(root);

	if(debugon) {
	fprintf(stdout, "DEPTH=%d,  #STATES = %d\n",maxd ,i);
	fprintf(stdout, "ARRAY: %d bytes malloced\n", arraybytesmalloced);
	fprintf(stdout, "TREE: %d bytes malloced\n", nodebytesmalloced);
	}

	return result;
}

#ifdef notdef
char *wordlist1[] = {
"commit",
"create",
"crash",
"setoption",
"configfile",
"session",
"echoFile",
"endechofile",
"abort",
"append",
"commit",
"crashserver",
"connect",
"checkpoint",
"destroy",
"delete",
"echo",
"exit",
"flush",
"frequency",
"help",
"insert",
"list",
"noecho",
"print",
"random",
"read",
"scan",
"set",
"serverstats",
"sleep",
"validate",
"ua",
"ca",
"version",
"write",
"zero",
 0
};
char *wordlist2[] = {
 "a",
 "i",
 "he",
 "be",
 "of",
 "with",
 "as",
 "is",
 "or",
 "at",
 "it",
 "to",
 "by",
 "in",
 "on",
 "was",
 "had",
 "the",
 "his",
 "not",
 "and",
 "her",
 "you",
 "for",
 "are",
 "but",
 "from",
 "that",
 "this",
 "have",
 "with",
 "which",
 "had",
 0
};

main(int argc, char **argv)
{
	TRIE	trie;
	register int i;
	static char inputLine[100];

	while(argc>1) {
		argc--;
		if(strncmp(argv[argc], "-d", 2)==0) 
			debugfile = stderr;
	}
#define WORDLIST wordlist1

	trie = trie_make( WORDLIST );

	for(;;) {
		fprintf(stderr, "\n%s > ", argv[0]);
	again:
		if(gets(inputLine) == NULL)
			break;
		i= trie_search(trie, strtok(inputLine, " 	"), stdout);
		if(i == NOMATCH) {
			fprintf(stdout, "\ntry again >");
			goto again;
		} else{
			fprintf(stdout, "MATCHES %d: %s\n", i, WORDLIST[i]);
		}
	}
}
#endif

