/*
 *	This file defines operations on widgets
 */
#include <stdio.h>
#include "sm_client.h"

#include <stdlib.h>
#include <strings.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "widget.h"
#include "error_handler.h"

int		global_bufgroup;
VOLID  	global_volid;
TID  	global_tid = NULL_TID;


/*
 *	Mount the volume containing the widgets
 */
int findWidgetVolume()
{
	char *volidStr = NULL;

	volidStr = getenv("EVOLID");

    if (volidStr == 0)  {
		fprintf(stderr, "Please specify a volume ID in EVOLID\n");
		return(esmBADVOLID);
    }
	global_volid = atoi(volidStr);

	return(esmNOERROR);
}


/*
 *	Get the ID of the file containing the widgets
 */
int getWidgetFile(FID* fid, char* name)
{

	int	rootLength;

retry_getWidgetFile: /* come here to retry transaction */

	/*
	 *	start the transation
	 */
	if (sm_BeginTransaction( &global_tid ) != esmNOERROR) {
		HANDLE_ERROR(sm_errno);
		return(sm_errno);
	}

	/*
	 *	Get the root entry containing the file id of the widget file.
	 */
 	rootLength = sizeof(*fid);
	if (sm_GetRootEntry(global_volid, name, fid, &rootLength) != esmNOERROR) {

		if (sm_errno != esmBADROOTNAME) {
			HANDLE_ERROR_ABORTED(sm_errno, retry_getWidgetFile);
			return(sm_errno);
		}
	}

	/*
	 *	See if the widget file already exists
	 */
	if ( (sm_errno == esmBADROOTNAME) || (rootLength == 0) ) {

		/*
		 * Create the widget file.
		 */
		printf("creating widget file\n");
		if (sm_CreateFile(global_bufgroup, global_volid, fid) != esmNOERROR) {
			HANDLE_ERROR_ABORTED(sm_errno, retry_getWidgetFile);
			return(sm_errno);
		}

		/*
		 * Put the datafile's fid in the root page.
		 */
		if (sm_SetRootEntry(global_volid, name, (char*)fid, sizeof(FID)) != esmNOERROR)	{
			HANDLE_ERROR_ABORTED(sm_errno, retry_getWidgetFile);
			return(sm_errno);
		}
	}

    /*
     *	Commit the transaction
     */
    if (sm_CommitTransaction(global_tid))	{

		HANDLE_ERROR_ABORTED(sm_errno, retry_getWidgetFile);
		return(sm_errno);
	}

	return(esmNOERROR);
}


/*
 *	Destroy the file containing the widgets
 */
int destroyWidgetFile(FID* fid)
{

	printf("attempting to destroy widget file\n");
	if (sm_DestroyFile(global_bufgroup, fid) != esmNOERROR) {
		HANDLE_ERROR(sm_errno);
		SM_ABORT_TRANSACTION(sm_errno);
	}

	/*
	 * Mark that the file has been destroyed.
	 */
	if ( sm_RemoveRootEntry(global_volid, WIDGET_ROOT_ENTRY) != esmNOERROR)	{
		HANDLE_ERROR(sm_errno);
		SM_ABORT_TRANSACTION(sm_errno);
	}

	return(esmNOERROR);
}


/*
 *	Produce a widget
 */
int	produce(FID* widgetFid, int numWidgets, int widgetSize, 
			char* producerName)
{

	int			count;
	widget_t	*widget;
	int			fillerSize;
	int			i;
	OID			widgetOid;

	INVALIDATE_TID(global_tid);

	if (widgetSize < sizeof(widget_t)) {
		widgetSize = sizeof(widget_t);
	}

	/* allocate space for a widget with the proper filler size */
	widget = (widget_t*) malloc(widgetSize);

	/* initialize the widget */
	strncpy(widget->producerName, producerName, MAX_WIDGET_NAME);
	fillerSize = widgetSize - sizeof(*widget) + sizeof(widget->filler);
	for (i = 0; i < fillerSize; i++) {
		widget->filler[i] = i % 256;
	}

	count = 0;  /* no widgets produced yet */
	while (count < numWidgets) {

		/* if there is already a transaction running, abort it */
		if (!TID_IS_INVALID(global_tid)) {
			if (sm_AbortTransaction(global_tid) != esmNOERROR)	{
				HANDLE_ERROR(sm_errno);
				exit(1);
			}
		}

		/* start a new transaction */
		if (sm_BeginTransaction( &global_tid ) != esmNOERROR) {
			HANDLE_ERROR(sm_errno);
			continue;
		}

		/* initaalize a widget, and create it */
		widget->widgetNum = count;
		if (sm_CreateObject(global_bufgroup, widgetFid, NEAR_LAST_PHYSICAL, NULL,
						NULL, widgetSize, (char*)widget, &widgetOid) != esmNOERROR)	{
			HANDLE_ERROR(sm_errno);
			continue;
		}

		/* Commit the transaction */
		if (sm_CommitTransaction(global_tid) != esmNOERROR)	{
			HANDLE_ERROR(sm_errno);
			continue;
		}
		INVALIDATE_TID(global_tid);

		printf("Made widget %d size = %d\n", widget->widgetNum, widgetSize);
		count++;
    }

    if (widget) free(widget);

    return(esmNOERROR);
}


/*
 *	Consume a widget
 */
int	consume(FID* widgetFid, int numWidgets, char* consumerName)
{

	int			count;
	widget_t	*widget;
	int			fillerSize;
	int			i;
	OID			widgetOid;
	OBJHDR		objHdr;
	USERDESC	*userDesc;
	BOOL		eof;
	int			abortCode;

	printf("Consumer %s is consuming %d widgets.\n", consumerName, numWidgets);
	count = 0;  /* no widgets produced yet */
	while (count < numWidgets) {


		SM_BEGIN_TRANSACTION(&global_tid, abortCode)

			/* get the oid of the first widget in the file */
			if (sm_GetFirstOid(global_bufgroup, widgetFid, &widgetOid, &objHdr, &eof) != esmNOERROR)	{
				HANDLE_ERROR(sm_errno);
				SM_ABORT_TRANSACTION(sm_errno);
			}


			if (eof) {
				/* the file is empty, so destroy it */
				destroyWidgetFile(widgetFid);
			} else {
				/* obtain a pointer to the object */
				if (sm_ReadObject(global_bufgroup, &widgetOid, 0, objHdr.size, &userDesc) != esmNOERROR)	{
					HANDLE_ERROR(sm_errno);
					SM_ABORT_TRANSACTION(sm_errno);
				}
				widget = (widget_t*) userDesc->basePtr;

				/* 	validate the widget */
				fillerSize = objHdr.size - sizeof(*widget) + sizeof(widget->filler);
				for (i = 0; i < fillerSize; i++) {
					if (widget->filler[i] != i % 256) {
						printf("Widget error: incorrect filler for widget %d from %s\n", widget->widgetNum, widget->producerName);
					}
				}

				printf("found widget #%d from %s\n", widget->widgetNum, widget->producerName);

				/* unfix the widget and destroy it */
				if (sm_ReleaseObject(userDesc) != esmNOERROR) {
					HANDLE_ERROR(sm_errno);
					exit(1);	/* there should be no error from this call */
				}
				if (sm_DestroyObject(global_bufgroup, &widgetOid) != esmNOERROR) {
					HANDLE_ERROR(sm_errno);
					SM_ABORT_TRANSACTION(sm_errno);
				}

			}
	
		SM_COMMIT_TRANSACTION(global_tid)
		if (abortCode == esmNOERROR) {
			/* transaction was committed */
			if (eof) {
				printf("destroyed widget file #%d\n", widgetFid->pid.page);
				return(esmNOERROR);
			} else {
				printf("consumed widget #%d from %s\n", widget->widgetNum, widget->producerName);
			}
			count++;
		} else {
			/* transaction was aborted */
			if (sm_errno == esmBADFID) {
				/* the file has been destroyed, so leave */
				printf("Widget file destroyed.\n");
				return(esmNOERROR);	
			}
			HANDLE_ERROR(sm_errno);
		}
    }

    return(esmNOERROR);
}



