-------------------------------------------------------------------------------
--                                                                           --
--               Ada95 Bindings to Linux SVGA Library                        --
--                                                                           --
--                         Darren J Moffat                                   --
--                      moffatd@dcs.gla.ac.uk                                --
--                                                                           --
-- These bindings may be distributed under the same agreement as holds for   --
-- the current version of SVGALib.  With the additional condition that my    --
-- name must not be removed from any of the Ada source files.                --
--                                                                           --
-------------------------------------------------------------------------------
-- Last Modified: 
-- $RCSfile: svgalib.ads,v $
-- $Revision: 1.1 $
-- $Author: djm $
-- $Log: svgalib.ads,v $
-- Revision 1.1  1995/02/08  00:57:42  djm
-- entered into RCS
--
 
with 
  Interfaces.C;
use
  Interfaces.C;

package SVGALib is

   ----------------------------------------------------------------------
   -- Exceptions.
   ----------------------------------------------------------------------
   
   Invalid_Vga_Mode : exception;

   -----------------------------------------------------------------------
   -- Types for SVGALib.
   -----------------------------------------------------------------------

   -- Vga_Mode is a type representing all? of the modes currently supported
   -- by SVGALib, these were generated from the vga.h header file.
   -- My personal view is that a mechanism more like X would be nicer,
   -- since there is no guarantee that these modes are available on a
   -- given video card. NB Please use Has_Mode() and don't assume!
   type VgaMode is (TEXT,
                    G320x200x16,
		    G640x200x16,
		    G640x350x16,
                    G640x480x16,
                    G320x200x256,
                    G320x240x256,
                    G320x400x256,
                    G360x480x256,
                    G640x480x2,
                    G640x480x256,
                    G800x600x256,
                    G1024x768x256, 
                    G1280x1024x256,
                    G320x200x32K,
                    G320x200x64K,
                    G320x200x16M,
                    G640x480x32K,
                    G640x480x64K,
                    G640x480x16M,
                    G800x600x32K,
                    G800x600x64K,
                    G800x600x16M,
                    G1024x768x32K,
                    G1024x768x64K,
                    G1024x768x16M,
                    G1280x1024x32K,
                    G1280x1024x64K,
                    G1280x1024x16M,
                    G800x600x16,
                    G1024x768x16,
                    G1280x1024x16,
                    G720x348x2,
                    G320x200x16M32,
                    G640x480x16M32,
                    G800x600x16M32,
                    G1024x768x16M32,
                    G1280x1024x16M32);

   for VgaMode'Size use Int'Size;
   for VgaMode use (TEXT => 0,
                    G320x200x16 => 1,
		    G640x200x16 => 2,
		    G640x350x16 => 3,
                    G640x480x16 => 4,
                    G320x200x256 => 5,
                    G320x240x256 => 6,
                    G320x400x256 => 7,
                    G360x480x256 => 8,
                    G640x480x2 => 9,
                    G640x480x256 => 10,
                    G800x600x256 => 11,
                    G1024x768x256 => 12, 
                    G1280x1024x256 => 13,
                    G320x200x32K => 14,
                    G320x200x64K => 15,
                    G320x200x16M => 16,
                    G640x480x32K => 17,
                    G640x480x64K => 18,
                    G640x480x16M => 19,
                    G800x600x32K => 20,
                    G800x600x64K => 21,
                    G800x600x16M => 22,
                    G1024x768x32K => 23,
                    G1024x768x64K => 24,
                    G1024x768x16M => 25,
                    G1280x1024x32K => 26,
                    G1280x1024x64K => 27,
                    G1280x1024x16M => 28,
                    G800x600x16 => 29,
                    G1024x768x16 => 30,
                    G1280x1024x16 => 31,
                    G720x348x2 => 32,
                    G320x200x16M32 => 33,
                    G640x480x16M32 => 34,
                    G800x600x16M32 => 35,
                    G1024x768x16M32 => 36,
                    G1280x1024x16M32 => 37);
		     

   -- Probably a minimum of 0 and a max of 1280 but won't set limits.
   -- Use Integer rather than Natural since they are int in C.
   
   type Point is record
      X, Y : Integer;
   end record;
   
   subtype RGBNumber is Integer range 0..255;
   type RGBColour is record
      Red : RGBNumber;
      Green : RGBNumber;
      Blue : RGBNumber;
   end record;


   -----------------------------------------------------------------------
   -- Mode Settings
   -----------------------------------------------------------------------

   -- The following were defined in vga.h to return int,
   -- As far as I can tell they are really boolean success values
   -- so they have been given return types of Boolean for Ada.

   procedure Set_Mode (Mode : VgaMode := G320x200x16);

   function Has_Mode (Mode : VgaMode := G320x200x16) return Boolean;


   -- Not sure what this does => correct type for this parameter.
   function Set_Flip_Char (C : Integer) return Boolean;
   
   -- vga_flip and vga_clear were defined to return int, I feel
   -- that these should really be procedures with no in or out params.

   procedure Clear;

   procedure Flip;

   function Get_Xdim return Integer;

   function Get_Ydim return Integer;

   function Get_Colours return Natural;

   

   -----------------------------------------------------------------------   
   -- Stuff for Palettes
   -----------------------------------------------------------------------
   procedure Set_Palette (Palette : in Natural; Colour : in RGBColour); 
   procedure Get_Palette (Palette : in Natural; Colour : out RGBColour);
   
   -- Not sure what these are for ??
   -- int vga_setpalvec(int start, int num, int *pal);
   -- int vga_getpalvec(int start, Int num, int *pal);

   
   -----------------------------------------------------------------------
   -- Turn the Screen Display On/Off
   -----------------------------------------------------------------------
   procedure Screen_Off;

   procedure Screen_On;


   -----------------------------------------------------------------------
   -- Drawing Functions.
   -----------------------------------------------------------------------

   -- Is this the same number from set/get palette ?
   procedure Set_Colour (Colour : in Integer);

   procedure Set_Color (Colour : in Integer) renames Set_Colour;

   procedure Draw_Pixel (P : in Point);
   procedure Draw_Point (P : in Point) renames Draw_Pixel;   

   procedure Draw_Line (T,B : in Point);


   -- Not sure how usefull these are and what their params mean.

   -- int vga_drawscanline(int line, unsigned char* colors);

   -- procedure Draw_Scan_Line (Line : in Integer; Colours : ???);

   -- int vga_drawscansegment(unsigned char* colors, int x, int y, int length)



   pragma Import (C, Has_Mode, "vga_hasmode");
   pragma Import (C, Set_Flip_Char, "vga_setflipchar");
   pragma Import (C, Clear, "vga_clear");
   pragma Import (C, Flip, "vga_flip");
   pragma Import (C, Get_Xdim, "vga_getxdim");
   pragma Import (C, Get_Ydim, "vga_getydim");
   pragma Import (C, Get_Colours, "vga_getcolors");
   pragma Import (C, Screen_Off, "vga_screenoff");
   pragma Import (C, Screen_On, "vga_screenon");
   pragma Import (C, Set_Colour, "vga_setcolor");


end SVGALib;


