//:ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLProcess.h,v 0.27 1995/01/20 15:15:32 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLProcess --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************/

#ifndef __CNSDLProcess_h
#define __CNSDLProcess_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>

#include <CNCL/Object.h>		// Base class

#include <CNCL/DLList.h>
#include <CNCL/SDLTimer.h>
#include <CNCL/SDLManager.h>

extern CNClassDesc CN_SDLPROCESS;	// Class CNSDLProcess description


typedef int PId;
typedef int ProcessType;
typedef int SignalType;

class CNSDLProcess;
typedef int(CNSDLProcess::*PMF)(int); 


/*
 * The class CNSDLProcess
 */

class CNSDLProcess : public CNObject
{
    friend class CNSDLManager;      // Communication-Manager etc.
    friend class CNSDLEnvironment;  // needs access to 'System'
    friend class CNSDLTimer;        // needs save_list for reset

  private:
        // hey, i don't need the global identifier any longer,
        // just made it a static member
    static PMF _global_pointer_to_actual_sdlsignalhandler_;
    
  public:
    PId self();              // special PIds
    PId parent(); 
    PId offspring();
    PId sender();
    PId environment();
        
    ProcessType type();      // get process type
   
    char *get_name() const;

  protected:
    int state();     // get/set process state
    void state(int);
    
    void save(CNSDLSignal *sig);
    
    CNSimTime now();
    
    void procedure(PMF, SignalType); // procedure-call
                                     // (contains address of new sdl handler)
    SignalType from_procedure();     // returns from procedure

    virtual int process(int);     // process body
        

  private:
    void parent(PId);    // set PIds (only SDLManager)
    void offspring(PId);
    void sender(PId);
    void self(PId);
        
    void clear_savelist();  // (only SDLManager)
             
    int signalhandler(int);  // is called from SDLManager


  private:
    int actual_state;
    int p_type;  // process type
    char *p_name;

    CNDLList *save_list;  // list of saved signals
    CNSDLTimer *p_timer;  // timers of process

    struct stack_i { int old_state;    // stack used for procedure calls  
                     PMF old_handler;
                     SignalType old_signame;
                     int helpval;
                     stack_i *stack_end;
                   } *stack;
    
    PMF handler;   //  local pointer to signal handler
                   // (process, procedure)

    PId p_self;       // own PId 
    PId p_parent;     // the creator (is zero when created at system start)
    PId p_offspring;  // the last child created by this process
    PId p_sender;     // sender of last received signal
    

  protected:
    CNSDLManager *System;   // pointer to system's SDLManager
    
    int sub_state;    //this one is used 'cause procedures are okay!

    /***** Constructors ******************************************************/
  public:
    CNSDLProcess(CNParam *param) {}			// CNParam constructor
    CNSDLProcess(char*, CNSDLManager*, int);
      
    ~CNSDLProcess();
    
  protected:
    CNSDLProcess() {}                           // Default constructor

  
    /***** Member functions required by CNCL *********************************/
  public:
    virtual CNClassDesc class_desc() const	// CNClass description
    {
	return CN_SDLPROCESS;
    };
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    {
	return desc == CN_SDLPROCESS ? TRUE : CNObject::is_a(desc);
    };
        
    static CNSDLProcess *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
        return (CNSDLProcess *)obj;
#   else
        return (CNSDLProcess *)( !obj || obj->is_a(CN_SDLPROCESS)
               ? obj : fatal_type(obj->class_desc(), CN_SDLPROCESS) );
#   endif
    }
    
    
    // CNObject creation
    static CNObject *new_object(CNParam *param = NIL)
    { return param ? new CNSDLProcess(param) : new CNSDLProcess; }
     
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};



// now the inline functions

// get/set state
inline int CNSDLProcess::state() { return actual_state; }
inline void CNSDLProcess::state(int new_state) 
{
    actual_state = new_state;
    sub_state = 0;
}

inline int CNSDLProcess::type() { return p_type; }

// some special PId's
inline PId CNSDLProcess::self()        { return p_self; }
inline PId CNSDLProcess::parent()      { return p_parent; }
inline PId CNSDLProcess::offspring()   { return p_offspring; }
inline PId CNSDLProcess::sender()      { return p_sender; }
inline PId CNSDLProcess::environment() { return 1; }

inline void CNSDLProcess::self(PId se)      { p_self = se; }
inline void CNSDLProcess::parent(PId pa)    { p_parent = pa; }
inline void CNSDLProcess::offspring(PId ch) { p_offspring = ch; }
inline void CNSDLProcess::sender(PId sd)    { p_sender = sd; }

// called by SDLManager (indirect function call)
inline int CNSDLProcess::signalhandler(int signal)
{
    CNSDLProcess::_global_pointer_to_actual_sdlsignalhandler_ = handler;
    return (this->*_global_pointer_to_actual_sdlsignalhandler_)(signal);
}


#endif /**__CNSDLProcess_h**/
