//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLEnvironment.c,v 0.27 1995/01/20 15:15:32 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLEnvironment --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "SDLEnvironment.h"

/*
   here the environment is created; there is only one! environment process
   with the class name SDLProcess_environment; only one! instance of this
   class may exist; to ease communication with the environment a global
   pointer to the environment data is used.
*/

CNSDLEnvironment::CNSDLEnvironment(int systems, CNSDLProcess* env_proc, 
CNEventBaseSched* sched) : CNSDLManager(1, sched)
{
    if (pointer_to_environment_data)
        fatal("SDLEnvironment ","only one environment allowed");
    
    processes = new CNSDLProcessData *[pr_size = 2];
    signal_receiver = NIL; channels = NIL;
    sr_size = ch_size = 0;
        
    pointer_to_environment_data = processes[1] = 
                         new CNSDLProcessData(1, 1, 1);
    create (env_proc, 1); env_proc->System = this;
    
    SDLSystem = new CNSDLManager*[ sys_no = (systems+1) ];
    for (int i=0; i<sys_no; i++) SDLSystem[i] = NIL;
}    


CNSDLEnvironment::~CNSDLEnvironment()
{
      // delete all connected systems calling their destructors
    for (int i=0; i<sys_no; i++) delete SDLSystem[i];
    delete[] SDLSystem;
      // delete environment itself
    delete processes[1]; // may delete env_proc
    delete[] processes;	
      // mark environment as dead
    pointer_to_environment_data = NIL;
}



// connect SDLSystem with environment
void CNSDLEnvironment::install(int sys, CNSDLManager *sys_p)
{
    if (sys>=sys_no || sys<0)
        fatal("SDLEnvironment ","invalid system number");
    if (SDLSystem[sys])
        fatal("SDLEnvironment ","same system(number) installed twice");

    SDLSystem[ sys ] = sys_p;
}



// disconnect SDLSystem from environment (it will not be deleted though)
CNSDLManager *CNSDLEnvironment::deinstall(int sys)
{
    if (sys>=sys_no || sys<0)
    {
        error("SDLEnvironment ","invalid system number");
        return NIL;
    }
    
    CNSDLManager *disconnected = SDLSystem[sys];
    SDLSystem[sys] = 0;
    return disconnected;
}



/*
   the modified local output functions. Since the environment "system"
   contains only one process explicit addressing other than to PId 1 is 
   forbidden; implicit addressing always goes to PId 1; there are no 
   internal signalroutes so ProcessType must be 1 in every case.
*/
void CNSDLEnvironment::output_x(SignalType name, PId receiver, 
                                    CNObject *data)
{
    // here explicit addressing is used
    if (receiver != 1)
        fatal("SDLEnvironment::output ", "wrong signal addressing");
     
    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}


void CNSDLEnvironment::output(SignalType name, CNObject *data)
{
    // implicit addressing
    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}


void CNSDLEnvironment::output(SignalType name, ProcessType rec, 
                                  CNObject *data)
{
    if (rec!=1)
        fatal("SDLEnvironment::output ", "wrong signal addressing");


    send(new CNSDLSignal(name, now(), 1, 1, 1, data));
}


/*
   the modified channel output functions. Since the environment is connected
   to many systems it uses their channels; it also uses internal data of the
   systems to find out the right addresses for the signals, the target
   system has to be set here (unlike in SDL systems)
*/
void CNSDLEnvironment::output_x(SignalType name, PId receiver, 
                       CNSDLChannel *ch, CNObject *data, int target_system)
{
    if (target_system < 2)
        fatal("SDLEnvironment::output ", "wrong target system");

    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival = ch->delay(now());
    			ch->disturb(name, data);
    			
    if (receiver == 1)
        fatal("SDLEnvironment::output ", 
              "no channel to connect environment with environment");
    else 
    {
        ProcessType pt = 2;
        CNSDLManager *man = SDLSystem[target_system];
        
        while ( !(man->processes[ pt ]->is_of_this_type(receiver)) )
        {
            pt++;
            if (pt == man->pr_size)
                fatal("CNSDLEnvironment::output_x", " strange error");
        }

        if ( !(man->processes[ pt ]->get_process_by_PId(receiver)) )
            error("SDLEnvironment::output", " no receiver availlable");    
        else
            send(new CNSDLSignal(name, arrival, this, man,
                             1, receiver, 1, data));
    }
}

void CNSDLEnvironment::output(SignalType name, ProcessType rec, 
                         CNSDLChannel *ch, CNObject *data, int target_system)
{
    if (target_system < 2)
        fatal("SDLEnvironment::output ", "wrong target system");

    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival = ch->delay(now());
    			ch->disturb(name, data);
    			
    if (rec == 1)
        fatal("SDLEnvironment::output ", 
              "no channel to connect environment with environment");
    else 
    {
        CNSDLManager *target = SDLSystem[target_system];
        CNSDLProcess *proc_ad = target->processes[rec]->get_process_by_type();
        if (!proc_ad)
            error("SDLEnvironment::output", " no receiver availlable");
        else
        {
            PId receiver = proc_ad->self(); 
            send(new CNSDLSignal(name, arrival, this, target, 1, 
                                                      receiver, 1, data));
        }
    }                          
}


CNSDLProcessData *CNSDLEnvironment::pointer_to_environment_data = NIL;


/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNSDLEnvironment::print(ostream &strm) const
{
    strm << "SDLEnvironment:" << endl;
    CNSDLManager::print(strm);
}

// Debug output
void CNSDLEnvironment::dump(ostream &strm) const
{
    strm << "CNSDLEnvironment { $Revision: 0.27 $ ..." << endl;
    CNSDLManager::dump(strm);
    strm << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNSDLEnvironment
static CNClass CNSDLEnvironment_desc("CNSDLEnvironment", "$Revision: 0.27 $",
			    CNSDLEnvironment::new_object);

// "Type" for type checking functions
CNClassDesc CN_SDLENVIRONMENT = &CNSDLEnvironment_desc;
