/* Memory access checker.
   Copyright 1993 Tristan Gingold
		  Written September 1993 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as   Tristan Gingold
   			  8 rue Parmentier
   			  F91120 PALAISEAU
   			  FRANCE
*/
/* This file defines two important functions: 
 *   chkr_check_addr and chkr_set_right.
 * These functions handle the bitmaps.
 */

#define _MALLOC_INTERNAL
#define NEED_MM
#include "malloc.h"

#include <sys/shm.h>

#include "machine.h"
#include "errlist.h"
#include "message.h"

#ifndef MDCHECKER
#include "maccess.mes"

/* Number of stack frames saved for shm and mmap.  Could become a variable */
#define HISTORY_DEPTH 8

typedef unsigned char uchar;
/* typedef unsigned int uint; */

/* Defined in ../parse-args.c  Can disable some errors */
extern const int chkr_ld_options;

/* True if the text segment is writable: linked with ld -N */
int is_text_writable;

/* Number of memory access error displayed */
static uint nbr_rep_access_error;
/* Number of memory access error found */
static uint nbr_tot_access_error;

/* Differents type of bitmap.
   SegFinish : end of table
   SegVoid :   Can't be accessed (e.g. NULL pointers)
   SegText:    Text segment.
   SegROnly :  Can't be written.
   SegNormal : Normal segment (e.g. heap segment)
   SegRW :     Can be read and write but no bitmap (e.g. data segment)
   SegWOnly:   Can be only written.
   SegMmap:    memory allocated by mmap.
   SegShm:     memory allocated by shm.
 */
typedef enum
  {
    SegFinish, SegVoid, SegText, SegNormal,
    SegRW, SegROnly, SegWOnly, SegMmap, SegShm
  } MapType;

/* Defined latter in this file. */
struct BitMem;
struct BitMemInfo;
struct Mmapinfo;
struct Shminfo;
struct Heapinfo;

/* Anything you want to know about a bitmap */
typedef struct MAPInfo
{
  PTR base;			/* base of the segment */
  PTR real_base;		/* real base of the segment */
  uint length;			/* length of the segment */
  uchar *bmbase;		/* base of the bitmap */
  MapType type;			/* type of the segment */
  const char *name;		/* name of the segment */
  uint right;			/* right (for executing) */
  int (*efunc) (struct BitMem const *bm,
                 struct BitMemInfo *bmi); /* func to call in case of error */
  int (*idfunc) (struct BitMem const *bm,
		 struct BitMemInfo *bmi); /* func to call to name the error */
  /* If the previous infos are not enough, you can use this: */
  union
  {
    struct Mmapinfo *mmapinfo;		/* for SegMmap */
    struct Shminfo *shminfo;		/* for SegShm */
    struct Heapinfo *heapinfo;		/* for heap */
  } info;
#ifdef CHKR_PROFILE
  uint addressed;	/* how many times the bitmap was handled */
#endif
} MapInfo;

/* The Mmap and shm zone have no bitmaps but some other informations:
 * REAL_LENGTH is the length specified by the user; the OS have to round
 *   the length owing to pages size.
 * HISTORY points to a zone where the stack frames are written (usually just
 *   after these structures). 
 * RIGHT is the right the user want.
 */
struct Mmapinfo
{
  uint real_length;
  void *history;
};

struct Shminfo
{
  uint real_length;
  void *history;
  int shmid;
};

struct Heapinfo
{
 struct mdesc *mdp;
};

/* This structure is filled by SetBitMem, which `converts' an address
 *  into a bitmap and an offset in this bitmap.
 */
struct BitMem
{
  uint seg_offset;		/* Offset in the segment */
  uint bm_offset;		/* Offset in the bitmap */
  uint bit_offset;		/* Offset in the byte */
  uint length;			/* How many bytes to the end */
  MapInfo *info;		/* Description of the bitmap */
};

/* This structure is filled by chkr_check_addr when an error is found. */
struct BitMemInfo
{
  PTR ptr;
  int arq;
  int remain;
  int size;
  int val;
};

/* Contains ranges of disabled addresses. */
struct mem_range {
 int first;
 int last;
};
static struct mem_range (*disable)[];
/* Number of range. */
static int disable_cur;
/* Max number of range allowed before having to call sys_realloc(). */
static int disable_max;

/* A pointer to the maps descriptors table. */
static MapInfo *Fmapinfo;
/* Current number of descriptors. */
static uint nbr_mapinfo;
/* Maximum number of descriptors before having to call sys_realloc(). */
static uint max_mapinfo;
/* A pointer to the desciptors of always-defined maps. */
static MapInfo **mapinfo;

/* Here are the offset of these always-defined maps. */
#define NULLBM  0
#define TEXTBM  1
#define DATABM  2
#define STACKBM 3

/* A special descriptor returned by SetBitMem when an address is bad (ie: it
 *   cannot be found in any map). */
static MapInfo null_mapinfo = 
		{(PTR) 0, (PTR) 0, 0, (uchar *) 0, SegFinish, M_NO_MEM};

/* Number of byte(s) regrouped by a state.  Only 1 is tested. */
static int bytes_per_state = 1;
/* Log of bytes_per_state */
static int log_bytes_per_state = 0;
/* Number of bytes per byte of bitmap -1 */
static int bm_round;
/* Number of bytes per byte of bitmap. */
static int log_bytes_per_bbm;

/* The current stack pointer.  It is ajusted by the stubs. */
PTR known_stack_limit;

/* The limit of the bitmap for the stack.  The stub can reduce this value if
 *  the stack use less space.  However, only adjust_bm() can expand it.
 */
PTR stack_bitmapped;

/* The function called by chkr_check_addr() when an error has been found.  */
static int chkr_access_error (const PTR ptr, int arq, struct BitMem const *bm,
			      char val, int size, int len);

/* The function which can expand the stack bitmap. */
static void adjust_bm (void);

/* This function returns TRUE if the error E is disable */
extern int is_error_disable(int e);

#ifdef CHKR_STACKBITMAP
/* Alloc space for the stack bitmap. */
void adjust_stackbitmap (PTR ptr);
void adjust_bm_when_stack_reduce (void);

/* Function called by chkr_access_error() to give more informations. */
static int chkr_access_error_stack (struct BitMem const *bm, 
                                    struct BitMemInfo *bmi);
                                    
#endif /* CHKR_STACKBITMAP */

/* Functions called by chkr_access_error() to give more informations. */
static int chkr_access_error_heap (struct BitMem const *bm, 
                                   struct BitMemInfo *bmi);
static int chkr_access_error_mmap (struct BitMem const *bm,
                                   struct BitMemInfo *bmi);
static int chkr_access_error_shm (struct BitMem const *bm,
                                  struct BitMemInfo *bmi);
static int chkr_access_error_null (struct BitMem const *bm,
                                   struct BitMemInfo *bmi);
static int chkr_access_error_text (struct BitMem const *bm,
                                   struct BitMemInfo *bmi);

#ifdef CHKR_PROFILE
/* Number of times adjust_bm() was called. */
static uint adjust_bm_called = 0;
#endif

/* Returns informations in BITMEM about ADDR: which bitmap, which offset... */
static void
SetBitMem (const PTR addr, struct BitMem *bitmem)
{
  register MapInfo *tmpinfo = Fmapinfo;

#ifdef CHKR_STACKBITMAP
  if (known_stack_limit != stack_bitmapped)
    {
      if (known_stack_limit < stack_bitmapped)
        adjust_bm ();
      else
        adjust_bm_when_stack_reduce();
    }
#endif

  /* Search the good descriptor */
  while (tmpinfo->type != SegFinish)
    {
      /* Test if ADDR is inside the zone specify by TMPINFO. */
      if (!(addr >= tmpinfo->real_base 
          && addr < (tmpinfo->base + tmpinfo->length)))
        {
	  tmpinfo++;
	  continue;
        }
      bitmem->seg_offset = addr - tmpinfo->base;
      bitmem->length = tmpinfo->length - bitmem->seg_offset;
      bitmem->bit_offset = bitmem->seg_offset & bm_round;
      bitmem->bm_offset = bitmem->seg_offset >> log_bytes_per_bbm;
      bitmem->info = tmpinfo;
#ifdef CHKR_PROFILE
      tmpinfo->addressed++;
#endif
      return;
    }
  /* Not found. */
  bitmem->info = &null_mapinfo;
  return;
}

/* Adjust all bitmaps (Useful for the stack).
 *  Alloc more memory for the bitmap if needed.
 *  Set the rights.
 */
static void
adjust_bm (void)
{
#ifdef CHKR_STACKBITMAP
  /* set correctly the stack bitmap */
  int bit_off;
  unsigned char *bm_offset2;
  unsigned char *bm_offset1;
      
  adjust_bm_called++;	/* for profile infos */

  /* Alloc (or dealloc) space for the bitmap */
  /* FIXME: avoid to call this function if not necessary. */
  adjust_stackbitmap (known_stack_limit);
      
  /* Compute the offset. */
  bit_off = ((int) stack_bitmapped) & bm_round;
  bm_offset2 = (char*)MM_STACK - 1 -
	((STACK_LIMIT - (int) stack_bitmapped) >> log_bytes_per_bbm);
  bm_offset1 = (char*)MM_STACK - 1 -
	((STACK_LIMIT - (int) known_stack_limit) >> log_bytes_per_bbm);
  if (bit_off != 0)
    {
      *bm_offset2 &= (bit_off == 1) ? 0xfc :
	  ((bit_off == 2) ? 0xf0 : 0xc0);
      *bm_offset2 |= ((bit_off == 1) ? 0x01 :
	  ((bit_off == 2) ? 0x05 : 0x15)) * CHKR_WO;
      bm_offset2--;
    }
  /* Set the rights */
  if (bm_offset2 > bm_offset1)
    memset (bm_offset1, (int)0x55 * CHKR_WO, (int) bm_offset2 - (int) bm_offset1 + 1);
    
  mapinfo[STACKBM]->real_base = known_stack_limit;
  stack_bitmapped = known_stack_limit;
#endif
}

/* Called by codecheck.S */
void
adjust_bm_when_stack_reduce (void)
{
#if CHKR_STACKBITMAP
  mapinfo[STACKBM]->real_base = known_stack_limit;
  stack_bitmapped = known_stack_limit;
#endif
}

/* Disp the memory map.
 */
/* Used by qsort */
static int
map_compare(const void *a, const void *b)
{
  if ( (*((MapInfo**)a))->base > (*((MapInfo**)b))->base )
    return 1;
  else if ( (*((MapInfo**)a))->base == (*((MapInfo**)b))->base )
    return 0;
  else
    return -1;
}

void
__chkr_disp_map(void)
{
  static char *right_name[] = { "--", "r-", "-w", "rw"};
  MapInfo **buf;
  int nbr_map = nbr_mapinfo - 1;
  int i;
  PTR base = (PTR*)0;
  
  /* Fill a tab with the addresses of the segments and sort it. */
  buf = (MapInfo**) sys_malloc (nbr_map * sizeof (MapInfo*));
  for (i = 0; i < nbr_map; i++)
    buf[i] = &Fmapinfo[i];
  qsort(buf, nbr_map, sizeof(MapInfo*), map_compare);
  
  chkr_header("Memory map:\n");
  chkr_printf("Name           | Base       | End        | Size        | Right |\n");
  chkr_printf("---------------+------------+------------+-------------+-------+\n");
  for (i = 0; i < nbr_map; i++)
    {
      if (buf[i]->real_base > base)
        chkr_printf("%14s | 0x%-8x | 0x%-8x | %8d kb | ---   |\n", 
                  "  (nothing)",
                  base, 
                  buf[i]->real_base,
                  ((uint)buf[i]->real_base - (uint)base) / 1024);
      base = buf[i]->base  + buf[i]->length + 1;
      chkr_printf("%14s | 0x%-8x | 0x%-8x | %8d kb | %s%c   |\n", 
                  buf[i]->name,
                  buf[i]->real_base, 
                  buf[i]->base + buf[i]->length,
                  ((uint)(buf[i]->base + buf[i]->length) - (uint)buf[i]->real_base) / 1024,
                  right_name[buf[i]->right & CHKR_MASK],
                  buf[i]->right & CHKR_EXEC ? 'x' : '-');
    }
  sys_free(buf);
}

/* Return the size of the memory used. */
uint
get_total_mem(void)
{
  uint total = 0;
  int i;
  
  for (i = 0; i < nbr_mapinfo; i++)
    {
      total += Fmapinfo[i].length 
               - ((uint)Fmapinfo[i].real_base - (uint)Fmapinfo[i].base);
    }
  return total;
}

/* Disp the bitmap from addr ptr.  LEN is the number of bytes.
 */
void
__chkr_disp_right (const PTR ptr, int len)
{
  struct BitMem bitmem;
  int bm_offset;
  uchar *base;
  int val;
  int i, j;
  uint addr = (int) ptr & (~bm_round);	/* round ptr */

  /* Search info about the bitmap */
  SetBitMem ((PTR) addr, &bitmem);
  base = bitmem.info->bmbase;
  bm_offset = bitmem.bm_offset;

  /* Skip bad addresses. */
  if (base == (uchar *) 0)
    {
      chkr_printf (M_NO_BM_4ADDRESS);
      return;
    }
  if (bitmem.info->type != SegNormal)
    {
      chkr_printf (M_NO_BM_4ADDRESS);
      return;
    }
    
  if (bitmem.bit_offset != 0)
    chkr_printf (M_BO_NOT_ALIGNED);

  addr -= bitmem.bit_offset << log_bytes_per_state;

  /* print the header */
  chkr_printf ("          ");
  for (i = 0; i < 16; i++)
    chkr_printf ("%02x ", ((addr & 0x0f) + (i << log_bytes_per_state)) &
		 (0x0f << log_bytes_per_state));
  chkr_printf ("\n");

  for (i = 0; len > 0; i++)
    {
      /* address of the line */
      chkr_printf ("%08x: ", addr + i * (16 << log_bytes_per_state));
      /* bitmap */
      /* val = ((unsigned int*)base)[bm_offset]; */
      val = base[bm_offset] + (base[bm_offset + 1] << 8) + (base[bm_offset + 2] << 16)
	    + (base[bm_offset + 3] << 24);
      bm_offset += 4;
      for (j = 0; j < 16; j++)
        {
	  if ((addr + i * (16 << log_bytes_per_state) + j) >= (uint) ptr)
	    {
	      chkr_printf ("%s ", short_right_name[val & CHKR_MASK]);
	      len--;
	    }
	  else
	    chkr_printf ("?? ");
	  val >>= 2;
	}
      chkr_printf ("\n");
    }
}

/* Test the execution right */
void
chkr_check_exec (const PTR ptr)
{
  struct BitMem bitmem;
  SetBitMem (ptr, &bitmem);
  if (!(bitmem.info->right & CHKR_EXEC) && chkr_report(M_E_ZNE_VD_ET))
    {
      chkr_printf (M_TRY_TO_EXECUTE, ptr, bitmem.info->name);
      chkr_printf (M_PROD_SEG_FAULT);
#ifdef CHKR_SAVESTACK
      chkr_frames_to_forget = 1;
      chkr_show_frames ();
      chkr_remove_symtabfile ();
#endif /* CHKR_SAVESTACK */
    }
}

/* Bit tabs used by chkr_set_right and chkr_check_addr */
static const uchar tab_beg_offset[4] =
{0x00, 0x54, 0x50, 0x40};
static const uchar tab_end_offset[4] =
{0x00, 0x01, 0x05, 0x15};
static const uchar val_right[4] = 
{0x55 * CHKR_UN, 0x55 * CHKR_RO, 0x55 * CHKR_WO, 0x55 * CHKR_RW};

/* Set the right of a zone of length LEN, starting at PTR */
void
chkr_set_right (const PTR ptr, int len, int right)
{
  struct BitMem bitmem;
  uchar *base;			/* Bitmap base */
  uchar val = 0;		/* Val to set in the bitmap */
  uchar mask = 0;		/* Mask corresponding to the val */

  /* Search info about ptr */
  SetBitMem (ptr, &bitmem);
  base = bitmem.info->bmbase;

  /* Be sure about right.  Right is 2 bits length. */
  right &= CHKR_MASK;

  /* When the program contains standard file, Checker is B&W ! */
  if (chkr_ld_options & 1)
    right = (right != CHKR_UN) ? CHKR_RW : CHKR_UN;

  switch (bitmem.info->type)
    {
    /* The rights of these segments can't be changed. */
    case SegFinish:
    case SegVoid:
    case SegText:
    case SegMmap:
    case SegShm:
      chkr_printf (M_ERR_SET_RIGHT);
      return;
    case SegRW:
      return;
    case SegROnly:
      if (right & CHKR_WO)
	chkr_printf (M_ERR_SET_RIGHT);
      return;
    case SegWOnly:
      if (right & CHKR_RO)
	chkr_printf (M_ERR_SET_RIGHT);
      return;
    case SegNormal:
      if (len > bitmem.length || len == 0)
	{
	  chkr_printf (M_BAD_LEN_IN_SR);
	  return;
	}
      /* The processus is split into 3 parts: the first byte of rights,
       *  the middle and the last.  This is due to alignment. */
      /* The first byte of right */
      if (bitmem.bit_offset > 0)
	{
	  /* len must be divided */
	  if ((len >> log_bytes_per_state) + bitmem.bit_offset < bm_round + 1)
	    {
	      /* Only one byte of right. */
	      val = (tab_beg_offset[bitmem.bit_offset] -
		     tab_beg_offset[bitmem.bit_offset + (len >> log_bytes_per_state)]) * right;
	      mask = (tab_beg_offset[bitmem.bit_offset] -
		      tab_beg_offset[bitmem.bit_offset + (len >> log_bytes_per_state)]) * CHKR_RW;
	      len = 0;		/* finish */
	    }
	  else
	    {
	      val = tab_beg_offset[bitmem.bit_offset] * right;
	      mask = tab_beg_offset[bitmem.bit_offset] * CHKR_RW;
	      len -= (bm_round + 1) - (bitmem.bit_offset << log_bytes_per_state);
	    }
	  mask = ~mask;
	  base[bitmem.bm_offset] &= mask;
	  base[bitmem.bm_offset] |= val;
	  /* If finish, then return */
	  if (len <= 0)
	    return;
	  /* next byte to set */
	  bitmem.bm_offset++;
	}
	
      /* Now, we set bytes of bitmap. So len is decrease by bm_round + 1 */
      if (len > bm_round)
	{
	  val = val_right[right]; /* = 0x55 * right; */
	  while (len > bm_round)
	    {
	      base[bitmem.bm_offset++] = val;
	      len -= bm_round + 1;
	    }
	  if (len <= 0)
	    return;
	}
      /* Now, the end. Set the last byte of bitmap */
      val = tab_end_offset[len] * right;
      mask = tab_end_offset[len] * CHKR_RW;
      mask = ~mask;
      base[bitmem.bm_offset] &= mask;
      base[bitmem.bm_offset] |= val;
      return;
    }
}

/* Check the right of a zone of length len, starting at ptr */
/*				rights in bitmap
 *                   +-----------+-------------+-------------+-------------+
 *  r                |  CHKR_UN  |   CHKR_RO   |   CHKR_WO   |   CHKR_RW   |
 *  i    +-----------+-----------+-------------+-------------+-------------+
 *  g    |  CHKR_UN  |                   not accepted                      |
 *  h    +-----------+-----------+-------------+-------------+-------------+
 *  t    |  CHKR_RO  |   error   |     OK      |    error    |     OK      |
 *       +-----------+-----------+-------------+-------------+-------------+
 *  t    |  CHKR_WO  |   error   |    error    | -> CHKR_RW  |     OK      |
 *  e    +-----------+-----------+-------------+-------------+-------------+
 *  s    |  CHKR_TW  |   error   |    error    |     OK      |     OK      |
 *  t    +-----------+-----------+-------------+-------------+-------------+
 *  e    |  CHKR_RW  |   error   |    error    |    error    |     OK      |
 *  d    +-----------+-----------+-------------+-------------+-------------+
 */
void
chkr_check_addr (const PTR ptr, int len, int right1)
{
  struct BitMem bitmem;
  uchar *base;			/* Bitmap base */
  uchar val = 0;		/* Val to set in the bitmap */
  uchar mask;
  int size = len;		/* save */
  int right;			/* Good Value of right1 */

  /* Be sure about right.  Right is 2 bits length. */
  right = right1 & CHKR_MASK;

  /* Checks for stupidities.  Could be removed. */
  if (right == 0)
    {
      chkr_perror (M_I_IEB_MA_ET); /* Bad check, must never happen.  */
      chkr_abort ();
    }
  /* This test must be done only for syscalls, otherwise it is an error. */
  if (len == 0)
    return;		/* Can be used by syscall, such as read().  FIXME */

  /* Search info about ptr */
  SetBitMem (ptr, &bitmem);
  base = bitmem.info->bmbase;

  switch (bitmem.info->type)
    {
    case SegNormal:
      if (len > bitmem.length)
        {
          chkr_perror (M_A_SBV_SG_ET);
          return;
        }
      if (bitmem.bit_offset > 0)
	{
	  /* len must be divided */
	  if ((len >> log_bytes_per_state) + bitmem.bit_offset < bm_round + 1)
	    {
	      val = (tab_beg_offset[bitmem.bit_offset] -
		     tab_beg_offset[bitmem.bit_offset + (len >> log_bytes_per_state)]) * right;
	      mask = (tab_beg_offset[bitmem.bit_offset] -
		      tab_beg_offset[bitmem.bit_offset + (len >> log_bytes_per_state)]) * CHKR_RW;
	      len = 0;		/* finish */
	    }
	  else
	    {
	      val = tab_beg_offset[bitmem.bit_offset] * right;
	      mask = tab_beg_offset[bitmem.bit_offset] * CHKR_RW;
	      len -= (bm_round + 1) - (bitmem.bit_offset << log_bytes_per_state);
	    }
	  if ((base[bitmem.bm_offset] & val) != val)
	    {
	      /* len = size for chkr_access_error(), since len was decreased
              * too early. */
	      if (chkr_access_error (ptr, right, &bitmem, val, size, size))
		return;		/* check fails */
	    }
	  if (right1 == CHKR_WO)
	    base[bitmem.bm_offset] |= mask;	/* CHKR_WO -> CHKR_RW */
	  /* If finish, then return */
	  if (len <= 0)
	    return;
	  /* next byte to set */
	  bitmem.bm_offset++;
	}
      /* Now, we set bytes of bitmap. So len is decrease by bm_round + 1 */
      if (len > bm_round)
	{
	  val = val_right[right]; /* = 0x55 * right; */
	  while (len > bm_round)
	    {
	      if ((base[bitmem.bm_offset] & val) != val)
		{
		  if (chkr_access_error (ptr, right, &bitmem, val, size, len))
		    return;	/* check fails */
		}
	      if (right1 == CHKR_WO)
		base[bitmem.bm_offset] |= 0x55 * CHKR_RW;
	      bitmem.bm_offset++;
	      len -= bm_round + 1;
	    }
	  if (len <= 0)
	    return;
	}
      /* Now, the end. Set the last byte of bitmap */
      val = tab_end_offset[len] * right;
      if ((base[bitmem.bm_offset] & val) != val)
	{
	  if (chkr_access_error (ptr, right, &bitmem, val, size, len))
	    return;		/* check fails */
	}
      if (right1 == CHKR_WO)
	base[bitmem.bm_offset] |= tab_end_offset[len] * CHKR_RW;
      return;

    case SegFinish:
    case SegVoid:
      chkr_access_error (ptr, right, &bitmem, 0, size, 0);
      return;			/* always return */

    case SegText:
      if (!is_text_writable)
	if (right != CHKR_RO)
	  chkr_access_error (ptr, right, &bitmem, CHKR_RO * 0x55, size, 0);
      return;			/* always return */

    case SegRW:
      return;
      
    case SegMmap:
      /* FIXME: cross */
      if (len > bitmem.length)
        {
          chkr_perror (M_A_SBV_SG_ET);
          return;
        }
      if (ptr >= bitmem.info->base + bitmem.info->info.mmapinfo->real_length)
        chkr_access_error (ptr, right, &bitmem, 0, size, 0);
      if ((right & bitmem.info->right) != right)
        chkr_access_error (ptr, right, &bitmem, 0, size, 0);
      return;
    
    case SegShm:
      if ((right & bitmem.info->right) != right)
        chkr_access_error (ptr, right, &bitmem, 0, size, 0);
      return;

    case SegROnly:
    case SegWOnly:
      chkr_abort ();		/* Not implemented */
    }
}

/* Check for a string.
 * Is this implementation poor ?
 */
void
chkr_check_str (const PTR ptr, int right)
{
  chkr_check_addr (ptr, strlen(ptr) + 1, right);
}

/* Copy a part of bitmap.  Must be aligned.
 * This is used by realloc.
 */
void
chkr_copy_aligned_bm (PTR dest, PTR src, uint len)
{
  struct BitMem bmsrc, bmdest;
  static uchar tab_offset[] =
  {0x00, 0x03, 0x0f, 0x3f};
  uint bm_len;

  if (len == 0)
    return;
    
  SetBitMem (src, &bmsrc);
  if (bmsrc.info->type != SegNormal || bmsrc.length < len)
    {
      chkr_printf (M_SRC_IN_CP_BM);
      return;
    }
  SetBitMem (dest, &bmdest);
  if (bmdest.info->type != SegNormal || bmdest.length < len)
    {
      chkr_printf (M_DEST_IN_CP_BM);
      return;
    }
  if (bmsrc.bit_offset != 0 || bmdest.bit_offset != 0)
    {
      chkr_printf (M_ALIGN_IN_CP_BM);
      return;
    }
  bm_len = len / (bm_round + 1);
  memcpy (&(bmdest.info->bmbase[bmdest.bm_offset]),
	  &(bmsrc.info->bmbase[bmsrc.bm_offset]),
	  bm_len);
  if ((len & bm_round) == 0)
    return;
  bmdest.info->bmbase[bmdest.bm_offset + bm_len] &= ~(tab_offset[len & bm_round]);
  bmdest.info->bmbase[bmdest.bm_offset + bm_len] |=
    tab_offset[len & bm_round] & bmsrc.info->bmbase[bmsrc.bm_offset + bm_len];
}

/* Copy a part of bitmap. Can be not aligned.
 */
void
chkr_copy_bitmap (PTR dest, PTR src, uint len)
{
  struct BitMem bmsrc, bmdest;
  static uchar tab_offset[] =
  {0x01, 0x04, 0x10, 0x40};
  uchar right = 0;
  
  if (len == 0)
    return;
    
  SetBitMem (src, &bmsrc);
  if (bmsrc.length < len)
    {
      chkr_printf (M_SRC_IN_CP_BM);
      return;
    }
  switch (bmsrc.info->type)
    {
      case SegFinish:
      case SegVoid:
        chkr_printf (M_SRC_IN_CP_BM);
        return;
      case SegText:
        if (is_text_writable)
          right = CHKR_RW;
        else
          right = CHKR_RO;
        break;
      case SegRW:
        right = CHKR_RW;
        break;
      case SegROnly:
        right = CHKR_RO;
        break;
      case SegWOnly:
        right = CHKR_WO;
        break;
      case SegMmap:
        right = bmsrc.info->right;
        break;
      case SegShm:
        right = bmsrc.info->right;
        break;
      case SegNormal:
        break;
    }
  right &= CHKR_MASK;
  
  SetBitMem (dest, &bmdest);
  if (bmdest.length < len)
    {
      chkr_printf (M_DEST_IN_CP_BM);
      return;
    } 
  switch (bmdest.info->type)
    {
      case SegFinish:
      case SegVoid:
        chkr_printf (M_DEST_IN_CP_BM);
        return;
      case SegText:
        if (!is_text_writable)
          chkr_printf (M_DEST_IN_CP_BM);
        return;
      case SegRW:
      case SegWOnly:
      case SegShm:
      case SegMmap:
      case SegROnly:
        return;
      case SegNormal:
        break;
    }
  len >>= log_bytes_per_state;
  while (len > 0)
    {
      if (bmsrc.info->type == SegNormal)
        {
          /* Read a right */
          right = bmsrc.info->bmbase[bmsrc.bm_offset] & tab_offset[bmsrc.bit_offset];
          right >>= 2 * bmsrc.bit_offset;
          right &= CHKR_MASK;
          /* Update for the next right */
          if (bmsrc.bit_offset == 3)
            {
              bmsrc.bit_offset = 0;
              bmsrc.bm_offset++;
            }
          else
            bmsrc.bit_offset++;
        }
      /* Write the right */
      right <<= 2 * bmdest.bit_offset;
      bmdest.info->bmbase[bmdest.bm_offset] &= ~tab_offset[bmdest.bit_offset];
      bmdest.info->bmbase[bmdest.bm_offset] |= right;
      /* Update for the next right */
      if (bmdest.bit_offset == 3)
        {
          bmdest.bit_offset = 0;
          bmdest.bm_offset++;
        }
      else
        bmdest.bit_offset++;
      len--;
    }
}

#ifdef CHKR_PROFILE
/* Display profile.  Called by chkr_do_end() only is profile_flag is true. */
void
display_profile ()
{
  register MapInfo *tmpinfo = Fmapinfo;
  unsigned int total = 0;

  chkr_header (M_PROF_INFO_HERE);
  while (tmpinfo->type != SegFinish)
    {
      chkr_printf (M_HANDLED_N_TIME, tmpinfo->name,
		   tmpinfo->addressed, (tmpinfo->addressed > 1) ? "s" : "");
      total += tmpinfo->addressed;
      tmpinfo++;
    }
  chkr_printf (M_HANDLED_MEM, total);
  chkr_printf (M_BM_CALLED, adjust_bm_called);
  chkr_printf (M_ACC_ERR_REP, nbr_rep_access_error);
  chkr_printf (M_ACC_ERR_TOT, nbr_tot_access_error);
}
#endif

/* Disable errors.  STR is an address or a range. STR can be modified.
 *  Called by error.c
 */
int
parse_disable_range (char *str)
{
  /* STR is either an address or a range. */
  char *c;
  int low, high;

  c = strchr(str, '-');
  if (c == (char*)0)
    low = high = atod(str);	/* An address. */
  else
    {
      /* A range. */
      *c = '\0';
      low = atod(str);
      high = atod(c+1);
    }
  if (disable_cur + 1 > disable_max)
    {
      /* The table named disable must be expanded. */
      disable = sys_realloc(disable, 
                      (disable_max + 16) * sizeof(struct mem_range));
      disable_max += 16;
    }
  (*disable)[disable_cur].first = low;
  (*disable)[disable_cur].last = high;
  disable_cur++;
  return 1;	/* OK */
}  

/* Functions for Mmap */
/* This function is a member of a descriptor and is used to specify the
 * error.  Called only by chkr_access_error(). */
static int
ename_mmapseg (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  /* If PTR is after the end of the zone, this is a violation. */
  if (bmi->ptr >= (bm->info->base + bm->info->info.mmapinfo->real_length))
    return M_U_BVM_MM_ET;
  /* if PTR + SIZE ... */
  if ((bmi->ptr + bmi->size) > (bm->info->base + bm->info->info.mmapinfo->real_length))
    return M_U_BVM_MM_ET;
  /* Otherwise, the user has not this right. */
  return M_U_APD_MM_ET;
}

/* Just add a new descriptor at the end of the list (but before the last).
 * It returns an index for mapinfo[].  The mapinfo[] table is updated.
 */
static int
add_slot(void)
{
  int shift;		/* To update the tab */
  int i;
  MapInfo *old_Fmapinfo;
  
  if (nbr_mapinfo == max_mapinfo)
    {
      max_mapinfo += 4;
      old_Fmapinfo = Fmapinfo;
      Fmapinfo = (MapInfo *) sys_realloc (Fmapinfo, max_mapinfo * sizeof (MapInfo));
      shift = (uint)Fmapinfo - (uint)old_Fmapinfo;
      mapinfo = (MapInfo **) sys_realloc (mapinfo, max_mapinfo * sizeof (MapInfo*));
      /* Update the always-defined pointers. */
      for (i = 0; i < nbr_mapinfo; i++)
        if (mapinfo[i])
          mapinfo[i] = (PTR)mapinfo[i] + shift;
      /* Initialize the other pointers. */
      for (i = nbr_mapinfo; i < max_mapinfo; i++)
        mapinfo[i] = NULL;
    }
  Fmapinfo[nbr_mapinfo] = Fmapinfo[nbr_mapinfo-1];
  nbr_mapinfo++;
  /* Find a free slot in mapinfo[] */
  for (i = 0; i < max_mapinfo; i++)
    if (mapinfo[i] == NULL)
      break;
  mapinfo[i] = &Fmapinfo[nbr_mapinfo - 2];
  return i;
}

/* Just remove a descriptor. */
static void
remove_slot(MapInfo *tinfo)
{
  int i;
  
  for (i = 0; i < max_mapinfo; i++)
    if (mapinfo[i] == tinfo)
      {
        mapinfo[i] = NULL;
        break;
      }
  *tinfo = Fmapinfo[nbr_mapinfo-2];
  Fmapinfo[nbr_mapinfo-2] = Fmapinfo[nbr_mapinfo-1];
  for (i = 0; i < nbr_mapinfo; i++)
    if (mapinfo[i] == &Fmapinfo[nbr_mapinfo - 2])
      {
        mapinfo[i] = tinfo;
        break;
      }
  nbr_mapinfo--;
}

/* Split a mmap zone into the old and shorter zone and a new zone.
 * Create a new descriptor. */
static int 
split_segmmap(MapInfo *tinfo, uint len)
{
  int slot;
  slot = add_slot();
  memcpy(mapinfo[slot], tinfo, sizeof(MapInfo));
  /* Duplicate the history. */
  mapinfo[slot]->info.mmapinfo = sys_dupalloc(tinfo->info.mmapinfo);
  if (tinfo->length <= len)
    chkr_abort();
  mapinfo[slot]->base += len;
  mapinfo[slot]->length -= len;
  tinfo->length -= len;
  return slot;
}

/* Called usually by munmap() and new_segmap() when an mmap zone has
 *  desappeared.  The history is also forgotten FIXME. */
void
remove_mmap(PTR addr, uint len)
{
  MapInfo *tmpinfo = Fmapinfo;
  uint round_len = ( len + PAGESIZE - 1) & (~(PAGESIZE-1));
  int slot;

  while (tmpinfo->type != SegFinish)
    if (tmpinfo->type == SegMmap)
      {
        /* The zone removed can: */
        /*  * recover a zone */ 
        if (addr <= tmpinfo->base 
            && (addr + round_len) >= (tmpinfo->base + tmpinfo->length))
          {
            /* Remove this mapinfo (aka descriptor). */
            sys_free(tmpinfo->info.mmapinfo);
            remove_slot(tmpinfo);
            /* No tmpinfo++ */
            continue;
          }
        /*  * recover the beginning of a zone */
        if (addr <= tmpinfo->base
            && (addr + round_len) > tmpinfo->base
            && (addr + round_len) < (tmpinfo->base + tmpinfo->length))
          {
            tmpinfo->length -= addr + round_len - tmpinfo->base;
            tmpinfo->base = addr + round_len;
            tmpinfo++;
            continue;
          }
        /*  * recover the end of a zone */
        if (addr > tmpinfo->base
            && addr < (tmpinfo->base + tmpinfo->length)
            && (addr + round_len) >= (tmpinfo->base + tmpinfo->length)) 
          {
            tmpinfo->length = addr - tmpinfo->base;
            tmpinfo++;
            continue;
          }
        /*  * be inside a zone */
        if (addr > tmpinfo->base
            && (addr + round_len) < (tmpinfo->base + tmpinfo->length))
          {
            /* In this case, the zone must be split. */
            slot = split_segmmap(tmpinfo, addr - tmpinfo->base);
            mapinfo[slot]->base = addr + round_len;
            mapinfo[slot]->length -= round_len;
            tmpinfo++;
            continue;
          }
      }
    else
      tmpinfo++;
}

/* Create a new descriptor for a mmap zone.  Called just after mmap(). */
void
new_segmmap(PTR addr, uint len, int prot, int flags, int filedes, uint off)
{
  uint round_len = ( len + PAGESIZE - 1) & ~(PAGESIZE-1);
  struct Mmapinfo *minfo;
  int slot;

  if (len == 0)
    return;	/* Nothing to do */
   
  /* Remove the zones recovered. */
  remove_mmap(addr, round_len);
  
  /* Add a new descriptor and fill it. */
  slot = add_slot();
  mapinfo[slot]->name = M_MMAP_SEGMENT;
  mapinfo[slot]->base = addr;
  mapinfo[slot]->real_base = addr;
  mapinfo[slot]->length =  round_len;
  mapinfo[slot]->bmbase = (uchar *) 0;
  mapinfo[slot]->type = SegMmap;
  mapinfo[slot]->idfunc = ename_mmapseg;
  mapinfo[slot]->efunc = chkr_access_error_mmap;
  
  /* Alloc space for more infos and the history. */
  minfo = (struct Mmapinfo*) sys_malloc(sizeof(struct Mmapinfo) + HISTORY_DEPTH * sizeof(void*));
  minfo->history = (PTR*)(minfo + 1);
#ifdef CHKR_SAVESTACK
  chkr_save_stack(minfo->history, 3, HISTORY_DEPTH);
#endif
  mapinfo[slot]->info.mmapinfo = minfo;
  
  /* compute the len */
  if (!(flags & MAP_ANONYMOUS))
    {
      /* The memory mapped is a file.  The real length is determined by the
       *  file length. */
      struct stat buf;
      
      fstat(filedes, &buf);
      if (len + off > buf.st_size)
        {
          if (off > buf.st_size)
            len = 0;	/* Must never happen. */
          else
            len = buf.st_size - off;	/* Set the new real length. */
        }
    }
  minfo->real_length = len;
  /* Set the right according to the args. */
  mapinfo[slot]->right = (prot & PROT_READ ? CHKR_RO : 0)
                         | (prot & PROT_WRITE ? CHKR_WO : 0)
                         | (prot & PROT_EXEC ? CHKR_EXEC : 0);
}

/* For the heap. */
static int
ename_heap (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  struct malloc_header *block;
  PTR ptr;
  int mask;
  int i;

  /* If there is no heap, this is really strange. */
  if (_firstmdesc == NULL_MDESC)
    return M_IE_ET;
    
  /* PTR contains the exact address. */
  ptr = bmi->ptr + bmi->size - bmi->remain;
  /* The good right in the bitmap must be watched: we can have this:
   * RW RW RW RO
   * ^
   * PTR
   */
  if (bmi->size > 1)
    {
      mask = 3;
      for (i = 0; i < 4; i++)
	{
	  if (((bm->info->bmbase)[bm->bm_offset] & (bmi->val & mask))
	      != (bmi->val & mask))
	    {
	      ptr += i << log_bytes_per_state;
	      mask = 0;		/* a flag */
	      break;
	    }
	  mask = (mask << 2) | 3;
	}
      if (i == 3 && mask)
	chkr_abort ();
      if (bmi->size == bmi->remain)
	ptr -= bm->bit_offset << log_bytes_per_state;
    }

  block = find_header ((struct mdesc*)0, ptr, 0);
  if (block == NULL_HEADER)
    return M_U_BVH_HE_ET;
  switch (block->state)
    {
    case MDFREE:
    case MDAGED:
      switch (bmi->arq)
	{
	case CHKR_RO:
	  return M_R_RFB_HE_ET;	/* Read inside a Free Block. */
	case CHKR_RW:
	case CHKR_WO:
	  return M_W_WFB_HE_ET;	/* Write inside a Free Block. */
	default:
	  return M_IE_ET;	/* Internal Error. */
	}
    case MDBRK:
      return M_IE_ET;		/* BRK zone are always RW. */
    case MDBUSY:
      switch (bmi->arq)
	{
	case CHKR_RW:
	case CHKR_RO:
	  return M_R_RUH_HE_ET;	/* Read Unitialized data. */
	default:
	  return M_IE_ET;
	}
    default:
      return M_IE_ET;
    }
}

/* Search a free zone between MM_LOW and MM_HIGH which may contains a bitmap.
 *  BASE is used to compute a first address.  While this address is bad, we
 *  look further. */
char *
find_bmbase(PTR base)
{
  PTR res;
  int i;
  int npages;
#define PAGE_ALIGN(x)  (((unsigned int)(x) + PAGESIZE - 1) & ~(PAGESIZE-1))
  res = (PTR)MM_HEAP + PAGE_ALIGN (((uint)base - objects->bss_end) >> log_bytes_per_bbm);
 again:
  if (res < (PTR)MM_LOW || res > (PTR)MM_HIGH)
    chkr_abort();
  for (i = 0; i < nbr_mapinfo; i++)
    {
      if (Fmapinfo[i].type != SegNormal)
        continue;
      npages = PAGE_ALIGN(Fmapinfo[i].length >> log_bytes_per_bbm);
      if (res >= (PTR)Fmapinfo[i].bmbase && res <= ((PTR)Fmapinfo[i].bmbase + npages))
        {
          res += npages;
          goto again;
        }
    }
  if (res < (PTR)MM_LOW || res > (PTR)MM_HIGH)
    chkr_abort();
  return res;
}

/* Add a new descriptor for an heap.
 * If BRK is true, this is an heap based on sbrk().  This arg is only used to
 *   set the name of the segment.
 * BMBASE is the base of the bitmap. 
 * It return a slot number for mapinfo[]. */
int
new_heap(int brk, struct mdesc *mdp)
{
  MapInfo *res;
  int slot = add_slot();
  res = mapinfo[slot];
  res->info.heapinfo = sys_malloc(sizeof(struct Heapinfo));
  res->info.heapinfo->mdp = mdp;
  res->name = brk ? M_HEAP_BRK_SEGMENT : M_HEAP_MAP_SEGMENT;
  res->base = mdp->base;
  res->real_base = mdp->base;
  res->length = mdp->breakval - mdp->base;
  res->bmbase = mdp->info.inmem.bitmap->base;
  res->type = SegNormal;
  res->efunc = chkr_access_error_heap;
  res->idfunc = ename_heap;
  res->right = CHKR_RW;  
  return slot;
}

/* Remove an heap. */
void
remove_heap(int slot)
{
  MapInfo *map = mapinfo[slot];
  sys_free(map->info.heapinfo);
  remove_slot(map);
}

/* Functions for shm */
/* Function to specify an error. */
static int
ename_shmseg (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  /* FIXME */
  return M_U_WRS_SH_ET;
}

/* Create a new descriptor for a shm (SHared Memory). */
void
new_segshm(int shmid, PTR addr, int flags)
{
  struct Shminfo *sinfo;
  struct shmid_ds shmbuf;
  int slot;

  /* Used to know the size of the zone. */
  shmctl(shmid, IPC_STAT, &shmbuf);
  
  /* Create a new descriptor and fill it. */
  slot = add_slot();  
  mapinfo[slot]->name = M_SHM_SEGMENT;
  mapinfo[slot]->base = addr;
  mapinfo[slot]->real_base = addr;
  mapinfo[slot]->length = shmbuf.shm_segsz;
  mapinfo[slot]->bmbase = (uchar *) 0;
  mapinfo[slot]->type = SegShm;
  mapinfo[slot]->idfunc = ename_shmseg;
  mapinfo[slot]->efunc = chkr_access_error_shm;
  /* Alloc space for more infos and the history. */
  sinfo = (struct Shminfo*) sys_malloc(sizeof(struct Shminfo) + HISTORY_DEPTH * sizeof(void*));
  sinfo->history = (PTR*)(sinfo + 1);
#ifdef CHKR_SAVESTACK
  chkr_save_stack(sinfo->history, 3, HISTORY_DEPTH);
#endif  
  mapinfo[slot]->info.shminfo = sinfo;
  sinfo->real_length = shmbuf.shm_segsz;
  mapinfo[slot]->right = (flags & SHM_RDONLY ? CHKR_RO : CHKR_RW);
  sinfo->shmid = shmid;
}

/* Remove a descriptor of shm. */
void
remove_shm(PTR addr)
{
  MapInfo *tmpinfo = Fmapinfo;

  while (tmpinfo->type != SegFinish)
    if (tmpinfo->type == SegShm && tmpinfo->base == addr)
      {
        /* Free the additional infos. */
        sys_free(tmpinfo->info.shminfo);
        remove_slot(tmpinfo);
        break;
      }
    else
      tmpinfo++;
}

/* Functions to identify error types. */
/* For the NULL zone. */
static int
ename_null (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  switch (bmi->arq)
    {
    case CHKR_RO:
    case CHKR_WO:
    case CHKR_RW:
      return M_U_NZA_NZ_ET;
    default:
      return M_IE_ET;	/* Internal Error. */
    }
}

/* For the text segment. */
static int
ename_textseg (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  switch (bmi->arq)
    {
    case CHKR_WO:
    case CHKR_RW:
      return M_W_WOB_TE_ET;
    default:
      return M_IE_ET;
    }
}

/* For the data segment. */
static int
ename_dataseg (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  return M_IE_ET;		/* Internal Error at this time */
}

/* For the stack segment. */
static int
ename_stack (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  int mask;
  int fl = 0;
  int i;

  /* PTR contains the exact addr.  See ename_heap(). */
  mask = 0xc0;
  for (i = 0; i < 4; i++)
    {
      if (((bm->info->bmbase)[bm->bm_offset] & (bmi->val & mask))
	  != (bmi->val & mask))
	{
	  fl = 1;
	  break;
	}
      mask >>= 2;
    }
  if (!fl)
    chkr_abort ();

  fl = (bm->info->bmbase)[bm->bm_offset] & mask;
  while (mask != 3)
    {
      mask >>= 2;
      fl >>= 2;
    }
  /* now fl contains the right of the error */
  switch (bmi->arq)
    {
    case CHKR_RO:
      switch (fl)
	{
	case CHKR_WO:
	  return M_R_RUS_ST_ET;		/* Read Uninitialized bytes. */
	case 0:
	  return M_R_RZS_ST_ET;		/* Read in a red Zone. */
	default:
	  return M_IE_ET;
	}
    case CHKR_WO:
      if (fl == 0)
	return M_W_WZS_ST_ET;		/* Write into a red Zone. */
      else
	return M_IE_ET;
    case CHKR_RW:
      switch (fl)
	{
	case CHKR_WO:
	  return M_M_WUS_ST_ET;		/* Modify */
	case 0:
	  return M_M_MZS_ST_ET;
	default:
	  return M_IE_ET;
	}
    default:
      return M_IE_ET;
    }
}

/* Initialize the informations for the descriptors. */
static void
init_mapinfos (void)
{
  int i;
  unsigned int text_beg;
  
  max_mapinfo = 6;
  
  Fmapinfo = (MapInfo *) sys_malloc (max_mapinfo * sizeof (MapInfo));
  mapinfo = (MapInfo**) sys_malloc (max_mapinfo * sizeof (MapInfo*));
  for (i = 0; i < max_mapinfo; i++)
    mapinfo[i] = NULL;
  
  /* Set the always-defined pointers. */
  i = 0;
#ifdef CHKR_STACKBITMAP
  mapinfo[STACKBM] = &Fmapinfo[i++];
#endif
#if 0
#ifdef CHKR_HEAPBITMAP
  mapinfo[HEAPBM] = &Fmapinfo[i++];
#endif
#endif
#ifdef CHKR_DATABITMAP
  mapinfo[DATABM] = &Fmapinfo[i++];
#endif
  mapinfo[TEXTBM] = &Fmapinfo[i++];
  mapinfo[NULLBM] = &Fmapinfo[i++];

  Fmapinfo[i].type = SegFinish;
  
  nbr_mapinfo = i + 1;
  text_beg = objects->text_beg ? objects->text_beg : null_pointer_zone;

  /* NULL zone */
  mapinfo[NULLBM]->name = M_NULL_ZONE;
  mapinfo[NULLBM]->base = (PTR) 0x0;
  mapinfo[NULLBM]->real_base = (PTR) 0x0;  
  mapinfo[NULLBM]->length = text_beg;
  mapinfo[NULLBM]->bmbase = (uchar *) 0;
  mapinfo[NULLBM]->type = SegVoid;
  mapinfo[NULLBM]->idfunc = ename_null;
  mapinfo[NULLBM]->efunc = chkr_access_error_null;
  mapinfo[NULLBM]->right = 0;

  /* text segment */
  mapinfo[TEXTBM]->name = M_TEXT_SEGMENT;
  mapinfo[TEXTBM]->base = (PTR) text_beg;
  mapinfo[TEXTBM]->real_base = (PTR) text_beg;
  mapinfo[TEXTBM]->length = (uint)(objects->text_end) - text_beg;
  mapinfo[TEXTBM]->bmbase = (uchar *) 0;
  mapinfo[TEXTBM]->type = SegText;
  mapinfo[TEXTBM]->idfunc = ename_textseg;
  mapinfo[TEXTBM]->efunc = chkr_access_error_text;
  mapinfo[TEXTBM]->right = CHKR_RO | CHKR_EXEC | (is_text_writable ? CHKR_WO : 0);

#ifdef CHKR_DATABITMAP
  /* data segment */
  mapinfo[DATABM]->name = M_DATA_SEGMENT;
  mapinfo[DATABM]->base = (PTR)(objects->data_beg);
  mapinfo[DATABM]->real_base = (PTR)(objects->data_beg);
  mapinfo[DATABM]->length = (uint)(objects->bss_end) - (uint)(objects->data_beg);
  mapinfo[DATABM]->bmbase = (uchar *) 0;
  mapinfo[DATABM]->type = SegRW; /*SegNormal;*/
  mapinfo[DATABM]->idfunc = ename_dataseg;
  mapinfo[DATABM]->right = CHKR_RW;
#endif
#ifdef CHKR_STACKBITMAP
  mapinfo[STACKBM]->name = M_STACK_SEGMENT;
  mapinfo[STACKBM]->base = (PTR) STACK_LIMIT;
  mapinfo[STACKBM]->real_base = (PTR) STACK_LIMIT;
  mapinfo[STACKBM]->length = 0;
  mapinfo[STACKBM]->bmbase = (uchar *) 0;
  mapinfo[STACKBM]->type = SegNormal;
  mapinfo[STACKBM]->efunc = chkr_access_error_stack;
  mapinfo[STACKBM]->idfunc = ename_stack;
  mapinfo[STACKBM]->right = CHKR_RW | CHKR_EXEC;
#endif
}


/* Function called to give more informations.
 * If the result is different from 0, the error must cause a core dump.
 */
static int
chkr_access_error_null (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  chkr_printf (M_USE_NULL_PTR);
  chkr_printf (M_PROD_SEG_FAULT);
  return 1; /* Can produce a segmentation fault. So, remove the temp file */
}

static int
chkr_access_error_text (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  chkr_printf (M_CANT_MODIFY_IT);
  chkr_printf (M_PROD_SEG_FAULT);
  return 1; /* Can produce a segmentation fault. So, remove the temp file */
}

static int
chkr_access_error_mmap (struct BitMem const *bm, struct BitMemInfo *bmi)
{
#if CHKR_SAVESTACK
  /* Display the history. */
  chkr_load_symtab ();
  chkr_printf (M_MAPPED_BY);
  disp_block_history (bm->info->info.mmapinfo->history);
  chkr_unload_symtab();
#endif
  return 0;	/* FIXME: violation */
}

static int
chkr_access_error_shm (struct BitMem const *bm, struct BitMemInfo *bmi)
{
#if CHKR_SAVESTACK
  /* Display the history. */
  chkr_load_symtab ();
  chkr_printf (M_ATTACHED_BY);
  disp_block_history (bm->info->info.shminfo->history);
  chkr_unload_symtab();
#endif
  return 1;
}

#ifdef CHKR_STACKBITMAP
static int
chkr_access_error_stack (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  chkr_printf (M_KNOW_STACK_LIM, known_stack_limit);
  __chkr_disp_right (bmi->ptr, 24);
  return 0;
}
#endif

static int
chkr_access_error_heap (struct BitMem const *bm, struct BitMemInfo *bmi)
{
  struct malloc_header *block;
  struct mdesc *mdp = bm->info->info.heapinfo->mdp;
  PTR *ptr_on_block;
  int block_size;
  PTR ptr = bmi->ptr;

  if (_lastmdesc == NULL_MDESC)
    return 0;
  if (ptr < low_addr_heap || ptr > high_addr_heap)
    return 0;			/* Not inside the heap. */

  block = find_header (mdp, ptr, 1);
  if (block == NULL_HEADER)
    {
      chkr_printf (M_AFTER_LAST_BLOCK);
      return 0;
    }

  /* Don't forget that ptr can point anywhere. */
  if (block->state == MDFREE)
    {
      /* Pointer to a free block. */
      chkr_printf (M_INSIDE_FBLOCK);
      return 0;
    }

  /* Compute the size of the block. */
  if (block->state == MDAGED)
    block_size = block->size - be_red_zone - af_red_zone - block->s_diff;
  else
    block_size = block->info.busy.real_size;

  if (ptr < ((PTR) block + be_red_zone + HEADER_SIZE))
    chkr_printf (M_BEFORE_BLOCK,
	       (int) ((int) block + be_red_zone + HEADER_SIZE - (int) ptr));
  else if (ptr < ((PTR) block + be_red_zone + HEADER_SIZE + block_size))
    chkr_printf (M_INSIDE_BLOCK,
	     (int) (ptr) - (int) ((int) block + be_red_zone + HEADER_SIZE));
  else
    chkr_printf (M_AFTER_BLOCK,
		 (int) (ptr) -
	  (int) ((PTR) block + be_red_zone + HEADER_SIZE + block_size));
	  
  /* Display size of the block and its beginning. */
  chkr_printf (M_BLOCK_ID, (uint) block + HEADER_SIZE + be_red_zone,
  			   block_size,
  			   mdp == __mmalloc_default_mdp ? 0 : mdp);
	  
#if CHKR_SAVESTACK
  /* Display the history. */
  ptr_on_block = (PTR *) ((int) block + block->size + HEADER_SIZE - af_red_zone);
  chkr_load_symtab ();
  chkr_printf (M_BLOCK_ALLO_FRM);
  disp_block_history (ptr_on_block);
  if (block->state == MDAGED)
    {
      chkr_printf (M_N_FREE_CALLED, get_age ((struct mdesc*)0, block));
      ptr_on_block = (PTR *) ((int) block + HEADER_SIZE);
      disp_block_history (ptr_on_block);
    }
#if 0   
  if (mdp != __mmalloc_default_mdp)
    {
      chkr_printf (M_MDESC_CREATED_BY);
      disp_block_history(bm->info->info.heapinfo.history);
    }
#endif
  chkr_unload_symtab ();
#endif /* CHKR_SAVESTACK */
  return 0;
}

/* The user can set a breakpoint to this function, which is called just
 * after a report.
 */
void
__chkr_pause(void)
{
 /* nothing */
}

/* This function is called by chkr_check_addr, when an error has been
 * discovered. It calls chkr_access_error_*
 * return 0: check other error
 * return 1: don't
 */
static int
chkr_access_error (const PTR ptr, int arq, struct BitMem const *bm,
		   char val, int size, int len)
{
  int must_remove = 0;
  struct BitMemInfo bmi;
  int i;

  /* Without inserted code */
  if (!(chkr_ld_options & 2)
      || ((chkr_ld_options & 1) && bm->info->type != SegNormal))
    return 1;

  /* An error has been found! */
  nbr_tot_access_error++;
  
  /* Check for disabled ip ('--disable=' option) */
  for (i = 0; i < disable_cur; i++)
    if ((int) chkr_frames_ip >= (*disable)[i].first
	&& (int) chkr_frames_ip <= (*disable)[i].last)
      return 1;			/* The ip doesn't change. */

  bmi.ptr = (PTR)ptr;
  bmi.arq = arq;
  bmi.remain = len;
  bmi.size = size;
  bmi.val = val;
  /* Call the function which identify the error. */
  if (bm->info->idfunc)
    i = (bm->info->idfunc) (bm, &bmi);
  else
    i = M_U_NMA_VD_ET;	/* FIXME */
    
  /* If the error is disabled, return immidiately. */
  if (is_error_disable(i))
    return 0;
    
  /* The error is going to be reported. */
  nbr_rep_access_error++;
  chkr_report (i);
  chkr_printf (M_ACC_ERR_WHERE, right_name[arq], size, ptr, bm->info->name);
  if (bm->info == &null_mapinfo)
    {
      chkr_printf (M_PROD_SEG_FAULT);
      must_remove = 1;
    }
  else if (bm->info->efunc)
    must_remove = (bm->info->efunc) (bm, &bmi);
    
#ifdef CHKR_SAVESTACK
  chkr_frames_to_forget = 5;
  chkr_show_frames ();
  if (must_remove)
    chkr_remove_symtabfile ();
#endif /* CHKR_SAVESTACK */
  __chkr_pause();
  return 1;
}

#else /* !MDCHECKER */

/* Return the size of the memory used. */
uint
get_total_mem(void)
{
  uint total = 0;
  total = (uint)chkr_sbrk(0) - (uint)(objects->data_beg);
  return total;
}

#endif /* !MDCHECKER */

#if 1
#include "macc-mmap.h"
#else
#include "macc-brk.h"
#endif
