/*
    client  : Browsing an Analysis Tool for C-source code;
	      Test-Client-Application

    Copyright (C) 1993  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* This is just an example of how to use the toolkit-functions in your
   own applications.

   Most of the code is purely for the "nice" output of the information
   returned by these functions.

   look in the main-function first to see how to use them
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <ncurses.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>

#include "c-bat.h"
#include "clients.h"


#define LINESIZE  256  /* maximum of 256 characters per line */

#define L_MEM_INIT_SIZE   1024    /* initial values for data-output memory */
#define LL_MEM_INIT_SIZE    32

#define L_MEM_EXPAND      1024
#define LL_MEM_EXPAND       32

#define R_MEM_INIT_SIZE   1024
#define RL_MEM_INIT_SIZE    32

#define R_MEM_EXPAND      1024
#define RL_MEM_EXPAND       32

#define F_MEM_INIT_SIZE   1024   /* initial memory for file view memory */
#define FL_MEM_INIT_SIZE    32   /* choosen small for testing purposes */

#define F_MEM_EXPAND      1024
#define FL_MEM_EXPAND       32

#define SORT_BY_ITEM         1
#define SORT_BY_FILE         2

#define TAB_EXP_VALUE            8  /* value to expand TAB characters    */
#define INTERESTING_LINE_OUTPUT  0  /* first line is the interesting one */

#define INTERESTING_LINE_FILE    9  /* we print 18 lines, the middle is 9*/

static char *l_mem  = NULL;  /* memory for left part of lines    */
static long *ll_mem = NULL;  /* memory for left-line-indizes     */
static long l_size = 0;      /* size of l_mem-memory allocated   */
static long ll_size = 0;     /* size of ll_mem-memory allocated  */
static long cnt_ll = 0;      /* number of left lines             */

static char *r_mem  = NULL;  /* memory for right part of lines   */
static long *rl_mem = NULL;  /* memory for right-line-indizes    */
static long r_size = 0;      /* size of r_mem-memory allocated   */
static long rl_size = 0;     /* size of rl_mem-memory allocated  */
static long cnt_rl = 0;      /* number of right lines            */

static char *f_mem  = NULL;  /* memory for file to view          */
static long *fl_mem = NULL;  /* memory for lines                 */
static long f_size = 0;      /* size of f_mem-memory allocated   */
static long fl_size = 0;     /* size of fl_mem-memory allocated  */
static long cnt_fl = 0;      /* number of lines in file          */

static short vi_flag = 0;           /* if 1, use vi as viewer */
static short Sort_flag = 0;         /* No Sorting             */
static char  vi_call[LINESIZE+2];   /* line to call vi        */

static  char funct[STRINGSIZE],     /* Strings to hold brs-parameter    */
             file[STRINGSIZE],      /* could be a local variable, but   */
             module[STRINGSIZE],    /* we need it in so many functions, */
             left[ 2 * STRINGSIZE], /* so I made it global              */
             right[2 * STRINGSIZE];

/* last error-message, printed on the screen
 */
static char last_error_msg[LINESIZE+2] = "No Errors";

/* global data for file-viewer
 */
static char last_filename[MAX_FILENAME] = "";
static char master_dir[MAX_FILENAME] = "";
static char new_dir[MAX_FILENAME] = "";
static char interesting_file[MAX_FILENAME] = "";
static long interesting_line = 0;

/* marking of an item in the window:
   don't have to type item in again
 */
static long marked_number = -1; /* -1 ... marking not allowed, cnt_ll ... marking allowed */

static char marked_file[MAX_FILENAME] = "";
static char marked_item[STRINGSIZE] = "";
  
static long  interesting_line_output; /* The highlighted line in the output-window */

/* possible types:  
   1 ... file
   2 ... function
   3 ... variable
   4 ... struct, enum, union, typedef
 */
#define MARK_NULL       0
#define MARK_FILE       1
#define MARK_FUNCTION   2
#define MARK_VARIABLE   3
#define MARK_TYPE       4
#define MARK_MACRO      5

#define MAX_MARKED  0x7fffffff

static short type_marked_item = MARK_NULL;
static short type_marked_file = MARK_NULL;
static short next_type_m_item = MARK_NULL;
static short next_type_m_file = MARK_NULL;

#define SET_MARK_TYPES(x,y);  {next_type_m_item = x; next_type_m_file = y;}
#define SET_MARK();           {type_marked_item = next_type_m_item; \
                               type_marked_file = next_type_m_file;}
#define CHECK_MARK_ITEM_TYPE(x)  ((type_marked_item == (x)) && marked_item[0])
#define CHECK_MARK_FILE_TYPE(x)  ((type_marked_file == (x)) && marked_file[0])
#define MARKING_ALLOWED()         marked_number = MAX_MARKED;
#define MARKING_NOT_ALLOWED()     marked_number = -1

extern int timeout_value; /* timeout used for client-server connection */

#ifndef min 
#define min(x,y) ((x < y)? (x):(y))
#define max(x,y) ((x > y)? (x):(y))
#endif 


char * menu_functions =
"             FUNCTION-MENU\n\n"
"   'F'  Print all functions defined in project\n"
"   'c'  Print all functions called by a given function\n"
"   'C'  Like 'c', but print EVERY function call position)\n"
"   'u'  Print all variables used by a function\n"
"   'U'  Like 'U', but print EVERY usage position\n"
"   'b'  Print all functions calling a given function\n"
"   'B'  Like 'b', but for EVERY calling position\n"
"   'd'  Print definition position of a function\n\n"
"   '?'  Help screen\n"
"   '+'  Next menu-screen (variable-menu)\n"
"   '-'  Previous menu-screen (misc-menu)\n"  ;

char * menu_variables =
"             VARIABLE-MENU\n\n"
"   'V'  Print all variables in project\n"
"   't'  Print all structs in project\n"
"   'n'  Print all unions in project\n"
"   'm'  Print all enums in project\n"
"   'y'  Print all typdefs in project\n"
"   's'  Print all functions using a variable\n"
"   'S'  Like 's', but for EVERY usage position\n"
"   'D'  Print definition position of a variable\n"
"   '.'  Print usage of a struct component\n"
"   ':'  Print usage of a struct component with a variable\n\n"
"   '?'  Help screen\n"
"   '+'  Next menu-screen (file-menu)\n"
"   '-'  Previous menu-screen (function-menu)\n"  ;

char * menu_files =
"               FILE-MENU\n\n"
"   'f'  Print all source-files in project\n"
"   'i'  Print all includes of a source-file\n"
"   'I'  Print all source files including a given file\n"
"   '#'  Print all macro defininitions in project\n"
"   '*'  Print position of a macro defininition\n"
"   'W'  What is ...\n"
"   'w'  Print working directory of a file\n\n"
"   '?'  Help screen\n"
"   '+'  Next menu-screen (misc-menu)\n"
"   '-'  Previous menu-screen (variable-menu)\n"  ;

char * menu_misc =
"               MISC-MENU\n\n"
"   'l'  Clear Marking\n"
"   'o'  Toggle Sort-flags (%s)\n"
"   'T'  Set new Timeout-Time\n"
"   'a'  Get all messages in queue\n"
"   '['  Change viewer (currently: %s)\n"
"   'R'  Set new browsing-datafile in actual server\n"
"   'r'  Switch to a new server\n"
"   'k'  Kill server and exit\n"
"   'q'  Exit client (not killing server)\n\n"
"   '?'  Help screen\n"
"   '+'  Next menu-screen (function-menu)\n"
"   '-'  Previous menu-screen (files-menu)\n"  ;

extern void help(int nr);

/* reset curses
 */
void Exit(int val )
{  erase();
   refresh();
   nocbreak();
   echo();
   exit( val );
 }


/* Initialize memory for line-output
 */
void init_lines(void)
{
   l_mem = malloc(L_MEM_INIT_SIZE);
   if( !l_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   l_size = L_MEM_INIT_SIZE;
   
   ll_mem = malloc(LL_MEM_INIT_SIZE);
   if( !ll_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   ll_size = LL_MEM_INIT_SIZE;
   
   ll_mem[0] = 0;
   
   r_mem = malloc(R_MEM_INIT_SIZE);
   if( !r_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   r_size = R_MEM_INIT_SIZE;
   
   rl_mem = malloc(RL_MEM_INIT_SIZE);
   if( !rl_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   rl_size = RL_MEM_INIT_SIZE;
   
   rl_mem[0] = 0;
 }

/* Initialize memory for file-view
 */
void init_file_memory(void)
{
   f_mem = malloc(F_MEM_INIT_SIZE);
   if( !f_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   f_size = F_MEM_INIT_SIZE;
   
   fl_mem = malloc(FL_MEM_INIT_SIZE);
   if( !fl_mem )
     { fprintf(stderr, "Not enough memory available: aborting !\n" );
       Exit(1);
      }
   fl_size = FL_MEM_INIT_SIZE;
   
   fl_mem[0] = 0;
   
 }



/* Checks for sufficient memory to store cnt_needed bytes
 */
void check_memory(void *ptr, long used, long cnt_needed)
{ long *p_size,
        exp_value,
        to_expand = 0;
  long **p_long = NULL;
  char **p_char = NULL;
   
  if( ptr == l_mem )
   { p_size = &l_size;
     p_char = &l_mem;
     exp_value = L_MEM_EXPAND;
    }
  else if( ptr == ll_mem )
   { p_size = &ll_size;
     p_long = &ll_mem;
     exp_value = LL_MEM_EXPAND;
    }
  else if( ptr == r_mem )
   { p_size = &r_size;
     p_char = &r_mem;
     exp_value = R_MEM_EXPAND;
    }
  else if( ptr == rl_mem )
   { p_size = &rl_size;
     p_long = &rl_mem;
     exp_value = RL_MEM_EXPAND;
    }
  else if( ptr == f_mem )
   { p_size = &f_size;
     p_char = &f_mem;
     exp_value = F_MEM_EXPAND;
    }
  else if( ptr == fl_mem )
   { p_size = &fl_size;
     p_long = &fl_mem;
     exp_value = FL_MEM_EXPAND;
    }
  else
   { fprintf(stderr, "UNKNOWN POINTER: expand failed\n" );
     exit(1);
    }
   
  /* calculate number of bytes we have to expand
   */
  while( (used + cnt_needed) >= (*p_size + to_expand) )
   { to_expand += exp_value;
    }

  /* if needed, expand memory 
   */  
  if( to_expand )
   { if( p_char )
       { *p_char = (char *)realloc( *p_char, *p_size + to_expand );
         if( !*p_char )
           { fprintf(stderr, "Memory-expand not possible\n" );
             Exit(1);
            }
        }
     else if( p_long )
       { *p_long = (long *)realloc( *p_long, *p_size + to_expand );
         if( !*p_long )
           { fprintf(stderr, "Memory-expand not possible\n" );
             Exit(1);
            }
        }
      
     *p_size += to_expand;
    }
   
}       
    
/* checks for sufficient memory to store strings p1, p2, p3 and
   terminating '\0'-bytes
   for adding 4 Bytes, call with: (.., ..., " ", NULL, NULL)
 */
void check_line_memory(void *ptr, long used, char *p1, char *p2, char *p3)
{ long cnt;
   
  cnt =  (p1) ? strlen(p1) + 1 : 1 ;
  cnt += (p2) ? strlen(p2) + 1 : 1 ;
  cnt += (p3) ? strlen(p3) + 1 : 1 ;
   
  check_memory( ptr, used, cnt);
  }

/* loads a file into memory (if not allready in memory)

   returns 0 on success, else error-code

   -1  file error
   -2  directory error
 */
short load_file( char *filename )
{ struct stat  stat_buffer;
  FILE  *fp; 
  off_t  size_left;
  register long   p = 0, i;
  size_t cnt_read;
   
  if( !strcmp(last_filename, filename) )
    return( 0 ); /* allready loaded */
   
  if( brs_get_file_dir( filename, new_dir) )
     return( -2 );
  
  if( !getcwd( master_dir, MAX_FILENAME) )
    return( -2 );
   
  if( chdir( new_dir ) )
    { /* error: cannot set new working directory
       */
      return( -2 );
     } 
   
  if(  stat( filename, &stat_buffer )  )
     return( -1 );  /* file does not exist */
   
  size_left = stat_buffer.st_size;
   
  fp = fopen( filename, "r" );
   
  cnt_fl = 0;     /* clear memory */
  fl_mem[0] = 0; 
   
  p = 0;  /* start at the beginning */
   
  while( size_left )
   { /* Check for memory to store next block
      */
     check_memory( f_mem , p , stat_buffer.st_blksize );
      
     cnt_read = fread( f_mem + p, 1, stat_buffer.st_blksize, fp );
      
     size_left -= cnt_read;
     
     p += cnt_read;
    }
   
  /* We have the file in memory, now calculate the lines
   */
   
  for( i = 0; i < p; i++ )
   { if( f_mem[i] == '\n' )
       { /* Check for memory to store next line-offset
          */
         check_memory( fl_mem, cnt_fl * sizeof(long), sizeof(long) );
         
         /* Next line starts one character after the Newline
          */
         cnt_fl++;
         fl_mem[cnt_fl] = i + 1;
        }
    }

  cnt_fl++;           /* Numer of lines */
  fl_mem[cnt_fl] = p; /* end of text */

  fclose( fp );
  strcpy(last_filename, filename);
   
  if( chdir( master_dir ) )
    { /* error: cannot set back old working directory
         ignore, because loading of the file suceeded !
       */
      return( 0 );
     } 
  
  return( 0 );
   
}


static char blank_string[LINESIZE+2];

/* prints the file in file-memory beginning from line 'line' with the 
   column offset 'col_offset'. This is to be used for viewing lines
   which are longer than the size of the window

   TAB's are expanded to TAB_EXP_VALUE blanks

   cnt_lines  ...  number of lines in the window
   cnt_cols   ...  numer of cols in the window
   line       ...  starting line
   col_offset ...  number of visible outermost left column (normally 0)
   int_line   ...  interesting line
 */
void print_file_win(short cnt_lines, short cnt_cols, long line, 
                    short col_offset, long int_line, char *identifier, short *token_no)
{ register long i,j, k, l, m;
  char line_buf[LINESIZE+2];
  char  *l1, *file_buffer;
  long  ll1, end, k_start;
  short   tmp;
   
   cnt_cols = min( cnt_cols, LINESIZE );
   
   cnt_cols -= 6; /* 6 Bytes for Linenumber */
   
   for( j = 0, i = line; (i < line + cnt_lines) ; i++, j++ )
     { move( j + 1, 0 );
      
       if( i == int_line )
         attron( A_REVERSE );
       else
         attroff( A_REVERSE );

       /* we start line-counting with 0, but the compiler with 1 !
        */
       printw("%5d ", i + 1); 
      
       if( i == int_line )
         attroff( A_REVERSE );

       if( i < cnt_fl )
         { /* The current line is a valid linenumber !
            */
           l1  = f_mem + fl_mem[i] ;
           ll1 = strlen( l1 );

           l = 0;  /* true column number (TAB's count as blanks) */
      
          /* walk the whole line, expand TAB's and start to write
             char's into line_buf as l exceeds col_offset
           */
          line_buf[0] = 0;

          end = fl_mem[i+1];

          k_start = fl_mem[i];
          file_buffer = f_mem;
         }
       else
         { /* "faked" line - since we do not clear the output window (due to output
               speed), we have to fill up the sub-bottom-lines with blanks
            */
           k_start = 0;
           end = 1;
           file_buffer = "\n";
          }

       for( k = k_start; k < end; k++ )
          { switch( file_buffer[k] )
              { case '\t' : /* TAB: expand by tmp blanks 
                             */
                            tmp = 8 - (l % 8);
                            for( m = 0; m < tmp; m++ )
                              { if( (l - col_offset) >= cnt_cols )
                                   break;
                                if( l >= col_offset )
                                  line_buf[l-col_offset] = ' ';
                                l++;
                               }
                            break;

               case '\n':   /* end of line */
                            line_buf[l - col_offset] = 0;
                            l = cnt_cols + col_offset; /* stop line */
                            break;
            
               default:     if( (l - col_offset) >= cnt_cols )
                               break; /* end of line reached */
            
                            if( l >= col_offset )
                              { /* after first visible column 
                                 */
                                if( isprint( f_mem[k] ) )
                                  line_buf[l-col_offset] = file_buffer[k];
                                else
                                  line_buf[l-col_offset] = ' ';
                               }
                            l++;
                            break;
               }
            
            if( (l - col_offset) >= cnt_cols )
              break; /* line ends */
           } /* next */
       line_buf[l - col_offset] = 0;

       /* do not use printw, because this results in troubles with 
          '%..' in the strings
        */
       if( *token_no >= 0 && i == int_line )
         { short state = 0;
         
           /* finit-state-automaton
         
              state-0  --[char]---------> state-1
              state-1  --[char/digit]---> state-1
              state-1  --[else]---------> state-0
            */
           for( l = 0, m = 0, k = 0; k < strlen(line_buf); k++ )
             { switch( state )
                { case 0: if( isalpha(line_buf[k]) || line_buf[k] == '_' )
                            { state = 1;
                              if( l == *token_no )
                                { attron( A_REVERSE );
                                  identifier[m++] = line_buf[k];
                                 }
                             }
                          break;
                  case 1: if( isalnum(line_buf[k]) || line_buf[k] == '_' )
                            { if( m )
                                identifier[m++] = line_buf[k];
                              break;
                             }
                          else
                            { state = 0;
                              if( l == *token_no )
                                attroff( A_REVERSE );
                              l++;
                              if( m )
                                { identifier[m] = 0;
                                  m = 0;
                                 }
                             }
                 }
               addch( line_buf[k] );
              }
            if( l <= *token_no )
              *token_no = -1;
          }
       else
         { /* normal output */
           addstr( line_buf );
          }
       addstr(blank_string + LINESIZE - cnt_cols + strlen(line_buf) );
     }
   
  attroff( A_REVERSE );
  
  refresh();
}  

/* used to store the file-call-stack
   if user requests to follow a calling
   structure
 */
struct file_stack
{ char *filename;
  long  line;
  long  start;
  short col;
 };

#define MAX_STACK         50  
#define MAX_IDENTIFIER    50

/* loads and views a given file and views at the
   given line (note: counting begins with 0)
 */
short print_file(char *filename, long line)
{ long  start_line = 0, init_start_line, init_line, line_end;
  short start_col = 0, ret;
  short win_size;
  short ident_mode = 0; /* 0 .. normal mode,  1 .. ident mode,  2 .. view position */
  int   c;
  struct file_stack f_stack[MAX_STACK];
  char   identifier[MAX_IDENTIFIER];
  char   info_line[80] = "";
  short  stack_pointer = 0;
  short  token_no = -1;
   
  memset( blank_string, ' ', LINESIZE );

  if( vi_flag )
    { /* use vi instead of the internal viewer
       */
      if( brs_get_file_dir( filename, new_dir) )
         return( -1 );
  
      if( !getcwd( master_dir, MAX_FILENAME) )
        return( -1 );
   
      if( chdir( new_dir ) )
        { /* error: cannot set new working directory
           */
          return( -1 );
         } 
      sprintf( vi_call, "vi -R +%ld %s", line + 1, filename );
      system( vi_call );
      
      chdir( master_dir );
      return( 0 );
     }
    
  strcpy(file, filename);

  if( load_file( filename )  )
    return( -1 );

  start_line = line - INTERESTING_LINE_FILE; 
   
  if( start_line < 0 )
     start_line = 0;
   
  if( start_line > cnt_fl - 1) 
    start_line = cnt_fl - 1;
   
  init_start_line = start_line;
  init_line       = line;

  win_size = LINES - 6;
  
  erase();
 
  while(1)
   { /* erase(); */
     move( 0, (COLS - strlen(file)) / 2);
     addstr(file);
     print_file_win( win_size + 1, COLS - 1 /*79*/, start_line, 
                     start_col, line, identifier, &token_no);
      
     move( LINES-4 /*20*/, 0 );
     if( !ident_mode )
      { printw("k ... Up,      j ... Down,    ^U ... Page up,     ^D ... Page down      \n");
        printw("l ... Right,   h ... Left,     ^ ... Beg. of line  q ... Quit   ?...Help\n");
        printw("t ... Top,     b ... Bottom,   B ... Back          i ... Ident-mode     ");
       }
     else if( ident_mode == 1 )
      { printw("c ... Call (%2d)  d ... Definition     i ... Next token    ? ... Help     \n", 
                stack_pointer);
        printw("r ... Return     w ... What is        q ... Quit Mode     ? ... Help     \n");
        printw("Info: %s", info_line);
        addstr(blank_string + LINESIZE - COLS + strlen(info_line) + 6 );
        info_line[0] = 0;
       }
     else 
      { printw("                                                                         \n");
        printw("r ... Return                                                             \n");
        printw("Info: %s", info_line);
        addstr(blank_string + LINESIZE - COLS + strlen(info_line) + 6 );
        info_line[0] = 0;
       }
     refresh();

     c = getch();
      
     if( ident_mode == 2 )
       { /* In view position mode just cursor-keys allowed
            every other key returns to the previous position
          */
         switch( c )
           { case 'k': case KEY_UP:
             case 'j': case KEY_DOWN:
             case 0x15: case KEY_PPAGE:
             case 0x4: case KEY_NPAGE:
             case '^':
                   break;
             default: c = 'r';  /* return */
                      ident_mode = 1;
                      break;
            }
        }
      
     switch( c )
      { case '?': help(5);
                  break;
        case 'c': /* call function */
                  if( ident_mode )
                   { /* push current file */
                     if( stack_pointer >= MAX_STACK - 1 )
                        break;
                     f_stack[stack_pointer].filename = strdup(file);
                     f_stack[stack_pointer].line     = line;
                     f_stack[stack_pointer].start    = start_line;
                     f_stack[stack_pointer].col      = start_col;
                     stack_pointer++;
                     
                     /* Try to find the definition-position of this function
                        first in the file, later global
                      */
                     strcpy( left, file );
                     ret = brs_get_funct_pos( identifier, left, file, 
                           &line, &line_end, module);
                     if( ret )
                       { ret = brs_get_funct_pos( identifier, "", file, 
                           &line, &line_end, module);
                         if( ret )
                           { sprintf(info_line, "Definition position of %s unknown", 
                                             identifier );
                             stack_pointer--;
                             strcpy( file, f_stack[stack_pointer].filename );
                             free(f_stack[stack_pointer].filename);
                             break;
                            }
                        }
            
                     line--; /* Line counting starts with 1, here with 0 */
            
                     if( load_file( file )  )
                       goto ret_pos;

                     start_line = line - INTERESTING_LINE_FILE; 
   
                     if( start_line < 0 )
                        start_line = 0;
   
                     if( start_line > cnt_fl - 1) 
                       start_line = cnt_fl - 1;
   
                     token_no = 0;
                    }
                  break;
         
        case 'k': case KEY_UP:
                  line -= (line > 0 ) ? 1 : 0;
                  if( line < start_line )
                     start_line -= (start_line > 0) ? 1 : 0;
                  break;
         
        case 'j': case KEY_DOWN:
                  line += (line < cnt_fl - 1 ) ? 1 : 0;
                  if( line > start_line + win_size)
                     start_line += (start_line < cnt_fl - 1) ? 1 : 0;
                  break;
         
        case 'i': if( !ident_mode )
                    token_no = 0; /* first time ident_mode */
                  else
                    token_no++;
                  ident_mode = 1;
                  break;
         
        case 0x15: case KEY_PPAGE:
                  start_line -= (start_line > win_size) ? win_size : start_line;
                  line -= (line > win_size) ? win_size : line;
                  break;
         
        case 0x4: case KEY_NPAGE:
                  start_line += (start_line < cnt_fl - win_size) ? win_size : 0;
                  line += (line < cnt_fl - win_size) ? win_size : 0;
                  break;
         
        case '^': /* begin of line */
                  if( ident_mode )
                     break;
                  start_col = 0;
                  break;
         
        case 'l': case KEY_RIGHT:
                  if( ident_mode )
                     break;
                  start_col += (start_col < 256) ? 1 : 0;
                  break;
         
        case 'h': case KEY_LEFT:
                  if( ident_mode )
                     break;
                  start_col -= (start_col > 0) ? 1 : 0;
                  break;
         
        case 'q': if( ident_mode ) /* quit */
                    { ident_mode = 0;
                      token_no = -1;
                     }
                  else
                    { erase();
                      refresh();
                      return;
                     }
                  break;
         
        case 't': case KEY_HOME: /* Top of file */
                  start_line = 0;
		  line = 0;
                  break;
         
        case 'b': case '~': /* end of file */
                  start_line = cnt_fl - win_size;
		  line = cnt_fl - 1;
                  break;
         
        case 'r': /* return */
                  if( ident_mode )
                   { /* pop file */
ret_pos:
                     if( stack_pointer == 0 )
                        break;
                     stack_pointer--;
                     strcpy( file, f_stack[stack_pointer].filename);
                     line       = f_stack[stack_pointer].line;
                     start_line = f_stack[stack_pointer].start;
                     start_col  = f_stack[stack_pointer].col;
                     free(f_stack[stack_pointer].filename);
                                 
                     if( load_file( file )  )
                      return;
                     token_no = 0;
                    }
                  break;

        case 'd': /* jump to definition position */
                  if( !ident_mode )
                     break;
                  /* 
                   */
                  strcpy( left, file );
                  strcpy(info_line, "" );
                  ret = brs_get_item_pos( identifier, file, line + 1,
                                          left, &line_end);
                  if( !ret )
                    { /* Jump to this position */
                      if( stack_pointer >= MAX_STACK - 1 )
                         break;
                      f_stack[stack_pointer].filename = strdup(file);
                      f_stack[stack_pointer].line     = line;
                      f_stack[stack_pointer].start    = start_line;
                      f_stack[stack_pointer].col      = start_col;
                      stack_pointer++;
                     
                      line = line_end - 1; /* counting starts with 1, here with 0 */

                      strcpy( file, left );
                      if( load_file( file )  )
                        goto ret_pos;

                      start_line = line - INTERESTING_LINE_FILE; 
   
                      if( start_line < 0 )
                         start_line = 0;
   
                      if( start_line > cnt_fl - 1) 
                        start_line = cnt_fl - 1;

                      ident_mode = 2;
                      token_no = 0;
                     }
                  else
                    sprintf( info_line, "Error: %d %s", ret, left );
                  break;

        case 'w': /* print what is */
                  if( !ident_mode )
                     break;
                  /* 
                   */
                  strcpy( left, file );
                  strcpy(info_line, "" );
                  ret = brs_what_is( identifier, file, line,
                                     left, &line_end, right );

                  switch( ret )
                    { case RET_MACRO:
                         strcat(info_line, "Macro" );
                         break;
                      case RET_VAR:
                         strcat(info_line, "Variable" );
                         break;
                      case RET_FUNCT:
                         strcat(info_line, "Function" );
                         break;
                      case RET_STRUCT:
                         strcat(info_line, "Struct" );
                         break;
                      case RET_UNION:
                         strcat(info_line, "Union" );
                         break;
                      case RET_ENUM:
                         strcat(info_line, "Enum" );
                         break;
                      case RET_TYPEDEF:
                         strcat(info_line, "Typedef" );
                         break;
                      case RET_COMP:
                         strcat(info_line, "struct/union comp" );
                         break;
                      case RET_ENUM_C:
                         strcat(info_line, "enum constant" );
                         break;
                      case RET_END: 
                      case RET_ERROR:
                      case RET_TIMEOUT:
                         info_line[0] = 0;
                         break;
                      default:
                         break;
                     }
                 if( ret != RET_END && ret != RET_ERROR && ret != RET_TIMEOUT )
                    { sprintf( right, " %s %ld", left, line_end );
   
                      /* Quick & Dirty-test not to overwrite info_line-buffer */
                      if( strlen( info_line ) + strlen( right ) < 50 )
                         strcat( info_line, right );
                     }

                  while( ret != RET_END )
                     ret = brs_what_is( identifier, file, line,
                                     left, &line_end, right );

                  break;

        case 'B': /* back */
                  if( ident_mode )
                    break;
                  start_line = init_start_line;
                  line       = init_line;
                  break;

        default : break;
       }
    }
}
   

/* used to get two strings and store them in the data-output-memory
   for the left part of the lines
 */
void push_data_left(char *left1, char *left2)
{ long p;
  char empty[3] = ""; 
      
  p = ll_mem[cnt_ll];
   
  check_line_memory( l_mem, p, left1, left2, NULL);
   
  /* check for 4 Bytes */
  check_memory( ll_mem, cnt_ll * sizeof(long), sizeof(long) );
   
  if( !left1 )
    left1 = empty;
  memcpy( l_mem + p, left1, strlen(left1) + 1 );
  p += strlen( left1 ) + 1;
   
  if( !left2 )
    left2 = empty;
  memcpy( l_mem + p, left2, strlen(left2) + 1 );
  p += strlen( left2 ) + 1;
   
  cnt_ll++;
  ll_mem[cnt_ll] = p;
 }


/* used to get three strings and store them in the data-output-memory
   for the right part of the lines
 */
void push_data_right(char *right1, char *right2, char *right3)
{ long p;
  char  empty[3] = "";
   
  p = rl_mem[cnt_rl];
   
  check_line_memory( r_mem, p, right1, right2, right3);
   
  /* check for 4 Bytes */
  check_memory( rl_mem, cnt_rl * sizeof(long), sizeof(long));
   
  if( !right1 )
    right1 = empty;
   
  memcpy( r_mem + p, right1, strlen(right1) + 1 );
  p += strlen( right1 ) + 1;
   
  if( !right2 )
    right2 = empty;
  memcpy( r_mem + p, right2, strlen(right2) + 1 );
  p += strlen( right2 ) + 1;

  if( !right3 )
    right3 = empty;
  memcpy( r_mem + p, right3, strlen(right3) + 1 );
  p += strlen( right3 ) + 1;
   
  cnt_rl++;
  rl_mem[cnt_rl] = p;
 }


/* save output-window to a file
 */
void output_to_file(char *head, short cnt_lines, short cnt_cols, long line )
{ char filename[MAX_FILENAME] = "";
  FILE  *fp;
  int   c = 'o';
  long i,j;
  char line_buf[LINESIZE+2];
  char  *l1, *l2, *r1, *r2, *r3;
  long  ll1, ll2, rl1, rl2, rl3, l_right;
  
  move( LINES-4, 0 );
  printw("                                                                            \n");
  printw("                                                                            ");
  move( LINES-4, 0 );
  printw("Filename : " );
  refresh();
  
  getstr(filename);
   
  fp = fopen( filename, "r" );
  if( fp )
    { /* filename does exist, ask if overwrite or append
       */
      fclose(fp);
      move( LINES-3, 0 );
      printw("'%s' exists : (o)verwrite, (a)ppend or (q)uit", filename );
      refresh();
      do
       { c = getch();
        } while ( c != 'o' && c != 'a' && c != 'q' );
      if( c == 'q' )
        { strcpy( last_error_msg, "Command aborted !");
          return;
         }
     }
   
  if( c == 'o' )
    fp = fopen( filename, "w" );
  else
    fp = fopen( filename, "a" );
      
  if( !fp )
    {  sprintf(last_error_msg, "
          Could not open '%s' for writing (errno: %d)", 
          filename, errno );
       return;
     }
   
  /* first print headline : */
  fprintf( fp, "\n%s\n\n", head );
   
  /* This is mainly the code from print_output_win:
   */
   
  cnt_cols = min( cnt_cols, LINESIZE );
 
  for( j = 0, i = line; (i < line + cnt_lines) && (i < cnt_ll); i++, j++ )
    { fprintf( fp, "%4d ", i + 1);
      
      l1  = l_mem + ll_mem[i] ;
      ll1 = strlen( l1 );
      l2  = l1 + strlen(l1) + 1;
      ll2 = strlen( l2 );
      
      r1  = r_mem + rl_mem[i] ;
      rl1 = strlen( r1 );
      r2  = r1 + strlen( r1 ) + 1;
      rl2 = strlen( r2 );
      r3  = r2 + strlen( r2 ) + 1;
      rl3 = strlen( r3 );

      if( rl1 + rl2 + rl3 > 0 )
        memset( line_buf, '.', cnt_cols - 5);
      else
        memset( line_buf, ' ', cnt_cols - 5);
      
      
      memcpy(line_buf, l1, ll1 );
      line_buf[ll1] = ' ';
      if( l2[0] )
       { memcpy(line_buf + ll1 + 1, l2, ll2 );
         line_buf[ll1 + 1 + ll2] = ' ';
        }
   
      l_right = rl1;
      if( r2[0] )
        { l_right += 1 + rl2;
          if( r3[0] )
            l_right += 1 + rl3;
         }
   
      if( r3[0] )
        sprintf(line_buf + cnt_cols - 6 - l_right, " %s %s %s", r1, r2, r3 );
      else if( r2[0] )
        sprintf(line_buf + cnt_cols - 6 - l_right, " %s %s", r1, r2 );
      else if( r1[0] )
        sprintf(line_buf + cnt_cols - 6 - l_right, " %s", r1 );
      else
        sprintf(line_buf + cnt_cols - 6 - l_right, " " );
      

      fprintf( fp, "%s\n", line_buf );
    }
   
  fclose( fp );
  sprintf( last_error_msg, "%d lines saved !", j + 3);
   
}
  
   
  
   
/* views one window of output-data 
(combined of left and right parts)
   may be sorted if an index-array is given
*/
void print_output_win(long *idx_array, short cnt_lines, short cnt_cols, 
                      long line, short file_name_info, short line_name_info)
{ long i,j, k;
  char line_buf[LINESIZE+2];
  char  *l1, *l2, *r1, *r2, *r3;
  long  ll1, ll2, rl1, rl2, rl3, l_right;
    
   cnt_cols = min( cnt_cols, LINESIZE );
   
   interesting_line_output = cnt_lines / 2;
   interesting_line_output = min( interesting_line_output, cnt_ll - line - 1);
   
   if( line < 0 )
     { /* go to a line above the middle-line */
       interesting_line_output = min(cnt_lines / 2, cnt_ll - 1) + line;
       line = 0;
       if( interesting_line_output < 0 )
         interesting_line_output = 0;
      }
   
   for( j = 0, i = line; (i < line + cnt_lines) && (i < cnt_ll); i++, j++ )
     { move( j + 1, 0 );
       if( j == interesting_line_output  /* INTERESTING_LINE_OUTPUT */ )
         attron( A_REVERSE );
       else
         attroff( A_REVERSE );
      
       if( idx_array )
         k = idx_array[i];
       else
         k = i;
            
       l1  = l_mem + ll_mem[k] ;
       ll1 = strlen( l1 );
       l2  = l1 + strlen(l1) + 1;
       ll2 = strlen( l2 );
      
       r1  = r_mem + rl_mem[k] ;
       rl1 = strlen( r1 );
       r2  = r1 + strlen( r1 ) + 1;
       rl2 = strlen( r2 );
       r3  = r2 + strlen( r2 ) + 1;
       rl3 = strlen( r3 );
      
       if( marked_number == i ) 
         { printw("%4d*", i + 1);
           /* assume marked item in l1, interesting file in r2
            */
           strcpy( marked_item, l1 );
           strcpy( marked_file, r1 );
           SET_MARK();
          }
       else
         printw("%4d ", i + 1);
      


       if( j == interesting_line_output /* INTERESTING_LINE_OUTPUT */ )
         { switch( file_name_info )
             { case 0:  strcpy( interesting_file, l1 ); break;
               case 1:  strcpy( interesting_file, l2 ); break;
               case 2:  strcpy( interesting_file, r1 ); break;
               case 3:  strcpy( interesting_file, r2 ); break;
               case 4:  strcpy( interesting_file, r3 ); break;
               default: strcpy( interesting_file, "" ); break;
              }
           switch( line_name_info )
             { case 0:  interesting_line = atoi(l1); break;
               case 1:  interesting_line = atoi(l2); break;
               case 2:  interesting_line = atoi(r1); break;
               case 3:  interesting_line = atoi(r2); break;
               case 4:  interesting_line = atoi(r3); break;
               default: interesting_line = 1;
              }
          }
      
       if( rl1 + rl2 + rl3 > 0 )
         memset( line_buf, '.', cnt_cols - 5);
       else
         memset( line_buf, ' ', cnt_cols - 5);
      
      
       memcpy(line_buf, l1, ll1 );
       line_buf[ll1] = ' ';
       if( l2[0] )
        { memcpy(line_buf + ll1 + 1, l2, ll2 );
          line_buf[ll1 + 1 + ll2] = ' ';
         }
   
       l_right = rl1;
       if( r2[0] )
         { l_right += 1 + rl2;
           if( r3[0] )
             l_right += 1 + rl3;
          }
   
       if( r3[0] )
         sprintf(line_buf + cnt_cols - 6 - l_right, " %s %s %s", r1, r2, r3 );
       else if( r2[0] )
         sprintf(line_buf + cnt_cols - 6 - l_right, " %s %s", r1, r2 );
       else if( r1[0] )
         sprintf(line_buf + cnt_cols - 6 - l_right, " %s", r1 );
       else
         sprintf(line_buf + cnt_cols - 6 - l_right, " " );
      

       printw( line_buf );
     }
   
  if( j == interesting_line_output /* INTERESTING_LINE_OUTPUT */ + 1 )
    attroff( A_REVERSE );
  
  refresh();
}


void reset_line_data(void)
{
  cnt_ll = 0;
  cnt_rl = 0;
 }


/* print_output stores the number of the right-line-part, where
   the filename is stored
 */
static short  local_file_name_info;

/* compare-function for sorting output-lines
 */
int output_compare(const void * buff_1, const void * buff_2 )
{ char  *l1, *l2, *r1, *r2;
  long  idx1, idx2, rl1, rl2;
      
  idx1 = *((long *)buff_1);
  idx2 = *((long *)buff_2);
   
  if( Sort_flag == SORT_BY_ITEM || local_file_name_info < 2)
   { /* Sort by ITEM or if filename is not in the right part
      */
     l1  = l_mem + ll_mem[idx1];
     l2  = l_mem + ll_mem[idx2];
     return( strcmp( l1, l2) );
    }
  else /* filename is in the right part of the line */
   { r1  = r_mem + rl_mem[idx1];
     r2  = r_mem + rl_mem[idx2];
      
     if( local_file_name_info == 2 ) /* filename in right1 */
       return( strcmp( r1, r2) );
      
     rl1 = strlen( r1 );
     rl2 = strlen( r2 );
     r1  = r1 + strlen( r1 ) + 1;
     r2  = r2 + strlen( r2 ) + 1;
      
     if( local_file_name_info == 3 ) /* filename in right2 */
       return( strcmp( r1, r2) );
      
     /* else: filename in right-3 */
     rl1 = strlen( r1 );
     rl2 = strlen( r2 );
     r1  = r1 + strlen( r1 ) + 1;
     r2  = r2 + strlen( r2 ) + 1;
     return( strcmp( r1, r2) );
    }
 }


/* prints the output generated by push_data_left and push_data_right.

   file_name_info tells this routine, which part of the line is the 
   filename if the user wants to view the file

   line_name_info tells this routine, which part of the line is the 
   linenumber if the user wants to view the file at this position

   possible values:

   0  ...  left 1
   1  ...  left 2
   2  ...  right 1
   3  ...  right 2
   4  ...  right 3

  -1  ...  no given
*/
void print_output(char *head, short file_name_info, short line_name_info)
{ long start_line = 0, i;
  int   c;
  short  win_size;
  static long *sort_indices = NULL;
  static long cnt_sort_idx = 0;

  win_size = LINES - 6; /* 18 lines on 24-line-terminal */
   
  if( Sort_flag )
    { if( cnt_sort_idx < max( cnt_ll, cnt_rl) )
       { if( sort_indices )
            free( sort_indices );
         sort_indices = malloc( 2 * max( cnt_ll, cnt_rl) * sizeof( long ) );
         if( !sort_indices )
           { fprintf(stderr, "Not enough memory available: aborting !\n" );
             Exit(1);
            }
         cnt_sort_idx = 2 * max( cnt_ll, cnt_rl);
        }
      for( i = 0; i < cnt_sort_idx; i++ )
        sort_indices[i] = i;

      local_file_name_info = file_name_info;
      
      /* Now sort the sort_indices, but use the ll_mem-buffer for the comparing
       */
      qsort( sort_indices, cnt_ll, sizeof(long), output_compare );
     }
   
  last_error_msg[0] = 0; 
  while(1)
   { erase();
     move( 0, (COLS - strlen(head)) / 2);
     addstr(head);
     print_output_win( (Sort_flag) ? sort_indices : NULL, win_size, COLS-1 /*79*/, 
                       start_line, file_name_info, line_name_info);
      
     move( LINES-4/*20*/, 0 );
     printw("k .. Up,       j .. Down,        ^U .. Pg up,       ^D .. Pg down  ? .. Help\n");
     printw("v .. View src  f .. Win to file,  F .. All to file   q .. Quit     %s",
             (marked_number == -1) ? "" : "m  .. mark" );
     
     move( LINES-2/*22*/, 15 );
     printw("Error: %s", last_error_msg );
     move( LINES-2/*22*/, 0 );
     printw("command: " );
     refresh();
     last_error_msg[0] = 0;
      
     c = getch();
      
     switch( c )
      { case '?': help(4);
                  break;
        case 't': case KEY_HOME:
                  start_line = 0;
                  break;
        case 'b': case '~':
                  start_line = cnt_ll - win_size;
                  break;
        case 'k': case KEY_UP:
                  start_line -= (start_line > (- min(win_size / 2, cnt_ll - 1) )) ? 1 : 0;
                  break;
        case 'j': case KEY_DOWN:
                  start_line += (start_line < cnt_ll - 1) ? 1 : 0;
                  break;
        case 0x15: case KEY_PPAGE:
                  start_line -= (start_line > win_size) ? win_size : start_line;
                  break;
        case 0x4: case KEY_NPAGE:
                  start_line += (start_line < cnt_ll - win_size) ? win_size : 0;
                  break;
        case 'q': erase();
                  refresh();
                  return;
        case 'v': if( interesting_file[0] )
                    { if( print_file( interesting_file, 
                                     interesting_line - 1) )
                        { /* error loading file */
                          sprintf(last_error_msg, 
                                  "Error loading file \"%s\" !",
                                  interesting_file );
                         }
                     }
                  else
                    { /* not possible */                          
                      sprintf(last_error_msg, 
                              "Operation not allowed here !" );
                     }
                  break;
        case 'f': /* output to file (just the window)
                   */
                  output_to_file(head, win_size, COLS-1 , start_line );
                  break;
         
        case 'F': /* output to file (all)
                   */
                  output_to_file(head, cnt_ll, COLS-1 , 0 );
                  break;
        
        case 'm': if( marked_number == -1 ) 
                    /* marking not allowed */
                    break;
                  marked_number = start_line + interesting_line_output;
                  break;
         
        default : 
                   sprintf(last_error_msg, 
                            "Unknown Command '%c' (0x%02x)", c, c );
                   break;
       }
    }
}
   
/* Print a context sensitive help screen 
   Well, this is more than exagerating :-)
 */
void help(int nr)
{ int in;
           
   erase();
   switch( nr )
     { case 0:
          printw("BRS_CLIENT              OVERVIEW                      HELP-SCREEN\n\n");
          printw("This is the main menu of the browser-client. If you have started a\n");
          printw("browser-server with a browser-control-file, there should be an\n");
          printw("established IPC-connection (no connection is reported otherwise).\n");
          printw("The menu is divided into 4 pages (accessable with '+' or '-'), but\n");
          printw("you can type any command at any time.\n");
          printw("After pressing one of the Print-Commands, the client issues a query\n");
          printw("to the server and will print a list with the query-results (in the\n");
          printw("order of the occurence - this can be changed by the 'o' command)\n");
          printw("This may take a little time, depending on the size of the project.\n");
          printw("You can choose one of the results by the KEY-UP/KEY-DOWN (k/j) keys\n");
          printw("and view the source-position. With 'm', you can mark the highlighted\n");
          printw("item so it will be used as argument for the next command.\n\n");
          printw("The 4 menu-pages are:\n");
          printw("  FUNCTION-menu : definition position and calls of functions \n");
          printw("  VARIABLE-menu : definition and usage of variables and data-types\n");
          printw("  FILE-menu     : include files / #define's \n");
          printw("  MISC-menu     : communication, sorting and misc. options\n");
          break;
       case 1:
          printw("BRS_CLIENT             VARIABLE-MENU                  HELP-SCREEN\n\n");
          printw("This menu shows all commands concerning variables and data-types.\n");
          printw("Most of them should be clear. An interesting command is the \n");
          printw("possibility to show the usage of a struct or union component, esp.\n");
          printw("with a specified variable-identifier ('.' and ':')\n");
          break;
       case 2:
          printw("BRS_CLIENT               FILE-MENU                    HELP-SCREEN\n\n");
          printw("This menu shows all commands concerning source-files and macros.\n");
          printw("('#define'). The 'What is' command tells the type of an identifier.\n");
          printw("Since there is no possibility to have two source-files with the same\n");
          printw("name in a project (at least at the moment), you can retrieve the\n");
          printw("working directory of a specified source file by entering its name.");
          break;
       case 3:
          printw("BRS_CLIENT               MISC-MENU                    HELP-SCREEN\n\n");
          printw("This menu shows all commands which do not fit into one of the others.\n");
          printw("With 'l', you clear the marking you have set within a result-list.\n");
          printw("By pressing 'o', you cycle between 'no order', 'sort by identifier'\n");
          printw("and 'sort by filename'. You can set a new Timeout-value for the\n");
          printw("client-server communication. The 'a' command is mainly for debugging\n");
          printw("purposes, but we still are in ALPHA-stage :-). You can switch between\n");
          printw("the very slow internal viewer (with special functions) and the 'vi'.\n");
          printw("Since there is a support for multiple servers at a time, you can \n");
          printw("connect to a different server or tell the actual server to process a\n");
          printw("new project.\n");
          printw("At last, you can quit this program with ('k') or without ('q') killing\n");
          printw("the server. When killing the server, the IPC connection will be cleaned\n");
          break;
       case 4:
          printw("BRS_CLIENT               RESULT-LIST                    HELP-SCREEN\n\n");
          printw("This is a list with the result of the command issued. You can move \n");
          printw("around with the KEY-UP/KEY-DOWN keys ('k'/'j' respectivly), the PAGE-\n");
          printw("UP/PAGE-DOWN keys (CTRL-U / CTRL-D), the 't' (top of list) and the 'b'\n");
          printw("(bottom of list) keys. The highlighted line is the currently active\n");
          printw("one. You can mark it with the 'm' key, so this file and the identifier\n");
          printw("will be stored in a buffer and will be used as argument for the next \n");
          printw("command. By pressing 'v', the position in the source file will be dis-\n");
          printw("played in the file-windows.\n");
          printw("You can save the results (currently displayed window only or all \n");
          printw("results to a file by pressing 'w' or 'f'\n");
          break;
       case 5:
          printw("\nBRS_CLIENT               FILE-WINDOW                   HELP-SCREEN\n\n");
          printw("You can see the source-file at the selected position. You can move\n");
          printw("around like in all other windows. With the 'B'-command, you get back\n");
          printw("to the line you started.\n");
          printw("By pressing 'i', you enter the IDENT-Mode and the first valid token\n");
          printw("in the current line (marked linenumber) will be highlighted. Typing\n");
          printw("'i' again, you select the next token. If the token is a valid function\n");
          printw("identifier, you can 'jump' the the definition position of the function\n");
          printw("with 'c' (call). The current position will be saved in a stack, so you\n");
          printw("can return to it by pressing 'r' (return). The definition position of\n");
          printw("any other identifier can be viewed by 'd'. You can also issue the 'What\n");
          printw("is'-command for this identifier.");
          break;
      }
   move(LINES - 3,0);
   printw("   press a key: ");
   refresh();

   in =  getch();
 }
   
   
/* print simple menu
 */
int menu(void)
{ int in = 0;
  static short  menu_nr = 0;
       
  do { 
     erase();
     move( LINES - 1, 0  );
     printw("last Error: %s", last_error_msg );
   
     move(0,0);
     printw("Choose command:" );
     if( marked_item[0] )
       printw(" (marked: '%s' '%s')\n\n", marked_item, marked_file);
     else
       printw("                                            \n\n");
      
     switch( menu_nr )
      { case 0 : /* function menu */
             printw( menu_functions );
             break;
        case 1 :
             printw( menu_variables);
             break;
        case 2 :
             printw( menu_files);
             break;
        case 3 :
             printw( menu_misc, (Sort_flag == SORT_BY_ITEM) ? "sort by item" :
                                ((Sort_flag == SORT_BY_FILE) ? "sort by file" : "none"),
                                (vi_flag) ? "vi" : "internal");
             break;
       }
         
     move(LINES - 3,0);
     printw("   Your choice: ");

     refresh();

     in =  getch();
     switch( in )
       { case '+': 
            menu_nr = (menu_nr < 3) ? menu_nr + 1 : 0;
            in = 0;
            break;
         case '-': 
            menu_nr = (menu_nr > 0) ? menu_nr - 1 : 3;
            in = 0;
            break;
         case '[': 
            vi_flag = !vi_flag;
            in = 0;
            break;
         case 'o': 
            Sort_flag++;
            if( Sort_flag > SORT_BY_FILE && Sort_flag > SORT_BY_ITEM )
              Sort_flag = 0;
            in = 0;
            break;
         case '?':
            help( menu_nr );
            in = 0;
            break;
                        
        }
         
    } while( !in );
   
   erase();
   
   return( in );
 }


main( int argc, char *argv[] )
{ int  server;
  short  ret,
         code,
         static_flag;
  long   line,
         line_end,
         m_id;
   
  short  cmd, flag;
  int    pid, mtype, cnt, m_char;
  char   buffer[INTRFACE_BUFSIZE];

   
  initscr();
  cbreak();
  noecho();
  keypad( stdscr, TRUE );
 
  init_lines();   
  init_file_memory();   
   
  while(1)     
   {  m_char = menu();
      strcpy( last_error_msg, "No Error" );
      MARKING_ALLOWED();
      SET_MARK_TYPES( MARK_FUNCTION, MARK_FILE);
      
      switch( m_char )
       { case 'F': /* Print all functions in project */
           do 
             { ret = brs_get_all_funct( funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               if( ret == RET_VAR )
                 ret = 0;
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
            
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   print_output( "All functions in project", 2, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_get_all_funct( NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case 'c': case 'C': /* print functions called by target */
           MARKING_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_FUNCTION) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter function-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              getstr( right );
              noecho();
             }
           SET_MARK_TYPES( MARK_FUNCTION, MARK_FILE);
           do 
             { /* if 'c', set the sym_once-flag causing the browser to
                  just return the first calling-position instead of all */
               ret = brs_get_funct_calls( left, right, 
                                 ( m_char == 'c' ) ? 1 : 0,
                                 funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   sprintf( right, "Functions called by '%s'%s", left, 
                              (m_char == 'c') ? "" : " (all)");
                   print_output( right, 2, 3);
                   break;

              /* case RET_ERROR: */
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         

           /* reset */
           brs_get_funct_calls( NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case 'b': case 'B': /* print functions calling target-function */
           MARKING_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_FUNCTION) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter function-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              getstr( right );
              noecho();
             }
           SET_MARK_TYPES( MARK_FUNCTION, MARK_FILE);
           do 
             { /* if 'b', set the sym_once-flag causing the browser to
                  just return the first calling-position instead of all */
               ret = brs_get_funct_called_by( left, right, 
                                 ( m_char == 'b' ) ? 1 : 0,
                                 funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: file r2: line 
                    */
                   sprintf( right, "Functions calling '%s'%s", left, 
                              (m_char == 'b') ? "" : " (all)");
                   print_output( right, 2, 3);
                   break;

              /* case RET_ERROR: */
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         

           /* reset */
           brs_get_funct_called_by( NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case 's': case 'S': /* print functions using target-variable */
           MARKING_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_VARIABLE) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter variable-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              getstr( right );
              noecho();
             }
           SET_MARK_TYPES( MARK_FUNCTION, MARK_FILE);
           do 
             { /* if 'c', set the sym_once-flag causing the browser to
                  just return the first calling-position instead of all */
               ret = brs_get_var_used_by( left, right, 
                                 ( m_char == 's' ) ? 1 : 0,
                                 funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: file r2: line 
                    */
                   sprintf( right, "Functions using '%s'%s", left, 
                              (m_char == 's') ? "" : " (all)");
                   print_output( right, 2, 3);
                   break;

              /* case RET_ERROR: */
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         

           /* reset */
           brs_get_var_used_by( NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case 'd':  /* print definition position */
           MARKING_NOT_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_FUNCTION) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter function-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              getstr( right );
              noecho();
             }
           ret = brs_get_funct_pos( left, right, file, &line, &line_end, module);
         
           if( !ret )
             { push_data_left( left, "" );
               sprintf(right, "%ld - %ld", line, line_end );
               push_data_right(file, right, module );
               /* r1: file r2: line 
                */
               sprintf( right, "Definition of function '%s'", left );
               print_output( right, 2, 3);
              }
      
           else
            { switch( ret )
               { case RET_TIMEOUT: 
                      sprintf( last_error_msg, "No connection !\n");
                      break;
         
                 case RET_ERROR:
                      sprintf( last_error_msg, "%s %s !\n", file, module);
                      break;
         
                 /* case RET_ERROR: */
         
                 default:
                      sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, file, module );
                }
             }
         

           reset_line_data();
           break;

         case 'u': case 'U':  /* get variables used by a function */
           MARKING_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_FUNCTION) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter function-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              gets( right );
              noecho();
             }
           SET_MARK_TYPES( MARK_VARIABLE, MARK_FILE);
           
           do 
             { ret = brs_get_var_uses( left, right, 
                                 ( m_char == 'u' ) ? 1 : 0,
                               funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   sprintf( right, "Variables used by '%s'%s", left, 
                              (m_char == 'u') ? "" : " (all)");
                   print_output( right, 2, 3);
                   break;

              /* case RET_ERROR: */
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         

           /* reset */
           brs_get_var_uses( NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL);
                    
           reset_line_data();
           break;
         
         case 'V': /* print all variables in project */
           MARKING_ALLOWED();
           SET_MARK_TYPES( MARK_VARIABLE, MARK_FILE);
           do 
             { ret = brs_get_all_global_vars( funct, file, &line, module, &static_flag);
               if( !ret )
                 { push_data_left( funct, (static_flag) ? "(static)" : "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   print_output( "All global variables defined in project", 2, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_get_all_global_vars( NULL, NULL, NULL, NULL, NULL);
         
           reset_line_data();
           break;
         
         case 'D':  /* definition position of a variable */
           MARKING_NOT_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_VARIABLE) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter variable-name      : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename if static symbol: ");
              refresh();
              getstr( right );
              noecho();
             }
           ret = brs_get_var_pos( left, right, file, &line, module);
           if( !ret )
             { push_data_left( left, "" );
               sprintf(right, "%ld", line );
               push_data_right(file, right, module );
               /* r1: filem r2: line 
                */
               sprintf( right, "Definition of variable '%s'", left );
               print_output( right, 2, 3);
              }
      
           else
            { switch( ret )
               { case RET_TIMEOUT: 
                      sprintf( last_error_msg, "No connection !\n");
                      break;
         
                 case RET_ERROR:
                      sprintf( last_error_msg, "%s %s !\n", file, module);
                      break;
         
                 /* case RET_ERROR: */
         
                 default:
                      sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, file, module );
                }
             }
         
           reset_line_data();
           break;
         
         case 't': /* structs  */
         case 'n': /* unions   */
         case 'm': /* enums    */
         case 'y': /* typedefs */
           MARKING_NOT_ALLOWED();  /* for now, maybe later allowed if we reuse types */
         
           switch( m_char )
            { case 't': code = 0; strcpy( left, "All structs in project" ); break;
              case 'n': code = 1; strcpy( left, "All unions in project" ); break;
              case 'm': code = 2; strcpy( left, "All enums in project" ); break;
              case 'y': code = 3; strcpy( left, "All typedefs in project" ); break;
             }
             
           do 
             { ret = brs_get_all_types( code, funct, file, &line);
               if( !ret )
                 { push_data_left( funct, "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, "" );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s!\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   print_output( left, 2, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_get_all_types( 0, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case '.': /* reference components  */
         case ':': /* reference components with variables  */
           MARKING_NOT_ALLOWED();  /* for now, maybe later allowed if we reuse types */
         
           left[0] = 0;
           right[0] = 0;
           printw("\nEnter component-name      : ");
           refresh();
           echo();
           getstr( left );
           noecho();
         
           switch( m_char )
            { case '.': 
                  code = 1;
                  if( CHECK_MARK_ITEM_TYPE( MARK_FUNCTION) )
                    { strcpy( right, marked_item );
                     }
                  else
                   { 
                     printw("\n\nFunction-name (or \"\" if global) : ");
                     refresh();
                     echo();
                     gets( right );
                     noecho();
                    }
            
                  if( right[0] == 0 )
                   { if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                       { strcpy( right, marked_item );
                        }
                     else
                      { 
                        printw("\n\nNo Function specified, select file-name \n");
                        printw("\nFile-name (or \"\" if global) : ");
                        refresh();
                        echo();
                        gets( right );
                        noecho();
                       }
                     code = 0;
                    }
                  break;
            
              case ':': 
                  code = 2;
            
                  if( CHECK_MARK_ITEM_TYPE( MARK_VARIABLE) )
                    { strcpy( right, marked_item );
                     }
                  else
                   { 
                     printw("\n\nVariable-name          : ");
                     refresh();
                     echo();
                     gets( right );
                     noecho();
                    }
                  break;
             }
             
           do 
             { ret = brs_get_component_use( code, left, right, file, &line, funct);
               if( !ret )
                 { push_data_left( file, "" );
                   sprintf(right, "%ld", line );
                   push_data_right(funct, right, "" );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s!\n", funct, file);
                   break;
         
              case RET_END:
                   /* l1: filem r2: line 
                    */
                   sprintf(right, "Usage of component '%s'", left);
                   print_output( right, 0, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_get_component_use( 0, NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;
         
         case 'f': /* Print all files in project */
           MARKING_NOT_ALLOWED();
           do 
             { ret = brs_get_all_files( file, module);
               if( !ret )
                 { push_data_left( file, "" );
                   push_data_right( module, "", "" );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
            
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", file, module);
                   break;
         
              case RET_END:
                   /* l1: filem  line: none
                    */
                   print_output( "All files in project", 0, -1);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, file, module );
             }
         
           /* reset function for next call
            */
           brs_get_all_files( NULL, NULL);

           reset_line_data();
           break;

         case 'W': /* Print Whatis */
           MARKING_NOT_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           printw("\nEnter ident-name      : ");
           refresh();
           echo();
           getstr( left );
           printw("\nFilename (or \"\")      : ");
           refresh();
           getstr( right );
           noecho();
           do 
             { ret = brs_what_is( left, right, 0, file, &line, module );
               switch( ret )
                 { case RET_MACRO:
                      push_data_left( left, "Macro" );
                      break;
                   case RET_VAR:
                      push_data_left( left, "Variable" );
                      break;
                   case RET_FUNCT:
                      push_data_left( left, "Function" );
                      break;
                   case RET_STRUCT:
                      push_data_left( left, "Struct" );
                      break;
                   case RET_UNION:
                      push_data_left( left, "Union" );
                      break;
                   case RET_ENUM:
                      push_data_left( left, "Enum" );
                      break;
                   case RET_TYPEDEF:
                      push_data_left( left, "Typedef" );
                      break;
                   case RET_COMP:
                      push_data_left( left, "struct/union comp" );
                      break;
                   case RET_ENUM_C:
                      push_data_left( left, "enum constant" );
                      break;
                   case RET_END: 
                   case RET_ERROR:
                   case RET_TIMEOUT:
                      break;
                   default:
                      break;
                  }
              if( ret != RET_END && ret != RET_ERROR && ret != RET_TIMEOUT )
                 { sprintf(right, "%ld", line );
                   push_data_right(file, right, module );
                  }
              } while( ret != RET_END && ret != RET_ERROR && ret != RET_TIMEOUT );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
            
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   print_output( "Whatis ...", 2, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_what_is( NULL, NULL, 0, NULL, NULL, NULL );

           reset_line_data();
           break;

         case '#': /* Print all macro-definitions in project */
           MARKING_ALLOWED();
           SET_MARK_TYPES( MARK_MACRO, MARK_FILE);
           do 
             { ret = brs_get_all_macro( funct, file, &line);
               if( !ret )
                 { push_data_left( funct, "" );
                   sprintf(right, "%ld", line );
                   push_data_right(file, right, "" );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
            
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", funct, file);
                   break;
         
              case RET_END:
                   /* r1: filem r2: line 
                    */
                   print_output( "All macro definitions in project", 2, 3);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, funct, file );
             }
         
           /* reset function for next call
            */
           brs_get_all_funct( NULL, NULL, NULL, NULL, NULL);

           reset_line_data();
           break;

         case '*':  /* definition position of a macro */
           MARKING_NOT_ALLOWED();
           left[0] = 0;
           right[0] = 0;
           if( CHECK_MARK_ITEM_TYPE( MARK_MACRO) )
             { strcpy( left, marked_item );
               if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
                 strcpy( right, marked_file );
              }
           else
            { printw("\nEnter macro-name  : ");
              refresh();
              echo();
              getstr( left );
              printw("\nFilename          : ");
              refresh();
              getstr( right );
              noecho();
             }
           ret = brs_get_macro_pos( left, right, file, &line, module);
           if( !ret )
             { push_data_left( left, "" );
               sprintf(right, "%ld", line );
               push_data_right(file, right, module );
               /* r1: filem r2: line 
                */
               sprintf( right, "Definition of macro '%s'", left );
               print_output( right, 2, 3);
              }
      
           else
            { switch( ret )
               { case RET_TIMEOUT: 
                      sprintf( last_error_msg, "No connection !\n");
                      break;
         
                 case RET_ERROR:
                      sprintf( last_error_msg, "%s %s !\n", file, module);
                      break;
         
                 /* case RET_ERROR: */
         
                 default:
                      sprintf( last_error_msg, "\n**** Error: Code %hd %s %s\n", ret, file, module );
                }
             }
         
           reset_line_data();
           break;
         
         case 'i':  /* include files */
           MARKING_NOT_ALLOWED();
           funct[0] = 0;
           if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
             { strcpy( funct, marked_file );
              }
           else
            { printw("\nEnter main filename: ");
              refresh();
              echo();
              getstr( funct );
              noecho();
             }
           
           do 
             { ret = brs_get_includes( funct, file, module, &static_flag);
               if( !ret )
                 { push_data_left( file, (static_flag) ? "(system)" : "" );
                   push_data_right( module, "", "" );
                  }
               } while( !ret );     
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", file, module);
                   break;
         
              case RET_END:
                   /* l1: file  no line
                    */
                   sprintf( right, "Include-files of '%s'", funct );
                   print_output( right, 0, -1);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s\n", ret, file );
                   break;
             }
         
           /* reset function for next call
            */
           brs_get_includes( NULL, NULL, NULL, NULL);
                  
           reset_line_data();
           break;
         
         case 'w':  /* working directory of a file */
           MARKING_NOT_ALLOWED();
           file[0] = 0;
           if( CHECK_MARK_FILE_TYPE( MARK_FILE) )
             { strcpy( file, marked_file );
              }
           else
            { printw("\nEnter filename: ");
              refresh();
              echo();
              getstr( file );
              noecho();
             }
           
           ret = brs_get_file_dir( file, module );
           if( !ret )
             { push_data_left( file, "" );
               push_data_right( module, "", "" );
              }
      
           switch( ret )
            { case RET_TIMEOUT: 
                   sprintf( last_error_msg, "No connection !\n");
                   break;
         
              case RET_ERROR:
                   sprintf( last_error_msg, "%s %s !\n", file, module);
                   break;
         
              case 0:
                   /* l1: file  no line
                    */
                   sprintf( right, "Working directory of '%s'", file );
                   print_output( right, -1, -1);
                   break;
         
              default:
                   sprintf( last_error_msg, "\n**** Error: Code %hd %s\n", ret, file );
                   break;
             }
         
           reset_line_data();
           break;
         
         case 'l': /* clear marking */
           SET_MARK_TYPES( MARK_NULL, MARK_NULL);
           marked_item[0] = 0;
           marked_file[0] = 0;
           break;
         
         case 'T': /* new timeout*/
           right[0] = 0;
           printw("Old Timeout-value: %ld seconds \n", timeout_value);
           printw("Enter new value:  ");
           refresh();
           echo();
           getstr( right );
           noecho();
           timeout_value = atoi( right );
           sprintf( last_error_msg, "New Timeout set to %ld seconds", timeout_value );
           break;
         
         case 'R':  /* browser filename */
           printw("Enter new browser filename: ");
           refresh();
           echo();
           getstr( funct );
           noecho();
           ret = brs_reinit_browser( funct, left, right );
           if( ret )
             { printw("\nError: Code %hd %s %s\n", ret, left, right );
               Exit(1);
              }
           break;
         
         case 'r':  /* New server */
           printw("Enter browser filename to look for: ");
           refresh();
           echo();
           getstr( funct );
           noecho();
           ret = brs_get_server_id( funct );
           if( ret < 0 )
             { printw("\nError: Code %hd \n", ret );
               Exit(1);
              }
           brs_disconnect_server();
           server = brs_connect_server( ret );
           if( server != 0 )
             { printw("\nError: Code %hd \n", server );
               Exit(1);
              }
           break;
         
         case 'k':
           ret = brs_kill_server();
           if( ret )
             { printw("\nError: Code %hd \n", ret );
               Exit(1);
              }
           Exit(0);
         
         case 'q': /* quit */
           brs_disconnect_server();
           Exit(0);
         
         case 'a': /* get all messages in queue */
           while(1)
            { buffer[0] = 0;
              cnt = brs_get_any_message( &mtype, &cmd, &flag, &pid, &m_id, buffer);
              if( cnt == -1 )
                break;
              sprintf(left, "Message (%d bytes): mtype: %d, cmd: %hd, flag %hx, pid: %d, msg-id: %ld",
                               cnt, mtype, cmd, flag, pid, m_id );
              sprintf(right, "Buffer            : \"%s\"", buffer );
            
              push_data_left( left, "" );
              push_data_right("", "", "" );
            
              push_data_left( right, "" );
              push_data_right("", "", "" );
            
             }
         
           print_output( "All messages in message queue", -1, -1);
           reset_line_data();
           break; 
           
         default:
           sprintf( last_error_msg, "Unknown Command '%c' (0x%02x)", m_char );
           break;
        }
    } /* endwhile */
}
         
   
