/*
    brs_server : Browsing an Analysis Tool for C-source code;
	         Browser-Engine, runs in background

    Copyright (C) 1993  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

%{



   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>
   #include <time.h>
   #include <unistd.h>
   #include <errno.h>

   #include "hash.h"
   #include "c-bat.h"

   /*
   #define YY_USER_ACTION  {if( command != CMD_INIT ) 
      fprintf(stderr,"\"%s\" %d\n",yytext, yy_act);}
    */
   
   /* undefine yywrap ! Will be a function to switch
      form one browser-file to the next
    */
   
   #undef   yywrap()

   
   /* Declaration of global variables
    */

   #define TYPE_UNKNOWN           0x0 /* invald entry ! */
   #define TYPE_FUNCTION_CALL     0x1 /* types can be OR'ed together */
   #define TYPE_FUNCTION_DEF      0x2
   #define TYPE_MAIN_FILE         0x4
   #define TYPE_FUNCT_CALLED      0x8
   #define TYPE_MODULE           0x10
   #define TYPE_MODULE_FILE      0x20
   #define TYPE_STATIC           0x40
   #define TYPE_SOURCE_FILE      0x80
   #define TYPE_GLOBAL_VAR_DEF  0x100
   #define TYPE_LOCAL_VAR_DEF   0x200
   #define TYPE_ENUM            0x400
   #define TYPE_STRUCT          0x800
   #define TYPE_UNION          0x1000
   #define TYPE_ENUM_CONST     0x2000
   #define TYPE_COMPONENT      0x4000
   #define TYPE_TYPEDEF        0x8000
   #define TYPE_MACRO         0x10000

   #define ILL_F_POS            (-1L) /* flag for illegal file-position */
   
   typedef unsigned long  t_type;

   /* currently not used:
   #define TYPE_GLOBAL_VAR_USE    0x8
   #define TYPE_LOCAL_VAR_USE    0x10
   */

   /* Flags used in Hash-Table-Entry
    */
   #define FLAG_PRINTED  0x1     /* Print Identifier  */
   #define FLAG_IGNORE   0x2     /* Ignore Identifier */
   #define FLAG_IN_HASH  0x4     /* Allready inserted in Hash-Index-Array */


   /* Flags used in Hash-Add-Entry
    */
   #define HD_FLAG_STATIC  0x1 /* This instance is static within file */

   #ifdef DEBUG
   int  debug = 1;
   #else
   int  debug = 0;
   #endif

   static char main_input_filename[MAX_FILENAME]; /* Main Source File            */
   static char input_filename[MAX_FILENAME];      /* act. Source or Include-File */

   static char funct_filename[MAX_FILENAME];      /* File, where search-function */

   static char var_filename[MAX_FILENAME];        /* File, where variable defined*/

   static char target_filename[MAX_FILENAME];     /* filename of target_function */
					          /* (specified by the user)     */
   static char module_file[MAX_FILENAME];         /* file with module-file-assign*/
   static char exclude_file[MAX_FILENAME];        /* list of exclude symbols     */
   static char browser_file[MAX_FILENAME];        /* Name of browser controlfile */

   static char *stop_file_name;                   /* Pointer to argv of stop-file*/

   static char act_function[MAX_IDENT];           /* act. Function Ident         */
   static char act_f_call[MAX_IDENT];             /* Ident of called function    */

   static char act_variable[MAX_IDENT];           /* act. Variable Ident         */
   static char act_v_use[MAX_IDENT];              /* Ident of used variable      */

   static char act_component[MAX_IDENT];          /* Ident of actual component   */

   static char act_module[MAX_IDENT];             /* Ident of actual module      */
   static char target_function[MAX_IDENT];        /* Ident of searched function  */
   static char target_variable[MAX_IDENT];        /* Ident of searched variable  */
   static char target_component[MAX_IDENT];       /* Ident of searched component */

   static long   target_function_start;  /* line, where target-function starts */
   static long   target_function_end;    /* line, where target-function end    */
   static long   function_start;         /* line, where normal function starts */
   static long   function_end;           /* line, where normal function end    */

   static long   f_line;     /* Source-Line, where function called/defined */
   static long   f_line2;    /* Source-Line, where called function defined */
   static long   v_line;     /* Source-Line, where variable used/defined   */
   static long   ev_line;    /* Source-Line, where variable used           */
   static long   target_line;  /* Target-Line fuer CMD_DEF_POS             */

   #define  CR_SIZE   1   /* CR/LF 2 Bytes under DOS, 1 under UNIX */
   static long   lineno = 1;
   static long   file_pos = 0;
   static long   goto_file_pos = 0;
   
   static long   start_file_pos;  /* starting filepos of actual tag */

   static short  command;    /* command read in by Interface-routines */
   
   time_t  lex_time;


   /* FLAGS
    */
   
   /* set to 0, 1 or 2: control of yy_flex_debug - flag
    */
   static short flex_debug_flag = 0;
   
   /* set to 1 if scanner should print unrecognized characters
    */
   static short flex_debug2_flag = 0;
   
   /* [EOF] in browser-file found -> according to actual implementation,
      where all browser-files are just concatenated together, we can find
      out that way, where a new browser-file begins    */
   static short  eof_detected = 1;

   /* Set to 1, if we found allready the target-function to encounter
      a double defined target-function -> would produce an error    */
   static short  function_found = 0;

   /* Set to 1, if we are in the interesting function
    */
   static short  interesting_function = 0;

   /* Set to 1, if we can stop after reaching end of actual main-file
    */
   static short  act_mainfile_only = 0;

   /* set to 1 , if we are at the level of the main input file
    */
   static short  main_flag = 0;

   /* set to 1, if actual include is a system-include
    */
   static short  sys_include = 0;

   /* char of [E?]-Tag
    */
   static char  var_c;

   /* set to 1, if actual function is static
    */
   static short static_function = 0;

   /* set to 1, if actual variable is static
    */
   static short static_variable = 0;
   
   /* set to 1 if we changed the flags of hash-symbols
    */
   static short  hash_table_changed = 0;
   
   /* set to 1 if we changed the flags of hash-symbols
    */
   static short  clear_invalid_hash_table_entrys = 0;

   /* set to 1 if every symbol should only printed once
    */
   static short  symbols_only_once = 0;

   /* All symbols, which have to be printed, are stored in the hashtable
      and aditionally stored in this array (by their hash-indizes).
      So we don't have to check the whole hash-table for nonnull-entries
    */
   static long *hash_array = NULL;
   static long hash_array_cnt = 0;
   static long hash_array_size = 0;
   
   /* All symbols, that has been changed in the hash-table (either added or
      changed flag)
    */
   static long *hash_changed_array = NULL;
   static long hash_changed_array_cnt = 0;
   static long hash_changed_array_size = 0;
   
   static int  cont_state;  /* used to continue rules, which can occure in */
                            /* two or more states                          */

   /* Additional data, stored in Hash-Table. Here used to store position
      of any function defined in the source-project for the output
      "defined in ... at line ..."

      One identifier may appear in the Hashtable only once although it might
      have differen meanings in several scopes (e.g. static/global function
      with the same name)

      Although things like that belong not to a good programming style,
      we have to handle this with this.

      t_hash_add is also used for linked data strutures like modules, which
      consist of many files.
    */
   typedef struct hash_add_data  t_hash_add;

   struct hash_add_data
     { long        file;  /* File, where Function is defined (hash-index) */
       long        line;  /* Line, where Function is defined              */
       long     end_line; /* Line, where Function ends (Functions only)   */
       t_hash_add  *next; /* next position; used for static symbols       */
       long        f_pos; /* Position in Scanner-File                     */
       t_type       type;  /* Type of identifier (not or-ed together)      */
      };
   
   /* This struct is used for storing the browser-files
    */
   typedef struct brs_file  t_brs_file;
   
   struct brs_file
     { char        *file;       /* filename                         */
       char        *dir;        /* working directory of this file   */
       long        pos_offset;  /* offset of this file (CMD_INIT)   */
       t_brs_file  *next;       /* next brs-file or NULL at the end */
      };

   /* List with all browser-files and their working directories
    */
   static t_brs_file  *brs_file_list = NULL;
   static t_brs_file  *act_brs_file  = NULL; 
   static char act_brs_filename[MAX_FILENAME]; 
   static char act_brs_dir[MAX_FILENAME]; 
   
   /* This pointer is set to the hash-data of the actual 
      processed function. It is used to write the line of the
      function end
    */
   static t_hash_add  *data_of_act_function = NULL;
   
   /* prototypes
    */
   void reset_hashtable(void);
   long Add_ident_to_hash(char *ident, short flag, void *add_data, t_type type );
   void get_file_or_module ( t_hash_data  *data, long index);
   void Add_to_Hash_Index_Array(long index);
   void Add_to_Hash_changed_Index_Array(long index);
   void Add_all_function_to_hash( char *function );
   void Add_all_variables_to_hash( char *variable );
   void Add_all_items_to_hash( char *item, short static_flag, t_type type );
   void f_call_found( char *function );
   void v_use_found( char * variable );
   void Store_Source_File( short main_flag );
   short end_f_call(void);
   short end_v_use(void);
   short end_f_called(void);
   short end_reference(void);
   short get_item_pos(char *arg1, char *arg2, t_type type,
			 t_hash_add *h_erg1, t_hash_add *h_erg2, short strict_flag);
   void return_f_pos( char *function, char *file, t_type type );
   int  set_browser_position(long new_pos);
   void reset_brs_file_list();
   void Add_to_brs_file_list( char * brs_file, char *brs_dir);
   t_brs_file * get_brs_list_entry(long pos);
   
   extern void set_browser_filename(char *filename);
   extern int client_pid; /* pid of actual client process */

%}

DIGIT   [0-9]
ID      [a-zA-Z_][a-zA-Z_0-9]*
INT     {DIGIT}*
FILEN   [^\"<>]*
WHITE   [\t ]*

%x toplev inc_file f_def f_decl in_funct f_call f_call_end1 f_call_end2 v_def lv_def
%x v_use v_use_end1 v_use_end2 d_enum d_comp d_const f_call2 f_call2_end1 f_call2_end2
%x control control2 definition component

%%

          
<toplev,in_funct>\n      {
	lineno++;
	file_pos += yyleng;
	}

\n      {
	lineno++;
	file_pos += yyleng;
	}

<control>"\""{FILEN}"\""	 {  /* name of the browser-file */
        BEGIN( control2 );
        strcpy( act_brs_filename, yytext + 1 );
        act_brs_filename[yyleng-2] = 0;
        }

<control,control2>{WHITE}	 {     /* eat up whitespaces */
        ;
        }

<control,control2>\n		 {     /* eat up newline */
        ;
        }

<control2>"\""{FILEN}"\""	{ /* read working dir */
        BEGIN( control );
        strcpy( act_brs_dir, yytext + 1 );
        act_brs_dir[yyleng-2] = 0;
        Add_to_brs_file_list( act_brs_filename, act_brs_dir );
        }


<toplev>"[I0]\""	{ /* Include-File started */
	sys_include = 0;
	BEGIN(inc_file);

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[R]\""{ID}"\""	{ 
        /* Reference to a struct or union component */
        strcpy( act_component, yytext + 4 );
        act_component[yyleng - 5] = 0;
	file_pos += yyleng;
	cont_state = toplev;
        BEGIN(component);
	}
   
<in_funct>"[R]\""{ID}"\""	{ 
        /* Reference to a struct or union component */
        strcpy( act_component, yytext + 4 );
        act_component[yyleng - 5] = 0;
	file_pos += yyleng;
	cont_state = in_funct;
        BEGIN(component);
	}

<component>"\""{ID}"\""	{ 
        /* Reference to a struct or union component */
        strcpy( act_variable, yytext + 1 );
        act_variable[yyleng - 2] = 0;
	switch( command )
	  { case CMD_INIT: /* Beim Init ist nichts zu tun */
			   break;
      
            case CMD_REFERENCE_F: 
            case CMD_REFERENCE_V: 
            case CMD_REFERENCE:   
                 /* struct or union component use global or file
                  */
                 /* I know, I do abuse act_f_call, but I need a
                    free string variable to do the sprintf
                  */
                 if( !strcmp(act_component, target_component) )
                   { long tmp_index;
         
                     tmp_index = search_token(input_filename, NULL);
                     if( tmp_index != TOKEN_NOT_FOUND && 
                         (get_token_flag(tmp_index) & FLAG_IGNORE) )
                       break;
         
                     tmp_index = search_token(act_function, NULL);
                     if( tmp_index != TOKEN_NOT_FOUND && 
                         (get_token_flag(tmp_index) & FLAG_IGNORE) )
                       break;
         
                     sprintf(act_f_call, "%ld", f_line );
                     if( command == CMD_REFERENCE_V &&
                          strcmp( target_variable, act_variable) )
                        break;
         
                     send_brs_ret( RET_REFERENCE, 0, client_pid, 
                            input_filename, act_f_call , act_function, "" );
                    }
		 break;
      
	    default:       break;
	   }
	file_pos += yyleng;
	BEGIN( cont_state );
	}

<toplev>"[I1]\""	{ /* Include-File started */
	sys_include = 1;
	BEGIN(inc_file);

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<in_funct>"[f]["{INT}        { /* implicite Function declaration */
	/* Deklaration inside function means call to a unprototyped funct. */
	f_line = atoi( yytext + 4 );

	BEGIN(f_call);
	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[f]["{INT}     { /* Function deklaration */
	f_line = atoi( yytext + 4 );

	BEGIN(f_decl);
	file_pos += yyleng;
	}
   
<toplev>"[P]["{INT}     { /* definition */
	f_line = atoi( yytext + 4 );

	BEGIN(definition);
	file_pos += yyleng;
	}
   
<in_funct>"[DV]["{INT}"]"    { /* local variable defined */
	v_line = atoi( yytext + 5 );

	BEGIN( lv_def );
	static_variable = 0;

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<in_funct>"[DVs]["{INT}"]"    { /* local variable defined
				      BUG: static doesn't work !
				    */
	v_line = atoi( yytext + 6 );

	BEGIN( lv_def );
	/* static_variable = 1; */

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[DV]["{INT}"]"    { /* variable defined */
	f_line = atoi( yytext + 5 );

	BEGIN( v_def );
	static_variable = 0;

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[DVs]["{INT}"]"    { /* static variable defined */
	f_line = atoi( yytext + 6 );

	BEGIN( v_def );
	static_variable = 1;

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[Fs]["{INT}    { /* Static function definition */
	f_line = atoi( yytext + 5 );

	BEGIN(f_def);
	static_function = 1;

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev>"[F]["{INT}     { /* Function definition */
	f_line = atoi( yytext + 4 );

        target_function_start = f_line;
	BEGIN(f_def);
	static_function = 0;

	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<in_funct>"[fe]["{INT}"]"    {

	f_line = atoi( yytext + 5 );
	target_function_end = f_line;
	start_file_pos = file_pos;

	switch( command )
	  { case CMD_INIT: /* store end-of-function lineno */
                 if( data_of_act_function )
                    data_of_act_function->end_line = f_line;
	         break;

	    case CMD_CALLS: /* are we examining target function ?  */
            case CMD_USES:
            case CMD_CALLS_ONCE:
            case CMD_USES_ONCE:
            case CMD_REFERENCE_F:

		 /* If we are in interesting function, we can stop now
		  */
		 if( interesting_function )
		   { interesting_function = 0;

		     /* Tell interface-module that search-command is
			completed. If nothing else to do, we can quit yylex
		      */
                     if( command == CMD_CALLS || command == CMD_CALLS_ONCE )
		       { if( end_f_call() )
                           return;
                        }
                     else if( command == CMD_REFERENCE_F )
		       { if( end_reference() )
                           return;
                        }
                     else
		       { if( end_v_use() )
                           return;
                        }
		    }
		 break;

	    default:       break;
	   }

	BEGIN( toplev );
        act_function[0] = 0;
	file_pos += yyleng;
	}

<toplev>"[E"(f|F)("]["{INT})   { /* function pointer outside function */
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

	BEGIN(f_call2);
	file_pos += yyleng;
	}

<in_funct>"[E"(f|F)("]["{INT})   {
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

	BEGIN(f_call);
	file_pos += yyleng;
	}

<toplev>"[E"(V|v)("]["{INT})   {
	f_line = atoi( yytext + 5 );
      
        /* if we are NOT in the same line, reset the ev_line. if
           we are in the same line, keep this value (in case of
           array-references)
         */

	start_file_pos = file_pos;

	var_c = yytext[2];  /* store character V/v */

	BEGIN(v_use);
	file_pos += yyleng;
        cont_state = toplev;
	}

<in_funct>"[E"(V|v)("]["{INT})   {
	f_line = atoi( yytext + 5 );
      
        /* if we are NOT in the same line, reset the ev_line. if
           we are in the same line, keep this value (in case of
           array-references)
         */

	start_file_pos = file_pos;

	var_c = yytext[2];  /* store character V/v */

	BEGIN(v_use);
	file_pos += yyleng;
        cont_state = in_funct;
	}

<toplev>"[EOF"{INT}"]"                  {
	eof_detected = 1;

	switch( command )
	  {
	    case CMD_CALLED_BY_ONCE:
	    case CMD_CALLED_BY:
		 /* If we look for a static function (act_mainfile_only),
		    we can stop now
		  */
                 if( act_mainfile_only )
                    { if( end_f_called() )
                        return;
                     }
                 break;

	    case CMD_USED_BY_ONCE:
	    case CMD_USED_BY:
		 /* If we look for a static variable (act_mainfile_only),
		    we can stop now
		  */
                 if( act_mainfile_only )
                    { if( end_f_called() )
                        return;
                     }
                 break;
      
	    case CMD_REFERENCE:
	    case CMD_REFERENCE_F:
	    case CMD_REFERENCE_V:
                 if( act_mainfile_only )
		   { if( end_reference() )
                       return;
                    }
      
            case CMD_INCLUDES: /* get includes of file */

                 if( act_mainfile_only )
                    { send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );
		      return;
                     }
                 break;
      
            case CMD_WHAT_IS: /* WHAT_IS */

                 if( act_mainfile_only )
                    { send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );
		      return;
                     }
                 break;
      
            case CMD_DEF_POS: /* Scoping and print definition position */
                 /* last try, it may be a Macro or something else stored
                    in the hash-table
                  */
                 { long tmp_index;
         
                   tmp_index = search_token( target_function, NULL);
                   if( tmp_index == TOKEN_NOT_FOUND )
                      send_brs_ret( RET_ERROR, 0, client_pid, "Item not found", "", "", "" );
                   else
                     { t_hash_data   data;
                       t_hash_add    *h_data;
                       char *ptr;
            
                       get_token( &data, tmp_index );
               
                       h_data = (t_hash_add *)data.data;
                       ptr = get_symbol( h_data->file );
                       if( ptr && ptr[0] )
                         { sprintf(act_f_call, "%ld", h_data->line );
                           send_brs_ret( RET_POS, 0x0, client_pid, ptr, 
                                                 act_f_call, "", "" );
                          }
                      }
		   return;
                  }

	    default:
		 break;
	   }


	start_file_pos = file_pos;
	file_pos += yyleng;
	}

<toplev,in_funct>"[Dv"s?("]["{INT}"]\""{ID}"\"")	{ /* Declaration of var */
	/* ignore at the moment */
	file_pos += yyleng;
        }

<toplev>"[D"(E|S|T|U)("]["{INT})  	{ /* Definition of enum,struct or union */
	f_line = atoi( yytext + 5 );

        var_c = yytext[2];
	start_file_pos = file_pos;

        cont_state = toplev;
	BEGIN(d_enum);
	file_pos += yyleng;
	}

<in_funct>"[D"(E|S|T|U)("]["{INT})	{ /* Definition of enum,struct or union */
	f_line = atoi( yytext + 5 );

        var_c = yytext[2];
	start_file_pos = file_pos;

        cont_state = in_funct;
	BEGIN(d_enum);
	file_pos += yyleng;
	}
	   
<toplev>"[De]["{INT}  		{ /* Definition of enum-const */
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

        cont_state = toplev;
	BEGIN(d_const);
	file_pos += yyleng;
	}

<in_funct>"[De]["{INT}		{ /* Definition of enum-const */
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

        cont_state = in_funct;
	BEGIN(d_const);
	file_pos += yyleng;
	}
	   
<toplev>"[SC]["{INT}  		{ /* Definition of stuct or union component */
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

        cont_state = toplev;
	BEGIN(d_comp);
	file_pos += yyleng;
	}

<in_funct>"[SC]["{INT}		{ /* Definition of stuct or union component */
	f_line = atoi( yytext + 5 );

	start_file_pos = file_pos;

        cont_state = in_funct;
	BEGIN(d_comp);
	file_pos += yyleng;
	}
	   
<toplev,in_funct>"[E"(P|T|C)("]["{INT}"]".*)$	{ /* Usage */
	f_line = atoi( yytext + 5 );
        
        if( yytext[2] == 'P' )
          { /* Usage of formal parameter */
      
            /* if we are NOT in the same line, reset the ev_line. if
               we are in the same line, keep this value (in case of
               array-references)
             */
      
            if( strchr( yytext, '"' ) )
              { strcpy( act_v_use, strchr( yytext , '"') + 1 );
                act_v_use[ strlen(act_v_use) - 1] = 0;
               }
            else
              act_v_use[0] = 0;
      
            if( command == CMD_DEF_POS && f_line == target_line )
              { if( !strcmp(act_v_use, target_function) )
                   { sprintf(act_f_call, "%ld", target_function_start );
                     send_brs_ret( RET_POS, 0x0, client_pid, input_filename, 
                                                 act_f_call, "", "" );
                     return;
                    }
               }
      
           }
   
	/* ignore at the moment */
	file_pos += yyleng;
        if( yytext[yyleng-1] == '\n' )
          lineno++;
        }
   
   
<inc_file>({FILEN}|"<predefined>")"\""	{

	if( yyleng > 1)
          { if(eof_detected)
	      { /* Last File-Tag was "EOF", so  the next File-Tag is 
                   from a main (*.brs) file
	         */
	        eof_detected = 0;
	        strcpy( main_input_filename, yytext );
	        main_input_filename[yyleng - 1] = 0;
	        strcpy( input_filename, main_input_filename );
	        main_flag = 1;
	       }
	    else
	      { /* normal include-file, first copy previous file 
                   for CMD_INCLUDES
	         */
                strcpy( act_variable, input_filename );
	        strcpy( input_filename, yytext );
	        input_filename[yyleng - 1] = 0;
	        main_flag = 0;
	       }
           }
        else
         { /* if yyleng == 1: it is a illegale file (the file
               with the includes)
             */
           input_filename[0] = 0;
           main_input_filename[0] = 0;
          }

	switch( command )
	  { case CMD_INIT: if( input_filename[0] )
                              Store_Source_File( main_flag );
			   break;

            case CMD_INCLUDES: /* get includes of file */
                           if( strcmp(input_filename, main_input_filename)  )
                             { /* just list the include-files, not the mainfile.
                                  Return actual inputfile as module, so we can
                                  detect nested includes
                                */
                               short flags;
                               long  index;
         
                               index = search_token( input_filename, NULL);
                               if( index != TOKEN_NOT_FOUND )
                                 { /* This token has not been printed yet
                                    */
                                   flags = get_token_flag( index );
                                     if( !(flags & FLAG_PRINTED) &&
                                         !(flags & FLAG_IGNORE)     )
                                       { send_brs_ret( RET_FILE, 
                                            (sys_include) ? 0x03 : 0x01, 
                                            client_pid, input_filename, 
                                            act_variable , "", "" );
                                         flags |= FLAG_PRINTED;
                                         set_token_flag( index, flags );
                                         Add_to_Hash_changed_Index_Array( index);
                                         hash_table_changed = 1;
                                        }
                                  }
                              }
	    default:       break;
	   }

	BEGIN( toplev );
	file_pos += yyleng;
	}

<f_decl>"]\""{ID}"\""    { /* Function deklaration */
	strcpy( act_function, yytext + 2 );
	act_function[yyleng-3] = 0;

	BEGIN( toplev );
	file_pos += yyleng;
	}

<definition>"]\""{ID}"\""    { /* macro definition */
	strcpy( act_variable, yytext + 2 );
	act_variable[yyleng-3] = 0;
     
        Add_all_items_to_hash( act_variable, 0, TYPE_MACRO );

	BEGIN( toplev );
	file_pos += yyleng;
	}

<d_enum>"]\""{ID}?"\""    { /* definition of enum,struct or union */
	t_type  local_type = 0;

	strcpy( act_variable, yytext + 2 );
	act_variable[yyleng-3] = 0;

        switch( var_c )
         { case 'E': local_type = TYPE_ENUM;   break;
	   case 'S': local_type = TYPE_STRUCT; break;
	   case 'U': local_type = TYPE_UNION;  break;
	   case 'T': local_type = TYPE_TYPEDEF;  break;
          }

	switch( command )
	  { case CMD_INIT: Add_all_items_to_hash( act_variable, 0, local_type );
			   break;

	    default:       break;
	   }

	BEGIN( cont_state );
	file_pos += yyleng;
	}

<d_comp>"]\""{ID}?"\""    { /* definition of struct or union component */

	strcpy( act_variable, yytext + 2 );
	act_variable[yyleng-3] = 0;

	switch( command )
	  { case CMD_INIT: /* nothing to do */
			   break;
            case CMD_WHAT_IS:
               /* definition of a component */
               if( ! strcmp( act_variable, target_variable) )
                 { sprintf(act_f_call, "%ld", f_line );
         
                   send_brs_ret( RET_COMP, 0, client_pid, 
                          input_filename, act_f_call , act_variable, "" );
                  }
               break;

	    default:       break;
	   }

	BEGIN( cont_state );
	file_pos += yyleng;
	}


<d_const>"]\""{ID}"\""    { /* definition of enum-const */

	strcpy( act_variable, yytext + 2 );
	act_variable[yyleng-3] = 0;

	switch( command )
	  { case CMD_INIT: /* nothing to do */
			   break;

            case CMD_WHAT_IS:
               /* definition of a enum constant */
               if( ! strcmp( act_variable, target_variable) )
                 { sprintf(act_f_call, "%ld", f_line );
         
                   send_brs_ret( RET_ENUM_C, 0, client_pid, 
                          input_filename, act_f_call , act_variable, "" );
                  }
               break;
   
	    default:       break;
	   }

	BEGIN( cont_state );
	file_pos += yyleng;
	}

<lv_def>"\""{ID}"\""     { /* local variable definition */
	strcpy( act_variable, yytext + 1 );
	act_variable[yyleng-2] = 0;


	switch( command )
	  { case CMD_INIT: /* local variables not into hash-table */
			   break;
            case CMD_WHAT_IS:
                 /* local variable */
                 if( ! strcmp( act_variable, target_variable) )
                   { sprintf(act_f_call, "%ld", v_line );
         
                     send_brs_ret( RET_VAR, 0, client_pid, 
                            input_filename, act_f_call , act_variable, "" );
                    }
                 break;

	    default:       break;
	   }


	BEGIN(in_funct);
	file_pos += yyleng;
	}

<v_def>"\""{ID}"\""     { /* variable definition */
	strcpy( act_variable, yytext + 1 );
	act_variable[yyleng-2] = 0;


	switch( command )
	  { case CMD_INIT: Add_all_items_to_hash( act_variable, static_variable, 
                                                  TYPE_GLOBAL_VAR_DEF );
			   break;

	    default:       break;
	   }


	BEGIN( toplev );
	file_pos += yyleng;
	}

<f_def>"]\""{ID}"\""     { /* Function definition */
	strcpy( act_function, yytext + 2 );
	act_function[yyleng-3] = 0;

	switch( command )
	  { case CMD_INIT: Add_all_function_to_hash( act_function );
			   break;
   
	    default:       break;
	   }


	BEGIN(in_funct);
	file_pos += yyleng;
	}

<f_call>"]\""{ID}"\""   { /* Function call */
	strcpy( act_f_call, yytext + 2 );
	act_f_call[yyleng-3] = 0;

	BEGIN(f_call_end1);
	file_pos += yyleng;
	}

<f_call_end1>("["{INT}"]")?$         { /* function with file-scope */

	BEGIN( in_funct );
	funct_filename[0] = 0;

	switch( command )
	  { case CMD_INIT:  /* Nothing to do when initializing */
			    break;

	    case CMD_CALLS: /* are we examining target function ?  */
	    case CMD_CALLS_ONCE:

		 /* If we are in interesting function, add this call to
		    return list.
		  */
		 if( interesting_function )
		    f_call_found( act_f_call );
		 break;
      
   	    case CMD_CALLED_BY: /* functions calling target funtion */
   	    case CMD_CALLED_BY_ONCE:

		 /* If we found a call for the interesting function, return 
                    actual function as caller
		  */
		 if( !strcmp( target_function, act_f_call)  )
		    f_call_found( act_function );
		 break;
      
            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_f_call, target_function) )
                   { strcpy(act_f_call, yytext + 1 );
                     act_f_call[strlen(act_f_call) - 1] = 0;
                     send_brs_ret( RET_POS, 0x0, client_pid, input_filename, 
                                             act_f_call, "", "" );
                     return;
                    }
                 break;

	    default:       break;
	   }

	file_pos += yyleng;
	}

<f_call_end1>"["{INT}"<"         { /* ext. function with global scope */

	f_line2 = atoi( yytext + 1 );

	BEGIN( f_call_end2 );
	file_pos += yyleng;
	}

<f_call_end2>({FILEN}|"<built-in>")(">]")   { /* ext. function with global scope */

	strcpy( funct_filename, yytext );
	funct_filename[yyleng-2] = 0;

	switch( command )
	  { case CMD_INIT:  /* Nothing to do when initializing */
			    break;

	    case CMD_CALLS: /* function called by target function  */
	    case CMD_CALLS_ONCE:

		 /* If we are in interesting function, add this call to
		    return list.
		  */
                 { long tmp_index;

                   /* if this function has been defined (or declared)
                      within a file with FLAG_IGNORE-status, ignore this
                      call
                    */      
                   tmp_index = search_token(funct_filename, NULL);
                   if( tmp_index != TOKEN_NOT_FOUND && 
                       (get_token_flag(tmp_index) & FLAG_IGNORE) )
                     break;
                  }
      
		   if( interesting_function )
		      f_call_found( act_f_call );
		 break;
   
   	    case CMD_CALLED_BY: /* functions calling target funtion */
   	    case CMD_CALLED_BY_ONCE:

		 /* If we found a call for the interesting function, return 
                    actual function as caller
		  */
		 if( !strcmp( target_function, act_f_call)  )
		    f_call_found( act_function );
		 break;

            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_f_call, target_function) )
                   { sprintf(act_f_call, "%ld", f_line2 );
                     send_brs_ret( RET_POS, 0x0, client_pid, funct_filename, 
                                                 act_f_call, "", "" );
                     return;
                    }
                 break;
   
	    default:       break;
	   }

	BEGIN( in_funct );
	file_pos += yyleng;
	}

<f_call2>"]\""{ID}"\""   { /* Function call outside function */
	strcpy( act_f_call, yytext + 2 );
	act_f_call[yyleng-3] = 0;

	BEGIN(f_call2_end1);
	file_pos += yyleng;
	}

<f_call2_end1>("["{INT}"]")?$         { /* function with file-scope */

	BEGIN( toplev );
	funct_filename[0] = 0;

	switch( command )
	  { case CMD_CALLS: /* are we examining target function ?  */
	    case CMD_CALLS_ONCE:

		 /* We are outside all functions, seems to be a reference
                    to a function pointer
		  */
                 if( !interesting_function )
                   f_call_found( act_f_call );
		 break;
      
            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_f_call, target_function) )
                   { strcpy(act_f_call, yytext + 1 );
                     act_f_call[strlen(act_f_call) - 1] = 0;
                     send_brs_ret( RET_POS, 0x0, client_pid, input_filename, 
                                             act_f_call, "", "" );
                     return;
                    }
                 break;
      
	    default:       break;
	   }

	file_pos += yyleng;
	}

<f_call2_end1>"["{INT}"<"         { /* ext. function with global scope */

	f_line2 = atoi( yytext + 1 );

	BEGIN( f_call2_end2 );
	file_pos += yyleng;
	}

<f_call2_end2>({FILEN}|"<built-in>")(">]")   { /* ext. function with global scope */

	strcpy( funct_filename, yytext );
	funct_filename[yyleng-2] = 0;

	switch( command )
	  { case CMD_CALLS: /* function called by target function  */
	    case CMD_CALLS_ONCE:

		 /* If we are in interesting function, add this call to
		    return list.
		  */
                 { long tmp_index;

                   /* if this function has been defined (or declared)
                      within a file with FLAG_IGNORE-status, ignore this
                      call
                    */      
                   tmp_index = search_token(funct_filename, NULL);
                   if( tmp_index != TOKEN_NOT_FOUND && 
                       (get_token_flag(tmp_index) & FLAG_IGNORE) )
                     break;
                  }
   
		 /* We are outside any function: return
		  */
                 if( !interesting_function )
                   f_call_found( act_f_call );
		 break;
   
            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_f_call, target_function) )
                   { sprintf(act_f_call, "%ld", f_line2 );
                     send_brs_ret( RET_POS, 0x0, client_pid, funct_filename, 
                                                 act_f_call, "", "" );
                     return;
                    }
                 break;
   
	    default:       break;
	   }

	BEGIN( toplev );
	file_pos += yyleng;
	}


<v_use>"]\""{ID}"\""   { /* Variable use */
	strcpy( act_v_use, yytext + 2 );
	act_v_use[yyleng-3] = 0;

	BEGIN(v_use_end1);
	file_pos += yyleng;
	}

<v_use_end1>"["{INT}"]"         { /* variable with file-scope */

	funct_filename[0] = 0;
        f_line2 = atoi( yytext + 1 );
        funct_filename[0] = 0;

	switch( command )
	  { case CMD_INIT:  /* Nothing to do when initializing */
			    break;

	    case CMD_USES: /* are we examining target function ?  */
	    case CMD_USES_ONCE:

		 /* If we are in interesting function, add this call to
		    return list.
		  */
		 if( interesting_function )
		    v_use_found( act_v_use );

		 break;
      
   	    case CMD_USED_BY: /* functions using target variable */
   	    case CMD_USED_BY_ONCE:

		 /* If we found the usage of the interesting variable, return 
                    actual function as "user"
		  */
		 if( !strcmp( target_variable, act_v_use)  )
		    f_call_found( act_function );
		 break;

            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_v_use, target_function) )
                   { strcpy(act_f_call, yytext + 1 );
                     act_f_call[strlen(act_f_call) - 1] = 0;
                     send_brs_ret( RET_POS, 0x0, client_pid, input_filename, 
                                             act_f_call, "", "" );
                     return;
                    }
                 break;
      
	    default:       break;
	   }

	BEGIN( cont_state );
	file_pos += yyleng;
	}

<v_use_end1>"["{INT}"<"         { /* ext. variable with global scope */

	f_line2 = atoi( yytext + 1 );

	BEGIN( v_use_end2 );
	file_pos += yyleng;
	}

<v_use_end2>{FILEN}">]"         { /* ext. variable with global scope */

	strcpy( funct_filename, yytext );
	funct_filename[yyleng-2] = 0;

	switch( command )
	  { case CMD_INIT:  /* Nothing to do when initializing */
			    break;

	    case CMD_USES: /* are we examining target function ?  */
	    case CMD_USES_ONCE:

		 /* If we are in interesting function, add this call to
		    return list.
		  */
                 { long tmp_index;

                   /* if this function has been defined (or declared)
                      within a file with FLAG_IGNORE-status, ignore this
                      call
                    */      
                   tmp_index = search_token(funct_filename, NULL);
                   if( tmp_index != TOKEN_NOT_FOUND && 
                       (get_token_flag(tmp_index) & FLAG_IGNORE) )
                     break;
                  }
   
		 /* If we are in interesting function, add this usage to
		    return list.
		  */
		 if( interesting_function )
		    v_use_found( act_v_use );
		 break;
   
   	    case CMD_USED_BY: /* functions using target variable */
   	    case CMD_USED_BY_ONCE:

		 /* If we found the usage of the interesting variable, return 
                    actual function as "user"
		  */
		 if( !strcmp( target_variable, act_v_use)  )
		    f_call_found( act_function );
		 break;

            case CMD_DEF_POS: /* Scoping and print definition position */
                 if( f_line == target_line &&
                     !strcmp(act_v_use, target_function) )
                   { sprintf(act_f_call, "%ld", f_line2 );
                     send_brs_ret( RET_POS, 0x0, client_pid, funct_filename, 
                                                 act_f_call, "", "" );
                     return;
                    }
                 break;
   
	    default:       break;
	   }


	BEGIN( cont_state );
	file_pos += yyleng;
	}

"[browser_control]"\n	{  /* browser-control-file */
        if( command != 0 )
          { fprintf(stderr, "This file (%s)is a control-file, not a brs-file !\n",
                             act_brs_file->file);
            exit(1);
           }
        BEGIN( control );
        }
        
.	{ /* initial rule , if file-positioning works, this rule is obsolete
             and just consist of an unput and a BEGIN(toplev)
           */
        if( command == 0 )
          { fprintf(stderr, "This file (%s) is NOT a browser-control-file ! !\n",
                             browser_file);
            exit(1);
           }
         unput( yytext[0] ); 
         BEGIN( toplev );
	}

<in_funct,d_enum,d_comp,d_const,f_call2,f_call2_end1>.   { /* Nothing to do else */
	BEGIN( in_funct );
   
	if( flex_debug2_flag )
           fprintf(stderr, "%s", yytext); 
	if( flex_debug2_flag == 2 )
           fprintf(stderr, " %s:%ld (%d)\n", act_brs_filename, lineno, yy_start ); 
   
	file_pos += yyleng;
	}

<toplev,f_call2_end2,control,control2,definition>.     { /* Nothing to do else */
	if( flex_debug2_flag )
           fprintf(stderr, "%s", yytext); 
	if( flex_debug2_flag == 2 )
           fprintf(stderr, " %s:%ld (%d)\n", act_brs_filename, lineno, yy_start ); 
	file_pos += yyleng;
	}

%%


/* This function is called by the scanner and 
   switches to the next brs-file
 */
int yywrap(void)
{
  /* close actual file if open
   */
  if( yyin )
   { fclose( yyin );
     yyin = NULL;
    }
   
  if( command == 0 )
    { /* flag: initial scanning of brs-command-file
       */
      return(1);
     }
   
  if( act_brs_file )
   { /* nornal yywrap function ! next file
      */
     act_brs_file = act_brs_file->next;
    }
  else
   { /* should not happen */
     return(1);
    }
  
  if( act_brs_file->file )
    { if( chdir( act_brs_file->dir ) )
        { /* error: cannot set new working directory
           */
          fprintf(stderr, "[Brs_server]: Cannot change to dir: \"%s\" !\n",
            act_brs_file->dir);
          fprintf(stderr, "              errno: %d (%s)\n", 
                                      errno, strerror(errno) );
          exit( 1 );
         }
      
      yyin = fopen( act_brs_file->file, "r" );
      strcpy( act_brs_filename, act_brs_file->file );
      if( !yyin )
        { /* error: cannot open file 
           */
          fprintf(stderr, "[Brs_server]: Cannot open brs-file: \"%s\" !\n",
            act_brs_file->file);
          fprintf(stderr, "              errno: %d (%s)\n", 
                                      errno, strerror(errno) );
          exit( 1 );
         }
      
      /* first time: set actual file-position as offset
       */
      if( act_brs_file->pos_offset == -1 )
        act_brs_file->pos_offset = file_pos;
      
      lineno = 1;
      
      return(0);  /* next input-file */
     }
  else
   { /* reset !
      */
     act_brs_file = NULL;
     
     return(1);
    }
 }


/* add browser-file read in from the browser-control-file with
   the working directory to list of browser-files
 */
void Add_to_brs_file_list( char * brs_file, char *brs_dir)
{ char        *ptr;
  t_brs_file  *info_ptr;
   
  if( !brs_file_list )
   { /* Create the list if there does not exist one
        (and set first entry to NULL)
      */
     brs_file_list = malloc( sizeof( t_brs_file ) );
     if( !brs_file_list )
       { fprintf(stderr, "[brs_server]: Not enough memory !\n" );
         exit(1);
        }
     act_brs_file = brs_file_list;
     act_brs_file->file = NULL;
    }
  
  /* first look if we have this browserfile allready in our
     brs-list (it is not possible to have two files with
     the same filename in one project !)
   */
  info_ptr = brs_file_list;
  while( info_ptr->file )
   { if( !strcmp( info_ptr->file, brs_file ) )
       { /* we found a allready stored filename
            check forworking directory
          */
         if( strcmp(info_ptr->dir, brs_dir) )
           { /* error: same filename, but in different directories 
              */
             fprintf(stderr, "[brs_server]: Double brs-file: \"%s\" (dir \"%s\") !\n",
                              brs_file, brs_dir );
             exit(1);
            }
         else
           { /* same filename and same dir: ignore !
              */
             return;
            }
        }
      
     if( !info_ptr->next )
       break;
     info_ptr = info_ptr->next;
    }
    
  act_brs_file->file = strdup( brs_file );
  act_brs_file->dir  = strdup( brs_dir  );
  act_brs_file->pos_offset = -1;  /* invalid offset */
  act_brs_file->next = malloc( sizeof( t_brs_file ) );
  if( !act_brs_file->next )
    { fprintf(stderr, "[brs_server]: Not enough memory !\n" );
      exit(1);
     }
  act_brs_file = act_brs_file->next;
  /* reset file-entry of new created entry
   */
  act_brs_file->file = NULL;
  act_brs_file->next = NULL;
}
 

/* This function sets the browser-position to the given
   new_pos-value. It sets the new working directory and
   opens the correct browserfile.
 */  
int set_browser_position(long new_pos)
{ t_brs_file  *brs_ptr;
  long         pos;
  
  if( new_pos < 0 )
    new_pos = 0;
   
  /* search in the list for the correct file
   */
  brs_ptr = get_brs_list_entry( new_pos );
  act_brs_file = brs_ptr;
   
  /* set new browser-file
   */  
  strcpy( act_brs_filename, brs_ptr->file );
  strcpy( act_brs_dir, brs_ptr->dir );
   
  if( chdir( act_brs_file->dir ) )
    { /* error: cannot set new working directory
       */
      fprintf(stderr, "[Brs_server]: Cannot change to dir: \"%s\" !\n",
                       act_brs_file->dir);
      fprintf(stderr, "              errno: %d (%s)\n", 
                                  errno, strerror(errno) );
      exit( 1 );
     }
   
  yyin = fopen( act_brs_file->file, "r" );
  if( !yyin )
    { /* error: cannot open file 
       */
      fprintf(stderr, "[Brs_server]: Cannot open brs-file: \"%s\" !\n",
        act_brs_file->file);
      fprintf(stderr, "              errno: %d (%s)\n", 
                                  errno, strerror(errno) );
      exit( 1 );
     }

  if( act_brs_file->pos_offset != -1 )
     pos = new_pos - act_brs_file->pos_offset;
  else 
     pos = new_pos;
   
  if( lseek( fileno(yyin), pos, SEEK_SET ) == -1 )
     { fprintf(stderr, "[brs_server]: Error fseeking yyin \n" );
       exit(1);
      }
   
  file_pos = new_pos;
  return(0);
}
    


/* Looks in the hash-table for the entry of a file. If this file is part
   of a module, it sets the global string act_module to the module-name
 */
void get_file_or_module ( t_hash_data  *data, long index)
{ t_hash_data      module_data;
  t_hash_add       *h_data;

  get_token ( data, index);

  h_data = (t_hash_add *)data->data;
   
  /* If this hash-add-entry has a valid ->file entry,
     this file belongs to a module
   */
  if( h_data && h_data->file != TOKEN_NOT_FOUND )
    { /* File belongs to a module */
      get_token ( &module_data, h_data->line);
      strcpy( act_module, module_data.ident );
     }
  else
    { /* No Module for this file defined
       */
      strcpy( act_module, "" );
     }
 }


void Add_to_Hash_Index_Array(long index)
{

  hash_array[hash_array_cnt] = index;
  hash_array_cnt++;

  if( hash_array_cnt == hash_array_size)
    { hash_array_size += HASH_ARRAY_INIT_SIZE;
      hash_array = realloc( hash_array,
		       sizeof( long ) * hash_array_size );

      if( !hash_array )
	{ fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		   sizeof( long ) * hash_array_size );
	  exit(1);
	 }
     }
 }

void Add_to_Hash_changed_Index_Array(long index)
{

  hash_changed_array[hash_changed_array_cnt] = index;
  hash_changed_array_cnt++;

  if( hash_changed_array_cnt == hash_changed_array_size)
    { hash_changed_array_size += HASH_ARRAY_INIT_SIZE;
      hash_changed_array = realloc( hash_changed_array,
		       sizeof( long ) * hash_changed_array_size );

      if( !hash_changed_array )
	{ fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		   sizeof( long ) * hash_changed_array_size );
	  exit(1);
	 }
     }
 }


/* If a function-definition has been found, it adds it's identifier to the
   hash-table. So for every function-call, there can be printed the position
   in the source-file
 */
void Add_all_function_to_hash( char *function )
{ t_hash_add  *h_data, *h2;
  t_hash_data  h_entry;
  long         index;
  short        flag = 0;
  t_type       type = TYPE_FUNCTION_DEF;

  if( function[0] == 0 )
     return;
   
  if( static_function )
    type |= TYPE_STATIC;

  h_data  = (t_hash_add *)malloc( sizeof(t_hash_add) );
  h_data->file = search_token( input_filename, &h_entry);
  h_data->line = f_line;
  h_data->f_pos = start_file_pos;
  h_data->next = NULL;
  h_data->type = type;
   
  /* to store the line-number of the end of this function
     detected by [fe]
   */
  data_of_act_function = h_data;

  if( h_data->file == TOKEN_NOT_FOUND )
    { /* Filename not found -> Add to hashtable
       */
      h_data->file = Add_ident_to_hash( input_filename,
					0, NULL, TYPE_SOURCE_FILE);
     }
   
  /* if the file is marked FLAG_IGNORE, this symbol will also marked 
     FLAG_IGNORE
   */
  if( get_token_flag(h_data->file) & FLAG_IGNORE )
    flag |= FLAG_IGNORE;

  /* look, if symbol is allready in Hash-Table
   */
  index = search_token (function, &h_entry);

  if( index == TOKEN_NOT_FOUND )
    { /* function-name not found -> add to hash-table
       */

      index = Add_ident_to_hash( function, flag, h_data, type);
      get_token ( &h_entry, index );
     }

  else
    { /* function-name has allready been used (probably in a function call)
	 or a previous definition of a static function:
	 just updates the existing hash-record
       */
      if( !(h_entry.data) )
	{ /* function has probably been used, so it is allready in the
	     hashtable, but there's no definition position. update
	     definition position
	   */
	  h_entry.data  = h_data;
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );
	 }
      else
	{
	 h2 = (t_hash_add *)h_entry.data;
	 if( h2->file != h_data->file ||
	     h2->line != h_data->line    )
	    { /* redefinition ! might be a static function which is defined
		 more than once
	       */

	      while( h2->next )
		{ /* get to the end of the list
		   */
		  h2 = h2->next;
		 }

	      h2->next = h_data;
	     }
	  else
	    { /* h_data is allready in this list -> free h_data
	       */
	      free( h_data );
	     }

	  /* Do update the type of the ident
	   */
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );

	 }
     }

  /* Add function-symbol to hash-index-array
   */
  if( !(h_entry.flags & FLAG_IN_HASH) )
    { Add_to_Hash_Index_Array( index);
      h_entry.flags  |= FLAG_IN_HASH;
      update_token( &h_entry, index );
     }

 }

/* If a variable-definition has been found, it adds it's identifier to the
   hash-table. So for every function-call, there can be printed the position
   in the source-file
 */
void Add_all_variables_to_hash( char *variable )
{ t_hash_add  *h_data, *h2;
  t_hash_data  h_entry;
  long         index;
  short        flag = 0;
  t_type       type = TYPE_GLOBAL_VAR_DEF;

  if( variable[0] == 0 )
     return;
   
  if( static_variable )
    type |= TYPE_STATIC;

  h_data  = (t_hash_add *)malloc( sizeof(t_hash_add) );
  h_data->file = search_token( input_filename, &h_entry);
  h_data->line = f_line;
  h_data->f_pos = start_file_pos;
  h_data->next = NULL;
  h_data->type = type;


  if( h_data->file == TOKEN_NOT_FOUND )
    { /* Filename not found -> Add to hashtable
       */
      h_data->file = Add_ident_to_hash( input_filename,
					0, NULL, TYPE_SOURCE_FILE);
     }

  /* if the file is marked FLAG_IGNORE, this symbol will also marked 
     FLAG_IGNORE
   */
  if( get_token_flag(h_data->file) & FLAG_IGNORE )
    flag |= FLAG_IGNORE;
   
  /* look, if symbol is allready in Hash-Table
   */
  index = search_token (variable, &h_entry);

  if( index == TOKEN_NOT_FOUND )
    { /* function-name not found -> add to hash-table
       */

      index = Add_ident_to_hash( variable, flag, h_data, type);
      get_token ( &h_entry, index );
     }

  else
    { /* variable-name has allready been used (probably a
	 a previous definition of a static variable:
	 just updates the existing hash-record
       */
      if( !(h_entry.data) )
	{ /* there's no definition position. update
	     definition position
	   */
	  h_entry.data  = h_data;
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );
	 }
      else
	{
	 h2 = (t_hash_add *)h_entry.data;
	 if( h2->file != h_data->file ||
	     h2->line != h_data->line    )
	    { /* redefinition ! might be a static variable which is defined
		 more than once
	       */

	      while( h2->next )
		{ /* get to the end of the list
		   */
		  h2 = h2->next;
		 }

	      h2->next = h_data;
	     }
	  else
	    { /* h_data is allready in this list -> free h_data
	       */
	      free( h_data );
	     }

	  /* Do update the type of the ident
	   */
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );

	 }
     }

  /* Add variable symbol to hash-index-array
   */
  if( !(h_entry.flags & FLAG_IN_HASH) )
    { Add_to_Hash_Index_Array( index);
      h_entry.flags  |= FLAG_IN_HASH;
      update_token( &h_entry, index );
     }

 }

   
/* If a function-definition has been found, it adds it's identifier to the
   hash-table. So for every function-call, there can be printed the position
   in the source-file
 */
void Add_all_items_to_hash( char *item, short static_flag, t_type type )
{ t_hash_add  *h_data, *h2;
  t_hash_data  h_entry;
  long         index;
  short        flag = 0;

  if( item[0] == 0 )
     return;
   
  if( static_flag )
    type |= TYPE_STATIC;

  h_data  = (t_hash_add *)malloc( sizeof(t_hash_add) );
  h_data->file = search_token( input_filename, &h_entry);
  h_data->line = f_line;
  h_data->f_pos = start_file_pos;
  h_data->next = NULL;
  h_data->type = type;

  if( h_data->file == TOKEN_NOT_FOUND )
    { /* Filename not found -> Add to hashtable
       */
      h_data->file = Add_ident_to_hash( input_filename,
					0, NULL, TYPE_SOURCE_FILE);
     }
   
  /* if the file is marked FLAG_IGNORE, this symbol will also marked 
     FLAG_IGNORE
   */
  if( get_token_flag(h_data->file) & FLAG_IGNORE )
    flag |= FLAG_IGNORE;

  /* look, if symbol is allready in Hash-Table
   */
  index = search_token ( item, &h_entry);

  if( index == TOKEN_NOT_FOUND )
    { /* item-name not found -> add to hash-table
       */

      index = Add_ident_to_hash( item, flag, h_data, type);
      get_token ( &h_entry, index );
     }

  else
    { /* item-name has allready been used or a previous definition 
         of an item with same identifier:
	 just updates the existing hash-record
       */
      if( !(h_entry.data) )
	{ /* item has probably been used, so it is allready in the
	     hashtable, but there's no definition position. create
             entry for definition position
	   */
	  h_entry.data  = h_data;
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );
	 }
      else
	{ /* there is a hash_add-entry, look if there is a matching one.
             If not, add a new entry to the chain, otherwise ignore
           */
         
	 h2 = (t_hash_add *)h_entry.data;
	 if( h2->file != h_data->file ||
	     h2->line != h_data->line    )
	    { /* redefinition ! might be item with same identifier or
                 item, which has to be defined for every source-file
                 (for eg. a type or struct or a static function)
	       */

	      while( h2->next )
		{ /* walk thru the list and test for identical
                     position. If identical position found, 
                     ignore this position
		   */
		  h2 = h2->next;
               
                  if( h2->file == h_data->file &&
            	      h2->line == h_data->line    )
                    { /* identical entry found: ignore !
                       */
                      h2 = NULL;
                      break;
                     }
		 }
              if( h2 )
	        { /* if we are at the end of the list and did not find
                     an identical entry: add the new entry to this list
                   */
                  h2->next = h_data;
               
                  /* DO NOT change the flag-entry in h_entry ! This may
                     cause an ident, which should be ignored to be printed
                     but it is better than ignoring an ident, which should
                     have been printed ! THIS IS A BUG WHICH HAS TO BE 
                     REMOVED SOON !
                   */
                  flag = h_entry.flags;
                 }
	     }
	  else
	    { /* h_data is allready in this list -> free h_data
	       */
	      free( h_data );
	     }

	  /* Do update the type of the ident
	   */
	  h_entry.flags |= flag;
	  h_entry.type  |= type;
	  update_token( &h_entry, index );

	 }
     }

  /* Add symbol to hash-index-array
   */
  if( !(h_entry.flags & FLAG_IN_HASH) )
    { Add_to_Hash_Index_Array( index);
      h_entry.flags  |= FLAG_IN_HASH;
      update_token( &h_entry, index );
     }

 }
   
   
   
void Store_Source_File( short main_flag )
{ t_hash_add  *h_data, *h2;
  t_hash_data  h_entry;
  long         index;
  short        flag = 0;
  t_type       type = TYPE_SOURCE_FILE;

  if( main_flag )
    type |= TYPE_MAIN_FILE;


  /* look, if symbol is allready in Hash-Table
   */
  index = search_token (input_filename, &h_entry);


  if( index == TOKEN_NOT_FOUND )
    { /* token not found -> add to hash-table 
         main_flag-set:      it is a mainfile
         main_flag not set:  it is an include file. We DO store the 
                             position to have a "handle" for the working
                             directory of this include-file
       */
       h_data  = (t_hash_add *)malloc( sizeof(t_hash_add) );
       h_data->file  = TOKEN_NOT_FOUND;
       h_data->line  = 0;
       h_data->f_pos = start_file_pos;
       h_data->next  = NULL;
       h_data->type  = type;

      index = Add_ident_to_hash( input_filename, flag, h_data, type);
      get_token ( &h_entry, index );
     }

  else
    { /* token has been used: if just a module-entry, update 
         hash-add-entry only, if f_pos is valid, create a new hash-add-
         entry
       */
      h_entry.flags       |= flag;
      h_entry.type        |= type;
      if( (h2 = h_entry.data) && h2->f_pos == ILL_F_POS )
        { /* none-valid f_pos-entry: this entry belongs only
             to a module entry
           */
          h2->f_pos  = start_file_pos;
         }
      else
        { /* valid entry ! This file has allready been inserted to
             hash-table: ignore !
           */
          ;
         }
      
      update_token( &h_entry, index );
     }

  /* Add variable symbol to hash-index-array
   */
  if( !(h_entry.flags & FLAG_IN_HASH) )
    { Add_to_Hash_Index_Array( index);
      h_entry.flags  |= FLAG_IN_HASH;
      update_token( &h_entry, index );
     }

 }



/* This function add's a symbol to the symbol-table. If the flag for
   printing is set, the index is added to the hash-index-array which
   causes the symbol to be printed after processing

   return-value is the index in the hash-table
 */
long Add_ident_to_hash(char *ident, short flag, void *add_data, t_type type )
{ long index ;
  t_hash_data  h_entry;

  if( ident[0] == 0 )
    return; /* nothing to do with an empty item */
   
  strncpy( h_entry.ident, ident, SZE_HASH_IDENT - 1 );
  h_entry.ident[SZE_HASH_IDENT - 1] = 0;
  h_entry.type  = type;
  h_entry.flags = flag;
  h_entry.data  = add_data;
  index = insert_token( &h_entry );

  return( index );
}


/* This function is called if a function-call is found within 
   target-function. It sends the position to frontend-modules
 */
void f_call_found( char *function )
{ t_hash_data  h_entry_1, h_entry_2;
  t_hash_add   *h_add_1, *h_add_global = NULL;
  long         index,
	       file_index;
  short        flag = 0;
  char         arg1[MAX_IDENT],    /* return argument for identifier */
	       arg2[MAX_FILENAME], /* return argument for filename   */
	       arg3[MAX_IDENT],   /* return argument for linenumber */
	       arg4[MAX_FILENAME]; /* return argument for modulename */


  arg1[0] = 0;
  arg2[0] = 0;
  arg3[0] = 0;
  arg4[0] = 0;


  strcpy( arg1, function );

  /* look, if symbol is allready in Hash-Table
   */
  index = search_token ( function, &h_entry_1);
   
  if( (index != TOKEN_NOT_FOUND) && (h_entry_1.flags & FLAG_IGNORE) )
    { /* ignore this symbol
       */
      return;
     }
      
  if( index == TOKEN_NOT_FOUND )
    { /* function not defined in project
       */

      flag    = 0x0;
      strcpy(arg2, input_filename );
      sprintf( arg3, "%ld", f_line);
      arg4[0] = 0;
      
      /* Check if filename is in Hash-Table and is marked FLAG_IGNORE.
         If so, ignore this symbol
       */
      index = search_token( input_filename, NULL );
      if( index != TOKEN_NOT_FOUND && (get_token_flag(index) & FLAG_IGNORE) )
        { /* The file is marked IGNORE, so ignore also this symbol
           */
          return;
         }
      
      send_brs_ret( RET_FUNCT, flag, client_pid, arg1, arg2, arg3, arg4 );
      
      if( symbols_only_once )
        { /* add to hash because we want functions also only.
             We have to store the FLAG_PRINTED_FLAG
           */
          index = Add_ident_to_hash( function, FLAG_PRINTED, NULL, TYPE_UNKNOWN);
          Add_to_Hash_changed_Index_Array( index);
          clear_invalid_hash_table_entrys = 1;
         }
      
      return;
     }
   
  if( symbols_only_once )
    { if( h_entry_1.flags & FLAG_PRINTED )
        { /* This symbol has allready been printed 
           */
          return;
         }
      
      h_entry_1.flags |= FLAG_PRINTED;
      set_token_flag( index, h_entry_1.flags );
      Add_to_Hash_changed_Index_Array( index);
      hash_table_changed = 1;
     }

  /* Function is defined in project -> return known position
   */

  file_index = search_token( input_filename, &h_entry_2);

  h_add_1 = (t_hash_add *)h_entry_1.data;

  while( 1 )
   { if( h_add_1->type == (TYPE_FUNCTION_DEF | TYPE_STATIC) )
       { /* static function: check for matching files
          */
	 if( file_index == h_add_1->file )
	    break;
        }

     if( h_add_1->type == TYPE_FUNCTION_DEF )
       { /* global function: store position, but wait, maybe
            a static function overrides this global function
          */
	 h_add_global = h_add_1;
        }
      
     if( h_add_1-> next )
	 h_add_1 = h_add_1-> next;
     else
       { /* no matching file ?
	  */
	    
         if( h_add_global )
           { /* we found a global symbol and no static function
                did override this function
              */
             h_add_1 = h_add_global;
             break;
            }
	 else
	 /* Matching type or file not found: ignore
	  */
           return;
	}
    }

  /************************
     Get filename where function is defined ?
     for f_call, it might be more intersting, at which line
     this function is called 
      
     get_file_or_module ( &h_entry_2, h_add_1->file);

     strcpy( arg2, h_entry_2.ident );

     sprintf( arg3, "%ld", h_add_1->line );

     if( act_module[0] )
       { strcpy( arg4, act_module );
	 flag |= 0x02;
	}

   *****************/
  if( h_add_1->type & TYPE_STATIC )
    flag |= 0x04;

  strcpy(arg2, input_filename );
  sprintf( arg3, "%ld", f_line);
  arg4[0] = 0;
  send_brs_ret( RET_FUNCT, flag, client_pid, arg1, arg2, arg3, arg4 );


 }


/* This function is called if a variable usage is found within
   target-function. It sends the position to frontend-modules
 */
void v_use_found( char * variable )
{ t_hash_data  h_entry_1, h_entry_2;
  t_hash_add   *h_add_1, *h_add_global = NULL;
  t_type       type;
  long         index,
	       file_index;
  short        flag = 0;
  char         arg1[MAX_IDENT],    /* return argument for identifier */
	       arg2[MAX_FILENAME], /* return argument for filename   */
	       arg3[MAX_IDENT],   /* return argument for linenumber */
	       arg4[MAX_FILENAME]; /* return argument for modulename */


  arg1[0] = 0;
  arg2[0] = 0;
  arg3[0] = 0;
  arg4[0] = 0;


  strcpy( arg1, variable);

  /* look, if symbol is allready in Hash-Table
   */
  index = search_token ( variable, &h_entry_1);
  if( index == TOKEN_NOT_FOUND || !(h_entry_1.data) )
    { /* variable not defined as global: must be local: ignore
       */

      return;
     }
   
  if( h_entry_1.flags & FLAG_IGNORE )
    { /* ignore this symbol
       */
      return;
     }
   
  if( symbols_only_once )
    { if( h_entry_1.flags & FLAG_PRINTED )
        { /* This symbol has allready been printed 
           */
          return;
         }
      
      h_entry_1.flags |= FLAG_PRINTED;
      set_token_flag( index, h_entry_1.flags );
      Add_to_Hash_changed_Index_Array( index);
      hash_table_changed = 1;
     }
   
  /* Variable is defined in project -> return known position
   */

  file_index = search_token( input_filename, &h_entry_2);

  h_add_1 = (t_hash_add *)h_entry_1.data;

  while( 1 )
   { if( h_add_1->type == (TYPE_GLOBAL_VAR_DEF | TYPE_STATIC) )
       { /* static variable: check for matching files
          */
	 if( file_index == h_add_1->file )
	    break;
        }

     if( h_add_1->type == TYPE_GLOBAL_VAR_DEF )
       { /* global variable: store position, but wait, maybe
            a static variable overrides this global one
          */
	 h_add_global = h_add_1;
        }
      
     if( h_add_1-> next )
	 h_add_1 = h_add_1-> next;
     else
       { /* no matching file ?
	  */
	    
         if( h_add_global )
           { /* we found a global symbol and no static variable
                did override the scope of this variable
              */
             h_add_1 = h_add_global;
             break;
            }
	 else
	 /* Matching type or file not found: ignore
	  */
           return;
	}
    }
      
  if( h_add_1->type & TYPE_STATIC )
    flag |= 0x04;

  strcpy(arg2, input_filename );
  sprintf( arg3, "%ld", f_line);
  arg4[0] = 0;

  send_brs_ret( RET_VAR, flag, client_pid, arg1, arg2, arg3, arg4 );

 }

/* Send the item with the given index to the client application.
   If there are more than one matching item with this hash-index,
   this function returns all of them
 */
void dump_hash_item( long  index, t_type out_type )
{ t_hash_data  h_entry_1, h_entry_2;
  t_hash_add   *h_add_1;
  long	       file_index;
  short        flag = 0,
               found_flag = 0;
  t_type       type;
  char         arg1[MAX_IDENT],    /* return argument for identifier */
	       arg2[MAX_FILENAME], /* return argument for filename   */
	       arg3[MAX_IDENT],   /* return argument for linenumber */
	       arg4[MAX_FILENAME]; /* return argument for modulename */


  arg1[0] = 0;
  arg2[0] = 0;
  arg3[0] = 0;
  arg4[0] = 0;


  get_token ( &h_entry_1, index);
   
  if( h_entry_1.flags & FLAG_IGNORE )
    { /* ignore this symbol
       */
      return;
     }
   
  strcpy( arg1, h_entry_1.ident );

  if( !(h_entry_1.data) )
    { /* function not defined in project
         This might not be possible at all !!!
       */

      flag    = 0x1;
      arg2[0] = 0;
      arg3[0] = 0;
      arg4[0] = 0;
      
      if( h_entry_1.type & TYPE_STATIC )
	flag |= 0x04;

      if( h_entry_1.type & (TYPE_GLOBAL_VAR_DEF | TYPE_LOCAL_VAR_DEF) )
          send_brs_ret( RET_VAR, flag, client_pid, arg1, arg2, arg3, arg4 );

      else if( h_entry_1.type & TYPE_FUNCTION_DEF )
          send_brs_ret( RET_FUNCT, flag, client_pid, arg1, arg2, arg3, arg4 );

     }
  else
    { /* Item is defined in project -> return known position
       */

      if( funct_filename[0] )
	{ /* filename given: look for a static item in given file
	   */
	  file_index = search_token( funct_filename, &h_entry_2);
	 }
      else
	{ /* item is external: look for global
	   */
	  file_index = TOKEN_NOT_FOUND; /* don't care */
	 }

      h_add_1 = (t_hash_add *)h_entry_1.data;

      while( 1 )
       { flag = 0;
         if( ( h_add_1->type == out_type ) &&
	     (file_index == TOKEN_NOT_FOUND || h_add_1->file == file_index) )
	   { /* We found an item with matching type and matching file (if given)
                Get filename where item is defined
              */
             get_file_or_module ( &h_entry_2, h_add_1->file);

             strcpy( arg2, h_entry_2.ident );

             sprintf( arg3, "%ld", h_add_1->line );

             if( act_module[0] )
	       { strcpy( arg4, act_module );
	         flag |= 0x02; /* module given */
	        }
             else
                strcpy( arg4, "" );

             if( h_add_1->type & TYPE_STATIC )
	       flag |= 0x04;

             if( h_add_1->type & (TYPE_GLOBAL_VAR_DEF | TYPE_LOCAL_VAR_DEF) )
               { found_flag = 1;
                 send_brs_ret( RET_VAR, flag, client_pid, arg1, arg2, arg3, arg4 );
                }

             else if( h_add_1->type & TYPE_FUNCTION_DEF )
               { found_flag = 1;
                 send_brs_ret( RET_FUNCT, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
             else if( h_add_1->type & TYPE_TYPEDEF ) 
               { found_flag = 1;
                 send_brs_ret( RET_TYPEDEF, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
             else if( h_add_1->type & TYPE_STRUCT ) 
               { found_flag = 1;
                 send_brs_ret( RET_STRUCT, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
             else if( h_add_1->type & TYPE_ENUM ) 
               { found_flag = 1;
                 send_brs_ret( RET_ENUM, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
             else if( h_add_1->type & TYPE_UNION ) 
               { found_flag = 1;
                 send_brs_ret( RET_UNION, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
             else if( h_add_1->type & TYPE_MACRO ) 
               { found_flag = 1;
                 send_brs_ret( RET_MACRO, flag, client_pid, arg1, arg2, arg3, arg4 );
                }
            
            }

	 if( h_add_1-> next )
	     h_add_1 = h_add_1-> next;
	 else
	   { /* Matching type or file not found: ignore
	      */
	     break;
	    }
	}
     }
 }



/* This function sends the definition position of the interesting function
   (or a variable) to the frontend-modules
 */
void return_f_pos( char *function, char *file, t_type type )
{ t_hash_add   h_erg1, h_erg2;
  t_hash_data  h_entry;
  long         index;
  short        flag = 0;
  char         arg1[MAX_IDENT],    /* return argument for identifier */
	       arg2[MAX_FILENAME], /* return argument for filename   */
	       arg3[MAX_IDENT],   /* return argument for linenumber */
	       arg4[MAX_FILENAME]; /* return argument for modulename */


  arg1[0] = 0;
  arg2[0] = 0;
  arg3[0] = 0;
  arg4[0] = 0;


  /* look for the function in this specified file (strict-flag set)
   */
  if( get_item_pos( function, file, type, &h_erg1, &h_erg2, 1) )
    { /* Error: allready reportet to front-end
       */
      return;
     }

  get_file_or_module ( &h_entry, h_erg1.file);

  if( file[0] )
    strcpy( arg1, file );
  else
    { /* get filename from hash_add-entry
       */
      strcpy( arg1, h_entry.ident );
     }

  sprintf( arg2, "%ld", h_erg1.line );

  if( type == TYPE_FUNCTION_DEF )
    { sprintf( arg3, "%ld", h_erg1.end_line );

      if( act_module[0] )
        { strcpy( arg4, act_module );
          flag |= 0x01; /* module given */
         } 

      send_brs_ret( RET_FPOS, flag, client_pid, arg1, arg2, arg3, arg4 );
     }
  else
    { if( act_module[0] )
        { strcpy( arg3, act_module );
          flag |= 0x01; /* module given */
         } 

      send_brs_ret( RET_POS, flag, client_pid, arg1, arg2, arg3, arg4 );
     }

 }


/* Interesting function has completly scanned. So send the END-Flag to
   frontend-modules
 */
short end_v_use(void)
{
  send_brs_ret( RET_END, 0, client_pid, "", "","","" );

  /* quit yylex
   */
  return( 1 );
 }


/* Interesting function has completly scanned. So send the END-Flag to
   frontend-modules
 */
short end_f_call(void)
{
  send_brs_ret( RET_END, 0, client_pid, "", "","","" );

  /* quit yylex
   */
  return( 1 );
 }

/* file of static function has completly scanned. So send the END-Flag to
   frontend-modules
 */
short end_f_called(void)
{
  send_brs_ret( RET_END, 0, client_pid, "", "","","" );

  /* quit yylex
   */
  return( 1 );
 }

/* Interesting function has completly scanned. So send the END-Flag to
   frontend-modules
 */
short end_reference(void)
{
  send_brs_ret( RET_END, 0, client_pid, "", "","","" );

  /* quit yylex
   */
  return( 1 );
 }


/* Clear all found function calls ! Functions, which should not be printed,
   (they were read in by read_exclude_symbols) still stay in the hash-table
 */
void reset_hashtable(void)
{ long          i;
  t_hash_data   data;
  t_hash_add    *h_add1, *h_next;

  for( i = 0; i < hash_array_cnt; i++ )
    { get_token ( &data, hash_array[i]);
      
      if( data.data )
        { h_add1 = data.data;
          do
           { h_next = h_add1->next;
             free( h_add1 );
             h_add1 = h_next;
            } while( h_next );
         } 

      clear_token (hash_array[i]);
     }

  hash_array_cnt = 0;
 }

/* remove the brs-file-list ! First step: free all filenames,
   and build a second linklink-direction
 */
void reset_brs_file_list(void)
{ long          i;
  t_brs_file    *info_ptr,
                *old_ptr;
    
  info_ptr = brs_file_list;
   
  while( info_ptr->file )
   { 
     free( info_ptr->file ); /* free filename */
     free( info_ptr->dir ); /* free filename */
      
     if( !info_ptr->next )
       break;

     /* get the sucessor-entry and then free this entry
      */
     old_ptr = info_ptr;
      
     if( info_ptr->next )
       info_ptr = info_ptr->next;
     else
      { free( old_ptr );
        break;
       }
      
     free( old_ptr );
    }
   
  if( info_ptr->next )
    free( info_ptr->next );
   
  brs_file_list = NULL;

 }

/* returns the brs_list-entry of the browser-file with the
   given position
 */
t_brs_file * get_brs_list_entry(long pos)
{ t_brs_file  *brs_ptr,
              *last_ptr;
  
  if( pos < 0 )
    pos = 0;
  
  if( pos == 0 )
    return( brs_file_list );
   
  /* search in the list for the correct file
   */
  brs_ptr = brs_file_list;
  last_ptr = NULL;
   
  while( pos && brs_ptr->file )
   { if( brs_ptr->pos_offset > pos )
       { /* offset is higher than the searched: we have to take
            the last one
          */
         brs_ptr = last_ptr;
         last_ptr = NULL; /* flag, that we found the target */
         break;
        }
     else
       { /* test the next one, but save this file, because it might
            be the correct one
          */
         last_ptr = brs_ptr;
         
         if( brs_ptr->next )
           { if( !((brs_ptr->next)->file) )
               { /* next entry is not valid ! We are at the end of
                    the list 
                  */
                 last_ptr = NULL;
                 break;
                }
             else
               { /* next entry */
                  last_ptr = brs_ptr;
                  brs_ptr = brs_ptr->next;
                }
            }
         else /* should never happen: ->next == NULL */ 
           break;
        }
    } 
   
  if( last_ptr )
   { /* error: tried to set position after browser-file
      */
      fprintf(stderr, "[brs_server]: Illegal browser-positioning %ld !\n",
                          pos );
      exit(1);
     }
   
  return( brs_ptr );
 }
   

/* opens a file with exclude-symbols
 */
void read_exclude_symbols(char *file)
{ long         index;
  t_hash_data  data;
  FILE        *fp;


  /* save filename for later use
   */
  if( !exclude_file[0] )
    strcpy( exclude_file, file );

  fp = fopen( file, "r" );
  if( !fp )
    { fprintf(stderr, "Cannot open Exclude-File %s !\nAborting !", file );
      exit(1);
     }

  while( !feof(fp) )
    { if( fgets( act_function, MAX_IDENT, fp ) )
	{ if( act_function[strlen(act_function)-1] == '\n' )
	     act_function[strlen(act_function)-1] = 0;

	  Add_ident_to_hash( act_function, FLAG_IGNORE, NULL, 0 );
	 }
     }

  fclose(fp);
 }

/* opens a file with module-names

   Modules:
     - 1 Entry for the module-ident
     - 1 Entry for every file (TYPE_MODULE_FILE)

     t_hash_add:    Module:    file    Index of first file in this module
                               f_pos   ILL_F_POS  as flag
                               next    unused

                    M-Files:   file    Index of next file in module-chain
                                       (points to itself: end of chain)
                               line    Index of module-entry
                               f_pos   ILL_F_POS  as flag
                               next    unused
 */
void read_modules(char *file)
{ long         index, module_index;
  t_hash_data  data;
  FILE        *fp;
  char         module_name[MAX_IDENT];
  char         file_name[MAX_FILENAME];
  t_hash_add  *h_data, *h_data2;


  /* save filename for later use
   */
  if( !module_file[0] )
    strcpy( module_file, file );

  fp = fopen( file, "r" );
  if( !fp )
    { fprintf(stderr, "Cannot open Module-File %s !\nAborting !", file );
      exit(1);
     }

  while( !feof(fp) )
    { /* Get the name of the module
       */
      if( !fgets( module_name, MAX_IDENT, fp ) )
	 break;

      /* Allocate memory for at least one Filename
       */
      h_data  = (t_hash_add *)malloc( sizeof(t_hash_add) );

      h_data->line  = 0;
      h_data->f_pos = ILL_F_POS; /* unknown position */
      h_data->next  = NULL;
      h_data->type  = TYPE_MODULE;

      if( module_name[strlen(module_name)-1] == '\n' )
	 module_name[strlen(module_name)-1] = 0;

      index = search_token( module_name, &data);

      if( index != TOKEN_NOT_FOUND )
	{ /* Error: A Module-Name was defined twice */
	  fprintf(stderr, "Modulname %s defined twice in %s\n",
			  module_name, file );
	  exit(1);
	 }

      module_index = Add_ident_to_hash( module_name, 0, h_data, TYPE_MODULE);

      while(1)
       { if( feof( fp ) || !fgets( file_name, MAX_FILENAME, fp ) )
	   break;

	 if( file_name[strlen(file_name)-1] == '\n' )
	    file_name[strlen(file_name)-1] = 0;

	 if( file_name[0] == 0 )
	   break; /* empty line: break */

	 index = search_token( file_name, &data);

	 if( index != TOKEN_NOT_FOUND )
	   { /* Error: A File has been assigned to two modules */
	     if( data.type & TYPE_MODULE_FILE )
		fprintf(stderr, "File %s has defined twice in %s\n",
			     file_name, module_name );
	     else
		fprintf(stderr, "Strange: File %s is allready defined !\n",
			     file_name);
	     exit(1);
	    }

	 h_data2  = (t_hash_add *)malloc( sizeof(t_hash_add) );
         
         h_data2->f_pos = ILL_F_POS;
         h_data2->next  = NULL;
         h_data2->type  = TYPE_MODULE_FILE;

	 index = Add_ident_to_hash( file_name, 0, h_data2, TYPE_MODULE_FILE);


	 /* build chain: store in the last entry (or the module-
	    entry) the index of the next entry */
	 h_data->file = index;

	 h_data2->file = index;  /* Mark End: Point to itself */

	 /* Store Index of Module in line-Entry
	  */
	 h_data2->line = module_index;

	 /* Build chain of Module-Files for this module */
	 h_data = h_data2;
	}

     }

  fclose(fp);
 }


/* This function gets function- and file-ident and returns information
   stored in hashtable (position, line, ...)

   If symbol could not be found, it returns 1 so we can wait for the
   next browser-command

   INPUT:

   arg1 ... item-name
   arg2 ... file-name where item defined (or where it has been found)

   type ... type of the item
   strict_flag ... if set to 1, the definition file and arg2 (if given) have to
                   match, otherwise an error will be returned
                   if set to 0, the definition file and the file in arg2 may
                   differ. However, if the item is defined more than once, the
                   definition position in arg2 will returned if possible, otherwise
                   the first occurence in every other file !
                   (static/global scope-precedence)

   OUTPUT:
   h_erg1  ... hash_add_data of the right item
   h_erg2  ... hash_add_data of the right file
 */
short get_item_pos(char *arg1, char *arg2, t_type type,
			 t_hash_add *h_erg1, t_hash_add *h_erg2, short strict_flag)
{ long         index_item, 
               index_file;
  t_hash_data  h_entry_item, 
               h_entry_file;
  t_hash_add   *h_add_item = NULL,
               *h_add_global = NULL,
	       *h_add_file = NULL;

  index_item = search_token( arg1, &h_entry_item );
  if( index_item == TOKEN_NOT_FOUND )
    { /* function not found
       */
      send_brs_ret( RET_ERROR, 0, client_pid, "Symbol unknown","","","" );

      return( 1 );
     }

  if( h_entry_item.flags & FLAG_IGNORE )
    { /* function is in exclude-list
       */
      send_brs_ret( RET_ERROR, 0, client_pid, "Symbol ignored","","","" );

      return( 1 );
     }

  if( !h_entry_item.data || !(h_entry_item.type & type))
    { /* no definition position specified -> this */
      /* function is not specified in target      */
      send_brs_ret( RET_ERROR, 0, client_pid, "No Function","","","" );

      return( 1 );
     }

  if( arg2[0] )
    { /* filename is specified: look for this file
       */
      index_file = search_token( arg2, &h_entry_file);
      if( index_file == TOKEN_NOT_FOUND )
	{ /* function not found
	   */
	  send_brs_ret( RET_ERROR, 0, client_pid, "File not found", "","","" );

	  return( 1 );
	 }

      /* If filename is given, assume static function - obsolete
       type |= TYPE_STATIC; */
      
      h_add_file = (t_hash_add *)h_entry_file.data;

     }
  else
    index_file = TOKEN_NOT_FOUND;

  h_add_item = (t_hash_add *)h_entry_item.data;

  while( 1 )
   { if( (h_add_item->type & type) &&
	 (index_file == TOKEN_NOT_FOUND || h_add_item->file == index_file)  )
       /* symbol in given file found or no file specified
        */
       break;
      
     if( !strict_flag && (h_add_item->type & type) )
       { /* we found the symbol, but the filename does not match. Store
            and wait, maybe we get the one in the right file too, otherwise,
            return this item, but set the correct file.
         
            But this item has to be a global one !!!
          */
         if( !(h_add_item->type & TYPE_STATIC ) )
            h_add_global = h_add_item;
        }

     if( h_add_item-> next )
	 h_add_item = h_add_item-> next;
     else
       { /* no matching file
	  */
         if( !h_add_global )
	   { send_brs_ret( RET_ERROR, 0, client_pid, 
                   "Item not found", "","","" );
             return(1);
            }
         else
          { /* We found that item, but in an other file. Assume global
               item and return correct file */
            h_add_item = h_add_global;
            if( h_add_item->file != TOKEN_NOT_FOUND )
              { get_token( &h_entry_file, h_add_item->file );
               
                /*  WE DO NOT NEED THIS ANYMORE !!! if( arg2)
                   strcpy( arg2, h_entry_file.indent ); */
               
                h_add_file = (t_hash_add *)h_entry_file.data;
               }
            break;
           }
	}
    }

  if( h_add_item && h_erg1 )
    *h_erg1 = *h_add_item;

  if( h_add_file && h_erg2 )
    *h_erg2 = *h_add_file;

  return( 0 );
 }




main( int argc, char *argv[] )
{ short        i, flag;
  char         *ptr,
               arg1[MAX_IDENT],
	       arg2[MAX_FILENAME];
   
  t_hash_add   h_add1,
	       h_add2,
               *p_h_add;

  t_hash_data  h_data;
   
  long         index;
  t_type       type;

   if( argc <= 1 || argv[1][1] == '?' )
     { fprintf(stderr, "Usage: c-bat_e [Options] -b <browser-file>\n" );
       fprintf(stderr, "Browser Engine: runs in background and processes requests by front- \n");
       fprintf(stderr, "end-modules via file- or IPC-communication.\n");
       fprintf(stderr, "This program should be run in background only !\n");
       fprintf(stderr, "\nOptions:\n");
       fprintf(stderr, "Overall Options:\n");
       fprintf(stderr, "-b <browser-file>  Specify Browser-File\n");
       fprintf(stderr, "-x <exclude-file>  Exclude symbols given in exclude-file\n");
       fprintf(stderr, "    format of exclude-file : one symbol per line\n");
       fprintf(stderr, "    if symbol is a filename: all symbols in this file will be ignored\n");
       fprintf(stderr, "-m <module-file>   Combine source-files to modules given in <module-file>\n");
       fprintf(stderr, "                   format: first module-name, source-files every line,\n");
       fprintf(stderr, "                           empty line to distiguish modules\n");
       exit(1);
      }

   yyin = stdin;

   module_file[0]  = 0;
   exclude_file[0] = 0;

   init_hashtable ();

   for( i = 1; i < argc; i++ )
     { /* ordinary option given: */
       switch( argv[i][1] )
	 { case 'b' : /* Browser-Filename */
	     if( argv[i][2] )
	       { /* Filename given without blank */
		 yyin = fopen( argv[i] + 2, "r" );
                 strcpy( browser_file, argv[i] + 2 ); 
                 set_browser_filename( browser_file);
		 if( !yyin )
		   { fprintf(stderr,
		       "Could not open Browser-file \"%s\" !\n",
		       argv[i]+2);
		     exit(1);
		    }
		}
	     else
	       { /* Browser-filename is in the next argument
		  */
		 i++;
		 yyin = fopen( argv[i], "r" );
                 strcpy( browser_file, argv[i] ); 
                 set_browser_filename( browser_file);
		 if( !yyin )
		   { fprintf(stderr,
		       "Could not open Browser-file \"%s\" !\n",argv[i]);
		     exit(1);
		    }
		}
	     break;

	   case 'x' : /* Exclude-Filename */
	     if( argv[i][2] )
	       { /* Filename given without blank */
		 read_exclude_symbols( argv[i] + 2 );
		}
	     else
	       { /* Browser-filename is in the next argument
		  */
		 i++;
		 read_exclude_symbols( argv[i] );
		}
	     break;

	   case 'm' : /* Module-Filename */
	     if( argv[i][2] )
	       { /* Filename given without blank */
		 read_modules( argv[i] + 2 );
		}
	     else
	       { /* Browser-filename is in the next argument
		  */
		 i++;
		 read_modules( argv[i] );
		}
	     break;

           case 'd' : /* Debug Flex */
	     switch( argv[i][2] )
              { case '0':  /* -d0 ... no yy_flex_debug */
                   flex_debug_flag = 0;
                   break;
                case '1':  /* -d1 ... yy_flex_debug allways on */
                   flex_debug_flag = 1;
                   break;
                case '2':  /* -d2 ... yy_flex_debug only after command */
                   flex_debug_flag = 2;
                   break;
                case '3':  /* -d3 ... output not recognized scanner input */
                   flex_debug2_flag = 1;
                   break;
                case '4':  /* -d4 ... output not recognized scanner input (with line)*/
                   flex_debug2_flag = 2;
                   break;
		}
	     break;

	   default:
	     fprintf(stderr, "Unknown Option \"%s\" !\n", argv[i] );
	     fprintf(stderr, "Type \"c-bat -?\" for help !\n" );
	     exit(1);
	  }
      }


   hash_array      = malloc(sizeof(long) * HASH_ARRAY_INIT_SIZE);
   if( !hash_array )
     { fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		       sizeof( long ) * HASH_ARRAY_INIT_SIZE );
       exit( 1 );
      }
   hash_array_size = HASH_ARRAY_INIT_SIZE;
   hash_array_cnt  = 0;

   hash_changed_array      = malloc(sizeof(long) * HASH_ARRAY_INIT_SIZE);
   if( !hash_changed_array )
     { fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		       sizeof( long ) * HASH_ARRAY_INIT_SIZE );
       exit( 1 );
      }
   hash_changed_array_size = HASH_ARRAY_INIT_SIZE;
   hash_changed_array_cnt  = 0;
   
   if( flex_debug_flag == 1 )
     yy_flex_debug = 1;
   else
     yy_flex_debug = 0;


   /* Init IPC now, because scanning may take more time than the user
      needs to start a client
    */
   init_server();

   /* scan brs-control-file to get all the browser-files
    */
   command = 0;
   
   yylex();
   
   fclose( yyin ); /* close control-file */
   yyin = NULL;
   
   /* Now do the initial scanning !
    */
   command = CMD_INIT;
   
   set_browser_position( 0);

   yyrestart( yyin );
   yy_init = 1;
   yy_start = 0;
   
   lex_time = time(NULL);
   yylex();
   
   /* Get time (in seconds) we used for complete scanning. Will
      be told to clients for recommended minimal timeout
    */
   lex_time = time(NULL) - lex_time;
   /* fprintf(stderr, "Needed %ld seconds to scan .brs-file !\n", lex_time); */

   while( 1 )
     { command = get_brs_cmd( arg1, arg2);

       switch( command )
	 { case CMD_INIT:
		/* Init: Open new Browser-File and do a re-scan
		 */
		yyin = fopen( arg1, "r" );
		if( !yyin )
		  { send_brs_ret( RET_ERROR, 0x1, client_pid, 
                                        "could not open", arg1,"", "" );
                    break;
		   }
         
		reset_hashtable();
                reset_brs_file_list();

		if( exclude_file[0] )
                   read_exclude_symbols( exclude_file );
		if( module_file[0] )
                   read_modules( module_file );

                strcpy( browser_file, arg1 ); 
                set_browser_filename( browser_file);
                act_brs_file = NULL;
         
                /* scan browser-control-file
                 */
                command = 0;

                yyrestart( yyin );
                yy_init = 1;
                yy_start = 0;
                yylex();
         
                fclose(yyin);
                yyin = NULL;
         
		set_browser_position( 0 );
         
                send_brs_ret( RET_OK, 0, client_pid, "", "", "", "" );
                eof_detected = 1;
                lineno = 1;
                goto_file_pos = 0;
                start_file_pos;
                interesting_function = 0;
                main_flag = 0;
                sys_include = 0;
                var_c;
                static_function = 0;
                static_variable = 0;
                command = CMD_INIT;
         
		break;

	   case CMD_CALLS_ONCE:
                symbols_only_once = 1;
                /* no break here !!! */
         
	   case CMD_CALLS:
		/* get all function that are called by function in arg1/arg2
                   if arg1 == "", return functions called outside any function
                   (function pointer)
		 */
                if( arg1[0] )
		  { /* first look for function in symbol table and return the
		       right one (defined in the given file or global scope)
		     */
		    if( get_item_pos(arg1, arg2,
				       TYPE_FUNCTION_DEF, &h_add1, &h_add2, 0) )
		      { /* we could not find the correct function. Error message
		           has been sent, so we just have to return
		         */
		        command = 0;
		        break;
		       }
         
                    get_file_or_module ( &h_data, h_add1.file);

		    /* Now we know what to do (print calls of a function)
		       and where the function starts. position file pointer
		       of yyin to this position and start scanning
		     */

		    interesting_function = 1;

                    /* only one file (exactly one function) hast to be 
                       processed -> prevents double sending of RET_END
                     */
                    act_mainfile_only = 1;
                  }
                else
                  { /* scan whole browser-file for functions "called" outside
                       a function. use interesting_function as a flag
                     */
                    interesting_function = 0;
                    act_mainfile_only = 0;
                    h_add1.f_pos = 0;
                   }
         
                /*** fclose( yyin );
                     yyin = fopen( browser_file, "r"); ***/
         
		set_browser_position( (long)h_add1.f_pos );
         
                eof_detected = 1;
                strcpy( main_input_filename, h_data.ident );
                strcpy( input_filename, h_data.ident );
         
		break;

	   case CMD_CALLED_BY_ONCE:
                symbols_only_once = 1;
                /* no break here !!! */
         
	   case CMD_CALLED_BY:
		/* get all functions that call the function in arg1/arg2
		 */

		/* first look for function in symbol table and return the
		   right one (defined or used in the given file)
		 */

		if( get_item_pos(arg1, arg2,
				   TYPE_FUNCTION_DEF, &h_add1, &h_add2, 0) )
		  { /* we could not find the correct function. Error message
		       has been sent, so we just have to return
		     */
		    command = 0;
		    break;
		   }

		strcpy(target_function, arg1);

		strcpy(target_filename, arg2);

		goto_file_pos = 0; /* start search from beginning */
                act_mainfile_only = 0;

		if( h_add1.type & TYPE_STATIC )
		  { /* static function, we can restrict the search on this
		       source-file
		     */
                    goto_file_pos = h_add2.f_pos;
                    act_mainfile_only = 1;            
		   }

		/* Now we know what to do (print calls of a function)
		   and where the function starts. position file pointer
		   of yyin to this position and start scanning
		 */

		set_browser_position( goto_file_pos );

                goto_file_pos = 0;
 
		break;

	   case CMD_FUNCT_POS:
		/* get position of a function
		 */

		return_f_pos( arg1, arg2, TYPE_FUNCTION_DEF );
                command = 0;
		break;

	   case CMD_VAR_POS:
		/* return definition position of a global variable
		 */

		return_f_pos( arg1, arg2, TYPE_GLOBAL_VAR_DEF );
                command = 0;
		break;

	   case CMD_MACRO_POS:
		/* return definition position of a global variable
		 */

		return_f_pos( arg1, arg2, TYPE_MACRO );
                command = 0;
		break;

	   case CMD_USES_ONCE:
                symbols_only_once = 1;
                /* no break here !!! */
         
	   case CMD_USES:
		/* get all variables that are used by function in arg1/arg2
		 */

		/* first look for function in symbol table and return the
		   right one (defined in the given file or global scope)
		 */

		if( get_item_pos(arg1, arg2,
				   TYPE_FUNCTION_DEF, &h_add1, &h_add2, 0) )
		  { /* we could not find the correct function. Error message
		       has been sent, so we just have to return
		     */
		    command = 0;
		    break;
		   }
         
                get_file_or_module ( &h_data, h_add1.file);

		/* Now we know what to do (print variables used in function)
		   and where the function starts. position file pointer
		   of yyin to this position and start scanning
		 */

		interesting_function = 1;
         
		set_browser_position( (long)h_add1.f_pos );
         
                eof_detected = 1;
                act_mainfile_only = 1;
                strcpy( main_input_filename, h_data.ident );
                strcpy( input_filename, h_data.ident );
		break;

	   case CMD_USES_ALL:
		break;

	   case CMD_USED_BY_ONCE:
                symbols_only_once = 1;
                /* no break here !!! */
         
	   case CMD_USED_BY:
		/* get all functions that use the global variable in arg1/arg2
		 */

		/* first look for variable in symbol table and return the
		   right one (in the given file). This function is originally
                   written for functions, but it should work for global
                   data as well.
         
                   (defined in the given file or global scope)
		 */

		if( get_item_pos(arg1, arg2,
				   TYPE_GLOBAL_VAR_DEF, &h_add1, &h_add2, 0) )
		  { /* we could not find the correct variable. Error message
		       has been sent, so we just have to return
		     */
		    command = 0;
		    break;
		   }

		strcpy(target_variable, arg1);

		strcpy(target_filename, arg2);

		goto_file_pos = 0; /* start search from beginning */
                act_mainfile_only = 0;

		if( h_add1.type & TYPE_STATIC )
		  { /* static variable, we can restrict the search on this
		       source-file
		     */
                    goto_file_pos = h_add2.f_pos;
                    act_mainfile_only = 1;            
		   }

		/* Now we know what to do (print calls of a function)
		   and where the function starts. position file pointer
		   of yyin to this position and start scanning
		 */

		set_browser_position( goto_file_pos );
         
                goto_file_pos = 0;
 
		break;

	   case CMD_FILES:
                for( i = 0; i < hash_array_cnt; i++)
                  {
                    index = hash_array[i];

                    get_token ( &h_data, index);

                    if( h_data.type & TYPE_MAIN_FILE )
                      { get_file_or_module( &h_data, index );
                        if( act_module[0] )
                           send_brs_ret( RET_FILE, 1, client_pid, h_data.ident, 
                                                      act_module, "", "");
                        else
                           send_brs_ret( RET_FILE, 0, client_pid, h_data.ident, 
                                                      "", "", "");
                       }

                   }

                send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );

                command = 0;
		break;
         
           case CMD_WORK_DIR: 
                /* get the working directory of a given file
                 */
                { t_brs_file  *brs_ptr;
            
                  /* filename is in arg1: look for this file
                   */
                  index = search_token( arg1, &h_data);
                  if( index == TOKEN_NOT_FOUND ||
                      ! h_data.data )
	            { /* file not found
	               */
	              send_brs_ret( RET_ERROR, 0, client_pid,
                                    "File not found", "","","" );

	              command = 0;
                      break;
	             }

                  /* get definition postion in browser-files
                     and do calculate the correct brs-file.
                     Return the working-directory of this
                     browser-file
                   */
                  h_add1 = *((t_hash_add *)(h_data.data));
                  brs_ptr = get_brs_list_entry( h_add1.f_pos);

                  send_brs_ret( RET_DIR, 0, client_pid, brs_ptr->dir, 
                                                      "", "", "");

                  command = 0;
		  break;
                 }
                

	   case CMD_INCLUDES:
                /* List all include-files of a given file
                 */

                /* filename is in arg1: look for this file
                 */
                index = search_token( arg1, &h_data);
                if( index == TOKEN_NOT_FOUND ||
                    !(h_data.type & TYPE_MAIN_FILE) ||
                    ! h_data.data )
	          { /* file not found
	             */
	            send_brs_ret( RET_ERROR, 0, client_pid,
                                  "File not found", "","","" );

	            command = 0;
                    break;
	           }

                h_add1 = *((t_hash_add *)(h_data.data));
                goto_file_pos = h_add1.f_pos;
                         
		set_browser_position( goto_file_pos );
         
                goto_file_pos = 0;
                act_mainfile_only = 1;
                eof_detected = 1;
                strcpy( main_input_filename, h_data.ident );
                strcpy( input_filename, h_data.ident );
		break;

	   case CMD_FUNCTIONS:
                funct_filename[0] = 0; /* not only static functions */
                for( i = 0; i < hash_array_cnt; i++)
                  {
                    index = hash_array[i];

                    get_token ( &h_data, index);

                    if( h_data.type & TYPE_FUNCTION_DEF )
                      { dump_hash_item( index, TYPE_FUNCTION_DEF );
                        dump_hash_item( index, TYPE_FUNCTION_DEF | TYPE_STATIC );
                       }

                   }

                send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );

                command = 0;
		break;

	   case CMD_VARIABLES:
                funct_filename[0] = 0; /* not only static variables */
                for( i = 0; i < hash_array_cnt; i++)
                  {
                    index = hash_array[i];

                    get_token ( &h_data, index);

                    if( h_data.type & TYPE_GLOBAL_VAR_DEF )
                      { dump_hash_item( index, TYPE_GLOBAL_VAR_DEF );
                        dump_hash_item( index, TYPE_GLOBAL_VAR_DEF | TYPE_STATIC );
                       }

                   }
                send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );

                command = 0;
		break;
         
	   case CMD_ENUMS:
	   case CMD_STRUCTS:
	   case CMD_UNIONS:
	   case CMD_TYPEDEFS:
	   case CMD_MACROS:
                funct_filename[0] = 0; /* global symbols */
         
                switch( command )
                  { case CMD_ENUMS:     type = TYPE_ENUM; break;
                    case CMD_STRUCTS:   type = TYPE_STRUCT; break;
                    case CMD_UNIONS:    type = TYPE_UNION; break;
                    case CMD_TYPEDEFS:  type = TYPE_TYPEDEF; break;
                    case CMD_MACROS:     type = TYPE_MACRO; break;
                   }
            
                for( i = 0; i < hash_array_cnt; i++)
                  {
                    index = hash_array[i];

                    get_token ( &h_data, index);

                    if( h_data.type & type )
                      { dump_hash_item( index, type );
                       }

                   }
                send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );

                command = 0;
		break;
         
	   case CMD_WHAT_IS:
                /* look if we find the symbol in the hash-table
                 */
                strcpy( funct_filename, arg2 );
         
                index = search_token( arg1, &h_data);
                if( index != TOKEN_NOT_FOUND )
                  { /* test for every possible item in hash-table 
                     */
                    if( h_data.type & TYPE_FUNCTION_DEF)
                      { dump_hash_item( index, TYPE_FUNCTION_DEF );
                        dump_hash_item( index, TYPE_FUNCTION_DEF | TYPE_STATIC );
                       }
            
                    if( h_data.type & TYPE_MODULE)
                      dump_hash_item( index, TYPE_MODULE );
            
                    if( h_data.type & TYPE_SOURCE_FILE )
                      dump_hash_item( index, TYPE_SOURCE_FILE );
            
                    if( h_data.type & TYPE_GLOBAL_VAR_DEF )
                      { dump_hash_item( index, TYPE_GLOBAL_VAR_DEF );
                        dump_hash_item( index, TYPE_GLOBAL_VAR_DEF | TYPE_STATIC );
                       }
            
                    if( h_data.type & TYPE_LOCAL_VAR_DEF )
                      dump_hash_item( index, TYPE_LOCAL_VAR_DEF );
            
                    if( h_data.type & TYPE_ENUM)
                      dump_hash_item( index, TYPE_ENUM );
            
                    if( h_data.type & TYPE_STRUCT)
                      dump_hash_item( index, TYPE_STRUCT );
            
                    if( h_data.type & TYPE_UNION)
                      dump_hash_item( index, TYPE_UNION );
            
                    if( h_data.type & TYPE_ENUM_CONST)
                      dump_hash_item( index, TYPE_ENUM_CONST );
            
                    if( h_data.type & TYPE_COMPONENT)
                      dump_hash_item( index, TYPE_COMPONENT );
            
                    if( h_data.type & TYPE_MACRO)
                      dump_hash_item( index, TYPE_MACRO );
                   }
                if( !(arg2[0]) )
                  { /* no filename specified: send return
                     */
                    send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );
                    command = 0;
                   }
                else
                  { /* filename specified ! Now look for a local variable
                       in this file (later we can implement scoping)
                     */
                    strcpy( target_variable, arg1 );
            
                    index = search_token( arg2, &h_data);
                    if( index == TOKEN_NOT_FOUND ||
                        !(h_data.type & TYPE_SOURCE_FILE) ||
                        ! h_data.data )
	              { /* file not found
	                 */
                        send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );
                        command = 0;
                        break;
	               }

                    h_add1 = *((t_hash_add *)(h_data.data));
                    goto_file_pos = h_add1.f_pos;
                         
		    set_browser_position( goto_file_pos );
         
                    goto_file_pos = 0;
                    act_mainfile_only = 1;
                    eof_detected = 1;
                    strcpy( main_input_filename, h_data.ident );
                    strcpy( input_filename, h_data.ident );
                   }
                    
                break;

           case CMD_REFERENCE_F: 
		/* get component references inside a function
		 */

		/* first look for function in symbol table and return the
		   right one (global functions only)
		 */
		if( get_item_pos(arg2, "",
				   TYPE_FUNCTION_DEF, &h_add1, &h_add2, 0) )
		  { /* we could not find the correct function. Error message
		       has been sent, so we just have to return
		     */
		    command = 0;
		    break;
		   }
         
                get_file_or_module ( &h_data, h_add1.file);

		/* Position file pointer of yyin to this position and 
                   start scanning
		 */

		interesting_function = 1;
         
		set_browser_position( (long)h_add1.f_pos );
         
                eof_detected = 1;
                act_mainfile_only = 1;
                strcpy( main_input_filename, h_data.ident );
                strcpy( input_filename, h_data.ident );
                strcpy( target_component, arg1 );
                break;
         
           case CMD_REFERENCE:   
           case CMD_REFERENCE_V: 
		
		goto_file_pos = 0; /* start search from beginning */
                act_mainfile_only = 0;
         
                if( command == CMD_REFERENCE_V )
                    strcpy(target_variable, arg2);
                else
                  { /* just in the given file
                       filename is in arg2: look for this file
                     */
                    if( arg2[0] ) 
                      { index = search_token( arg2, &h_data);
                        if( index != TOKEN_NOT_FOUND )
                         {  h_add1 = *((t_hash_add *)(h_data.data));
                            goto_file_pos = h_add1.f_pos;
                            act_mainfile_only = 1;
                            strcpy( main_input_filename, h_data.ident );
                            strcpy( input_filename, h_data.ident );
                          }
                       }
            
                   }

                strcpy( target_component, arg1 );

		set_browser_position( goto_file_pos );
         
                goto_file_pos = 0;
                act_mainfile_only = 0;
		break;
         
           case CMD_DEF_POS:
                /* Definition position of an item
                 */
                ptr = strchr( arg1, '*');
                if( ! ptr )
	          { /* Arguments are not seperated by an asterix
	             */
	            send_brs_ret( RET_ERROR, 0, client_pid,
                                  "Arguments not seperated by '*' !", "","","" );

	            command = 0;
                    break;
	           }
         
                *ptr = 0;
                  
                strcpy( target_function, ptr + 1 );
         
                target_line = atoi( arg2 );
         
                /* filename is in arg1: look for this file
                 */
                index = search_token( arg1, &h_data);
                if( index == TOKEN_NOT_FOUND ||
                    !(h_data.type & TYPE_MAIN_FILE) ||
                    ! h_data.data )
	          { /* file not found
	             */
	            send_brs_ret( RET_ERROR, 0, client_pid,
                                  "File not found", "","","" );

	            command = 0;
                    break;
	           }

                h_add1 = *((t_hash_add *)(h_data.data));
                goto_file_pos = h_add1.f_pos;
                         
		set_browser_position( goto_file_pos );
         
                goto_file_pos = 0;
                act_mainfile_only = 1;
                eof_detected = 1;
                strcpy( main_input_filename, h_data.ident );
                strcpy( input_filename, h_data.ident );
                break;
         
	   default:
		fprintf(stderr, "Unknown command-flag: %hd\nAborting ...\n", command);
		break;
	  }


       if( command )
	 { if( flex_debug_flag == 2 )
              yy_flex_debug = 1;
         
           yyrestart( yyin );
           yy_init = 1;
           yy_start = 0;
	   yylex();
         
           /* close the yyin-file ( if open )
            */
           if( yyin )
             { fclose( yyin );
               yyin = NULL;
              }
         
           if( !act_mainfile_only )
            { /* if we scanned complete input, tell client that
                 we are finished now
               */
              switch( command )
                { case CMD_CALLED_BY:
                  case CMD_CALLED_BY_ONCE:
                  case CMD_USED_BY:
                  case CMD_USED_BY_ONCE:
                     end_f_called();
                     break;
                  case CMD_CALLS: 
                  case CMD_CALLS_ONCE:
                     end_f_call();
                     break;
                  case CMD_REFERENCE_F: 
                  case CMD_REFERENCE_V: 
                  case CMD_REFERENCE: 
                     end_reference();
                     break;
                  case CMD_INCLUDES:
                  case CMD_WHAT_IS:
                     send_brs_ret( RET_END, 0, client_pid, "", "", "", "" );
                     break;
                 }
             }
           else
             act_mainfile_only = 0;
         
           symbols_only_once = 0;
         
           if( hash_changed_array_cnt )
            { /* If we changed the Hash-table, we have to clear
                 the FLAG_PRINTED-bits
                 might be more efficient, if we use an own array for
                 the changed hash-indizes $$$
               */
              hash_table_changed = 0;
              clear_invalid_hash_table_entrys = 0;
              
              for( i = 0; i < hash_changed_array_cnt; i++)
                { index = hash_changed_array[i];

                  get_token ( &h_data, index);
                  if( h_data.type == TYPE_UNKNOWN )
                    { /* delete this entry from symbol table and 
                         hash-array
                       */
                      clear_token ( index);
                      continue;
                     }
                  else
                    { /* just reset the FLAG_PRINTED-bit
                       */
                      flag = h_data.flags;
                      if( flag  & FLAG_PRINTED )
                        { flag ^= FLAG_PRINTED;
                          set_token_flag( index, flag );
                         }
                     }

                 } /* next */
            
              /* reset this array */
              hash_changed_array_cnt = 0;
             }
         
           if( flex_debug_flag != 1 )
              yy_flex_debug = 0;
	  }
      }
 }

