/*
    clients.c : Browsing an Analysis Tool for C-source code;
	        Client-Application-Library

    Copyright (C) 1993  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "hash.h"
#include "c-bat.h"

/*********************************************************
 Short overview:

 A Client has first to connect to a server to which it
 can send information requests and get the results.

 Normally, the functions defined in this module do
 automatically connect to the master-server, which is
 the first (or only) server started. However, a client
 can connect to an other server if it knows it's server-ID.

 This ID can be obtained by the get_server_id function
 which asks the master server for the ID of the server
 which processes a special browser file.

 Each client can only be connected to one server at a given
 time. However, you can switch to an other server by first 
 disconnect the old one and then connect to the new
 server.

 All functions send the request to the server when they are
 called for the first time. If they can return more than
 one return-value, the have to be called multiple times.
 After the last value has been returned, these functions
 return RET_END instead of 0.

 For the documentation of the functions, please look in the
 comments preceeding these functions.

 ****************************************/

static int connected = 0;

/* ID of the server to which this client is connected
 */
static int Server_ID = MASTER_ID;

/* Timeout in seconds. Default is set to SECONDS_TIMEOUT
   can be overridden if server recommends higher value
 */
extern int  timeout_value;

/* connects to server1;

   return:  0    ...  O.K.
            else ...  Return-value from server
*/
short brs_connect_server(int server_id)
{ char   arg1[STRINGSIZE],
         arg2[STRINGSIZE],
         arg3[STRINGSIZE],
         arg4[STRINGSIZE];
  short  ret, flag;
  int    server, t_out_val;
 
   
  if( connected )
    return( 0 );
   
  /* send command to print all functions in project
   */
  send_brs_cmd( server_id, CMD_CONNECT, 0, "", "" );
  ret = get_brs_ret( CMD_CONNECT, &server, &flag, arg1, arg2, arg3, arg4);
  
   
  if( ret != RET_CONNECT )
   { /* Error */
     /* printf("Connection failed: Code: %hd (%s)\n", ret, arg1); */
     return( ret );
    } 
   
  /* Change timeout if server recommends higher value
   */
  t_out_val = atoi( arg1 ); 
  if( t_out_val > timeout_value )
    timeout_value = t_out_val;
   
  connected = 1;
  Server_ID = server_id;
  return( (flag) ? -1 : 0 );
}

/* disconnects actual server

   A client may only be connected to one server at a time

*/
void brs_disconnect_server(void)
{ 
   
  if( !connected )
    return;
   
  /* send command to print all functions in project
   */
  send_brs_cmd( Server_ID, CMD_RELEASE, 0, "", "" );
  
  connected = 0;
  return;
}

/* sends a browser filename to the master and returns
   the ID of the server processing this file

   return-value -1 or -2 : Error
 */
int brs_get_server_id(char *browser_file)
{ char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  int server; 
   
  send_brs_cmd( MASTER_ID, CMD_SERVER_ID, 0, browser_file, "" );
    
  ret = get_brs_ret( CMD_SERVER_ID, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_SERVER_ID:  return( flag );
      
     case RET_ERROR: return( -1 );
     default:        return( -2 );
    }
 }
  


/* This function kills server. Return-values like connect_server
   return:  0    ...  O.K.
            else ...  Return-value from server
 */
short brs_kill_server(void)
{ short erg;
   
  if( erg = brs_connect_server( Server_ID ) )
   { return( erg );
    }
   
  send_brs_cmd( Server_ID, CMD_TERMINATE, 0, "", "" );
   
  return( 0 );
 }



/* This function returns all functions defined in the project.
   strings should be have a size of at least STRINGSIZE Bytes

   funct        ...   holds the name of the function,
   file         ...   the file, where the function is defined (or "" if unknown)
   line         ...   the line in the file, where function is defined (if known)
                      linenumber is the first line after the ')' of the arguments
   module       ...   module to which this file belongs to (or "" if not known)
   static_flag  ...   1 if function is static, 0 else

   The first time, this function is called, it issues the 
   CMD_FUNCTIONS-command to the server.
   This function should be called with a NULL-pointer in funct
   to reset internal flags if you want to get all functions a
   second time. However, it does a self-reset, it the RET_END-value 
   is returned by the server.

   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
                          error-message in funct/file
 */
short brs_get_all_funct( char *funct, char *file, long *line, char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !funct )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_FUNCTIONS, 0, "", "" );
      first = 0;
     }
   
  ret = get_brs_ret( CMD_FUNCTIONS, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FUNCT:
          strcpy( funct, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;
         
     case RET_ERROR:
          strcpy( funct, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns all local variables defined in project.

   it works like "get_all_funct" !
 */
short brs_get_all_global_vars( char *funct, char *file, long *line, char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !funct )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_VARIABLES, 0, "", "" );
      first = 0;
     }
   
  ret = get_brs_ret( CMD_VARIABLES, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_VAR:
          strcpy( funct, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;
         
     case RET_ERROR:
          strcpy( funct, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}


/* This function returns all functions called by a specific
   function
   strings should be have a size of at least STRINGSIZE Bytes

   INPUT:
   funct_in     ...   target-function, (just interesting for the first time)
   file_in      ...   file, where target-function is defined (or called: in this
                      case the one with the right scope (global/static) will be
                      taken)
                      may be "", if function is just defined once
   sym_once     ...   if set to 1, every function is just returned once,
                      otherwise every single call will be returned.

   OUTPUT 
   (if ret-value is 0:)
   funct        ...   holds the name of the function,
   file         ...   the file, where the function is called
   line         ...   linenumber, where function is called
   module       ...   always ""
   static_flag  ...   1 if function is static, 0 else

   (if ret-value if RET_ERROR)
   funct        ...   Error message 1
   file         ...   Error message 2 (might be "")


   The first time, this function is called, it issues the 
   CMD_FUNCTIONS-command to the server.
   This function should be called with a NULL-pointer in funct
   to reset internal flags if you want to get all functions a
   second time. However, it does a self-reset, it the RET_END-value 
   is returned by the server.

   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
 */
short brs_get_funct_calls( char *funct_in, char *file_in, short sym_once,
                     char *funct, char *file, long *line, 
                     char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !funct_in )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, (sym_once) ? CMD_CALLS_ONCE : CMD_CALLS, 
                    0, funct_in, file_in );
      first = 0;
     }
   
  ret = get_brs_ret( (sym_once) ? CMD_CALLS_ONCE : CMD_CALLS,
                     &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FUNCT:
          strcpy( funct, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;

     case RET_ERROR:
          strcpy( funct, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;

     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function gets all global variables that are used
   by the given function.

   It works like "get_funct_calls", so see above
   for dokumentation.
 */
short brs_get_var_uses( char *funct_in, char *file_in, short sym_once,
                     char *var, char *file, long *line, 
                     char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !funct_in )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, (sym_once) ? CMD_USES_ONCE : CMD_USES, 
                    0, funct_in, file_in );
      first = 0;
     }
   
  ret = get_brs_ret( (sym_once) ? CMD_USES_ONCE : CMD_USES, 
                     &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_VAR:
          strcpy( var, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;

     case RET_ERROR:
          strcpy( var, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;

     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}


/* This function returns every function, that calls the specific
   function
   strings should be have a size of at least STRINGSIZE Bytes

   INPUT:
   funct_in     ...   target-function, (just interesting for the first time)
   file_in      ...   file, where target-function is defined (or called: in this
                      case the one with the right scope (global/static) will be
                      taken).
                      may be "", if function is just defined once
   sym_once     ...   if set to 1, every function is just returned once,
                      otherwise every single call will be returned.

   OUTPUT 
   (if ret-value is 0:)
   funct        ...   holds the name of the function,
   file         ...   the file, where the target-function is called
   line         ...   linenumber, where target-function is called
   module       ...   always ""
   static_flag  ...   1 if function is static, 0 else

   (if ret-value if RET_ERROR)
   funct        ...   Error message 1
   file         ...   Error message 2 (might be "")


   The first time, this function is called, it issues the 
   CMD_...-command to the server.
   This function should be called with a NULL-pointer in funct
   to reset internal flags if you want to get all functions a
   second time. However, it does a self-reset, it the RET_END-value 
   is returned by the server.

   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
 */
short brs_get_funct_called_by( char *funct_in, char *file_in, short sym_once,
                     char *funct, char *file, long *line, 
                     char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !funct_in )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, (sym_once) ? CMD_CALLED_BY_ONCE : CMD_CALLED_BY, 
                    0, funct_in, file_in );
      first = 0;
     }
   
  ret = get_brs_ret( (sym_once) ? CMD_CALLED_BY_ONCE : CMD_CALLED_BY,
                     &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FUNCT:
          strcpy( funct, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;

     case RET_ERROR:
          strcpy( funct, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;

     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns every function, that uses the specific
   variable
   strings should be have a size of at least STRINGSIZE Bytes

   INPUT:
   var_in       ...   target-variable, (just interesting for the first time)
   file_in      ...   file, where target-variable is defined (or used: in this
                      case the one with the right scope (global/static) will be
                      taken).
                      may be "", if variable is just defined once
   sym_once     ...   if set to 1, every function is just returned once,
                      otherwise every single usage will be returned.

   OUTPUT 
   (if ret-value is 0:)
   funct        ...   holds the name of the function,
   file         ...   the file, where the target-variable is used
   line         ...   linenumber, where target-variable is used
   module       ...   always ""
   static_flag  ...   1 if function is static, 0 else

   (if ret-value if RET_ERROR)
   funct        ...   Error message 1
   file         ...   Error message 2 (might be "")


   The first time, this function is called, it issues the 
   CMD_...-command to the server.
   This function should be called with a NULL-pointer in funct
   to reset internal flags if you want to get all functions a
   second time. However, it does a self-reset, it the RET_END-value 
   is returned by the server.

   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
 */
short brs_get_var_used_by( char *var_in, char *file_in, short sym_once,
                     char *funct, char *file, long *line, 
                     char *module, short *static_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !var_in )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, (sym_once) ? CMD_USED_BY_ONCE : CMD_USED_BY, 
                    0, var_in, file_in );
      first = 0;
     }
   
  ret = get_brs_ret( (sym_once) ? CMD_USED_BY_ONCE : CMD_USED_BY,
                     &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FUNCT:
          strcpy( funct, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          if( static_flag )
            *static_flag = (flag & 0x04) ? 1 : 0;
          break;

     case RET_ERROR:
          strcpy( funct, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;

     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}


/* This function returns all include-files of a given file.

   Every Include is named just once

   INPUT:
   file_in      ...   filename of main-file ( xxx.c ), where the browser
                      should look for includes

   OUTPUT 
   (if ret-value is 0:)
   file         ...   include-filename
   inc_file     ...   file, where #include-directive was found
                      (may be mainfile or an other headerfile in 
                      case of nested include-levels)
   sys_flag     ...   1 if include-file is a system-header

   (if ret-value if RET_ERROR)
   file         ...   Error message 1
   inc_file     ...   Error message 2


   The first time, this function is called, it issues the 
   CMD_FUNCTIONS-command to the server.
   This function should be called with a NULL-pointer in funct
   to reset internal flags if you want to get all functions a
   second time. However, it does a self-reset, it the RET_END-value 
   is returned by the server.


   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
 */
short brs_get_includes(  char *file_in,
                     char *file, char *inc_file, short *sys_flag)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !file_in )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_INCLUDES, 0, file_in, "" );
      first = 0;
     }
   
  ret = get_brs_ret( CMD_INCLUDES, &server, &flag, arg1, arg2, arg3, arg4);
  
  switch( ret )
   { case RET_FILE:
          strcpy( file , arg1 );
          if( arg2[0] )
             strcpy( inc_file , arg2 );
          if( sys_flag )
            *sys_flag = (flag & 0x02) ? 1 : 0;
          break;

     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( inc_file, arg2 );
          first = 1;
          return(ret);
          break;

     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns the position, where a given function
   is defined

   INPUT:
   funct_in     ...   target-function
   file_in      ...   file, where target-function is defined (and defined only !)
                      may be "", if function is just defined once

   OUTPUT 
   (if ret-value is 0:)
   file         ...   the file, where the function is defined
   line         ...   linenumber, where function is defined
   end_line     ...   linenumber, where function ends
   module       ...   module where file belongs to or ""

   (if ret-value if RET_ERROR)
   file         ...   Error message 1
   module       ...   Error message 2 (might be "")


   return:  0        ...  O.K., function-info is stored in arguments
            else     ...  error returned by get_brs_ret
 */
short brs_get_funct_pos( char *funct_in, char *file_in,
                     char *file, long *line, long *end_line, char *module)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;

  /* send command to retrieve position of function
   */
  if( (ret = brs_connect_server( Server_ID )) > 0 )
   { return( ret );
    }
      
  send_brs_cmd( Server_ID, CMD_FUNCT_POS, 0, funct_in, file_in );
   
  ret = get_brs_ret( CMD_FUNCT_POS, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FPOS:
          strcpy( file , arg1 );
          if( arg2[0] )
            *line = atoi( arg2 );
          if( arg3[0] )
            *end_line = atoi( arg3 );
          strcpy( module, arg4 );
          break;

     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( module, arg2 );
          return(ret);
          break;

     case RET_END:
          return( ret );
          break;
         
     default:
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns the position where given global variable
   is defined

   INPUT:
   var_in       ...   target-variable
   file_in      ...   file, where target-variable is defined (defined only !)
                      may be "", if variable is just defined once

   OUTPUT 
   (if ret-value is 0:)
   file         ...   the file, where the variable is defined
   line         ...   linenumber, where variable is defined
   module       ...   module where file belongs to or ""

   (if ret-value if RET_ERROR)
   file         ...   Error message 1
   module       ...   Error message 2 (might be "")


   return:  0        ...  O.K., variable-info is stored in arguments
            else     ...  error returned by get_brs_ret
 */
short brs_get_var_pos( char *var_in, char *file_in,
                     char *file, long *line, char *module)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;

  /* send command to retrieve position of function
   */
  if( (ret = brs_connect_server( Server_ID )) > 0 )
   { return( ret );
    }
      
  send_brs_cmd( Server_ID, CMD_VAR_POS, 0, var_in, file_in );
   
  ret = get_brs_ret( CMD_VAR_POS, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_POS:
          strcpy( file , arg1 );
          if( arg2[0] )
            *line = atoi( arg2 );
          strcpy( module, arg3 );
          break;

     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( module, arg2 );
          return(ret);
          break;

     case RET_END:
          return( ret );
          break;
         
     default:
          return( ret );
          break;
    }

  return( 0 );         
  
}



/* This function tells the browser-engine (server1) to re-init with
   a new browser-file

   return value:
   0     ... O.K.
   else  ... Error, if RET_ERROR, in err_msg1 and err_msg2 errormessages 
             from server
 */
short brs_reinit_browser( char *file, char *err_msg1, char *err_msg2)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret,
         flag;
  short  erg;
    
  if( (erg = brs_connect_server( Server_ID )) > 0 )
   { return( erg );
    }
   
  /* send command to print all functions in project
   */
  send_brs_cmd( Server_ID, CMD_INIT, 0, file, "" );
  ret = get_brs_ret( CMD_INIT, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_OK:
          ret = 0;
          break;
      
     case RET_ERROR:
          strcpy( err_msg1, arg1);
          strcpy( err_msg2, arg2);
          break;
         
     default:
          break;
    }
   
  return( ret );
}


/* This function returns all types (structs, unions, typedefs, enums)
   defined in the project.
   strings should be have a size of at least STRINGSIZE Bytes

   INPUT: 
   type_code    ...   0 ... structs
                      1 ... unions
                      2 ... enums
                      3 ... typedefs

   OUTPUT:
   type_name    ...   holds the name of the type,
   file         ...   the file, where the type is defined (or "" if unknown)
   line         ...   the line in the file, where type is defined (if known)
                      linenumber is the first line after the ')' of the arguments


   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
                          error-message in funct/file
 */
short brs_get_all_types( short type_code, char *type_name, char *file, long *line)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag, cmd;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !type_name )
   { first = 1;
     return( 0 );
    }

  switch( type_code )
    { case 0: cmd = CMD_STRUCTS;  break;
      case 1: cmd = CMD_UNIONS;   break;
      case 2: cmd = CMD_ENUMS;    break;
      case 3: cmd = CMD_TYPEDEFS; break;      
      default: cmd = CMD_STRUCTS; break;
     }

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, cmd, 0, "", "" );
      first = 0;
     }
   
  ret = get_brs_ret( cmd, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_STRUCT:
     case RET_UNION:
     case RET_ENUM:
     case RET_TYPEDEF:
          strcpy( type_name, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          break;
         
     case RET_ERROR:
          strcpy( type_name, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns the usage of components.

   INPUT: 
   type      0   Get all components in a file (or global if arg2 == "")
             1   Get all component references in a function 
             2   Get all components in same line like variable

   component_name  ...   name of the target component of a struct/union
 
   arg2          depends on type:  0 ... filename or ""
                                   1 ... function name
                                   2 ... variable name

   OUTPUT:
   file         ...   the file, where component used
   line         ...   the line in the file, where component used
   function     ...   ident of function where component used

   return:  0        ...  O.K.
            RET_END  ...  No more items found
            else     ...  error returned by get_brs_ret
                          error-message in file/function
 */
short brs_get_component_use( int type, char *comp_name, char *f_in, 
                         char *file, long *line, char *function)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag, cmd;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !comp_name )
   { first = 1;
     return( 0 );
    }


  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      switch( type )
        { case 0: 
             send_brs_cmd( Server_ID, CMD_REFERENCE  , 0, comp_name, f_in );
             cmd = CMD_REFERENCE;
             break;
          case 1: 
             send_brs_cmd( Server_ID, CMD_REFERENCE_F, 0, comp_name, f_in );
             cmd = CMD_REFERENCE_F;
             break;
          case 2: 
             send_brs_cmd( Server_ID, CMD_REFERENCE_V, 0, comp_name, f_in );
             cmd = CMD_REFERENCE_V;
             break;
          default: 
             break;
         }
      first = 0;
     }
   
  ret = get_brs_ret( cmd, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_REFERENCE:
          strcpy( file, arg1 );
          *line = atoi( arg2 );
          strcpy( function , arg3 );
          break;
         
     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( function, arg3 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns all files in a project (just main-files, not the
   includes)
   strings should be have a size of at least STRINGSIZE Bytes

   The first time, this function is called, it issues the 
   CMD_FILES-command to the server.

   Return-values:

     file    ...  filename
     module  ...  module, where file belongs to

   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
                          error-message in funct/file
 */
short brs_get_all_files( char *file, char *module)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !file )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_FILES, 0, "", "" );
      first = 0;
     }
   
  ret = get_brs_ret( CMD_FILES, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_FILE:
          strcpy( file , arg1 );
          strcpy( module, arg2 );
          break;
         
     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( module, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}


/* This function returns the working directory of a given 
   source-file

   INPUT:
   file_in      ...   source-file

   OUTPUT 
   (if ret-value is 0:)
   dir          ...   working-directory

   (if ret-value if RET_ERROR)
   dir          ...   Error message 1


   return:  0        ...  O.K., working-directory stored
            else     ...  error returned by get_brs_ret
 */
short brs_get_file_dir( char *file_in, char *dir)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;

  /* send command to retrieve position of function
   */
  if( (ret = brs_connect_server( Server_ID )) > 0 )
   { return( ret );
    }
      
  send_brs_cmd( Server_ID, CMD_WORK_DIR, 0, file_in, "" );
   
  ret = get_brs_ret( CMD_WORK_DIR, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_DIR:
          strcpy( dir , arg1 );
          break;

     case RET_ERROR:
          strcpy( dir, arg1 );
          return(ret);
          break;

     case RET_END:
          return( ret );
          break;
         
     default:
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns all macros defined in the project.
   strings should be have a size of at least STRINGSIZE Bytes

   macro        ...   holds the name of the macro
   file         ...   the file, where the macro is defined (or "" if unknown)
   line         ...   the line in the file, where macro is defined (if known)


   return:  0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
                          error-message in macro/file
 */
short brs_get_all_macro( char *macro, char *file, long *line )
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !macro )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_MACROS, 0, "", "" );
      first = 0;
     }
   
  do
   { ret = get_brs_ret( CMD_MACROS, &server, &flag, arg1, arg2, arg3, arg4);
            
    } while( !ret );
  switch( ret )
   { case RET_MACRO:
          strcpy( macro, arg1 );
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          break;
         
     case RET_ERROR:
          strcpy( macro, arg1 );
          strcpy( file, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( 0 );         
  
}


/* This function returns definition positions of the unknown item. To handle
   scoping of local items, you can specify the position, where you found that 
   item.
   Note: scoping not implemented yet. The line_in is currently ignored

   item         ...   holds the name of the unknown item
   file_in      ...   file, where item was found (may be "")
   line_in      ...   line, where item was found (or -1)
                      only valid if a filename is specified

   file         ...   file, where item was defined
   line         ...   the line in the file, where item is defined
   module       ...   module, the file belongs to (or "")

   return:  

   RET_VAR           Return variable-position

   RET_FUNCT         Return function-position

   RET_TYPEDEF       Returns typedef

   RET_STRUCT        Returns struct-identifier

   RET_ENUM          Returns enum-type-identifier

   RET_UNION         Returns union-identifier

   RET_MACRO         Returns macro definition
                     Arguments: <macro>,<file>,<line>

       0        ...  O.K., function-info is stored in arguments
            RET_END  ...  No more functions found
            else     ...  error returned by get_brs_ret
                          error-message in macro/file
 */
short brs_what_is( char *item, char *file_in, long line_in,
                   char *file, long *line, char *module )
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;
  static short first = 1;

  /* NULL-Pointer in funct used for reset this function
   */
  if( !item )
   { first = 1;
     return( 0 );
    }
       

  if( first )    
    { /* First time this function is called: 
         send command to print all functions in project
       */
      if( (ret = brs_connect_server( Server_ID )) > 0 )
       { return( ret );
        }
      
      send_brs_cmd( Server_ID, CMD_WHAT_IS, 0, item, file_in );
      first = 0;
     }
   
  do
   { ret = get_brs_ret( CMD_WHAT_IS, &server, &flag, arg1, arg2, arg3, arg4);
            
    } while( !ret );
  switch( ret )
   { case RET_MACRO:
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          break;
         
     case RET_FUNCT:
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          break;
      
     case RET_VAR:
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          strcpy( module, arg4 );
          break;
      
     case RET_STRUCT:
     case RET_UNION:
     case RET_ENUM:
     case RET_TYPEDEF:
          strcpy( file , arg2 );
          if( arg2[0] )
            *line = atoi( arg3 );
          break;
      
     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( module, arg2 );
          first = 1;
          return(ret);
          break;
      
     case RET_END:
          first = 1;
          return( ret );
          break;
         
     default:
          first = 1;
          return( ret );
          break;
    }

  return( ret );         
  
}


/* This function returns the position where the given macro
   is defined

   INPUT:
   macro        ...   target-macro
   file_in      ...   file, where target-macro is defined
                      may be "", if macro is just defined once

   OUTPUT 
   (if ret-value is 0:)
   file         ...   the file, where the macro is defined
   line         ...   linenumber, where macro is defined
   module       ...   module where file belongs to or ""

   (if ret-value if RET_ERROR)
   file         ...   Error message 1
   module       ...   Error message 2 (might be "")

   return:  0        ...  O.K., variable-info is stored in arguments
            else     ...  error returned by get_brs_ret
 */
short brs_get_macro_pos( char *macro, char *file_in,
                     char *file, long *line, char *module)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;

  /* send command to retrieve position of function
   */
  if( (ret = brs_connect_server( Server_ID )) > 0 )
   { return( ret );
    }
      
  send_brs_cmd( Server_ID, CMD_MACRO_POS, 0, macro, file_in );
   
  ret = get_brs_ret( CMD_MACRO_POS, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_POS:
          strcpy( file , arg1 );
          if( arg2[0] )
            *line = atoi( arg2 );
          strcpy( module, arg3 );
          break;

     case RET_ERROR:
          strcpy( file, arg1 );
          strcpy( module, arg2 );
          return(ret);
          break;

     case RET_END:
          return( ret );
          break;
         
     default:
          return( ret );
          break;
    }

  return( 0 );         
  
}

/* This function returns the definition position of a given item
   which was found at given position (within an expression). It
   differs from the other brs_get_..._pos functions, because it
   provides a kind of scoping. It returns the position of the
   item which is valid in the context of the given position.

   It is also independend from the type of the item so it can be
   used for an identifier of an unknown type.

   INPUT:
   item         ...   item
   file_in      ...   file, where item was found
   line_in      ...   line, where item was found

   OUTPUT 
   (if ret-value is 0:)
   file         ...   the file, where the item is defined
   line         ...   linenumber, where item is defined

   (if ret-value if RET_ERROR)
   file         ...   Error message 1

   return:  0        ...  O.K., variable-info is stored in arguments
            else     ...  error returned by get_brs_ret
 */
short brs_get_item_pos( char *item, char *file_in, long line_in,
                        char *file, long *line)
{ int  server;
  char arg1[STRINGSIZE],
       arg2[STRINGSIZE],
       arg3[STRINGSIZE],
       arg4[STRINGSIZE];
  short  ret, flag;

  /* send command to retrieve position of function
   */
  if( (ret = brs_connect_server( Server_ID )) > 0 )
   { return( ret );
    }

  if( strlen(item) + strlen(file_in) + 1 < STRINGSIZE )      
    { strcpy(arg1, file_in);
      strcat(arg1, "*"); /* seperator: '.' */
      strcat(arg1, item );
     }
  else
    { strcpy(file, "Arguments too long !");
      return(RET_ERROR);
     }
   
  sprintf(arg2, "%ld", line_in );
  send_brs_cmd( Server_ID, CMD_DEF_POS, 0, arg1, arg2 );
   
  ret = get_brs_ret( CMD_DEF_POS, &server, &flag, arg1, arg2, arg3, arg4);
  switch( ret )
   { case RET_POS:
          strcpy( file , arg1 );
          if( arg2[0] )
            *line = atoi( arg2 );
          break;

     case RET_ERROR:
          strcpy( file, arg1 );
          return(ret);
          break;

     case RET_END:
          strcpy( file, "" );
          return( ret );
          break;
         
     default:
          strcpy( file, "" );
          return( ret );
          break;
    }

  return( 0 );         
  
}
