/* exc.c */

/*
 * This software is in the public domain and may be freely copied and
 * distributed.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY. IN PARTICULAR, THE AUTHORS DO NOT MAKE ANY REPRESENTATION OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "exc.h"
#undef return

extern int snprintf (char *str, size_t n, const char *fmt, ...);
extern int vsnprintf (char *str, size_t n, const char *fmt, va_list ap);
static void _exc_msg (const char *f, int l, const char *fmt, ...);

/* Pointer to the top of the exception context frame stack. */
_exc_frame_t *_exc_stack = 0;

/*
 * Global used to test if we are in a try statement and, if so, to also record
 * whether we are lexically nested inside a try-finally statement.
 */
char __exc_in_try = 0;

/* Dummy used to test if we executed ereturn or ereturn2. */
char __exc_in_ereturn;

/* Enable debugging if != 0. */
int exc_debug = 0;

/* Disallow use of goto inside a try statement if != 0. */
int exc_no_goto_in_try = 0;

/* Error message strings. */
const char _exc_ereturn_msg[]	= "Called ereturn inside try-finally.";
const char _exc_break_msg[]	= "Unmatched break inside try.";
const char _exc_cont_msg[]	= "Unmatched continue inside try.";
const char _exc_finexc_msg[]	= "Combined finally and except.";
const char _exc_return_msg[]	= "Called return inside try.";
const char _exc_goto_msg[]	= "Called goto inside try.";
const char _exc_except_msg[]	= "Too many exception handlers "
				  "(increase EXC_ARRAY_SZ).";

/* The catch-all exception. */
exc_t exc_any = { "exc_any", 0 }; /* parent field MUST be 0 */

/*
 * _EXC_CHK_SHADOW:
 *     Abort if any handler in the current try statement is shadowed by an
 *     earlier handler in the current try statement.
 */
void
_exc_chk_shadow (file, line)
    const char *file;
    int line;
{
    _exc_frame_t *ctx = _exc_stack;
    int i, j, err=0;
    exc_t *a;

    for (i=1; i<ctx->nx; i++) {		   	   /* for each handler */
	for (j=0; j<i; j++) {			   /* for each prev handler */
	    for (a=ctx->excs[i]; a; a=a->parent) { /* for each ancestor */
	        if (a == ctx->excs[j]) {
		    _exc_msg (file, line, "Handler %s shadowed by handler %s.",
			      ctx->excs[i]->name, a->name);
		    err = 1;
	        }
	    }
	}
    }
    if (err) abort ();
}

/*
 * _EXC_FIND_HANDLER:
 *    Find a handler for exception e, and return a pointer to its enclosing
 *    try statement's context frame.
 */
static _exc_frame_t *
_exc_find_handler (e)
    exc_t *e;
{
    _exc_frame_t *xb;
    exc_t *a;
    int i;

    /*
     * Walk down the context frame stack. For each exception handler in a
     * context frame, check if it can handle e. It can do so if it is either a
     * literal handler for e, or a handler for an ancestor of e.
     */

    for (xb=_exc_stack; xb; xb=xb->prev) {	/* for each frame */
	for (i=0; i<xb->nx; i++) {		/* for each handler */
	    for (a=e;				/* for each ancestor */
		 a && (a != xb->excs[i]);
		 a=a->parent)
		; /* empty body */
	    if (a) { /* found a handler */
		xb->handler = xb->excs[i];
		return xb;
	    }
	}
    }

    /* No handler for exception e was found. */
    return 0;
}

/*
 * _EXC_ERAISE:
 *     Raise exception e.
 */
void
_exc_eraise (e, info, file, line, search)
    exc_t *e;		/* Exception being raised. */
    const char *info;	/* Optional, additional exception information. */
    const char *file;	/* File in which exception is being raised. */
    int line;		/* Line number at which exception is being raised. */
    int search;		/* Non-zero means search for a handler. */
{
    static _exc_frame_t *_exc_handler = 0;
    _exc_frame_t *cb;

    if (search || !_exc_handler) { /* Find a handler for e. */

	if (!(_exc_handler = _exc_find_handler (e))) {
	    _exc_error (file, line, "Unhandled exception %s.", e->name);
	}

	if (search) { /* Save exception details in handler frame. */
            _exc_handler->file = file;
            _exc_handler->line = line;
            if (info) strcpy (_exc_handler->info, info);
	    else *_exc_handler->info = '\0';
	}
    }

    /*
     * Execute all finalization statements between the top of the exception
     * stack and the handler for exception e.
     */

    for (cb = _exc_stack;
	 cb != _exc_handler && !(cb->state & _exc_Finally);
	 cb = cb->prev)
	; /* empty body */

    _exc_stack = cb; /* pop exception frame stack */

    if (cb == _exc_handler) _exc_handler = 0; /* clear cache */

    /*
     * Remember the exception being raised. Note that unlike other exception
     * details, which are recorded only in the exception frame containing the
     * handler, the exception being raised must be saved in intermediate frames
     * as well since it is (possibly) reraised in the end section of each
     * intermediate try-finally statement.
     */

    cb->exc = e;

    _exc_longjmp (cb->jmp, 1);

    /* NOTREACHED */
    _exc_error (__FILE__, __LINE__, "internal error: longjmp botch.");
}

/*
 * _EXC_FMT:
 *    Format additional exception information when raising an exception using
 *    eraise2. Return a pointer to a static buffer. The formatted information
 *    is truncated to at most EXC_INFO_SZ bytes.
 */
char *
_exc_fmt (const char *fmt, ...)
{
    static char _exc_info[EXC_INFO_SZ];
    va_list args;

    va_start (args, fmt);
    (void) vsnprintf (_exc_info, sizeof _exc_info, fmt, args);
    va_end (args);

    return _exc_info;
}

/*
 * _EXC_VMSG:
 */
static void
_exc_vmsg (const char *file, int line, const char *fmt, va_list args)
{
    fprintf (stderr, "exc: %s:%d: ", file, line);
    vfprintf (stderr, fmt, args);
    putc ('\n', stderr);
}

/*
 * _EXC_MSG:
 */
static void
_exc_msg (const char *file, int line, const char *fmt, ...)
{
    va_list args;

    va_start (args, fmt);
    _exc_vmsg (file, line, fmt, args);
    va_end (args);
}

/*
 * _EXC_ERROR:
 *     It's All Over Now, Baby Blue.
 */
int
_exc_error (const char *file, int line, const char *fmt, ...)
{
    va_list args;

    va_start (args, fmt);
    _exc_vmsg (file, line, fmt, args);
    va_end (args);

    abort ();

    /* NOTREACHED */
    return 0;
}
