/* exc.h - a touch of evil */

/*
 * This software is in the public domain and may be freely copied and
 * distributed.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY. IN PARTICULAR, THE AUTHORS DO NOT MAKE ANY REPRESENTATION OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

#ifndef _EXC_H
#define _EXC_H

/* EXC requires an ANSI C compiler and preprocessor. */
#ifndef __STDC__
you lose
#endif

#include <setjmp.h>
#include "config.h"

#ifndef EXC_NO_POLLUTE

#define any		exc_any
#define eraise(e)	exc_eraise (e)
#define eraise2(e, x)	exc_eraise2 (e, x)
#define ereraise	exc_ereraise
#define ereturn		exc_ereturn
#define try		exc_try
#define finally		exc_finally
#define except(e)	exc_except (e)
#define end		exc_end
#ifdef __GNUC__
#define ereturn2(e)	exc_ereturn2 (e)
#endif /* __GNUC__ */

#endif /* EXC_NO_POLLUTE */

/*
 * try/eraise are implemented using setjmp/longjmp, therefore, we need to worry
 * about saving/restoring the signal mask. See file config.h for what needs to
 * be defined.
 */

#if defined (EXC_USE_SIGSETJMP)

    /*
     * If EXC_SAVE_SIGMASK is defined save/restore the signal mask.
     */

    #if defined (EXC_SAVE_SIGMASK)
    #define _EXC_SAVE_SIGMASK	1
    #else
    #define _EXC_SAVE_SIGMASK	0
    #endif

    #define _exc_jmpbuf			sigjmp_buf
    #define _exc_setjmp(env)		sigsetjmp (env, _EXC_SAVE_SIGMASK)
    #define _exc_longjmp(env, val)	siglongjmp (env, val)

#elif defined (EXC_USE_SETJMP)

    /*
     * Don't save/restore the signal mask, unless this is a BSD system.
     */

    #define _exc_jmpbuf			jmp_buf
    #define _exc_setjmp(env)		setjmp (env)
    #define _exc_longjmp(env, val)	longjmp (env, val)

#elif defined (EXC_USE__SETJMP)

    /*
     * This better be a BSD system - don't save/restore the signal mask.
     */

    #if defined (EXC_SAVE_SIGMASK)
    #error Cannot define both EXC_SAVE_SIGMASK && EXC_USE__SETJMP !
    #endif

    #define _exc_jmpbuf			jmp_buf
    #define _exc_setjmp(env)		_setjmp (env)
    #define _exc_longjmp(env, val)	_longjmp (env, val)

#else /* !EXC_USE_SIGSETJMP && !EXC_USE_SETJMP && !EXC_USE__SETJMP */
    #error Must define one of EXC_USE_{,_,SIG}SETJMP !
#endif

/*
 * The exception structure.
 */

typedef struct exc_t exc_t;

struct exc_t {
    const char	*name;		/* Useful for identifying an exception. */
    exc_t	*parent;	/* Parent exception or exc_any. */
};

/*
 * Macros to declare exceptions. exc_decl declares a "non-derived" exception e
 * whose only ancestor is the catch-all exception exc_any. exc_decl_d declares
 * a "derived" exception e with p as its parent.
 *
 * A strictly conforming ANSI C compiler will not allow exc_decl_d to be used
 * inside a function definition since the initializer for the parent field is a
 * non-constant value - the parent field has to be explicitly initialized in
 * such cases using exc_parent. However, this shouldn't be inconvenient in
 * practice since exc_decl_d can be used to declare global derived exceptions,
 * and most exceptions are usually global (since exceptions declared inside a
 * function aren't visible outside that function).
 */

#define exc_decl(e)		exc_t e = { #e, &exc_any }
#define exc_decl_d(e, p)	exc_t e = { #e, &p }

/*
 * Macro to initialize a derived exception's parent field in cases where
 * exc_decl_d can't be used - inside function definitions. Declare p to be e's
 * parent. This should rarely, if ever, be useful in practice.
 */

#define exc_parent(e, p)	e.parent = &p

/*
 * The state of the current try statement is a bitmask of these values.
 * For EXC internal use only.
 */

#define _exc_Initialize    0x01  /* Initializing. */
#define _exc_Finally	   0x02  /* Is a try-finally statement. */
#define _exc_NestedFinally 0x04  /* Nested within a try-finally statement. */
#define _exc_TopLevel	   0x08  /* try statement is not lexically nested. */
#define _exc_Exception     0x10  /* An exception has been raised. */
#define _exc_EvalBody      0x20  /* Execute user's code in try statement. */
#define _exc_Popped        0x40  /* The context frame has been popped. */
#define _exc_Handled	   0x80  /* Exception has been handled. */
#define _exc_While	   0x100 /* Enter try statement while loop. */
#define _exc_Break	   0x200 /* Exit try statement while loop. */

/*
 * Exception context frame allocated for each try statement. The top of the
 * frame stack is pointed to by the global variable _exc_stack.
 */

typedef struct _exc_frame_t _exc_frame_t;

struct _exc_frame_t {
    _exc_jmpbuf	 jmp;		/* setjmp context. */
    int		 nx;		/* Number of slots used in array `excs'. */
    exc_t	 *exc;		/* The exception that was raised, if any. */
    const char	 *file;		/* File in which exception was raised. */
    int		 line;		/* Line number where exception was raised. */
    exc_t	 *handler;	/* Handler for exception `exc'. */
    int		 state;		/* Current state of try statement. */ 
    _exc_frame_t *prev;		/* Link to previous frame on _exc_stack. */
    exc_t	 *excs[EXC_ARRAY_SZ]; /* Array of exception handlers. */
    char	 info[EXC_INFO_SZ];   /* eraise2 information, if any. */
};

/*
 * Raise an exception. Named eraise, instead of raise, to avoid confusion with
 * ANSI raise(3).
 */

#define exc_eraise(e) _exc_eraise (&e, 0, __FILE__, __LINE__, 1)

/*
 * Similar to eraise, but provide some additional exception information "x"
 * which must be a *parenthesized*, printf style argument list,
 * e.g., eraise2 (efile, ("%s: can't open (errno=%d)", fname, errno));
 *
 * The formatted exception information string can be accessed within a handler
 * using exc_info. eraise2 is a limited, but still useful, way of passing
 * information back from the point an exception is raised to its handler.
 */

#define exc_eraise2(e, x)						\
    _exc_eraise (&e, _exc_fmt x, __FILE__, __LINE__, 1);

/*
 * Internal macro to test if we are in a try statement. The global variable
 * __exc_in_try is declared with type char. Whenever we enter a try statement,
 * declare a shadow-variable with the same name, but with type int. If we are
 * in a try statement, __exc_in_try is also used to record if the current try
 * statement is lexically nested within a try-finally statement.
 */

extern char __exc_in_try;
#define _exc_in_try (sizeof (__exc_in_try) > 1)

/*
 * Internal macro that aborts the program when ereturn or ereturn2 is executed
 * within statement S1 of a "try S1 finally S2 end" statement. In such cases
 * the finalization statement S2 might not be executed, which would violate the
 * definition of finally.
 */

#define _exc_chk_ereturn()						\
    do {								\
	if (_exc_stack) {						\
	    if (_exc_stack->state & (_exc_NestedFinally | _exc_Finally)) {\
	        _exc_error (__FILE__, __LINE__, _exc_ereturn_msg);	\
	    }								\
	}								\
    } while (0)

/*
 * Internal macro to unwind the exception stack past all the lexically nested
 * try statements. Used by ereturn and ereturn2.
 */

#define _exc_unwind_top()						\
    do {								\
	int state;							\
	if (_exc_stack) {						\
	    do {							\
	        state = _exc_stack->state;				\
	        _exc_stack = _exc_stack->prev;				\
	    } while (!(state & _exc_TopLevel) && _exc_stack);		\
	}								\
    } while (0)

/*
 * Return from the current (void) function. If we are in a try statement,
 * unwind the exception stack before returning. ereturn may be used outside a
 * try statement, in which case it should be optimized by the compiler into an
 * ordinary C return statement.
 */

#define exc_ereturn							\
    do {								\
	int __exc_in_ereturn;						\
	if (_exc_in_try) {						\
	    _exc_chk_ereturn ();					\
	    _exc_unwind_top ();						\
	}								\
	return;								\
    } while (0)

/*
 * Return the value of expression e from the current (non-void) function. We
 * can't unwind the stack and then do "return e" since e may raise an
 * exception - so keep the stack intact until after e has been successfully
 * evaluated. This requires us to save the value returned from evaluating e
 * somewhere - we use GCC's __typeof__ operator to declare a variable
 * _exc_returnv with the type of expression e, assign the result of evaluating
 * e to _exc_returnv, unwind the stack and then return the value saved in
 * _exc_returnv. However, if an exception is raised while evaluating e, the
 * return doesn't happen - we just go into the exception handling code.
 * ereturn2 may be used outside a try statement, in which case it should be
 * optimized by the compiler into an ordinary C "return e" statement.
 */

#ifdef __GNUC__

#define exc_ereturn2(e)							\
    do {								\
	int __exc_in_ereturn;						\
	if (_exc_in_try) {						\
	    __typeof__ (e) _exc_returnv;				\
	    _exc_chk_ereturn ();					\
	    _exc_returnv = e; /* eval e */				\
	    _exc_unwind_top ();						\
	    return _exc_returnv;					\
	}								\
	else return e;							\
    } while (0)

#endif /* __GNUC__ */

/*
 * Internal macro to test if we have executed ereturn or ereturn2. The global
 * variable __exc_in_ereturn is declared with type char. Whenever we execute
 * ereturn or ereturn2, declare a shadow-variable with the same name, but with
 * type int. This variable is used prevent the return macro defined below from
 * erroneously inferring that we are invoking an ordinary C return statement
 * within a try statement, which is a checked runtime error.
 */

extern char __exc_in_ereturn;
#define _exc_in_ereturn (sizeof (__exc_in_ereturn) > 1)

/*
 * An ordinary C return statement should never be used to break out of a try
 * statement - doing so will leave the exception stack in a mess. Define return
 * as a macro to ensure that this doesn't happen. This code should be optimized
 * by the compiler into an ordinary C return statement outside a try statement,
 * so it should impose no overhead at runtime.
 */

#define return								\
    switch ((_exc_in_try && !_exc_in_ereturn) ?				\
	    _exc_error (__FILE__, __LINE__, _exc_return_msg) : 0)	\
	default: return

/*
 * An ordinary C goto statement should never be used to break out of a try
 * statement. To disallow any use of goto within a try statement (even a
 * possibly legal goto that doesn't transfer control out of a try statement),
 * set the flag exc_no_goto_in_try to a non-zero value. This code should be
 * optimized by the compiler into an ordinary C goto statement outside a try
 * statement, so it should impose no overhead at runtime.
 */

#define goto								\
    switch (_exc_in_try && exc_no_goto_in_try ?				\
	    _exc_error (__FILE__, __LINE__, _exc_goto_msg) : 0)		\
	default: goto

/*
 * Internal macro to push a new exception context frame onto the exception
 * stack. Remember if the current try statement is lexically nested within
 * another try statement - this is used to unwind the exception stack when we
 * see ereturn or ereturn2. Also remember if the current try statement is
 * lexically nested within a try-finally statement - this is used to check if
 * ereturn or ereturn2 are executed within the current try statement. Note that
 * we use the value of __exc_in_try that was set in the enclosing try statement
 * (if any).
 */

#define _exc_push_ctx()							\
    do {								\
	_exc_ctx.state =						\
	    _exc_Initialize |						\
	    (_exc_in_try ? __exc_in_try : _exc_TopLevel);		\
        _exc_ctx.nx = 0;						\
        _exc_ctx.exc = _exc_ctx.handler = 0;				\
        _exc_ctx.prev = _exc_stack;					\
        _exc_stack = &_exc_ctx;						\
    } while (0)

/*
 * Internal macro to pop an exception context frame off the exception stack,
 * unless it has already been popped.
 */

#define _exc_pop_ctx()							\
    do {								\
	if (!(_exc_ctx.state & _exc_Popped)) {				\
            _exc_stack = _exc_ctx.prev;					\
            _exc_ctx.state |= _exc_Popped;				\
	}								\
    } while (0)

/*
 * Internal macro that aborts the program if a break statement that is not
 * matched by a corresponding for or while loop *was* executed within a try
 * statement - since try is implemented using a while loop, this will cause the
 * try statement's while loop to be to be exited prematurely, not the user's
 * enclosing while or for loop. By "unmatched", we mean a break statement whose
 * containing for or while statement is not contained within the try statement,
 * e.g., "for (;;) try if (...) break; end"
 */

#define _exc_chk_break()						\
    do {								\
	if (!(_exc_ctx.state & _exc_Break)) {				\
	    _exc_error (__FILE__, __LINE__, _exc_break_msg);		\
	}								\
    } while (0)

/*
 * Internal macro that aborts the program if an unmatched continue statement
 * *was* executed within a try statement. Since try is implemented using a
 * while loop, this will cause the try statement's while loop to be executed
 * again, not the user's enclosing while or for loop.
 */

#define _exc_chk_continue()						\
    do {								\
        if (!(_exc_ctx.state & _exc_While)) {				\
	    _exc_error (__FILE__, __LINE__, _exc_cont_msg);		\
        }								\
        _exc_ctx.state &= ~_exc_While;					\
    } while (0)

/*
 * Begin an exception handling context. There are 2 phases - in phase I we
 * allocate an exception context frame and initialize it by collecting the
 * try statement's exception handlers; in phase II we execute the user code in
 * the try statement.
 */

#define exc_try								\
    { 									\
	_exc_frame_t _exc_ctx;						\
	_exc_push_ctx ();						\
	if (_exc_setjmp (_exc_ctx.jmp)) { /* an exception was raised */	\
	    _exc_ctx.state &= ~_exc_EvalBody;				\
	    _exc_ctx.state |= _exc_Exception;				\
	}								\
	_exc_ctx.state |= _exc_While;					\
	while (1) {							\
	    volatile int __exc_in_try;					\
	    _exc_chk_continue ();					\
	    if (_exc_ctx.state & _exc_EvalBody) {			\
		__exc_in_try = /* for any contained _exc_push_ctx */	\
		    _exc_ctx.state & (_exc_NestedFinally | _exc_Finally) ? \
		    _exc_NestedFinally : 0;
		/* try code goes here */

/*
 * A try-finally statement contains finalization code that should be executed
 * before the try-finally statement is exited, whether or not an exception was
 * raised.
 */

#define exc_finally							\
	    } 								\
	    if (_exc_ctx.state & _exc_Initialize) {			\
		_exc_ctx.state |= _exc_Finally;				\
	    }								\
	    else {							\
		_exc_pop_ctx ();					\
		if (!(_exc_ctx.state & _exc_NestedFinally)) {		\
		    __exc_in_try = 0;					\
		}							\
		/* finalization code goes here */

/*
 * Declare a handler for exception e and all its descendants. Abort the program
 * if the handler array is full. Within the handler, variables exc_line,
 * exc_file, exc_name and exc_info may be used to obtain more details about the
 * exception that is being handled. The exception being handled may be reraised
 * using ereraise. The order of handlers is important if either derived
 * exceptions or the catch-all handler "any" are used (if exc_debug != 0, then
 * the call to _exc_chk_shadow in "end" will check for handlers in this try
 * statement being shadowed by previous handlers in this try statement).
 */

#define exc_except(e)							\
	    }								\
	    if (_exc_ctx.state & _exc_Initialize) {			\
        	if (_exc_ctx.nx >= EXC_ARRAY_SZ) {			\
	    	    _exc_error (__FILE__, __LINE__, _exc_except_msg);	\
        	}							\
        	_exc_ctx.excs[_exc_ctx.nx++] = &e;			\
	    }								\
	    else if (_exc_ctx.handler == &e) {				\
		exc_t *_exc = _exc_ctx.exc;				\
		int exc_line = _exc_ctx.line;				\
		const char *exc_name = _exc->name,			\
		    	   *exc_file = _exc_ctx.file,			\
		    	   *exc_info = _exc_ctx.info;			\
		(void) exc_name; (void) exc_file; (void) exc_info;	\
		(void) exc_line; /* -Wno-unused */			\
		_exc_pop_ctx ();					\
		_exc_ctx.state |= _exc_Handled;
		/* except code goes here */

/*
 * Reraise the current exception. Can only be used within a handler where we're
 * sure an exception has been raised. This macro is written so that if it is
 * used outside an exception handler, the compiler will issue an error message
 * about variables _exc, exc_info, ... being undeclared.
 */

#define exc_ereraise							\
    _exc_eraise (_exc, exc_info, exc_file, exc_line, 1)

/*
 * End an exception handling context. In phase I, the initialization phase,
 * (1) check that both a finalization statement and exception handlers have not
 * been declared and (2) if debugging is enabled, check that no handler in the
 * current try statement shadows a previous handler in the current try
 * statement. In phase II (1) pop the current exception context frame unless it
 * has already been popped (2) if an exception was raised but not caught,
 * reraise the exception and (3) check that the try statement while loop was
 * not exited prematurely by an unmatched break statement in the user code,
 * rather than the break statement below in exc_end.
 */

#define exc_end								\
	    }								\
	    if (_exc_ctx.state & _exc_Initialize) {			\
		if ((_exc_ctx.state & _exc_Finally) && _exc_ctx.nx) {	\
		    _exc_error (__FILE__, __LINE__, _exc_finexc_msg);	\
		}							\
		if (exc_debug) _exc_chk_shadow (__FILE__, __LINE__);	\
		_exc_ctx.state &= ~_exc_Initialize;			\
		_exc_ctx.state |= _exc_EvalBody | _exc_While;		\
	    }								\
	    else {							\
		_exc_pop_ctx ();					\
		if ((_exc_ctx.state & _exc_Exception)			\
		 && !(_exc_ctx.state & _exc_Handled)) {			\
		    _exc_eraise (_exc_ctx.exc, 0, 0, 0, 0);		\
		}							\
		_exc_ctx.state |= _exc_Break;				\
		break;							\
	    }								\
	}								\
	_exc_chk_break ();						\
    }

/* Top of the stack of exception context frames. */
extern _exc_frame_t *_exc_stack;

/* The catch-all exception. */
extern exc_t exc_any;

/* EXC error messages. */
extern const char
    _exc_ereturn_msg[], _exc_break_msg[], _exc_cont_msg[], _exc_finexc_msg[],
    _exc_except_msg[], _exc_goto_msg[], _exc_return_msg[];

/*
 * Enable debugging if this flag is non-zero. Currently, the only debugging
 * check is for exception handler shadowing.
 */
extern int exc_debug; /* default=0 */

/*
 * Disallow use of goto within a try statement if this flag is non-zero.
 */
extern int exc_no_goto_in_try; /* default=0 */

/*
 * Users should not call these routines directly. We need to declare them here
 * since the exception macros use them.
 */

extern void
    _exc_eraise (exc_t *e, const char *info, const char *f, int l, int search),
    _exc_chk_shadow (const char *f, int l);

extern int
    _exc_error (const char *f, int l, const char *fmt, ...);

extern char *
    _exc_fmt (const char *fmt, ...);

#endif /* _EXC_H */
