--
-- STRING:  String handling
--
-- %Z%%Y%:%M%	%I%

class STRING

inherit ANY
	redefine copy, is_equal, out
	end

creation

   make,
   make_from_c,
   make_from_int,
   make_from_real,
   make_from_double,
   make_from_boolean,
   make_from_string

feature

   make ( n: INTEGER) is
         -- Allocate space for at least `n' characters
      require
         non_negative_size: n >= 0
      do
         if
            space = Void
         then
            !!space.allocate (n)	-- This can't be right
         end
         if 
            capacity < n
         then
            space.allocate ( n)
         end
      ensure
         space /= Void
         capacity >= n
      end -- make

   make_from_c ( c_string: POINTER) is
         -- initialise Current from the C-style string
         -- `c_string', creating current if Void
      do
         make ( c_strlen ( c_string) + 1)
         if
            capacity > 0
         then
            c_strset ( space.space, c_string)
         end
      end -- make_from_c

   make_from_string ( str:STRING) is
		-- initialise from another string
	do
		make_from_c ( str.to_c)

	end -- make_from_string
		

   make_from_int ( value: INTEGER) is
         -- initialise Current as a string representation
         -- of the integer value `value', creating current if Void
      do
         make ( 12)
         if
            capacity > 0
         then
            c_itoa ( space.space, value)
         end
      end -- make_from_int

   make_from_real ( value: REAL) is
         -- initialise Current as a string representation
         -- of the real value `value', creating current if Void
      do
         make ( 24)
         if
            capacity > 0
         then
            c_rtoa ( space.space, value)
         end
      end -- make_from_real

   make_from_double ( value: DOUBLE) is
         -- initialise Current as a string representation
         -- of the double value `value', creating current if Void
      do
         make ( 24)
         if
            capacity > 0
         then
            c_dtoa ( space.space, value)
         end
      end -- make_from_double

   make_from_boolean ( value: BOOLEAN) is
         -- initialise Current as a string representation
         -- of the boolean value `value', creating current if Void
      do
         make ( 7)
         if
            capacity > 0
         then
            c_btoa ( space.space, value)
         end
      end -- make_from_boolean

   from_c ( c_string: POINTER): STRING is
         -- return a new STRING instance initialised from the
         -- C-style string `c_string'
      do
         !!Result.make ( c_strlen ( c_string))
         c_strset ( Result.space.space, c_string)
      end -- from_c

   infix "<=" ( other: like Current): BOOLEAN is
         -- Is current string less than or equal to `other'?
      do
		if
			c_strcmp ( space.space, other.space.space) >= 0
		then
			Result := True
		else
			Result := False
		end

      ensure
         Result implies not ( Current > other)
      end -- infix "<="

   infix "<" ( other: like Current): BOOLEAN is
         -- Is current string lexicographically lower than `other'?
      do
		if
			c_strcmp ( space.space, other.space.space) < 0
		then
			Result := True
		else
			Result := False
		end
      end -- infix "<"

   infix ">=" ( other: like Current): BOOLEAN is
         -- Is current string greater than or euqual to `other'?
      do
		if
			c_strcmp ( space.space, other.space.space) >= 0
		then
			Result := True
		else
			Result := False
		end
      ensure
         Result implies not ( Current < other)
      end -- infix ">="

   infix ">" ( other: like Current): BOOLEAN is
         -- Is current string greater than `other'?
      do
		if
			c_strcmp ( space.space, other.space.space) > 0
		then
			Result := True
		else
			Result := False
		end
      ensure
         Result implies not ( Current <= other)
      end -- infix ">"

   adapt ( s: STRING): like Current is
         -- Object of a type conforming to the type of `s'
         -- initialised with attributes from `s'
      do
      end -- adapt

   append ( s: STRING) is
         -- Append a copy of `s' at the end of current string
      require
         argument_not_void: s /= Void
      do
         space.expand ( count + s.count + 1)
         if
            capacity > 0
         then
            c_strcat ( space.space, s.to_c)
         end
      ensure
         -- count = old count + s.count   REMOVED UNTIL OLD WORKS!!
      end -- append

   capacity: INTEGER is
         -- Number of characters guaranteed to fit in space
         -- currently allocated
      do
         Result := space.capacity
      end

   clear is
         -- Clear out string
      do
	    if 
		  count /= 0
	    then
		  space.release
         end

      ensure
         count = 0
      end -- clear

   copy ( other: STRING) is
         -- Reinitialise with copy of `other'
      require else
         other /= Void
      do
      ensure then
         count = other.count
         -- Forall i: 1 .. count, item(i) = other.item(i)
      end -- copy

   count: INTEGER is
         -- Actual number of characters making up the string
      do
         if
            capacity > 0
         then
            Result := c_strlen ( space.space)
         end
      ensure
         Result >= 0
      end -- count

   empty: BOOLEAN is
         -- Is the string empty?
      do
         if
            count = 0
         then
            Result := True
         end
      end -- empty

   extend ( c: CHARACTER) is
         -- Add `c' at end
      do
      ensure
         count = old count + 1
      end -- extend

   fill_blank is
         -- Fill with blanks
      do
      ensure
         -- Forall i: 1 .. capacity, item(i) = Blank
      end -- fill_blank

   hash_code: INTEGER is
         -- hash code value of current string
      do
      end -- hash_code

   head ( n: INTEGER) is
         -- Remove all but the first `n' characters,
         -- do nothing if `n' >= `count'
      require
         non_negative_argument: n >= 0
      do
      ensure
         -- count = min ( n, old count)
      end -- head

   index_of ( c: CHARACTER; i: INTEGER): INTEGER is
         -- Index of the first occurrence of `c', starting at
         -- position `i'; 0 if not present in current string
      require
          index_large_enough: i >= 1
          index_small_enough: i <= count
      do
      ensure
          -- Result = 0 or item ( Result) = c
      end -- index_of

   is_equal ( other: STRING): BOOLEAN is
         -- Is current string made of same character sequence as `other'?
      do
         if
            c_strcmp ( Current.to_c, other.to_c) = 0
         then
            Result := True
         end
      end -- is_equal

   item, infix "@" ( i: INTEGER): CHARACTER is
         -- Character at position `i'
      require
         index_large_enough: i >= 1
         index_small_enough: i <= count
      do
         Result := c_item ( space.space, i)
      end -- item, infix "@"

   item_code ( i: INTEGER): INTEGER is
         -- Numeric code of character at position `i'
      require
         index_large_enough: i >= 1
         index_small_enough: i <= count
      do
      end -- item_code

   left_adjust is
         -- Remove leading blanks
      do
      ensure
         (count /= 0) implies (item(1) /= ' ')
      end -- left_adjust

   precede ( c: CHARACTER) is
         -- Add `c' at front
      do
      ensure
         count = old count + 1
      end -- precede

   prepend ( s: STRING) is
         -- Prepend a copy of `s' at front of current string
      require
         argument_not_void: s /= Void
      do
      ensure
         count = old count + s.count
      end -- prepend

   put ( c: CHARACTER; i: INTEGER) is
         -- Replace by `c' character at position `i'
      require
         index_large_enough: i >= 1
         index_small_enough: i <= count
      do
      ensure
         item ( i) = c
      end -- put

   remove ( i: INTEGER) is
         -- Remove `i'-th character
      require
         index_large_enough: i >= 1
         index_small_enough: i <= count
      do
      ensure
         count = old count - 1
      end -- remove

   remove_all_occurrences ( c: CHARACTER) is
         -- Remove all occurrences of `c' from current string
      do
      ensure
         -- Forall i: 1 .. count, item(i) /= c
         -- count = old count - `number of occurrecnes of c in old Current'
      end -- remove_all_occurrences

   resize ( newsize: INTEGER) is
         -- Reallocate if needed to accomodate at least `newsize'
         -- Do not lose any characters from the existing string
      require
         new_size_non_negative: newsize >= 0
      do
         if
            newsize > capacity
         then
            space.expand ( newsize)
         end
      ensure
         count >= newsize
         count >= old count
      end -- resize

   right_adjust is
         -- Remove trailing blanks
      do
      ensure
         -- count /= 0 implies item ( count) /= ' '
      end -- right_adjust

   shrink ( s: STRING; n1, n2: INTEGER) is
         -- Reset from `s', removing characters outside of interval `n1' .. `n2'
      require
         argument_not_void: s /= Void
         meaningful_origin: 1 <= n1
         meaningful_interval: n1 <= n2
         meaningful_end: n2 <= s.count
      do
      ensure
         is_equal ( s.substring ( n1, n2))
      end -- shrink

   substring ( n1, n2: INTEGER): STRING is
         -- Copy of sub-string of current string containing all
         -- characters at indicies between `n1' and `n2'
      require
         meaningful_origin: 1 <= n1
         meaningful_interval: n1 <= n2
         meaningful_end: n2 <= count
      do
      ensure
         Result.count = n2 - n1 + 1
         -- Forall i: 1 .. n2-n1, Result.item(i) = item(n1 + i - 1)
      end -- substring

   tail ( n: INTEGER) is
         -- Remove all characters except for the last `n'
         -- do nothing if `n' >= `count'
      require
         non_negative_argument: n >= 0
      do
      ensure
         -- count = min ( n, old count)
      end -- tail

   to_c: POINTER is
         -- A value which a C function may cast into a pointer to a
         -- C form of the string.  Useful only for interfacing with C software.
         -- Caveat: because C uses null characters as end marker for strings,
         -- a null character occurring in an Eiffel string passed to C will
         -- prevent C functions from accessing any character past the null
      do
         Result := space.space
      end -- to_c

   out: STRING is
          -- Printable representation of current string (redefined from ANY)
      do
--
-- DANGER - SHOULD CHECK TO SEE IF A STRING HAS BEEN DEFINED
--
         Result := Current
      end -- out

   to_integer: INTEGER is
         -- Integer value of current string, assumed to contain digits only
         -- when applied to "123" will yield 123
      require
         -- String containing digits only
      do
         Result := c_atoi ( space.space)
      end -- to_integer

   to_lower is
         -- Convert string to lower-case
      do
      end -- to_lower

   to_upper is
         -- Convert string to upper-case
      do
      end -- to_upper

feature { STRING }

   space: CORE
         -- string stored in here

feature { NONE }

   c_strlen ( s: POINTER): INTEGER is
         -- count of bytes until first 0 in `space'
      external "C"
      end -- c_strlen

   c_strcmp ( s1, s2: POINTER): INTEGER is
         -- compare `s1' and `s2', returning 0 if they match
      external "C"
      end -- c_strcmp

   c_strset ( target, source: POINTER) is
         -- copy string from `source' (a C-string) into `target'
      external "C"
      end -- c_strset

   c_strcat ( target, source: POINTER) is
         -- concaternate string from `source' (a C-string) onto `target'
      external "C"
      end -- c_strcat

   c_atoi ( str: POINTER): INTEGER is
         -- return integer representation of string's contents
      external "C"
      end -- c_atoi

   c_itoa ( sp: POINTER; value: INTEGER) is
         -- set C-string `sp' from integer `value'
      external "C"
      end -- c_itoa

   c_rtoa ( sp: POINTER; value: REAL) is
         -- set C-string `sp' from real `value'
      external "C"
      end -- c_rtoa

   c_dtoa ( sp: POINTER; value: DOUBLE) is
         -- set C-string `sp' from double `value'
      external "C"
      end -- c_dtoa

   c_btoa ( sp: POINTER; value: BOOLEAN) is
         -- set C-string `sp' from boolean `value'
      external "C"
      end -- c_btoa

   c_item ( str: POINTER; index: INTEGER): CHARACTER is
         -- return integer representation of string's contents
      external "C"
      end -- c_item

invariant

   0 <= count
   count <= capacity

end -- class STRING
