require 5.000;

# PerlVision - A class library to do ANSI graphics and textmode GUI
# By Ashish Gulhati (hash@well.sf.ca.us)
# V.0.1.0
#
# (C) Ashish Gulhati, 1995. All Rights Reserved.
#
# Free electronic distribution permitted. You are free to use
# PerlVision in your own code so long as this copyright message stays
# intact. PerlVision or derived code may not be used in any commercial
# product without my prior written or PGP-signed consent. Please e-mail 
# me if you make significant changes, or just want to let me know what 
# you're using PerlVision for.

require "pvbasic.pl";

package PV_Static;		# Trivial static text class for dialog boxes

sub new {
    my $type=shift;
    my @params=@_;
    my $self=\@params;
    bless $self;
}

sub place {
    my $self=shift;
    my ($message,$x1,$y1,$x2,$y2)=@$self[0..4];
    my @message=split("\n",$message);
    my $width=$x2-$x1;
    my $depth=$y2-$y1;
    my $i=$y1;
    &pv::fgcolor(0);
    &pv::bgcolor(6);
    foreach (@message[0..$depth]) {
	&pv::set_cur_pos($x1,$i);
	&pv::pvprint (substr ($_,0,$width));
	$i++;
    }
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

package PV_Checkbox;

sub new {			# Creates your basic check box
    my $type = shift;		# $foo = new PV_Checkbox (Label,x,y,stat);
    my @params = @_;		
    my $self = \@params;
    bless $self;
    return $self;
}

sub place {			
    my $self = shift;		
    pv::set_cur_pos($$self[1],$$self[2]); 
    pv::bgcolor(6); pv::fgcolor(15); &pv::pvprint("["); pv::fgcolor(0);
    ($$self[3]) && &pv::pvprint($pv::TICK);
    ($$self[3]) || &pv::pvprint(" ");
    pv::fgcolor(15); &pv::pvprint("]"); pv::fgcolor(0); 
    &pv::pvprint(" $$self[0]");
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {			# Refreshes display of your check box
    my $self = shift;
    pv::set_cur_pos($$self[1]+1,$$self[2]); 
    pv::bgcolor(6); pv::fgcolor(0);
    ($$self[3]) && &pv::pvprint($pv::TICK);
    ($$self[3]) || &pv::pvprint(" ");
    pv::set_cur_pos($$self[1]+1,$$self[2]); 
    &pv::refresh();
}

sub activate {			# Makes checkbox active
    my $self = shift;		# $foo->activate;
    my @key;
    $self->refresh;
    &pv::refresh_cursor();
    while (@key = pv::getkey()) {

	if ($key[1]==7) {	# UpArrow
	    return 1;
	}
	elsif ($key[1]==8) {	# DnArrow
	    return 2;
	}
	elsif ($key[1]==9) {	# RightArrow
	    return 3;
	}
	elsif ($key[1]==10) {	# LeftArrow
	    return 4;
	}
	elsif ($key[1]==18) {	# Help
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    return 6;
	}
	elsif (($key[0] eq "\t") && ($key[1]==200)) { 
	    return 7;
	}

	elsif (($key[0] eq ' ') && ($key[1]==200)) {
	    $self->select;
	}
	$self->refresh;
	&pv::refresh_cursor();
    }
}

sub select {			# Toggles checkbox status
    my $self = shift;
    $$self[3] = ($$self[3] ? 0 : 1);
}

sub stat {			# Returns status of checkbox
    my $self = shift;		# $bar = $foo->status;
    return $$self[3];
}

package PV_Radio;

@ISA = (PV_Checkbox);

sub new {			# Creates your basic radio button
    my $type = shift;		# $foo = new PV_Radio (Label,x,y,stat);
    my @params = (@_,0);
    my $self = \@params;
    bless $self;
    return $self;
}

sub place {			# Displays a radio button
    my $self = shift;		# $foo->display;
    pv::set_cur_pos($$self[1],$$self[2]); 
    pv::bgcolor(6); pv::fgcolor(15); &pv::pvprint("("); pv::fgcolor(0);
    ($$self[3]) && &pv::pvprint($pv::MARK);
    ($$self[3]) || &pv::pvprint(" ");
    pv::fgcolor(15); &pv::pvprint(")"); pv::fgcolor(0); 
    &pv::pvprint(" $$self[0]");
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {			# Refreshes display of your check box
    my $self = shift;
    pv::set_cur_pos($$self[1]+1,$$self[2]); 
    pv::bgcolor(6); pv::fgcolor(0);
    ($$self[3]) && &pv::pvprint($pv::MARK);
    ($$self[3]) || &pv::pvprint(" ");
    pv::set_cur_pos($$self[1]+1,$$self[2]); 
    &pv::refresh();
}

sub group {			# Puts the button in a group
    my $self = shift;		# Should not be called from user programs
    $$self[5] = shift;
}

sub select {			# Turn radio button on
    my $self = shift;
    unless ($$self[3]) {
	$$self[5]->blank if $$self[5];
	$$self[3] = 1;
	$$self[5]->refresh;
    }
}

sub unselect {			# Turn radio button off
    my $self = shift;
    $$self[3] = 0;
}

package PV_RadioG;
		    
sub new {			# Creates your basic radio button group
    my $type = shift;		# $foo = new PV_RadioG (rb1, rb2, rb3...)
    my @params = @_;		# where rbn is of class PV_Radio
    my $self = \@params;
    my $i;
    bless $self;
    foreach $i (@$self) {
	($i->group($self));
    }
    return $self;
}

sub place {
    my $self = shift;
    my $i;
    foreach $i (@$self) {
	$i->display;
    }
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {
    my $self = shift;
    my $i;
    foreach $i (@$self) {
	$i->refresh;
    }
    &pv::refresh();
}

sub blank {			# Unchecks all buttons in the group
    my $self = shift;
    my $i;
    foreach $i (@$self) {
	$i->unselect;
    }
}
    
sub stat {			# Returns label of selected radio button
    my $self = shift;
    my $i;
    foreach $i (@$self) {
	($i->stat) && (return $$i[0]);
    }
    return undef;
}

package PV_Pushbutton;

sub new {			# Creates a basic pushbutton
    my $type = shift;		# PV_Pushbutton ("Label",x1,y1);
    my @params= @_;
    my $self = \@params;
    bless $self;
}

sub place {
    my $self=shift;
    pv::box(@$self[1..2],$$self[1]+length($$self[0])+3,$$self[2]+2,1,7);
    pv::fgcolor(15); pv::set_cur_pos($$self[1]+2,$$self[2]+1);
    &pv::pvprint($$self[0]);
}    

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub press {
    my $self=shift;
    pv::box(@$self[1..2],$$self[1]+length($$self[0])+3,$$self[2]+2,0,7);
    pv::fgcolor(0); pv::set_cur_pos($$self[1]+2,$$self[2]+1);
    pv::pvprint($$self[0]);
    pv::refresh();
}

sub active {
    my $self=shift;
    pv::bgcolor(7);
    pv::fgcolor(0); pv::set_cur_pos($$self[1]+2,$$self[2]+1);
    &pv::pvprint($$self[0]);
    pv::refresh();
}

sub activate {
    my $self=shift;
    $self->active;
    while (@key = pv::getkey()) {

	if ($key[1]==7) {	# UpArrow
	    $self->display;
	    return 1;
	}
	elsif ($key[1]==8) {	# DnArrow
	    $self->display;
	    return 2;
	}
	elsif ($key[1]==9) {	# RightArrow
	    $self->display;
	    return 3;
	}
	elsif ($key[1]==10) {	# LeftArrow
	    $self->display;
	    return 4;
	}
	elsif ($key[1]==18) {	# Help
	    $self->display;
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    $self->display;
	    return 6;
	}
	elsif (($key[0] eq "\t") && ($key[1]==200)) { 
	    $self->display;
	    return 7;
	}

	elsif (($key[0] =~ /[ \n]/) && ($key[1]==200)) {
	    $self->press;
	    return 8;
	}
    }
}

package PV_Cutebutton;

@ISA = (PV_Pushbutton);

sub new {			# A smaller, cuter pushbutton
    my $type = shift;		# PV_Pushbutton ("Label",x1,y1);
    my @params= @_;
    my $self = \@params;
    bless $self;
}

sub place {
    my $self=shift;
    pv::fgcolor(15); pv::set_cur_pos($$self[1],$$self[2]);
    &pv::pvprint("  ".$$self[0]." "); pv::fgcolor(0); pv::pvprint($pv::VT);
    pv::fgcolor(15); pv::set_cur_pos($$self[1],$$self[2]+1);
    &pv::pvprint($pv::BL);pv::fgcolor(0);
    &pv::pvprint(($pv::HZ x (length($$self[0])+2)).$pv::BR);
}    

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub press {
    my $self=shift;
    pv::fgcolor(0); pv::set_cur_pos($$self[1],$$self[2]);
    &pv::pvprint(($pv::TL.($pv::HZ x (length($$self[0])+2))));
    pv::fgcolor(15); pv::pvprint($pv::TR);
    pv::set_cur_pos($$self[1],$$self[2]+1); pv::fgcolor(0);
    &pv::pvprint($pv::VT);
    pv::fgcolor(4); pv::pvprint (" ".$$self[0]."  ");
    pv::refresh();
}

sub active {
    my $self=shift;
    pv::fgcolor(4); pv::set_cur_pos($$self[1]+2,$$self[2]);
    &pv::pvprint($$self[0]);
    pv::refresh();
}

package PV_Plainbutton;

@ISA = (PV_Pushbutton);

sub new {			# A minimal pushbutton
    my $type = shift;		# PV_Pushbutton ("Label",x1,y1);
    my @params= @_;
    my $self = \@params;
    bless $self;
}

sub place {
    my $self=shift;
    pv::fgcolor(15); pv::bgcolor(6); pv::set_cur_pos($$self[1],$$self[2]);
    &pv::pvprint($$self[0])
}    

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub press {
}

sub active {
    my $self=shift;
    pv::bgcolor(4); pv::fgcolor(15); pv::set_cur_pos($$self[1],$$self[2]);
    &pv::pvprint($$self[0]);
    pv::refresh();
}

package PV_SListbox;

sub new {			# Creates a superclass list box
    my $type = shift;		# PV_SListbox (Head,top,x1,y1,x2,y2,list)
    my $head = shift;
    my @params = ($head,0,@_);	# where list is (l1,s1,l2,s2,...)
    my $self = \@params;	# Do not use from outside
    bless $self;
}

sub place {
    my $self = shift;
    my ($top,$x1,$y1,$x2,$y2) = @$self[1..5];
    $self->draw_border;
    my $i = shift;
    $i *= 2;
    $x1++; $y1++;
    while (($y1 < $y2) && ($i+6 < $#$self)) {
	($$self[7+$i]) && ($self->selected($y1,$i));
	($$self[7+$i]) || ($self->unselected($y1,$i));
	$y1++;
	$i += 2;
    }
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {
    my $self = shift;
    my ($top,$x1,$y1,$x2,$y2) = @$self[1..5];
    my $i = shift;
    unless ($i==$top) {
	$$self[1]=$i;
	$i *= 2;
	$x1++; $y1++;
	while (($y1 < $y2) && ($i+6 < $#$self)) {
	    ($$self[7+$i]) && ($self->selected($y1,$i));
	    ($$self[7+$i]) || ($self->unselected($y1,$i));
	    $y1++;
	    $i += 2;
	}
    }
    &pv::refresh();
}

sub unhighlight {
    my $self = shift;
    my ($ypos,$i) = @_;
    ($$self[7+$i]) && ($self->selected($ypos,$i));
    ($$self[7+$i]) || ($self->unselected($ypos,$i));
    &pv::refresh();
}

sub highlight {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    my ($x1,$x2) = @$self[2,4];
    $x1++;
    pv::bgcolor(4); pv::fgcolor(15);
    pv::set_cur_pos($x1+1,$ypos);
    &pv::pvprint (substr ($$self[6+$i],0,$x2-$x1-2).
		 " " x 
		 ($x2-$x1-2-length(substr($$self[6+$i],0,$x2-$x1-2))));
    &pv::refresh();
}

sub selected {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    $self->unselected($ypos,$i);
}

sub reset {
    my $self = shift;
    my $i;
    for ($i=7; $i <= $#$self; $i +=2) {
	$$self[$i] = 0;
    }
    $self->refresh(0);
}

sub stat {
    my $self = shift;
    my $i;
    my @returnlist = ();
    for ($i=7; $i <= $#$self; $i +=2) {
	($$self[$i]) && (@returnlist = (@returnlist,$$self[$i-1]));
    }
    $self->reset;
    return @returnlist;
}

sub done {
    my $self = shift;
    my $i = shift;
    $$self[$i*2+7]=1;
    $self->refresh(0);
}

sub deactivate {
    my $self = shift;
    $self->reset();
}

sub unselected {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    my ($x1,$x2) = @$self[2,4];
    $x1++;
    pv::bgcolor(6); pv::fgcolor(0);
    pv::set_cur_pos($x1+1,$ypos);
    &pv::pvprint (substr ($$self[6+$i],0,$x2-$x1-2).
		 " " x 
		 ($x2-$x1-2-length(substr($$self[6+$i],0,$x2-$x1-2))));
}

sub activate {
    my $self = shift;
    my ($x1,$y1,$x2,$y2) = @$self[2..5];
    my $i = 0;
    my @key;
    $x1++; $y1++;
    my $ypos=$y1;
    $self->refresh($i);
    $self->highlight($y1,$i*2);
    while (@key = pv::getkey()) {

	if ($key[1]==18) {	# Help
	    $self->unhighlight($ypos,$i*2);
	    $self->deactivate();
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    $self->unhighlight($ypos,$i*2);
	    $self->deactivate();
	    return 6;
	}
	elsif ($key[1]==9) {	# RightArrow
	    $self->unhighlight($ypos,$i*2);
	    $self->deactivate();
	    return 3;
	}
	elsif ($key[1]==10) {	# LeftArrow
	    $self->unhighlight($ypos,$i*2);
	    $self->deactivate();
	    return 4;
	}
	elsif (($key[0] eq "\t") && ($key[1]==200)) { 
	    $self->unhighlight($ypos,$i*2);
	    $self->deactivate();
	    return 7;
	}
        elsif (($key[0] eq "\n") && ($key[1] == 200)) {
	    $self->unhighlight($ypos,$i*2);
	    $self->done($i);
	    return 8;		
	}
	elsif (($key[0] eq " ") && ($key[1] == 200)) {
	    $self->select($i);
	    $self->highlight($ypos,$i*2);
	}
	elsif (($key[1] == 7) && ($i != 0)) { # Up
	    ($ypos == $y1) || do {$self->unhighlight($ypos,$i*2); $ypos--};
	    $i--;
	    $self->refresh($i-$ypos+$y1);
	    $self->highlight($ypos,$i*2);
	}
	elsif (($key[1] == 8) && (($i*2+7) < $#$self)) { # Down
	    ($ypos == $y2-1) || do {$self->unhighlight($ypos,$i*2); $ypos++};
	    $i++;
	    $self->refresh($i-$ypos+$y1);
	    $self->highlight($ypos,$i*2);
	}
    }
}

sub draw_border {
    my $self = shift;
    pv::box(@$self[2..5],0,6);
    pv::fgcolor(15); pv::set_cur_pos($$self[2],$$self[3]);
    &pv::pvprint($$self[0]);
}

sub select {
}

package PV_Listbox;

@ISA = (PV_SListbox);

sub new {			# Basic single selection listbox
    my $type = shift;		# PV_Listbox (Head,x1,y1,x2,y2,list)
    my @params = @_;		# where list is (l1,s1,l2,s2,...)
    my $self = new PV_SListbox(@params);
    bless $self;
}

package PV_Mlistbox;

@ISA = (PV_SListbox);

sub new {			# A multiple selection listbox
    my $type = shift;		# PV_Mlistbox (Head,x1,y1,x2,y2,list)
    my @params = @_;		# where list is (l1,s1,l2,s2,...)
    my $self = new PV_SListbox(@params);
    bless $self;
}

sub select {
    my $self = shift;
    my $i = shift;
    if ($$self[7+$i*2]) {
	$$self[7+$i*2] = 0;
    }
    else {
	$$self[7+$i*2] = 1;
    }
}

sub selected {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    my ($x1,$x2) = @$self[2,4];
    $x1++;
    pv::bgcolor(6); pv::fgcolor(10);
    pv::set_cur_pos($x1+1,$ypos);
    &pv::pvprint (substr ($$self[6+$i],0,$x2-$x1-2).
		 " " x 
		 ($x2-$x1-2-length(substr($$self[6+$i],0,$x2-$x1-2))));
}

sub highlight {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    my ($x1,$x2) = @$self[2,4];
    $x1++;
    pv::bgcolor(4); pv::fgcolor(15-5*$$self[7+$i]);
    pv::set_cur_pos($x1+1,$ypos);
    &pv::pvprint (substr ($$self[6+$i],0,$x2-$x1-2).
		 " " x 
		 ($x2-$x1-2-length(substr($$self[6+$i],0,$x2-$x1-2))));
    &pv::refresh();
}

sub deactivate {
    my $self = shift;
    $self->refresh();
}

sub done {
    my $self = shift;
    $self->refresh();
}

package PV_Pulldown;

@ISA = (PV_SListbox);

sub new {			# A pulldown menu box. Used by PV_Menubar
    my $type = shift;		# Don't use from outside
    my @params = (@_);
    my $self = new PV_SListbox(@params);
    bless $self;
}

sub draw_border {
    my $self = shift;
    pv::set_cur_pos(@$self[2..3]);
    &pv::bgcolor(7);
    pv::fgcolor(15);
    &pv::pvprint (($$self[2] == 2) ? $pv::VT : $pv::TR);
    pv::fgcolor(0);
    &pv::pvprint(" " x ($$self[4]-$$self[2]-1).(($$self[4] == 79) ? $pv::VT : $pv::TL));
    my $lines=$$self[4]-$$self[2];
    my $j;
    for ($j=$$self[3]+1; $j<$$self[5]; $j++) {
	&pv::set_cur_pos($$self[2],$j);
	&pv::fgcolor (15); &pv::pvprint ($pv::VT);
	&pv::pvprint (" " x ($lines-1));
	&pv::fgcolor (0); &pv::pvprint ($pv::VT); 
    }
    &pv::set_cur_pos($$self[2],$$self[5]); 
    &pv::fgcolor (15); &pv::pvprint ($pv::BL); 
    &pv::fgcolor (0); &pv::pvprint ($pv::HZ x ($lines-1));
    &pv::pvprint ($pv::BR);
}

sub unselected {
    my $self = shift;
    my $ypos = shift;
    my $i = shift;
    my ($x1,$x2) = @$self[2,4];
    $x1++;
    pv::bgcolor(7); pv::fgcolor(4);
    pv::set_cur_pos($x1+1,$ypos);
    &pv::pvprint (substr ($$self[6+$i],0,$x2-$x1-2).
		 " " x 
		 ($x2-$x1-2-length(substr($$self[6+$i],0,$x2-$x1-2))));
}

sub activate {
    my $self=shift;
    my $savestate=&pv::pv_tellregion(@$self[2..3],$$self[4]+1,$$self[5]);
    $self->display();
    my $ret=$self->PV_SListbox::activate();
    &pv::pv_putregion(@$self[2..3],$$self[4]+1,$$self[5],$savestate);
    &pv::refresh;
    return ($ret,$self->stat());
}

package PV_Menubar;		

sub new {			# A menu bar with pulldowns
    my $type=shift;		# new PV_Menubar(Head,width,depth,l,0,l,0,l,0,l,0,l);
    my @params=@_;
    my $pulldown = new PV_Pulldown ($params[0],2,3,$params[1]+2,$params[2]+3,@params[3..$#params]);
    my $self=[$pulldown];
    bless $self;
}

sub add {			# Add a pulldown to the menubar
    my $self=shift;		# $foo->add(Head,width,depth,l,0,l,0,l,0,l,0,l);
    my @params=@_;
    my $pulldown = new PV_Pulldown ($params[0],2+(10*($#$self+1)),3,
				    $params[1]+2+(10*($#$self+1)),$params[2]+3,
				    @params[3..$#params]);
    $$self[$#$self+1]=$pulldown;
}

sub highlight {
    my $self=shift;
    my $i=shift;
    &pv::set_cur_pos (4+10*$i,2);
    &pv::bgcolor(4); &pv::fgcolor(14);
    &pv::pvprint($$self[$i][0]);
    &pv::refresh();
}

sub unhighlight {
    my $self=shift;
    my $i=shift;
    &pv::set_cur_pos (4+10*$i,2);
    &pv::bgcolor(7); &pv::fgcolor(0);
    &pv::pvprint($$self[$i][0]);
    &pv::refresh();
}

sub activate {
    my $self=shift;
    my $i=0;
    my @key;
    my @ret;
    $self->highlight($i);
    while (@key = pv::getkey()) {

	if ($key[1]==18) {	# Help
	    $self->unhighlight($i);
	    return 5;
	}
	elsif ($key[1]==9) {	# RightArrow
	    $$self[$i]->reset();
	    $self->unhighlight($i);
	    $i = ($i==$#$self ? 0 : $i+1);
	    $self->highlight($i);
	}
	elsif ($key[1]==10) {	# LeftArrow
	    $$self[$i]->reset();
	    $self->unhighlight($i);
	    $i = ($i==0 ? $#$self : $i-1);
	    $self->highlight($i);
	}
	elsif (($key[0] eq "\t") && ($key[1]==200)) { 
	    $self->unhighlight($i);
	    return 7;
	}
        elsif ((($key[0] eq "\n") && ($key[1] == 200)) || ($key[1] == 8))  {
	    while (@ret = ($$self[$i]->activate())) {
		if ($ret[0]==3) {
		    $$self[$i]->reset();
		    $self->unhighlight($i);
		    $i = ($i==$#$self ? 0 : $i+1);
		    $self->highlight($i);
		}
		elsif ($ret[0]==4) {
		    $$self[$i]->reset();
		    $self->unhighlight($i);
		    $i = ($i==0 ? $#$self : $i-1);
		    $self->highlight($i);
		}
		else {
		    last;
		}
	    }
	    if ($ret[0] == 5) {
		$self->unhighlight($i);
		return 5;
	    }
	    elsif ($ret[0] == 8) {
		$self->unhighlight($i);
		return (8,$$self[$i][0].":".$ret[1]);
	    }
	}
    }
}

sub place {
    my $self=shift;
    my ($i);
    &pv::box (2,1,79,3,1,7);
    for ($i=0; $i <=$#$self; $i++) {
	&pv::set_cur_pos (4+10*$i,2);
	&pv::pvprint($$self[$i][0]);
    }
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

package PV_Entryfield;

sub new {			# Creates your basic text entry field
    my $type = shift;		# new PV_Entryfield(x,y,len,start,label,value);
    my @params = @_;
    my $self = \@params;
    bless $self;
}

sub place {
    my $self = shift;
    my $start = shift;
    my ($x,$y,$len,$max,$label,$value)=@$self;
    pv::set_cur_pos($x,$y); pv::bgcolor(6); pv::fgcolor(0);
    &pv::pvprint($label." "); pv::bgcolor(4); pv::fgcolor(15); &pv::pvprint(" ");
    &pv::pvprint(substr($value,$start,$len)); 
    &pv::pvprint("." x ($len - length(substr($value,$start,$len)))); 
    &pv::pvprint (" ");
    pv::bgcolor (6);
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {
    my $self = shift;
    my $start = shift;
    my $i=shift;
    my ($x,$y,$len,$oldstart,$label,$value)=@$self;
    if ($oldstart == $start) {
        pv::set_cur_pos($x+length($label)+2+$i-$start,$y); 
        pv::bgcolor(4); pv::fgcolor(15);
	&pv::pvprint(substr($value,$i,$len-($i-$start))); 
	&pv::pvprint("." x ($len-($i-$start)-length(substr($value,$i,$len)))); 
        pv::bgcolor (6);
    }
    else {
	$$self[3]=$start;
	pv::set_cur_pos($x+length($label)+2,$y); 
        pv::bgcolor(4); pv::fgcolor(15);
	&pv::pvprint(substr($value,$start,$len)); 
	&pv::pvprint("." x ($len - length(substr($value,$start,$len)))); 
        pv::bgcolor (6);
    }
    &pv::refresh();
}

sub activate {			# Makes entryfield active
    my $self = shift;
    my $OVSTRK_MODE=0;
    my ($x,$y,$len,$max,$label)=@$self;
    my $i=0;
    $x += length($label)+2;
    my $start=0; my $savestart=0;
    my $jump=(($len % 2) ? ($len+1)/2 : $len/2);
    $self->refresh($start,$i);
    pv::set_cur_pos($x,$y);
    &pv::refresh_cursor();
    while (@key = pv::getkey()) {

	if ($key[1]==7) {	# UpArrow
	    $self->refresh(0,0);
	    return 1;
	}
	elsif ($key[1]==8) {	# DnArrow
	    $self->refresh(0,0);
	    return 2;
	}
	elsif ($key[1]==18) {	# Help
	    $self->refresh(0,0);
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    $self->refresh(0,0);
	    return 6;
	}

	($key[1]) || do {	# Control-char
	    (($key[0] eq "") || ($key[0] eq "")) && do {
		if ($i) {
		    $i--;
		    substr ($$self[5],$i,1) = "";
		    ($i<$start) && ($start -= $jump);
		    ($start <0) && ($start = 0);
		    $self->refresh($start,$i);
  		    pv::set_cur_pos($x+$i-$start,$y);
		    &pv::refresh_cursor();
		}
	    }
	};
	($key[1]==200) && do {
	    if ($key[0] =~ /[\n\r\t\f]/) {
		($key[0] eq "\t") && do {
		    $self->refresh(0,0);
		    return 7;
		};
		(($key[0] eq "\n") || ($key[0] eq "\r")) && do {
		    $self->refresh(0,0);
		    return 8;
		};
		($key[0] eq "\f") && do {

		};
	    }
	    else {
		substr ($$self[5],$i,$OVSTRK_MODE) = $key[0];
		($i-$start >= $len) && ($start += $jump);
		$self->refresh($start,$i);
		$i++;
	        pv::set_cur_pos($x+$i-$start,$y); 
		&pv::refresh_cursor();
	    }
	};
	($key[1]==1) && do {	# Home
	    ($start) && ($self->refresh(0,0));
	    $i=0; $start=0;
	    pv::set_cur_pos($x,$y);
	    &pv::refresh_cursor();
	};
	($key[1]==2) && do {	# Insert
	    $OVSTRK_MODE = ($OVSTRK_MODE ? 0 : 1);
	};
	($key[1]==3) && do {	# Del
	    if ($i < length($$self[5])) {
		substr ($$self[5],$i,1) = "";
		$self->refresh($start,$i);
    	        pv::set_cur_pos($x+$i-$start,$y); 
		&pv::refresh_cursor();
	    }
	};
	($key[1]==4) && do {	# End
	    $i=length($$self[5]); 
	    $savestart=$start;
	    ($start+$len <= length($$self[5])) && 
	     (($start=$i-$len+1) < 0) && ($start = 0);
	    ($savestart != $start) && ($self->refresh($start,$i));
	    pv::set_cur_pos($x+$i-$start,$y); 
	    &pv::refresh_cursor();
	};
	($key[1]==9) && do {	# RightArrow
	    if ($i < length($$self[5])) {
		$i++;
		$savestart=$start;
		($i-$start >= $len) && ($start += $jump);
		($savestart != $start) && ($self->refresh($start,$i));
	        pv::set_cur_pos($x+$i-$start,$y);
		&pv::refresh_cursor();
	    }
	};
	($key[1]==10) && do {	# LeftArrow
	    if ($i) {
		$i--;
		$savestart=$start;
		($i<$start) && ($start -= $jump);
		($start <0) && ($start = 0);
		($savestart != $start) && ($self->refresh($start,$i));
	        pv::set_cur_pos($x+$i-$start,$y); 
		&pv::refresh_cursor();
	    }
	};
    }
}

sub stat {
    my $self = shift;
    return $$self[5];
}

package PV_Password;

@ISA = (PV_Entryfield);

sub new {			# Creates your basic hidden text entry field
    my $type = shift;		# new PV_Entryfield(x,y,len,max,label,value);
    my @params = @_;
    my $self = \@params;
    bless $self;
}

sub place {
    my $self = shift;
    my $start = shift;
    my ($x,$y,$len,$max,$label,$value)=@$self;
    pv::set_cur_pos($x,$y); pv::bgcolor(6); pv::fgcolor(0);
    &pv::pvprint($label." "); pv::bgcolor(4); pv::fgcolor(15); &pv::pvprint(" ");
    &pv::pvprint("*" x (length(substr($value,$start,$len)))); 
    &pv::pvprint("." x ($len - length(substr($value,$start,$len)))); 
    &pv::pvprint (" ");
    pv::bgcolor (6);
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {
    my $self = shift;
    my $start = shift;
    my $i=shift;
    my ($x,$y,$len,$oldstart,$label,$value)=@$self;
    if ($oldstart == $start) {
        pv::set_cur_pos($x+length($label)+2+$i-$start,$y); 
        pv::bgcolor(4); pv::fgcolor(15);
	&pv::pvprint("*" x (length (substr($value,$i,$len-($i-$start))))); 
	&pv::pvprint("." x ($len-($i-$start)-length(substr($value,$i,$len)))); 
        pv::bgcolor (6);
    }
    else {
	$$self[3]=$start;
	pv::set_cur_pos($x+length($label)+2,$y); 
        pv::bgcolor(4); pv::fgcolor(15);
	&pv::pvprint("*" x (length(substr($value,$start,$len)))); 
	&pv::pvprint("." x ($len - length(substr($value,$start,$len)))); 
        pv::bgcolor (6);
    }
    &pv::refresh();
}

package PV_Combobox;

sub new {			# A basic combo-box
}

package PV_Viewbox;		

sub new {			# A readonly text viewer
    my $type=shift;		# PV_Viewbox (x1,y1,x2,y2,text,top);
    my @params=(@_,[],[]);
    my $self=\@params;
    $$self[4]=~s/[\r\0]//g;	# Strip nulls & DOShit.
    $$self[4]=~s/\t/        /g;	# TABs = 8 spaces.
    $$self[4].="\n";
    my $text = $$self[4];
    $text=~s/\n/\n\t/g;
    @{$$self[6]}=split("\t",$text);
    @{$$self[7]}=();
    bless $self;
}

sub place {
    my $self=shift;
    my ($x1,$y1,$x2,$y2,$text,$start)=@$self;
    my $lines=$y2-$y1-2;
    my $i=0;
    $y1++;
    pv::box(@$self[0..3],0,6);
    $self->refresh(1);
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub refresh {
    my $self=shift;
    my $display=shift;
    ($$self[5]>($#{$$self[6]}-$$self[3]+$$self[1]+2)) && 
	($$self[5]=$#{$$self[6]}-$$self[3]+$$self[1]+2);
    ($$self[5]<0) && ($$self[5]=0);
    my ($x1,$y1,$x2,$y2,$text,$start)=@$self;
    my $lines=$y2-$y1-2;
    my $l;
    my $i=0;
    $y1++; my $len=0;
    pv::bgcolor(6); pv::fgcolor(0);
    foreach (@{$$self[6]}[$start..$start+$lines]) {
	unless ($$self[7][$i] eq $_) {
	    pv::set_cur_pos($x1+2,$y1+$i);
  	    $l=$_;
	    $len=length ($$self[7][$i]);
	    $$self[7][$i] = $l;
	    chop ($l);
	    (length($l) > $x2-$x1-3) && ($l=substr($l,0,$x2-$x1-3));
	    &pv::pvprint($l); 
  	    if (length($l) < $x2-$x1-3) {
		&pv::pvprint (" " x ($x2-$x1-3 - length ($l)));
	    }
	}
    $i++;
    }
    $self->statusbar;
    ($display) || (&pv::refresh());
}

sub statusbar {
}

sub activate {			# Makes viewer active
    my $self = shift;
    my ($x1,$y1,$x2,$y2,$text,$start)=@$self;
    $self->refresh;
    while (@key = pv::getkey()) {

	if ($key[1]==18) {	# Help
	    $self->refresh;
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    $self->refresh;
	    return 6;
	}
	($key[1]==200) && do {
	    if ($key[0] =~ /[\r\t\f]/) {
		($key[0] eq "\t") && do {
		    $self->refresh;
		    return 7;
		};
	    }
	};

	($key[1]==1) && do {	# Home
	    $$self[5]=0;
	    $self->refresh;
	};
	($key[1]==4) && do {	# End
	    $$self[5]=$#{$$self[6]}-$y2+$y1+2;
	    $self->refresh;
	};
	($key[1]==5) && do {	# PgUp
	    $$self[5]-=$y2-$y1-2;
	    $self->refresh;
	};
	($key[1]==6) && do {	# PgDown
	    $$self[5]+=$y2-$y1-2;
	    $self->refresh;
	};
	($key[1]==7) && do {	# UpArrow
	    $$self[5]--;
	    $self->refresh;
	};
	($key[1]==8) && do {	# DownArrow
	    $$self[5]++;
	    $self->refresh;
	};
    }
}

package PV_Editbox;

sub new {			# More or less a complete editor
    my $type=shift;		# PV_Editbox (x1,y1,x2,y2,m,text,index,top);
    my @params=(@_,[],[],0);
    my $self=\@params;
    $$self[5]=~s/[\r\0]//g;	# Strip nulls & DOShit.
    $$self[5]=~s/\t/        /g;	# TABs = 8 spaces.
    $$self[5].="\n";
    bless $self;
    $self->justify(1);
    return $self;
}

sub place {
    my $self=shift;
    my ($x1,$y1,$x2,$y2,$margin,$text,$index,$start)=@$self;
    my $lines=$y2-$y1-2;
    my $i=0;
    $y1++;
    pv::box(@$self[0..3],0,6);
    $self->refresh(1);
}

sub display {
    my $self=shift;
    $self->place;
    &pv::refresh();
}

sub statusbar {
}

sub refresh {
    my $self=shift;
    my $display=shift;
    my ($x1,$y1,$x2,$y2,$margin,$text,$index,$start)=@$self;
    my @visible=@{$$self[9]};
    my $lines=$y2-$y1-2;
    my $i=0; my $l;
    $y1++;
    pv::bgcolor(6); pv::fgcolor(0);
    foreach (@{$$self[8]}[$start..$start+$lines]) {
	unless ($visible[$i] eq $_) {
	    $$self[9][$i] = $_;
	    pv::set_cur_pos($x1+2,$y1+$i);
 	    $l=$_;
	    chop ($l);
	    &pv::pvprint($l); &pv::pvprint (" " x (length ($visible[$i]) - length ($l)));
	}
    $i++;
    }
    $self->statusbar;
    ($display) || (&pv::refresh());
}

sub process_key {
}

sub justify {
    my $self=shift;
    my $mode=shift;
    my ($x1,$y1,$x2,$y2,$margin,$text,$index)=@$self;
    my ($i,$j)=(0,0); my $line; my @text; my $ta; my $tb;
    my @textqq;
    substr ($text,$index,0)="\0";
    $text=~s/ *\n/\n/g;
    if ($mode) {
	$ta="";
	$tb="";
    }
    else {
	$mode=length($text);
	($ta,$tb)=split("\0",$text);
	$ta=$ta."\0";$tb="\0".$tb;
	$ta=~s/(.*)\n\s.*/$1/s; ($ta=~/\0/) && ($ta="");
	$tb=~s/.*?\n\s//s; ($tb=~/\0/) && ($tb="");
	$text=substr($text,length($ta),$mode-(length($ta)+length($tb)));
	$mode=0;
    }
    $text=~s/\n/\n\t/g;
    my @text=split("\t",$text);
    my $j=0;
    for ($i=0; $j<=$#text; $i++) {
	if (($text[$j] eq "\n") || ($text[$j] eq "\0\n")) {
	    $textqq[$i]=$text[$j];
	}
	else {
	    if (length($text[$j]) > $margin) {
		$line=$text[$j];
		$text[$j]=substr($text[$j],0,$margin);
		$text[$j]=~s/^(.*\s+)\S*$/$1/;
		$line=substr($line,length($text[$j])); 
		$line=~s/^\s*//;
		$text[$j]=~s/\s*$/\n/;
		if (($j==$#text) && ($line)) {
		    $text[$j+1]=$line;
		    @textqq[$i]=$text[$j];
		}
		elsif (($line) && 
		       ($text[$j+1]=~/^[\s\0]/)) {
		    $textqq[$i]=$text[$j];
		    $text[$j]=$line; $j--;
		}
		else {
		    $line=~s/\n$//;
		    $line=~s/(\S)$/$1 /;
		    $textqq[$i]=$text[$j];
		    $text[$j+1]=$line.$text[$j+1];
		}
	    }
	    elsif ((!$mode) && 
		   ($j < $#text) &&  
		   (length($text[$j])+
		    length ((split(" ",$text[$j+1]))[0]) < $margin) && 
		   ($text[$j+1] =~ /^[^\s\0]/)) { 

		chop ($text[$j]);
		($text[$j]=~/\s$/) || ($text[$j].=" ");
		$text[$j].=$text[$j+1];
		$textqq[$i]=$text[$j];
		$text[$j+1]=$text[$j];
		$i--;
	    }
	    else {
		$textqq[$i]=$text[$j];
	    }
	}
	$j++;
    }
    $text=join("",@textqq);
    $text=$ta.$text.$tb;
    $index=length((split("\0",$text))[0]);
    substr($text,$index,1)="";
    $$self[6]=$index;
    $$self[5]=$text;
    $text =~ s/\n/\n\t/g;
    @{$$self[8]}=split("\t",$text);
}

sub cursor {
    my $self=shift;
    my ($x1,$y1,$x2,$y2,$margin,$text,$index,$start)=@$self;
    my $textthis=substr($text,0,$index+1);
    my $col=0;
    my $line=($textthis =~ tr/\n//);
    if ($textthis=~/\n$/) {
	($line) && ($line--);
	$col++;
    }
    my $len=(length($$self[8][$line])-1);
    $col+=(length((split("\n",$textthis))[$line]));
    if ($line<$start) {
	$start=$line;
    }
    elsif ($line>=$start+$y2-$y1-1) {
	(($start=$line-$y2+$y1+2) <0) && ($start=0);
    }
    ($$self[7]!=$start) && do {
	$$self[7]=$start;
	$self->refresh;
    };
    pv::set_cur_pos($col+$x1+1,$y1+$line-$start+1);
    return ($col,$line,$len);
}

sub linemove {
    my $self=shift;
    my $dir=shift;
    my $count=shift;
    my ($col, $line, $len) = $self->cursor;
    if ($dir) {
	($line+$count >$#{$$self[8]}) && ($count = $#{$$self[8]} - $line);
	if ($count) {
	    $$self[6]+=($len-$col+1);
	    (length ($$self[8][$line+$count]) < $col) && 
		($col=length ($$self[8][$line+$count]));
	    $$self[6]+=$col;
	    $count--;
	    while ($count) {
		$$self[6]+=length($$self[8][$count+$line]);
		$count--;
	    }
	}
    }
    elsif ($line) {
	($line - $count <0) && ($count = $line);
	$$self[6]-=($col+length($$self[8][$line-$count]));
	(length ($$self[8][$line-$count]) < $col) && 
	    ($col=length ($$self[8][$line-$count]));
	$$self[6]+=$col;
	$count--;
	while ($count) {
	    $$self[6]-=length($$self[8][$line-$count]);
	    $count--;
	}
    }
}

sub e_bkspc {
    my $self = shift;
    my ($col, $line, $len) = $self->cursor;
    if ($$self[6]) {
	$$self[6]--;
	if (substr ($$self[5],$$self[6],1) eq "\n") {
	    substr ($$self[5],$$self[6],1) = "";
	    $self->justify;
	}
	else {
	    substr ($$self[5],$$self[6],1) = "";
	    substr ($$self[8][$line],$col-2,1) = "";
	}
	$self->refresh;
    }
}

sub e_del {
    my $self=shift;
    my ($col, $line, $len) = $self->cursor;
    unless ($$self[6]==length($$self[5])-1) {
	if (substr ($$self[5],$$self[6],1) eq "\n") {
	    substr ($$self[5],$$self[6],1) = "";
	    $self->justify;
	}
	else {
	    substr ($$self[5],$$self[6],1) = "";
	    substr ($$self[8][$line],$col-1,1) = "";
	}
	$self->refresh;
    }
}

sub e_ins {
    my $self = shift;
    my $keystroke = shift;
    my ($col, $line, $len) = $self->cursor;
    if (substr ($$self[5],$$self[6],1) eq "\n") {
	substr ($$self[5],$$self[6],0) = $keystroke;
	substr($$self[8][$line],$col-1,0)=$keystroke;
    }
    else {
	substr ($$self[5],$$self[6],$$self[10]) = $keystroke;
	substr($$self[8][$line],$col-1,$$self[10])=$keystroke;
    }
    $$self[6]++;
    if ((length($$self[8][$line]) >= $$self[4]) || 
	($keystroke eq "\n")) {
	$self->justify;
    }
    $self->refresh;
}

sub stat {
    my $self=shift;
    return $$self[5];
}

sub activate {			# Makes editbox active
    my $self = shift;
    my ($y1,$y2,$margin)=($$self[1],$$self[3],$$self[4]);
    my $exitcode;
    $self->refresh;
    my ($col, $line, $len) = $self->cursor;
    &pv::refresh_cursor();
    while (@key = pv::getkey()) {

	if ($key[1]==18) {	# Help
	    $self->refresh;
	    return 5;
	}
	elsif ($key[1]==19) {	# Menu
	    $self->refresh;
	    return 6;
	}
	else {			# Process key hook for subclasses
	    @exitcode = ($self->process_key (@key));
	    if ($exitcode[0] == 1) {
		$self->refresh;
		return 8;
	    }
	    elsif ($exitcode[0] == 2) {
	    }
	    else {		# Now defaults for the editbox.
		if ($exitcode[0] == 3) {
		    @key = @exitcode[1..2];
		}

		((!$key[1]) && (($key[0] eq "") || ($key[0] eq ""))) && ($self->e_bkspc());
		(($key[1]==200) && ($key[0] eq "\t")) && do {$self->refresh; return 7;};
		(($key[1]==200) && ($key[0] =~ /\r\f/)) && do {pv::redraw(); last;};
		($key[1]==200) && ($self->e_ins($key[0]));
		(($key[1]==2) || ($key[1]==21)) && ($$self[10] = ($$self[10] ? 0 : 1)); 
		(($key[1]==3) || (($key[0] eq "") && (!$key[1]))) && ($self->e_del());
		
		(($key[1]==1) || (($key[0] eq "") && (!$key[1]))) && do {	# Home
		    $$self[6]-=(($self->cursor)[0]-1);
		};
		(($key[1]==4) || (($key[0] eq "") && (!$key[1]))) && do {	# End
		    $$self[6]+=(($self->cursor)[2] - (($self->cursor)[0]-1));
		};
		(($key[1]==5) || ($key[1]==15)) && do {	# PgUp
		    $self->linemove(0,$y2-$y1-2);
		};
		(($key[1]==6) || (($key[0] eq "") && (!$key[1]))) && do {	# PgDown
		    $self->linemove(1,$y2-$y1-2);
		};
		(($key[1]==7) || (($key[0] eq "") && (!$key[1]))) && do {	# UpArrow
		    $self->linemove(0,1);
		};
		(($key[1]==8) || (($key[0] eq "") && (!$key[1]))) && do {	# DownArrow
		    $self->linemove(1,1);
		};
		(($key[1]==9) || (($key[0] eq "") && (!$key[1]))) && do {	# RightArrow
		    unless ($$self[6]==length($$self[5])-1) {
			$$self[6]++;
		    }
		};
		(($key[1]==10) || (($key[0] eq "") && (!$key[1]))) && do {	# LeftArrow
		    if ($$self[6]) {
			$$self[6]--;
		    }
		};
		$self->cursor;
		$self->statusbar;
		($col, $line, $len) = $self->cursor;
		&pv::refresh_cursor();
	    }
	}
    }
}

package PV_Dialog;

sub new {			# The dialog box object
    my $type=shift;		# PV_Dialog ("Label",x1,y1,x2,y2,style,color,
    my @params=(0,@_);		#            Control1,1,2,3,4,5,6,7,8,
    my $self=\@params;		#            Control2,1,2,3,4,5,6,7,8,...)
    bless $self;      
}

sub display {
    my $self=shift;
    $$self[0]=&pv::pv_tellregion($$self[2],$$self[3],$$self[4]+1,$$self[5]);
    &pv::box(@$self[2..7]);
    my $i=8;
    while ($i+7 < $#$self) {
	($$self[$i])->place;
	$i+=9;
    }
    &pv::refresh;
}

sub activate {
    my $self=shift;
    $self->display;
    my $i=1; my @last=();
    while ($i) {
	@last=($i,($$self[8+(($i-1)*9)]->activate));
	$i=$$self[8+(($i-1)*9)+$last[1]];
    }
    $self->hide;
    &pv::refresh();
    return (@last);
}

sub hide {
    my $self=shift;
    ($$self[0]) && (&pv::pv_putregion($$self[2],$$self[3],$$self[4]+1,$$self[5],$$self[0]));
    $$self[0]=0;
}

package PVD;			# Two commonly needed dialog box types

sub message {
    my ($message,$width,$depth)=@_;
    ($width<11) && ($width=11);
    $depth+=4;
    my $x1=int ((80-$width)/2);
    my $y1=4 + int ((19-$depth)/2);
    my $x2=$x1+$width;
    my $y2=$y1+$depth;
    my $static=new PV_Static($message,$x1+2,$y1+1,$x2,$y2-4);
    my $ok = new PV_Cutebutton(" OK ",$x1+int($width/2)-3,$y2-2);
    my $dialog = new PV_Dialog ("",$x1,$y1,$x2,$y2,1,6,
				$ok,1,1,1,1,1,1,1,0,
				$static,0,0,0,0,0,0,0,0);
    $dialog->activate;
}

sub yesno {
    my ($message,$width,$depth)=@_;
    my @message=split("\n",$message);
    ($width<21) && ($width=21);
    $depth+=4;
    my $x1=int ((80-$width)/2);
    my $y1=4 + int ((19-$depth)/2);
    my $x2=$x1+$width;
    my $y2=$y1+$depth;
    my $static=new PV_Static($message,$x1+2,$y1+1,$x2,$y2-4);
    my $yes = new PV_Cutebutton (" YES ",$x1+int($width/2)-9,$y2-2);
    my $no = new PV_Cutebutton (" NO ",$x1+int($width/2)+2,$y2-2);
    my $dialog = new PV_Dialog ("",$x1,$y1,$x2,$y2,1,6,
				$yes,1,1,2,1,1,1,2,0,
				$no,2,3,2,1,2,2,1,0,
				$static,0,0,0,0,0,0,0,0);
    my $stat=($dialog->activate)[0];
    ($stat==2) && ($stat=0);
    return $stat;
}

"PerlVision. (C) Ashish Gulhati, 1995";
