# tixBalloon -
#
#	The help widget. It provides both "balloon" type of help message
# and "status bar" type of help message. You can use this widget to indicate
# the function of the widgets inside your application.
#
#

tixWidgetClass tixBalloon {
    -classname TixBalloon
    -superclass tixShell
    -method {
	bind post unbind
    }
    -flag {
	-installcolormap -initwait -state -statusbar
    }
    -configspec {
	{-installcolormap installColormap InstallColormap false}
	{-initwait initWait InitWait 200}
	{-state state State both}
	{-statusbar statusBar StatusBar {}}

 	{-cursor cursor Cursur left_ptr}
    }
    -default {
	{*background 			#ffff60}
	{*foreground 			black}
	{*borderWidth 			0}
	{.borderWidth 			1}
	{.background 			black}
    }
}


proc tixBalloon::InitWidgetRec {w} {
    upvar #0 $w data

    tixChainMethod $w InitWidgetRec

    set data(popped)    0
    set data(statusSet) 0
    set data(serial)    0
    set data(fakeEnter) 0
    set data(curWidget) {}
}

proc tixBalloon::ConstructWidget {w} {
    upvar #0 $w data

    tixChainMethod $w ConstructWidget

    wm overrideredirect $w 1
    wm withdraw $w

    # Frame 1 : arrow
    frame $w.f1 -bd 0
    set data(w:label) [label $w.f1.lab -bd 0 -relief flat \
		       -bitmap [tix getbitmap balArrow]]
    pack $data(w:label) -side left -padx 1 -pady 1
    
    # Frame 2 : Message
    frame $w.f2 -bd 0
    set data(w:message) [message $w.f2.message -padx 0 -pady 0 -bd 0]
    pack $data(w:message) -side left -expand yes -fill both -padx 10 -pady 1

    # Pack all
    pack $w.f1 -fill both
    pack $w.f2 -fill both

}

bind TixBalloon <Leave> "tixBalloon::Leave %W %W %X %Y"
bind TixBalloon <Visibility> "raise %W"

#----------------------------------------------------------------------
# Config:
#----------------------------------------------------------------------

proc tixBalloon::config-state {w value} {
    upvar #0 $w data

    case $value {
	{none balloon status both} {}
	default {
	   error "invalid value $value, must be none, balloon, status, or both"
	}
    }
}

#----------------------------------------------------------------------
# PrivateMethods:
#----------------------------------------------------------------------

proc tixBalloon::ClientDestroy {w client} {
    upvar #0 $w data

    if {$data(curWidget) == $client} {
	tixBalloon::Popdown $w
    }

    # Maybe thses have already been unset by the Destroy method
    #
    catch {unset data(m:$client)}
    catch {unset data(s:$client)}
}

# Handle the mouse pointer entering the client widget
#
proc tixBalloon::Enter {w client} {
    upvar #0 $w data

    if {$data(fakeEnter) == 1} {
	# The mouse pointer just left either the balloon window or the
	# client window: do nothing; otherwise the balloon will flash
	#
	set data(fakeEnter) 0
	return
    }
    if {$data(-state) != "none"} {
    	set data(popped)    0
    	set data(statusSet) 0
	set data(curWidget) $client
	incr data(serial)
    	after $data(-initwait) tixBalloon::Activate $w $data(serial)
    }
}

#
#  This proc is a big hack
proc tixBalloon::post {w client} {
    upvar #0 $w data

    if {![info exists data(m:$client)]} {
	return
    }
    tixBalloon::Enter $w $client
    set data(fakeEnter) 1
}

proc tixBalloon::Leave {w client rootX rootY} {
    upvar #0 $w data

    set cw [winfo containing $rootX $rootY]
    if {$cw == "." && $data(curWidget) == "."} {
	# The root window have this problem: [winfo containing] may
	# return "." even the mouse is outside the root window
	#
	set rx1 [winfo rootx .]
	set ry1 [winfo rooty .]
	set rw [winfo width .]
	set rh [winfo height .]
	set rx2 [expr $rx1+$rw]
	set ry2 [expr $ry1+$rh]

	if {$rootX > $rx1 && $rootX < $rx2 && $rootY > $ry1 && $rootY < $ry2} {
	    # mouse moves from balloon to the root window (current client)
	    # This causes a fake enter to the root window, let's
	    # do nothing.
	    #
	    set data(fakeEnter) 1
	    return 
	}
    }

    if {$data(curWidget) == $cw || [string match $data(curWidget).* $cw]} {
	# mouse moves from balloon to current client widget
	# This causes a fake enter to the client widget, let's
	# do nothing.
	#
	set data(fakeEnter) 1
	return
    }

    if {$w == $cw || [string match $w.* $cw]} {
	# mouse moves from current client widget to balloon
	#
	set data(fakeEnter) 1
	return
    }

    if {$data(popped) == 1 || $data(statusSet) == 1} {
	if {$data(popped) == 1} {
	    set data(fakeEnter) 0
	    tixBalloon::Popdown $w
	}
	if {$data(statusSet) == 1} {
	    tixBalloon::ClearStatus $w
	}
    } elseif {$client == $data(curWidget) || $client == $w} {
	# have to make sure that previous popup's are cancelled
	# just make sure previous
	#
	incr data(serial)
    }
}

proc tixBalloon::Activate {w serial} {
    if {![winfo exists $w]} {
	return
    }

    upvar #0 $w data
    
    if {$serial != $data(serial)} {
	# a new balloon will be activated by the latest call
	#
	return	
    } else {
	if {$data(-state) == "both" || $data(-state) == "balloon"} {
	    tixBalloon::Popup $w
	}
	if {$data(-state) == "both" || $data(-state) == "status"} {
	    tixBalloon::SetStatus $w
	}
    }
}

proc tixBalloon::Popup {w} {
    upvar #0 $w data

    if [tixGetBoolean -nocomplain $data(-installcolormap)] {
	wm colormapwindows [winfo toplevel $data(curWidget)] $w
    }

    # trick: the following lines allow the balloon window to
    # acquire a stable width and height when it is finally
    # put on the visible screen
    #
    set client $data(curWidget)
    $data(w:message) config -text $data(m:$client)
    wm geometry $w +10000+10000
    wm deiconify $w
    raise $w
    update

    # Put it on the visible screen
    #
    set x [expr [winfo rootx $client]+[winfo width  $client]/2]
    set y [expr int([winfo rooty $client]+[winfo height $client]/1.3)]

    wm geometry $w +$x+$y

    set data(popped) 1
}

proc tixBalloon::Popdown {w} {
    upvar #0 $w data

    wm withdraw $w
    set data(popped) 0

}

proc tixBalloon::SetStatus {w} {
    upvar #0 $w data

    if {$data(-statusbar) != {}} {
	$data(-statusbar) config -text $data(s:$data(curWidget))
    }
    set data(statusSet) 1
}

proc tixBalloon::ClearStatus {w} {
    upvar #0 $w data

    # Clear the StatusBar widget
    #
    if {$data(-statusbar) != {}} {
	$data(-statusbar) config -text ""
    }
}

proc tixBalloon::BindOneWidget {w client subwidget} {
    upvar #0 $w data

    set class [winfo class $subwidget]

    bind TixBalloon$client <Any-Enter>  "tixBalloon::Enter $w $client"
    bind TixBalloon$client <Any-Leave>  "tixBalloon::Leave $w $client %X %Y"
    bind TixBalloon$client <Destroy>    "tixBalloon::ClientDestroy $w $client"

    tixAppendBindTag $client TixBalloon$client
}

#----------------------------------------------------------------------
# PublicMethods:
#----------------------------------------------------------------------

# %% if balloon is already popped-up for this client, change mesage
#
proc tixBalloon::bind {w client args} {
    upvar #0 $w data

    if [info exists data(m:$client)] {
	set alreadyBound 1
    } else {
	set alreadyBound 0
    }

    set opt(-balloonmsg) {}
    set opt(-statusmsg)  {}
    set opt(-msg)        {}

    tixHandleOptions opt {-balloonmsg -msg -statusmsg} $args

    if {$opt(-balloonmsg) != {}} {
	set data(m:$client) $opt(-balloonmsg)
    } else {
	set data(m:$client) $opt(-msg)
    }
    if {$opt(-balloonmsg) != {}} {
	set data(s:$client) $opt(-statusmsg)
    } else {
	set data(s:$client) $opt(-msg)
    }

    # Set up the bindings of the widget, in which the balloon should appear
    #
    if {! $alreadyBound} {
	if [tixWInfo tix $client] {
	    # This is a Tix style compound widgets, it needs special attention:
	    # we must bind all its descendants
	    #
	    foreach subwidget [tixDescendants $client] {
		tixBalloon::BindOneWidget $w $client $subwidget
	    }
	} else {
	    tixBalloon::BindOneWidget $w $client $client
	}
    }
}

proc tixBalloon::unbind {w client} {
    upvar #0 $w data

    if [info exists data(m:$client)] {
	catch {unset data(m:$client)}
	catch {unset data(s:$client)}

	if [winfo exists $client] {
	    tixDeleteBindTag $client TixBalloon$client
	}
    }
}

