/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to report errors
 */

#include <ctype.h>
#include <errno.h>
#include <ac/stddef.h>
#include <stdio.h>
#include <ac/stdlib.h>
#include <ac/string.h>
#include <ac/time.h>

#include <arglex.h>
#include <error.h>
#include <mprintf.h>
#include <s-v-arg.h>


/*
 * NAME
 *	wrap - wrap s string over lines
 *
 * SYNOPSIS
 *	void wrap(char *);
 *
 * DESCRIPTION
 *	The wrap function is used to print error messages onto stderr
 *	wrapping ling lines.
 *
 * CAVEATS
 *	Line length is assumed to be 80 characters.
 */

#define PAGE_WIDTH 79

static void wrap _((char *));

static void
wrap(s)
	char	*s;
{
	static char escapes[] = "\rr\nn\ff\bb\tt";
	char	tmp[PAGE_WIDTH + 2];
	int	first_line;
	char	*tp;
	int	bomb_later;
	
	/*
	 * Flush stdout so that errors are in sync with the output.
	 * If you get an error doing this, whinge about it _after_ reporting
	 * the originating error.  Also, clear the error on stdout to 
	 * avoid getting caught in an infinite loop.
	 */
	star_eoln();
	if (fflush(stdout) || ferror(stdout))
	{
		bomb_later = errno;
		clearerr(stdout);
	}
	else
		bomb_later = 0;

	first_line = 1;
	while (*s)
	{
		char	*ep;
		int	ocol;

		/*
		 * Work out how many characters fit on the line.
		 */
		if (first_line)
			ocol = strlen(progname) + 2;
		else
			ocol = 8;
		for (ep = s; *ep; ++ep)
		{
			int	cw;
			int	c;

			c = (unsigned char)*ep;
			if (isprint(c))
				cw = 1 + (c == '\\');
			else
				cw = (strchr(escapes, c) ? 2 : 4);
			if (ocol + cw > PAGE_WIDTH)
				break;
			ocol += cw;
		}

		/*
		 * see if there is a better place to break the line
		 */
		if (*ep && *ep != ' ')
		{
			char	*mp;

			for (mp = ep; mp > s; --mp)
			{
				if (strchr(" /", mp[-1]))
				{
					ep = mp;
					break;
				}
			}
		}

		/*
		 * ignore trailing blanks
		 */
		while (ep > s && ep[-1] == ' ')
			ep--;
		
		/*
		 * print the line
		 */
		if (first_line)
			sprintf(tmp, "%s: ", progname);
		else
			strcpy(tmp, "\t");
		tp = tmp + strlen(tmp);
		while (s < ep)
		{
			int	c;

			c = (unsigned char)*s++;
			if (isprint(c))
			{
				if (c == '\\')
					*tp++ = '\\';
				*tp++ = c;
			}
			else
			{
				char	*esc;

				esc = strchr(escapes, c);
				if (esc)
				{
					*tp++ = '\\';
					*tp++ = esc[1];
				}
				else
				{
					sprintf(tp, "\\%3.3o", c);
					tp += strlen(tp);
				}
			}
		}
		*tp++ = '\n';
		*tp = 0;
		fputs(tmp, stderr);
		if (ferror(stderr))
			break;

		/*
		 * skip leading spaces for subsequent lines
		 */
		while (*s == ' ')
			s++;
		first_line = 0;
	}
	if (fflush(stderr) || ferror(stderr))
	{
		/* don't print why, there is no point! */
		quit(1);
	}
	if (bomb_later)
	{
		errno = bomb_later;
		nfatal("(stdout)");
	}
}


static void double_jeopardy _((void));

static void
double_jeopardy()
{
	char	buffer[200];

	sprintf
	(
		buffer,
		"while attempting to construct an error message: %s (fatal)",
		strerror(errno)
	);
	wrap(buffer);
	quit(1);
}


static char *copy_a_string _((char *));

static char *
copy_a_string(s)
	char		*s;
{
	char		*cp;

	errno = 0;
	cp = malloc(strlen(s) + 1);
	if (!cp)
	{
		if (!errno)
			errno = ENOMEM;
		double_jeopardy();
	}
	strcpy(cp, s);
	return cp;
}


/*
 * NAME
 *	error - place a message on the error stream
 *
 * SYNOPSIS
 *	void error(char *s, ...);
 *
 * DESCRIPTION
 *	Error places a message on the error output stream.
 *	The first argument is a printf-like format string,
 *	optionally followed by other arguments.
 *	The message will be prefixed by the program name and a colon,
 *	and will be terminated with a newline, automatically.
 *
 * CAVEAT
 *	Things like "error(filename)" blow up if the filename
 *	contains a '%' character.
 */

void
error(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	char		*msg;

	sva_init(ap, s);
	msg = vmprintf(s, ap);
	va_end(ap);
	if (!msg)
		double_jeopardy();
	wrap(msg);
}


/*
 * NAME
 *	nerror - place a system fault message on the error stream
 *
 * SYNOPSIS
 *	void nerror(char *s, ...);
 *
 * DESCRIPTION
 *	Nerror places a message on the error output stream.
 *	The first argument is a printf-like format string,
 *	optionally followed by other arguments.
 *	The message will be prefixed by the program name and a colon,
 *	and will be terminated with a text description of the error
 *	indicated by the 'errno' global variable, automatically.
 *
 * CAVEAT
 *	Things like "nerror(filename)" blow up if the filename
 *	contains a '%' character.
 */

/*VARARGS1*/
void
nerror(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	int		n;
	char		*msg;

	n = errno;
	sva_init(ap, s);
	msg = vmprintf(s, ap);
	va_end(ap);
	if (!msg)
		double_jeopardy();
	msg = copy_a_string(msg);
	error("%s: %s", msg, strerror(n));
	free(msg);
}


/*
 * NAME
 *	nfatal - place a system fault message on the error stream and exit
 *
 * SYNOPSIS
 *	void nfatal(char *s, ...);
 *
 * DESCRIPTION
 *	Nfatal places a message on the error output stream and exits.
 *	The first argument is a printf-like format string,
 *	optionally followed by other arguments.
 *	The message will be prefixed by the program name and a colon,
 *	and will be terminated with a text description of the error
 *	indicated by the 'errno' global variable, automatically.
 *
 * CAVEAT
 *	Things like "nfatal(filename)" blow up if the filename
 *	contains a '%' character.
 *
 *	This function does NOT return.
 */

/*VARARGS1*/
void
nfatal(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	int		n;
	char		*msg;

	n = errno;
	sva_init(ap, s);
	msg = vmprintf(s, ap);
	va_end(ap);
	if (!msg)
		double_jeopardy();
	msg = copy_a_string(msg);
	error("%s: %s", msg, strerror(n));
	free(msg);
	quit(1);
}


/*
 * NAME
 *	fatal - place a message on the error stream and exit
 *
 * SYNOPSIS
 *	void fatal(char *s, ...);
 *
 * DESCRIPTION
 *	Fatal places a message on the error output stream and exits.
 *	The first argument is a printf-like format string,
 *	optionally followed by other arguments.
 *	The message will be prefixed by the program name and a colon,
 *	and will be terminated with a newline, automatically.
 *
 * CAVEAT
 *	Things like "error(filename)" blow up if the filename
 *	contains a '%' character.
 *
 *	This function does NOT return.
 */

void
fatal(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	char		*msg;

	sva_init(ap, s);
	msg = vmprintf(s, ap);
	va_end(ap);
	if (!msg)
		double_jeopardy();
	wrap(msg);
	quit(1);
}


/*
 * NAME
 *	assert - make an assertion
 *
 * SYNOPSIS
 *	void assert(int condition);
 *
 * DESCRIPTION
 *	Assert is a handy tool for a programmer to guarantee the internal
 *	consistency of their program.  If "-DDEBUG" is specified on
 *	the compiler's command line, then assert will generate code to verify
 *	the assertios made. If no DEBUG is defined, assertions will generate
 *	no code.
 *
 * CAVEAT
 *	If the assertion fails, a fatal diagnostic is issued.
 *
 *	The #define's which control the compilation may be found in "error.h".
 *
 */

#ifdef	DEBUG

void
cook_assert(c, s, file, line)
	int	c;
	char	*s;
	char	*file;
	int	line;
{
	if (c)
		return;
	error("%s: %d: assertion \"%s\" failed (bug)", file, line, s);
	abort();
}

#endif /* DEBUG */


/*
 * NAME
 *	quit_handler
 *
 * SYNOPSIS
 *	int quit_handler(quit_ty);
 *
 * DESCRIPTION
 *	The quit_handler function registers the function pointed to by func,
 *	to be called without arguments at normal program termination.
 */

static	quit_ty	quit_list[32];
static	int	quit_list_length;

void
quit_handler(f)
	quit_ty	f;
{
	assert(quit_list_length < SIZEOF(quit_list));
	quit_list[quit_list_length++] = f;
}


/*
 * NAME
 *	quit - leave program
 *
 * SYNOPSIS
 *	void quit(int status);
 *
 * DESCRIPTION
 *	The quit function causes normal program termination to occur.
 *
 *	First, all functions registered by the quit_handler function are
 *	called, in the reverse order of their registration.
 *
 *	Next, the program is terminated using the exit() function.
 *
 * CAVEAT
 *	The quit function never returns to its caller.
 */

void
quit(n)
	int	n;
{
	star_eoln();
	while (quit_list_length > 0)
		(*quit_list[--quit_list_length])();
	exit(n);
}


static int	star_flag;
static int	star_col;
static time_t	star_time;


void
star()
{
	time_t	now;

	if (!star_flag)
		return;
	time(&now);
	if (now < star_time)
		return;
	if (!star_col)
	{
		fprintf(stderr, "%s: ", progname);
		star_col = strlen(progname) + 2;
	}
	fputc('*', stderr);
	star_col++;
	if (star_col >= PAGE_WIDTH)
	{
		fputc('\n', stderr);
		star_col = 0;
	}
	fflush(stderr);
	star_time = now + 1;
}


void
star_eoln()
{
	time_t		now;

	if (!star_flag)
		return;
	if (star_col)
	{
		fputc('\n', stderr);
		star_col = 0;
		fflush(stderr);
	}
	time(&now);
	star_time = now + 1;
}


void
star_sync()
{
	time_t		when;

	if (!star_flag)
		return;
	assert(star_col == 0);
	if (star_col)
	{
		fputc('\n', stderr);
		star_col = 0;
		fflush(stderr);
	}
	time(&when);
	when += 5;
	if (when > star_time)
		star_time = when;
}


static void star_quit_handler _((void));

static void
star_quit_handler()
{
	star_flag = 0;
	if (star_col)
	{
		fputc('\n', stderr);
		star_col = 0;
	}
}


void
star_enable()
{
	if (star_flag)
		return;
	star_flag = 1;
	quit_handler(star_quit_handler);
}
