/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin execute function
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/execute.h>
#include <error.h>
#include <expr.h>
#include <option.h>
#include <os.h>


/*
 * NAME
 *	builtin_execute - execute a command
 *
 * SYNOPSIS
 *	int builtin_execute(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Exec is a built-in function of cook, described as follows:
 *	This function requires at least one argument, and
 *	executes the command given by the arguments.
 *
 * RETURNS
 *	If the executed command returns an error code the resulting value
 *	is "" (false), otherwise it is "1" (true).
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_execute(result, args)
	wlist		*result;
	wlist		*args;
{
	wlist		wl;
	int		j;
	string_ty	*s;
	int		silent;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires at least one argument",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	wl_zero(&wl);
	for (j = 1; j < args->wl_nwords; j++)
		wl_append(&wl, args->wl_word[j]);
	option_set(OPTION_SILENT, OPTION_LEVEL_EXECUTE, 1);
	silent = option_test(OPTION_SILENT);
	if (!silent)
	{
		string_ty	*s;

		s = wl2str(&wl, 0, wl.wl_nwords - 1, (char *)0);
		error("%s", s->str_text);
		str_free(s);
	}
	j = os_execute(&wl, (string_ty *)0, 0);
	if (!silent)
		star_sync();
	option_undo_level(OPTION_LEVEL_EXECUTE);
	wl_free(&wl);
	if (j < 0)
		return -1;
	s = (j ? str_false : str_true);
	wl_append(result, s);
	return 0;
}
