/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin text functions
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <ac/stdlib.h>
#include <ac/string.h>

#include <builtin/text.h>
#include <error.h>
#include <expr.h>
#include <mem.h>


/*
 * NAME
 *	builtin_upcase - upcase strings
 *
 * SYNOPSIS
 *	int builtin_upcase(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires one or more arguments,
 *	which will bu upcased.
 *
 * RETURNS
 *	It returns the arguments upcased.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_upcase(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty *s;

		s = str_upcase(args->wl_word[j]);
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_downcase - downcase strings
 *
 * SYNOPSIS
 *	int builtin_downcase(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of cook, described as follows:
 *	This function requires one or more arguments,
 *	which will bu downcased.
 *
 * RETURNS
 *	It returns the arguments downcased.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_downcase(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty	*s;

		s = str_downcase(args->wl_word[j]);
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_prepost - add prefix and suffix
 *
 * SYNOPSIS
 *	int builtin_prepost(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Prepost is a built-in function of cook, described as follows:
 *	This function must have at least two arguments.
 *	The first argument is a prefix and the second argument is a suffix.
 *
 * RETURNS
 *	The resulting word list is the third and later arguments each given
 *	the prefix and suffix as defined by the first and second arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_prepost(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 3)
	{
		expr_error
		(
			"%s: requires at least two arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 3; j < args->wl_nwords; j++)
	{
		string_ty	*s;

		s =
			str_cat_three
			(
				args->wl_word[1],
				args->wl_word[j],
				args->wl_word[2]
			);
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_head - head of a wordlist
 *
 * SYNOPSIS
 *	int builtin_fromto(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Head is a built-in function of cook, described as follows:
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	The wordlist returned is empty if there were no arguemnts,
 *	or the first argument if there were arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_head(result, args)
	wlist		*result;
	wlist		*args;
{
	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords >= 2)
		wl_append(result, args->wl_word[1]);
	return 0;
}


/*
 * NAME
 *	builtin_tail - tail of a wordlist
 *
 * SYNOPSIS
 *	int builtin_tail(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Tail is a built-in function of cook, described as follows:
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	The word list returned will be empty if
 *	there is less than two arguemnts,
 *	otherwise it will consist of the second and later arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_tail(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	for (j = 2; j < args->wl_nwords; j++)
		wl_append(result, args->wl_word[j]);
	return 0;
}


/*
 * NAME
 *	builtin_catenate - catenate a wordlist
 *
 * SYNOPSIS
 *	int builtin_catenate(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Catenate is a built-in function of cook, described as follows:
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	A word list containg zero words if there were no arguments,
 *	or a single word which is the catenation of the arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_catenate(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	static char	*tmp;
	static size_t	tmplen;
	size_t		length;
	char		*pos;
	string_ty	*s;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
		return 0;
	if (args->wl_nwords == 2)
	{
		wl_append(result, args->wl_word[1]);
		return 0;
	}

	length = 0;
	for (j = 1; j < args->wl_nwords; j++)
		length += args->wl_word[j]->str_length;
	if (!tmp)
	{
		tmplen = length;
		if (tmplen < 16)
			tmplen = 16;
		tmp = mem_alloc(tmplen);
	}
	else
	{
		if (tmplen < length)
		{
			tmplen = length;
			tmp = mem_change_size(tmp, tmplen);
		}
	}
	pos = tmp;
	for (j = 1; j < args->wl_nwords; j++)
	{
		s = args->wl_word[j];
		memcpy(pos, s->str_text, s->str_length);
		pos += s->str_length;
	}
	s = str_n_from_c(tmp, length);
	wl_append(result, s);
	str_free(s);
	return 0;
}


/*
 * NAME
 *	builtin_count - length of a word list
 *
 * SYNOPSIS
 *	int builtin_count(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Count is a built-in function of cook, described as follows:
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	A word list containg a single word containing the (decimal)
 *	length of the argument list.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_count(result, args)
	wlist		*result;
	wlist		*args;
{
	string_ty	*s;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	s = str_format("%ld", args->wl_nwords - 1);
	wl_append(result, s);
	str_free(s);
	return 0;
}


/*
 * NAME
 *	builtin_quote - quote the arguments
 *
 * SYNOPSIS
 *	int builtin_quote(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	The quote function is a built-in of cook, described as follows:
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing the values of the arguments
 *	surrounded by double quotes.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_quote(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	static char	*tmp;
	static size_t	tmp_len;
	size_t		len;
	char		*cp1;
	char		*cp2;
	char		*cp3;
	string_ty	*s;
	static char	special[] = "\bb\ff\nn\rr\tt";

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	for (j = 1; j < args->wl_nwords; ++j)
	{
		len = 2;
		for (cp1 = args->wl_word[j]->str_text; *cp1; ++cp1)
		{
			if (*cp1 < ' ' || *cp1 > '~')
			{
				if (strchr(special, *cp1))
					len += 2;
				else
					len += 4;
			}
			else
			{
				if (strchr("\"\\", *cp1))
					++len;
				++len;
			}
		}
		if (len > tmp_len)
		{
			tmp_len = len;
			tmp = mem_change_size(tmp, tmp_len);
		}
		cp2 = tmp;
		*cp2++ = '"';
		for (cp1 = args->wl_word[j]->str_text; *cp1; ++cp1)
		{
			if (*cp1 < ' ' || *cp1 > '~')
			{
				cp3 = strchr(special, *cp1);
				if (cp3)
				{
					*cp2++ = '\\';
					*cp2++= cp3[1];
				}
				else
				{
					*cp2++ = '\\';
					*cp2++ = '0' + ((*cp1 >> 6) & 3);
					*cp2++ = '0' + ((*cp1 >> 3) & 7);
					*cp2++ = '0' + (*cp1 & 7);
				}
			}
			else
			{
				if (strchr("\"\\", *cp1))
					*cp2++ = '\\';
				*cp2++ = *cp1;
			}
		}
		*cp2 = '"';
		s = str_n_from_c(tmp, len);
		wl_append(result, s);
		str_free(s);
	}
	return 0;
}


/*
 * NAME
 *	builtin_sort - sort the arguments
 *
 * SYNOPSIS
 *	int builtin_sort(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	The builtin_sort function is a built-in of cook, described as follows:
 *	sorts the arguments lexicagraphically.
 *	This function requires zero or more arguments.
 *
 * RETURNS
 *	A sorted word list.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	string_ty	*a;
	string_ty	*b;

	a = *(string_ty **)va;
	b = *(string_ty **)vb;
	return strcmp(a->str_text, b->str_text);
}


int
builtin_sort(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	int		start;

	assert(result);
	assert(args);
	switch (args->wl_nwords)
	{
	case 0:
		assert(0);

	case 1:
		return 0;

	case 2:
		wl_append(result, args->wl_word[1]);
		return 0;
	}
	start = result->wl_nwords;
	for (j = 1; j < args->wl_nwords; ++j)
		wl_append(result, args->wl_word[j]);
	qsort
	(
		&result->wl_word[start],
		args->wl_nwords - 1,
		sizeof(result->wl_word[0]),
		cmp
	);
	return 0;
}
