/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to cook targets
 *
 * the kitchen
 *
 * This file contains the part of cook that actually decides which
 * recipes will be cooked.
 *
 * When cook has a target, cook performs the following actions in the order
 * given:
 *
 * 1.	Cook scans through the instanciated prerequisite recipes. All
 *	prerequisite recipes with the target in their target list are used.
 *
 *	If the recipe is used, any prerequisite files are recursively
 *	cooked, then if any of the prerequisite files were out of date, then
 *	all other explicit or implicit recipes with the same target will be
 *	deemed to be out of date.
 *
 * 2.	Cook then scans through the instanciated explicit recipes. All
 *	explicit recipes with the target in their target list are used.
 *
 *	If the recipe is used, any prerequisite files are recursively
 *	cooked, then if any prerequisites were out of date (including those
 *	of prerequisite recipes) then the actions bound to this recipe will
 *	be evaluated.
 *
 *	If there are no ingredients, then it is not out-of-date.  The
 *	body will be performed if (a) the target does not yet exist, or
 *	(b) the "force" flag is set, usually in the "set" clause of
 *	the recipe.
 *
 * 3.	If the target was not the subject of any explicit recipe, cook then
 *	scans the instanciated implicit recipes. Only first implicit recipe
 *	for which cook knows how to cook will be used.
 *
 *	Implicit recipe targets and prerequisites may contain a wilcard
 *	character (%), which is why they are implicit. If more than one
 *	wildcard character appears, only the last is considered the wilcard
 *	charcater.
 *
 *	If an implicit recipe is used, when expressions are evaluaded into
 *	word lists, any word containing the wildcard charcater (%) will be
 *	expanded out by the current wildcard expansion.
 *
 * 4.	If the target is not the subject of any prerequisite or explicit
 *	recipe, and no implicit recipes can be applied, then two things can
 *	happen.
 *		a. If the file exists, then it is up to date, or
 *		b. If the file does not exist then cook doesn't know how.
 *
 * If a command in the actions bound to any recipe fail, cook will not
 * evaluate those actions any further, and will not evaluate the actions
 * of any recipe for which the target of the failed actions was a
 * prerequisite.
 *
 * Cook will trap recursive looping of targets. If a recursion loop is
 * detected, then
 *	1. If the file exists, the it is up to date, or
 *	2. If the file does not exist then cook doesn't know how.
 */

#include <ac/stddef.h>
#include <stdio.h>
#include <ac/time.h>
#include <signal.h>

#include <cook.h>
#include <error.h>
#include <id.h>
#include <main.h>
#include <match.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <symtab.h>
#include <trace.h>


/*
 *  Cooker results.
 *  The better the result, the higher then number.
 */
enum cooker_result_ty
{
	COOK_DONTKNOW,
	COOK_ERROR,
	COOK_DONE,
	COOK_DONE_UPTODATE,
	COOK_UPTODATE,
	COOK_BACKTRACK
};
typedef enum cooker_result_ty cooker_result_ty;


rlist		explicit;	/* the explicit recipes */
rlist		implicit;	/* the implicit recipes */
static int	max_tag;
int		desist;
static symtab_ty *already;
static wlist	cook_auto_list;


static cooker_result_ty *already_copy _((cooker_result_ty));

static cooker_result_ty *
already_copy(n)
	cooker_result_ty n;
{
	cooker_result_ty *result;

	result = mem_alloc(sizeof(cooker_result_ty));
	*result = n;
	return result;
}


static void already_assign _((string_ty *, cooker_result_ty));

static void
already_assign(path, level)
	string_ty	*path;
	cooker_result_ty level;
{
	if (!already)
	{
		already = symtab_alloc(100);
		already->reap = mem_free;
	}
	symtab_assign(already, path, already_copy(level));
}


static int already_search _((string_ty *, cooker_result_ty *));

static int
already_search(path, result)
	string_ty	*path;
	cooker_result_ty *result;
{
	cooker_result_ty *data;

	if (!already)
		return 0;
	data = symtab_query(already, path);
	if (!data)
		return 0;
	*result = *data;
	return 1;
}


static void already_delete _((string_ty *));

static void
already_delete(path)
	string_ty	*path;
{
	if (!already)
		return;
	symtab_delete(already, path);
}


/*
 * NAME
 *	interrupt - handle interrupts by signals
 *
 * SYNOPSIS
 *	void interrupt(int);
 *
 * DESCRIPTION
 *	The interrupt function is used to set the appropriate flags when an
 *	interrupt occurs.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Don't use too extravagantly, because it just returns!
 */

static RETSIGTYPE interrupt _((int));

static RETSIGTYPE
interrupt(n)
	int		n;
{
	desist++;
	error("interrupted by %s", signal_name(n));
	option_set_errors();
	signal(n, interrupt);
}


/*
 * NAME
 *	recipe_tag - make next one
 *
 * SYNOPSIS
 *	int recipe_tag(void);
 *
 * DESCRIPTION
 *	The recipe_tag function is used to generate successive recipe tages.
 *	Recipe tags are used during the -trace output to indicate which recipe
 *	is in use.  Recipes are not necessarily instanciated in the order in
 *	which they appear in the source file, hence the tagging.
 *
 * RETURNS
 *	int; unique tag
 */

int
recipe_tag()
{
	return ++max_tag;
}


/*
 * NAME
 *	cook_status_name - give name of status
 *
 * SYNOPSIS
 *	char *cook_status_name(int);
 *
 * DESCRIPTION
 *	The cook_status_name function is used to map a cook status name into a
 *	more informative string equivalent of that status.
 *
 * RETURNS
 *	char *; pointer to C string.
 *
 * CAVEAT
 *	Do not pass the returned string to free().
 *	Do not alter the contents of the string pointed to by the return value.
 *	This function is obly available when synbol DEBUG is defined.
 *	Assumes that the argument will be a valid status value.
 */

#ifdef DEBUG

static char *cook_status_name _((cooker_result_ty));

static char *
cook_status_name(n)
	cooker_result_ty n;
{
	switch (n)
	{
	case COOK_DONTKNOW:
		return "COOK_DONTKNOW";

	case COOK_ERROR:
		return "COOK_ERROR";

	case COOK_DONE:
		return "COOK_DONE";

	case COOK_DONE_UPTODATE:
		return "COOK_DONE_UPTODATE";

	case COOK_UPTODATE:
		return "COOK_UPTODATE";

	case COOK_BACKTRACK:
		return "COOK_BACKTRACK";
	}
	return "unknown (bug)";
}

#endif

static cooker_result_ty cooker _((string_ty *, int, int)); /* forward */


/*
 * NAME
 *	cook_search_list
 *
 * SYNOPSIS
 *	cook_search_list(wlist *slp);
 *
 * DESCRIPTION
 *	The cook_search_list function is used to get the search list
 *	from the "search_list" variable.
 *
 *	Defaulting and clean-up are done here, also.
 *	If absent, defaults to ".".
 *	If the first element is not "." then it is inserted.
 *
 * ARGUMENTS
 *	slp - where to put the result
 */

static void cook_search_list _((wlist *slp));

static void
cook_search_list(slp)
	wlist		*slp;
{
	string_ty	*s;

	/*
	 * make sure the variable exists
	 */
	trace(("cook_search_list()\n{\n"/*}*/));
	if (!id_search(id_search_list, slp))
	{
		wl_zero(slp);
		s = str_from_c(".");
		wl_append(slp, s);
		str_free(s);
		id_assign(id_search_list, slp);
	}

	/*
	 * make sure the search list isn't empty
	 */
	if (!slp->wl_nwords)
	{
		s = str_from_c(".");
		wl_append(slp, s);
		str_free(s);
		id_assign(id_search_list, slp);
	}

	/*
	 * make sure the search list has "." as the first element
	 */
	if
	(
		slp->wl_word[0]->str_text[0] != '.'
	||
		slp->wl_word[0]->str_text[1]
	)
	{
		s = str_from_c(".");
		wl_prepend(slp, s);
		str_free(s);
		id_assign(id_search_list, slp);
	}
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	cook_mtime
 *
 * SYNOPSIS
 *	time_t cook_mtime(string_ty *path, long *depth_p);
 *
 * DESCRIPTION
 *	The cook_mtime function is used to scan the search path for
 *	a file to determine the last-modified time of the file.
 *
 * ARGUMENTS
 *	path	- file to get the mtime for
 *	depth_p	- where to put the depth
 *
 * RETURNS
 *	long; -1 on error, 0 if no such file, >0 for time
 *
 * CAVEAT
 *	The user must design recipes using the [resolve] function.
 */

time_t
cook_mtime_oldest(path, depth_p)
	string_ty	*path;
	long		*depth_p;
{
	time_t		result;

	trace(("cook_mtime_oldest(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (path->str_text[0] == '/')
		result = os_mtime_oldest(path);
	else
	{
		wlist		sl;
		long		j;

		result = 0;
		cook_search_list(&sl);
		for (j = 0; j < sl.wl_nwords; ++j)
		{
			string_ty	*s1;
			string_ty	*s2;
			time_t		t;

			s1 = sl.wl_word[j];
			if (s1->str_text[0] == '.' && !s1->str_text[1])
				s2 = str_copy(path);
			else
				s2 = str_format("%S/%S", s1, path);
			t = os_mtime_oldest(s2);
			str_free(s2);
			if (t != 0)
			{
				result = t;
				*depth_p = j;
				break;
			}
		}
		wl_free(&sl);
	}
	trace(("return %ld (%d);\n", (long)result, *depth_p));
	trace((/*{*/"}\n"));
	return result;
}


time_t
cook_mtime_newest(path, depth_p)
	string_ty	*path;
	long		*depth_p;
{
	time_t		result;

	trace(("cook_mtime_newest(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (path->str_text[0] == '/')
		result = os_mtime_newest(path);
	else
	{
		wlist		sl;
		long		j;

		result = 0;
		cook_search_list(&sl);
		for (j = 0; j < sl.wl_nwords; ++j)
		{
			string_ty	*s1;
			string_ty	*s2;
			time_t		t;

			s1 = sl.wl_word[j];
			if (s1->str_text[0] == '.' && !s1->str_text[1])
				s2 = str_copy(path);
			else
				s2 = str_format("%S/%S", s1, path);
			t = os_mtime_newest(s2);
			str_free(s2);
			if (t != 0)
			{
				result = t;
				*depth_p = j;
				break;
			}
		}
		wl_free(&sl);
	}
	trace(("return %ld (%d);\n", (long)result, *depth_p));
	trace((/*{*/"}\n"));
	return result;
}


/*
 * NAME
 *	cook_mtime_resolve
 *
 * SYNOPSIS
 *	int cook_mtime_resolve(wlist *output, wlist *input);
 *
 * DESCRIPTION
 *	The cook_mtime_resolve function is used to
 *	resolve the name used for a file in the search list.
 *
 *	It implements the "resolve" built-in function.
 *
 * ARGUMENTS
 *	input - the function arguments (0 is the func name)
 *	output - where to put the results
 *
 * RETURNS
 *	int; 0 on success, -1 on error
 *
 * CAVEAT
 *	The user must design rules using the [resolve] function.
 */

int
cook_mtime_resolve(output, input, start)
	wlist		*output;
	wlist		*input;
	int		start;
{
	int		result;
	long		j;
	wlist		sl;

	trace(("cook_mtime_resolve(input = %08lX, output = %08lX)\n{\n"/*}*/,
		input, output));
	cook_search_list(&sl);
	result = 0;
	for (j = start; j < input->wl_nwords; ++j)
	{
		string_ty	*arg;

		arg = input->wl_word[j];
		if (arg->str_text[0] == '/')
			wl_append(output, arg);
		else
		{
			int		done;
			long		k;

			done = 0;
			for (k = 0; k < sl.wl_nwords; ++k)
			{
				string_ty	*s1;
				string_ty	*s2;
				time_t		t;

				s1 = sl.wl_word[k];
				if (s1->str_text[0] == '.' && !s1->str_text[1])
					s2 = str_copy(arg);
				else
					s2 = str_format("%S/%S", s1, arg);
				t = os_mtime_newest(s2);
				if (t < 0)
				{
					result = -1;
					str_free(s2);
					break;
				}
				if (t > 0)
				{
					wl_append(output, s2);
					str_free(s2);
					done = 1;
					break;
				}
				str_free(s2);
			}
			if (!done)
				wl_append(output, arg);
		}
		if (result < 0)
			break;
	}
	wl_free(&sl);
	trace(("return %d;\n", result));
	trace((/*{*/"}\n"));
	return result;
}


static void strip_dot _((string_ty **));

static void
strip_dot(p)
	string_ty	**p;
{
	char		*cp;
	size_t		len;
	string_ty	*tmp;

	cp = (*p)->str_text;
	len = (*p)->str_length;
	while (len >= 3 && cp[0] == '.' && cp[1] == '/')
	{
		cp += 2;
		len -= 2;
	}
	if (len == (*p)->str_length)
		return;

	tmp = str_n_from_c(cp, len);
	str_free(*p);
	*p = tmp;
}


/*
 * NAME
 *	chef - the bit that does the cooking
 *
 * SYNOPSIS
 *	int chef(string_ty *target, int dkwinge, recipe *rp,
 *		time_ty age, int forced, int build);
 *
 * DESCRIPTION
 *	The chef function is used to evaluate a recipe
 *	which has been determined by cooker() to be applicable.
 *
 * RETURNS
 *	The chef function returns an int; one of the cook status values.
 *
 * CAVEAT
 *	The `target' variable is assumed to have already been set.
 *	The match pattern is assumed to have been pushed already.
 *	The recipe flags apply to expression evaluations for ingredients, too.
 */

static cooker_result_ty chef _((string_ty *, int, recipe *, time_t, int, int,
	time_t *, long *));

static cooker_result_ty
chef(target, dkwinge, rp, age, forced, build, worst_age_p, worst_depth_p)
	string_ty	*target;
	int		dkwinge;
	recipe		*rp;
	time_t		age;
	int		forced;
	int		build;
	time_t		*worst_age_p;
	long		*worst_depth_p;
{
	int		j;
	int		k;
	cooker_result_ty best;
	char		*recipe_type_name;
	wlist 		friends;
	wlist		need;
	wlist		younger;
	match_ty	*field;
	time_t		mtime;
	time_t		*friends_mtime;
	int		fp;
	long		initial_depth;
	long		depth;

	trace(("chef(target = %08lX, dkwinge = %d, rp = %08lX, age = %ld, \
forced = %d, build = %d)\n{\n"/*}*/, target, dkwinge, rp, age, forced, build));
	trace_string(target->str_text);
	friends_mtime = 0;
	fp = option_test(OPTION_FINGERPRINT);
	wl_zero(&need);
	wl_zero(&younger);
	wl_zero(&friends);
	field = match_top();
	if (field)
	{
		wl_reconstruct(&friends, &rp->r_target, field);
		recipe_type_name = "implicit";
	}
	else
	{
		wl_copy(&friends, &rp->r_target);
		recipe_type_name = "explicit";
	}
	id_assign_push(id_friend, &friends);

	/*
	 * age should be set to the worst case of all the targets
	 *
	 * The depth of the target search should be less than or equal
	 * to the depth of the worst (shallowest) ingredients search.
	 * This is to guarantee that when ingredients change they
	 * result in targets shallower in the path being updated.
	 */
	initial_depth = *worst_depth_p;
	if (fp)
		friends_mtime = mem_alloc(friends.wl_nwords * sizeof(time_t));
	for (j = 0; j < friends.wl_nwords; ++j)
	{
		string_ty	*s;
		time_t		age2;

		s = friends.wl_word[j];
		if (fp)
		{
			depth = *worst_depth_p;
			age2 = cook_mtime_oldest(s, &depth);
			if (age2 < 0)
			{
				best = COOK_ERROR;
				goto ret;
			}
			friends_mtime[j] = age2;
		}
		depth = *worst_depth_p;
		age2 = cook_mtime_newest(s, &depth);
		if (age2 < 0)
		{
			best = COOK_ERROR;
			goto ret;
		}
		if (depth > *worst_depth_p)
		{
			age2 = 0;
			if (option_test(OPTION_TRACE))
			{
				error
				(
    "\"%s\" is out of date because an ingredients recipe was shallower (trace)",
					target->str_text
				);
			}
			forced = 1;
		}
		if (age2 < age)
			age = age2;
	}

	/*
	 * Flags apply to the precondition and to the ingredients evaluation.
	 * That is why the grammar puts them first.
	 */
	cook_flags(rp->r_flags, OPTION_LEVEL_RECIPE);

	/*
	 * wander through the ingredients, seeing if we know how to cook them
	 *	the first set is used to determine if to use this recipe
	 *	the second set must be cookable
	 */
	if (*worst_age_p >= age)
	{
		if (option_test(OPTION_TRACE))
		{
			error
			(
      "\"%s\" is out of date because an ingredients recipe was younger (trace)",
				target->str_text
			);
		}
		forced = 1;
	}
	if (option_test(OPTION_FORCE))
	{
		if (option_test(OPTION_TRACE))
		{
			error
			(
 "\"%s\" is out of date because the recipe has the \"forced\" flag set (trace)",
				target->str_text
			);
		}
		forced = 1;
	}
	best = COOK_UPTODATE;
	trace(("check first ingredients\n"));
	for (j = 0; j < rp->r_need.el_nexprs; ++j)
	{
		wlist	wl;

		wl_zero(&wl);
		if (expr_eval(&wl, rp->r_need.el_expr[j]))
		{
			wl_free(&wl);
			best = COOK_ERROR;
			goto ret;
		}
		for (k = 0; k < wl.wl_nwords; ++k)
		{
			string_ty	*target2;
			cooker_result_ty best2;

			if (option_test(OPTION_STRIP_DOT))
				strip_dot(&wl.wl_word[k]);
			target2 = wl.wl_word[k];
			if (!os_legal_path(target2))
			{
				best = COOK_BACKTRACK;
				wl_free(&wl);
				goto ret;
			}
			wl_append_unique(&need, target2);
			assert(rp->r_tag <= max_tag);
			if (option_test(OPTION_TRACE))
			{
				error
				(
				     "\"%s\" %s \"%s\" by %s recipe %d (trace)",
					target->str_text,
					(dkwinge ? "requires" : "may require"),
					target2->str_text,
					recipe_type_name,
					rp->r_tag
				);
			}
			if (field)
			{
				match_ty	*inhibit;

				/*
				 * if any of the ingredients of an
				 * implicit recipe match the targets of
				 * that same recipe, inhibit the
				 * recursive application the recipe.
				 */
				inhibit = wl_match(&rp->r_target, target2);
				if (inhibit)
				{
					rp->inhibit = 1;
					match_free(inhibit);
				}
			}
			option_undo_level(OPTION_LEVEL_RECIPE);
			best2 = cooker(target2, dkwinge, build);
			cook_flags(rp->r_flags, OPTION_LEVEL_RECIPE);
			rp->inhibit = 0;
			if (best2 == COOK_DONTKNOW)
				best2 = COOK_ERROR;
			if (best2 < best)
				best = best2;
			switch (best2)
			{
			case COOK_BACKTRACK:
				best = best2;
				wl_free(&wl);
				goto ret;

			default:
				if (!option_test(OPTION_PERSEVERE))
				{
					wl_free(&wl);
					goto ret;
				}
				break;

			case COOK_UPTODATE:
			case COOK_DONE_UPTODATE:
				depth = *worst_depth_p;
				mtime = cook_mtime_oldest(target2, &depth);
				if (mtime < 0)
				{
					wl_free(&wl);
					best = COOK_ERROR;
					goto ret;
				}
				if (mtime > *worst_age_p)
					*worst_age_p = mtime;
				if (depth < *worst_depth_p)
					*worst_depth_p = depth;
				if (mtime < age)
					break;
				if (option_test(OPTION_TRACE))
				{
					error
					(
		      "\"%s\" is out of date because \"%s\" is younger (trace)",
						target->str_text,
						target2->str_text
					);
				}
				forced = 1;
				wl_append_unique(&younger, target2);
				break;

			case COOK_DONE:
				depth = *worst_depth_p;
				mtime = cook_mtime_oldest(target2, &depth);
				if (mtime < 0)
				{
					wl_free(&wl);
					best = COOK_ERROR;
					goto ret;
				}
				if (mtime > *worst_age_p)
					*worst_age_p = mtime;
				if (depth < *worst_depth_p)
					*worst_depth_p = depth;
				if (option_test(OPTION_TRACE))
				{
					error
					(
   "\"%s\" is out of date because \"%s\" was cooked and is now younger (trace)",
						target->str_text,
						target2->str_text
					);
				}
				forced = 1;
				wl_append_unique(&younger, target2);
				break;
			}
		}
		wl_free(&wl);
	}
	trace(("check second ingredients\n"));
	for (j = 0; j < rp->r_need2.el_nexprs; ++j)
	{
		wlist		wl;

		wl_zero(&wl);
		if (expr_eval(&wl, rp->r_need2.el_expr[j]))
		{
			wl_free(&wl);
			best = COOK_ERROR;
			goto ret;
		}
		for (k = 0; k < wl.wl_nwords; ++k)
		{
			string_ty	*target2;
			cooker_result_ty best2;

			if (option_test(OPTION_STRIP_DOT))
				strip_dot(&wl.wl_word[j]);
			target2 = wl.wl_word[k];
			if (!os_legal_path(target2))
			{
				best = COOK_BACKTRACK;
				wl_free(&wl);
				goto ret;
			}
			wl_append_unique(&need, target2);
			if (option_test(OPTION_TRACE))
			{
				error
				(
			       "\"%s\" requires \"%s\" by %s recipe %d (trace)",
					target->str_text,
					target2->str_text,
					recipe_type_name,
					rp->r_tag
				);
			}
			option_undo_level(OPTION_LEVEL_RECIPE);
			best2 = cooker(target2, 1, build);
			cook_flags(rp->r_flags, OPTION_LEVEL_RECIPE);
			if (best2 == COOK_BACKTRACK || best2 == COOK_DONTKNOW)
				best2 = COOK_ERROR;
			if (best2 < best)
				best = best2;
			switch (best2)
			{
			default:
				if (!option_test(OPTION_PERSEVERE))
				{
					wl_free(&wl);
					goto ret;
				}
				break;

			case COOK_UPTODATE:
			case COOK_DONE_UPTODATE:
				depth = *worst_depth_p;
				mtime = cook_mtime_oldest(target2, &depth);
				if (mtime < 0)
				{
					wl_free(&wl);
					best = COOK_ERROR;
					goto ret;
				}
				if (mtime > *worst_age_p)
					*worst_age_p = mtime;
				if (depth < *worst_depth_p)
					*worst_depth_p = depth;
				if (mtime < age)
					break;
				if (option_test(OPTION_TRACE))
				{
					error
					(
		      "\"%s\" is out of date because \"%s\" is younger (trace)",
						target->str_text,
						target2->str_text
					);
				}
				forced = 1;
				wl_append_unique(&younger, target2);
				break;

			case COOK_DONE:
				depth = *worst_depth_p;
				mtime = cook_mtime_oldest(target2, &depth);
				if (mtime < 0)
				{
					wl_free(&wl);
					best = COOK_ERROR;
					goto ret;
				}
				if (mtime > *worst_age_p)
					*worst_age_p = mtime;
				if (depth < *worst_depth_p)
					*worst_depth_p = depth;
				if (option_test(OPTION_TRACE))
				{
					error
					(
   "\"%s\" is out of date because \"%s\" was cooked and is now younger (trace)",
						target->str_text,
						target2->str_text
					);
				}
				forced = 1;
				wl_append_unique(&younger, target2);
				break;
			}
		}
		wl_free(&wl);
	}

	/*
	 * see if this recipe applies
	 */
	trace(("check precondition\n"));
	if (rp->r_precondition)
	{
		int		result;

		id_assign_push(id_need, &need);
		id_assign_push(id_younger, &younger);
		result = expr_eval_condition(rp->r_precondition);
		id_delete(id_need);
		id_delete(id_younger);
		switch (result)
		{
		case -1:
			best = COOK_ERROR;
			goto ret;

		case 0:
			if (option_test(OPTION_TRACE))
			{
				error
				(
			 "\"%s\" precondition for %s recipe %d rejects (trace)",
					target->str_text,
					recipe_type_name,
					rp->r_tag
				);
			}
			best = COOK_BACKTRACK;
			goto ret;
		}
	}

	/*
	 * if the target depth became lower,
	 * rescan the targets to see if this implies out-of-date
	 *
	 * The depth of the target search should be less than or equal
	 * to the depth of the worst (shallowest) ingredients search.
	 * This is to guarantee that when ingredients change they
	 * result in targets shallower in the path being updated.
	 */
	if (initial_depth != *worst_depth_p && !forced)
	{
		for (j = 0; j < friends.wl_nwords; ++j)
		{
			string_ty	*s;
			time_t		age2;

			s = friends.wl_word[j];
			depth = *worst_depth_p;
			age2 = cook_mtime_newest(s, &depth);
			if (age2 < 0)
			{
				best = COOK_ERROR;
				goto ret;
			}
			if (depth > *worst_depth_p)
			{
				if (option_test(OPTION_TRACE))
				{
					error
					(
	     "\"%s\" is out of date because an ingredient is shallower (trace)",
						s->str_text
					);
				}
				forced = 1;
				break;
			}
		}
	}

	/*
	 * See if we need to perform the actions attached to this recipe.
	 */
	if (forced && rp->r_action)
	{
		trace(("do recipe body\n"));
		if (build)
		{
			if (option_test(OPTION_TOUCH))
			{
				for (k = 0; k < rp->r_target.wl_nwords; k++)
				{
					if (!option_test(OPTION_SILENT))
					{
						error
						(
							"touch %s",
					       rp->r_target.wl_word[k]->str_text
						);
					}
					if (os_touch(rp->r_target.wl_word[k]))
						best = COOK_ERROR;
					else
						best = COOK_DONE;
				}
			}
			else
			{
				int		result;

				id_assign_push(id_need, &need);
				id_assign_push(id_younger, &younger);
				result = stmt_eval(rp->r_action);
				id_delete(id_need);
				id_delete(id_younger);
				switch (result)
				{
				case STMT_BACKTRACK:
					best = COOK_DONTKNOW;
					break;

				case STMT_OK:
					best = COOK_DONE;
					break;

				case STMT_ERROR:
					if (option_test(OPTION_ERROK))
						best = COOK_DONE;
					else
						best = COOK_ERROR;
					break;

				default:
					best = COOK_ERROR;
					break;
				}
				if
				(
					best == COOK_ERROR
				&&
					!option_test(OPTION_PRECIOUS)
				)
				{
					for (k = 0; k < friends.wl_nwords; k++)
						os_delete(friends.wl_word[k]);
				}
			}
		}
		else
			best = COOK_DONE;
	}

	/*
	 * This recipe is being used, so
	 * perform its 'use' action.
	 *
	 * Ignore the 'touch' option,
	 * ignore the 'errok' option,
	 * don't delete files on errors.
	 */
	if (rp->r_use_action && build)
	{
		int		result;

		trace(("perform ``use'' clause\n"));
		id_assign_push(id_need, &need);
		id_assign_push(id_younger, &younger);
		result = stmt_eval(rp->r_use_action);
		id_delete(id_need);
		id_delete(id_younger);
		if (result != STMT_OK)
			best = STMT_ERROR;
	}

ret:
	option_undo_level(OPTION_LEVEL_RECIPE);
	wl_free(&need);
	wl_free(&younger);
	if (best == COOK_DONE && rp->r_action && build)
	{
		mtime = *worst_age_p + 1;
		if (fp)
		{
			/*
			 * Update the times, and see if the pringerprint
			 * changed.  If the fingerprint did not change
			 * on any target, use the DONE_UPTODATE result,
			 * not DONE.
			 */
			for (j = 0; j < friends.wl_nwords; ++j)
			{
				string_ty	*s;
				time_t		t;

				s = friends.wl_word[j];
				os_clear_stat(s);
				if (os_mtime_adjust(s, mtime))
				{
					best = COOK_ERROR;
					continue;
				}
				if (friends_mtime[j] <= 0)
				{
					already_assign(s, COOK_DONE);
					continue;
				}
				t = os_mtime_oldest(s);
				if (t < 0)
				{
					best = COOK_ERROR;
					continue;
				}
				if (t != friends_mtime[j])
					already_assign(s, COOK_DONE);
				else
					already_assign(s, COOK_DONE_UPTODATE);
			}
			if (best != COOK_ERROR)
				already_search(target, &best);
		}
		else
		{
			/*
			 * update the times if it worked
			 * and if it was not an ingredients recipe
			 */
			for (j = 0; j < friends.wl_nwords; j++)
			{
				string_ty	*s;

				s = friends.wl_word[j];
				if (os_mtime_adjust(s, mtime))
					best = COOK_ERROR;
				else
					already_assign(s, COOK_DONE);
			}
		}
	}
	if
	(
		rp->r_action
	&&
		best != COOK_BACKTRACK
	&&
		best != COOK_DONE
	)
	{
		for (j = 0; j < friends.wl_nwords; j++)
			already_assign(friends.wl_word[j], best);
	}
	if (best == COOK_UPTODATE)
		star();
	wl_free(&friends);
	id_delete(id_friend);
	if (friends_mtime)
		mem_free(friends_mtime);
	trace(("return %s;\n", cook_status_name(best)));
	trace((/*{*/"}\n"));
	return best;
}


/*
 * NAME
 *	cooker - cook a target
 *
 * SYNOPSIS
 *	int cooker(string_ty *target, int dkwinge, int build);
 *
 * DESCRIPTION
 *	The cooker function is used to find and perform the actions required
 *	to bring the given target up-to-date.
 *
 * RETURNS
 *	The cook function returns int; one of the status return values
 *
 *	COOK_BACKTRACK	the given target was not possble, try something else
 *	COOK_UPTODATE	the given target required no action
 *	COOK_DONE_UPTODATE
 *			the given target required some action,
 *			but the results are identical to previous
 *	COOK_DONE	some action was performed to bring it up to date
 *	COOK_ERROR	an error was found when performing some action
 *	COOK_DONTKNOW	don't know how to cook the given target
  */

static cooker_result_ty cooker _((string_ty *, int, int));

static cooker_result_ty
cooker(target, dkwinge, build)
	string_ty	*target;	/* file which we want to cook */
	int		dkwinge;	/* complain if don't know how to */
	int		build;		/* build if true, EWOULDBUILD if not */
{
	time_t		age;		/* age of the target file */
	int		j;		/* looping temporary */
	cooker_result_ty best;		/* return status */
	cooker_result_ty best2;
	int		outofdate;
	int		used_explicit_recipe;
	int		used_ingredients_recipe;
	int		used_implicit_recipe;
	wlist		target_list;
	time_t		worst_age;
	long		worst_depth;

	trace(("cooker(target = %08lX, dkwinge = %d, build = %d)\n{\n"/*}*/,
		(long)target, dkwinge, build));
	trace_string(target->str_text);
	if (option_test(OPTION_STRIP_DOT))
		strip_dot(&target);
	wl_zero(&target_list);
	wl_append(&target_list, target);
	id_assign_push(id_target, &target_list);
	wl_free(&target_list);
	used_ingredients_recipe = 0;
	used_explicit_recipe = 0;
	used_implicit_recipe = 0;

	/*
	 * Check to see if this one has been cooked already.
	 * It may have failed, too.  If it is currently being cooked
	 * a value of COOK_BACKTRACK will be returned, this is to
	 * trap recursive recipes.
	 */
	if (already_search(target, &best))
	{
		if (best == COOK_BACKTRACK)
		{
			error
			(
				"%s: subject of recipe infinite loop",
				target->str_text
			);
			option_set_errors();
			best = COOK_ERROR;
		}
		goto ret;
	}

	/*
	 * add it to the "currently being cooked" list
	 */
	best = COOK_UPTODATE;
	worst_depth = 32767;
	age = cook_mtime_newest(target, &worst_depth);
	if (age < 0)
	{
		best = COOK_ERROR;
		goto ret;
	}
	best2 = COOK_BACKTRACK;
	already_assign(target, best2);
	worst_age = 0;

	/*
	 * Look for the named file in the recipes.
	 * If it matches, cook its prerequisites.
	 * It is an explicit recipe if there is any action attached to
	 * any recipe targeting the named file.
	 */
	outofdate = !age || option_test(OPTION_FORCE);
	trace(("outofdate = %d;\n", outofdate));

	/*
	 * Cook scans through the instanciated ingredients recipes.  All
	 * ingredients recipes with the target in their target list are used.
	 *
	 * If the recipe is used, any prerequisite files are recursively
	 * cooked, then if any of the prerequisite files were out of date, then
	 * all other explicit or implicit recipes with the same target will be
	 * deemed to be out of date.
	 */
	trace(("check for ingredients recipes\n"));
	for (j = 0; j < explicit.rl_nrecipes; j++)
	{
		if (desist)
		{
			trace(("interrupted, desisting\n"));
			best = COOK_ERROR;
			goto ret;
		}

		if
		(
			explicit.rl_recipe[j].r_action
		||
			!wl_member(&explicit.rl_recipe[j].r_target, target)
		)
			best2 = COOK_BACKTRACK;
		else
		{
			match_push((match_ty *)0);
			best2 =
				chef
				(
					target,
					dkwinge,
					&explicit.rl_recipe[j],
					age,
					outofdate,
					build,
					&worst_age,
					&worst_depth
				);
			match_pop();
		}
		if (best2 == COOK_BACKTRACK)
			continue;
		used_ingredients_recipe = 1;
		if (best2 < best)
			best = best2;
		if (best2 < COOK_DONE && !option_test(OPTION_PERSEVERE))
			goto ret;
	}
	if (best == COOK_DONE)
		outofdate = 1;
	trace(("outofdate = %d;\n", outofdate));

	/*
	 * Cook then scans through the instanciated explicit recipes.
	 * Keep looping until a single-colon recipe is met.
	 *
	 * If the recipe is used, any prerequisite files are recursively
	 * cooked, then if any prerequisites were out of date (including those
	 * of prerequisite recipes) then the actions bound to this recipe will
	 * be evaluated.
	 */
	trace(("check explicit recipes\n"));
	for (j = 0; j < explicit.rl_nrecipes; j++)
	{
		if (desist)
		{
			best = COOK_ERROR;
			goto ret;
		}

		if
		(
			!explicit.rl_recipe[j].r_action
		||
			!wl_member(&explicit.rl_recipe[j].r_target, target)
		)
			best2 = COOK_BACKTRACK;
		else
		{
			match_push((match_ty *)0);
			best2 =
				chef
				(
					target,
					dkwinge,
					&explicit.rl_recipe[j],
					age,
					outofdate,
					build,
					&worst_age,
					&worst_depth
				);
			match_pop();
		}
		if (best2 == COOK_BACKTRACK)
			continue;
		used_explicit_recipe = 1;
		if (best2 < best)
			best = best2;
		if (best2 < COOK_DONE && !option_test(OPTION_PERSEVERE))
			goto ret;
		if (!explicit.rl_recipe[j].r_multiple)
			goto ret;
	}
	if (used_explicit_recipe)
		goto ret;
	trace(("outofdate = %d;\n", outofdate));

	/*
	 * None of the recipes specified an action,
	 * although prerequisites may have been specified.
	 * Check out the implicit recipes, instead.
	 *
	 * The tricky part is when going through several
	 * layers of recipes.  If a cook of a prerequisite comes back
	 * with BACKTRACK, then try something else.
	 *
	 * Only the first applicable implicit recipe is used.
	 */
	trace(("check implicit recipes\n"));
	for (j = 0; j < implicit.rl_nrecipes; j++)
	{
		match_ty *mbuf;

		if (desist)
		{
			best = COOK_ERROR;
			goto ret;
		}

		if (implicit.rl_recipe[j].inhibit)
			continue;
		mbuf = wl_match(&implicit.rl_recipe[j].r_target, target);
		if (mbuf)
		{
			match_push(mbuf);
			best2 =
				chef
				(
					target,
					0,
					&implicit.rl_recipe[j],
					age,
					outofdate,
					build,
					&worst_age,
					&worst_depth
				);
			match_pop();
			match_free(mbuf);
		}
		else
			best2 = COOK_BACKTRACK;
		if (best2 == COOK_BACKTRACK)
			continue;
		used_implicit_recipe = 1;
		if (best2 < best)
			best = best2;
		if (!implicit.rl_recipe[j].r_multiple)
			goto ret;
	}
	if (used_implicit_recipe)
		goto ret;
	trace(("outofdate = %d;\n", outofdate));

	/*
	 * None of the implicit recipes worked, either
	 * (perhapse none applied)
	 * so we don't know how to make this one.
	 *
	 * If it already exists, it must be up-to-date;
	 * but if it doesn't exist, then we may want to backtrack.
	 */
	trace(("no recipe applied\n"));
	if (age <= 0 && !used_ingredients_recipe)
		best = (dkwinge ? COOK_DONTKNOW : COOK_BACKTRACK);
	if (best != COOK_BACKTRACK)
		already_assign(target, best);

ret:
	if (already_search(target, &best2) && best2 == COOK_BACKTRACK)
		already_delete(target);
	id_delete(id_target);
	if (option_test(OPTION_TRACE))
	{
		char		*name;

		switch (best)
		{
		case COOK_DONTKNOW:
			name = "don't know how";
			break;

		case COOK_ERROR:
			name = "error";
			break;

		case  COOK_DONE:
			name = "done";
			break;

		case COOK_DONE_UPTODATE:
			name = "done and found to be up to date";
			break;

		case COOK_UPTODATE:
			name = "up to date";
			break;

		case COOK_BACKTRACK:
			name = "backtracking";
			break;

		default:
			name = "unknown (bug)";
			break;
		}
		error("\"%s\" %s (trace)", target->str_text, name);
	}
	else
	{
		switch (best)
		{
		case COOK_DONTKNOW:
			if (dkwinge)
				error("%s: don't know how", target->str_text);
			break;

		case COOK_ERROR:
			error
			(
				"%s: not done because of errors",
				target->str_text
			);
			break;

		default:
			break;
		}
	}
	trace(("return %s;\n", cook_status_name(best)));
	trace((/*{*/"}\n"));
	return best;
}


/*
 * NAME
 *	rl_append - append a recipe
 *
 * SYNOPSIS
 *	void el_append(rlist *rlp, recipe *rp);
 *
 * DESCRIPTION
 *	Rl_append is used to append a recipe to a recipe list.
 *
 * CAVEAT
 *	Recipes need to overwrite other recipes with
 *	identical target and prerequisite lists.
 */

void
rl_append(rlp, rp)
	rlist		*rlp;
	recipe		*rp;
{
	size_t		nbytes;
	size_t		j;

	trace(("rl_append(rlp = %08lX, rp = %08lX)\n{\n"/*}*/, rlp, rp));
	assert(rlp);
	assert(rp);
	assert(!rlp->rl_nrecipes == !rlp->rl_recipe);
	assert(rp->r_tag > 0 && rp->r_tag <= max_tag);

	if (option_test(OPTION_STRIP_DOT))
		for (j = 0; j < rp->r_target.wl_nwords; ++j)
			strip_dot(&rp->r_target.wl_word[j]);
	rp->inhibit = 0;

	nbytes = (rlp->rl_nrecipes + 1) * sizeof(recipe);
	rlp->rl_recipe = mem_change_size(rlp->rl_recipe, nbytes);
	rlp->rl_recipe[rlp->rl_nrecipes++] = *rp;
	trace((/*{*/"}\n"));
}


static void recipe_free _((recipe *));

static void
recipe_free(rp)
	recipe	*rp;
{
	wl_free(&rp->r_target);
	el_free(&rp->r_need);
	el_free(&rp->r_need2);
	if (rp->r_action)
		stmt_free(rp->r_action);
	if (rp->r_use_action)
		stmt_free(rp->r_use_action);
	if (rp->r_precondition)
		expr_free(rp->r_precondition);
}


static void rl_free _((rlist *));

static void
rl_free(rlp)
	rlist		*rlp;
{
	long		j;

	trace(("rl_free(rlp = %08lX)\n{\n"/*}*/, (long)rlp));
	assert(rlp);
	assert(!rlp->rl_nrecipes == !rlp->rl_recipe);
	for (j = 0; j < rlp->rl_nrecipes; ++j)
		recipe_free(&rlp->rl_recipe[j]);
	if (rlp->rl_recipe)
		mem_free(rlp->rl_recipe);
	rlp->rl_recipe = 0;
	rlp->rl_nrecipes = 0;
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	cook_flags - set them
 *
 * SYNOPSIS
 *	void cook_flags(int mask, option_levelk_ty level);
 *
 * DESCRIPTION
 *	The cook_flags function is used to take a flags variable and set the
 *	appropriate options at the given level.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Use the option_undo_level function to remove the flag settings.
 */

void
cook_flags(mask, level)
	int		mask;
	int		level;
{
	trace(("cook_flags(mask = 0x%04X, level = %d)\n{\n"/*}*/, mask, level));
	if (mask & RF_CLEARSTAT)
		option_set(OPTION_INVALIDATE_STAT_CACHE, level, 1);
	if (mask & RF_CLEARSTAT_OFF)
		option_set(OPTION_INVALIDATE_STAT_CACHE, level, 0);
	if (mask & RF_ERROK)
		option_set(OPTION_ERROK, level, 1);
	if (mask & RF_ERROK_OFF)
		option_set(OPTION_ERROK, level, 0);
	if (mask & RF_FINGERPRINT)
		option_set(OPTION_FINGERPRINT, level, 1);
	if (mask & RF_FINGERPRINT_OFF)
		option_set(OPTION_FINGERPRINT, level, 0);
	if (mask & RF_FORCE)
		option_set(OPTION_FORCE, level, 1);
	if (mask & RF_FORCE_OFF)
		option_set(OPTION_FORCE, level, 0);
	if (mask & RF_METER)
		option_set(OPTION_METER, level, 1);
	if (mask & RF_METER_OFF)
		option_set(OPTION_METER, level, 0);
	if (mask & RF_PRECIOUS)
		option_set(OPTION_PRECIOUS, level, 1);
	if (mask & RF_PRECIOUS_OFF)
		option_set(OPTION_PRECIOUS, level, 0);
	if (mask & RF_SILENT)
		option_set(OPTION_SILENT, level, 1);
	if (mask & RF_SILENT_OFF)
		option_set(OPTION_SILENT, level, 0);
	if (mask & RF_UPDATE)
		option_set(OPTION_UPDATE, level, 1);
	if (mask & RF_UPDATE_OFF)
		option_set(OPTION_UPDATE, level, 0);
	if (mask & RF_STRIPDOT)
		option_set(OPTION_STRIP_DOT, level, 1);
	if (mask & RF_STRIPDOT_OFF)
		option_set(OPTION_STRIP_DOT, level, 0);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	cook - construct files
 *
 * SYNOPSIS
 *	int cook(wlist *targets);
 *
 * DESCRIPTION
 *	The cook function is used to cook the given set of targets.
 *
 * RETURNS
 *	The cook function returns 0 if all of the targets cooked sucessfully,
 *	or 1 if there was any problem.
 *
 * CAVEAT
 *	This function must be called after evrything has been initialized,
 *	and the cookbook read in.
 */

int
cook(wlp)
	wlist		*wlp;
{
	int		retval;
	int		j;

	/*
	 * set interrupts to catch
	 *
	 * Note that tee(1) [see listing.c] must ignore them
	 * for the generated messages to appear in the log file.
	 */
	trace(("cook(wlp = %08lX)\n{\n"/*}*/, wlp));
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, interrupt);
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
		signal(SIGHUP, interrupt);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
		signal(SIGTERM, interrupt);

	/*
	 * cook each of the targets
	 */
	retval = 0;
	for (j = 0; j < wlp->wl_nwords; j++)
	{
		switch (cooker(wlp->wl_word[j], 1, 1))
		{
		case COOK_ERROR:
		case COOK_DONTKNOW:
		case COOK_BACKTRACK:
			retval = 1;
			break;

		case COOK_UPTODATE:
			error
			(
				"%s: already up to date",
				wlp->wl_word[j]->str_text
			);
			break;

		case COOK_DONE_UPTODATE:
		case COOK_DONE:
			break;
		}
		if (retval > 0 && !option_test(OPTION_PERSEVERE))
			break;
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	isit_uptodate - test construct files
 *
 * SYNOPSIS
 *	int isit_uptodate(string_ty *target);
 *
 * DESCRIPTION
 *	The isit_uptodate function is used test if the giben target
 *	is up to date.
 *
 * RETURNS
 *	The isit_uptodate function returns
 *		-1 if any error is encountered
 *		0 if not up to date, or don't know how
 *		1 if target is uptodate.
 *
 * CAVEAT
 *	This function must be called after everything has been initialized,
 *	and the cookbook read in.
 */

int
isit_uptodate(target)
	string_ty	*target;
{
	int		retval;

	/*
	 * set interrupts to catch
	 *
	 * Note that tee(1) [see listing.c] must ignore them
	 * for the generated messages to appear in the log file.
	 */
	trace(("isit_uptodate(target = %08lX)\n{\n"/*}*/, target));
	trace_string(target->str_text);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, interrupt);
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
		signal(SIGHUP, interrupt);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
		signal(SIGTERM, interrupt);

	/*
	 * cook the target
	 */
	switch (cooker(target, 0, 0))
	{
	default:
		retval = -1;
		break;

	case COOK_UPTODATE:
	case COOK_DONE_UPTODATE:
		retval = 1;
		break;

	case COOK_DONE:
	case COOK_DONTKNOW:
		retval = 0;
		break;
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	cando - test construct files
 *
 * SYNOPSIS
 *	int cando(string_ty *target);
 *
 * DESCRIPTION
 *	The cando function is used to test if the given target is cookable.
 *
 * RETURNS
 *	The cando function returns
 *		-1 if any error was encountered
 *		0 if can't cook the target
 *		1 if can cook target
 *
 * CAVEAT
 *	This function must be called after everything has been initialized,
 *	and the cookbook read in.
 */

int
cando(target)
	string_ty	*target;
{
	int		retval;

	/*
	 * set interrupts to catch
	 *
	 * Note that tee(1) [see listing.c] must ignore them
	 * for the generated messages to appear in the log file.
	 */
	trace(("cando(target = %08lX)\n{\n"/*}*/, target));
	trace_string(target->str_text);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, interrupt);
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
		signal(SIGHUP, interrupt);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
		signal(SIGTERM, interrupt);

	/*
	 * cook the target
	 */
	switch (cooker(target, 0, 0))
	{
	default:
		retval = -1;
		break;

	case COOK_DONTKNOW:
		retval = 0;
		break;

	case COOK_UPTODATE:
	case COOK_DONE_UPTODATE:
	case COOK_DONE:
		retval = 1;
		break;
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


void
cook_auto(wlp)
	wlist		*wlp;
{
	long		j;

	for (j = 0; j < wlp->wl_nwords; ++j)
	{
		wl_append_unique(&cook_auto_list, wlp->wl_word[j]);
	}
}


int
cook_auto_required()
{
	int		result;
	long		j;

	result = 0;
	option_set(OPTION_ACTION, OPTION_LEVEL_AUTO, 1);
	option_set(OPTION_TOUCH, OPTION_LEVEL_AUTO, 0);
	for (j = 0; j < cook_auto_list.wl_nwords; ++j)
	{
		int	status;

		status = cooker(cook_auto_list.wl_word[j], 1, 1);
		switch (status)
		{
		default:
			result = -1;
			break;

		case COOK_UPTODATE:
		case COOK_DONE_UPTODATE:
			continue;

		case COOK_DONE:
			result = 1;
			continue;
		}
		break;
	}
	option_undo(OPTION_ACTION, OPTION_LEVEL_AUTO);
	option_undo(OPTION_TOUCH, OPTION_LEVEL_AUTO);
	return result;
}


void
cook_reset()
{
	wl_free(&cook_auto_list);
	rl_free(&explicit);
	rl_free(&implicit);
	max_tag = 0;
	if (already)
	{
		symtab_free(already);
		already = 0;
	}
}


void
cook_find_default(wlp)
	wlist		*wlp;
{
	size_t		j;
	recipe		*rp;

	/*
	 * use the forst one
	 * explicitly flagged default
	 */
	for (j = 0; j < explicit.rl_nrecipes; ++j)
	{
		rp = &explicit.rl_recipe[j];
		if (rp->r_flags & RF_DEFAULT)
		{
			wl_copy(wlp, &rp->r_target);
			return;
		}
	}

	/*
	 * use the first one
	 * not flagged nodefault
	 */
	for (j = 0; j < explicit.rl_nrecipes; ++j)
	{
		rp = &explicit.rl_recipe[j];
		if (!(rp->r_flags & RF_DEFAULT_OFF))
		{
			wl_copy(wlp, &rp->r_target);
			return;
		}
	}

	/*
	 * fatal error otherwise
	 */
	fatal("no default target");
}
