/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate the symbol table
 */

#include <ac/stddef.h>
#include <ac/stdlib.h>

#include <error.h>
#include <id.h>
#include <mem.h>
#include <option.h>
#include <symtab.h>
#include <version-stmp.h>
#include <word.h>

static	symtab_ty	*symtab;
	string_ty	*id_need;
	string_ty	*id_younger;
	string_ty	*id_target;
	string_ty	*id_friend;
	string_ty	*id_search_list;


static void reap _((void *));

static void
reap(p)
	void	*p;
{
	wlist	*wlp;

	wlp = p;
	wl_free(wlp);
	mem_free(wlp);
}


static wlist *mem_copy_wlist _((wlist *));

static wlist *
mem_copy_wlist(wlp)
	wlist	*wlp;
{
	wlist	*result;

	result = mem_alloc(sizeof(wlist));
	wl_copy(result, wlp);
	return result;
}


/*
 * NAME
 *	id_initialize - start up symbol table
 *
 * SYNOPSIS
 *	void id_initialize(void);
 *
 * DESCRIPTION
 *	The id_initialize function is used to create the hash table.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Assumes the str_initialize function has been called already.
 */

void
id_initialize()
{
	id_need = str_from_c("need");
	id_younger = str_from_c("younger");
	id_target = str_from_c("target");
	id_friend = str_from_c("targets");
	id_search_list = str_from_c("search_list");

	id_reset();
}


void
id_reset()
{
	wlist		wl;
	string_ty	*s;

	if (symtab)
		symtab_free(symtab);
	symtab = symtab_alloc(100);
	symtab->reap = reap;

	/*
	 * set the "version" predefined variable
	 */
	wl_zero(&wl);
	s = str_from_c(version_stamp());
	wl_append(&wl, s);
	str_free(s);
	s = str_from_c("version");
	id_assign(s, &wl);
	str_free(s);
	wl_free(&wl);

	/*
	 * set the "self" predefined variable
	 */
	s = str_from_c(progname);
	wl_append(&wl, s);
	str_free(s);
	s = str_from_c("self");
	id_assign(s, &wl);
	str_free(s);
	wl_free(&wl);

#ifdef __STDC__
	/*
	 * This symbol is only defined if we have a conforming C
	 * compiler.  This is support for the C recipes.
	 */
	wl_append(&wl, str_true);
	s = str_from_c("__STDC__");
	id_assign(s, &wl);
	str_free(s);
	wl_free(&wl);
#endif
	
	/*
	 * set the default search list
	 * in the "search_list" predefined variable
	 */
	s = str_from_c(".");
	wl_append(&wl, s);
	str_free(s);
	id_assign(id_search_list, &wl);
	wl_free(&wl);
}


/*
 * NAME
 *	id_search - search for a variable
 *
 * SYNOPSIS
 *	int id_search(string_ty *name, wlist *value);
 *
 * DESCRIPTION
 *	Id_search is used to reference a variable.
 *
 * RETURNS
 *	If the variable has been defined, the function returns a non-zero value
 *	and the value is returned through the 'value' pointer.
 *	If the variable has not been defined, it returns zero,
 *	and 'value' is unaltered.
 *
 * CAVEAT
 *	The value returned from this function, when returned, is allocated
 *	in dynamic memory (it is a copy of the value remembered by this module).
 *	It is the responsibility of the caller to free it when finished with,
 *	by a wl_free() call.
 */

int
id_search(name, arg)
	string_ty	*name;
	wlist		*arg;
{
	wlist		*data;

	assert(symtab);
	data = symtab_query(symtab, name);
	if (!data)
		return 0;
	wl_copy(arg, data);
	return 1;
}


/*
 * NAME
 *	id_assign - assign a variable
 *
 * SYNOPSIS
 *	void id_assign(string_ty *name, id_class_ty class, ...);
 *
 * DESCRIPTION
 *	Id_assign is used to assign a value to a given variable.
 *
 * CAVEAT
 *	The name and value are copied by id_assign, so the user may
 *	modify or free them at a later date without affecting the
 *	variable.
 */

void
id_assign(name, arg)
	string_ty	*name;
	wlist		*arg;
{
	assert(symtab);
	symtab_assign(symtab, name, mem_copy_wlist(arg));
}


/*
 * NAME
 *	id_assign_push - assign a variable, remembering old value
 *
 * SYNOPSIS
 *	void id_assign_push(string_ty *name, id_class_ty class, ...);
 *
 * DESCRIPTION
 *	Id_assign is used to assign a value to a given variable.
 *
 * CAVEAT
 *	The name and value are copied by id_assign, so the user may
 *	modify or free them at a later date without affecting the
 *	variable.
 */

void
id_assign_push(name, arg)
	string_ty	*name;
	wlist		*arg;
{
	assert(symtab);
	symtab_assign_push(symtab, name, mem_copy_wlist(arg));
}


/*
 * NAME
 *	id_delete - delete a variable
 *
 * SYNOPSIS
 *	void id_delete(string_ty *name, id_class_ty class);
 *
 * DESCRIPTION
 *	Id_delete is used to delete variables.
 *
 * CAVEAT
 *	No complaint is made if the variable does not exist,
 *	since the user wanted it to not exist.
 */

void
id_delete(name)
	string_ty	*name;
{
	assert(symtab);
	symtab_delete(symtab, name);
}


/*
 * NAME
 *	id_dump - dump id table
 *
 * SYNOPSIS
 *	void id_dump(char *title, int mask);
 *
 * DESCRIPTION
 *	The id_dump function is used to dump the contents of the id table.
 *	The title will be used to indicate why the table was dumped.  The mask
 *	may be used to selectively dump the table, 0 means everything, bits
 *	correspond directly with ID_CLASS defines.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	This function is only available when symbol DEBUG is defined.
 */

#ifdef DEBUG

void
id_dump(s)
	char		*s;
{
	assert(symtab);
	symtab_dump(symtab, s);
}

#endif
