/*
 *	cook - file construction tool
 *	Copyright (C) 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: operating system start point, and parse command line arguments
 */

#include <ac/stddef.h>
#include <stdio.h>
#include <ac/string.h>
#include <ac/stdlib.h>

#include <sys/types.h>
#include <utime.h>
#include <sys/stat.h>

#include <arglex.h>
#include <date.h>
#include <error.h>
#include <help.h>
#include <main.h>
#include <str.h>
#include <trace.h>
#include <version.h>
#include <word.h>


static void usage _((void));

static void
usage()
{
	fprintf(stderr, "usage: %s [ <option>... ] <filename>\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERsion\n", progname);
	exit(1);
}


static void main_help _((void));

static void
main_help()
{
	static char *text[] =
	{
#include <../man1/cooktime.h>
	};

	trace(("c_incl_help()\n{\n"/*}*/));
	help(text, SIZEOF(text), usage);
	trace((/*{*/"}\n"));
}


enum
{
	arglex_token_access,
	arglex_token_modify,
	arglex_token_report
};

static	arglex_table_ty	argtab[] =
{
	{ "-Access",	(arglex_token_ty)arglex_token_access,	},
	{ "-Modify",	(arglex_token_ty)arglex_token_modify,	},
	{ "-Report",	(arglex_token_ty)arglex_token_report,	},
	{ 0, (arglex_token_ty)0, }, /* end marker */
};


int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	wlist		filename;
	string_ty	*s;
	size_t		j;
	int		mtime_set;
	long		mtime;
	int		atime_set;
	long		atime;
	int		report;

	arglex_init(argc, argv, argtab);
	str_initialize();
	switch (arglex())
	{
	case arglex_token_help:
		main_help();
		exit(0);

	case arglex_token_version:
		version();
		exit(0);

	default:
		break;
	}

	report = 0;
	atime_set = 0;
	atime = 0;
	mtime_set = 0;
	mtime = 0;
	wl_zero(&filename);
	while (arglex_token != arglex_token_eoln)
	{
		switch(arglex_token)
		{
		default:
			error
			(
				"misplaced \"%s\" command line argument",
				arglex_value.alv_string
			);
			usage();

		case arglex_token_string:
			s = str_from_c(arglex_value.alv_string);
			wl_append(&filename, s);
			str_free(s);
			break;

		case arglex_token_report:
			if (report)
			{
				duplicate:
				fatal
				(
					"duplicate \"%s\" option",
					arglex_value.alv_string
				);
			}
			report++;
			break;

		case arglex_token_access:
			if (atime_set)
				goto duplicate;
			if (arglex() != arglex_token_string)
				fatal("-Access requires a string argument");
			atime_set = 1;
			atime = date_scan(arglex_value.alv_string);
			if (atime < 0)
				fatal
				(
					"the string \"%s\" is not a valid date",
					arglex_value.alv_string
				);
			break;

		case arglex_token_modify:
			if (mtime_set)
				goto duplicate;
			if (arglex() != arglex_token_string)
				fatal("-Modify requires a string argument");
			mtime_set = 1;
			mtime = date_scan(arglex_value.alv_string);
			if (mtime < 0)
				fatal
				(
					"the string \"%s\" is not a valid date",
					arglex_value.alv_string
				);
			break;

#ifdef DEBUG
		case arglex_token_tracing:
			if (arglex() != arglex_token_string)
				fatal("-TRACIng requires one or more string arguments");
			for (;;)
			{
				trace_enable(arglex_value.alv_string);
				if (arglex() != arglex_token_string)
					break;
			}
			continue;
#endif
		}
		arglex();
	}
	if (!filename.wl_nwords)
		fatal("no files named");
	if (!report && !mtime_set && !atime_set)
	{
		mtime_set = 1;
		mtime = date_scan("now");
	}

	for (j = 0; j < filename.wl_nwords; ++j)
	{
		struct stat	st;
		struct utimbuf	ut;

		s = filename.wl_word[j];
		if (stat(s->str_text, &st))
			nfatal("%s", s->str_text);
		if (mtime_set)
			ut.modtime = mtime;
		else
			ut.modtime = st.st_mtime;
		if (atime_set)
			ut.actime = atime;
		else
			ut.actime = st.st_atime;
		if ((atime_set || mtime_set) && utime(s->str_text, &ut))
			nfatal("%s", s->str_text);
		if (report)
		{
			printf("%s\n", s->str_text);
			printf("\taccess %s\n", date_string(ut.actime));
			printf("\tmodify %s\n", date_string(ut.modtime));
		}
	}
	exit(0);
	return 0;
}
