/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to perform lexical analysis on Makefiles
 */

#include <ctype.h>
#include <stdio.h>
#include <ac/string.h>

#include <blob.h>
#include <error.h>
#include <lex.h>
#include <mem.h>
#include <s-v-arg.h>
#include <stmt.h>
#include <symtab.h>
#include <trace.h>
#include <gram.gen.h> /* must be last */

static string_ty *file_name;
static FILE	*input;
static long	line_number;
static int	error_count;
static int	bol;
static int	first;
static int	colon_special;
static int	within_define;
static long	sa_data_length;
static long	sa_data_max;
static char	*sa_data;


static void notify _((void));

static void
notify()
{
	if (!file_name)
		return;
	if (++error_count >= 20)
		fatal("%S: too many errors, bye!", file_name);
}


void
lex_open(s)
	char		*s;
{
	trace(("lex_open()\n{\n"/*}*/));
	assert(!input);
	blob_error_notify(notify);
	if (s)
	{
		input = fopen(s, "r");
		if (!input)
			nfatal("%s", s);
		file_name = str_from_c(s);
	}
	else
	{
		file_name = str_from_c("standard input");
		input = stdin;
	}
	line_number = 1;
	bol = 1;
	first = 1;
	colon_special = 1;
	error_count = 0;
	trace((/*{*/"}\n"));
}


void
lex_close()
{
	trace(("lex_close()\n{\n"/*}*/));
	assert(input);
	if (error_count)
	{
		fatal
		(
			"%S: found %d fatal error%s",
			file_name,
			error_count,
			(error_count == 1 ? "" : "s")
		);
	}
	if (input != stdin)
		fclose(input);
	input = 0;
	str_free(file_name);
	file_name = 0;
	line_number = 0;
	bol = 0;
	first = 0;
	trace((/*{*/"}\n"));
}


void
gram_error(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	string_ty	*buffer;
	int		len;

	sva_init(ap, s);
	buffer = str_vformat(s, ap);
	va_end(ap);
	len = buffer->str_length;
	while (len > 0 && isspace(buffer->str_text[len - 1]))
		--len;
	error("%S: %ld: %.*S", file_name, line_number, len, buffer);
	str_free(buffer);
	notify();
}


static int byte _((void));

static int
byte()
{
	int		c;

	assert(input);
	c = getc(input);
	switch (c)
	{
	case EOF:
		if (ferror(input))
			nfatal("read %s", file_name->str_text);
		bol = 1;
		break;
	
	case '\n':
		++line_number;
		bol = 1;
		first = 1;
		colon_special = 1;
		break;

	case ' ':
	case '\t':
	case '\f':
#if __STDC__ >= 1
	case '\v':
#endif
		bol = 0;
		break;

	default:
		bol = 0;
		first = 0;
		break;
	}
	return c;
}


static void byte_undo _((int));

static void
byte_undo(c)
	int		c;
{
	switch (c)
	{
	case EOF:
		break;

	case '\n':
		--line_number;
		ungetc(c, input);
		break;
	
	default:
		ungetc(c, input);
		break;
	}
}


/*
 * NAME
 *	lex_trace - debug output
 *
 * SYNOPSIS
 *	void lex_trace(char *s, ...);
 *
 * DESCRIPTION
 *	The lex_trace function is used to format and output yyparse's trace
 *	output.  The printf's in yyparse are #define'd into lex_trace calls.
 *	Unfortunately yacc's designer did not take trace output redirection
 *	into account, to this function is a little tedious.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	This function is only available when the DEBUG symbol is #define'd.
 */

#ifdef DEBUG

void
gram_trace(s sva_last)
	char		*s;
	sva_last_decl
{
	va_list		ap;
	string_ty	*buffer;
	char		*cp;
	static char	line[1024];

	sva_init(ap, s);
	buffer = str_vformat(s, ap);
	va_end(ap);
	strcat(line, buffer->str_text);
	str_free(buffer);
	cp = line + strlen(line) - 1;
	if (cp > line && *cp == '\n')
	{
		*cp = 0;
		trace_printf
		(
			"%s: %ld: %s\n",
			file_name->str_text,
			line_number,
			line
		);
		line[0] = 0;
	}
}


void
gram_trace2(garbage, s sva_last)
	void		*garbage;
	char		*s;
	sva_last_decl
{
	va_list		ap;
	string_ty	*buffer;
	char		*cp;
	static char	line[1024];

	sva_init(ap, s);
	buffer = str_vformat(s, ap);
	va_end(ap);
	strcat(line, buffer->str_text);
	str_free(buffer);
	cp = line + strlen(line) - 1;
	if (cp > line && *cp == '\n')
	{
		*cp = 0;
		error("%S: %ld: %s", file_name, line_number, line);
		line[0] = 0;
	}
}

#endif


static int reserved _((string_ty *s));

static int
reserved(s)
	string_ty	*s;
{
	typedef struct table_ty table_ty;
	struct table_ty
	{
		char	*name;
		int	token;
	};

	static table_ty table[] =
	{
		{ "override",	OVERRIDE,	},
		{ "include",	INCLUDE2,	},
		{ "-include",	INCLUDE3,	},
		{ ".include",	INCLUDE,	},
		{ "vpath",	VPATH,		},
		{ "VPATH",	VPATH2,		},
		{ "ifdef",	IF,		},
		{ "ifndef",	IF,		},
		{ "ifeq",	IF,		},
		{ "ifneq",	IF,		},
		{ "else",	ELSE,		},
		{ "endif",	ENDIF,		},
		{ "endef",	ENDDEF,		},
		{ "define",	DEFINE,		},
		{ "export",	EXPORT,		},
		{ "unexport",	UNEXPORT,	},
	};

	static symtab_ty *symtab;
	int		*data;
	string_ty	*name;
	char		*cp;

	if (!symtab)
	{
		table_ty	*tp;

		symtab = symtab_alloc(SIZEOF(table));
		for (tp = table; tp < ENDOF(table); ++tp)
		{
			name = str_from_c(tp->name);
			symtab_assign(symtab, name, &tp->token);
			str_free(name);
		}
	}

	cp = strchr(s->str_text, '('/*)*/);
	if (cp)
	{
		name = str_n_from_c(s->str_text, cp - s->str_text);
		data = symtab_query(symtab, name);
		str_free(name);
	}
	else
		data = symtab_query(symtab, s);
	if (data)
		return *data;
	return 0;
}


static void sa_open _((void));

static void
sa_open()
{
	sa_data_length = 0;
}


static string_ty *sa_close _((void));

static string_ty *
sa_close()
{
	string_ty	*s;

	s = str_n_from_c(sa_data, sa_data_length);
	sa_data_length = 0;
	return s;
}


static void sa_char _((int));

static void
sa_char(c)
	int		c;
{
	if (sa_data_length >= sa_data_max)
	{
		sa_data_max = sa_data_max * 2 + 16;
		sa_data = mem_change_size(sa_data, sa_data_max);
	}
	sa_data[sa_data_length++] = c;
}


int
gram_lex()
{
	static char	*paren;
	static long	paren_max;
	long		paren_depth;
	int		c;
	long		linum;
	int		bol_was;
	int		first_was;
	string_ty	*s;
	int		token;

	trace(("gram_lex()\n{\n"/*}*/));
	for (;;)
	{
		linum = line_number;
		bol_was = bol;
		first_was = first;
		c = byte();
		switch (c)
		{
		case EOF:
			token = 0;
			goto done;

		case '\t':
			if (!bol_was || within_define)
				continue;
			sa_open();
			for (;;)
			{
				c = byte();
				switch (c)
				{
				case EOF:
				case '\n':
					break;

				case '\\':
					c = byte();
					if (c == EOF)
						break;
					if (c == '\n')
						sa_char(' ');
					else
					{
						sa_char('\\');
						sa_char(c);
					}
					continue;

				case ' ':
				case '\t':
				case '\f':
#if __STDC__ >= 1
				case '\v':
#endif
					if (sa_data_length)
						sa_char(c);
					continue;

				default:
					sa_char(c);
					continue;
				}
				break;
			}
			gram_lval.lv_line =
				blob_alloc(sa_close(), file_name, linum);
			token = COMMAND;
			goto done;

		case '#':
			sa_open();
			for (;;)
			{
				c = byte();
				switch (c)
				{
				case EOF:
				case '\n':
					break;

				case '#':
					if (sa_data_length)
						sa_char(c);
					continue;

				case ' ':
				case '\t':
				case '\f':
#if __STDC__ >= 1
				case '\v':
#endif
					if (sa_data_length)
						sa_char(' ');
					continue;

				default:
					sa_char(c);
					continue;
				}
				break;
			}
			gram_lval.lv_line =
				blob_alloc(sa_close(), file_name, linum);
			if (!first_was)
			{
				blob_free(gram_lval.lv_line);
				byte_undo('\n');
				continue;
			}
			token = COMMENT;
			goto done;

		case ' ':
		case '\f':
#if __STDC__ >= 1
		case '\v':
#endif
			break;

		case '\n':
			token = EOLN;
			goto done;

		case ';':
			byte_undo('\t');
			bol = 1;
			first = 1;
			colon_special = 1;
			token = EOLN;
			goto done;

		case ':':
			if (!colon_special)
				goto normal;
			c = byte();
			if (c == ':')
			{
				token = COLON_COLON;
				goto done;
			}
			if (c == '=')
			{
				token = COLON_EQUALS;
				colon_special = 0;
				goto done;
			}
			byte_undo(c);
			token = COLON;
			goto done;

		case '=':
			token = EQUALS;
			colon_special = 0;
			goto done;

		case '\\':
			c = byte();
			if (c == '\n')
				continue;
			byte_undo(c);
			c = '\\';
			goto normal;

		case '+':
			c = byte();
			if (c == '=')
			{
				token = PLUS_EQUALS;
				colon_special = 0;
				goto done;
			}
			byte_undo(c);
			c = '+';
			/* fall through... */
		
		default:
			normal:
			sa_open();
			paren_depth = 0;
			for (;;)
			{
				switch (c)
				{
				case '\\':
					c = byte();
					if (c == EOF)
						c = '\\';
					else if (c == '\n')
						c = ' ';
					sa_char(c);
					c = byte();
					continue;

				case EOF:
				case '\n':
					break;

				case ' ':
				case '\t':
				case '\f':
#if __STDC__ >= 1
				case '\v':
#endif
				case ';':
					if (!within_define && !paren_depth)
						break;
					sa_char(c);
					c = byte();
					continue;

				case ':':
				case '=':
					if
					(
						colon_special
					&&
						!within_define
					&&
						!paren_depth
					)
						break;
					sa_char(c);
					c = byte();
					continue;

				default:
					sa_char(c);
					c = byte();
					continue;

				case '('/*)*/:
					sa_char(c);
					if (paren_depth >= paren_max)
					{
						paren_max = paren_max * 2 + 16;
						paren = mem_change_size(paren, paren_max);
					}
					paren[paren_depth++] = /*(*/')';
					c = byte();
					continue;
				
				case /*(*/')':
				case /*{*/'}':
					sa_char(c);
					if
					(
						paren_depth
					&&
						c == paren[paren_depth - 1]
					)
						--paren_depth;
					c = byte();
					continue;

				case '{'/*}*/:
					sa_char(c);
					if (paren_depth >= paren_max)
					{
						paren_max = paren_max * 2 + 16;
						paren = mem_change_size(paren, paren_max);
					}
					paren[paren_depth++] = /*{*/'}';
					c = byte();
					continue;
				}
				break;
			}
			byte_undo(c);
			s = sa_close();
			if (first_was && (token = reserved(s)) != 0)
			{
				switch (token)
				{
				case DEFINE:
					str_free(s);
					++within_define;
					break;
				
				case ENDDEF:
					str_free(s);
					--within_define;
					break;

				case IF:
					gram_lval.lv_line =
						blob_alloc(s, file_name, linum);
					break;
				
				case VPATH:
					colon_special = 0;
					break;

				default:
					str_free(s);
					break;
				}
				goto done;
			}
			gram_lval.lv_line = blob_alloc(s, file_name, linum);
			token = WORD;
			goto done;
		}
	}

	/*
	 * here for all exits
	 */
	done:
#ifdef DEBUG
	if (token == WORD || token == COMMENT || token == COMMAND)
		trace(("text = \"%s\";\n", gram_lval.lv_line->text->str_text));
#endif
	trace(("return %d;\n", token));
	trace((/*{*/"}\n"));
	return token;
}
