/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate if statements
 */

#include <ctype.h>

#include <emit.h>
#include <stmt/if.h>
#include <variable.h>

typedef struct stmt_if_ty stmt_if_ty;
struct stmt_if_ty
{
	STMT
	blob_list_ty	*condition;
	stmt_ty		*then_clause;
	stmt_ty		*else_clause;
};


static void destructor _((stmt_ty *));

static void
destructor(that)
	stmt_ty		*that;
{
	stmt_if_ty	*this;

	this = (stmt_if_ty *)that;
	blob_list_free(this->condition);
	stmt_free(this->then_clause);
	if (this->else_clause)
		stmt_free(this->else_clause);
}


static void emit _((stmt_ty *));

static void
emit(that)
	stmt_ty		*that;
{
	stmt_if_ty	*this;
	size_t		j;

	this = (stmt_if_ty *)that;
	emit_line_number
	(
		this->condition->list[0]->line_number,
		this->condition->list[0]->file_name
	);
	emit_str("#if");
	for (j = 0; j < this->condition->length; ++j)
	{
		emit_char(' ');
		emit_string(this->condition->list[j]->text);
	}
	emit_char('\n');

	stmt_emit(this->then_clause);
	emit_bol();

	if (this->else_clause)
	{
		emit_str("#else\n");
		stmt_emit(this->else_clause);
		emit_bol();
	}
	emit_str("#endif\n");
}


static void regroup _((stmt_ty *));

static void
regroup(that)
	stmt_ty		*that;
{
	stmt_if_ty	*this;

	this = (stmt_if_ty *)that;
	stmt_regroup(this->then_clause);
	if (this->else_clause)
		stmt_regroup(this->else_clause);
}


static void sort _((stmt_ty *));

static void
sort(that)
	stmt_ty		*that;
{
	stmt_if_ty	*this;

	this = (stmt_if_ty *)that;
	stmt_sort(this->then_clause);
	if (this->else_clause)
		stmt_sort(this->else_clause);
}


static stmt_method_ty method =
{
	sizeof(stmt_if_ty),
	"if",
	0, /* constructor */
	destructor,
	emit,
	regroup,
	sort,
};


stmt_ty *
stmt_if_alloc(condition, then_clause, else_clause)
	blob_ty		*condition;
	stmt_ty		*then_clause;
	stmt_ty		*else_clause;
{
	stmt_if_ty	*result;
	blob_list_ty	*c2;

	c2 = blob_list_alloc();
	result = (stmt_if_ty *)stmt_alloc(&method);
	variable_rename(condition, c2, &result->ref);
	blob_free(condition);
	result->condition = c2;
	result->then_clause = then_clause;
	result->else_clause = else_clause;

	stmt_variable_merge((stmt_ty *)result, then_clause);
	if (else_clause)
		stmt_variable_merge((stmt_ty *)result, else_clause);
	return (stmt_ty *)result;
}
