/*  $Id$

    Designed and implemented by Jan Wielemaker
    E-mail: jan@swi.psy.uva.nl

    Copyright: GPL version 2
*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
This file defines a console for porting (unix) stream-based applications
to MS-Windows.  It redefines  read()  and   write()  to  redirect I/O to
descriptors 0, 1, 2 to the console window.  
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#include <windows.h>
#include <setjmp.h>
#include <stdlib.h>
#include <io.h>
#include <string.h>
#include <malloc.h>
#define _MAKE_DLL 1
#undef _export
#include "console.h"

					/* Microsoft VC++ 2.0 stuff */
#ifdef __WATCOMC__
#define POINTS		POINT
#define MAKEPOINTS	MAKEPOINT
#define WinAPI		long FAR PASCAL _export
#else
#define WinAPI		int PASCAL
#define MK_FP32(x)	(x)
#define SIGUSR1		20		/* seems free; cannot above 23!? */
#endif

#ifndef MAXPATHLEN
#define MAXPATHLEN 256
#endif

#ifndef CHAR_MAX
#define CHAR_MAX 256
#endif

#define CMD_INITIAL	0
#define CMD_ESC		1
#define CMD_ANSI	2

#define GWL_DATA	0		/* offset for client data */

#define CHG_RESET	0		/* unchenged */
#define CHG_CHANGED	1		/* changed, but no clear */
#define CHG_CLEAR	2		/* clear */
#define CHG_CARET	4		/* caret has moved */

#define SEL_CHAR	0		/* character-unit selection */
#define SEL_WORD	1		/* word-unit selection */
#define SEL_LINE	2		/* line-unit selection */

#define TagInt(i)	(((i)<<1) | 1)
#define UnTagInt(i)	((i)>>1)

#define NextLine(b, i) ((i) < (b)->height-1 ? (i)+1 : 0)
#define PrevLine(b, i) ((i) > 0 ? (i)-1 : (b)->height-1)
#define Bounds(v, mn, mx) ((v) < (mn) ? (mn) : (v) > (mx) ? (mx) : (v))

#define Control(x) ((x) - '@')
#define Meta(x)	   ((x) + 128)

#define streq(s, q) (strcmp((s), (q)) == 0)

static HANDLE _rlc_hinstance;

typedef struct
{ int		size;		/* #characters in line */
  char *	text;		/* the storage */
  int		adjusted;	/* line has been adjusted? */
  int		changed;	/* line needs redraw */
  int		softreturn;	/* wrapped line */
} text_line, *TextLine;


typedef struct
{ int		height;			/* number of lines in buffer */
  int		width;			/* #characters ler line */
  int		first;			/* first line of ring */
  int		last;			/* last line of ring */
  int		caret_x;		/* cursor's x-position */
  int		caret_y;		/* its line */
  int		window_start;		/* start line of the window */
  int		window_size;		/* #lines on the window */
  TextLine 	lines;			/* the actual lines */
  int		sel_unit;		/* SEL_CHAR, SEL_WORD, SEL_LINE */
  int		sel_org_line;		/* line origin of the selection */
  int		sel_org_char;		/* char origin of the selection */
  int		sel_start_line;		/* starting line for selection */
  int		sel_start_char;		/* starting char for selection */
  int		sel_end_line;		/* ending line for selection */
  int		sel_end_char;		/* ending char for selection */
  int		cmdstat;		/* for parsing ANSI escape */
  int		argstat;		/* argument status ANSI */
  int		argc;			/* argument count for ANSI */
  int		argv[10];		/* argument vector for ANSI */
  int		scaret_x;		/* saved-caret X */
  int		scaret_y;		/* saved-caret Y */
  HWND		window;			/* MS-Window window handle */
  int		has_focus;		/* Application has the focus */
  HFONT		hfont;			/* Windows font handle */
  COLORREF	foreground;		/* Foreground (text) color */
  COLORREF	background;		/* Background color */
  int		cw;			/* character width */
  int		ch;			/* character height */
  int		cb;			/* baseline */
  int		changed;		/* changes to the whole screen */
  int		sb_lines;		/* #lines the scrollbar thinks */
  int		sb_start;		/* start-line scrollbar thinks */
  int		caret_is_shown;		/* is caret in the window? */
  RlcQueue	queue;			/* input stream */
} rlc_data, *RlcData;

static RlcData  _rlc_stdio;		/* the main buffer */
static jmp_buf	_rlc_jmp_context;	/* the longjmp context */
static RlcQueue	_rlc_queue;		/* Current Character Queue */
					/* options */
static int	rlc_x = CW_USEDEFAULT;	/* X-position (pixels) */
static int	rlc_y = CW_USEDEFAULT;	/* Y-position (pixels) */
static int	rlc_rows = 24;		/* #rows of the window */
static int	rlc_cols = 80;		/* #columns of the window */
static int	rlc_savelines = 200;	/* #saved lines */

static char	_rlc_word_chars[CHAR_MAX]; /* word-characters (selection) */

static WinAPI	rlc_wnd_proc(HWND win, UINT msg, UINT wP, LONG lP);

static void	rlc_place_caret(RlcData b);
static void	rlc_resize_pixel_units(RlcData b, int w, int h);
static RlcData	rlc_make_buffer(int w, int h);
static int	rlc_count_lines(RlcData b, int from, int to);
static void	rlc_open_line(RlcData b);
static void	rlc_update_scrollbar(RlcData b);
static void	rlc_paste(RlcData b);
static void	rlc_init_text_dimensions(RlcData b);
static void	rlc_get_options();
static void	rlc_progbase(char *path, char *base);
static int	rlc_add_queue(RlcQueue q, int chr);
static int	rlc_add_lines(RlcData b, int here, int add);
static void	rlc_start_selection(RlcData b, int x, int y);
static void	rlc_extend_selection(RlcData b, int x, int y);
static void	rlc_word_selection(RlcData b, int x, int y);
static void	rlc_copy(RlcData b);
static void	rlc_destroy(void);
static void	rlc_request_redraw(RlcData b);
static void	rlc_redraw(RlcData b);
static int	rlc_breakargs(char *program, char *line, char **argv);
static void	rlc_resize(RlcData b, int w, int h);
static void	rlc_adjust_line(RlcData b, int line);
extern int	main();

static RlcUpdateHook	_rlc_update_hook;
static RlcTimerHook	_rlc_timer_hook;
static RlcRenderHook	_rlc_render_hook;
static RlcRenderAllHook _rlc_render_all_hook;
static int _rlc_copy_output_to_debug_output=0;	/* != 0: copy to debugger */
static int	emulate_three_buttons;
static HWND	emu_hwnd;		/* Emulating for this window */

int
rlc_main(HANDLE hInstance, HANDLE hPrevInstance,
	 LPSTR lpszCmdLine, int nCmdShow,
	 RlcMain mainfunc, HICON icon)
{ HWND		hwnd;
  WNDCLASS	wndClass;
  static char	class[] = "Console";
  char *	argv[100];
  int		argc;
  char		program[MAXPATHLEN];
  char	 	progbase[20];

  _rlc_hinstance = hInstance;

  if ( !hPrevInstance )
  { wndClass.lpszClassName	= class;
    wndClass.style		= CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS;
    wndClass.lpfnWndProc	= (LPVOID) rlc_wnd_proc;
    wndClass.cbClsExtra		= 0;
    wndClass.cbWndExtra		= sizeof(long);
    wndClass.hInstance		= hInstance;
    if ( icon )
      wndClass.hIcon		= icon;
    else
      wndClass.hIcon		= LoadIcon(NULL, IDI_APPLICATION);
    wndClass.hCursor		= LoadCursor(NULL, IDC_ARROW);
    wndClass.hbrBackground	= GetStockObject(WHITE_BRUSH);
    wndClass.lpszMenuName	= NULL;

    RegisterClass(&wndClass);
  }

  GetModuleFileName(hInstance, program, sizeof(program));
  rlc_progbase(program, progbase);
  rlc_get_options("CONSOLE");
  rlc_get_options(progbase);
  _rlc_stdio = rlc_make_buffer(rlc_cols, rlc_savelines);
  rlc_init_text_dimensions(_rlc_stdio);

  hwnd = CreateWindow(class, progbase,
		      WS_OVERLAPPEDWINDOW|WS_VSCROLL,
		      rlc_x, rlc_y,
		      (rlc_cols+2) * _rlc_stdio->cw,
		      (rlc_rows+1) * _rlc_stdio->ch,
		      NULL, NULL, hInstance, NULL);

  _rlc_stdio->window = hwnd;
  SetWindowLong(hwnd, GWL_DATA, (LONG) _rlc_stdio);
  atexit(rlc_destroy);

  _rlc_stdio->foreground = RGB(0,0,0);
  if ( GetSystemMetrics(SM_CMOUSEBUTTONS) == 2 )
    emulate_three_buttons = 120;

  ShowWindow(hwnd, nCmdShow);
  UpdateWindow(hwnd);

  { int rval;

    if ( (rval = setjmp(_rlc_jmp_context)) )
      return UnTagInt(rval);		/* what does this mean? */
  }

  argc = rlc_breakargs(program, lpszCmdLine, argv);

  return (*mainfunc)(argc, argv);
}


int
rlc_iswin32s()
{ if( GetVersion() & 0x80000000 && (GetVersion() & 0xFF) ==3)
    return TRUE;
  else
    return FALSE;
}


static void
rlc_progbase(char *path, char *base)
{ char *s;
  char *e;

  if ( !(s=strrchr(path, '\\')) )
    s = path;				/* takes the filename part */
  if ( !(e = strchr(s, '.')) )
    strcpy(base, s);
  else
  { strncpy(base, s, e-s);
    base[e-s] = '\0';
  }
}


static void
rlc_get_options(char *varname)
{ char *options = getenv(varname);

  while(options && *options)
  { int val;
    char name[100];
    char *s;

    if ( (s = strchr(options, ':')) )
    { strncpy(name, options, s-options);
      name[s-options] = '\0';

      val = atoi(&s[1]);
      if ( streq(name, "sl") )	      rlc_savelines = val;
      else if ( streq(name, "cols") ) rlc_cols = val;
      else if ( streq(name, "rows") ) rlc_rows = val;
      else if ( streq(name, "x") )    rlc_x = val;
      else if ( streq(name, "y") )    rlc_y = val;
    }
    
    options = strchr(options, ',');
    if ( options )
      options++;			/* skip the ',' */
  }
}


static int
rlc_breakargs(char *program, char *line, char **argv)
{ int argc = 1;

  argv[0] = program;

  while(*line)
  { while(*line && isspace(*line))
      line++;
    if ( *line )
    { argv[argc++] = line;
      while(*line && !isspace(*line))
	line++;
      if ( *line )
	*line++ = '\0';
    }
  }
  argv[argc] = NULL;			/* add trailing NULL pointer to argv */

  return argc;
}      
    

		 /*******************************
		 *	     ATTRIBUTES		*
		 *******************************/

void
rlc_foreground(COLORREF fg)
{ _rlc_stdio->foreground = fg;

  InvalidateRect(_rlc_stdio->window, NULL, FALSE);
}


		 /*******************************
		 *	 WINDOW PROCEDURE	*
		 *******************************/

static void
rlc_destroy()
{ if ( _rlc_stdio && _rlc_stdio->window )
    DestroyWindow(_rlc_stdio->window);
}


static WinAPI
rlc_wnd_proc(HWND hwnd, UINT message, UINT wParam, LONG lParam)
{ RlcData b = (RlcData) GetWindowLong(hwnd, GWL_DATA);
  if ( !b )
    b = _rlc_stdio;

  switch(message)
  { case WM_CREATE:
      b->queue    = rlc_make_queue(256);
      b->sb_lines = rlc_count_lines(b, b->first, b->last); 
      b->sb_start = rlc_count_lines(b, b->first, b->window_start); 
      SetScrollRange(hwnd, SB_VERT, 0, b->sb_lines, FALSE);
      SetScrollPos  (hwnd, SB_VERT, b->sb_start, TRUE);
      return 0;

    case WM_SIZE:
      if ( wParam != SIZE_MINIMIZED )
	rlc_resize_pixel_units(b, LOWORD(lParam), HIWORD(lParam));
      return 0;

    case WM_SETFOCUS:
      b->has_focus = TRUE;
      CreateCaret(hwnd, NULL, b->cw, b->ch);
      rlc_place_caret(b);
      return 0;

    case WM_KILLFOCUS:
      b->has_focus = FALSE;
      b->caret_is_shown = FALSE;
      HideCaret(hwnd);
      DestroyCaret();
      return 0;

    case WM_PAINT:
      rlc_redraw(b);
      return 0;

  { int chr; 

    case WM_KEYUP:
    { switch((int) wParam)
      { case VK_DELETE:	chr = 127;		goto out_chr;
	case VK_LEFT:	chr = Control('B');	goto out_chr;
	case VK_RIGHT:	chr = Control('F');	goto out_chr;
	case VK_UP:	chr = Control('P');	goto out_chr;
	case VK_DOWN:	chr = Control('N');	goto out_chr;
      }
      break;
    }
    case WM_SYSCHAR:	chr = Meta(wParam);	goto out_chr;
    case WM_CHAR:	chr = wParam;

  out_chr:
      if ( chr == Control('C') )
	raise(SIGINT);
      else if ( chr == Meta('y') )
	rlc_paste(b);
      else if ( _rlc_queue )
	rlc_add_queue(_rlc_queue, chr);
      else if ( b->queue )
	rlc_add_queue(b->queue, chr);

      return 0;
  }

					/* selection handling */
    case WM_MBUTTONDOWN:
    middle_down:
      return 0;

    case WM_MBUTTONUP:
    middle_up:
      rlc_paste(b);

      return 0;

    case WM_LBUTTONDOWN:
    { POINTS pt;

      if ( emulate_three_buttons )
      { MSG msg;

	Sleep(emulate_three_buttons);
	if ( PeekMessage(&msg, hwnd,
			 WM_RBUTTONDOWN, WM_RBUTTONDOWN, PM_REMOVE) )
	{ emu_hwnd = hwnd;
	  goto middle_down;
	}
      }

      pt = MAKEPOINTS(lParam);
      rlc_start_selection(b, pt.x, pt.y);

      return 0;
    }
    
    case WM_LBUTTONUP:
    case WM_RBUTTONUP:
    if ( emu_hwnd == hwnd )
    { if ( wParam & (MK_RBUTTON|MK_LBUTTON) )
	goto middle_up;
      else
      { emu_hwnd = 0;
	return 0;
      }
    } else
    { rlc_copy(b);

      return 0;
    }

    case WM_LBUTTONDBLCLK:
    { POINTS pt = MAKEPOINTS(lParam);

      rlc_word_selection(b, pt.x, pt.y);

      return 0;
    }

    case WM_RBUTTONDOWN:
    { POINTS pt;

      if ( emulate_three_buttons )
      { MSG msg;

	Sleep(emulate_three_buttons);
	if ( PeekMessage(&msg, hwnd,
			 WM_LBUTTONDOWN, WM_LBUTTONDOWN, PM_REMOVE) )
	{ emu_hwnd = hwnd;
	  goto middle_down;
	}
      }

      pt = MAKEPOINTS(lParam);
      rlc_extend_selection(b, pt.x, pt.y);

      return 0;
    }

    case WM_MOUSEMOVE:
    { POINTS pt = MAKEPOINTS(lParam);

      if ( (wParam & (MK_LBUTTON|MK_RBUTTON)) &&
	   (wParam & (MK_LBUTTON|MK_RBUTTON)) != (MK_LBUTTON|MK_RBUTTON) )
      { rlc_extend_selection(b, pt.x, pt.y);

	return 0;
      }

      break;
    }

					/* scrolling */
    case WM_VSCROLL:
    { switch( wParam )
      { case SB_LINEUP:
	  if ( b->window_start != b->first )
	    b->window_start = PrevLine(b, b->window_start);
	  break;
	case SB_LINEDOWN:
	  if ( b->window_start != b->last )
	    b->window_start = NextLine(b, b->window_start);
	  break;
	case SB_PAGEUP:
	{ int maxdo = rlc_count_lines(b, b->first, b->window_start);
	  int pagdo = b->window_size - 1;
	  b->window_start = rlc_add_lines(b, b->window_start,
					  -min(maxdo, pagdo));
	  break;
	}
	case SB_PAGEDOWN:
	{ int maxup = rlc_count_lines(b, b->window_start, b->last);
	  int pagup = b->window_size - 1;
	  b->window_start = rlc_add_lines(b, b->window_start,
					  min(maxup, pagup));
	  break;
	}
	case SB_THUMBTRACK:
	  b->window_start = rlc_add_lines(b, b->first, LOWORD(lParam));
	  break;
      }

      rlc_update_scrollbar(b);
      InvalidateRect(hwnd, NULL, FALSE);
    }

    case WM_TIMER:
      if ( _rlc_timer_hook && wParam >= RLC_APPTIMER_ID )
      { (*_rlc_timer_hook)((int) wParam);

	return 0;
      }
      break;

    case WM_RENDERALLFORMATS:
      if ( _rlc_render_all_hook )
      { (*_rlc_render_all_hook)();

        return 0;
      }
      break;

    case WM_RENDERFORMAT:
      if ( _rlc_render_hook && (*_rlc_render_hook)(wParam) )
        return 0;

      break;

    case WM_DESTROY:
      PostQuitMessage(0);
      return 0;
  }

  return DefWindowProc(hwnd, message, wParam, lParam);
}


void
rlc_dispatch(RlcQueue q)
{ MSG msg;
  RlcQueue oldq = _rlc_queue;

  if ( q )
    _rlc_queue = q;

  if ( GetMessage(&msg, NULL, 0, 0) )
  { TranslateMessage(&msg);
    DispatchMessage(&msg);
    _rlc_queue = oldq;
    return;
  }

  _rlc_queue = oldq;
  longjmp(_rlc_jmp_context, TagInt(0));
}


void
rlc_check_intr()
{ MSG msg;

  if ( PeekMessage(&msg, _rlc_stdio->window,
		   WM_KEYFIRST, WM_KEYLAST, PM_REMOVE) )
  { TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
}


RlcQueue
rlc_input_queue()
{ return _rlc_stdio->queue;
}


		 /*******************************
		 *	 CHARACTER TYPES	*
		 *******************************/

#define iswordchar(c) (_rlc_word_chars[(int)(c)])

static void
rlc_init_word_chars()
{ int i;

  for(i=0; i<CHAR_MAX; i++)
    _rlc_word_chars[i] = (isalnum(i) || i == '_') ? TRUE : FALSE;
}


void
rlc_word_char(int chr, int isword)
{ if ( chr > 0 && chr < CHAR_MAX )
    _rlc_word_chars[chr] = isword;
}


		 /*******************************
		 *	    SELECTION		*
		 *******************************/

#define SelLT(l1, c1, l2, c2) ((l1) < (l2) || (l1) == (l2) && (c1) < (c2))
#define SelEQ(l1, c1, l2, c2) ((l1) == (l2) && (c1) == (c2))

static int
rlc_min(RlcData b, int x, int y)
{ if ( rlc_count_lines(b, b->first, x) < rlc_count_lines(b, b->first, y) )
    return x;

  return y;
}


static int
rlc_max(RlcData b, int x, int y)
{ if ( rlc_count_lines(b, b->first, x) > rlc_count_lines(b, b->first, y) )
    return x;

  return y;
}


static void
rlc_changed_line(RlcData b, int i, int mask)
{ b->lines[i].changed |= mask;
}


static void
rlc_set_selection(RlcData b, int sl, int sc, int el, int ec)
{ int sch = rlc_min(b, sl, b->sel_start_line);
  int ech = rlc_max(b, el, b->sel_end_line);
  int i;
  int innow  = FALSE;
  int insoon = FALSE;

					/* find the lines that changed */
  for(i=sch; ; i = NextLine(b, i))
  { if ( i == sl )
    { insoon = TRUE;
      if ( i == b->sel_start_line )
      { innow = TRUE;
	if ( sc != b->sel_start_char )
	  rlc_changed_line(b, i, CHG_CHANGED);
      } else
	rlc_changed_line(b, i, CHG_CHANGED);
    } else if ( i == b->sel_start_line )
    { innow = TRUE;
      rlc_changed_line(b, i, CHG_CHANGED);
    }

    if ( i == el )
    { insoon = FALSE;
      if ( i == b->sel_end_line )
      { innow = FALSE;
	if ( ec != b->sel_end_char )
	  rlc_changed_line(b, i, CHG_CHANGED);
      } else
	rlc_changed_line(b, i, CHG_CHANGED);
    } else if ( i == b->sel_end_line )
    { innow = FALSE;
      rlc_changed_line(b, i, CHG_CHANGED);
    }

    if ( innow != insoon )
      rlc_changed_line(b, i, CHG_CHANGED);

    if ( i == ech )
      break;
  }

					/* update the attributes */
  b->sel_start_line = sl;
  b->sel_start_char = sc;
  b->sel_end_line   = el;
  b->sel_end_char   = ec;

					/* ... and request a repaint */
  rlc_request_redraw(b);
}


void
rlc_translate_mouse(RlcData b, int x, int y, int *line, int *chr)
{ *line = rlc_add_lines(b, b->window_start, y / b->ch);
  *chr  = (x - b->cw) / b->cw;
}


static void
rlc_start_selection(RlcData b, int x, int y)
{ int l, c;

  rlc_translate_mouse(b, x, y, &l, &c);
  b->sel_unit = SEL_CHAR;
  b->sel_org_line = l;
  b->sel_org_char = c;
  rlc_set_selection(b, l, c, l, c);
}


static void
rlc_end_selection(RlcData b, int x, int y)
{ int l, c;

  rlc_translate_mouse(b, x, y, &l, &c);
  if ( SelLT(l, c, b->sel_org_line, b->sel_org_char) )
    rlc_set_selection(b, l, c, b->sel_org_line, b->sel_org_char);
  else if ( SelLT(b->sel_org_line, b->sel_org_char, l, c) )
    rlc_set_selection(b, b->sel_org_line, b->sel_org_char, l, c);
  rlc_set_selection(b, l, c, l, c);
}


static int				/* v >= f && v <= t */
rlc_between(RlcData b, int f, int t, int v) 
{ int h = rlc_count_lines(b, b->first, v);

  if ( h >= rlc_count_lines(b, b->first, f) &&
       h <= rlc_count_lines(b, b->first, t) )
    return TRUE;

  return FALSE;
}


static void
rlc_word_selection(RlcData b, int x, int y)
{ int l, c;

  rlc_translate_mouse(b, x, y, &l, &c);
  if ( rlc_between(b, b->first, b->last, l) )
  { TextLine tl = &b->lines[l];

    if ( c < tl->size && iswordchar(tl->text[c]) )
    { int f, t;

      for(f=c; f>0 && iswordchar(tl->text[f-1]); f--)
	;
      for(t=c; t<tl->size && iswordchar(tl->text[t]); t++)
	;
      rlc_set_selection(b, l, f, l, t);
    }
  }

  b->sel_unit = SEL_WORD;
}


static void
rlc_extend_selection(RlcData b, int x, int y)
{ int l, c;

  rlc_translate_mouse(b, x, y, &l, &c);
  if ( SelLT(l, c, b->sel_org_line, b->sel_org_char) )
  { if ( b->sel_unit == SEL_WORD )
    { if ( rlc_between(b, b->first, b->last, l) )
      { TextLine tl = &b->lines[l];

	if ( c < tl->size && iswordchar(tl->text[c]) )
	  for(; c > 0 && iswordchar(tl->text[c-1]); c--)
	    ;
      }
    } else if ( b->sel_unit == SEL_LINE )
      c = 0;
    rlc_set_selection(b, l, c, b->sel_end_line, b->sel_end_char);
  } else if ( SelLT(b->sel_org_line, b->sel_org_char, l, c) )
  { if ( b->sel_unit == SEL_WORD )
    { if ( rlc_between(b, b->first, b->last, l) )
      { TextLine tl = &b->lines[l];

	if ( c < tl->size && iswordchar(tl->text[c]) )
	  for(; c < tl->size && iswordchar(tl->text[c]); c++)
	    ;
      }
    } else if ( b->sel_unit == SEL_LINE )
      c = b->width;
    rlc_set_selection(b, b->sel_start_line, b->sel_start_char, l, c);
  }
}


static char *
rlc_selection(RlcData b)
{ if ( SelEQ(b->sel_start_line, b->sel_start_char,
	     b->sel_end_line,   b->sel_end_char) )
    return NULL;
  else
  { int bufsize = 256;
    char *buf = malloc(bufsize);
    int i = 0;
    int l = b->sel_start_line;
    int c = b->sel_start_char;

    for( ; ; c = 0, l = NextLine(b, l))
    { TextLine tl = &b->lines[l];
      if ( tl )
      { int e = (l == b->sel_end_line ? b->sel_end_char : tl->size);

	if ( e > tl->size )
	  e = tl->size;

	while(c < e)
	{ if ( i >= bufsize )
	  { bufsize *= 2;
	    if ( !(buf = realloc(buf, bufsize)) )
	      return NULL;		/* not enough memory */
	  }
	  buf[i++] = tl->text[c++];
	}
      }
	
      if ( l == b->sel_end_line || l == b->last )
      { buf[i++] = '\0';
	return realloc(buf, i);
      }

      if ( tl && !tl->softreturn )
      { if ( i >= bufsize )
	{ bufsize *= 2;
	  if ( !(buf = realloc(buf, bufsize)) )
	    return NULL;		/* not enough memory */
	}
	buf[i++] = '\n';
      }
    }
  }
}


static void
rlc_copy(RlcData b)
{ char *sel = rlc_selection(b);

  if ( sel )
  { int size = strlen(sel);
    HGLOBAL mem = GlobalAlloc(GMEM_MOVEABLE, size + 1);
    char far *data;
    int i;

    if ( !mem )
    { MessageBox(NULL, "Not enough memory to paste", "Error", MB_OK);
      return;
    }
    data = MK_FP32(GlobalLock(mem));

    for(i=0; i<size; i++)
      *data++ = sel[i];
    *data = '\0';

    GlobalUnlock(mem);
    OpenClipboard(b->window);
    EmptyClipboard();
    SetClipboardData(CF_TEXT, mem);
    CloseClipboard();

    free(sel);
  }
}



		 /*******************************
		 *           REPAINT		*
		 *******************************/

static void
rlc_place_caret(RlcData b)
{ if ( b->has_focus )
  { int line = rlc_count_lines(b, b->window_start, b->caret_y);

    if ( line < b->window_size )
    { SetCaretPos((b->caret_x + 1) * b->cw, line * b->ch);
      if ( !b->caret_is_shown )
      { ShowCaret(b->window);
	b->caret_is_shown = TRUE;

	return;
      }
    } else
    { if ( b->caret_is_shown == TRUE )
      { HideCaret(b->window);
	b->caret_is_shown = FALSE;
      }
    }
  }

  b->caret_is_shown = FALSE;
}


static void
rlc_update_scrollbar(RlcData b)
{ int nsb_lines = rlc_count_lines(b, b->first, b->last);
  int nsb_start = rlc_count_lines(b, b->first, b->window_start);

  if ( nsb_lines != b->sb_lines ||
       nsb_start != b->sb_start )
  { SetScrollRange(b->window, SB_VERT, 0, nsb_lines, FALSE);
    SetScrollPos(  b->window, SB_VERT, nsb_start, TRUE);

    b->sb_lines = nsb_lines;
    b->sb_start = nsb_start;
  }
}


static void
rlc_redraw(RlcData b)
{ PAINTSTRUCT ps;
  HDC hdc = BeginPaint(b->window, &ps);
  int sl = max(0, ps.rcPaint.top/b->ch);
  int el = min(b->window_size, ps.rcPaint.bottom/b->ch);
  int l = rlc_add_lines(b, b->window_start, sl);
  int pl = sl;				/* physical line */
  RECT rect;
  HBRUSH white = GetStockObject(WHITE_BRUSH);
  int insel = FALSE;			/* selected lines? */

  SelectObject(hdc, b->hfont);
  SetTextColor(hdc, b->foreground);
  
  if ( b->has_focus && b->caret_is_shown )
  { HideCaret(b->window);
    b->caret_is_shown = FALSE;
  }

  if ( rlc_count_lines(b, b->first, b->sel_start_line) <
       rlc_count_lines(b, b->first, l) &&
       rlc_count_lines(b, b->first, b->sel_end_line) >
       rlc_count_lines(b, b->first, l) )
    insel = TRUE;

  for(; pl <= el; l = NextLine(b, l), pl++)
  { TextLine tl = &b->lines[l];

    if ( tl->text )
      TextOut(hdc, b->cw, b->ch*pl, tl->text, tl->size);
    else
      tl->size = 0;

    rect.top    = b->ch * pl;
    rect.bottom = rect.top + b->ch;
      
    if ( tl->size < b->width )
    { rect.left   = (tl->size+1) * b->cw;
      rect.right  = (b->width+1) * b->cw;
      FillRect(hdc, &rect, white);
    }
    tl->changed = CHG_RESET;

					/* selection start line */
    if ( l == b->sel_start_line )
    { int cf = b->sel_start_char;
      int ce = (b->sel_end_line != b->sel_start_line ? b->width
						     : b->sel_end_char);
      if ( cf != ce )
      { rect.left   = b->cw * (cf+1);
	rect.right  = b->cw * (ce+1);

	InvertRect(hdc, &rect);
      }

      if ( l != b->sel_end_line )
	insel = TRUE;
    } else if ( l == b->sel_end_line )	/* end of selection */
    { insel = FALSE;
      rect.left   = b->cw;
      rect.right  = b->cw * (b->sel_end_char+1);

      InvertRect(hdc, &rect);
    } else if ( insel )			/* entire line in selection */
    { rect.left   = b->cw;
      rect.right  = b->cw * (b->width+1);

      InvertRect(hdc, &rect);
    }

    if ( l == b->last )
    { rect.left   = b->cw;
      rect.right  = b->width * (b->cw+1);
      rect.top    = b->ch * (pl+1);
      rect.bottom = b->ch * (el+1);
      FillRect(hdc, &rect, white);

      break;
    }
  }
  rlc_place_caret(b);

  b->changed = CHG_RESET;

  EndPaint(b->window, &ps);

  rlc_update_scrollbar(b);
}


static void
rlc_request_redraw(RlcData b)
{ if ( b->changed & CHG_CHANGED )
  { if ( b->window )
      InvalidateRect(b->window, NULL, FALSE);
  } else
  { int i = b->window_start;
    int y = 0;
    RECT rect;
    int first = TRUE;
    int clear = FALSE;

    rect.left = b->cw;
    rect.right = (b->width+1) * b->cw;
    
    for(; y < b->window_size; y++, i = NextLine(b, i))
    { TextLine l = &b->lines[i];
      
      if ( l->changed & CHG_CHANGED )
      { if ( first )
	{ rect.top = y * b->ch;
	  rect.bottom = rect.top + b->ch;
	  first = FALSE;
	} else
	  rect.bottom = (y+1) * b->ch;

	if ( l->changed & CHG_CLEAR )
	  clear = TRUE;
      }
      if ( i == b->last )
	break;
    }

    if ( !first && b->window )
      InvalidateRect(b->window, &rect, FALSE); /*clear);*/
    else if ( b->changed & CHG_CARET )
      rlc_place_caret(b);
  }
}


static void
rlc_normalise(RlcData b)
{ if ( rlc_count_lines(b, b->window_start, b->caret_y) >= b->window_size )
  { b->window_start = rlc_add_lines(b, b->caret_y, -(b->window_size-1));
    b->changed |= CHG_CARET|CHG_CLEAR|CHG_CHANGED;
    rlc_request_redraw(b);
  }
}


static void
rlc_resize_pixel_units(RlcData b, int w, int h)
{ rlc_resize(b, max(20, w/b->cw)-2, max(1, h/b->ch));
#ifdef RDEF_STDIO
  raise(SIGWINCH);
#endif

  rlc_request_redraw(b);
}

static void
rlc_init_text_dimensions(RlcData b)
{ HDC hdc;
  TEXTMETRIC tm;

  b->hfont = GetStockObject(SYSTEM_FIXED_FONT);
  hdc = GetDC(NULL);
  SelectObject(hdc, b->hfont);
  GetTextMetrics(hdc, &tm);
  b->cw = tm.tmAveCharWidth;
  b->cb = tm.tmHeight;
  b->ch = tm.tmHeight + tm.tmExternalLeading;
  ReleaseDC(NULL, hdc);
}

static RlcData
rlc_make_buffer(int w, int h)
{ RlcData b = malloc(sizeof(rlc_data));
  int i;

  b->height         = h;
  b->width          = w;
  b->first          = b->last = 0;
  b->caret_x	    = 0;
  b->caret_y        = 0;
  b->window_start   = 0;
  b->window_size    = 25;		/* default? */
  b->lines          = malloc(sizeof(text_line) * h);
  b->cmdstat	    = CMD_INITIAL;
  b->changed	    = CHG_CARET|CHG_CHANGED|CHG_CLEAR;
  b->sel_start_line = 0;
  b->sel_start_char = 0;
  b->sel_end_line   = 0;
  b->sel_end_char   = 0;
  b->window	    = 0;
    
  for(i=0; i<h; i++)
  { b->lines[i].text = NULL;
    b->lines[i].size = 0;
    b->lines[i].adjusted = TRUE;
    b->lines[i].softreturn = FALSE;
  }

  rlc_init_word_chars();

  return b;
}


static void
rlc_shift_lines_down(RlcData b, int line)
{ int i = b->first;
  int p = PrevLine(b, i);

  if ( p != b->last )
  { b->first = p;
    b->lines[p] = b->lines[i];
  }
  for(p=i, i = NextLine(b, i); p != line; p=i, i = NextLine(b, i))
    b->lines[p] = b->lines[i];

  b->lines[line].text = NULL;
}


static void
rlc_shift_lines_up(RlcData b, int line)
{ int prev = PrevLine(b, line);

  while(line != b->first)
  { b->lines[line] = b->lines[prev];
    line = prev;
    prev = PrevLine(b, prev);
  }

  b->first = NextLine(b, b->first);
}



static void
rlc_resize(RlcData b, int w, int h)
{ int i;

  b->window_size = h;
  b->width = w;
  
  for(i = b->first; i != b->last; i = NextLine(b, i))
  { TextLine tl = &b->lines[i];

    if ( tl->text && tl->adjusted == FALSE )
      rlc_adjust_line(b, i);

    if ( tl->size > w )
    { if ( !tl->softreturn )		/* hard --> soft */
      { TextLine pl;

	rlc_shift_lines_down(b, i);
	pl = &b->lines[PrevLine(b, i)];
	tl->text = malloc(pl->size - w);
	tl->adjusted = TRUE;
	strncpy(tl->text, &pl->text[w], pl->size - w);
	pl->softreturn = TRUE;
	tl->size = pl->size - w;
	pl->size = w;
	i = pl - b->lines;
      } else				/* put in next line */
      { TextLine nl;

	if ( i == b->last )
	{ b->last = NextLine(b, b->last);
	  rlc_open_line(b);
	}
	nl = &b->lines[NextLine(b, i)];
	nl->text = realloc(nl->text, nl->size + tl->size - w);
	strncpy(&nl->text[tl->size - w], nl->text, nl->size);
	strncpy(nl->text, &tl->text[w], tl->size - w);
	tl->size = w;
      }	
    } else if ( tl->text && tl->softreturn && tl->size < w )
    { TextLine nl;

      if ( i == b->last )
      { b->last = NextLine(b, b->last);
	rlc_open_line(b);
      }
      nl = &b->lines[NextLine(b, i)];

      nl->text = realloc(nl->text, nl->size + tl->size);
      strncpy(&nl->text[tl->size], nl->text, nl->size);
      strncpy(nl->text, tl->text, tl->size);
      nl->size += tl->size;
      nl->adjusted = TRUE;
      rlc_shift_lines_up(b, i);
    }
  }

  if ( rlc_count_lines(b, b->first, b->last) < h )
    b->window_start = b->first;
  else
    b->window_start = rlc_add_lines(b, b->last, -(h-1));

  b->caret_y = b->last;
  b->caret_x = b->lines[b->last].size;

  b->changed |= CHG_CARET|CHG_CHANGED|CHG_CLEAR;
}


static void
rlc_free_line(RlcData b, int line)
{ if ( b->lines[line].text )
  { free(b->lines[line].text);
    b->lines[line].text = NULL;
  }
}


static void
rlc_adjust_line(RlcData b, int line)
{ TextLine tl = &b->lines[line];

  if ( tl->text && !tl->adjusted )
  { tl->text = realloc(tl->text, tl->size == 0 ? 4 : tl->size);	/* TBD */
    tl->adjusted = TRUE;
  }
}


static void
rlc_unadjust_line(RlcData b, int line)
{ TextLine tl = &b->lines[line];

  if ( tl->text )
  { if ( tl->adjusted )
    { tl->text = realloc(tl->text, b->width + 1);
      tl->adjusted = FALSE;
    }
  } else
  { tl->text = malloc(b->width + 1);
    tl->adjusted = FALSE;
    tl->size = 0;
  }
}


static void
rlc_open_line(RlcData b)
{ int i = b->last;

  if ( NextLine(b, i) == b->first )
  { rlc_free_line(b, b->first);
    b->first = NextLine(b, b->first);
  }
  b->lines[i].text = malloc(b->width + 1);
  b->lines[i].adjusted = FALSE;
  b->lines[i].size = 0;
  b->lines[i].softreturn = FALSE;
}


		 /*******************************
		 *	   CALCULATIONS		*
		 *******************************/

static int
rlc_count_lines(RlcData b, int from, int to)
{ if ( to >= from )
    return to-from;

  return to + b->height - from;
}


static int
rlc_add_lines(RlcData b, int here, int add)
{ here += add;
  while ( here < 0 )
    here += b->height;
  while ( here >= b->height )
    here -= b->height;

  return here;
}


		 /*******************************
		 *    ANSI SEQUENCE HANDLING	*
		 *******************************/

static void
rlc_need_arg(RlcData b, int arg, int def)
{ if ( b->argc < arg )
  { b->argv[arg-1] = def;
    b->argc = arg;
  }
}


static void
rlc_caret_up(RlcData b, int arg)
{ while(arg-- > 0 && b->caret_y != b->first)
    b->caret_y = PrevLine(b, b->caret_y);

  b->changed |= CHG_CARET;
}


static void
rlc_caret_down(RlcData b, int arg)
{ while ( arg-- > 0 )
  { if ( b->caret_y == b->last )
    { b->last = NextLine(b, b->last);
      rlc_open_line(b);
    }
    b->caret_y = NextLine(b, b->caret_y);
    b->lines[b->caret_y].softreturn = FALSE;
  }
  b->changed |= CHG_CARET;
					/* scroll? */
  if ( rlc_count_lines(b, b->window_start, b->caret_y) >= b->window_size )
  { b->window_start = rlc_add_lines(b, b->caret_y, -(b->window_size-1));
    b->changed |= CHG_CHANGED|CHG_CLEAR;
  }
}


static void
rlc_caret_forward(RlcData b, int arg)
{ while(arg-- > 0)
  { if ( ++b->caret_x == b->width )
    { b->lines[b->caret_y].softreturn = TRUE;
      rlc_caret_down(b, 1);
      b->caret_x = 0;
    }
  }

  b->changed |= CHG_CARET;
}


static void
rlc_caret_backward(RlcData b, int arg)
{ while(arg-- > 0)
  { if ( b->caret_x-- == 0 )
    { rlc_caret_up(b, 1);
      b->caret_x = b->width;
    }
  }

  b->changed |= CHG_CARET;
}


static void
rlc_cariage_return(RlcData b)
{ b->caret_x = 0;

  b->changed |= CHG_CARET;
} 


static void
rlc_tab(RlcData b)
{ TextLine tl = &b->lines[b->caret_y];

  do
  { rlc_caret_forward(b, 1);
  } while( (b->caret_x % 8) != 0 );

  while ( tl->size < b->caret_x )
    tl->text[tl->size++] = ' ';

  b->changed |= CHG_CARET;
}


static void
rlc_set_caret(RlcData b, int x, int y)
{ int cy = rlc_count_lines(b, b->window_start, b->caret_y);

  y = Bounds(y, 0, b->window_size);

  if ( y < cy )
    b->caret_y = rlc_add_lines(b, b->window_start, y);
  else
    rlc_caret_down(b, y-cy);

  b->caret_x = Bounds(x, 0, b->width-1);

  b->changed |= CHG_CARET;
}


static void
rlc_save_caret_position(RlcData b)
{ b->scaret_y = rlc_count_lines(b, b->window_start, b->caret_y);
  b->scaret_x = b->caret_x;
}


static void
rlc_restore_caret_position(RlcData b)
{ rlc_set_caret(b, b->scaret_x, b->scaret_y);
}


static void
rlc_erase_display(RlcData b)
{ int i = b->window_start;
  int last = rlc_add_lines(b, b->window_start, b->window_size);

  do
  { b->lines[i].size = 0;
    i = NextLine(b, i);
  } while ( i != last );

  b->changed |= CHG_CHANGED|CHG_CLEAR|CHG_CARET;

  rlc_set_caret(b, 0, 0);
}


static void
rlc_erase_line(RlcData b)
{ TextLine tl = &b->lines[b->caret_y];

  tl->size = b->caret_x;
  tl->changed |= CHG_CHANGED|CHG_CLEAR;
}


static void
rlc_putchar(RlcData b, int chr)
{ TextLine tl = &b->lines[b->caret_y];

  rlc_unadjust_line(b, b->caret_y);
  while( tl->size < b->caret_x )
    tl->text[tl->size++] = ' ';
  tl->text[b->caret_x] = chr;
  if ( tl->size <= b->caret_x )
    tl->size = b->caret_x + 1;
  tl->changed |= CHG_CHANGED;

  rlc_caret_forward(b, 1);
}


static void
rlc_putansi(RlcData b, int chr)
{ switch(b->cmdstat)
  { case CMD_INITIAL:
      switch(chr)
      { case '\b':
	  rlc_caret_backward(b, 1);
	  break;
        case Control('G'):
	  MessageBeep(MB_ICONEXCLAMATION);
	  break;
	case '\r':
	  rlc_cariage_return(b);
	  break;
	case '\n':
	  rlc_caret_down(b, 1);
	  break;
	case '\t':
	  rlc_tab(b);
	  break;
	case 27:			/* ESC */
	  b->cmdstat = CMD_ESC;
	  break;
	default:
	  rlc_putchar(b, chr);
	  break;
      }
      break;
    case CMD_ESC:
      switch(chr)
      { case '[':
	  b->cmdstat = CMD_ANSI;
	  b->argc    = 0;
	  b->argstat = 0;		/* no arg */
	  break;
	default:
	  b->cmdstat = CMD_INITIAL;
	  break;
      }
      break;
    case CMD_ANSI:
      if ( isdigit(chr) )
      { if ( !b->argstat )
	{ b->argv[b->argc] = (chr - '0');
	  b->argstat = 1;		/* positive */
	} else
	{ b->argv[b->argc] = b->argv[b->argc] * 10 + (chr - '0');
	}

	break;
      } 
      if ( !b->argstat && chr == '-' )
      { b->argstat = -1;		/* negative */
	break;
      }
      if ( b->argstat )
      { b->argv[b->argc] *= b->argstat;
	b->argc++;
	b->argstat = 0;
      }
      switch(chr)
      { case ';':
	  break;			/* wait for more args */
	case 'H':
	case 'f':
	  rlc_need_arg(b, 1, 0);
	  rlc_need_arg(b, 2, 0);
	  rlc_set_caret(b, b->argv[0], b->argv[1]);
	  break;
	case 'A':
	  rlc_need_arg(b, 1, 1);
	  rlc_caret_up(b, b->argv[0]);
	  break;
	case 'B':
	  rlc_need_arg(b, 1, 1);
	  rlc_caret_down(b, b->argv[0]);
	  break;
	case 'C':
	  rlc_need_arg(b, 1, 1);
	  rlc_caret_forward(b, b->argv[0]);
	  break;
	case 'D':
	  rlc_need_arg(b, 1, 1);
	  rlc_caret_backward(b, b->argv[0]);
	  break;
	case 's':
	  rlc_save_caret_position(b);
	  break;
	case 'u':
	  rlc_restore_caret_position(b);
	  break;
	case 'J':
	  if ( b->argv[0] == 2 )
	    rlc_erase_display(b);
	  break;
	case 'K':
	  rlc_erase_line(b);
	  break;
      }
      b->cmdstat = CMD_INITIAL;
  }
}


		 /*******************************
		 *	      CUT/PASTE		*
		 *******************************/

static void
rlc_paste(RlcData b)
{ HGLOBAL mem;

  OpenClipboard(b->window);
  if ( (mem = GetClipboardData(CF_TEXT)) )
  { char far *data = MK_FP32(GlobalLock(mem));
    int i;
    RlcQueue q = (_rlc_queue ? _rlc_queue : b->queue);

    if ( q )
      for(i=0; data[i]; i++)
	rlc_add_queue(q, data[i]);

    GlobalUnlock(mem);
  }
  CloseClipboard();
}


		 /*******************************
		 *	  WATCOM/DOS I/O	*
		 *******************************/

int
getch()
{ RlcQueue q = _rlc_stdio->queue;

  if ( _rlc_update_hook )
    (*_rlc_update_hook)();

  while( rlc_is_empty_queue(q) )
    rlc_dispatch(q);

  return rlc_from_queue(q);
}


int
getche()
{ int chr = getch();

  rlc_putansi(_rlc_stdio, chr);
  return chr;
}


		 /*******************************
		 *        GO32 FUNCTIONS	*
		 *******************************/

int
getkey()
{ return getch();
}


void
ScreenInit()
{
}


int
kbhit()
{ return !rlc_is_empty_queue(_rlc_stdio->queue);
}


void
ScreenGetCursor(int *row, int *col)
{ RlcData b = _rlc_stdio;

  *row = rlc_count_lines(b, b->window_start, b->caret_y) + 1;
  *col = b->caret_x + 1;
}


void
ScreenSetCursor(int row, int col)
{ RlcData b = _rlc_stdio;

  rlc_set_caret(b, col-1, row-1);
}


int
ScreenCols()
{ return _rlc_stdio->width;
}


int
ScreenRows()
{ return _rlc_stdio->window_size;
}

		 /*******************************
		 *	      QUEUE		*
		 *******************************/

#define QN(q, i) ((i)+1 >= (q)->size ? 0 : (i)+1)


RlcQueue
rlc_make_queue(int size)
{ RlcQueue q;

  if ( (q = malloc(sizeof(rlc_queue))) )
  { q->first = q->last = 0;
    q->size = size;

    if ( (q->buffer = malloc(sizeof(short) * size)) )
      return q;
  }

  return NULL;				/* not enough memory */
}


void
rlc_free_queue(RlcQueue q)
{ if ( q )
  { if ( q->buffer )
      free(q->buffer);
    free(q);
  }
} 


static int
rlc_add_queue(RlcQueue q, int chr)
{ if ( QN(q, q->last) != q->first )
  { q->buffer[q->last] = chr;
    q->last = QN(q, q->last);

    return TRUE;
  }

  return FALSE;
}


int
rlc_is_empty_queue(RlcQueue q)
{ if ( q->first == q->last )
    return TRUE;

  return FALSE;
}


void
rlc_empty_queue(RlcQueue q)
{ q->first = q->last = 0;
}


static int
rlc_from_queue(RlcQueue q)
{ if ( q->first != q->last )
  { int chr = q->buffer[q->first];

    q->first = QN(q, q->first);

    return chr;
  }

  return -1;
}


		 /*******************************
		 *	     UNIX I/O		*
		 *******************************/

int
rlc_read(char *buf, int count)
{ int rd = 0;
  char *ptr = buf;
  RlcData b = _rlc_stdio;

  while(rd < count)
  { int chr = getch();

    switch(chr)
    { case Control('C'):
	raise(SIGINT);
	continue;
      case Control('D'):
	if ( rd == 0 )
	  return 0;			/* end-of-file */
        break;
      case '\b':			/* erase character */
	if ( rd > 0 )
	{ rlc_caret_backward(b, 1);
	  rlc_erase_line(b);
	  rd--;
	  ptr--;
	  rlc_request_redraw(b);
	}
        continue;
      case Control('U'):		/* erase line */
        rlc_cariage_return(b);
	rlc_erase_line(b);
	rlc_request_redraw(b);
	rd = 0;
	ptr = buf;
	continue;
      case Control('W'):		/* erase word */
      { int ord = rd;

	while(rd > 0 && !isalnum(buf[rd-1]))
	  rd--;
	while(rd > 0 && isalnum(buf[rd-1]))
	  rd--;
	rlc_caret_backward(b, ord-rd);
	rlc_erase_line(b);
	ptr = &buf[rd];
	rlc_request_redraw(b);
	continue;
      }
      case Control('R'):
	b->changed |= CHG_CLEAR|CHG_CHANGED|CHG_CARET;
	rlc_request_redraw(b);
	continue;
      case '\r':			/* return: output */
	chr = '\n';
      case '\n':
	rlc_cariage_return(b);
	rlc_caret_down(b, 1);
	rlc_request_redraw(b);
	UpdateWindow(b->window);
	*ptr++ = chr;
        rd++;
	return rd;
    }

    *ptr++ = chr;
    rlc_putansi(b, chr);	/* or rlc_putchar()? */
    rd++;
    rlc_request_redraw(b);
  }

  return rd;
}


int
rlc_write(char *buf, unsigned int count)
{ unsigned int n = 0;
  RlcData b = _rlc_stdio;
  char *s = buf;

  while(n++ < count)
  { int chr = *s++;

    if ( chr == '\n' )
      rlc_putansi(b, '\r');
    rlc_putansi(b, chr);
  }

  if ( _rlc_copy_output_to_debug_output )
  { char *b2 = alloca(count*2+1);
    char *q;

    for(q=b2, s=buf; count > 0; count--)
    { if ( *s == '\n' )
	*q++ = '\r';
      *q++ = *s++;
    }
    *q = '\0';
    OutputDebugString(b2);
  }

  rlc_normalise(b);
  rlc_request_redraw(b);
  UpdateWindow(b->window);

  return count;
}

		 /*******************************
		 *	    MISC STUFF		*
		 *******************************/

#define RLC_TITLE_MAX 256

static char current_title[RLC_TITLE_MAX];

void
rlc_title(char *title, char *old, int size)
{ if ( old )
    strncpy(old, current_title, size);

  if ( title )
  { SetWindowText(_rlc_stdio->window, title);

    strncpy(current_title, title, RLC_TITLE_MAX);
  }
}


void
rlc_icon(HICON icon)
{ SetClassLong(rlc_hwnd(), GCL_HICON, (LONG) icon);
}


HANDLE
rlc_hinstance()
{ return _rlc_hinstance;
}


HWND
rlc_hwnd()
{ return _rlc_stdio->window;
}

		 /*******************************
		 *	 SETTING OPTIONS	*
		 *******************************/

int
rlc_copy_output_to_debug_output(int new)
{ int old = _rlc_copy_output_to_debug_output;

  _rlc_copy_output_to_debug_output = new;

  return old;
}

RlcUpdateHook
rlc_update_hook(RlcUpdateHook new)
{ RlcUpdateHook old = _rlc_update_hook;

  _rlc_update_hook = new;
  return old;
}

RlcTimerHook
rlc_timer_hook(RlcTimerHook new)
{ RlcTimerHook old = _rlc_timer_hook;

  _rlc_timer_hook = new;
  return old;
}

RlcRenderHook
rlc_render_hook(RlcRenderHook new)
{ RlcRenderHook old = _rlc_render_hook;

  _rlc_render_hook = new;
  return old;
}

RlcRenderAllHook
rlc_render_all_hook(RlcRenderAllHook new)
{ RlcRenderAllHook old = _rlc_render_all_hook;

  _rlc_render_all_hook = new;
  return old;
}


		 /*******************************
		 *	     READ/WRITE		*
		 *******************************/

#ifdef REDEFINE_STDIO

int
read(int fd, void *buf, unsigned int count)
{ if ( fd == fileno(stdin) )
    return rlc_read(buf, count);
  else
    return _read(fd, buf, count);
}


int
write(int fd, void *buf, unsigned int count)
{ if ( fd <= 2 )		/* stdio */
  { /*fflush(stdout);
    fflush(stderr); */

    return rlc_write(buf, count);
  } else
    return _write(fd, buf, count);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
The  functions  cwrite()  and  cread()  are  called  from  the  modified
fflush.obj and _filbuf.obj files from the   MSVC20  files.  They capture
stream-based I/O.
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int
cwrite(int fd, void *buf, unsigned int count)
{ if ( fd <= 2 )		/* stdio */
    return rlc_write(buf, count);
  else
    return _write(fd, buf, count);
}


int
cread(int fd, void *buf, unsigned int count)
{ printf("cread(%d, %d, %d)\n", fd, buf, count);
  printf("stdin = %d\n", stdin);
  printf("stdin->_ptr = %d\n", stdin->_ptr);
  printf("stdin->_cnt = %d\n", stdin->_cnt);
  printf("stdin->_base = %d\n", stdin->_base);
  printf("stdin->_flag = %d\n", stdin->_flag);
  printf("stdin->_file = %d\n", stdin->_file);
  printf("stdin->_charbuf = %d\n", stdin->_charbuf);
  printf("stdin->_bufsiz = %d\n", stdin->_bufsiz);
  printf("stdin->_tmpfname = %d\n", stdin->_tmpfname);

  return read(fd, buf, count);
}

#endif /*REDEFINE_STDIO*/
