/*
 *	aegis - project change supervisor
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate report lists
 */

#include <errno.h>
#include <stdio.h>
#include <ac/stdlib.h>
#include <ac/string.h>

#include <aer/list.h>
#include <arglex2.h>
#include <col.h>
#include <dir.h>
#include <error.h>
#include <gonzo.h>
#include <help.h>
#include <mem.h>
#include <os.h>
#include <rptidx.h>
#include <trace.h>
#include <word.h>




static void process _((string_ty *, char *, rptidx_where_list));

static void
process(dir, nondir, result)
	string_ty	*dir;
	char		*nondir;
	rptidx_where_list result;
{
	string_ty	*fn;
	int		err;
	rptidx		data;

	trace(("process((dir = \"%s\", nondir = \"%s\"))\n{\n"/*}*/,
		dir->str_text, nondir));
	fn = str_format("%S/%s", dir, nondir);
	os_become_orig();
	err = os_readable(fn);
	if (err)
	{
		os_become_undo();
		str_free(fn);
		trace((/*{*/"}\n"));
		return;
	}
	data = rptidx_read_file(fn->str_text);
	os_become_undo();
	str_free(fn);
	if (data->where)
	{
		size_t		j;

		for (j = 0; j < data->where->length; ++j)
		{
			rptidx_where	in;
			rptidx_where	out;
			rptidx_where	*out_p;
			type_ty		*type_p;

			in = data->where->list[j];
			if (!in->name || !in->filename)
				continue;
			assert(result);
			out_p =
				rptidx_where_list_type.list_parse
				(
					result,
					&type_p
				);
			assert(type_p == &rptidx_where_type);
			out = rptidx_where_type.alloc();
			*out_p = out;
			out->name = str_copy(in->name);
			if (in->description)
				out->description = str_copy(in->description);
			else
				out->description = str_copy(in->name);
			if (in->filename->str_text[0] == '/')
				out->filename = str_copy(in->filename);
			else
			{
				out->filename =
					str_format("%S/%S", dir, in->filename);
			}
			trace(("out->name = \"%s\";\n", out->name->str_text));
		}
	}
	rptidx_type.free(data);
	trace((/*{*/"}\n"));
}




static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	rptidx_where	a;
	rptidx_where	b;

	a = *(rptidx_where *)va;
	b = *(rptidx_where *)vb;
	return strcasecmp(a->name->str_text, b->name->str_text);
}


void
report_list(usage)
	void		(*usage)_((void));
{
	wlist		path;
	rptidx_where_list result;
	size_t		j;
	int		name_col;
	int		desc_col;
	int		path_col;

	/*
	 * read the rest of the command line
	 */
	trace(("report_list()\n{\n"/*}*/));
	arglex();
	while (arglex_token != arglex_token_eoln)
		generic_argument(usage);

	/*
	 * get all of the report names
	 */
	gonzo_report_path(&path);
	result = rptidx_where_list_type.alloc();
	for (j = 0; j < path.wl_nwords; ++j)
	{
		process(path.wl_word[j], "report.index", result);
		process(path.wl_word[j], "report.local", result);
	}
	wl_free(&path);
	trace(("result->length = %ld;\n", result->length));
	qsort(result->list, result->length, sizeof(result->list[0]), cmp);

	/*
	 * form the columns for the output
	 */
	col_open((char *)0);
	col_title("List of Reports", (char *)0);
	name_col = col_create(0, 15);
	desc_col = col_create(16, 47);
	path_col = col_create(48, 0);
	col_heading(name_col, "Name\n------");
	col_heading(desc_col, "Description\n-------------");
	col_heading(path_col, "Script File\n-------------");

	/*
	 * name each of the reports
	 */
	for (j = 0; j < result->length; ++j)
	{
		rptidx_where	p;

		p = result->list[j];
		col_puts(name_col, p->name->str_text);
		col_puts(desc_col, p->description->str_text);
		col_puts(path_col, p->filename->str_text);
		col_eoln();
	}
	rptidx_where_list_type.free(result);
	col_close();
	trace((/*{*/"}\n"));
}
