
// Written by Greg Lewis, gregl@umich.edu
// If you release any versions of this code, please include
// the author in the credits.  Give credit where credit is due!

#include <sys/param.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "dehacked.h"
#include "input.h"
#include "linux_text.h"

#undef TEXT
#undef WIDTH
#undef HEIGHT
#include "ttyobj.h"

// Confirms that the user wants to quit if changes have been made.

EBool ConfirmQuit(void)
{
	char buffer[2];

	// Ask the user if he wants to quit, only if changes have actually been
	// made.
	if (changes == NO)
		return YES;

	if ((Printinputwindow(buffer, "You have unwritten changes.  Quit?", YESNO, 1) == -1)
		 || (tolower(buffer[0]) == 'n'))
		return NO;
	else
		return YES;
}

// Gets an input string with certain specifications.

int Getinput(int promptlen, char *string, int maxlen, int type, int x1, char firstchar)
{
	int curx, cury, curpos = 0;
	unsigned char inputchar;
	unsigned int x, y;					// Mouse x and y
#ifdef HAVE_MOUSE
	EButton lbutton, rbutton;			// State of left and right buttons
#endif
	EBool ExitLoop = NO;
	EBool escape = NO;

	// Find where to position the cursor
	if (type == STRING)
	{
		cury = 26;
		curx = x1 + 2;
	}
	else
	{
		curx = x1+4+promptlen;
		cury = 25;
	}

	// A little kludge to handle the fact that when a user is in the
	// Thing name pick-list, and types a number, it must be entered into
	// the box as the first number.  So if there is a character passed to
	// this function, make it the first one in the string.
	if (firstchar)
	{
		curpos = 1;

		string[0] = firstchar;
		gotoxy(curx, cury);
		curx++;
		putch(firstchar);
	}

	// Keep getting input until done
	while (!ExitLoop)
	{
		string[curpos] = 0;
		gotoxy(curx, cury);
		_setcursortype(_NORMALCURSOR);

		// Get some input
		if (Waitforevent(NO))
		{
			// OK, it was a keypress
			_setcursortype(_NOCURSOR);
			inputchar = getch();

			// If it's an extended key, ignore it.
			if (inputchar == 0)
			{
				getch();
				continue;
			}
			else if (inputchar == ESC)
				// Escape exits
				escape = ExitLoop = YES;
			else if (inputchar == RET)
			{
				// Return acts like Escape if there hasn't been anything
				// entered at all.
				ExitLoop = YES;
				if (curpos == 0)
					escape = YES;
			}
			else if (inputchar == BKSP)
			{
				// Get rid of the last thing typed.
				if (curpos > 0)
				{
					string[--curpos] = 0;
					gotoxy(--curx, cury);
					putch(' ');
				}
			}
		}

		// Depending on what type of input we're accepting, check to see if
		// the user entered a valid character and process it.
		switch (type)
		{
			case STRING:
				// We're inputting a string, which means anything above a
				// space and not extended is OK.
				if (inputchar >= ' ' && inputchar <= 127)
				{
					if (curpos == maxlen)
					{
						gotoxy(curx-1, cury);
						string[curpos-1] = inputchar;
					}
					else
					{
						string[curpos++] = inputchar;
						curx++;
					}
					putch(inputchar);
				}
				break;
			case LONGINT:
				// We're inputting a long integer, which may or may not have
				// a leading sign indicator.  I think this will bomb if
				// the user enters a whole bunch of zeros.
				if (((inputchar == '+' || inputchar == '-') && (curpos == 0))
					 || isdigit(inputchar))
				{
					if (atol(string) <= LONG_MAX/10L &&
						 atol(string) >= LONG_MIN/10L)
					{
						string[curpos++] = inputchar;
						curx++;
					}
					else
					{
						gotoxy(curx-1, cury);
						string[curpos-1] = inputchar;
					}
					putch(inputchar);
				}
				break;
			case YESNO:
				// We want one character, only one, no more, no less, and it
				// has got to be a 'y' or an 'n', no doubts about it.
				if (toupper(inputchar) == 'Y' || toupper(inputchar) == 'N')
				{
					putch(inputchar);
					string[curpos] = inputchar;
					string[1] = 0;
					ExitLoop = YES;
				}
				break;
		}
	}

	// Escape is true if Esc was hit (or right button clicked)
	if (escape)
	{
		string[0] = 0;
		return -1;
	}
	else
		return 0;
}

// Gets the current Thing "window", which smaller window in the big
// Thing editing screen we're on.

int GetThingWindow(int curfield)
{
	int i;

	for (i=0; i<5; i++)
		if (curfield <= fieldinfo[i][1])
			return i;

	return 0;
}

// GoPageDown goes down (for Page-Down or arrow click)

void GoPageDown(void)
{
	int last = global[mode][GMAX] - global[mode][GMIN];

	// Note: I don't wanna comment this.  Take a look at GoPageUp for
	// details.  I'm lazy.
	if (mode == THING_EDIT || mode == AMMO_EDIT)
	{
		if (global[mode][CUR] < last)
		{
			global[mode][CUR]++;
			redraw = ALLDATA;
		}
	}
	else
	{
		redraw = ALLDATA;
		if (global[mode][CUR] == last)
			redraw = NOT;
		else if (global[mode][CUR] > last-38)
			global[mode][CUR] = last;
		else
			global[mode][CUR] += 38;
		if (global[mode][TOPROW] > last-76)
			global[mode][TOPROW] = last-37;
		else
			global[mode][TOPROW] += 38;
	}
}

// GoPageUp goes up (for Page-Up or arrow click)

void GoPageUp(void)
{
	// In the Thing and Weapon editing modes, "GoPageUp" actually means
	// "advance to next object".
	if (mode == THING_EDIT || mode == AMMO_EDIT)
	{
		// If we're not at the top, go up one.
		if (global[mode][CUR] > 0)
		{
			global[mode][CUR]--;
			redraw = ALLDATA;
		}
	}
	else
	{
		// We've gotta redraw all the data (might be changed later)
		redraw = ALLDATA;

		// If we're already AT the top, don't bother redrawing.
		// If we're less than a page away from the top, go to the top.
		// If we're more than a page away from the top, go up by one page
		// (which is 38, which should be a global define BTW).
		if (global[mode][CUR] == 0)
			redraw = NOT;
		else if (global[mode][CUR] < 38)
			global[mode][CUR] = 0;
		else
			global[mode][CUR] -= 38;

		// TOPROW is the number of the object at the top of the screen.
		// Set it to the correct value.
		if (global[mode][TOPROW] < 38)
			global[mode][TOPROW] = 0;
		else
			global[mode][TOPROW] -= 38;
	}
}

// Inputs a string for the Text data section

int Inputtext(EBool showonly)
{
	char *origtext;
	char *buffer;
	EBool ExitLoop = NO, changes = NO;
	int *image, *image2;					// Background of view/input windows
	int stringlen, textoff = 0;
	int i;
	int curpos = 0, maxlen, minlen;
	char xpos, ypos, inputchar;
	char answer[2] = {0, 0};
	unsigned int x, y;					// Mouse x and y
#ifdef HAVE_MOUSE
	EButton lbutton, rbutton;			// State of left and right buttons
#endif

	// Allocate necessary memory.
	image = new int[2000];
	if (showonly == NO)
		image2 = new int[2000];

	if (image == NULL || (image2 == NULL && showonly == NO))
		AbortProg("in Inputtext");

	// Save necessary background (which is basically the whole screen).
	gettext(1, 1, 80, 25, image);
	if (showonly == NO)
		gettext(1, 27, 80, 50, image2);

	// Get a concrete number for which string we're on, from the offset
	// in the text section.
	for (i=0; i<global[TEXT_EDIT][CUR]; i++)
	{
		stringlen = strlen(&(textdatap[textoff]));
		if ( Lnx_DOOM )
			textoff += stringlen + 1;
		else
			textoff += (stringlen & (~3)) + 4;
	}
	if ( Lnx_DOOM ) {
		maxlen = strlen(&(textdatap[textoff]));
		minlen = maxlen;
	} else {
		maxlen = (strlen(&(textdatap[textoff]))/4)*4+3;
		minlen = maxlen - 3;
	}

	// Get some memory for the old and new text strings, abort if none
	// is available.
	origtext = new char[strlen(&(textdatap[textoff]))+1];
	buffer = new char[maxlen+1];

	if (origtext == NULL || buffer == NULL)
		AbortProg("in Inputtext");

	strcpy(origtext, &(textdatap[textoff]));
	memset(buffer, 0, maxlen+1);

	// Draw the windowframes
	Drawframe(1, INFO, 1, 1, 80, 24);

	if (showonly == NO)
		Drawframe(1, INPUT, 1, 27, 80, 49);

	textattr(INFO);

	CPutsXY(3, 3, "This is the current text string:");

	xpos = 3;
	ypos = 5;

	// Go through the string character by character, and print "bad"
	// character in their C-style equivalent, in dark grey color.
	for (i=0; i<strlen(origtext); i++)
	{
		// Wrap around if the line is too long (bad idea?)
		if (xpos > 78)
		{
			xpos = 3;
			ypos++;
		}
		gotoxy(xpos, ypos);
		switch(origtext[i])
		{

			case '\n':
				textattr(INFDGRAY);
				cputs("\\n");
				xpos = 3;
				ypos++;
				break;
			case '\r':
				textattr(INFDGRAY);
				cputs("\\r");
				xpos = 3;
				ypos++;
				break;
			case '\t':
				textattr(INFDGRAY);
				cputs("\\t");
				xpos += 2;
				break;
			case '\b':
				textattr(INFDGRAY);
				cputs("\\b");
				xpos += 2;
				break;
			case '\f':
				textattr(INFDGRAY);
				cputs("\\f");
				xpos += 2;
				break;
			default:
				textattr(INFO);
				putch(origtext[i]);
				xpos++;
				break;
		}
	}

	// If all the user wants to do is see the current string, quit right
	// now.
	if (showonly == YES)
	{
		Waitforevent(YES);
		puttext(1, 1, 80, 25, image);
		delete origtext;
		delete buffer;
		delete image;
		return 0;
	}

	// OK, so he wants to enter some new text also...
	textattr(INPUT);

	gotoxy(3, 29);
	cprintf("Enter new text. ESC quits, %3d chars min, %3d chars max. Current len:",
			 minlen, maxlen);

	xpos = 3;
	ypos = 31;

	// Stay in the loop until the user exits
	while (!ExitLoop)
	{
		textattr(INPUT);
		CPrintfXY(75, 29, "%3d", curpos);

		gotoxy(xpos, ypos);
		_setcursortype(_NORMALCURSOR);

		// Wait for something to happen
		if (Waitforevent(NO))
		{
			_setcursortype(_NOCURSOR);
			inputchar = getch();

			// Ignore extended keys
			if (!inputchar)
			{
				inputchar = getch();
				continue;
			}

			// Depending on what the character was...
			switch(inputchar)
			{
				case ESC:
					ExitLoop = YES;
					break;
				case BKSP:
					// Check if we're at the beginning of the string and handle
					// those cases.
					if (curpos == 0)
						break;
					else if (curpos == 1)
						changes = NO;
					curpos--;

					// Yuck.  Try to handle the backspace over a newline elegantly.
					// It's tough cause of the cheap way I do the current position.
					if (buffer[curpos] == '\n')
					{
						for (i = curpos-1; i>=-1; i--)
							if (i == -1 || buffer[i] == '\n')
							{
								xpos = curpos-i+3;
								break;
							}
						gotoxy(xpos, --ypos);
						putch(' ');
					}
					gotoxy(--xpos, ypos);
					putch(' ');
					break;
				case RET:
					// Go to a new line on a return.
					if (curpos >= maxlen)
						break;
					changes = YES;
					buffer[curpos++] = '\n';
					textattr(INPDGRAY);
					cputs("\\n");
					xpos = 3;
					ypos++;
					break;
				default:
					// Take the character and enter it in the string.
					if (curpos >= maxlen)
						break;
					changes = YES;
					buffer[curpos++] = inputchar;
					textattr(INPUT);
					xpos++;
					putch(inputchar);
					break;
			}
		}
		else
		{
#ifdef HAVE_MOUSE
			// Handle the mouse-clicks.
			_setcursortype(_NOCURSOR);
			getLastEvent(&x, &y, &lbutton, &rbutton);

			if (LastEventButtons & RIGHTBUTTON && rbutton == buttonUp)
				ExitLoop = YES;
			else if (LastEventButtons & LEFTBUTTON && lbutton == buttonUp)
			{
				// Whoops, haven't handled this yet.
			}
#endif /* HAVE_MOUSE */
		}

		// Text for exit conditions, whether the user wants to
		// save the text changes.
		if (ExitLoop)
		{
			if (changes == YES)
				if (Printinputwindow(answer, "Save your text changes?", YESNO, 1) == -1)
					ExitLoop = NO;

			// Check to make sure it's not too short.
			if (tolower(answer[0]) == 'y')
			{
				if (curpos < minlen) {
					Printwindow("Text does not meet minimum length!", ERROR);
					ExitLoop = NO;
				} else {
					buffer[curpos] = 0;
					strcpy(&(textdatap[textoff]), buffer);
				}
			}
		}
	}

	// Restore the text
	puttext(1, 1, 80, 25, image);
	puttext(1, 27, 80, 50, image2);

	// Free memory
	delete origtext;
	delete buffer;

	if (showonly == NO)
		delete image2;
	delete image;

	// Return -1 if the user answered "no" to that question about saving
	// text changes.
	if (tolower(answer[0]) == 'y')
		return 0;
	else
		return -1;
}

// The 'Go to <Thing>' window

int Printgowindow(void)
{
	int  *image;
	char (*thingnames)[18];
	EBool ExitLoop = NO;
	EBool gotnum = NO;
	int i, curnumber = 0, firstnum = 0;
	int maxnum = numobj[THING][version];
	char key;
	unsigned int x, y;					// Mouse x and y
#ifdef HAVE_MOUSE
	EButton lbutton, rbutton;			// State of left and right buttons
#endif
	long elapsed;
	static didthisone = 0;
	char searchkey[18] = "";
	char curpos = 0;
	int thingnum;

	// Get some memory for the window, and for an array of all the text
	// strings
	image = new int[986];
	thingnames = new char[maxnum][18];

	// Abort if we can't get the memory
	if (image == NULL || thingnames == NULL)
		AbortProg("in PrintGoWindow");

	// Fill in the array of Thing names with the normal array of Thing
	// names.
	for (i=0; i<maxnum; i++)
	{
		if (version == DOOM1_2)
			strcpy(thingnames[i], namelist[thingconvar[i]]);
		else
			strcpy(thingnames[i], namelist[i]);
	}

	// Sort the array
	qsort(thingnames, maxnum, 18, Sortfunction);

	// Save the background text, and draw the window frame up there, with
	// the scroll arrows.
	gettext(27, 9, 55, 42, image);
	Drawframe(1, INPUT, 27, 9, 54, 41);
	textattr(INPUT);
	CPutsXY(54, 13, "");
	CPutsXY(54, 38, "");

	// This is the main input loop for this function
	while (ExitLoop == NO)
	{
		// Just draw everything every time, rather than computing
		// what needs to be done.  My drawing routines are fast enough
		// anyways.  :)
		textattr(INPUT);
		CPutsXY(29, 11, "Select a Thing to go to,");
		CPutsXY(29, 12, "or type a Thing number:");

		// Loop through, draw each Thing name
		for (i = 0; i < 26; i++)
		{
			if (curnumber == i + firstnum)
				textattr(INPHILIT);
			else
				textattr(INPUT);
			CPrintfXY(31, i+14, "%-18s", thingnames[i + firstnum]);
		}

		// Get a key/mouse click
		if (Waitforevent(NO))
		{
			key = getch();

			// !key means that a special (function, arrow) key was pressed.
			if (!key)
			{
				key = getch();
				switch (key)
				{
					case HOME:
						firstnum = curnumber = 0;
						break;
					case END:
						firstnum = maxnum - 26;
						curnumber = maxnum - 1;
						break;
					case PGUP:
						if (firstnum > 26)
							firstnum -= 26;
						else
							firstnum = 0;

						if (curnumber > 26)
							curnumber -= 26;
						else
							curnumber = 0;
						break;
					case PGDN:
						if (firstnum < maxnum - 52)
							firstnum += 26;
						else
							firstnum = maxnum-26;

						if (curnumber < maxnum - 26)
							curnumber += 26;
						else
							curnumber = maxnum-1;
						break;
					case UP:
						if (curnumber > 0)
							curnumber--;

						if (curnumber < firstnum)
							firstnum--;
						break;
					case DOWN:
						if (curnumber < maxnum-1)
							curnumber++;

						if (curnumber > firstnum + 25)
							firstnum++;
						break;
				}
			}
			else
			{
				if (key == ESC)
				{
					// Escape = exit without selecting a new Thing
					// Set gotnum to YES so that we don't try to find the
					// thing number, but leave it at -1 instead.
					gotnum = YES;
					thingnum = -1;
					ExitLoop = YES;
				}
				else if (key == RET)
					// Return = select this Thing
					ExitLoop = YES;
				else if (isdigit(key))
				{
					// Typing a digit brings up the Goto Thing box.  Provided
					// the user didn't escape out of it, return the number
					// Also remember to put the character back into the input
					// stream.
					thingnum = GotoObject(key);
					if (thingnum != -1)
					{
						gotnum = YES;
						ExitLoop = YES;
					}
				}
				else
				{
					// Backspace = chop a letter off the current search key
					if (key == BKSP)
					{
						if (curpos > 0)
							curpos--;
						searchkey[curpos] = 0;
					}
					else
					{
						// Otherwise, the default is just add it to the search key
						searchkey[curpos] = key;
						if (curpos < 19)
							curpos++;
					}

					// Find the closest match to what was typed.
					curnumber = -1;
					for (i=0; i<maxnum; i++)
						if (stricmp(searchkey, thingnames[i]) <= 0)
						{
							curnumber = i;
							break;
						}
					if (curnumber == -1)
						curnumber = maxnum - 1;

					// Make sure the new curnumber is onscreen.
					if ((curnumber < firstnum) || (curnumber > firstnum + 25))
					{
						if (curnumber < 13)
							firstnum = 0;
						else if (curnumber > maxnum-26)
							firstnum = maxnum-26;
						else
							firstnum = curnumber - 13;
					}
				}
			}
		}
#ifdef HAVE_MOUSE
		else
		{
			getLastEvent(&x, &y, &lbutton, &rbutton);

			if (LastEventButtons != eventButtons)
				didthisone = -1;

			// Otherwise, if the left or right button was let up, quit.
			if (eventButtons & LEFTBUTTON)
			{
				// Test the pageup/pagedown arrows
				if (x == 53 && y == 13)
				{
					elapsed = clock()-lclicktime;
					if (((elapsed == 0) || ((elapsed > 6) && (elapsed % 4 == 0))) &&
						 (didthisone != elapsed))
					{
						didthisone = elapsed;
						if (firstnum > 26)
							firstnum -= 26;
						else
							firstnum = 0;

						if (curnumber > 26)
							curnumber -= 26;
						else
							curnumber = 0;
					}
				}
				// Test the pageup/pagedown arrows
				else if (x == 53 && y == 38)
				{
					elapsed = clock()-lclicktime;
					if (((elapsed == 0) || ((elapsed > 6) && (elapsed % 4 == 0))) &&
						 (didthisone != elapsed))
					{
						didthisone = elapsed;
						if (firstnum < maxnum - 52)
							firstnum += 26;
						else
							firstnum = maxnum-26;

						if (curnumber < maxnum - 26)
							curnumber += 26;
						else
							curnumber = maxnum-1;
					}
				}
			}

			if (LastEventButtons & RIGHTBUTTON && rbutton == buttonUp)
			{
				curnumber = -1;
				ExitLoop = YES;
			}
			else if ((LastEventButtons & LEFTBUTTON) && (lbutton == buttonUp) &&
						 x > 29 && x < 47 && y > 12 && y < 39)
			{
				curnumber = firstnum + y - 13;
				ExitLoop = YES;
			}
		}
#endif /* HAVE_MOUSE */
	}

	// Slap the background text back up onscreen, free the memory.
	puttext(27, 9, 55, 42, image);
	delete image;

	// If the curnumber is valid (not -1), compares the chosen string with
	// each string in the regular Thing name list to find the correct Thing
	// to go to.
	if (gotnum == NO)
	{
		thingnum = -1;
		for (i=0; i<maxnum; i++)
			// Compares the chosen string with each string in the Thing name
			// list to find the correct Thing to go to.  Needs two separate
			// comparisons for Doom 1.2 and Doom anything else.
			if (((version != DOOM1_2) &&
				  (strcmp(thingnames[curnumber], namelist[i]) == 0)) ||
				 ((version == DOOM1_2) &&
				  (strcmp(thingnames[curnumber], namelist[thingconvar[i]]) == 0)))
			{
				thingnum = i;
				break;
			}
	}

	// Free memory, return value.
	delete thingnames;
	return thingnum;
}

// Prints the window to get input

int Printinputwindow(char *buffer, char *prompt, EInptype type, int length, char firstchar)
{
	int *image;
	int result, x1, x2, total;
	int y1 = 23, y2 = 27;

	// Set the max length according to the 'type' argument.
	if (type == LONGINT)
		length = 11;
	else if (type == YESNO)
	{
		length = 1;
		y2 += 2;
	}

	// Set the x-coord start and end of the input box.
	total = 6+strlen(prompt)+length;
	x1 = 40 - total/2;
	x2 = 40 + total/2;

	// Take some extra precautions for strings (ie, put them on a new line)
	if (type == STRING)
	{
		y2++;
		if (length < strlen(prompt))
		{
			x1 = 38 - strlen(prompt)/2;
			x2 = 42 + strlen(prompt)/2;
		}
		else
		{
			x1 = 38 - length/2;
			x2 = 42 + length/2;
		}
	}

	// Store the screen in memory
	image = new int[(x2-x1+2) * (y2-y1+2)];
	gettext(x1, y1, x2+1, y2+1, image);

	// Draw the frame up, put up the prompt and yes/no buttons if necessary.
	Drawframe(1, INPUT, x1, y1, x2, y2);
	textattr(INPUT);
	CPutsXY(x1+2, y1+2, prompt);
	if (type == YESNO)
	{
		// Put the clickable buttons up there.
		textattr(INFDGRAY);
		CPutsXY(30, y1+4, "  Yes  ");
		CPutsXY(44, y1+4, "  No  ");
		textattr(INFO);
		CPutsXY(32, y1+4, "Y");
		CPutsXY(46, y1+4, "N");
	}

	// Get the input
	result = Getinput(strlen(prompt), buffer, length, type, x1, firstchar);

	// Paste background back up
	puttext(x1, y1, x2+1, y2+1, image);
	delete image;

	return result;
}

// Act on a keypress

EBool ProcessKeypress(void)
{
	int  result;				// The result of inputs, used for error testing
	char buffer[160];			// Used for many string operations
	int key;						// For input
	int i;
	int last = global[mode][GMAX] - global[mode][GMIN];
	int &curfield = global[mode][FIELD];
	int &curobj = global[mode][CUR];

	key = getch();

	if (!key)
	{
		key = getch();

		// Unhighlight the current field
		Highlight(NORMAL);

		// Take care of arrow keys etc.
		switch (key)
		{
			case RIGHT:
				// Three cases for right arrow...
				// Thing editing mode, when we check the fieldinfo array
				// for the new position...
				if (mode == THING_EDIT)
				{
					i = GetThingWindow(curfield);

					if (i == 4)
						curfield = fieldinfo[0][0];
					else
						curfield = fieldinfo[i+1][0];
				}
				else if (mode != AMMO_EDIT)
				{
					// And everything else, when we just increment
					// the field count.
					if (curfield < fieldinfo2[mode])
						curfield++;
					else
						curfield = 1;
				}
				break;
			case LEFT:
				if (mode == THING_EDIT)
				{
					i = GetThingWindow(curfield);

					if (i == 0)
						curfield = fieldinfo[4][0];
					else
						curfield = fieldinfo[i-1][0];
				}
				else if (mode != AMMO_EDIT)
				{
					if (curfield > 1)
						curfield--;
					else
						curfield = fieldinfo2[mode];
				}
				break;
			case UP:
				// If this is the Thing mode, Up means go to a different
				// window.
				if (mode == THING_EDIT)
				{
					i = GetThingWindow(curfield);

					if (curfield > fieldinfo[i][0])
						curfield--;
					else
						curfield = fieldinfo[i][1];
				}
				// If this is the Ammo screen, Up means go to a different
				// field.
				else if (mode == AMMO_EDIT)
				{
					if (curfield > 1)
						curfield--;
					else
						curfield = 8;
				}
				// Otherwise, Up means go up on the list of items.
				else
				{
					if (curobj > 0)
						curobj--;

					if (curobj < global[mode][TOPROW])
					{
						global[mode][TOPROW]--;
						redraw = ALLDATA;
					}
				}
				break;
			case DOWN:
				if (mode == THING_EDIT)
				{
					i = GetThingWindow(curfield);

					if (curfield < fieldinfo[i][1])
						curfield++;
					else
						curfield = fieldinfo[i][0];
				}
				else if (mode == AMMO_EDIT)
				{
					if (curfield < 8)
						curfield++;
					else
						curfield = 1;
				}
				else
				{
					if (curobj < last)
						curobj++;

					if (curobj > global[mode][TOPROW]+37)
					{
						global[mode][TOPROW]++;
						redraw = ALLDATA;
					}
				}
				break;
			case PGUP:
				GoPageUp();
				break;
			case PGDN:
				GoPageDown();
				break;
			case HOME:
				// Go to the first object, if we're not already there, and
				// redraw.
				if (curobj != 0)
				{
					curobj = 0;
					redraw = ALLDATA;
				}
				global[mode][TOPROW] = 0;
				break;
			case END:
				// Go to the last object, if we're not already there, and
				// set the top row on the screen to 37 higher.
				if (curobj != last)
				{
					curobj = last;
					redraw = ALLDATA;
				}
				global[mode][TOPROW] = last-37;
				break;
			case F1:
				Printhelp();
				break;
			case F2:
			case F3:
			case F4:
			case F5:
			case F6:
			case F7:
			case F8:
				Changemode((EModes)(key - F2));
				break;
		}
	}
	else
	{
		// Handle normal keypresses
		switch (tolower(key))
		{
			case ESC:
				// Quit.  Return, do exit checking in Main.
				return ConfirmQuit();
			case RET:
				// Edit the current field, if applicable.
				if (Updatefunc[mode]() == 0)
				{
					changes = YES;
					redraw = DATA;
				}
				break;
			case ' ':
				// Show frames, text, and play sound, if applicable.
				result = -1;
				switch (mode)
				{
					case THING_EDIT:
						if (curfield >= fieldinfo[2][0] &&
							 curfield <= fieldinfo[2][1])
							result = Showframe(thingdata[curobj][thingorder[curfield-1]]);
						else if (curfield >= fieldinfo[1][0] &&
									curfield <= fieldinfo[1][1])
							Playsound(thingdata[curobj][thingorder[curfield-1]]);
						break;
					case FRAME_EDIT:
						if (curfield == 4)
							result = Showframe(framedata[curobj][NEXTFRAME]);
						else
							result = Showframe(curobj);
						break;
					case AMMO_EDIT:
						if (curfield >= 4 && curfield <= 8)
							result = Showframe(weapondata[curobj][curfield-3]);
						break;
					case SOUND_EDIT:
						Playsound(curobj+1);
						break;
					case TEXT_EDIT:
						Inputtext(YES);
						break;
				}
#ifdef __DOS__	// Linux doesn't need to redraw everything. :)
				if (result != -1)
					redraw = ALL;
#endif
				break;
			case 'a':
				// Print an about/status box.
				Printintro();
				break;
			case 'c':
				// Copy an entry.
				redraw = ALLDATA;
				if (Getcopyinfo() != -1)
					changes = YES;
				break;
			case 'd':
				// Dump the data to files.
				Dumpdata();
				break;
			case 'g':
				// Go to a specific object, by a list of names.

				// Unhighlight the current field
				Highlight(NORMAL);

				// Print the list of Thing names if applicable, otherwise
				// do the generic Go to box.
				if (mode == THING_EDIT)
				{
					if ((result = Printgowindow()) != -1)
					{
						curobj = result;
						redraw = ALLDATA;
					}
				}
				else if (GotoObject(0) != -1)
					redraw = ALLDATA;
				break;
			case 'j':
				// Jump to the info in whatever field is highlighted.

				// Unhighlight the current field first
				Highlight(NORMAL);

				// Now, depending on which mode and which field we're currently
				// on, do a bunch of different things...
				switch (mode)
				{
					case THING_EDIT:
						if (curfield >= fieldinfo[2][0] &&
							 curfield <= fieldinfo[2][1])
						{
							global[FRAME_EDIT][CUR] =
									(int)thingdata[curobj][thingorder[curfield-1]];
							Changemode(FRAME_EDIT);
						}
						else if (curfield >= fieldinfo[1][0] &&
									curfield <= fieldinfo[1][1] &&
									thingdata[curobj][thingorder[curfield-1]] != 0)
						{
							global[SOUND_EDIT][CUR] =
									(int)thingdata[curobj][thingorder[curfield-1]]-1;
							Changemode(SOUND_EDIT);
						}
						break;
					case FRAME_EDIT:
						switch (curfield)
						{
							case 1:
							case 2:
								global[SPRITE_EDIT][CUR] = (int)framedata[curobj][0];
								Changemode(SPRITE_EDIT);
								break;
							case 4:
								global[FRAME_EDIT][CUR] = (int)framedata[curobj][4];
								Changemode(FRAME_EDIT);
								redraw = ALLDATA;
								break;
						}
						break;
					case AMMO_EDIT:
						if (curfield >= 4 && curfield <= 8)
						{
							global[FRAME_EDIT][CUR] =
									weapondata[curobj][curfield-3];
							Changemode(FRAME_EDIT);
						}
						break;
					case SPRITE_EDIT:
						global[TEXT_EDIT][CUR] = Gettextnum((int)spritedata[curobj]);
						Changemode(TEXT_EDIT);
						break;
					case SOUND_EDIT:
						global[TEXT_EDIT][CUR] = Gettextnum((int)sounddata[curobj][TEXTP]);
						Changemode(TEXT_EDIT);
						break;
					case THING_LIST:
						global[THING_EDIT][CUR] = curobj;
						Changemode(THING_EDIT);
						break;
					case TEXT_EDIT:
						// What do we do here??  FIXME!!
						break;
				}
				break;
			case 'l':
				// Load a patch file.
				if (Printinputwindow(buffer, "Enter patch file to load:", STRING, 70) != -1)
				{
					Loadpatch(buffer);
					redraw = ALLDATA;
					changes = YES;
				}
				break;
			case 'o':
				// Old-style save patch.
				if (Printinputwindow(buffer, "Enter the filename to save a backwards compatible patch file to:", STRING, 70) != -1)
				{
					result = OldSave(buffer, NO);
					if (result == -1)
					{
						char temp[2];

						if ((Printinputwindow(temp, "File exists! Overwrite?", YESNO, 1) == -1) ||
							  tolower(temp[0]) != 'y')
						{
							strcpy(buffer, "Write canceled.");
							result = INFO;
						}
						else
							result = OldSave(buffer, YES);
					}

					if (result == -1)
						Printwindow("Unable to save patch file!", ERROR);
					else
						Printwindow(buffer, result);
				}
				break;
			case 'r':
				// Run Doom
				if (RunExe() != -1)
					redraw = ALL;
				break;
			case 's':
				// Save a patch
 				if (Printinputwindow(buffer, "Enter save patch filename:", STRING, 70) != -1)
				{
					result = Savepatch(buffer, NO);
					if (result == -1)
					{
						char temp[2];

						if ((Printinputwindow(temp, "File exists! Overwrite?", YESNO, 1) == -1) ||
							  tolower(temp[0]) != 'y')
						{
							strcpy(buffer, "Write canceled.");
							result = INFO;
						}
						else
							result = Savepatch(buffer, YES);
					}

					if (result == -1)
						Printwindow("Unable to save patch file!", ERROR);
					else
						Printwindow(buffer, result);
				}
				break;
			case 'u':
				// Reload from the Doom.exe file.
				Loaddoom(doomexefp);
				Printwindow("Doom exe data reloaded.", INFO);
				changes = NO;
				redraw = ALLDATA;
				break;
			case 'w':
				// Write changes (if we've made any).
				if (changes == NO)
					Printwindow("No changes made!", INFO);
				else
				{
					Writedoom();
					Printwindow("Changed data written to exe.", INFO);
					changes = NO;
				}
				break;
			case 'z':
				// Reload from original Doom.exe file.
				Loaddoom(doombakfp);
				Printwindow("Original Doom exe data loaded.", INFO);
				changes = YES;
				redraw = ALLDATA;
				break;
		}
	}

	return NO;
}

#ifdef HAVE_MOUSE
// Act on a mouse action

EBool ProcessMouse(void)
{
	int  result;				// The result of inputs, used for error testing
	unsigned int x, y;		// Mouse x and y
	EButton lbutton, rbutton;// State of left and right buttons
	int i;
	EBool Foundspot = NO;
	long elapsed;
	static int didthisone = -1;
	static long prevlclick = 0;

	getLastEvent(&x, &y, &lbutton, &rbutton);

	if (LastEventButtons != eventButtons)
		didthisone = -1;

	// OK, we're holding down the left button, so it's probably on an arrow
	// key somewhere.
	if (eventButtons & LEFTBUTTON)
	{
		// Test the pageup/pagedown arrows.  Yuck!  Bad code!
		// Bad bad!
		if (x == arrows[mode][0]-1 && y == arrows[mode][1]-1)
		{
			// This is the up button

			// Get the current time, and subtract the starting time from
			// it to find elapsed time (in clock ticks).
			elapsed = clock()-lclicktime;

			// If this is the third tick and we haven't done this tick's
			// action yet, then do it... slight pause at first for slow
			// clicking people.
			if (((elapsed == 0) || ((elapsed > 6) && (elapsed % 3 == 0))) &&
				 (didthisone != elapsed))
			{
				// Set didthisone to the elapsed time, so that we know
				// we've performed the action on this tick already.
				didthisone = elapsed;

				// And, of course, do the correct thing for whichever
				// editing screen.
				GoPageUp();
			}
		}
		else if (x == arrows[mode][2]-1 && y == arrows[mode][3]-1)
		{
			elapsed = clock()-lclicktime;
			if (((elapsed == 0) || ((elapsed > 6) && (elapsed % 3 == 0))) &&
				 (didthisone != elapsed))
			{
				didthisone = elapsed;
				GoPageDown();
			}
		}
	}

	if ((lbutton == buttonUp) && (LastEventButtons & LEFTBUTTON))
	{
		if (y == 49)
		{
			// The bar on the bottom of the screen.
			if (x > 6 && x < 17)
				Printhelp();
			else if (x > 24 && x < 54)
				Printintro();
			else if (x > 61 && x < 72)
				return ConfirmQuit();
		}
		else if (y == 0)
		{
			// The mode bar at the top of the screen.
			if (x > 0 && x < 10)
				Changemode(THING_EDIT);
			else if (x > 11 && x < 21)
				Changemode(FRAME_EDIT);
			else if (x > 22 && x < 33)
				Changemode(AMMO_EDIT);
			else if (x > 34 && x < 44)
				Changemode(SOUND_EDIT);
			else if (x > 45 && x < 56)
				Changemode(SPRITE_EDIT);
			else if (x > 57 && x < 65)
				Changemode(TEXT_EDIT);
			else if (x > 66 && x < 79)
				Changemode(THING_LIST);
		}
		else
		{
			// Switch to any applicable field that was clicked
			// (for editing).

			// Unhighlight the current field first
			Highlight(NORMAL);

			for (i=0; i < posinfo[mode][1]; i++)
			{
				if (x >= Fielddata[i+posinfo[mode][0]][1] &&
					 x <= Fielddata[i+posinfo[mode][0]][2])
				{
					if (posinfo[mode][2] == 1 && y > 6 && y < 75)
					{
						global[mode][FIELD] = i+1;
						global[mode][CUR] = global[mode][TOPROW]+y-7;
						if ((lclicktime-prevlclick < 6) && (Updatefunc[mode]() == 0))
						{
							changes = YES;
							redraw = DATA;
						}
					}
					else if (posinfo[mode][2] == 0 &&
								y == Fielddata[i+posinfo[mode][0]][0])
					{
						// Edit the current field, if applicable.
						global[mode][FIELD] = i+1;
						if ((lclicktime-prevlclick < 6) && (Updatefunc[mode]() == 0))
						{
							changes = YES;
							redraw = DATA;
						}
					}
				}

				if (Foundspot == YES)
					break;
			}

			// If the mouse click still hasn't been handled,
			// take care of special cases.
			if (Foundspot == NO)
			{
				switch (mode)
				{
					case THING_EDIT:
						if (x > 3 && x < 35 && y == 5)
						{
							if ((result = Printgowindow()) != -1)
							{
								global[THING_EDIT][CUR] = result;
								redraw = ALLDATA;
							}
							Foundspot = YES;
						}
						else if (x > 63 && x < 75 && y > 3 && y < 9)
						{
							Playsound(thingdata[global[mode][CUR]][thingorder[y+6]]);
							Foundspot = YES;
						}
						else if (x > 67 && x < 75 && y > 14 && y < 23)
						{
							if (Showframe(thingdata[global[mode][CUR]][thingorder[y]]) != -1)
								redraw = ALL;
							Foundspot = YES;
						}
						break;
					case FRAME_EDIT:
						if (x > 14 && x < 20 && y > 6 && y < 75)
						{
							if (Showframe(global[FRAME_EDIT][TOPROW]+y-7) != -1)
								redraw = ALL;
							Foundspot = YES;
						}
						break;
					case AMMO_EDIT:
						if (x > 51 && x < 58 && y > 21 && y < 30)
						{
							if (Showframe(weapondata[global[mode][CUR]][y-24]) != -1)
								redraw = ALL;
							Foundspot = YES;
						}
						break;
					case SOUND_EDIT:
						if (x > 24 && x < 32 && y > 6 && y < 75)
						{
							Playsound(global[SOUND_EDIT][CUR]+y-6);
							Foundspot = YES;
						}
						break;
				}
			}
		}
		prevlclick = lclicktime;
	}

	return NO;
}
#endif /* HAVE_MOUSE */

// Called by qsort to compare strings

int Sortfunction(const void *a, const void *b)
{
	return (stricmp((char *)a, (char *)b));
}

// Returns when key is pressed or mouse is clicked (or left mouse button
// is held down).

EBool Waitforevent(EBool eatevent)
{
	int hay_input;
	EBool KeyDown = NO;
	EBool ExitLoop = NO;

#ifdef HAVE_MOUSE
	showMouseCursor();
#endif

/* Do we really want to flush the input stream?
	// Flush the input stream
	while ( pending_input(0) ) {
		if ( getch() == 0 )
			getch();
	}
*/

	do
	{	// Mouse button presses and such should be seen asynchronously.
		// Wait for keypress...  (For a long time -- 10000 seconds)
		if ( (hay_input=pending_input(10000)) > 0 ) {
			KeyDown = YES;
			ExitLoop = YES;
		} else if ( hay_input < 0 ) { // Error or mouse event.
			ExitLoop = YES;
		}
	} while (!ExitLoop);

#ifdef HAVE_MOUSE
	hideMouseCursor();
#endif

	// Remove the event if we are supposed to.
	if (eatevent) {
		if (KeyDown) {
			if ( ! getch() )
				getch();
		}
#ifdef HAVE_MOUSE
		else
			LastEventButtons = eventButtons;
#endif
	}

	// Return if it was a keyboard press or not.
	return(KeyDown);
}
