/* $Id: signal.c,v 30000.23 1993/05/18 08:58:11 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/**************************************
 * Fugue signal handlers              *
 *                                    *
 * The following signals are trapped: *
 * SIGINT                             *
 * SIGTSTP                            *
 * SIGSEGV                            *
 * SIGBUS                             *
 * SIGQUIT                            *
 * SIGILL                             *
 * SIGTRAP                            *
 * SIGFPE                             *
 * SIGWINCH                           *
 * SIGPIPE                            *
 **************************************/

#include <signal.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "process.h"
#include "keyboard.h"
#include "output.h"
#include "socket.h"

#define TEST_SIG(sig) (pending_signals & (1 << ((sig) - 1)))
#define SET_SIG(sig) (pending_signals |= (1 << ((sig) - 1)))
#define CLR_SIG(sig) (pending_signals &= ~(1 << ((sig) - 1)))
#define ZERO_SIG() (pending_signals = 0)

static unsigned long pending_signals;
static void FDECL((*old_tstp_handler),(int sig));

static void NDECL(interrupt);
#ifndef NO_COREHANDLERS
static void FDECL(core_handler,(int sig));
static void FDECL(coremsg,(int sig));
#endif
static void FDECL(ignore_signal,(int sig));
static void FDECL(signal_scheduler,(int sig));

int  NDECL(suspend);
void NDECL(process_signals);
void NDECL(init_signals);
int  NDECL(interrupted);
void FDECL(core,(CONST char *why));


void init_signals()
{
    ZERO_SIG();
    signal(SIGINT  , signal_scheduler);
#ifdef SIGTSTP
    old_tstp_handler = signal(SIGTSTP , signal_scheduler);
#endif
#ifndef NO_COREHANDLERS
# ifdef SIGBUS /* not defined in Linux */
    signal(SIGBUS  , core_handler);
# endif
    signal(SIGSEGV , core_handler);
    signal(SIGQUIT , core_handler);
    signal(SIGILL  , core_handler);
    signal(SIGTRAP , core_handler);
    signal(SIGFPE  , core_handler);
#endif
    signal(SIGPIPE , ignore_signal);
#ifdef SIGWINCH
    signal(SIGWINCH, signal_scheduler);
#endif
}

static void ignore_signal(sig)
    int sig;
{
    signal(sig, ignore_signal);
}

static void interrupt()
{
    int c;

    if (visual) fix_screen();
    else clear_input_line();
    printf("C) continue; X) exit; T) disable triggers; P) kill processes ");
    fflush(stdout);
    c = igetchar();
    clear_input_line();
    if (strchr("xyXY", c)) die("Interrupt, exiting.\n");
    setup_screen();
    do_replace();
    if (c == 't' || c == 'T') {
        setivar("borg", 0, FALSE);
        oputs("% Cyborg triggers disabled");
    } else if (c == 'p' || c == 'P') {
        kill_procs();
    }
    oflush();       /* in case of output between SIGINT receipt and handling */
}

int suspend()
{
#ifdef SIGTSTP
    extern Stringp keybuf;
 
    if (old_tstp_handler == SIG_DFL) {        /* true for job-control shells */
        if (visual) fix_screen();
        reset_tty();
        kill(getpid(), SIGSTOP);
        cbreak_noecho_mode();
        if (keybuf->len) set_refresh_pending();
        get_window_size();
        setup_screen();
        do_replace();
        if (maildelay > 0) check_mail();
        return 1;
    }
#endif
    oputs("% Job control not supported.");
    return 0;
}

#ifndef NO_COREHANDLERS
static void coremsg(sig)
    int sig;
{
    printf("Core dumped - signal %d\n", sig);
    puts("Please report this and any other error messages to kkeys@ucsd.edu");
    puts("or hawkeye@glia.biostr.washington.edu.");
    puts("Read \"/help core\" first, if you can.");
}

static void core_handler(sig)
    int sig;
{
    cleanup();
    if (sig != SIGQUIT) coremsg(sig);
    signal(sig, SIG_DFL);
    kill(getpid(), sig);
}
#endif

static void signal_scheduler(sig)
    int sig;
{
    signal(sig, signal_scheduler);  /* restore handler (SysV) */
    SET_SIG(sig);                   /* set flag to deal with it later */
}

void process_signals()
{
    if (pending_signals == 0) return;

    if (TEST_SIG(SIGINT))   interrupt();
#ifdef SIGTSTP
    if (TEST_SIG(SIGTSTP))  suspend();
#endif
#ifdef SIGWINCH
    if (TEST_SIG(SIGWINCH)) get_window_size();
#endif
    ZERO_SIG();
}

int interrupted()
{
    return TEST_SIG(SIGINT);
}

void core(why)
    CONST char *why;
{
    cleanup();
    puts(why);
    signal(SIGQUIT, SIG_DFL);
    kill(getpid(), SIGQUIT);
}

