/* $Id: socket.c,v 30000.25 1993/05/25 01:07:06 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/***************************************************************
 * Fugue socket handling                                       *
 *                                                             *
 * Reception and transmission through sockets is handled here. *
 * This module also contains the main loop.                    *
 * Multiple sockets handled here.                              *
 * Autologin handled here.                                     *
 * Rewritten by Ken Keys to do non-blocking connect(), handle  *
 * background sockets, and do more efficient process running.  *
 ***************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <errno.h>
#include "port.h"
#include "fd_set.h"

#ifdef CONNECT_SVR4
# include <stropts.h>
#endif
#ifdef CONNECT_BSD
# include <sys/socket.h>
# include <sys/uio.h>
#endif

#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "history.h"
#include "world.h"
#include "tf.connect.h"
#include "socket.h"
#include "output.h"
#include "process.h"
#include "macro.h"
#include "keyboard.h"
#include "command2.h"
#include "special.h"
#include "signal.h"
#include "search.h"
#ifndef CONNECT
# include <sys/types.h>
# include <sys/socket.h>
# ifdef WINS
#  include <sys/in.h>
#  include <sys/inet.h>
# else
#  include <netinet/in.h>
# endif
# include "opensock.h"
#endif

static void  FDECL(wload,(World *w));
static Sock *FDECL(find_sock,(char *name));
static void  FDECL(announce_world,(Sock *s));
static void  FDECL(fg_sock,(Sock *sock));
static void  NDECL(bg_sock);
static void  NDECL(any_sock);
static int   FDECL(establish,(Sock *new));
static void  NDECL(nuke_dead_socks);
static void  FDECL(killsock,(Sock *sock));
static void  FDECL(nukesock,(Sock *sock));
static void  FDECL(magic_login,(Sock *sock));
static void  FDECL(do_prompt,(Sock *sock));
static void  FDECL(handle_socket_line,(Sock *sock, Stringp out));
static void  FDECL(handle_socket_input,(Sock *sock));
static void  FDECL(flush_output,(Sock *sock));
#ifdef CONNECT
static int   FDECL(nb_connect,(char *address, char *port, FILE **fpp,
             int *tfcerrnop));
static int   FDECL(recvfd,(int sockfd, int *tfcerrnop));
#endif

void   NDECL(init_sock);
void   NDECL(main_loop);
void   FDECL(readers_clear,(int fd));
void   FDECL(readers_set,(int fd));
TOGGLER(tog_background);
void   FDECL(mapsock,(void FDECL((*func),(World *world))));
World *NDECL(fworld);
World *NDECL(xworld);
int    FDECL(connect_to,(World *w, int autologin));
int    FDECL(disconnect,(char *args));
void   FDECL(movesock,(int dir));
void   NDECL(disconnect_all);
void   NDECL(listsockets);
void   FDECL(world_output,(World *w, char *str, int attrs));
int    FDECL(transmit,(char *s, unsigned int len));
void   NDECL(clear_refresh_pending);
void   NDECL(set_refresh_pending);
int    NDECL(is_refresh_pending);
void   NDECL(set_done);

#ifndef CONN_WAIT
#define CONN_WAIT 500000
#endif
#ifndef PROC_WAIT
#define PROC_WAIT 100000
#endif

extern Stringp lpprompt;
extern int input_cursor;                /* is cursor in input window? */
extern TIME_T proctime;                 /* when next process should run */

static fd_set readers;                  /* file descriptors we're watching */
static fd_set active;                   /* active file descriptors */
static int nfds;                        /* max # of readers */
static Sock *hsock = NULL;              /* head of socket list */
static Sock *tsock = NULL;              /* tail of socket list */
static int done = FALSE;                /* Are we all done? */
static int need_refresh;                /* Does input line need refresh? */
static int hold_refresh;                /* Should we wait to do refresh? */
static int dead_socks = 0;              /* Number of unnuked dead sockets */

/* These must correspond to the TFC_* errors in tf.connect.h */
static char *tfc_errlist[] = {
    "Error 0",
    "Can't find host",
    "No such service",
    "socket",
    "connect",
    TFCONNECT,
    "pipe",
    "socketpair",
    "popen",
    "recv",
    "recvmsg",
};

#define TFC_NERRORS (sizeof(tfc_errlist)/sizeof(char *))

#define TFCERROR(num) (((num) >= 0 && (num) < TFC_NERRORS) \
    ? tfc_errlist[(num)] : "Unknown error")

#define TELNET_ECHO    '\001'         /* echo option */
#define TELNET_SGA     '\003'         /* suppress GOAHEAD option */
#define TELNET_EOR_OPT '\031'         /* EOR option */

#define TELNET_EOR     '\357'         /* End-Of-Record */
#define TELNET_GA      '\371'         /* Go Ahead */
#define TELNET_WILL    '\373'         /* I offer to ~, or ack for DO */
#define TELNET_WONT    '\374'         /* I will stop ~ing, or nack for DO */
#define TELNET_DO      '\375'         /* Please do ~?, or ack for WILL */
#define TELNET_DONT    '\376'         /* Stop ~ing!, or nack for WILL */
#define TELNET_IAC     '\377'         /* telnet Is A Command character */

Sock *fsock = NULL;                   /* foreground socket */
Sock *xsock = NULL;                   /* current (transmission) socket */
int active_count = 0;                 /* # of (non-current) active sockets */
TIME_T mailtime = 0;                  /* next mail check (0==immdediately) */


/* initialize socket.c data */
void init_sock()
{
    FD_ZERO(&readers);
    FD_ZERO(&active);
    FD_SET(0, &readers);
    nfds = 1;
}

void main_loop()
{
    int count;
    struct timeval tv, *tvp;
    TIME_T now, earliest;
    Sock *sock;

    while (!done) {
        now = (TIME_T)time(NULL);
        if (!lpquote && now >= proctime) runall(now);
        /* figure out when next event is so select() can timeout then */
        earliest = proctime;
        if (maildelay > 0) {
            if (now >= mailtime) {
                check_mail();
                mailtime = now + maildelay;
            }
            if (!earliest || (mailtime < earliest))
                earliest = mailtime;
        }

        if (earliest) {
            tvp = &tv;
            tv.tv_sec = earliest - now;
            tv.tv_usec = 0;
            if (tv.tv_sec <= 0) {
                tv.tv_sec = 0;
            } else if (tv.tv_sec == 1) {
                tv.tv_sec = 0;
                tv.tv_usec = PROC_WAIT;
            }
        } else tvp = NULL;

        if (need_refresh || !input_cursor) {
            if (!tvp || tvp->tv_sec > 0 || tvp->tv_usec > refreshtime) {
                hold_refresh = 0;
                tvp = &tv;
                tv.tv_sec = 0;
                tv.tv_usec = refreshtime;
            } else hold_refresh = 1;
        }

        /* find descriptors that need to be read */
        active = readers;
        count = select(nfds, &active, NULL, NULL, tvp);

        if (count < 0) {
            if (errno != EINTR) {
                operror("TF/main_loop/select");
                die("% Failed select");
            }
        } else if (count == 0) {
            if (!hold_refresh) {
                if (need_refresh) do_line_refresh();
                else if (!input_cursor) ipos();
            }
        } else {
            /* check for user input */
            if (FD_ISSET(0, &active)) {
                count--;
                if (need_refresh) do_line_refresh();
                else if (!input_cursor) ipos();
                handle_keyboard_input();
            }
            /* check for input from foreground socket */
            if (count && fsock && FD_ISSET(fsock->fd, &active)) {
                count--;
                handle_socket_input(fsock);
                FD_CLR(fsock->fd, &active);
            }
            /* check for input from background sockets */
            for (sock = hsock; count && sock; sock = sock->next) {
                if (FD_ISSET(sock->fd, &active)) {
                    count--;
                    if (sock->flags & SOCKPENDING) {
                        establish(sock);
                    } else {
                        if (background) {
                            handle_socket_input(xsock = sock);
                            xsock = fsock;
                        } else FD_CLR(sock->fd, &readers);
                    }
                }
            }
        }

        /* deal with signals caught during last loop */
        process_signals();

        /* garbage collection */
        if (dead_socks) nuke_dead_socks();
        nuke_dead_macros();
    } 
    cleanup();

#ifdef DMALLOC
    {
        free_macros();
        purge_world("*");
        free_keyboard();
        free_prefixes();
        free_histories();
        free_term();
        free_vars();
        debug_mstats("tf");
    }
#endif
}

int is_active(fd)
    int fd;
{
    return FD_ISSET(fd, &active);
}

void readers_clear(fd)
    int fd;
{
    FD_CLR(fd, &readers);
}

void readers_set(fd)
    int fd;
{
    FD_SET(fd, &readers);
    if (fd >= nfds) nfds = fd + 1;
}

/* find existing open socket to world <name> */
static Sock *find_sock(name)
    char *name;
{
    Sock *sock;

    for (sock = hsock; sock; sock = sock->next) {
        if (sock->flags & (SOCKDEAD | SOCKPENDING)) continue;
        if (!name || cstrcmp(sock->world->name, name) == 0) break;
    }
    return sock;
}

void tog_background()
{
    Sock *sock;
    if (background)
        for (sock = hsock; sock; sock = sock->next)
            if (!(sock->flags & (SOCKDEAD | SOCKPENDING)))
                FD_SET(sock->fd, &readers);
}

/* Perform (*func)(world) on every open world */
void mapsock(func)
    void FDECL((*func),(World *world));
{
    Sock *sock;

    for (sock = hsock; sock; sock = sock->next)
        if (!(sock->flags & (SOCKDEAD | SOCKPENDING))) (*func)(sock->world);
}

/* foreground world */
World *fworld()
{
    return fsock ? fsock->world : NULL;
}

/* current transmission world */
World *xworld()
{
    return xsock ? xsock->world : NULL;
}
  
/* load macro file for a world */
static void wload(w)
    World *w;
{
    World *d;

#ifndef RESTRICT_FILE
    if (*w->mfile) do_file_load(w->mfile);
    else if ((d = get_default_world()) != NULL && *d->mfile)
        do_file_load(d->mfile);
#endif
}

/* announce foreground world */
static void announce_world(s)
    Sock *s;
{
    put_world(s ? s->world->name : NULL);
    if (!s)
        do_hook(H_WORLD, "---- No world ----", "");
    else if (s->flags & SOCKDEAD)
        do_hook(H_WORLD, "---- World %s (dead) ----", "%s", s->world->name);
    else do_hook(H_WORLD, "---- World %s ----", "%s", s->world->name);
}

/* bring a new socket into the foreground */
static void fg_sock(new)
    Sock *new;
{
    if (new) {
        xsock = fsock = new;
        FD_SET(new->fd, &readers);
        if (new->flags & SOCKACTIVE) put_active(--active_count);
        new->flags &= ~SOCKACTIVE;
        announce_world(new);
        flush_world(new->world);
        if (lpflag || new->state == TELNET_GA || new->state == TELNET_EOR) {
            if (new->current_output->len) setprompt(new->current_output->s);
            Stringterm(new->current_output, 0);
        }
        if (new->flags & SOCKDEAD) {
            nukesock(new);
            fsock = xsock = NULL;
            any_sock();
        }
    } else announce_world(NULL);
}

static void bg_sock()
{
    if (!fsock) return;
    if (lpprompt->len) {
        SStringcpy(fsock->current_output, lpprompt);
        setprompt("");
    }
    fsock = xsock = NULL;
}

void no_sock()
{
    if (!fsock) return;
    bg_sock();
    announce_world(NULL);
}
 
/* try to open a new connection */
int connect_to(w, autologin)
    World *w;
    int autologin;
{
    int fd, tfcerrno = TFC_OK;
    Sock *sock;
#ifdef CONNECT
    int count;
    FILE *fp;
    struct timeval tv;
    fd_set pending;
#else
    struct sockaddr_in addr;
#endif

    /* Does a socket to this world already exist? */
    for (sock = hsock; sock != NULL; sock = sock->next) {
        if (sock->world == w &&
          (!(sock->flags & SOCKDEAD) || sock->flags & SOCKACTIVE)) {
            if (sock == fsock) return 1;
            if (sock->flags & SOCKPENDING) {
                oputs("% Connection already in progress.");
                return 0;  /* ??? */
            }
            bg_sock();
            fg_sock(sock);
            if (sock == fsock && sockmload) wload(sock->world);
            return 1;
        }
    }

#ifdef CONNECT
    fd = nb_connect(w->address, w->port, &fp, &tfcerrno);
#else
    fd = open_sock(w->address, w->port, &addr, &tfcerrno);
#endif
    if (fd < 0) {
        do_hook(H_CONFAIL, "%% Connection to %s failed: %s: %s", "%s %s: %s",
            w->name, TFCERROR(tfcerrno),
            tfcerrno==TFC_CANT_FIND ? w->address :
            (tfcerrno == TFC_SERVICE ? w->port : STRERROR(errno)));
        return 0;
    }
#ifndef CONNECT
    if (connect(fd, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) < 0) {
        close(fd);
        do_hook(H_CONFAIL, "%% Connection to %s failed: %s: %s", "%s %s: %s",
            w->name, TFCERROR(TFC_ECONNECT), STRERROR(errno));
        return 0;
    }
#endif
    /* create and initialize new Sock */
    if (fd >= nfds) nfds = fd + 1;
    FD_SET(fd, &readers);
    sock = (Sock *) MALLOC(sizeof(struct Sock));
    if (hsock == NULL) {
        sock->prev = NULL;
        tsock = hsock = sock;
    } else {
        sock->prev = tsock;
        tsock = tsock->next = sock;
    }
    sock->state = '\0';
    sock->fd = fd;
    sock->flags = SOCKECHO | SOCKEDIT | SOCKTRAP | (autologin ? SOCKLOGIN : 0);
    sock->world = w;
    sock->world->socket = sock;
    Stringinit(sock->current_output);
    sock->next = NULL;
#ifdef CONNECT
    sock->flags |= SOCKPENDING;
    sock->fp = fp;
    FD_ZERO(&pending);
    FD_SET(fd, &pending);
    tv.tv_sec = 0;
    tv.tv_usec = CONN_WAIT;                /* give it a chance */
    count = select(fd + 1, &pending, NULL, NULL, &tv);
    if (count == 1) {
        return establish(sock);
    } else {
        do_hook(H_PENDING, "% Connection to %s in progress.", "%s",
            sock->world->name);
        return 1;  /* Maybe this should be something else. */
    }
#else
    return establish(sock);
#endif
}

/* establish a pending connection */
static int establish(sock)
    Sock *sock;
{
    int fg = TRUE;
#ifdef CONNECT
    int fd;
    int tfcerrno;

#if 0
    if (sock->flags & SOCKPENDING && !switch_on_connect) fg = FALSE;
#endif
    sock->flags &= ~SOCKPENDING;
    FD_CLR(sock->fd, &readers);
    pclose(sock->fp);                /* Let child die.  We don't need status. */
    if (sock->flags & SOCKDEAD) {    /* Sock was killed already. Nuke it now. */
        nukesock(sock);
        return 0;
    }
    fd = recvfd(sock->fd, &tfcerrno);
    if (fd < 0) {
        killsock(sock);
        do_hook(H_CONFAIL, "%% Connection to %s failed: %s: %s", "%s %s: %s",
            sock->world->name, TFCERROR(tfcerrno),
            tfcerrno==TFC_CANT_FIND ? sock->world->address :
            (tfcerrno == TFC_SERVICE ? sock->world->port : STRERROR(errno)));
        return 0;
    }
    close(sock->fd);
    sock->fd = fd;
    if (fd >= nfds) nfds = fd + 1;
    FD_SET(fd, &readers);
#endif
    if (quiet && sock->flags & SOCKLOGIN && *sock->world->character)
        sock->numquiet = MAXQUIET;
    else
        sock->numquiet = 0;

    /* skip any old undisplayed lines */
    sock->world->history->index = sock->world->history->pos;
    if (fg) {
        bg_sock();
        fg_sock(sock);
    }
    wload(sock->world);
    do_hook(H_CONNECT, NULL, "%s", sock->world->name);
    magic_login(sock);
    return 1;
}

/* bring next or previous socket in list into foreground */
void movesock(dir)
    int dir;
{
    Sock *sock, *stop;

    reset_outcount();
    if (!hsock) return;
    stop = sock = (fsock ? fsock : hsock);
    do {
        if (dir > 0) sock = sock && sock->next ? sock->next : hsock;
        else sock = sock && sock->prev ? sock->prev : tsock;
    } while ((sock->flags & SOCKPENDING) && sock != stop);
    if (sock != fsock) {
        if (fsock && (sock->flags & SOCKECHO) != (fsock->flags & SOCKECHO))
            set_refresh_pending();
        bg_sock();
        fg_sock(sock);
        if (sockmload) wload(fsock->world);
    }
}

/* mark socket as dead (but don't delete it yet) */
static void killsock(sock)
    Sock *sock;
{
    sock->flags |= SOCKDEAD;
    dead_socks++;
}

/* remove socket from list and free memory */
static void nukesock(sock)
    Sock *sock;
{
    if (sock->world->socket == sock) {
        /* not true if this sock is defunct and a later /world was successful */
        sock->world->socket = NULL;
    }
    if (sock->flags & SOCKPENDING) return;
    if (sock->world->flags & WORLD_TEMP) {
        nuke_world(sock->world);
        sock->world = NULL;
    }
    if (sock == hsock) hsock = sock->next;
    else sock->prev->next = sock->next;
    if (sock == tsock) tsock = sock->prev;
    else sock->next->prev = sock->prev; dead_socks--;
    FD_CLR(sock->fd, &readers);
    close(sock->fd);
    if (sock->flags & SOCKACTIVE) put_active(--active_count);
    Stringfree(sock->current_output);
    FREE(sock);
}

/* delete all dead sockets */
static void nuke_dead_socks()
{
    Sock *sock, *next;
    int reconnect = FALSE;

    if (fsock && (fsock->flags & SOCKDEAD)) {
        xsock = fsock = NULL;
        reconnect = TRUE;
    }
    for (sock = hsock; sock; sock = next) {
        next = sock->next;
        if (sock->flags & SOCKDEAD) {
            if (sock->flags & SOCKACTIVE) FD_CLR(sock->fd, &readers);
            else nukesock(sock);
        }
    }
    if (quitdone && !hsock) done = 1;
    else if (reconnect) any_sock();
}

/* find any socket that can be brought into foreground, and do it */
static void any_sock()
{
    Sock *sock;

    if ((sock = find_sock(NULL))) {
        fg_sock(sock);
        if (sockmload) wload(fsock->world);
    } else announce_world(NULL);
}

/* close all sockets */
void disconnect_all()
{
    Sock *sock, *next;

    for (sock = hsock; sock; sock = next) {
        next = sock->next;
        nukesock(sock);
    }
    hsock = tsock = xsock = fsock = NULL;
    if (quitdone) done = 1;
}

/* disconnect from a world */
int disconnect(args)
    char *args;
{
    Sock *s;

    if (!*args) {
        if (fsock) killsock(fsock);
        else return 0;
    } else if (cstrcmp(args, "-all") == 0) {
        if (hsock) {
            disconnect_all();
            announce_world(NULL);
        } else return 0;
    } else {
        for (s = hsock; s; s = s->next) {
            if (cstrcmp(s->world->name, args) == 0 && !(s->flags & SOCKDEAD))
                break;
        }
        if (s) {
            killsock(s);
            oprintf ("%% Connection to %s closed.", s->world->name);
        } else {
            tfprintf(tferr, "%% Not connected to %s", args);
            return 0;
        }
    }
    return 1;
}

void listsockets()
{
    Sock *sock;
    char *state, buffer[81];
    int defunct;

    if (hsock == NULL) {
        oputs("% Not connected to any sockets.");
        return;
    }

    for (sock = hsock; sock != NULL; sock = sock->next) {
        defunct = (sock->flags & SOCKPENDING) && (sock->flags & SOCKDEAD);
        if (sock == xsock) state = "current";
        else if (defunct) state = "defunct";
        else if (sock->flags & SOCKPENDING) state = "pending";
        else if (sock->flags & SOCKDEAD) state = "dead";
        else if (sock->flags & SOCKACTIVE) state = "active";
        else state = "idle";
        sprintf(buffer, "%% [%7s]  %15s %30s %s", state,
            sock->world->name, sock->world->address, sock->world->port);
        oputs(buffer);
    }
}

/* call the background hook */
void background_hook(line)
    char *line;
{
    if (xsock == fsock) return;     /* trigger was in fg world */
    if (background > 1) return;     /* trigger was caused by /trigger */
    do_hook(H_BACKGROUND, "%% Trigger in world %s", "%s %s",
        xsock->world->name, line);
}

/* handle /send command */
int do_send(args)
    char *args;
{
    Sock *save = xsock, *sock = xsock;
    unsigned int len;
    int opt, Wflag = FALSE, nflag = 0;

    if (!hsock) {
        tfputs("% Not connected to any sockets.", tferr);
        return 0;
    }
    startopt(args, "w:Wn");
    while ((opt = nextopt(&args, NULL))) {
        switch (opt) {
        case 'w':
            if (*args) {
                if (!(sock = find_sock(args))) {
                    tfprintf(tferr, "%% Not connected to %s", args);
                    return 0;
                }
            } else sock = xsock;
            break;
        case 'W':
            Wflag = TRUE;
            break;
        case 'n':
            nflag = 1;
            break;
        default:
            return 0;
        }
    }
    
    args[len = strlen(args)] = '\n';            /* be careful */
    if (Wflag) {
        for (xsock = hsock; xsock; xsock = xsock->next)
            transmit(args, len + 1 - nflag);
    } else {
        xsock = sock;
        transmit(args, len + 1 - nflag);
    }
    args[len] = '\0';                           /* restore end of string */
    xsock = save;
    return 1;
}

/* tramsmit text to current socket */
int transmit(str, numtowrite)
    char *str;
    unsigned int numtowrite;
{
    int numwritten;

    if (!xsock || xsock->flags & (SOCKDEAD | SOCKPENDING)) return 0;
    while (numtowrite) {
        numwritten = send(xsock->fd, str, numtowrite, 0);
        if (numwritten < 0) {
#ifdef EWOULDBLOCK
            if (errno == EWOULDBLOCK || errno == EAGAIN) numwritten = 0;
#else
            if (errno == EAGAIN) numwritten = 0;
#endif
            else {
                killsock(xsock);
                do_hook(H_DISCONNECT,
                    "%% Connection to %s closed by foreign host: %s",
                    "%s %s", xsock->world->name, STRERROR(errno));
                return 0;
            }
        }
        numtowrite -= numwritten;
        str += numwritten;
        if (numtowrite) sleep(1);
    }
    return 1;
}

/* call login hook */
static void magic_login(sock)
    Sock *sock;
{
    World *w;

    if (!(login && sock->flags & SOCKLOGIN)) return;
    w = (*sock->world->character) ? sock->world : get_default_world();
    if (!w) return;
    if (!*w->character) return;
    do_hook(H_LOGIN, NULL, "%s %s %s", w->name, w->character, w->pass);
}

void clear_refresh_pending()
{
    need_refresh = 0;
}

void set_refresh_pending()
{
    need_refresh = 1;
}

int is_refresh_pending()
{
    return need_refresh;
}

static void handle_socket_line(sock, out)
    Sock *sock;
    Stringp out;
{
    unsigned int len = out->len;
    short attrs = F_NORM;

    if (len && out->s[len - 1] == '\r') {
        while (--len && out->s[len - 1] == '\r');
        Stringterm(out, len);
    }
    attrs = special_hook(sock->world->history, out->s, &sock->numquiet);
    world_output(sock->world, out->s, attrs);
}

/* log, record, and display string as if it came from world w */
void world_output(w, str, attrs)
    World *w;
    char *str;
    int attrs;
{
    Aline *aline;

    if (gag && (attrs & F_GAG) && (attrs & F_NOHISTORY) && !w->history->logfile)
        return;
    aline = new_aline(str, attrs | F_NEWLINE);
    record_world(w->history, aline);
    if (!(gag && (attrs & F_GAG))) {
        if (fsock && w == fsock->world) {
            globalout(aline);
        } else {
            aline->links++;
            enqueue(w->queue, aline);
            if (!(w->socket->flags & SOCKACTIVE)) {
                w->socket->flags |= SOCKACTIVE;
                put_active(++active_count);
                do_hook(H_ACTIVITY, "%% Activity in world %s", "%s", w->name);
            }
        }
    }
}

static void do_prompt(sock)
    Sock *sock;
{
    if (lpquote) runall(time(NULL));
    check_trigger(sock->current_output->s);
    if (sock == fsock) {
        setprompt(sock->current_output->s);
        Stringterm(sock->current_output, 0);
    }
}

/* handle input from socket */
static void handle_socket_input(sock)
    Sock *sock;
{
    char *place;
    fd_set readfds;
    int count = 1;
    char buffer[512];
    struct timeval tv;
    Sock *oldsock = xsock;
    char cmd[4];

    xsock = sock;

    if (sock == fsock && lpprompt->len && sock->state != '\n'
        && sock->state != TELNET_EOR && sock->state != TELNET_GA) {
        /* what we assumed was a prompt really wasn't */
        SStringcpy(sock->current_output, lpprompt);
        setprompt("");
    }

    while (count > 0) {
        do count = recv(sock->fd, buffer, 512, 0);
            while (count < 0 && errno == EINTR);
        if (count <= 0) {
            if (count < 0) operror("recv failed");
            flush_output(sock);
            killsock(sock);
            do_hook(H_DISCONNECT, "%% Connection to %s closed.",
                "%s", sock->world->name);
            return;
        }

        place = buffer;
        while (place - buffer < count) {
            if (sock->state == TELNET_IAC) {
                sock->state = *place;
                if (*place == TELNET_IAC) {
                    /* Literal IAC.  Ignore it. */
                    sock->state = '\0';
                } else if (*place == TELNET_GA || *place == TELNET_EOR) {
                    /* This is definately a prompt. */
                    do_prompt(sock);
                } else if (*place == TELNET_DO) {
                } else if (*place == TELNET_DONT) {
                } else if (*place == TELNET_WILL) {
                } else if (*place == TELNET_WONT) {
                } else {
                    /* shouldn't happen. ignore it. */
                }
                place++;
            } else if (sock->state == TELNET_WILL) {
                if (*place == TELNET_ECHO) {
                    if (sock->flags & SOCKECHO) {
                        /* stop local echo, and acknowledge */
                        sock->flags &= ~SOCKECHO;
                        sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_DO, *place);
                        transmit(cmd, 3);
                    } else {
                        /* we already said DO ECHO, so ignore WILL ECHO */
                    }
                } else if (*place == TELNET_EOR_OPT) {
                    if (!(sock->flags & SOCKEOR)) {
                        sock->flags |= SOCKEOR;
                        sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_DO, *place);
                        transmit(cmd, 3);
                    } else {
                        /* we already said DO EOR_OPT, so ignore WILL EOR_OPT */
                    }
                } else {
                    /* don't accept other WILL offers */
                    sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_DONT, *place);
                    transmit(cmd, 3);
                }
                sock->state = '\0';
                place++;
            } else if (sock->state == TELNET_WONT) {
                if (*place == TELNET_ECHO) {
                    if (sock->flags & SOCKECHO) {
                        /* we're already echoing, so ignore WONT ECHO */
                    } else {
                        /* resume local echo, and acknowledge */
                        sock->flags |= SOCKECHO;
                        sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_DONT, *place);
                        transmit(cmd, 3);
                    }
                } else if (*place == TELNET_EOR_OPT) {
                    if (!(sock->flags & SOCKEOR)) {
                        /* we're in DONT EOR_OPT state, ignore WONT EOR_OPT */
                    } else {
                        /* acknowledge */
                        sock->flags &= ~SOCKEOR;
                        sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_DONT, *place);
                        transmit(cmd, 3);
                    }
                } else {
                    /* we're already in the WONT state, so ignore WONT */
                }
                sock->state = '\0';
                place++;
            } else if (sock->state == TELNET_DO) {
                /* refuse all DO requests */
                sprintf(cmd, "%c%c%c", TELNET_IAC, TELNET_WONT, *place);
                transmit(cmd, 3);
                sock->state = '\0';
                place++;
            } else if (sock->state == TELNET_DONT) {
                /* ignore all DONT requests (we're already in the DONT state) */
                sock->state = '\0';
                place++;
            } else if (*place == TELNET_IAC) {
                sock->state = *place++;
            } else if (*place == '\n') {
                /* Complete line received.  Process it. */
                handle_socket_line(sock, sock->current_output);
                Stringterm(sock->current_output, 0);
                sock->state = *place++;
            } else if (*place == '\r' || *place == '\0') {
                sock->state = *place++;
            } else if (*place == '\t') {
                Stringnadd(sock->current_output, ' ',
                    8 - sock->current_output->len % 8);
                sock->state = *place++;
            } else {
                Stringadd(sock->current_output, *place);
                sock->state = *place++;
            }
        }

        FD_ZERO(&readfds);
        FD_SET(sock->fd, &readfds);
        if (lpflag && sock->current_output->len && sock == fsock) {
            tv.tv_sec = prompt_sec;
            tv.tv_usec = prompt_usec;
        } else tv.tv_sec = tv.tv_usec = 0;

        while ((count = select(sock->fd + 1, &readfds, NULL, NULL, &tv)) < 0) {
            if (errno != EINTR) {
                operror("TF/receive/select");
                die("% Failed select");
            }
        }

    }
    xsock = oldsock;

    if (sock->current_output->len) {
        if (lpflag) {
            /* assume it's a prompt */
            do_prompt(sock);
        } else {
            /* Partial line.  Hold on to it pending arrival of rest of line. */
        }
    }
}

/* process any socket text that has been received but not processed yet */
static void flush_output(sock)
    Sock *sock;
{
    if (sock->current_output->len) {
        handle_socket_line(sock, sock->current_output);
        Stringterm(sock->current_output, 0);
    }
    setprompt("");
}

void set_done()
{
    done = TRUE;
}

#ifdef CONNECT
/* Get an open file descriptor or error from connector process */
/* Optimization breaks this (on BSD 4.4, at least) */

#ifdef CONNECT_SVR4
static int recvfd(sockfd, flag)
    int sockfd, *flag;
{
    struct strrecvfd rfd;

    if (ioctl(sockfd, I_RECVFD, (char*)&rfd) < 0) {
        *flag = TFC_ERECV;
    } else return rfd.fd;
    return -1;
}
#endif

#ifdef CONNECT_BSD
static int recvfd(sockfd, flag)
    int sockfd, *flag;
{
    struct iovec iov[1];
    struct msghdr msg;
    int error;
# ifdef CONNECT_BSD44                         /* BSD 4.3 Reno or later */
    struct cmsghdr cmptr[1];
# else
    int fd;
# endif

    if ((recv(sockfd, flag, sizeof(*flag), 0)) < 0) {
        *flag = TFC_ERECV;
    } else if (*flag == TFC_OK) {
        iov[0].iov_base = NULL;
        iov[0].iov_len = 0;
        msg.msg_name = NULL;
        msg.msg_namelen = 0;
        msg.msg_iov = iov;
        msg.msg_iovlen = 1;
# ifdef CONNECT_BSD44                         /* BSD 4.3 Reno or later */
        msg.msg_control = (caddr_t)cmptr;
        msg.msg_controllen = CONTROLLEN;
        if (recvmsg(sockfd, &msg, 0) >= 0) return *(int*)CMSG_DATA(cmptr);
# else
        msg.msg_accrights = (char*) &fd;
        msg.msg_accrightslen = sizeof(fd);
        if (recvmsg(sockfd, &msg, 0) >= 0) return fd;
# endif
        *flag = TFC_ERECVMSG;
    } else {
        if ((recv(sockfd, &error, sizeof(error), 0)) < 0) *flag = TFC_ERECV;
        errno = error;
    }
    return -1;
}
#endif

/*
 * nb_connect simulates a non-blocking connect() by forking a child to
 * do the actual connect().  The child sends the connected file descriptor
 * or error code back to the parent through a stream pipe when done.
 * The parent reads the fd or error from the pipe when the pipe selects
 * true for reading.  popen() and pclose() provide a portable way of
 * wait()ing for the child (we don't use the pipe it creates).
 */

static int nb_connect(address, port, fpp, tfcerrnop)
    char *address, *port;
    FILE **fpp;
    int *tfcerrnop;
{
    int sv[2];
    struct stat statbuf;
    STATIC_BUFFER(cmd);

    *tfcerrnop = TFC_ESTAT;
    if (stat(TFCONNECT, &statbuf) < 0) return -1;
#ifdef CONNECT_SVR4
    *tfcerrnop = TFC_EPIPE;
    if (pipe(sv) < 0) return -1;
#else
    *tfcerrnop = TFC_ESOCKETPAIR;
    if (socketpair(AF_UNIX, SOCK_STREAM, 0, sv) < 0) return -1;
#endif
    Sprintf(cmd, "%s %s %s %d", TFCONNECT, address, port, sv[0]);
    *tfcerrnop = TFC_EPOPEN;
    if ((*fpp = popen(cmd->s, "w")) == NULL) return -1;
    close(sv[0]);                     /* parent doesn't need this */
    *tfcerrnop = TFC_OK;
    return sv[1];
}
#endif /* CONNECT */

