/*
 * ------------------------------------------------------------------
 * Dice.cc - Dice Implementation
 * Created by Robert Heller on Sat Feb 25 18:08:34 1995
 * ------------------------------------------------------------------
 * Modification History: $Log: Dice.cc,v $
// Revision 1.3  1995/03/18  20:03:50  heller
// Added handle magic
//
// Revision 1.2  1995/02/26  14:56:55  heller
// Add I/O functions and tcl interface code
//
// Revision 1.1  1995/02/25  23:39:20  heller
// Initial revision
//
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Role Playing Database -- a program for maintaining a database
 *                              for RPG characters and monsters
 *     Copyright (C) 1995  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

static char rcsid[] = "$Id: Dice.cc,v 1.3 1995/03/18 20:03:50 heller Exp $";

#include <Dice.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <MLCG.h>
#include <fstream.h>
#include <strstream.h>
#include <tclExtend.h>

void Dice::_Dice()
{
	if (gen != NULL) delete gen;
	if (dice != NULL) delete dice;

	struct timeval now;
	struct timezone x;
	gettimeofday(&now,&x);
	gen = new MLCG(now.tv_sec,now.tv_usec);
	if (nsides == 0) dice = new RandomInteger(0,99,gen);
	else dice = new RandomInteger(1,nsides,gen);
}

Dice::~Dice()
{
	if (gen != NULL) delete gen;
	if (dice != NULL) delete dice;
}

unsigned int Dice::Roll()
{
	if (dice == NULL) return 0;
	if (nsides == 0) return (*dice)();
	else
	{
		unsigned int result = 0;
		for (int i = 0;i < ndice;i++) result += (*dice)();
		return result;
	}
}

static int WriteToFile(char *filename,Dice& d)
{
	ofstream stream(filename);
	if (!stream) return(0);
	stream << d;
	if (!stream) return(0);
	stream.close();
	if (!stream) return(0);
	return 1;
}

static int ReadFromFile(char *filename,Dice& d)
{
	ifstream stream(filename);
	if (!stream) return(0);
	stream >> d;
	if (!stream) return(0);
	stream.close();
	if (!stream) return(0);
	return 1;
}

static void_pt DiceHandles = NULL;

int Dice::TclFunction(Tcl_Interp *interp,int argc, char *argv[])
{
	if (argc == 1)
	{
		Tcl_AppendElement(interp, "Dice");
		if (nsides == 0)
		{
			Tcl_AppendElement(interp, "%");
		} else
		{
			{
				ostrstream stream;
				stream << ndice;
				_IO_ssize_t i = stream.pcount();
				char *s = stream.str();
				s[i] = '\0';
				Tcl_AppendElement(interp,s);
				stream.freeze(0);
			}
			{
				ostrstream stream;
				stream << nsides;
				_IO_ssize_t i = stream.pcount();
				char *s = stream.str();
				s[i] = '\0';
				Tcl_AppendElement(interp,s);
				stream.freeze(0);
			}
		}
		return TCL_OK;
	}
	if (strcmp(argv[1], "type") == 0)
	{
		Tcl_AppendResult(interp, "Dice",(char *) NULL);
		return TCL_OK;
	} else if (strcmp(argv[1], "write") == 0)
	{
		if (argc != 3)
		{
			Tcl_AppendResult(interp, "wrong # args: should be \"",
			 		  argv[0]," ",argv[1]," filename\"",
					  (char *) NULL);
			return TCL_ERROR;
		}
		if (!WriteToFile(argv[2],*this))
		{
			Tcl_AppendResult(interp, "Error writing file ",argv[2],
					 (char *) NULL);
			return TCL_ERROR;
		} else return TCL_OK;
	} else if (strcmp(argv[1], "read") == 0)
	{
		if (argc != 3)
		{
			Tcl_AppendResult(interp, "wrong # args: should be \"",
			 		  argv[0]," ",argv[1]," filename\"",
					  (char *) NULL);
			return TCL_ERROR;
		}
		if (!ReadFromFile(argv[2],*this))
		{
			Tcl_AppendResult(interp, "Error reading file ",argv[2],
					 (char *) NULL);
			return TCL_ERROR;
		} else return TCL_OK;
	} else if (strcmp(argv[1], "roll") == 0)
	{
		if (argc != 2)
		{
			Tcl_AppendResult(interp, "wrong # args: should be \"",
					 argv[0]," ",argv[1],"\"",
					 (char *) NULL);
			return TCL_ERROR;
		}
		ostrstream stream;
		stream << Roll();
		_IO_ssize_t i = stream.pcount();
		char *s = stream.str();
		s[i] = '\0';
		Tcl_AppendResult(interp,s,(char *) NULL);
		return TCL_OK;
	} else if (strcmp(argv[1], "delete") == 0)
	{
		if (argc != 2)
		{
			Tcl_AppendResult(interp, "wrong # args: should be \"",
					 argv[0]," ",argv[1],"\"",
					 (char *) NULL);
			return TCL_ERROR;
		}
		void_pt header = Tcl_HandleXlate (interp,DiceHandles,argv[0]);
		if (header == NULL) return TCL_ERROR;
		Tcl_HandleFree (DiceHandles,header);
		return Tcl_DeleteCommand(interp,argv[0]);
	} else
	{
		Tcl_AppendResult(interp, "Bad option: ",argv[1],(char *) NULL);
		return TCL_ERROR;
	}
}

static void deleteDice(ClientData clientData)
{
	register Dice *dice = (Dice*) clientData;
	delete dice;
}

static int diceCommand(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	register Dice *dice = (Dice*) clientData;
	return dice->TclFunction(interp,argc,argv);
}

static int DiceCreate(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	register Dice *dice;
	if (argc < 1 || argc > 3)
	{
		Tcl_AppendResult(interp, "Wrong # args: should be \"",
				 argv[0]," ?numsides numdice?\"",
				 (char*) NULL);
		return TCL_ERROR;
	}
	if (argc == 1)
	{
		/* nothing but the handle */
		dice = new Dice;
	} else if (argc == 2)
	{
		/* handle + numsides */
		int ns;
		if (Tcl_GetInt(interp,argv[1],&ns) != TCL_OK)
			return(TCL_ERROR);
		dice = new Dice(ns);
	} else
	{
		/* handle + numsides + numdice */
		int ns,nd;
		if (Tcl_GetInt(interp,argv[1],&ns) != TCL_OK)
			return(TCL_ERROR);
		if (Tcl_GetInt(interp,argv[2],&nd) != TCL_OK)
			return(TCL_ERROR);
		dice = new Dice(ns,nd);
	}
	static char handle[32];
	Dice** h = (Dice**) Tcl_HandleAlloc (DiceHandles,handle);
	*h = dice;
	Tcl_CreateCommand(interp,handle,(Tcl_CmdProc*)diceCommand,
			  (ClientData)dice,
			  (Tcl_CmdDeleteProc*)deleteDice);
	Tcl_AppendResult(interp,handle,(char *) NULL);
	return TCL_OK;
}

static int AllDice(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int walk;
	walk = -1;
	while (Tcl_HandleWalk(DiceHandles,&walk) != NULL)
	{
		char handle[32];
		Tcl_WalkKeyToHandle(DiceHandles,walk,handle);
		Tcl_AppendElement(interp,handle);
	}
	return TCL_OK;

}

static int FindDice(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int walk;
	int ns_pat,nd_pat;
	unsigned int ns,nd;
	Dice **dp;

	if (argc < 2 || argc > 3)
	{
		Tcl_AppendResult(interp, "Wrong # args: should be \"",
				 argv[0]," ?numsides numdice?\"",
				 (char*) NULL);
		return TCL_ERROR;
	}
	if (argc == 2)
	{
		nd_pat = 1;
		if (strcmp(argv[1],"%") == 0) ns_pat = 0;
		else
		{
			if (Tcl_GetInt(interp,argv[1],&ns_pat) != TCL_OK)
				return TCL_ERROR;
		}
	} else
	{
		if (Tcl_GetInt(interp,argv[1],&nd_pat) != TCL_OK)
			return TCL_ERROR;
		if (Tcl_GetInt(interp,argv[2],&ns_pat) != TCL_OK)
			return TCL_ERROR;
	}
	walk = -1;
	while ((dp = (Dice**) Tcl_HandleWalk(DiceHandles,&walk)) != NULL)
	{
		Dice *d = *dp;
		if (d->TypeOfDice(ns,nd))
		{
			if (ns == ns_pat && nd == nd_pat)
			{
				char handle[32];
				Tcl_WalkKeyToHandle(DiceHandles,walk,handle);
				Tcl_AppendElement(interp,handle);
			}
		}
	}
	return TCL_OK;
}



int Dice_Init(Tcl_Interp *interp)
{
	DiceHandles = Tcl_HandleTblInit("Dice",sizeof(Dice*),256);
	Tcl_CreateCommand(interp, "Dice", (Tcl_CmdProc*)DiceCreate,
			  (ClientData) NULL, (Tcl_CmdDeleteProc*) NULL);
	Tcl_CreateCommand(interp, "AllDice", (Tcl_CmdProc*)AllDice,
			  (ClientData) NULL, (Tcl_CmdDeleteProc*) NULL);
	Tcl_CreateCommand(interp, "FindDice", (Tcl_CmdProc*)FindDice,
			  (ClientData) NULL, (Tcl_CmdDeleteProc*) NULL);
	return TCL_OK;
}


ostream& operator << (ostream& s,Dice& d)
{
	if (d.nsides == 0) s << "1d%";
	else s << d.ndice << "d" << d.nsides;
}

istream& operator >> (istream& s,Dice& d)
{
	int nd, ns;
	char dch;
	s >> nd;
	s >> dch;
	if (dch != 'd')
	{
		s.set(ios::badbit);
		return s;
	}
	s >> dch;
	if (dch == '%')
	{
		if (nd != 1)
		{
			s.set(ios::badbit);
			return s;
		}
		d.ndice = 1;
		d.nsides = 0;
	} else
	{
		s.putback(dch);
		s >> ns;
		d.ndice = nd;
		d.nsides = ns;
	}
	d._Dice();
	return s;
}
