/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <stdlib.h>
#include "life.h"


/*
 * List of line command routines.
 */
static	void	CmdDestroy PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdEdit PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdDumpInitData PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdQuit PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdAdvance PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdFrequency PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdObjects PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdZero PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdRead PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdWrite PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdTtyInput PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdGrid PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdNoGrid PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdInsert PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdClearPath PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdRestorePath PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdShortenPath PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdCopy PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdCopyMarked PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdMove PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdMoveMarked PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdHelp PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdRule PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdEndInputLevel PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdUndo PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdRename PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdSet PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdVariables PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdType PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdUpdate PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdWait PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdMode PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdUnMark PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdInvertMarks PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdAssign PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdList PROTO((char *, VALUE, VALUE, BOOL, BOOL));
static	void	CmdDefault PROTO((char *, VALUE, VALUE, BOOL, BOOL));


/*
 * Flags for line commands.
 */
#define	F_NONE		0x00	/* no special condition */
#define	F_NOARG		0x01	/* must not have an argument */
#define	F_ARG1		0x02	/* must have at least one argument */
#define	F_NORUN		0x04	/* illegal if running generations */
#define	F_UPDATE	0x08	/* update status if command completes */
#define	F_REDRAW	0x10	/* redraw screen if command completes */
#define	F_ABBR		0x20	/* abbreviation works even if ambiguous */


/*
 * Dispatch table for line commands.  Those commands which are ambiguous
 * must be defined so as to be contiguous in the table.  A spaces delimits
 * the command itself from the help string for the command.
 */
typedef	struct	{
	char	*name;
	void	(*func) PROTO((char *, VALUE, VALUE, BOOL, BOOL));
	int	flags;
} CMDTAB;


static	CMDTAB	cmdtab[] = {
	{"advance (marked cells) expr (gens)", CmdAdvance, F_NORUN|F_ABBR},
	{"assign (button) num (to macro) ch", CmdAssign, F_ARG1|F_NORUN},
	{"clearpath (from current object)", CmdClearPath, F_NOARG|F_NORUN},
	{"copy (current object to) obj", CmdCopy, F_ARG1|F_NORUN|F_ABBR},
	{"copymarked (cells to) obj", CmdCopyMarked, F_ARG1|F_NORUN},
	{"destroy (object named) obj", CmdDestroy, F_ARG1|F_NORUN},
	{"default param (to) expr", CmdDefault, F_ARG1|F_ABBR},
	{"dump (initialization data to) file", CmdDumpInitData, F_ARG1|F_NORUN},
	{"edit (object named) obj", CmdEdit, F_NORUN|F_REDRAW|F_ABBR},
	{"endinputlevel", CmdEndInputLevel, F_NOARG|F_UPDATE},
	{"frequency (of typeout is) expr", CmdFrequency, F_UPDATE},
	{"grid (character is) char", CmdGrid, F_REDRAW},
	{"help", CmdHelp, F_NONE|F_ABBR},
	{"insert (object from) obj", CmdInsert, F_ARG1|F_NORUN|F_ABBR},
	{"invert (marks for all cells)", CmdInvertMarks, F_NOARG|F_NORUN|F_REDRAW},
	{"list (readable *.l files)", CmdList, F_REDRAW|F_ABBR},
	{"mode (for movement)", CmdMode, F_NORUN|F_UPDATE},
	{"move (current object to) obj", CmdMove, F_ARG1|F_NORUN|F_REDRAW|F_ABBR},
	{"movemarked (cells to) obj", CmdMoveMarked, F_ARG1|F_NORUN|F_REDRAW},
	{"nogrid", CmdNoGrid, F_NOARG|F_REDRAW},
	{"objects (are listed)", CmdObjects, F_ABBR},
	{"quit (program)", CmdQuit, F_NOARG|F_ABBR},
	{"read (commands from) file", CmdRead, F_ARG1|F_NORUN|F_REDRAW|F_ABBR},
	{"rename (current object to) obj", CmdRename, F_ARG1|F_UPDATE},
	{"restorepath (to current object)", CmdRestorePath, F_NOARG|F_NORUN},
	{"rule (for life are) born,live", CmdRule, F_NORUN|F_UPDATE},
	{"set (variable) name (to) expr", CmdSet, F_ARG1|F_ABBR},
	{"shortenpath (from current object)", CmdShortenPath, F_NORUN},
	{"ttyinput", CmdTtyInput, F_REDRAW},
	{"type (value of expression) expr", CmdType, F_ARG1|F_UPDATE|F_ABBR},
	{"undo (last change)", CmdUndo, F_NOARG|F_NORUN|F_REDRAW|F_ABBR},
	{"unmark (all cells)", CmdUnMark, F_NOARG|F_NORUN|F_REDRAW},
	{"update (view to be current)", CmdUpdate, F_NOARG},
	{"variables (are listed)", CmdVariables, F_NOARG|F_REDRAW|F_ABBR},
	{"wait (for computations)", CmdWait, F_NOARG},
	{"write (current object to) file", CmdWrite, F_ARG1|F_NORUN|F_ABBR},
	{"zero (current object)", CmdZero, F_NOARG|F_NORUN|F_REDRAW|F_ABBR},
	{0, 0, 0}				/* ends the table */
};


/*
 * Read and execute a line mode command.  This kind of command echoes,
 * and is terminated by an end of line.  Numeric arguments are available.
 */
void
DoLineCommand(arg1, arg2, got1, got2)
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	CMDTAB *	cmd;		/* command structure */
	CMDTAB *	cmdworks;	/* command that was found */
	CMDTAB *	cmdbest;	/* command found that is wanted */
	char *		name;		/* command name */
	char *		args;		/* arguments for command */
	int		flag;		/* flags for command */
	int		matches;	/* number of matches found */

	name = ReadString("command: ");

	while (isblank(*name))
		name++;

	if (*name == '\0')
		return;

	args = name + strlen(name);

	while (isblank(args[-1]))
		args--;

	*args = '\0';

	for (args = name; *args && !isblank(*args); args++)
		;

	while (isblank(*args))
		*args++ = '\0';

	/*
	 * Seatch the table for the command.
	 * Exact match always wins, followed by an indicated abbreviation,
	 * followed by a unique abbreviation.
	 */
	cmdbest = NULL;
	cmdworks = NULL;
	matches = 0;

	for (cmd = cmdtab; cmd->name != NULL; cmd++) {
		if (strcmp(name, cmd->name) == 0) {
			cmdbest = cmd;
			break;
		}

		if (!IsAbbreviation(name, cmd->name))
			continue;

		cmdworks = cmd;
		matches++;

		if (cmd->flags & F_ABBR)
			cmdbest = cmd;
	}

	cmd = cmdbest;

	if (cmd == NULL) {
		cmd = cmdworks;

		if (cmd == NULL)
			Error("Unknown line command");

		if (matches > 1)
			Error("Ambiguous line command");
	}

	/*
	 * OK, we have found the command.  Check it and execute it.
	 */
	flag = cmd->flags;

	if (flag & F_NORUN)
		CheckRun();

	if ((flag & F_ARG1) && (*args == '\0'))
		Error("Missing argument");

	if ((flag & F_NOARG) && *args)
		Error("Argument not allowed");

	(*cmd->func)(args, arg1, arg2, got1, got2);

	if (flag & F_UPDATE)
		curobj->update |= U_STAT;

	if (flag & F_REDRAW)
		curobj->update |= U_ALL;
}


/*
 * Advance marked cells by the specified number of generations.
 */
static void
CmdAdvance(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	COUNT	count;

	count = 1;

	if (*cp) {
		count = GetExpression(cp);

		if (count <= 0)
			return;
	}

	Backup();

	CopyMarkedObject(curobj, tempobject, MARK_USR);
	MoveMarkedObject(curobj, deleteobject, MARK_USR);

	genleft = count;

	while ((genleft > 0) && !stop)
		DoGeneration(tempobject);

	curobj->mark = MARK_USR;
	AddObject(tempobject, curobj, RELATIVE, FALSE);
	curobj->mark = MARK_ANY;
}


/*
 * Assign a button to a macro.
 */
static void
CmdAssign(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	VALUE	button;
	int	macro;

	button = 0;

	while (isdigit(*cp))
		button = button * 10 + *cp++ - '0';

	if (*cp && !isblank(*cp))
		Error("Illegal button value");

	while (isblank(*cp))
		cp++;

	macro = *cp;

	if (macro)
		cp++;

	while (isblank(*cp))
		cp++;

	if (*cp || !MacroIsValid(macro))
		Error("Illegal macro name");

	if (!(*dev->assign)(dev, button, macro))
		Error("Button is not assignable");
}


/*
 * Clear the path from the current object.
 */
static void
CmdClearPath(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	Backup();
	ClearPath(curobj);
}


/*
 * Shorten the path from the current object by one or more segments.
 */
static void
CmdShortenPath(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	COUNT	count;

	count = 1;

	if (*cp)
		count = GetExpression(cp);

	if (count <= 0)
		Error("Bad shorten count value");

	Backup();

	ShortenPath(curobj, count);
}


/*
 * Restore the path that was set for an object.
 */
static void
CmdRestorePath(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	Backup();

	if (!RestorePath(curobj))
		Error("There is no path to restore");
}


/*
 * Copy the current object into another object.
 */
static void
CmdCopy(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	CopyObject(curobj, GetObject(cp));
}


/*
 * Copy the currently marked cells into another object.
 */
static void
CmdCopyMarked(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	CopyMarkedObject(curobj, GetObject(cp), MARK_USR);
}


/*
 * Set a default value for new objects.
 */
static void
CmdDefault(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	char *	param;
	VALUE	val;

	param = cp;

	while (*cp && !isblank(*cp))
		cp++;

	while (isblank(*cp))
		*cp++ = '\0';

	if (*cp == '\0')
		cp = "1";

	if (IsAbbreviation(param, "scale")) {
		val = GetExpression(cp);

		if ((val == 0) || (val > MAXSCALE))
			Error("Bad scale value");

		if (val == -1)
			val = 1;

		defaultscale = val;

		return;
	}

	if (IsAbbreviation(param, "frequency")) {
		val = GetExpression(cp);

		if (val <= 0)
			Error("Illegal frequency value");

		defaultfrequency = val;

		return;
	}

	Error("Illegal parameter for defaulting");
}


/*
 * Destroy an existing object.
 */
static void
CmdDestroy(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	OBJECT *obj;

	obj = FindObject(cp);

	if (obj == NULL)
		Error("No such object");

	DestroyObject(obj);
}


/*
 * Dump things that the user might want initialized at startup to a file.
 * These are things like the current macros and button assignments.
 */
static void
CmdDumpInitData(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	FILE *	fp;

	fp = fopen(cp, "w");

	if (fp == NULL)
		Error("Cannot create file");

	WriteMacros(fp);
	(*dev->writeassign)(dev, fp);

	fflush(fp);

	if (ferror(fp)) {
		fclose(fp);
		Error("Write failed");
	}

	if (fclose(fp))
		Error("Close failed");
}


/*
 * Edit an object.  A null argument implies the previous object.
 */
static void
CmdEdit(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	OBJECT *obj;

	obj = prevobj;

	if (*cp) {
		obj = FindObject(cp);

		if (obj == NULL)
			obj = GetObject(cp);
	}

	if (obj == NULL)
		Error("No such object");

	SetObject(obj);
}


/*
 * Undo the last change made to the current object.
 */
static void
CmdUndo(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	MoveObject(curobj, tempobject);
	MoveObject(backupobject, curobj);
	MoveObject(tempobject, backupobject);
}


/*
 * End current input level.
 */
static void
CmdEndInputLevel(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if (curinput <= inputs)
		Error("Cannot end top level input");

	(*curinput->term)(curinput);
}


/*
 * Update the view even if inside of a loop or macro.
 */
static void
CmdUpdate(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	curobj->update |= U_ALL;	/* IS THIS NECESSARY?? */
	UpdateView();
}


/*
 * Wait until all outstanding generation computations are finished.
 */
static void
CmdWait(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	while (!stop && (genleft > 0)) {
		DoGeneration(curobj);
		UpdateView();
	}
}


/*
 * Set output frequency for object.
 */
static void
CmdFrequency(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	COUNT	freq;

	freq = 1;

	if (*cp)
		freq = GetExpression(cp);

	if (freq <= 0)
		Error("Illegal frequency value");

	curobj->frequency = freq;
	freqcount = freq;
}


/*
 * Set the mode associated with movement.
 */
static void
CmdMode(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if ((*cp == '\0') || IsAbbreviation(cp, "move")) {
		mode = M_MOVE;
		return;
	}

	if (IsAbbreviation(cp, "insert")) {
		mode = M_INSERT;
		return;
	}

	if (IsAbbreviation(cp, "delete")) {
		mode = M_DELETE;
		return;
	}

	Error("Illegal movement mode");
}


/*
 * Unmark all cells.
 */
static void
CmdUnMark(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	ClearMarks(curobj, MARK_SEE);
}


/*
 * Invert the marks for all cells.
 */
static void
CmdInvertMarks(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	InvertMarks(curobj, MARK_USR);
}


/*
 * Select the character used for the background of the screen.
 */
static void
CmdGrid(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if (*cp == '\0')
		cp = ".";

	if ((*cp < ' ') || (cp[1] != '\0'))
		Error("Bad grid character");

	if (*cp != gridchar)
		curobj->update |= U_REDRAW;

	gridchar = *cp;
}


/*
 * Set so we don`t see a grid on the screen.
 */
static void
CmdNoGrid(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if (gridchar != ' ')
		curobj->update |= U_REDRAW;

	gridchar = ' ';
}


/*
 * Type list of line commands.
 */
static void
CmdHelp(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	CMDTAB *	cmd;
	int		count;
	char		buf[80];

	ShowHelp("\
The following table lists all line mode commands.  Unique abbreviations are\n\
allowed.  Commands shown with '*' can be abbreviated even when ambiguous.\n");

	count = 0;

	for (cmd = cmdtab; cmd->name; cmd++) {
		if ((count++ % 2) == 0)
			ShowHelp("\n");

		sprintf(buf, "%c %-35s", ((cmd->flags & F_ABBR) ? '*' : ' '),
			cmd->name);

		ShowHelp(buf);
	}

	ShowHelp("\n");
	EndHelp();
}


/*
 * Display a list of the *.l files that can be read.
 * If a pattern is given, only list files matching that pattern.
 */
static void
CmdList(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	int	i;
	char *	dirname;
	LIST	names;

	names.argc = 0;
	names.maxargc = 0;
	names.used = 0;
	names.maxused = 0;

	/*
	 * Look for an option, which must be -d.
	 * If found, then look in that particular directory.
	 * Otherwise look in the normal places.
	 */
	if (*cp == '-') {
		dirname = cp;

		while (*cp && !isblank(*cp))
			cp++;

		if ((cp != dirname + 2) || (memcmp(dirname, "-d", 2) != 0))
			Error("Bad option for list command");

		while (isblank(*cp))
			cp++;

		dirname = cp;

		if (*dirname == '\0')
			dirname = ".";

		while (*cp && !isblank(*cp))
			cp++;

		if (*cp)
			*cp++ = '\0';

		while (isblank(*cp))
			cp++;

		GetFiles(&names, dirname, cp, strlen(dirname) + 1, 1);
	} else {
		GetFiles(&names, ".", cp, 2, 1);

		for (i = 0; i < userlibcount; i++) {
			GetFiles(&names, userlibs[i], cp,
				strlen(userlibs[i]) + 1, MAXLIST);
		}

		GetFiles(&names, LIFELIB, cp, strlen(LIFELIB) + 1, MAXLIST);
	}

	if (names.argc == 0)
		Error("No files found");

	ListFiles(&names);

	free(names.buf);
	free((char *) names.argv);
}


/*
 * Insert another object into this one.
 */
static void
CmdInsert(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	OBJECT *obj;

	obj = FindObject(cp);

	if (obj == NULL)
		Error("No such object");

	Backup();

	curobj->mark = MARK_USR;
	AddObject(obj, curobj, RELATIVE, FALSE);
	curobj->mark = MARK_ANY;
}


/*
 * Show list of objects.
 */
static void
CmdObjects(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	BOOL	all;

	all = ((*cp == '-') || (*cp == 'a'));

	ListObjects(all);
}


/*
 * Rename the current object to something else.
 */
static void
CmdRename(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if (curobj->reserved)
		Error("Cannot rename reserved object");

	if (strlen(cp) > MAXNAME)
		Error("Name too long");

	if (FindObject(cp))
		Error("Name already exists");

	if (BADNAME(cp))
		Error("Cannot create reserved name");

	strcpy(curobj->name, cp);
}


/*
 * Move current object into another object.
 */
static void
CmdMove(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	MoveObject(curobj, GetObject(cp));
}


/*
 * Move currently markeded cells into another object.
 */
static void
CmdMoveMarked(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	MoveMarkedObject(curobj, GetObject(cp), MARK_USR);
}


/*
 * Set the value of a variable.
 */
static void
CmdSet(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	char *	exp;

	for (exp = cp; *exp && !isblank(*exp) && (*exp != '='); exp++)
		;

	if (isblank(*exp)) {		/* skip spaces */
		*exp++ = '\0';

		while (isblank(*exp))
			exp++;
	}

	if (*exp == '\0') {		/* no expression, set to zero */
		SetVariable(cp, 0);
		return;
	}

	if (*exp == '=')
		*exp++ = '\0';

	SetVariable(cp, GetExpression(exp));
}


/*
 * Type the value of an expression.
 */
static void
CmdType(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	static	char	buf[30];

	sprintf(buf, "Value = %ld", GetExpression(cp));
	errorstring = buf;
}


/*
 * Display the values of all the variables.
 */
static void
CmdVariables(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	ListVariables();
}


/*
 * Quit program
 */
static void
CmdQuit(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	(*dev->close)(dev);

	exit(0);
}


/*
 * Read commands or object from file, defaulting extension if needed.
 */
static void
CmdRead(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	Backup();

	if (!SetFile(cp))
		Error("Cannot open input file");
}


/*
 * Set new life rule.
 */
static void
CmdRule(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if (*cp == '\0')
		cp = DEFAULTRULE;

	SetRule(cp);
}


/*
 * Read commands from the terminal.  Useful in loops or command files.
 * If the -c argument is given, we don't do it if no input is ready.
 */
static void
CmdTtyInput(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	if ((*cp == '-') || (*cp == 'c')) {	/* check for input */
		if (!(*dev->inputready)(dev))
			return;
	}

	if (!SetTty())
		Error("Nesting too deep");
}


/*
 * Write object to file.
 */
static void
CmdWrite(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	char *	method;

	if (!got1)
		arg1 = WRITECOLS;

	method = "rle";

	if (*cp == '-') {
		method = cp + 1;

		while (*cp && !isblank(*cp))
			cp++;

		while (isblank(*cp))
			*cp++ = '\0';
	}

	WriteObject(curobj, method, cp, arg1 ? WRITEROWS : 0, arg1);
}


/*
 * Zero out current object.
 */
static void
CmdZero(cp, arg1, arg2, got1, got2)
	char *	cp;
	VALUE	arg1;
	VALUE	arg2;
	BOOL	got1;
	BOOL	got2;
{
	OBJECT *obj;

	obj = curobj;

	Backup();

	ZeroObject(obj);

	obj->gen = 0;
	obj->born = 0;
	obj->died = 0;
	obj->currow = 0;
	obj->curcol = 0;
	obj->origrow = 0;
	obj->origcol = 0;
	obj->autoscale = FALSE;

	SetScale(obj, defaultscale);

	mode = M_MOVE;
	obj->frequency = defaultfrequency;
	freqcount = defaultfrequency;
}


/*
 * See if one string is an abbreviation of another.
 * Returns TRUE if first string is an abbreviation.
 */
BOOL
IsAbbreviation(str1, str2)
	char *	str1;
	char *	str2;
{
	if ((str1 == NULL) || (str2 == NULL))
		return FALSE;

	while (*str1) {
		if (*str1++ != *str2++)
			return FALSE;
	}

	return TRUE;
}

/* END CODE */
