/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Program to play Conway's Game of LIFE on an infinite board.
 */

#define	VERSION	"4.6"


#define	DEFINE_GLOBALS

#include <unistd.h>
#include "life.h"

#define	MAXINITS	10	/* number of init files */


static	BOOL	scaleset;
static	BOOL	inited;		/* initialization is complete */
static	BOOL	needscleanup;	/* need to cleanup after init file read */
static	char *	initfiles[MAXINITS];	/* user's initialization files */
static	int	initfilecount;		/* number of init files to do */
static	int	initfiledone;		/* number of init files done */
static	char *	loadfile;	/* file to be loaded on startup */


static	void	CheckLifeOpts PROTO((void));
static	void	ParseOptions PROTO((int, char **));
static	void	InterruptHandler PROTO((void));


int
main(argc, argv)
	int	argc;
	char **	argv;
{
	termcell = AllocateCell();
	termcell->next = termcell;
	termcell->col = INFINITY;

	termrow = AllocateRow();
	termrow->next = termrow;
	termrow->firstcell = termcell;
	termrow->row = INFINITY;

	mode = M_MOVE;
	gridchar = ' ';
	defaultscale = 1;
	defaultfrequency = 1;
	freqcount = 1;

	SetRule(DEFAULTRULE);

	reserve = TRUE;			/* creating special objects */

	deleteobject = GetObject("..delete");
	tempobject = GetObject("..temp");
	backupobject = GetObject("..backup");
	mainobject = GetObject("main");
	curobj = mainobject;
	prevobj = mainobject;

	reserve = FALSE;		/* no more special objects */

	curinput = &inputs[-1];		/* initialize for tty input */
	SetTty();

#ifdef	X11
	dev = &x11dev;
#else
	dev = &ttydev;
#endif

	CheckLifeOpts();
	ParseOptions(argc - 1, argv + 1);

	if (! (*dev->open)(dev))
		exit(1);

	if (!scaleset)
		defaultscale = dev->defaultscale;

	SetScale(deleteobject, defaultscale);	/* fix scale factors now */
	SetScale(tempobject, defaultscale);
	SetScale(backupobject, defaultscale);
	SetScale(mainobject, defaultscale);

	signal(SIGINT, InterruptHandler);
	ScanInit(ReadChar, ttyjmp);

	inited = TRUE;

	while (TRUE) {
		DoCommand();
		DoGeneration(curobj);
		UpdateView();
	}
}


/*
 * Parse the optional LIFEOPTS environment variable to define defaults.
 */
static void
CheckLifeOpts()
{
	char *	env;
	char *	cp;
	int	argc;
	char **	argv;
	static	char *	argtable[MAXARGS];

	env = getenv(LIFEOPTS);

	if (env == NULL)
		return;

	cp = malloc(strlen(env) + 1);

	if (cp == NULL) {
		fprintf(stderr, "No memory\n");
		exit(1);
	}

	strcpy(cp, env);

	argc = 0;
	argv = argtable;

	while (*cp) {
		while (isblank(*cp))
			cp++;

		if (*cp == '\0')
			break;

		if (++argc > MAXARGS) { 
			fprintf(stderr, "Too many arguments in LIFEOPTS\n");
			exit(1);
		}

		*argv++ = cp;

		while (*cp && !isblank(*cp))
			cp++;

		if (*cp)
			*cp++ = '\0';
	}

	ParseOptions(argc, argtable);
}


/*
 * Handle command line options.  Note: The first argument argv[0] is NOT
 * the usual program name, so that must have been removed if necessary.
 * We assume that the argument strings passed are permanent.
 */
static void
ParseOptions(argc, argv)
	int	argc;
	char **	argv;
{
	char *	str;
	char *	cp;
	VALUE	arg;
	BOOL	neg;

	while (argc-- > 0) {
		str = *argv++;

		/*
		 * If no option letter is used, then this means
		 * read from the specified file.
		 */
		if (*str != '-') {
			if (loadfile) {
				fprintf(stderr,
					"More than one initial file not allowed\n");
				exit(1);
			}

			loadfile = str;

			continue;
		}

		/*
		 * This is an option argument.  Parse each character of
		 * it, so that multiple options can be given at once.
		 */
		str++;
		while (*str) switch (*str++) {
			case 't':			/* use terminal */
				dev = &ttydev;
				break;

			case 'g':			/* use graphics */
#if defined(X11)
				dev = &x11dev;
#else
				fprintf(stderr, "Graphics is not available\n");
				exit(1);
#endif
				break;

			case 'l':			/* set library path */
				if ((argc <= 0) || (**argv == '-')) {
					fprintf(stderr, "Missing library path\n");
					exit(1);
				}

				if (userlibcount >= MAXLIBS) {
					fprintf(stderr, "Too many library directories\n");
					exit(1);
				}
	
				userlibs[userlibcount++] = *argv++;
				argc--;
				break;

			case 's':			/* set default scale */
				if (argc <= 0) {
					fprintf(stderr, "Missing scale\n");
					exit(1);
				}

				cp = *argv++;
				argc--;
				neg = FALSE;

				if (*cp == '-') {
					neg = TRUE;
					cp++;
				}

				arg = 0;

				while (isdigit(*cp))
					arg = arg * 10 + *cp++ - '0';

				if (*cp || (arg == 0)) {
					fprintf(stderr, "Bad scale factor\n");
					exit(1);
				}

				if (neg)
					arg = -arg;

				defaultscale = arg;
				scaleset = TRUE;
				break;

			case 'v':
				printf("Life version %s\n", VERSION);
				exit(0);

			case 'i':
				if (argc-- <= 0) {
					fprintf(stderr, "Missing file name\n");
					exit(1);
				}

				if (initfilecount >= MAXINITS) {
					fprintf(stderr, "Too many init files\n");
					exit(1);
				}

				initfiles[initfilecount++] = *argv++;
				break;

			default:
				fprintf(stderr, "Unknown option -%c\n", str[-1]);
				exit(1);
		}
	}
}


/*
 * Here to check on the startup files.
 * If the top level input level is reached, and there is another startup
 * file to process, then we set up to take input from it.
 */
void
CheckStartupFiles()
{
	char *	str;

	/*
	 * If not at the top level input level, then don't start
	 * reading another file.
	 */
	if (curinput != inputs)
		return;

	/*
	 * See if we have to clean up after an init file is finished.
	 * This just makes sure that we are on the main object and that
	 * it is empty.
	 */
	if (needscleanup) {
		needscleanup = FALSE;

		curobj = mainobject;
		prevobj = mainobject;

		ZeroObject(curobj);
		ZeroObject(deleteobject);

		curobj->gen = 0;
		curobj->born = 0;
		curobj->died = 0;
		curobj->currow = 0;
		curobj->curcol = 0;
	}

	/*
	 * Then look for another initialization file that hasn't been done.
	 */
	if (initfiledone < initfilecount) {
		if (!SetFile(initfiles[initfiledone++]))
			Error("Cannot read initialization file");

		needscleanup = TRUE;

		return;
	}

	/*
	 * Finally look for a file to be loaded on startup.
	 */
	str = loadfile;

	if (loadfile) {
		loadfile = NULL;

		if (!SetFile(str))
			Error("Cannot read initial file");
	}
}


/*
 * Here on an interrupt character.  Remember to stop what we are doing soon.
 * We cannot usually just longjmp away since things may be in an inconsistent
 * state.  The longjmp is only allowed if we are sitting in a terminal read.
 */
static void
InterruptHandler()
{
	signal(SIGINT, InterruptHandler);

	genleft = 0;
	stop = TRUE;
	curobj->update |= U_ALL;

	if (intjmpok) {			/* do longjmp only if set up */
		intjmpok = FALSE;
		longjmp(intjmp, 1);
	}
}


/*
 * Here on an error.  Close all but the top input level, cancel the
 * current command, beep, and set up to display the indicated message.
 * The message will remain until the next command is typed by the user.
 */
void
Error(str)
	char *	str;		/* message to type */
{
	if (!inited) {
		fprintf(stderr, "%s\n", str);
		exit(1);
	}

	while (curinput > inputs)
		(*curinput->term)(curinput);

	errorstring = str;
	curobj->update |= U_ALL;
	stop = FALSE;
	dowait = FALSE;
	Beep();

	ScanAbort();
}

/* END CODE */
