/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


/*
 * Find the given named object.  Returns NULL if nonexistant.  The special
 * name of "." means the current object.  The special name of ".." means
 * the previous object.
 */
OBJECT *
FindObject(str)
	char *	str;
{
	OBJECT *obj;

	if (str[0] == '.') {		/* check for "." or ".." */
		if (str[1] == '\0')
			return curobj;

		if ((str[1] == '.') && (str[2] == '\0'))
			return prevobj;
	}

	for (obj = objects; obj; obj = obj->next) {
		if (strcmp(obj->name, str) == 0)
			return obj;
	}

	return NULL;
}


/*
 * Create the given named object, or return it if it already exists.
 */
OBJECT *
GetObject(str)
	char *	str;
{
	OBJECT *obj;

	for (obj = objects; obj; obj = obj->next) {
		if (strcmp(obj->name, str) == 0)
			return obj;
	}

	if (strlen(str) > MAXNAME)
		Error("Object name too long");

	if (!reserve && BADNAME(str))
		Error("Cannot create reserved name");

	obj = AllocateObject();
	obj->next = objects;
	objects = obj;
	strcpy(obj->name, str);

	return obj;
}


/*
 * Set an object as the current one.  The old current object is remembered
 * so that it can be referenced using "..".  This cancels any insert mode.
 */
void
SetObject(obj)
	OBJECT *obj;
{
	mode = M_MOVE;

	if (obj == curobj)
		return;

	prevobj = curobj;
	curobj = obj;
	obj->update |= U_ALL;

	if (prevobj->path.count > 0)
		obj->update |= U_REDRAW;
}


/*
 * Delete all cells of an object.
 */
void
ZeroObject(obj)
	OBJECT *obj;
{
	ROW *	rp;

	ClearPath(obj);

	rp = obj->firstrow;

	if (rp == termrow)
		return;

	for (; rp != termrow; rp = rp->next) {
		if (rp->firstcell == termcell)
			continue;

		rp->lastcell->next = freecells;
		freecells = rp->firstcell;
		obj->count -= rp->count;
	}

	obj->lastrow->next = freerows;
	freerows = obj->firstrow;
	obj->firstrow = termrow;
	obj->lastrow = NULL;
}


/*
 * Destroy the existence of an object.  If it is the current object,
 * switch the current object back to the previous object.
 */
void
DestroyObject(obj)
	OBJECT *obj;
{
	OBJECT *pobj;

	if (obj == NULL)
		return;

	if (obj->reserved)
		Error("Cannot destroy reserved object");

	if (obj == prevobj)
		prevobj = mainobject;

	if (obj == curobj) {
		curobj = prevobj;
		prevobj = mainobject;
		obj->update |= U_ALL;
	}

	ZeroObject(obj);

	FreePath(obj);

	if (objects == obj) {		/* first object in list */
		objects = obj->next;
		obj->next = freeobjects;
		freeobjects = obj;

		return;
	}

	for (pobj = objects; pobj->next != obj; pobj = pobj->next)
		;

	pobj->next = obj->next;
	obj->next = freeobjects;
	freeobjects = obj;
}


/*
 * Move one object to another.  The source object is zeroed, and the
 * previous contents of the destination object are lost.
 */
void
MoveObject(sobj, dobj)
	OBJECT *sobj;		/* source object */
	OBJECT *dobj;		/* destination object */
{
	if (sobj == dobj)
		Error("Moving object to itself");

	ZeroObject(dobj);

	dobj->currow = sobj->currow;
	dobj->curcol = sobj->curcol;
	dobj->minrow = sobj->minrow;
	dobj->maxrow = sobj->maxrow;
	dobj->mincol = sobj->mincol;
	dobj->maxcol = sobj->maxcol;
	dobj->scale = sobj->scale;
	dobj->autoscale = sobj->autoscale;
	dobj->frequency = sobj->frequency;
	dobj->origrow = sobj->origrow;
	dobj->origcol = sobj->origcol;
	dobj->firstrow = sobj->firstrow;
	dobj->lastrow = sobj->lastrow;
	dobj->count = sobj->count;
	sobj->firstrow = termrow;
	sobj->lastrow = NULL;
	sobj->count = 0;

	CopyPath(sobj, dobj);
	ClearPath(sobj);
}


/*
 * Add one object to another.  The source object is unchanged.  The
 * destination object will get all cells from both objects.  If disp is
 * RELATIVE, the object is displaced as specified by the two object's cursor
 * positions. Otherwise, the addition is performed with absolute coordinates.
 * Domarks is TRUE if marks are to be copied for new cells.
 */
void
AddObject(sobj, dobj, disp, domarks)
	OBJECT *sobj;		/* source object */
	OBJECT *dobj;		/* destination object */
	COUNT	disp;
	BOOL	domarks;
{
	ROW *	rp;		/* current row */
	CELL *	cp;		/* current cell */
	COORD	newrow;		/* new row number */
	COUNT	rowdisp;	/* displacements */
	COUNT	coldisp;
	MARK	oldmark;

	if (sobj == dobj)
		Error("Adding object to itself");

	rowdisp = 0;
	coldisp = 0;
	oldmark = dobj->mark;

	if (disp == RELATIVE) {
		rowdisp = dobj->currow - sobj->currow;
		coldisp = dobj->curcol - sobj->curcol;
	}

	for (rp = sobj->firstrow; rp != termrow; rp = rp->next) {
		newrow = rp->row + rowdisp;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (domarks)
				dobj->mark = cp->marks;

			AddCell(dobj, newrow, cp->col + coldisp);
		}
	}

	dobj->mark = oldmark;
}


/*
 * Copy one object to another.  The source object is unchanged.
 * The current contents of the destination object are lost.
 */
void
CopyObject(sobj, dobj)
	OBJECT *sobj;		/* source object */
	OBJECT *dobj;		/* destination object */
{
	if (sobj == dobj)
		Error("Copying object to itself");

	ZeroObject(dobj);
	AddObject(sobj, dobj, ABSOLUTE, TRUE);

	dobj->currow = sobj->currow;
	dobj->curcol = sobj->curcol;
	dobj->minrow = sobj->minrow;
	dobj->maxrow = sobj->maxrow;
	dobj->mincol = sobj->mincol;
	dobj->maxcol = sobj->maxcol;
	dobj->viewrows = sobj->viewrows;
	dobj->viewcols = sobj->viewcols;
	dobj->scale = sobj->scale;
	dobj->autoscale = sobj->autoscale;
	dobj->frequency = sobj->frequency;
	dobj->origrow = sobj->origrow;
	dobj->origcol = sobj->origcol;

	CopyPath(sobj, dobj);

	memcpy(dobj->wherelocs, sobj->wherelocs, sizeof(dobj->wherelocs));
}


/*
 * Show the list of objects.  If all is nonzero, all objects will be
 * shown.  Otherwise, only objects not starting with a period are shown.
 */
void
ListObjects(all)
	BOOL	all;
{
	OBJECT *obj;
	int	ch;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
	char	buf[80];

	ShowHelp("cells	height	width	gen	scale	object\n");
	ShowHelp("-----	------	-----	---	-----	------\n");

	for (obj = objects; obj; obj = obj->next) {
		if (!all && (obj->name[0] == '.'))
			continue;

		ch = ' ';

		if (obj == prevobj)
			ch = '+';

		if (obj == curobj)
			ch = '*';

		MinMax(obj, &minrow, &maxrow, &mincol, &maxcol);

		sprintf(buf, "%ld\t%ld\t%ld\t%ld\t%d\t%c %s\n",
			obj->count, (maxrow - minrow + 1),
			(maxcol - mincol + 1), obj->gen,
			obj->scale, ch, obj->name);

		ShowHelp(buf);
	}

	ShowHelp("\n\nIn object column, '*' = current object, '+' = previous object\n");

	if (!all)
		ShowHelp("Use -a to show objects beginning with '.'\n");

	EndHelp();
}


/*
 * Find the minimum and maximum row and column numbers for an object.
 * If there are no cells in the object, the mins will be one more than
 * the maxes.  Returns FALSE if the object has no cells.
 */
BOOL
MinMax(obj, minrow, maxrow, mincol, maxcol)
	OBJECT *obj;
	COORD *	minrow;
	COORD *	maxrow;
	COORD *	mincol;
	COORD *	maxcol;
{
	ROW *	rp;
	COORD	maxr;
	COORD	minr;
	COORD	maxc;
	COORD	minc;
	BOOL	status;

	minr = INFINITY;
	maxr = -INFINITY;
	minc = INFINITY;
	maxc = -INFINITY;
	status = FALSE;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		if (rp->firstcell == termcell)
			continue;

		if (rp->row < minr)
			minr = rp->row;

		maxr = rp->row;

		if (rp->firstcell->col<minc)
			minc = rp->firstcell->col;

		if (rp->lastcell->col>maxc)
			maxc = rp->lastcell->col;

		status = TRUE;
	}

	if (!status) {			/* no cells in object */
		minr = 1;
		maxr = 0;
		minc = 1;
		maxc = 0;
	}

	*minrow = minr;
	*maxrow = maxr;
	*mincol = minc;
	*maxcol = maxc;

	return status;
}


/*
 * Search forwards for the nth next object, restarting at the top if necessary.
 * If all is nonzero, or if wrap around occurs, the search will be over all
 * objects.  Otherwise, objects found in previous searches will be skipped.
 * Returns FALSE if nothing was found.
 */
BOOL
SearchObject(obj, count, all)
	OBJECT *obj;
	COUNT	count;
	BOOL	all;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;
	COORD	col;

	if (obj->count <= 0)
		return FALSE;

	if (all)
		ClearMarks(obj, MARK_SRC);

	row = obj->currow;
	col = obj->curcol;

	for (rp = obj->firstrow; row > rp->row; rp = rp->next)
		;

	for (cp = rp->firstcell; col > cp->col; cp = cp->next)
		;

	if ((row == rp->row) && (col == cp->col) &&
		((cp->marks & MARK_SRC) == 0))
	    		count++;

	while (TRUE) {
		if (stop)
			return FALSE;

		if (cp == termcell) {
			rp = rp->next;

			if (rp == termrow) {
				ClearMarks(obj, MARK_SRC);
				rp = obj->firstrow;
			}

			cp = rp->firstcell;
			continue;
		}

		if ((cp->marks & MARK_SRC) == 0) {
			MarkObject(obj, rp->row, cp->col, 1, MARK_SRC);

			if (--count <= 0)
				break;
		}

		cp = cp->next;
	}

	obj->currow = rp->row;
	obj->curcol = cp->col;

	return TRUE;
}

/* END CODE */
