/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <fcntl.h>
#include <curses.h>
#include <sys/ioctl.h>
#include "life.h"


static	BOOL	TtyOpen PROTO((DEV *));
static	void	TtyClose PROTO((DEV *));
static	void	TtyUpdate PROTO((DEV *, BOOL));
static	void	TtyRefresh PROTO((DEV *));
static	BOOL	TtyInputReady PROTO((DEV *));
static	int	TtyReadChar PROTO((DEV *, BOOL));
static	void	TtyMoveCursor PROTO((DEV *, SCALE, COORD, COORD));
static	void	TtyShowView PROTO((DEV *, OBJECT *));
static	void	TtyShowStatus PROTO((DEV *, char *));
static	void	TtyAddStatus PROTO((DEV *, char *));
static	void	TtyShowHelp PROTO((DEV *));
static	void	TtyAddHelp PROTO((DEV *, char *));
static	BOOL	TtyAssign PROTO((DEV *, VALUE, UCHAR));
static	void	TtyWriteAssign PROTO((DEV *, FILE *));

static	void	ViewNormal PROTO((OBJECT *));
static	void	ViewScale PROTO((OBJECT *, SCALE));


DEV	ttydev = {
	TtyOpen, TtyClose, TtyUpdate, TtyRefresh, TtyInputReady, TtyReadChar,
	TtyMoveCursor, TtyShowView, TtyShowStatus, TtyAddStatus,
	TtyShowHelp, TtyAddHelp, TtyAssign, TtyWriteAssign,
	0, 0, 0, 0, 1, MAXSCALE, 1
};


static	WINDOW *statwin;
static	WINDOW *viewwin;

static	BOOL	havesavedchar;
static	int	savedchar;
static	int	ttyfd;


static BOOL
TtyOpen(dev)
	DEV *	dev;
{
#ifdef	O_NONBLOCK
	ttyfd = open("/dev/tty", O_RDONLY | O_NONBLOCK);

	if (ttyfd < 0) {
		fprintf(stderr, "Cannot set tty mode\n");

		return FALSE;
	}
#endif
	initscr();
	cbreak();
	noecho();

	statwin = newwin(1, COLS, 0, 0);
	viewwin = newwin(LINES - 1, COLS, 1, 0);

	dev->rows = LINES - 1;
	dev->cols = COLS - 1;
	dev->textrows = LINES - 1;
	dev->textcols = COLS - 1;

	return TRUE;
}


static void
TtyClose(dev)
	DEV *	dev;
{
	refresh();
	endwin();
}


static void
TtyUpdate(dev, inputflag)
	DEV *	dev;
	BOOL	inputflag;
{
	if (inputflag) {
		wrefresh(viewwin);
		wrefresh(statwin);
	} else {
		wrefresh(statwin);
		wrefresh(viewwin);
	}
}


static void
TtyRefresh(dev)
	DEV *	dev;
{
	wrefresh(curscr);
}


/*
 * Read the next character from the terminal, waiting if requested.
 * If not waiting and no character is ready, then EOF is returned.
 * When waiting and the read is interrupted, then also returns EOF.
 */
static int
TtyReadChar(dev, wait)
	DEV *	dev;
	BOOL	wait;
{
	UCHAR	ch;
	int	n;

	if (!wait && !TtyInputReady(dev))
		return EOF;

	if (havesavedchar) {
		havesavedchar = FALSE;

		return savedchar;
	}

	n = read(STDIN, &ch, 1);

	if (n <= 0)
		return EOF;

	if (ch == '\r')
		ch = '\n';

	return ch;
}


/*
 * See if input is ready from the terminal.
 */
static BOOL
TtyInputReady(dev)
	DEV *	dev;
{
	int	n;
	UCHAR	ch;

#ifdef O_NONBLOCK
	n = read(ttyfd, &ch, 1);

	if (n <= 0)
		return FALSE;

	if (ch == '\r')
		ch = '\n';

	savedchar = ch;
	havesavedchar = TRUE;

	return TRUE;
#endif

#ifdef FIONREAD
	return ((ioctl(STDIN, FIONREAD, &n) == 0) && (n > 0));
#endif

#ifdef FIORDCHK
	return (ioctl(STDIN, FIORDCHK, &n) > 0);
#endif

	Error("Cannot do non-blocking reads!!!!");
}


static BOOL
TtyAssign(dev, button, macro)
	DEV *	dev;
	VALUE	button;
	UCHAR	macro;
{
	return FALSE;
}


static void
TtyWriteAssign(dev, fp)
	DEV *	dev;
	FILE *	fp;
{
	fputs("! No button assignments for terminals\n", fp);
}


static void
TtyMoveCursor(dev, scale, row, col)
	DEV *	dev;
	SCALE	scale;
	COORD	row;
	COORD	col;
{
	if (scale < 1)
		scale = 1;

	wmove(viewwin, row / scale, col / scale);
}


/*
 * Show the view around the current window location.
 */
static void
TtyShowView(dev, obj)
	DEV *	dev;
	OBJECT *obj;
{
	wmove(viewwin, 0, 0);

	if (obj->scale <= 1)
		ViewNormal(obj);
	else
		ViewScale(obj, obj->scale);

	wclrtobot(viewwin);
}


/*
 * Show the specified status line.
 */
static void
TtyShowStatus(dev, str)
	DEV *	dev;
	char *	str;
{
	wmove(statwin, 0, 0);
	wclrtobot(statwin);
	wmove(statwin, 0, 0);
	waddstr(statwin, str);
}


/*
 * Add the specified status to the status line.
 */
static void
TtyAddStatus(dev, str)
	DEV *	dev;
	char *	str;
{
	waddstr(statwin, str);
}


/*
 * Setup to show help.
 */
static void
TtyShowHelp(dev)
	DEV *	dev;
{
	wmove(viewwin, 0, 0);
	wclrtobot(viewwin);
	wmove(viewwin, 0, 0);
}


/*
 * Add the specified information to the help display.
 */
static void
TtyAddHelp(dev, str)
	DEV *	dev;
	char *	str;
{
	waddstr(viewwin, str);
}


/*
 * Show the cells around the cursor normally (scale factor of 1).
 */
static void
ViewNormal(obj)
	OBJECT *obj;			/* current object */
{
	ROW *	rp;			/* current row */
	CELL *	cp;			/* current cell */
	COORD	row;			/* current row number */
	COORD	col;			/* current column number */
	char *	str;			/* characters for line */
	char *	endstr;			/* end of characters for line */

	rp = obj->firstrow;
	row = obj->minrow;
	seecount = 0;

	while (row > rp->row)
		rp = rp->next;

	for (; row <= obj->maxrow; row++) {
		if (row != rp->row) {			/* blank row */
			if (gridchar == ' ') {
				waddch(viewwin, '\n');

				continue;
			}

			str = stringbuf;

			for (col = obj->mincol; col <= obj->maxcol; col++) {
				*str++ = gridchar;
			}

			*str++ = '\n';
			*str = '\0';
			waddstr(viewwin, stringbuf);

			continue;
		}

		str = stringbuf;
		endstr = str;
		cp = rp->firstcell;
		col = obj->mincol;

		while (col > cp->col)
			cp = cp->next;

		for (; col <= obj->maxcol; col++) {
			if (col != cp->col) {		/* blank cell */
				*str++ = gridchar;

				if (gridchar != ' ')
					endstr = str;

				continue;
			}

			*str = 'o';

			if ((cp->marks & MARK_SEE) == 0)
				*str = 'O';

			endstr = ++str;
			seecount++;
			cp = cp->next;
		}

		*endstr++ = '\n';
		*endstr = '\0';
		waddstr(viewwin, stringbuf);
		rp = rp->next;
	}
}


/*
 * Show the view around the cursor with an arbitrary scale factor.
 * When in this mode, characters from 1 to 9 (or * if 10 or more)
 * are used to indicate how many cells are in each n by n square.
 */
static void
ViewScale(obj, scale)
	OBJECT *obj;
	SCALE	scale;
{
	ROW *	rp;			/* row pointer */
	CELL *	cp;			/* current cell structure */
	COORD	row;			/* current row number */
	COORD	col;			/* current column number */
	COUNT	sum;			/* number of cells in square */
	CELL **	cpp;			/* pointer into cell table */
	CELL **	endcpp;			/* end of cell table */
	char *	str;			/* buffer pointer */
	char *	endstr;			/* end of buffer */
	CELL *	cptab[MAXSCALE];	/* table of rows */

	row = obj->minrow;
	col = obj->mincol;
	endcpp = &cptab[scale];
	seecount = 0;

	for (rp = obj->firstrow; (rp->row < row); rp = rp->next)
		;

	while (row <= obj->maxrow) {
		/*
		 * If there is a large gap to the next row number then
		 * the terminal line is empty.
		 */
		if (rp->row >= (row + scale)) {	/* no rows here */
			if (gridchar == ' ') {
				waddch(viewwin, '\n');
				row += scale;

				continue;
			}

			str = stringbuf;

			for (col=obj->mincol; col<=obj->maxcol; col+=scale) {
				*str++ = gridchar;
			}

			*str++ = '\n';
			*str = '\0';
			waddstr(viewwin, stringbuf);
			row += scale;

			continue;
		}

		/*
		 * Collect the rows to be searched for one terminal line.
		 * Dummy up empty rows if necessary.
		 */
		for (cpp = cptab; cpp < endcpp; cpp++) {
			*cpp = termcell;

			if (rp->row > row++)
				continue;

			*cpp = rp->firstcell;
			rp = rp->next;
		}

		str = stringbuf;
		endstr = str;

		/*
		 * Advance along each row to the next range of columns,
		 * adding cells found to get the result for each square.
		 */
		for (col = obj->mincol; col <= obj->maxcol; col += scale) {
			sum = 0;

			for (cpp = cptab; cpp < endcpp; cpp++) {
				cp = *cpp;

				while (col > cp->col)
					cp = cp->next;

				while ((col + scale) >= cp->col) {
					sum++;
					cp = cp->next;
				}

				*cpp = cp;
			}

			if (sum == 0) {		/* no cells in square */
				*str++ = gridchar;

				if (gridchar != ' ')
					endstr = str;

				continue;
			}

			*str = '*';		/* show number of cells */

			if (sum <= 9)
				*str = '0' + sum;

			endstr = ++str;
			seecount += sum;
		}

		*endstr++ = '\n';
		*endstr = '\0';
		waddstr(viewwin, stringbuf);
	}
}

/* END CODE */
