/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


/*
 * Variable enum.
 */
typedef	enum	{
	VarMinx, VarMiny, VarMaxx, VarMaxy, VarCells, VarCx, VarCy, VarOx,
	VarOy, VarVminx, VarVmaxx, VarVminy, VarVmaxy, VarVcells, VarVx,
	VarVy, VarSminx, VarSmaxx, VarSminy, VarSmaxy, VarScells, VarGen,
	VarScale, VarFreq, VarBorn, VarDied, VarDefscale, VarDeffreq,
	VarGenleft, VarEndlist
} VARS;


/*
 * Table of multi-character variables.
 * This is arranged in listing order (three items per line).
 */
typedef	struct	{
	char *	name;			/* name of variable */
	VARS	type;			/* variable id */
} VARTAB;


static	VARTAB	vartab[] = {
	{"cx",		VarCx},
	{"vx",		VarVx},
	{"ox",		VarOx},

	{"cy",		VarCy},
	{"vy",		VarVy},
	{"oy",		VarOy},

	{"minx",	VarMinx},
	{"vminx",	VarVminx},
	{"sminx",	VarSminx},

	{"maxx",	VarMaxx},
	{"vmaxx",	VarVmaxx},
	{"smaxx",	VarSmaxx},

	{"maxy",	VarMaxy},
	{"vmaxy",	VarVmaxy},
	{"smaxy",	VarSmaxy},

	{"miny",	VarMiny},
	{"vminy",	VarVminy},
	{"sminy",	VarSminy},

	{"cells",	VarCells},
	{"vcells",	VarVcells},
	{"scells",	VarScells},

	{"gen",		VarGen},
	{"born",	VarBorn},
	{"died",	VarDied},

	{"scale",	VarScale},
	{"freq",	VarFreq},
	{"dscale",	VarDefscale},

	{"dfreq",	VarDeffreq},
	{"genleft",	VarGenleft},
	{NULL,		VarEndlist}
};


static	char *	curcp;		/* current character to parse */
static	VALUE	lowervars[26];	/* lower case single-char variable values */
static	VALUE	uppervars[26];	/* upper case single-char variable values */


static	VALUE	GetVariableByType PROTO((VARS));
static	VALUE	GetWhereVariable PROTO((char *));
static	VALUE	ParseSum PROTO((void));
static	VALUE	ParseProduct PROTO((void));
static	VALUE	ParseTerm PROTO((void));
static	VALUE	ParseName PROTO((void));


/*
 * Return the value of a multiple character variable name.  This can be
 * either a single character name, or else one of a fixed set of multi-
 * character names.  All variable names must start with either a letter
 * or a dollar sign followed by a letter.
 */
VALUE
GetVariable(cp)
	char *	cp;
{
	VARTAB *vp;

	if (*cp == '$')
		cp++;

	if (cp[1] == '\0')
		return GetVariable1(*cp);

	if (!islower(*cp) && !isupper(*cp))
		Error("Bad variable name");

	for (vp = vartab; ; vp++) {
		if (vp->name == NULL) {
			if (*cp == 'w')
				return GetWhereVariable(cp + 1);

			Error("Unknown variable");
		}

		if (strcmp(vp->name, cp) == 0)
			break;
	}

	return GetVariableByType(vp->type);
}


/*
 * Return the value of a variable given its enum value.
 */
static VALUE
GetVariableByType(type)
	VARS	type;
{
	OBJECT *obj;
	VALUE	value;
	COORD *	ptr;
	COORD *	sptr;
	int	sign;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;

	obj = curobj;
	value = 0;
	sign = 1;
	ptr = NULL;
	sptr = NULL;

	switch (type) {
		case VarCx:
			value = obj->curcol;
			break;

		case VarCy:
			value = -obj->currow;
			break;

		case VarOx:
			value = obj->origcol;
			break;

		case VarOy:
			value = -obj->origrow;
			break;

		case VarVx:
			value = (obj->mincol + obj->maxcol) / 2;
			break;

		case VarVy:
			value = -(obj->minrow + obj->maxrow) / 2;
			break;

		case VarVminx:
			value = obj->mincol;
			break;

		case VarVmaxx:
			value = obj->maxcol;
			break;

		case VarVminy:
			value = -obj->maxrow;
			break;

		case VarVmaxy:
			value = -obj->minrow;
			break;

		case VarVcells:
			value = MarkRegion(obj, MARK_ANY, obj->minrow,
				obj->maxrow, obj->mincol, obj->maxcol);

			break;

		case VarCells:
			value = obj->count;
			break;

		case VarGen:
			value = obj->gen;
			break;

		case VarGenleft:
			value = genleft;
			break;

		case VarBorn:
			value = obj->born;
			break;

		case VarDied:
			value = obj->died;
			break;

		case VarFreq:
			value = obj->frequency;
			break;

		case VarScale:
			value = obj->scale;
			break;

		case VarDeffreq:
			value = defaultfrequency;
			break;

		case VarDefscale:
			value = defaultscale;
			break;

		case VarMinx:
			ptr = &mincol;
			break;

		case VarMaxx:
			ptr = &maxcol;
			break;

		case VarMiny:
			ptr = &maxrow;
			sign = -1;
			break;

		case VarMaxy:
			ptr = &minrow;
			sign = -1;
			break;

		case VarScells:
			value = CountMarks(obj, MARK_SEE);
			break;

		case VarSminx:
			sptr = &mincol;
			break;

		case VarSmaxx:
			sptr = &maxcol;
			break;

		case VarSminy:
			sptr = &maxrow;
			sign = -1;
			break;

		case VarSmaxy:
			sptr = &minrow;
			sign = -1;
			break;

		default:
			return 0;
	}

	/*
	 * Call proper minmax routines if we need to
	 */
	if (ptr && MinMax(obj, &minrow, &maxrow, &mincol, &maxcol))
		value = *ptr;

	if (sptr && MarkMinMax(obj,MARK_SEE,&minrow,&maxrow,&mincol,&maxcol))
		value = *sptr;

	return (sign * value);
}


/*
 * Get the value of a 'where' variable.
 * The value is either the x or the y coordinate, depending on the argument.
 * The incoming string is the where variable letter followed by x or y.
 */
static VALUE
GetWhereVariable(cp)
	char *	cp;
{
	int	ch;

	ch = *cp++;

	if ((ch < 'a') || (ch >= 'z'))
		Error("Bad where variable name");

	if (((*cp != 'x') && (*cp != 'y')) || (cp[1] != '\0'))
		Error("Bad where coordinate selector");

	if (*cp == 'x')
		return curobj->wherelocs[ch - 'a'].col;

	return -curobj->wherelocs[ch - 'a'].row;
}


/*
 * Return the value of a single character variable name (a-z or A-Z).
 */
VALUE
GetVariable1(ch)
	int	ch;
{
	if (islower(ch))
		return lowervars[ch - 'a'];

	if (isupper(ch))
		return lowervars[ch - 'A'];

	Error("Bad variable name");

	return 0;
}


/*
 * Set the value of a variable name.  Multi-character names cannot be set.
 */
void
SetVariable(cp, value)
	char *	cp;
	VALUE	value;
{
	if (*cp == '$')
		cp++;

	if (cp[1] != '\0')
		Error("Cannot set multi-character variables");

	SetVariable1(*cp, value);
}


/*
 * Set the value of a single-character variable (a-z or A-Z).
 */
void
SetVariable1(ch, value)
	int	ch;
	VALUE	value;
{
	if (islower(ch)) {
		lowervars[ch - 'a'] = value;
		return;
	}

	if (isupper(ch)) {
		lowervars[ch - 'A'] = value;
		return;
	}

	Error("Bad variable name");
}


/*
 * Display the current values of the variables.  Show all multi-character
 * variable values, and those single character variables which have a
 * nonzero value.  The output is given three variables per line.
 */
void
ListVariables()
{
	VARTAB *	vp;
	VALUE *		var;
	int		count;
	int		i;
	char		buf[80];

	ShowHelp("Variable names and values:");

	count = 0;

	/*
	 * Show all fixed variables, even if their value is zero.
	 */
	for (vp = vartab; vp->name; vp++) {
		sprintf(buf, "%s%s\t%ld", (count++ % 3) ? "\t\t" : "\n",
			vp->name, GetVariableByType(vp->type));

		ShowHelp(buf);
	}

	/*
	 * Show the upper case and lower case single character variables
	 * which are nonzero.
	 */
	count = 0;

	for (var = uppervars; var < &uppervars[26]; var++) {
		if (*var == 0)
			continue;

		if (count == 0)
			ShowHelp("\n");

		sprintf(buf, "%s%c\t%ld", (count++ % 3) ? "\t\t" : "\n",
			'A' + (var - uppervars), *var);

		ShowHelp(buf);
	}

	for (var = lowervars; var < &lowervars[26]; var++) {
		if (*var == 0)
			continue;

		if (count == 0)
			ShowHelp("\n");

		sprintf(buf, "%s%c\t%ld", (count++ % 3) ? "\t\t" : "\n",
			'a' + (var - lowervars), *var);

		ShowHelp(buf);
	}

	/*
	 * Show all where locations which have a nonzero coordinate.
	 */
	count = 0;

	for (i = 0; i < 26; i++) {
		if (curobj->wherelocs[i].col) {
			if (count == 0)
				ShowHelp("\n");

			sprintf(buf, "%sw%cx\t%ld",
				(count++ % 3) ? "\t\t" : "\n",
				'a' + i, curobj->wherelocs[i].col);

			ShowHelp(buf);
		}

		if (curobj->wherelocs[i].row) {
			if (count == 0)
				ShowHelp("\n");

			sprintf(buf, "%sw%cy\t%ld",
				(count++ % 3) ? "\t\t" : "\n",
				'a' + i, -curobj->wherelocs[i].row);

			ShowHelp(buf);
		}
	}

	ShowHelp("\n\nMany names starting with 'v' refer to visible cells\n");
	ShowHelp("Many names starting with 's' refer to selected cells\n");
	ShowHelp("Also, 'where' variable coordinates may be referenced as\n");
	ShowHelp("w<ch>x or w<ch>y, where <ch> is the variable name.\n");

	EndHelp();
}


/*
 * Evaluate an expression and return its value.  The expression can contain
 * numbers, variables, the normal arithmetic operators, and parenthesized
 * expressions.  The usual precedence rules are used.  The string must be
 * writable.
 */
VALUE
GetExpression(str)
	char *	str;
{
	VALUE	value;

	while isblank(*str)
		str++;

	if (*str == '\0')
		Error("Null expression");

	curcp = str;
	value = ParseSum();

	if (*curcp != '\0')
		Error("Bad expression");

	return value;
}


/*
 * Parse the sum of products
 */
static VALUE
ParseSum()
{
	VALUE	value;

	while (isblank(*curcp))
		curcp++;

	switch (*curcp) {
		case '-':
			curcp++;
			value = -ParseProduct();
			break;

		case '+':
			curcp++;
			/* proceed into default case */

		default:
			value = ParseProduct();
	}

	while (TRUE) switch (*curcp++) {
		case '+':		/* sum of products */
			value += ParseProduct();
			continue;

		case '-':		/* difference of products */
			value -= ParseProduct();
			continue;

		case ' ':		/* space */
		case '\t':
			continue;

		default:		/* end of sum */
			curcp--;
			return value;
	}
}


/* 
 * Parse the product of terms
 */
static VALUE
ParseProduct()
{
	VALUE	value;
	VALUE	value2;

	value = ParseTerm();

	while (TRUE) switch (*curcp++) {
		case '*':			/* product of terms */
			value *= ParseTerm();
			continue;

		case '/':			/* division of terms */
			value2 = ParseTerm();

			if (value2 == 0)
				Error("Division by zero");

			value /= value2;
			continue;

		case '%':			/* modulo of terms */
			value2 = ParseTerm();

			if (value2 == 0)
				Error("Division by zero");

			value %= value2;
			continue;

		case ' ':			/* space */
		case '\t':
			continue;

		default:			/* end of product */
			curcp--;
			return value;
	}
}


/*
 * Parse a single term
 */
static VALUE
ParseTerm()
{
	VALUE	value;
	int	ch;

	while (isblank(*curcp))
		curcp++;

	ch = *curcp;

	if (isdigit(ch)) {		/* number */
		value = 0;

		do {
			value = (value * 10) + *curcp++ - '0';
		} while (isdigit(*curcp));

		return value;
	}

	if (ch == '(') {		/* parenthesized expression */
		curcp++;

		while (isblank(*curcp))
			curcp++;

		if (*curcp == ')')
			Error("Null expression");

		value = ParseSum();

		while (isblank(*curcp))
			curcp++;

		if (*curcp != ')')
			Error("Unmatched parenthesis");

		curcp++;

		return value;
	}

	if (ch == ')')
		Error("Unmatched parenthesis");

	return ParseName();
}


/*
 * Parse a variable name and return its value.
 */
static VALUE
ParseName()
{
	char *	cp;
	VALUE	value;
	int	oldch;

	cp = curcp;

	if (*cp == '$')
		cp++;

	while (islower(*cp) || isupper(*cp) || isdigit(*cp))
    		cp++;

	oldch = *cp;
	*cp = '\0';			/* this requires writable strings */

	value = GetVariable(curcp);

	*cp = oldch;
	curcp = cp;

	return value;
}

/* END CODE */
