/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


/*
 * Set the scaling factor for the specified object.  This also centers
 * the view around the current cursor location.  Positive scale factors
 * mean compress the view so that each displayed item represents multiple
 * cells.  Negative scale factors mean use multiple displayed items to
 * represent individual cells.
 */
void
SetScale(obj, scale)
	OBJECT *obj;
	SCALE	scale;
{
	COUNT	newrows;
	COUNT	newcols;

	if (scale < dev->minscale)
		scale = dev->minscale;

	if (scale > dev->maxscale)
		scale = dev->maxscale;

	if ((scale == 0) || (scale == -1))
		scale = 1;

	if (scale > 0) {
		newrows = dev->rows * scale;
		newcols = dev->cols * scale;
	} else {
		newrows = dev->rows / (-scale);
		newcols = dev->cols / (-scale);

		if (newrows <= 0)
			newrows = 1;

		if (newcols <= 0)
			newcols = 1;
	}

	obj->scale = scale;
	obj->minrow = obj->currow - (newrows / 2);
	obj->maxrow = obj->minrow + newrows - 1;
	obj->mincol = obj->curcol - (newcols / 2);
	obj->maxcol = obj->mincol + newcols - 1;
	obj->viewrows = newrows;
	obj->viewcols = newcols;
	obj->update |= U_ALL;
}


/*
 * Perform auto-scaling of the current object.  This implies picking a
 * scaling factor such that the whole object fits in the screen.  The
 * scale factor is never decreased.  When the scale factor is large,
 * convenient ones are picked.  Returns the new scale factor.
 */
SCALE
AutoScale()
{
	OBJECT *obj;
	SCALE	scale;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;

	obj = curobj;
	MinMax(obj, &minrow, &maxrow, &mincol, &maxcol);
	scale = obj->scale;

	if (mincol > maxcol)
		return scale;

	while ((scale <= dev->maxscale) &&
		((minrow < obj->minrow) || (maxrow > obj->maxrow) ||
		(mincol < obj->mincol) || (maxcol > obj->maxcol)))
	{
		scale++;

		if ((scale == -1) || (scale == 0))
			scale = 1;

		if (scale > 20)
			scale += (5 - (scale % 5));

		if (scale > 50)
			scale += (10 - (scale % 10));

		if (scale > 200)
			scale += (100 - (scale % 100));

		SetScale(obj, scale);
	}

	return obj->scale;
}


/*
 * Position the view of the current object to show both the given region and
 * the current cursor location.  If this is impossible, just the cursor
 * location will be positioned.
 */
void
PositionView(minrow, maxrow, mincol, maxcol)
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
{
	OBJECT *obj;

	obj = curobj;

	if (minrow > obj->currow)
		minrow = obj->currow;

	if (maxrow < obj->currow)
		maxrow = obj->currow;

	if (mincol > obj->curcol)
		mincol = obj->curcol;

	if (maxcol < obj->curcol)
		maxcol = obj->curcol;

	if ((maxrow - minrow) >= obj->viewrows) {	/* too many rows */
		minrow = obj->currow;
		maxrow = obj->currow;
	}

	if ((maxcol - mincol) >= obj->viewcols) {	/* too many columns */
		mincol = obj->curcol;
		maxcol = obj->curcol;
	}

	if (minrow < obj->minrow) {
		obj->minrow = minrow;
		obj->maxrow = minrow + obj->viewrows - 1;
		obj->update |= U_ALL;
	}

	if (maxrow > obj->maxrow) {
		obj->maxrow = maxrow;
		obj->minrow = maxrow - obj->viewrows + 1;
		obj->update |= U_ALL;
	}

	if (mincol < obj->mincol) {
		obj->mincol = mincol;
		obj->maxcol = mincol + obj->viewcols - 1;
		obj->update |= U_ALL;
	}

	if (maxcol > obj->maxcol) {
		obj->maxcol = maxcol;
		obj->mincol = maxcol - obj->viewcols + 1;
		obj->update |= U_ALL;
	}
}


/*
 * Show the view around the current window location if the view has changed.
 * The update flag indicates what parts of the display needs updating.
 * This is any of U_POS, U_STAT, and U_VIEW.
 */
void
UpdateView()
{
	OBJECT *obj;

	obj = curobj;

	if (!interact && (obj->update == 0))
		return;

	PositionView(obj->currow, obj->currow, obj->curcol, obj->curcol);

	if (obj->autoscale)
		AutoScale();

	if (obj->update & U_VIEW) {
		freqcount = obj->frequency;
		(*dev->showview)(dev, obj);
	}

	if (obj->update & (U_STAT | U_VIEW))
		ViewStatus();

	if (obj->update & (U_STAT | U_VIEW | U_POS)) {
		(*dev->movecursor)(dev, obj->scale,
			obj->currow - obj->minrow,
			obj->curcol - obj->mincol);

		(*dev->update)(dev, FALSE);
	}

	obj->update = 0;	/* no more updates until prodded */
	interact = FALSE;
}


/*
 * Update the status line for the object.
 */
void
ViewStatus()
{
	OBJECT *obj;
	char *	cp;
	char	buf[132];

	if (errorstring) {
		(*dev->showstatus)(dev, errorstring);
		return;
	}

	obj = curobj;
	cp = buf;
	sprintf(cp, "Gen:%ld cells:%ld", obj->gen, obj->count);
	cp += strlen(cp);

	if (obj->count > seecount) {
		sprintf(cp, "(%ldu)", obj->count - seecount);
		cp += strlen(cp);
	}

	if (obj->born) {
		sprintf(cp, " born:%ld", obj->born);
		cp += strlen(cp);
	}

	if (obj->died) {
		sprintf(cp, " died:%ld", obj->died);
		cp += strlen(cp);
	}

	if (obj->frequency > 1) {
		sprintf(cp, " freq:%ld", obj->frequency);
		cp += strlen(cp);
	}

	if ((obj->scale != 1) || (obj->autoscale)) {
		sprintf(cp, " %scale:%d",
			(obj->autoscale ? "autos" : "s") , obj->scale);
		cp += strlen(cp);
	}

	if (strcmp(rulestring, DEFAULTRULE)) {
		sprintf(cp, " rule:%s", rulestring);
		cp += strlen(cp);
	}

	if (curinput > inputs) {
		sprintf(cp, " cmd-nest:%d", curinput - inputs);
		cp += strlen(cp);
	}

	switch (curinput->type) {
		case INP_TTY:			/* reading from terminal */
			if (curinput != inputs)
				strcpy(cp, " tty-wait");
			break;

		case INP_FILE:			/* reading from file */
			strcpy(cp, " cmd-file");
			break;

		case INP_LOOP:			/* reading from loop */
			if (curinput->macro) {
				sprintf(cp, " macro-define-%c",
					curinput->macro);
				break;
			}

			sprintf(cp, " loop%s (curval:%ld end:%ld)",
				curinput->first ? "-define" : "",
				curinput->curval, curinput->endval);
			break;

		case INP_MACRO:			/* reading from macro */
			sprintf(cp, " macro-%c", curinput->macro);
			break;
	}

	cp += strlen(cp);

	if (mode == M_INSERT)
		strcpy(cp, " inserting");

	if (mode == M_DELETE)
		strcpy(cp, " deleting");

	cp += strlen(cp);

	if (curobj != mainobject) {
		sprintf(cp, " \"%s\"", curobj->name);
		cp += strlen(cp);
	}

	(*dev->showstatus)(dev, buf);
}

/* END CODE */
