/*
 * static char *rcsid_map_c =
 *   "$Id: map.c,v 1.6 1993/04/25 16:08:27 frankj Exp $";
 */

/*
    CrossFire, A Multiplayer game for X-windows

    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to frankj@ifi.uio.no.
*/

#include <global.h>
#include <object.h>
#include <funcpoint.h>

#ifdef NeXT
extern char *tempnam(char *dir, char *pfx);
#endif

#if defined (MACH) || defined (NeXT)
#ifndef S_IWOTH
#define S_IWOTH 0000200
#endif
#ifndef S_IWGRP
#define S_IWGRP 0000020
#endif
#ifndef S_IWUSR
#define S_IWUSR 0000002
#endif
#ifndef S_IROTH
#define S_IROTH 0000400
#endif
#ifndef S_IRGRP
#define S_IRGRP 0000040
#endif
#ifndef S_IRUSR
#define S_IRUSR 0000004
#endif
#endif
#if defined(MACH) || defined(vax) || defined(ibm032) || defined(NeXT)
#ifndef S_ISDIR
#define S_ISDIR(x) (((x) & S_IFMT) == S_IFDIR)
#endif
#endif

/*
 * Returns the mapstruct which has a name matching the given argument.
 */

mapstruct *has_been_loaded (char *name) {
  mapstruct *map;
  if (!name || !*name)
     return 0;
  for (map = first_map; map; map = map->next)
      if (!strcmp (name, map->path))
	  break;
  return (map);
}

/*
 * This makes a path absolute outside the world of Crossfire.
 * In other words, it prepends LIBDIR/MAPDIR/ to the given path
 * and returns the pointer to a static array containing the result.
 */

char *create_pathname (char *name) {
  static char buf[MAX_BUF];

  if (*name == '/')
      sprintf (buf, "%s/%s%s", LibDir, MapDir, name);
  else
      sprintf (buf, "%s/%s/%s", LibDir, MapDir, name);
  return (buf);
}

/*
 * This program checks if a file with the given path exists.
 * -1 is returned if it failes, otherwise the mode of the file
 * is returned.
 */

int check_path (char *name)
{
    char buf[MAX_BUF];
    struct stat statbuf;
    int mode = 0;

    strcpy (buf, create_pathname(name));
    if (stat (buf, &statbuf)) {
	strcat (buf, COMPRESS_SUFFIX);
	if (stat (buf, &statbuf))
	    return (-1);
    }
    if (!S_ISREG (statbuf.st_mode))
	return (0);

    if (((statbuf.st_mode & S_IRGRP) && getegid() == statbuf.st_gid) ||
	((statbuf.st_mode & S_IRUSR) && geteuid() == statbuf.st_uid) ||
	(statbuf.st_mode & S_IROTH))
	mode |= 4;

    if ((statbuf.st_mode & S_IWGRP && getegid() == statbuf.st_gid) ||
	(statbuf.st_mode & S_IWUSR && geteuid() == statbuf.st_uid) ||
	(statbuf.st_mode & S_IWOTH))
	mode |= 2;
    
    return (mode);
}

/*
 * Prints out debug-information about a map.
 */

void dump_map(mapstruct *m) {
  LOG(llevError,"Map %s status: %d.\n",m->path, m->in_memory);
  LOG(llevError,"Size: %dx%d Start: %d,%d\n",
      m->mapx, m->mapy,
      EXIT_X(m->map_object),EXIT_Y(m->map_object));
  if(m->map_object->msg!=NULL)
    LOG(llevError,"Message:\n%s",m->map_object->msg);
  if(m->tmpname!=NULL)
    LOG(llevError,"Tmpname: %s\n",m->tmpname);
}

/*
 * Prints otu debug-information about all maps.
 */

void dump_all_maps() {
  mapstruct *m;
  for(m=first_map;m!=NULL;m=m->next) {
    LOG(llevError,"Map %s status %d.\n",m->path,m->in_memory);
    LOG(llevError,"Tmpname: %s\n",m->tmpname);
  }
}

/*
 * Returns true if a wall is present in a given location.
 */

int wall(mapstruct *m, int x,int y) {
    if(out_of_map(m,x,y))
	return 1;
    return (get_map(m,x,y))->flags & P_NO_PASS;
}

/*
 * Returns true if it's impossible to see through the given coordinate
 * in the given map.
 */

int blocks_view(mapstruct *m, int x, int y) {
    if(out_of_map(m,x,y))
	return 1;
    return (get_map(m,x,y))->flags & P_BLOCKSVIEW;
}

/*
 * Returns true if the given coordinate in the given map blocks magic.
 */

int blocks_magic(mapstruct *m, int x, int y) {
    if(out_of_map(m,x,y))
	return 1;
    return (get_map(m,x,y))->flags & P_NO_MAGIC;
}

/*
 * Returns true if the given coordinate in the given map blocks passage.
 */

int blocked(mapstruct *m, int x, int y) {
    int r;
    MapLook *f;
    if(out_of_map(m,x,y))
	return 1;
    f = get_map(m,x,y);
    r = f->flags & (P_NO_PASS | P_IS_ALIVE);
    return r;
}

/*
 * Returns true if the given coordinate in the map where the given object
 * is, blocks the given object (which may be multi-part)
 */

int blocked_link(object *ob, int x, int y) {
  object *tmp;
  if(out_of_map(ob->map,x,y))
    return 1;
  if (!blocked(ob->map,x,y)) /* no need to go further */
      return 0; 
  if(ob->head != NULL)
    ob=ob->head;
  for(tmp = get_map_ob(ob->map,x,y); tmp!= NULL; tmp = tmp->above)
    if (NO_PASS(tmp) || (IS_ALIVE(tmp) && tmp->head != ob && tmp != ob))
      return 1;
  return 0;
}

/*
 * Eneq(@csd.uu.se): This is a new version of blocked, this one handles objects
 * that can be passed through by monsters with the CAN_PASS_THRU defined.
 */

int blocked_two(object *op, int x,int y) {
  object *tmp;
  if(out_of_map(op->map,x,y))
    return 1;
  tmp=get_map_ob(op->map,x,y);
  for(tmp=get_map_ob(op->map,x,y);tmp!=NULL;tmp=tmp->above)
    if((IS_ALIVE(tmp)&&tmp->type!=DOOR)||(NO_PASS(tmp)&&!PASS_THRU(tmp))||
       (NO_PASS(tmp)&&PASS_THRU(tmp)&&!CAN_PASS_THRU(op)))
      return 1;
  return 0;
}

/*
 * Returns true if the given archetype can't fit in the given spot.
 */

int arch_blocked(archetype *at,mapstruct *m,int x,int y) {
  archetype *tmp;
  if(at==NULL)
    return blocked(m,x,y);
  for(tmp=at;tmp!=NULL;tmp=tmp->more)
    if(blocked(m,x+tmp->clone.x,y+tmp->clone.y))
      return 1;
  return 0;
}

/*
 * Returns true if the given archetype can't fit into the map at the
 * given spot (some part of it is outside the map-boundaries).
 */

int arch_out_of_map(archetype *at,mapstruct *m,int x,int y) {
  archetype *tmp;
  if(at==NULL)
    return out_of_map(m,x,y);
  for(tmp=at;tmp!=NULL;tmp=tmp->more)
    if(out_of_map(m,x+tmp->clone.x,y+tmp->clone.y))
      return 1;
  return 0;
}

#ifndef SPEED_GAME

/*
 * Returns true if the given coordinates is outside the boundaries
 * of the given map.  This function is actually a macro now.
 */

int out_of_map(mapstruct *m, int x, int y) {
  if(m==NULL) {
    LOG(llevError,"Out of map called with NULL\n");
    return 1;
  }
  return x<0||x>=m->mapx||y<0||y>=m->mapy;
}

#endif

/*
 * Goes through all objects in the given map, and does a sanity-check
 * on all pointers.
 */

void refresh_map(mapstruct *m) {
  int x,y;
  object *tmp,*tmp2,*active=NULL;

  if(m==NULL || m->in_memory != MAP_IN_MEMORY)
    return;
  for(x=0;x<m->mapx;x++)
    for(y=0;y<m->mapy;y++) {
/* Eneq(@csd.uu.se): Hunting down inappropriate objects in the map. The game
   sometime hangs and tries to remove_removed objects etc. */
      for(active=tmp=get_map_ob(m,x,y);tmp!=NULL;tmp=tmp2)
      { /* how can we get objects with pointer<1024, mta */
        if (tmp < (object *) 1024) /* map is uninitialized? */
          break;
        tmp2=tmp->above;
        active=tmp;
        if (IS_REMOVED(tmp)) {
          LOG(llevError,"Crossfire: Found removed object in map.\n");
          active=(tmp->above==NULL?tmp->below:tmp->above);
          if (tmp->below==NULL&&tmp2==NULL)
            m->map_ob[x+m->mapx*y]=NULL;
          else if (tmp->below==NULL&&tmp2!=NULL) {
            tmp2->below=NULL;
            m->map_ob[x+m->mapx*y]=tmp2;
          } else if (tmp->below!=NULL&&tmp2==NULL)
            tmp->below->above=NULL;
          else if (tmp->below!=NULL&&tmp2!=NULL) {
            tmp->below->above=tmp2;
            tmp2->below=tmp->below;
          }
          if (!IS_FREED(tmp))
            free_object(tmp);
        }
        if (IS_FREED(tmp)&&!IS_REMOVED(tmp)) {
          LOG(llevError, "Crossfire: Found freed object in map.\n");
          remove_ob(tmp);
        }
      }

      if(active != NULL)
        update_object(active);
    }
}

/*
 * open_and_uncompress() first searches for the original filename.
 * if it exist, then it opens it and returns the file-pointer.
 * if not, it does two things depending on the flag.  If the flag
 * is set, it tries to create the original file by uncompressing a .Z file.
 * If the flag is not set, it creates a temporarily file and uncompresses
 * the .Z file into that file.  The temporary file will be deleted in
 * close_and_delete().
 * (Note, the COMPRESS_SUFFIX is used instead of ".Z", thus it can easily
 * be changed in the config file.)
 */

FILE *open_and_uncompress(char *name,int flag) {
  FILE *fp;
  char buf[MAX_BUF],buf2[MAX_BUF];
  if((fp=fopen(name,"r"))!=NULL)
    return fp;
  strcpy(buf,name);
  strcat(buf,COMPRESS_SUFFIX);
  if((fp=fopen(buf,"r"))==NULL)
    return NULL;
  fclose(fp);
  if((fp=fopen(UNCOMPRESS,"r"))==NULL) {
    LOG(llevError,"Error: Can't find uncompress(%s).\n",UNCOMPRESS);
    return NULL;
  }
  fclose(fp);
  if(editor) /* Hack that always uncompress the original file if in editor */
    flag=1;
  if(flag)
    sprintf(buf2,"%s %s",UNCOMPRESS,buf);
  else
    sprintf(buf2,"%s < %s > %s/uncomp.%d",UNCOMPRESS,buf,TMPDIR,getpid());
  system(buf2);
  if(flag) {
    chmod(name,0664);
    return fopen(name,"r");
  }
  delete_last_file=1;
  sprintf(buf,"%s/uncomp.%d",TMPDIR,getpid());
  return fopen(buf,"r");
}

/*
 * See open_and_uncompress().
 */

void close_and_delete(FILE *fp) {
  char buf[MAX_BUF];
  fclose(fp);
  if(!delete_last_file)
    return;
  delete_last_file=0;
  sprintf(buf,"%s/uncomp.%d",TMPDIR,getpid());
  unlink(buf);
}

void check_read_only(mapstruct *m) {
    int mode = check_path(m->path);

  if (mode < 0) {
      LOG(llevError,"Can't open %s\n",m->path);
      perror("stat");
      return;
  }
  m->read_only = (mode & 2);
}

/*
 * Loads (ands parses) the objects into a given map from the specified
 * file pointer.
 */

void load_objects (mapstruct *m, FILE *fp) {
    int i, j = 0;
    object *op, *prev=NULL,*last_more=NULL, *tmp, *otmp;
    
    op=get_object();
    op->map = m; /* To handle buttons correctly */

    while((i=load_object(fp,op))) {
	switch(i) {
	  case 1:
	    insert_ob_in_map(op,m);
	    for (tmp=op->inv, op->inv=NULL;tmp!=NULL;tmp=otmp) {
		if (tmp->nrof)
		    op->carrying-=tmp->weight*tmp->nrof;
		else
		    op->carrying-=tmp->weight+tmp->carrying;
		SET_NO_FIX_PLAYER(tmp);
		otmp=tmp->below;
		insert_ob_in_ob(tmp,op);
		UNSET_NO_FIX_PLAYER(tmp);
	    }
	    prev=op,last_more=op;
	    break;

	  case 2:
	    insert_ob_in_ob(op,prev);
	    break;

	  case 3:
	    insert_ob_in_map(op,m);
	    op->head=prev,last_more->more=op,last_more=op;
	    break;
	}
/*
	if(!editor && m->first_visit && op->arch!=NULL && 
	op->arch->randomitems!=NULL)
	    create_treasure(op->arch->randomitems,op,GT_INVENTORY,m->difficulty);
*/
	op=get_object();
        op->map = m;

#if CHECK_ACTIVE_MAPS
	if(j++>CHECK_ACTIVE_MAPS) {
	    j=0;
	    /* We don't want to freeze the whole game too long */
	    (*process_active_maps_func)(); 
	}
#endif
    }
    free_object(op);
    close_and_delete(fp);
}

void save_objects (mapstruct *m, FILE *fp) {
    int i, j = 0;
    object *op, *tmp, *otmp;

    /* first pass - save one-part objects */
    for(i = 0; i < m->mapx; i++)
	for (j = 0; j < m->mapy; j++)
	    for(op = get_map_ob (m, i, j); op; op = otmp) {
		otmp = op->above;

		if(op->type == PLAYER || op->head || op->more || op->owner)
		    continue;

		save_object(fp, op, 3);
		/* don't check and don't remove */
             }
    /* second pass - save multi-part objects */
    for(i = 0; i < m->mapx; i++)
        for (j = 0; j < m->mapy; j++)
            for(op = get_map_ob (m, i, j); op; op = otmp) {
                otmp = op->above;

                if(op->type == PLAYER || op->head || !op->more || op->owner)
                    continue;

                save_object(fp, op, 3);
                /* don't check and don't remove */

		for (tmp = op->more; tmp; tmp = tmp->more) {
		    fprintf (fp, "More\n");
		    save_object (fp, tmp, 3); 
		    /* don't check and don't remove */
		}
	    }
}

/*
 * Opens the file "filename" and reads information about the map
 * from the given file, and stores it in a newly allocated
 * mapstruct.  A pointer to this structure is returned, or NULL on failure.
 */

mapstruct *load_original_map(char *filename) {
    FILE *fp;
    mapstruct *m;
    object *op;
    
    if((fp=open_and_uncompress(filename,0))==NULL) {
	LOG(llevError,"Can't open %s\n",filename);
	perror("Can't read map file");
	return (NULL);
    }
        
    op = get_object();

    if (!load_object(fp,op) || op->arch->clone.type != MAP) {
	LOG(llevError,"Error in map '%s'\n", filename);
	return (NULL);
    }
    
    m = get_empty_map (op->x, op->y);
    
    strcpy (m->path, filename);
    m->difficulty=5; /* For create_object() in monsters */

    if (m->map_object)
	free_object (m->map_object);
    m->map_object = op;
    op->map = m;

    m->in_memory=MAP_LOADING;
    load_objects (m, fp);
    m->in_memory=MAP_IN_MEMORY;
    m->difficulty=calculate_difficulty(m);
	
    set_map_reset_time(m);

    return (m);
}

/*
 * Loads a map, which has been loaded earlier, from file.
 */

void load_temporary_map(mapstruct *m) {
    FILE *fp;
    object *op;
    
    if (!m->tmpname) {
	LOG(llevError, "No temporary filename for map %s\n", m->path);
        m = load_original_map(m->path);
	return;
    }

    if((fp=open_and_uncompress(m->tmpname,0))==NULL) {
	LOG(llevError,"Can't open %s\n",m->tmpname);
	perror("Can't read map file");
        m = load_original_map(m->path);
	return;
    }
    
    op = get_object();

    load_object(fp,op);
    if (op->arch == NULL || op->type != MAP) {
	LOG(llevError,"Error in temporary map '%s'\n", m->path);
        m = load_original_map(m->path);
	return;
    }

    if (m->map_object)
	free_object (m->map_object);
    m->map_object = op;

    allocate_map(m);
    clear_map(m);

    m->in_memory=MAP_LOADING;
    load_objects (m, fp);
    m->in_memory=MAP_IN_MEMORY;
}

/*
 * Saves a map to file.  If flag is set, it is saved into the same
 * file it was (originally) loaded from.  Otherwise a temporary
 * filename will be genarated, and the file will be stored there.
 * The temporary filename will be stored in the mapstructure.
 */

int new_save_map(mapstruct *m, int flag) {
    FILE *fp;
    char filename[MAX_BUF];
    
    if (flag && !*m->path) {
	LOG(llevError,"Tried to save map without path.\n");
	return -1;
    }
    
    if(flag) {
	strcpy (filename, create_pathname (m->path));
	make_path_to_file (filename);
    } else {
	if (!m->tmpname)
	    m->tmpname = tempnam(TMPDIR,NULL);
	strcpy(filename, m->tmpname);
    }
    LOG(llevDebug,"Saving map %s\n",m->path);
    
    if((fp=fopen(filename,"w"))==NULL) {
	perror("Can't open file for saving");
	return -1;
    }
    
    if (!m->map_object) {
	LOG(llevError, "no map object for map %s!\n", m->path);
	return -1;
    }

    save_object (fp, m->map_object, 3);    
    save_objects (m, fp);
    
    fclose(fp);

    if (flag && m->read_only)
	chmod (filename, 0644);
    else
	chmod (filename, 0644);
    return 0;
}
     
/*
 * If any directories in the given path doesn't exist, they are created.
 */

void make_path_to_file (char *filename)
{
    char buf[MAX_BUF], *cp = buf;
    struct stat statbuf;

    if (!filename || !*filename)
	return;
    strcpy (buf, filename);
    LOG(llevDebug, "make_path_tofile %s...", filename);
    while ((cp = strchr (cp + 1, (int) '/'))) {
	*cp = '\0';
	LOG(llevDebug, "Checking %s...", buf);
	if (stat (buf, &statbuf) || !S_ISDIR (statbuf.st_mode)) {
	    LOG(llevDebug, "Was not dir...");
	    if (mkdir (buf, 0777)) {
		perror ("Couldn't make path to file");
		return;
	    }
	    LOG(llevDebug, "Made dir.\n");
	} else
	    LOG(llevDebug, "Was dir\n");
	*cp = '/';
    }
}

/*
 * Used to fix an ancient bug in old maps.
 * Will soon be obsolete.
 */

/*
 * Clears the arrays containing object-pointers and outlook of a map.
 */

void clear_map(mapstruct *m) {
    MapLook *aptr, *endptr;

    endptr=m->map+m->mapx*m->mapy;
    for(aptr=m->map; aptr < endptr; ++aptr)
	*aptr = blank_look;
    memset(m->map_ob, 0,sizeof(object *)*m->mapx*m->mapy);
}

/*
 * This function relinks all _pointers_ to the objects from
 * one map to another.
 * Note: You can _not_ free the objects in the original map
 * after this function has been called.
 */

void copy_map(mapstruct *m1, mapstruct *m2) {
  int x,y;

  strncpy(m2->path,m1->path,BIG_NAME);
  x = m2->mapx;
  y = m2->mapy;
  copy_object (m1->map_object, m2->map_object);
  m2->mapx = x;
  m2->mapy = y;  

  for(x=0;x<m1->mapx&&x<m2->mapx;x++)
    for(y=0;y<m1->mapy&&y<m2->mapy;y++) {
      set_map(m2,x,y,get_map(m1,x,y));
    }
}

/*
 * Remove and free all objects in the inventory of the given object.
 * object.c ?
 */

void clean_object(object *op)
{
  object *tmp, *next;

  for(tmp = op->inv; tmp; tmp = next)
  {
    next = tmp->below;
    clean_object(tmp);
    remove_ob(tmp);
    free_object(tmp);
  }
}

#if 0 /* Why make it so difficult?  I believe the below is buggy */

void clean_object(object *op) {
  object *tmp;
  while((tmp=op->inv)!=NULL) {
    clean_object(tmp);

/* Eneq(@csd.uu.se): We'll have to move up the inventory or else op->inv
   always will point to the same object. */
    
    if (tmp->below!=NULL)
      tmp->below->above=NULL;
    op->inv=tmp->below;
    tmp->below=NULL;

    remove_ob(tmp);
    free_object(tmp);
  }
}

#endif

/*
 * Remove and free all objects in the given map.
 */

void free_all_objects(mapstruct *m) {
  int i,j;
  object *op;

  for(i=0;i<m->mapx;i++)
    for(j=0;j<m->mapy;j++) {
      object *previous_obj=NULL;
        while((op=get_map_ob(m,i,j))!=NULL) {
          if (op==previous_obj)
          {
            LOG(llevDebug, "free_all_objects: Link error, bailing out.\n");
            break;
          }
          previous_obj=op;
          if(op->head!=NULL)
            op = op->head;
          clean_object(op);
          remove_ob(op);
          free_object(op);
        }
      }
}

/*
 * This function moves all objects from one map to another.
 *
 * move_all_objects(): Only used from the editor(s?)
 * Yes. -Frank
 */

void move_all_objects(mapstruct *m1, mapstruct *m2) {
  int i,j;
  object *op;

  for(i=0;i<m1->mapx&&i<m2->mapx;i++)
    for(j=0;j<m1->mapy&&j<m2->mapy;j++) {
      while((op=get_map_ob(m1,i,j))!=NULL&&op->head==NULL) {
        remove_ob(op);
        op->x=i,op->y=j; /* Not really needed */
        insert_ob_in_map(op,m2);
      }
    }
  free_all_objects(m1);
}

/*
 * function: vanish mapstruct
 * m       : pointer to mapstruct, if NULL no action
 * Currently only used from editor, the game always remembers all maps.
 */

void delete_map(mapstruct *m) {
    mapstruct *tmp;

    if (!m)
      return;
    if (m->in_memory == MAP_IN_MEMORY)
	free_map (m, 1);
    if (m == first_map)
	first_map = m->next;
    else {
	for (tmp = first_map; tmp && tmp->next != m; tmp = tmp->next);
        if(tmp)
	  tmp->next = m->next;
    }
    free (m);
}

/*
 * Frees everything allocated by the given mapstructure.
 */

void free_map(mapstruct *m,int flag) {
  if (!m->in_memory) {
    LOG(llevError,"Trying to free freed map.\n");
    return;
  }
  if(flag)
    free_all_objects(m);
  m->in_memory=MAP_SWAPPED;
  if (m->map != NULL)
    CFREE(m->map);
  if (m->map_ob != NULL)
    CFREE(m->map_ob);
  m->map = NULL;/* It's being accessed again, incorrectly.  Maybe this will */
		/* make it get loaded back in. */
  m->map_ob = NULL;
  if (m->map_object)
      free_object (m->map_object);
  m->map_object=NULL;
  if (m->buttons)
    free_objectlinkpt(m->buttons);
  m->buttons = NULL;
}

/*
 * If there is a global message in the map where the given object is,
 * display the message to the object.
 */

void draw_map_message(object *op) {
  if(op->map->map_object->msg!=NULL)
    (*draw_info_func)(op,op->map->map_object->msg);
}

/*
 * Allocates, initialises, and returns a pointer to a mapstruct.
 */

mapstruct *get_linked_map(char *path) {
  mapstruct *map=(mapstruct *) CALLOC(1,sizeof(mapstruct));
  mapstruct *mp;

  if(map==NULL)
    fatal(OUT_OF_MEMORY);
  for(mp=first_map;mp!=NULL&&mp->next!=NULL;mp=mp->next);
  if(mp==NULL)
    first_map=map;
  else
    mp->next=map;
  map->next=NULL;

  map->path[0]='\0';
  map->tmpname=NULL;

#if 0
  map->mapx = map->mapy = 0;
  EXIT_X(map->map_object) = EXIT_Y(map->map_object) = 2;
#endif

  map->players=0;

  map->in_memory=MAP_SWAPPED;
  map->read_only=0;
  map->timeout=0;
  map->pending = (objectlink *) NULL;
  map->map_object = NULL;
  map->buttons = NULL;

  return map;
}

/*
 * Allocates the arrays contained in a mapstruct.
 */

void allocate_map(mapstruct *m) {
  if(m->in_memory != MAP_SWAPPED )
    return;
  m->in_memory = MAP_IN_MEMORY;
  m->map=(MapLook *) CALLOC(m->mapx*m->mapy,sizeof(MapLook));
  m->map_ob=(object **) CALLOC(m->mapx*m->mapy,sizeof(object *));
  if(m->map==NULL||m->map_ob==NULL)
    fatal(OUT_OF_MEMORY);
}

/*
 * Creates an empty map of the given size, and returns a pointer to it.
 */

mapstruct *get_empty_map (int sizex, int sizey) {
    mapstruct *m = get_linked_map (NULL);
    archetype *tmp;

    for (tmp = first_archetype; tmp; tmp = tmp->next)
	if (tmp->clone.type == MAP)
	    break;
    if (tmp) {
	m->map_object = ObjectCreateArch (tmp);
	m->mapx = sizex;
	m->mapy = sizey;
	allocate_map (m);
	clear_map (m);
    } else {
	m->map_object = NULL;
	LOG (llevError, "no map archetypes\n");
    }
    return m;
}

/*
 * Makes sure the given map is loaded and swapped in.
 * Returns a pointer to the given map.
 */

mapstruct *ready_map_name(char *name, int flag) {
    mapstruct *m;

    if (!name || check_path(name) < 4)
	return (NULL);

    /* Have we been at this level before? */
    m = has_been_loaded (name);
    if (m && (m->in_memory == MAP_LOADING || m->in_memory == MAP_IN_MEMORY))
      return m;

    if (flag || !m || (m->reset_time != -1 && seconds() > m->reset_time)) {

	/* first visit or time to reset */
	if (m) {
	    clean_tmp_map(m);	/* Doesn't make much difference */
	    delete_map(m);
	}

	/* create and load a map */
	if (!(m = load_original_map(create_pathname(name))))
	    return (NULL);

	strcpy (m->path, name);
	m->first_visit=0;

	(*fix_auto_apply_func)(m); /* Chests which open as default */

    } else {

	load_temporary_map (m);

	clean_tmp_map(m);
	m->in_memory = MAP_IN_MEMORY;
	m->tmpname = NULL;
	/* It's going to be saved anew anyway */
    }

    /* In case other objects press some buttons down */
    (*update_buttons_func)(m);

    return m;
}


#ifndef SPEED_GAME

/*
 * Sets how the map looks at a specified coordinate.
 * This function is a macro now.
 */

void set_map(mapstruct *m, int x, int y, MapLook *c) {
  if(m==NULL) {
    LOG(llevError,"Error: set_map called with NULL\n");
    return;
  }
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Error: set_map out of boundary: %d,%d of %d,%d\n",
            x,y,m->mapx,m->mapy);
    return;
  }
  if(m->in_memory!=MAP_IN_MEMORY && m->in_memory!=MAP_LOADING) {
    LOG(llevError,"Error: set_map called when mas was not in memory.\n");
    return;
  }
  m->map[x+m->mapx*y]=*c;
}

/*
 * Returns how a map looks at a spacified coordinate.
 * This function is a macro now.
 */

MapLook *get_map(mapstruct *m, int x, int y) {
  if(m==NULL) {
    LOG(llevError,"Error: get_map called with NULL\n");
    abort();
    return &blank_look;
  }
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Error: get_map out of boundary: %d,%d of %d,%d\n",
            x,y,m->mapx,m->mapy);
    abort();
    return  &blank_look;
  }
  if(m->in_memory!=MAP_IN_MEMORY && m->in_memory!=MAP_LOADING) {
    LOG(llevError,"Error: get_map called when mas was not in memory.\n");
    abort();
    return  &blank_look;
  }
  return &m->map[x+m->mapx*y];
}

/*
 * Links an object to a map.
 * Overwrite the link to any previous objects at that position.
 * This function is a macro now.
 */

void set_map_ob(mapstruct *m, int x, int y, object *tmp) {
  if(m==NULL) {
    LOG(llevError,"Error: set_map_ob called with NULL\n");
    abort();
    return;
  }
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Error: set_map_ob out of boundary: %d,%d of %d,%d\n",
            x,y,m->mapx,m->mapy);
    abort();
    return;
  }
  if(m->in_memory!=MAP_IN_MEMORY && m->in_memory!=MAP_LOADING) {
    dump_object(tmp);
    LOG(llevError,"Error: set_map_ob called when mas was not in memory.\n%s\n",
        errmsg);
    abort();
    return;
  }
  m->map_ob[x+m->mapx*y]=tmp;
}

/*
 * Returns the first object in the linked list at a specific position.
 * This function is a macro now.
 */

object *get_map_ob(mapstruct *m, int x, int y) {
  if(m==NULL) {
    LOG(llevError,"Error: get_map_ob called with NULL\n");
    abort();
    return NULL;
  }
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Error: get_map_ob out of boundary: %d,%d of %d,%d\n",
            x,y,m->mapx,m->mapy);
    abort();
    return NULL;
  }
  if(m->in_memory!=MAP_IN_MEMORY && m->in_memory!=MAP_LOADING) {
    LOG(llevError,"Error: get_map_ob called when mas was not in memory.\n");
    abort();
    return NULL;
  }
  /* Only with debugging malloc lib eanders@cmu.edu */
#ifdef with_debugging_malloc_lib_eanders_cmu_edu
  ASSERT(m->map_ob[x+m->mapx*y]!=0x55555555,"Youch, dangling pointer\n");
#endif
  return m->map_ob[x+m->mapx*y];
}

#endif

void no_maps_file(char *filename) {
  LOG(llevError,"Can't open the %s file.\n",filename);
  LOG(llevError,"If you have no maps, you must either make them yourself\n");
  LOG(llevError,"with the mapeditor, or fetch them from ftp.ifi.uio.no\n");
  LOG(llevError,"in the /pub/crossfire directory.  Be sure not to fetch\n");
  LOG(llevError,"maps belonging to a later version than this version.\n");
  exit(-1);
}

void set_map_reset_time(mapstruct *map) {
  if(MAP_RESET == 0)
    map->reset_time = (-1); /* Will never be reset */
  else
    map->reset_time = seconds() + MAP_RESET;
}

/*
 * This routine is supposed to find out which level the players should have
 * before visiting this map.  It is used to calculate which bonuses to put
 * on magic items.
 */

int calculate_difficulty(mapstruct *m) {
  object *op;
  archetype *at;
  int x,y;
  int diff=0;
  int total_exp=0,exp_pr_sq;
  int i;

  for(x=0;x<m->mapx;x++)
    for(y=0;y<m->mapy;y++)
      for(op=get_map_ob(m,x,y);op!=NULL;op=op->above) {
        if(IS_MONSTER(op))
          total_exp+=op->stats.exp;
        if(IS_GENERATOR(op)) {
          total_exp+=op->stats.exp;
          at=type_to_archetype(GENERATE_TYPE(op));
          if(at!=NULL)
            total_exp+=at->clone.stats.exp*8;
        }
      }
#ifdef NEWCALC
  (int)exp_pr_sq=((double)1000*total_exp)/(m->mapx*m->mapy+1);
  for(i=20;i>0;i--)
    if(exp_pr_sq>level_exp(i)) {
      diff=i;
      break;
    }
#else
  exp_pr_sq=((double)1000*total_exp)/(m->mapx*m->mapy+1);
  diff=20;
  for(i=1;i<20;i++)
    if(exp_pr_sq<=level_exp(i)) {
      diff=i;
      break;
    }
#endif
  return diff;
}

void clean_tmp_map(mapstruct *m) {
  if(m->tmpname == NULL)
    return;
  (void) unlink(m->tmpname);
}


object * MapGetObjectZ (mapstruct * emap, int x, int y, int z)
{
    object *op;

    if (!emap || out_of_map (emap, x, y))
        return (NULL);
    op = get_map_ob (emap, x, y);
    while (op && op->above)
        op = op->above;
    while (op && z-- > 0)
        op = op->below;
    return (op);
}

/*
 * member: copy by translate objects from source to a new map
 * source: -map
 * width : width of target map
 * height: height of target map
 * dx    : positive translate to right
 * dy    : positive translate to down
 */
mapstruct *MapMoveScrollResize(mapstruct *source, 
				int width, int height, int dx, int dy) 
{
    mapstruct *target;
    object *obj,*prt; /* PaRT of obj */
    int x,y,sx = source->mapx, sy = source->mapy;
    int linked = 0, link;

    if (!width) width = sx;
    if (!height) height = sy;
    target = get_empty_map (width, height);

    strncpy (target->path, source->path, BIG_NAME);

    copy_object (source->map_object, target->map_object);
    target->mapx = width;
    target->mapy = height;  

    if(dx < 0) dx += target->mapx;
    if(dy < 0) dy += target->mapy;

    for(y=0; y < sy && y < target->mapy; y++)
	for(x=0; x < sx && x < target->mapx; x++)
	    while((obj = get_map_ob(source,x,y)) && !obj->head) {
		if ((linked = IS_LINKED (obj))) {
		    link = get_button_value (obj);
		    remove_button_link (obj);
		}
		remove_ob(obj);
		for(prt = obj; prt; prt = prt->more) {
		    prt->x += dx;
		    prt->x %= target->mapx; /* it can be split by edge */
		    prt->y += dy;           /* designers problem to fix */
		    prt->y %= target->mapy;
		}
		insert_ob_in_map(obj,target);
		if (linked)
		    add_button_link(obj, target, link);
	    }
    /*free_all_objects(source);*/
    free_map (source, 1);
    delete_map (source);
    return target;
}

/*
 * member: copy by translate objects from source to target
 * target: -map
 * source: -map
 * dx    : positive translate to right
 * dy    : positive translate to down
 */
void MapMoveScroll(mapstruct *target, mapstruct *source, int dx, int dy) 
{
    object *obj,*prt; /* PaRT of obj */
    int x,y;

    if(dx < 0) dx += target->mapx;
    if(dy < 0) dy += target->mapy;

    for(y=0; y < source->mapy && y < target->mapy; y++)
	for(x=0; x < source->mapx && x < target->mapx; x++)
	    while((obj = get_map_ob(source,x,y)) && !obj->head) {
		remove_ob(obj);
		for(prt = obj; prt; prt = prt->more) {
		    prt->x += dx;
		    prt->x %= target->mapx; /* it can be split by edge */
		    prt->y += dy;           /* designers problem to fix */
		    prt->y %= target->mapy;
		}
		insert_ob_in_map(obj,target);
	    }
    free_all_objects(source);
}

object * MapGetRealObject (mapstruct * emap, int x, int y, int z)
{
    object *tmp = MapGetObjectZ (emap, x, y, z);
    return tmp ? (tmp->head ? tmp->head : tmp) : tmp;
}

int MapInsertObjectZ(mapstruct *emap,object *o,int x, int y, int z)
{
    object *op, *above, *below;

    if (o->more)
        MapInsertObjectZ (emap,o->more, x, y, z);

    o->x += x;
    o->y += y;
    o->map = emap;
#ifndef SPEED_GAME
    o->ox = x;
    o->oy = y;
#endif
    UNSET_REMOVED (o);

    op = get_map_ob (emap, o->x, o->y);
    if (z < 0) {
	above = op;
	below = NULL;
    } else {
	while (op && op->above)
	    op = op->above;
    
	above = NULL;
	below = op;
	while (op && z-- > 0) {
	    above = op;
	    below = op = op->below;
	}
    }
    o->below = below;
    o->above = above;

    if (above)
        above->below = o;
    else {
	MapLook f;
	f.flags = 0;
	f.face = o->face;
	set_map (emap, o->x, o->y, &f);
    }
    if (below)
        below->above = o;
    else
        set_map_ob (emap, o->x, o->y, o);
    
    return (0);
}

int MapObjectOut (mapstruct *target, object *obj, int x, int y) {
    object *tmp;
    for(tmp = obj; tmp; tmp = tmp->more)
        if(out_of_map(target,x + tmp->x,y + tmp->y)) return 1;
    return 0;
}
