/*
 * static char *rcsid_object_c =
 *   "$Id: object.c,v 1.5 1993/04/25 16:08:29 frankj Exp $";
 */

/*
    CrossFire, A Multiplayer game for X-windows

    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to frankj@ifi.uio.no.
*/

/* Eneq(@csd.uu.se): Added weight-modifiers in environment of objects.
   sub/add_weight will transcend the environment updating the carrying
   variable. */

#include <stdio.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <global.h>
#include <spells.h>
#include <graphics.h>
#include <object.h>
#include <funcpoint.h>

static char numbers[21][20] = {
  "no","a","two","three","four","five","six","seven","eight","nine","ten",
  "eleven","twelve","thirteen","fourteen","fifteen","sixteen","seventeen",
  "eighteen","nineteen","twenty"
};

int freearr_x[SIZEOFFREE]=
  {0,0,1,1,1,0,-1,-1,-1,0,1,2,2,2,2,2,1,0,-1,-2,-2,-2,-2,-2,-1,
   0,1,2,3,3,3,3,3,3,3,2,1,0,-1,-2,-3,-3,-3,-3,-3,-3,-3,-2,-1};
int freearr_y[SIZEOFFREE]=
  {0,-1,-1,0,1,1,1,0,-1,-2,-2,-2,-1,0,1,2,2,2,2,2,1,0,-1,-2,-2,
   -3,-3,-3,-3,-2,-1,0,1,2,3,3,3,3,3,3,3,2,1,0,-1,-2,-3,-3,-3};
static int maxfree[SIZEOFFREE]=
  {0,9,10,13,14,17,18,21,22,25,26,27,30,31,32,33,36,37,39,39,42,43,44,45,
  48,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49,49};
static int freedir[SIZEOFFREE]= {
  0,1,2,3,4,5,6,7,8,1,2,2,2,3,4,4,4,5,6,6,6,7,8,8,8,
  1,2,2,2,2,2,3,4,4,4,4,4,5,6,6,6,6,6,7,8,8,8,8,8};


object objarray[STARTMAX]; /* All objects, allocated this way at first */
object *objects;           /* Pointer to the list of used objects */
object *free_objects;      /* Pointer to the list of unused objects */

int nroffreeobjects = STARTMAX;  /* How many OBs allocated and free (free) */
int nrofallocobjects = STARTMAX; /* How many OBs allocated (free + used) */

/*
 * Eneq(@csd.uu.se): Since we can have items buried in a character we need
 * a better check
 */

object *is_player_inv (object *op) { 
    for (;op!=NULL&&op->type!=PLAYER; op=op->env)
      if (op->env==op)
      op->env = NULL;
    return op;
}

/*
 * Used by Crossedit?
 * The result of the dump is stored in the static global errmsg array.
 */

void dump_object2(object *op) {
  char *cp;
  object *tmp;
  if(op->arch!=NULL) {
      strcat(errmsg,"arch ");
      strcat(errmsg,op->arch->name);
      strcat(errmsg,"\n");
      if((cp=get_ob_diff(op,&empty_archetype->clone))!=NULL)
        strcat(errmsg,cp);
#if 0
      for (tmp=op->inv; tmp; tmp=tmp->below)
        dump_object2(tmp);
#endif
      strcat(errmsg,"end\n");
  }
}

/*
 * Dumps an object.  Returns output in the static global errmsg array.
 */

void dump_object(object *op) {
  if(op==NULL) {
    strcpy(errmsg,"[NULL pointer]");
    return;
  }
  errmsg[0]='\0';
  dump_object2(op);
}

/*
 * This is really verbose...Can be triggered by the P key while in DM mode.
 * All objects are dumped to stderr (or alternate logfile, if in server-mode)
 */

void dump_all_objects() {
  object *op;
  for(op=objects;op!=NULL;op=op->next) {
    dump_object(op);
    fprintf(logfile, "Object %d\n:%s\n", op->count, errmsg);
  }
}

/*
 * animate_object(object) updates the face-variable of an object.
 * If the object is the head of a multi-object, all objects are animated.
 * If the object has the IS_TURNING() flag, that is taken into consideration.
 */

void animate_object(object *op) {
  if(op->arch==NULL) {
    dump_object(op);
    LOG(llevError,"Tried to animate object without archetype:\n%s\n", errmsg);
  }
  if(!op->arch->animations) {
    LOG(llevError,"Archtype lacks animation.\n");
    dump_arch(op->arch);
  }
  if(++op->state>=(IS_TURNING(op)?op->arch->animations/2:op->arch->animations))
    op->state=0;
  op->face.number=op->arch->faces[op->state+(IS_TURNING(op)?
           ((op->head!=NULL?op->head:op)->value?op->arch->animations/2:0):0)];
  if(op->face.number==blank_face.number)
    op->invisible=1;
#if 1
  else if(IS_ALIVE((&op->arch->clone))) {
    if(!op->face.number) {
      op->invisible=1;
      UNSET_ALIVE(op);
    } else {
      op->invisible=0;
      SET_ALIVE(op);
    }
  }
#endif
  update_object(op);
  if(op->more)
    animate_object(op->more);
}

/*
 * get_nearest_part(multi-object, object 2) returns the part of the
 * multi-object 1 which is closest to the second object.
 * If it's not a multi-object, it is returned.
 */

object *get_nearest_part(object *op,object *pl) {
  object *tmp,*closest;
  int last_dist,i;
  if(op->more==NULL)
    return op;
  for(last_dist=distance(op,pl),closest=op,tmp=op->more;tmp!=NULL;tmp=tmp->more)
    if((i=distance(tmp,pl))<last_dist)
      closest=tmp,last_dist=i;
  return closest;
}

/*
 * Returns the object which has the count-variable equal to the argument.
 */

object *find_object(int i) {
  object *op;
  for(op=objects;op!=NULL;op=op->next)
    if(op->count==i)
      break;
 return op;
}

/*
 * Returns the first object which has a name equal to the argument.
 * Used only by the patch command, but not all that useful.
 * Enables features like "patch <name-of-other-player> food 999"
 */

object *find_object_name(char *str) {
  char *name=add_string(str);
  object *op;
  for(op=objects;op!=NULL;op=op->next)
    if(op->name==name)
      break;
  free_string(name);
  return op;
}

/*
 * Returns the object which this object marks as being the owner.
 * A id-scheme is used to avoid pointing to objects which have been
 * freed and are now reused.  If this is detected, the owner is
 * set to NULL, and NULL is returned.
 * (This scheme should be changed to a refcount scheme in the future)
 */

object *get_owner(object *op) {
  if(op->owner==NULL)
    return NULL;
  if(!IS_FREED(op->owner) && op->owner->count==op->ownercount)
    return op->owner;
  op->owner=NULL,op->ownercount=0;
  return NULL;
}

/*
 * Sets the owner of the first object to the second object.
 * Also checkpoints a backup id-scheme which detects freeing (and reusage)
 * of the owner object.
 * See also get_owner() and unset_owner().
 */

void set_owner(object *op, object *owner) {
  if(owner==NULL||op==NULL)
    return;
  op->owner=owner;
  op->ownercount=owner->count;
  owner->refcount++;
}

/*
 * clear_object() frees everything allocated by an object, and also
 * clears all variables and flags to default settings.
 */

void clear_object(object *op) {
  if(op->name!=NULL) {
    free_string(op->name);
    op->name=NULL;
  }
  if(op->race!=NULL) {
    free_string(op->race);
    op->race=NULL;
  }
  if(op->slaying!=NULL) {
    free_string(op->slaying);
    op->slaying=NULL;
  }
  if(op->msg!=NULL) {
    free_string(op->msg);
    op->msg=NULL;
  }
  op->flags=0; op->flags2=0; op->flags3=0;
  op->map=NULL;
  op->below=NULL;
  op->above=NULL;
  op->owner=NULL;
  op->refcount=0;
  op->inv=NULL;
  op->env=NULL;
  op->container=NULL;
  op->more=NULL;
  op->head=NULL;
  op->arch=NULL;
  op->other_arch=NULL;
  op->enemy=NULL;
  op->weight=0,op->carrying=0,op->weight_limit=0;
  op->anim_speed=0;
  op->level=0;
  op->bonus=0;
  op->value=op->run_away=0;
  op->invisible=0;
  op->last_heal=0,op->last_sp=0,op->last_eat=0;
  op->nrof=0;
  op->ownercount=0;
  op->immune=0,op->protected=0,op->attacktype=0,op->vulnerable=0;
  op->stats.exp=0;
  op->x=0; op->y=0; op->ox=0; op->oy=0;
  op->stats.dam=0,op->stats.wc=0,op->stats.ac=0;
  op->armour=0;
  op->stats.luck=0;
  op->stats.food=0;
  op->stats.sp=op->stats.maxsp=op->stats.hp=op->stats.maxhp=0;
  op->direction=0;
#ifdef NPC_PROG
  op->npc_status=0;
  op->npc_program=0;
#endif
  op->stats.Str=op->stats.Dex=op->stats.Con=0;
  op->stats.Wis=op->stats.Cha=op->stats.Int=0;
  op->material=op->magic=op->state=op->type=0;
  op->face = blank_face;
  op->speed=op->speed_left=0;
  op->pick_up=0;
  op->can_apply=0;
  op->will_apply=0;
  op->move_status = 0;
  op->move_type = 0;
}

/*
 * copy object first frees everything allocated by the second object,
 * and then copies the contends of the first object into the second
 * object, allocating what needs to be allocated.
 */

void copy_object(object *op2, object *op) {
  int is_freed=IS_FREED(op),is_removed=IS_REMOVED(op);
  if(op->name!=NULL)
    free_string(op->name);
  if(op->race!=NULL)
    free_string(op->race);
  if(op->slaying!=NULL)
    free_string(op->slaying);
  if(op->msg!=NULL)
    free_string(op->msg);
  (void) memcpy((char *)op +offsetof(object,name),
                (char *)op2+offsetof(object,name),
                sizeof(object)-offsetof(object,name));
  if(is_freed)
    SET_FREED(op);
  if(is_removed)
    SET_REMOVED(op);
  if(op->name!=NULL)
    add_refcount(op->name);
  if(op->race!=NULL)
    add_refcount(op->race);
  if(op->slaying!=NULL)
    add_refcount(op->slaying);
  if(op->msg!=NULL)
    add_refcount(op->msg);
  if(op2->speed<0)
    op->speed_left=op2->speed_left-RANDOM()%200/100.0;
}

/*
 * expand_objects() allocates more objects for the list of unused objects.
 * It is called from get_object() if the list unused list is empty.
 */

void expand_objects() {
  int i;
  object *new;
  new = (object *) CALLOC(OBJ_EXPAND,sizeof(object));

  if(new==NULL)
    fatal(OUT_OF_MEMORY);
  free_objects=new;
  new[0].prev=NULL;
  new[0].next= &new[1],
  new[0].flags|=(F_REMOVED|F_FREED);
  for(i=1;i<OBJ_EXPAND-1;i++)
    new[i].next= &new[i+1],
    new[i].prev= &new[i-1],
    new[i].flags|=(F_FREED|F_REMOVED);
  new[OBJ_EXPAND-1].prev= &new[OBJ_EXPAND-2],
  new[OBJ_EXPAND-1].next=NULL,
  new[OBJ_EXPAND-1].flags|=(F_FREED|F_REMOVED);
  nrofallocobjects += OBJ_EXPAND;
  nroffreeobjects += OBJ_EXPAND;
}

/*
 * get_object() grabs an object from the list of unused objects, makes
 * sure it is initialised, and returns it.
 * If there are no free objects, expand_objects() is called to get more.
 */

object *get_object() {
  object *op;
  if(free_objects==NULL) {
    expand_objects();
  }
  op=free_objects;
  if(!IS_FREED(op)) {
    LOG(llevError,"Fatal: Getting busy object.\n");
  }
  free_objects=op->next;
  if(free_objects!=NULL)
    free_objects->prev=NULL;
  op->count= ++ob_count;
  op->name=NULL;
  op->race=NULL;
  op->slaying=NULL;
  op->msg=NULL;
  op->next=objects;
  op->prev=NULL;
  if(objects!=NULL)
    objects->prev=op;
  objects=op;
  clear_object(op);
  SET_REMOVED(op);
  nroffreeobjects--;
  return op;
}

/*
 * If an object with the IS_TURNABLE() flag needs to be turned due
 * to the closest player being on the other side, this function can
 * be called to update the face variable, _and_ how it looks on the map.
 */

void update_turn_face(object *op) {
  if(!IS_TURNABLE(op)||op->arch==NULL)
    return;
  op->face.number=op->arch->faces[op->direction];
  update_object(op);
}

/*
 * update_object() updates the array which represents the map.
 * It takes into account invisible objects (and represent squares covered
 * by invisible objects by whatever is below them (unless it's another
 * invisible object, etc...))
 * If the object being updated is beneath a player, the look-window
 * of that player is updated (this might be a suboptimal way of
 * updating that window, though, since update_object() is called _often_)
 */

void update_object(object *op) {
  object *tmp;

  if(op->env!=NULL) {
    if((tmp=is_player_inv(op->env)))
      (*draw_inventory_faces_func)(tmp);
    return;
  }

  update_position (op->map, op->x, op->y);

  if(op->more!=NULL)
    update_object(op->more);
}

void update_position (mapstruct *m, int x, int y) {
    object *tmp, *last = 0;
    unsigned int flags = 0;
    MapLook f;

    for (tmp = get_map_ob (m, x, y); tmp; last = tmp, tmp = tmp->above) {
	if (tmp->type==PLAYER)
	    (*draw_look_faces_func)(tmp);
	if (tmp==tmp->above) {
	    LOG(llevError, "Error in structure of map\n");
	    exit (-1);
	}
      
	if (IS_ALIVE (tmp))
	    flags |= P_IS_ALIVE;
	if (NO_PASS (tmp))
	    flags |= P_NO_PASS;
	if (NO_MAGIC (tmp))
	    flags |= P_NO_MAGIC;
	if (PASS_THRU (tmp))
	    flags |= P_PASS_THRU;
	if (BLOCKSVIEW (tmp))
	    flags |= P_BLOCKSVIEW;
    }

    if (!editor)
	while (last && (last->invisible || !last->face.number))
	    last = last->below;

    f.face = last ? last->face : blank_face;
    f.flags = flags;
    set_map (m, x, y, &f);
    return;
}

/*
 * free_object() frees everything allocated by an object, removes
 * it from the list of used objects, and puts it on the list of
 * free objects.  The IS_FREED() flag is set in the object.
 * The object must have been removed by remove_ob() first for
 * this function to succeed.
 */

void free_object(object *ob) {
  object *tmp,*op;
  if(IS_FRIENDLY(ob)) {
    LOG(llevMonster,"Warning: tried to free friendly object.\n");
    (*remove_friendly_object_func)(ob);
  }
  if(IS_FREED(ob)) {
    dump_object(ob);
    LOG(llevError,"Trying to free freed object.\n%s\n",errmsg);
    return;
  }
  if(ob->more!=NULL) {
    free_object(ob->more);
    ob->more=NULL;
  }
  if (ob->inv) {
    if (ob->map==NULL || ob->map->in_memory!=MAP_IN_MEMORY ||
       wall(ob->map,ob->x,ob->y))
    {
      op=ob->inv;
      while(op!=NULL) {
        tmp=op->below;
        remove_ob(op);
        free_object(op);
        op=tmp;
      }
    }
    else {
      op=ob->inv;
      while(op!=NULL) {
        tmp=op->below;
        remove_ob(op);
        if(STARTEQUIP(op)||NO_DROP(op))
          free_object(op);
        else {
          op->x=ob->x,op->y=ob->y;
          insert_ob_in_map(op,ob->map); /* Insert in same map as the envir */
        }
        op=tmp;
      }
    }
  }
  SET_FREED(ob);
  /* First free the object from the used objects list: */
  if(ob->prev==NULL) {
    objects=ob->next;
    if(objects!=NULL)
      objects->prev=NULL;
  }
  else {
    ob->prev->next=ob->next;
    if(ob->next!=NULL)
      ob->next->prev=ob->prev;
  }
  
  /* Now link it with the free_objects list: */
  ob->prev=NULL;
  ob->next=free_objects;
  if(free_objects!=NULL)
    free_objects->prev=ob;
  free_objects=ob;
  ob->speed=0; /* Just to be on the safe side... */
  if(ob->name!=NULL) {
    free_string(ob->name);
    ob->name=NULL;
  }
  nroffreeobjects++;
}

/*
 * count_free() returns the number of objects on the list of free objects.
 */

int count_free() {
  int i=0;
  object *tmp=free_objects;
  while(tmp!=NULL)
    tmp=tmp->next, i++;
  return i;
}

/*
 * count_used() returns the number of objects on the list of used objects.
 */

int count_used() {
  int i=0;
  object *tmp=objects;
  while(tmp!=NULL)
    tmp=tmp->next, i++;
  return i;
}

/*
 * sub_weight() recursively (outwards) subtracts a number from the
 * weight of an object (and what is carried by it's environment(s)).
 */

void sub_weight (object *op, signed long weight) {
  while (op != NULL) {
    weight=(signed long)(weight*(100-op->bonus)/100);
    op->carrying-=weight;
    op = op->env;
  }
}

/* remove_ob(op):
 *   This function removes the object op from the linked list of objects
 *   which it is currently tied to.  When this function is done, the
 *   object will have no environment.  If the object previously had an
 *   environment, the x and y coordinates will be updated to
 *   the previous environment.
 *   Beware: This function is called from the editor as well!
 */

void remove_ob(object *op) {
  object *tmp,*last=NULL;
  object *otmp;

  if(IS_REMOVED(op)) {
    dump_object(op);
    LOG(llevError,"Trying to remove removed object.\n%s\n",errmsg);
    if(op->map!=NULL && op->map->in_memory == MAP_IN_MEMORY) {
      op->map->need_refresh=1;
      LOG(llevDebug,"Marked map %s for refresh.\n",op->map->path);
    }
    if(get_map_ob(op->map,op->x,op->y)==op)
    { /* we are at bottom [Smart! -Frank] */
       if (op->below)
	  set_map_ob(op->map,op->x,op->y,op->below);
       else if (op->above)
	  set_map_ob(op->map,op->x,op->y,op->above);
       else 
	  set_map_ob(op->map,op->x,op->y,NULL);
    }
    return; /* Why did you comment this away at DoCS?? It just makes it worse */
  }
#ifndef SPEED_GAME
  if(op->map==NULL && op->env==NULL) {
    dump_object(op);
    LOG(llevError,"Trying to remove object without map.\n%s\n",errmsg);
    abort();
    return;
  }
#if 0 /* inconsitent check with multipart objects ? */
  if(op->x!=op->ox||op->y!=op->oy) {
    LOG(llevError,"Object %d moved from %d,%d to %d,%d while not removed:\n",
            op->count,op->x,op->y,op->ox,op->oy);
    dump_object(op);
    LOG(llevError,"%s\n",errmsg);
    abort();
  }
#endif
#endif
  if(op->more!=NULL)
    remove_ob(op->more);
  SET_REMOVED(op);
  if(op->env!=NULL) {
    if(op->nrof)
      sub_weight(op->env, op->weight*op->nrof);
    else
      sub_weight(op->env, op->weight+op->carrying);
    if ((otmp=is_player_inv(op->env))!=NULL)
      fix_player(otmp);
    if(op->above!=NULL)
      op->above->below=op->below;
    else
      op->env->inv=op->below;
    if(op->below!=NULL)
      op->below->above=op->above;
    op->x=op->env->x,op->y=op->env->y;
    op->ox=op->x,op->oy=op->y;
    op->map=op->env->map;
    op->above=NULL,op->below=NULL;
    if(otmp!=NULL)
      (*draw_inventory_func)(otmp);
    op->env=NULL;
    return;
  }
  if(out_of_map(op->map,op->x,op->y))
    op->x=0,op->y=0,op->ox=0,op->oy=0;
  if(op->above!=NULL) {                /* Don't change map, we're not on top */
    op->above->below=op->below;        /* Link above with below */
    if(op->below!=NULL) {              /* Something below us? */
      op->below->above=op->above;      /* Yes, link above with below */
      op->below=NULL;
    }
    else {                             /* Nothing below, tell the floor what */
      if(get_map_ob(op->map,op->x,op->y)!=op) {
        dump_object(op);
        LOG(llevError,"Object squizez another object:\n%s\n",errmsg);
        dump_object(get_map_ob(op->map,op->x,op->y));
        LOG(llevError,"%s\n",errmsg);
      }
      set_map_ob(op->map,op->x,op->y,op->above);  /* goes on above it. */
    }
    op->above=NULL;                      
  }
  else if(op->below!=NULL) {
    op->below->above=NULL,             /* It is now on top */
    op->below=NULL;
  } else {
      if(get_map_ob(op->map,op->x,op->y)!=op) {
        LOG(llevError,"Object squizez another object:\n");
        dump_object(op);
        LOG(llevError,"%s\n",errmsg);
        dump_object(get_map_ob(op->map,op->x,op->y));
        LOG(llevError,"%s\n",errmsg);
      }
    set_map_ob(op->map,op->x,op->y,NULL);           /* No more objects here */
  }
  for(tmp=get_map_ob(op->map,op->x,op->y);tmp!=NULL;tmp=tmp->above) {
    if(tmp->type==PLAYER)
      (*draw_look_func)(tmp);
    if(IS_FLYING(op)?FLY_OFF(tmp):WALK_OFF(tmp))
      (*apply_func)(op,tmp);

    /* Eneq(@csd.uu.se): Fixed this to skip tmp->above=tmp */

    if(tmp->above == tmp)
      tmp->above = NULL;
    last=tmp;
  }  
  if(last==NULL)	/* No more objects present, draw the background */
    set_map(op->map,op->x,op->y,&blank_look);
  else			/* Redraw the object at the top */
    update_object(last);
  if(BLOCKSVIEW(op))		/* Should be harmless from editor, */
    update_all_los(op->map);	/* first_player is no longer set there */
}

/*
 * merge_ob(op,top):
 *
 * This function goes through all objects below and including top, and
 * merges op to the first matching object.
 * If top is NULL, it is calculated.
 * Returns true if it succeded in the merge.
 */

int merge_ob(object *op, object *top) {
  if(!op->nrof)
    return 0;
  if(top==NULL)
    for(top=op;top!=NULL&&top->above!=NULL;top=top->above);
  for(;top!=NULL;top=top->below) {
    if(top==op)
      continue;
    if (CAN_MERGE(op,top))
    {
      top->nrof+=op->nrof;
      UNSET_STARTEQUIP(top);
      op->weight = 0; /* Don't want any adjustements now */
      remove_ob(op);
      free_object(op);
      return 1;
    }
  }
  return 0;
}

/*
 * insert_ob_in_map(op, map):
 * This function inserts the object in the two-way linked list
 * which represents what is on a map.
 * The second argument specifies the map, and the x and y variables
 * in the object about to be inserted specifies the position.
 */

void insert_ob_in_map(object *op,mapstruct *m) {
  object *tmp, *top;
  if(m==NULL) {
    dump_object(op);
    LOG(llevError,"Trying to insert in null-map!\n%s\n",errmsg);
    return;
  }
  if(out_of_map(m,op->x,op->y)) {
    dump_object(op);
    LOG(llevError,"Trying to insert object outside the map.\n%s\n", errmsg);
    return;
  }
  if(!IS_REMOVED(op)) {
    dump_object(op);
    LOG(llevError,"Trying to insert (map) inserted object.\n%s\n", errmsg);
    return;
  }
  if(op->more!=NULL)
    insert_ob_in_map(op->more,m);
  UNSET_REMOVED(op);
  if(op->nrof)
    for(tmp=get_map_ob(m,op->x,op->y);tmp!=NULL;tmp=tmp->above)
      if (CAN_MERGE(op,tmp))
      {
        op->nrof+=tmp->nrof;
        UNSET_STARTEQUIP(op);
        remove_ob(tmp);
        free_object(tmp);
      }
  op->ox=op->x,op->oy=op->y;
  op->map=m;
  UNSET_APPLIED(op); /* hack for fixing F_APPLIED in items of dead people */
  if((top=get_map_ob(op->map,op->x,op->y))!=NULL) { /* If anything below then */
    /*
     * There is at least one object present, so we must figure out the order.
     * First figure out what is on top;
     */

    while (top->above != NULL)
      top = top->above;

    tmp = top;

    /*
     * Rule 1: Alive objects stay on top.
     */
    while(tmp != NULL && IS_ALIVE(tmp) && !IS_ALIVE(op) && !IS_FLYING(op))
      top = tmp, tmp = tmp->below;

    /*
     * Link the new object in the two-way list:
     */

    if (top == tmp) {
      /*
       * Simple case, insert new object above tmp.
       */
      if (tmp->above != NULL)
        LOG(llevError, "insert_ob_in_map: top == tmp && tmp->above != NULL.\n");
      else {
        tmp->above = op;
        op->below = tmp;
        op->above = (object *) NULL;
      }
    } else {
      /*
       * Not so simple case, insert between top and tmp.
       */
      if (tmp == NULL) { /* Insert at bottom of map */
        top->below = op;
        op->above = top;
        op->below = (object *) NULL;
        set_map_ob(op->map, op->x, op->y, op);
      } else { /* Insert between top and tmp */
        top->below = op;
        op->above = top;
        tmp->above = op;
        op->below = tmp;
      }
    }
  }
  else
    set_map_ob(op->map,op->x,op->y,op);   /* Tell the map that we're here */
  update_object(op);
  check_walk_on(op);
  if(IS_FREED(op))	/* check_walk_on() killed it */
    return;
  if(op->type==PLAYER)
    op->contr->do_los=1;
  for(tmp=get_map_ob(op->map,op->x,op->y);tmp!=NULL;tmp=tmp->above)
    switch(tmp->type) {
    case PLAYER:
      (*draw_look_func)(tmp);
      break;
    case BUTTON:
    case PEDESTAL:
      update_button(tmp);
      break;
    }
}

/*
 * get_split_ob(ob,nr) splits up ob into two parts.  The part which
 * is returned contains nr objects, and the remaining parts contains
 * the rest (or is removed and freed if that number is 0).
 * On failure, NULL is returned, and the reason put into the
 * global static errmsg array.
 */

object *get_split_ob(object *orig_ob,int nr) {
  object *tmp, *otmp;
  if(orig_ob->nrof<nr) {
    sprintf(errmsg,"There are only %d %ss.",
            orig_ob->nrof?orig_ob->nrof:1, orig_ob->name);
    return NULL;
  }
  tmp=get_object();
  copy_object(orig_ob,tmp);
  if((orig_ob->nrof-=nr)<1) {
    orig_ob->weight=orig_ob->nrof=0;
    remove_ob(orig_ob);
    free_object(orig_ob);
  }
  else if(!IS_REMOVED(orig_ob)){
    if(orig_ob->env!=NULL)
      sub_weight (orig_ob->env,orig_ob->weight*nr);
    if((otmp=is_player_inv (orig_ob->env))!=NULL)
      (*draw_inventory_func)(otmp);
    else if (orig_ob->env == NULL) {
      object *tmp;
      if (orig_ob->map->in_memory!=MAP_IN_MEMORY) {
        strcpy(errmsg, "Tried to split object whose map is not in memory.");
	LOG(llevDebug,
            "Error, Tried to split object whose map is not in memory.\n");
        return NULL;
#if 0 /* Have to check this first */
	load_omap(orig_ob->map,0);
#endif
      }
      for(tmp=get_map_ob(orig_ob->map,orig_ob->x,orig_ob->y);
          tmp!=NULL;tmp=tmp->above)
        if(tmp->type==PLAYER)
          (*draw_look_func)(tmp);
    }
  }
  tmp->nrof=nr;
  return tmp;
}

/*
 * decrease_ob_nr(object, number) decreases a specified number from
 * the amount of an object.  If the amount reaches 0, the object
 * is subsequently removed and freed.
 */

void decrease_ob_nr(object *op,int i) {
  object *tmp;

  if((long)(op->nrof-=i)<1) {
    remove_ob(op);
    free_object(op);
  }
  else
    if(op->env!=NULL) {
      sub_weight(op->env,op->weight*i);
      tmp=is_player_inv (op->env);
      if(tmp!=NULL)
        (*draw_inventory_func)(tmp);
    }
}

/*
 * decrease_ob(object) works like decrease_ob_nr(object, 1).
 */

void decrease_ob(object *op) {
  object *tmp;

  if(--(op->nrof)<1) {
    remove_ob(op);
    free_object(op);
  }
  else
    if(op->env!=NULL) {
      sub_weight(op->env,op->weight);
      tmp=is_player_inv (op->env);
      if(tmp!=NULL)
          (*draw_inventory_func)(tmp);
    } else {
      for(tmp=op->above;tmp!=NULL;tmp=tmp->above)
        if(tmp->type==PLAYER)
          (*draw_look_func)(tmp);
    }
}

/*
 * add_weight(object, weight) adds the specified weight to an object,
 * and also updates how much the environment(s) is/are carrying.
 */

void add_weight (object *op, signed long weight) {
  while (op!=NULL) {
    weight=(signed long)(weight*(100-op->bonus)/100);
    op->carrying+=weight;
    op=op->env;
  }
}

/*
 * insert_ob_in_ob(op,environment):
 *   This function inserts the object op in the linked list
 *   inside the object environment.  If environment is NULL,
 *   the object will be stacked at the map.
 *
 *
 * Eneq(@csd.uu.se): Altered insert_ob_in_ob to make things picked up enter 
 * the inventory at the last position or next to other objects of the same
 * type.
 * Frank: Now sorted by type, archetype and magic!
 */

void insert_ob_in_ob(object *op,object *where) {
  object *tmp, *otmp;
  if(!IS_REMOVED(op)) {
    dump_object(op);
    LOG(llevError,"Trying to insert (ob) inserted object.\n%s\n", errmsg);
    return;
  }
  if(where==NULL) {
    dump_object(op);
    LOG(llevError,"Trying to put object in NULL.\n%s\n", errmsg);
  }
  if (where->head)
  {
    LOG(llevDebug, "Tried to insert object wrong part of multipart object.\n");
    where = where->head;
  }
#if 0 /* this check to upper level, can't put titan in chest in editor :) */
  if (NO_PICK(op)) {  /* hevi@lut.fi */
    LOG(llevError, "Tried to insert object with NO_PICK(): %s (%d)\n",
        op->name, op->count);
    return;
  }
#endif
  if (op->more) {
    LOG(llevError, "Tried to insert multipart object %s (%d)\n",
        op->name, op->count);
    return;
  }
  UNSET_REMOVED(op);
  if(op->nrof) {
    for(tmp=where->inv;tmp!=NULL;tmp=tmp->below)
      if (CAN_MERGE(tmp,op))
      {
        op->nrof+=tmp->nrof;
        UNSET_STARTEQUIP(op);
        remove_ob(tmp);
        free_object(tmp);
      }
    /* I assume combined objects have no inventory */
    add_weight (where, op->weight*op->nrof);
  } else
    add_weight (where, (op->weight+op->carrying));

  otmp=is_player_inv(where);
  if (otmp&&otmp->contr!=NULL&&!NO_FIX_PLAYER(op))
      fix_player(otmp);

  op->map=NULL;
  op->env=where;
  op->above=NULL;
  op->below=NULL;
  op->x=0,op->y=0;
  op->ox=0,op->oy=0;
  if (where->inv==NULL)
      where->inv=op;
  else {
    tmp = where->inv;
    if (! tmp->invisible) {
      /*
       * Rule 1: Move to the similar type
       */
      while (tmp->below != NULL &&
             tmp->type != op->type)
        tmp = tmp->below;

      /*
       * Rule 2: Within similar type, sort by archetype
       */
      while (tmp->below != NULL &&
             tmp->below->type == op->type &&
             tmp->arch != op->arch)
        tmp = tmp->below;

      /*
       * Rule 3: Within similar type & archetype, sort by magic
       * First check a special case, then scroll down until
       * we are below items with more magic.
       */
      if (tmp->type == op->type &&
          tmp->arch == op->arch &&
          tmp->magic < op->magic)
        tmp = tmp->above; /* Beware: tmp might be NULL now. */
      else {
        while (tmp != NULL &&
               tmp->below != NULL &&
               tmp->below->type == op->type &&
               tmp->below->arch == op->arch &&
               tmp->magic > op->magic)
          tmp = tmp->below;

        /*
         * Rule 4: Insert what seems like a similar item below.
         */
        while (tmp != NULL &&
               tmp->below != NULL &&
               tmp->below->type == op->type &&
               tmp->below->arch == op->arch &&
               tmp->below->magic == op->magic)
          tmp = tmp->below;
      }
    }

    /*
     * Now link the new object into the two-way list, below tmp.
     */
    if (tmp == NULL) {
      op->below = where->inv;
      op->below->above = op;
      where->inv = op;
    } else if (tmp->above == NULL) {
      where->inv = op;
      op->above = NULL;
      op->below = tmp;
      tmp->above = op;
    } else if (tmp->below != NULL) {
      tmp->below->above = op;
      op->below = tmp->below;
      tmp->below = op;
      op->above = tmp;
    } else {
      tmp->below = op;
      op->above = tmp;
    }
  }
  tmp=is_player_inv(where);
  if (tmp)
      (*draw_inventory_func)(tmp);
}

/*
 * Checks if any objects which has the WALK_ON() (or FLY_ON() if the
 * object is flying) flag set, will be auto-applied by the insertion
 * of the object into the map (applying is instantly done).
 * Any speed-modification due to SLOW_MOVE() of other present objects
 * will affect the speed_left of the object.
 */

void check_walk_on(object *op) {
  object *tmp;
  if(NO_APPLY(op))
    return;
  for(tmp=op->below;tmp!=NULL;tmp=tmp->below) {
    if (tmp==tmp->below)	/* why? mta */
       break;
    if(SLOW_MOVE(tmp)&&!IS_FLYING(op))
      op->speed_left -= SLOW_PENALTY(tmp)*FABS(op->speed);
    if(IS_FLYING(op)?FLY_ON(tmp):WALK_ON(tmp)) {
      if(op->type==PLAYER)
        (*draw_func)(op);
      if((*apply_func)(op,tmp)==2)
        break;
    }
  }
}

/*
 * present_arch(arch, map, x, y) searches for any objects with
 * a matching archetype at the given map and coordinates.
 * The first matching object is returned, or NULL if none.
 */

object *present_arch(archetype *at, mapstruct *m, int x, int y) {
  object *tmp;
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Present_arch called outside map.\n");
    return NULL;
  }
  for(tmp=get_map_ob(m,x,y); tmp != NULL; tmp = tmp->above)
    if(tmp->arch == at)
      return tmp;
  return NULL;
}

/*
 * present(type, map, x, y) searches for any objects with
 * a matching type variable at the given map and coordinates.
 * The first matching object is returned, or NULL if none.
 */

object *present(unsigned char type,mapstruct *m, int x,int y) {
  object *tmp;
  if(out_of_map(m,x,y)) {
    LOG(llevError,"Present called outside map.\n");
    return NULL;
  }
  for(tmp=get_map_ob(m,x,y);tmp!=NULL;tmp=tmp->above)
    if(tmp->type==type)
      return tmp;
  return NULL;
}

/*
 * present_in_ob(type, object) searches for any objects with
 * a matching type variable in the inventory of the given object.
 * The first matching object is returned, or NULL if none.
 */

object *present_in_ob(unsigned char type,object *op) {
  object *tmp;
  for(tmp=op->inv;tmp!=NULL;tmp=tmp->below)
    if(tmp->type==type)
      return tmp;
  return NULL;
}

/*
 * present_arch_in_ob(archetype, object) searches for any objects with
 * a matching archetype in the inventory of the given object.
 * The first matching object is returned, or NULL if none.
 */

object *present_arch_in_ob(archetype *at, object *op)  {
  object *tmp;
  for(tmp=op->inv;tmp!=NULL;tmp=tmp->below)
    if( tmp->arch == at)
      return tmp;
  return NULL;
}

/*
 * query_weight(object) returns a character pointer to a static buffer
 * containing the text-representation of the weight of the given object.
 * The buffer will be overwritten by the next call to query_weight().
 */

char *query_weight(object *op) {
  static char buf[10];
  int i=op->nrof?op->nrof*op->weight:op->weight+op->carrying;

  if(op->weight<0)
    return "      ";
  if(i%1000)
    sprintf(buf,"%6.1f",i/1000.0);
  else
    sprintf(buf,"%4d  ",i/1000);
  return buf;
}

/*
 * get_number(integer) returns the text-representation of the given number
 * in a static buffer.  The buffer might be overwritten at the next
 * call to get_number().
 * It is currently only used by the query_name() function.
 */

char *get_number(int i) {
  if(i<=20)
    return numbers[i];
  else {
    static char buf[MAX_BUF];
    sprintf(buf,"%d",i);
    return buf;
  }
}

/*
 * query_name(object) returns a character pointer pointing to a static
 * buffer which contains a verbose textual representation of the name
 * of the given object.  The buffer will be overwritten at the next
 * call to query_name().
 */

char *query_name(object *op) {
  static char buf[MAX_BUF];
  char buf2[MAX_BUF];
  if(op->name == NULL)
    return "(null)";
  if(!op->nrof&&!op->magic&&!IS_APPLIED(op)&&op->weight<1&&!IS_UNPAID(op))
    return op->name; /* To speed things up */
  if(op->nrof) {
    strcpy(buf,get_number(op->nrof));
    if(op->nrof==1&&NEED_AN(op))
      strcat(buf,"n");
    strcat(buf," ");
    strcat(buf,op->name);
    if(op->nrof!=1)
      if(NEED_IE(op)) {
        char *cp=strrchr(buf,'y');
        if(cp!=NULL)
          *cp='\0'; /* Strip the 'y' */
        strcat(buf,"ies");
      } else if (buf[strlen(buf)-1]!='s') /* if the item ends in 's', then */
        strcat(buf,"s");		  /* adding another one is not the */
				/* way to pluralize it.  The only item */
				/* where this matters (that I know of) is */
				/* bracers, as they start of plural */
  } else
      strcpy(buf,op->name);
  switch(op->type) {
  case SCROLL:
  case WAND:
  case SPELLBOOK:
    strcat(buf," of ");
    strcat(buf,spells[op->stats.sp].name);
    break;
  case AMULET:
  case RING:
#if 0 /* The name becomes too long, especially when unpaid */
    strcat(buf," ");
    strcat(buf,describe_item(op));
#endif
    break;
  default:
    if(op->magic) {
      sprintf(buf2," %+d",op->magic);
      strcat(buf,buf2);
    }
  }
  if(IS_APPLIED(op)) {
    switch(op->type) {
    case BOW:
    case WAND:
      strcat(buf," (readied)");
      break;
    case WEAPON:
      strcat(buf," (wielded)");
      break;
    case ARMOUR:
    case HELMET:
    case SHIELD:
    case RING:
    case BOOTS:
    case GLOVES:
    case AMULET:
    case GIRDLE:
    case BRACERS:
      strcat(buf," (worn)");
      break;
    default:
      strcat(buf," (undefined)");
    }
  }
  if(IS_UNPAID(op))
    strcat(buf," (unpaid)");
/*
  if(op->weight>0) {
    sprintf(buf2," (%d)",op->nrof?op->weight*op->nrof:op->weight+op->carrying);
    strcat(buf,buf2);
  }
*/
  return buf;
}

/*
 * set_cheat(object) sets the cheat flag (WAS_WIZ) in the object and in
 * all it's inventory (recursively).
 * If checksums are used, a player will get set_cheat called for
 * him/her-self and all object carried by a call to this function.
 */

void set_cheat(object *op) {
  object *tmp;
  SET_WAS_WIZ(op);
  if(op->inv)
    for(tmp = op->inv; tmp != NULL; tmp = tmp->below)
      set_cheat(tmp);
}

/*
 * find_free_spot(archetype, map, x, y, start, stop) will search for
 * a spot at the given map and coordinates which will be able to contain
 * the given archetype.  start and stop specifies how many squares
 * to search (see the freearr_x/y[] definition).
 * It returns a random choice among the alternatives found.
 */

int find_free_spot(archetype *at, mapstruct *m,int x,int y,int start,int stop) {
  int i,index=0;
  static int altern[SIZEOFFREE];
  for(i=start;i<stop;i++) {
    if(!arch_blocked(at,m,x+freearr_x[i],y+freearr_y[i]))
      altern[index++]=i;
    else if(wall(m,x+freearr_x[i],y+freearr_y[i])&&maxfree[i]<stop)
      stop=maxfree[i];
  }
  if(!index) return 0;
  return altern[RANDOM()%index];
}

/*
 * find_first_free_spot(archetype, mapstruct, x, y) works like
 * find_free_spot(), but it will search max number of squares.
 * But it will return the first available spot, not a random choice.
 */

int find_first_free_spot(archetype *at, mapstruct *m,int x,int y) {
  int i;
  for(i=0;i<SIZEOFFREE;i++)
    if(!arch_blocked(at,m,x+freearr_x[i],y+freearr_y[i]))
      return i;
  return 0;
}

/*
 * find_dir(map, x, y, exclude) will search some close squares in the
 * given map at the given coordinates for live objects.
 * It will not considered the object given as exlude among possible
 * live objects.
 * It returns the direction toward the first/closest live object if finds
 * any, otherwise 0.
 */

int find_dir(mapstruct *m, int x, int y, object *exclude) {
  int i,max=SIZEOFFREE;
  object *tmp;

  for(i=1;i<max;i++) {
    if(wall(m, x+freearr_x[i],y+freearr_y[i]))
      max=maxfree[i];
    else {
      tmp=get_map_ob(m,x+freearr_x[i],y+freearr_y[i]);
      while((tmp!=NULL&&!IS_MONSTER(tmp)&&tmp->type!=PLAYER) || tmp == exclude)
        tmp=tmp->above;
      if(tmp!=NULL)
        return freedir[i];
    }
  }
  return 0;
}

/*
 * distance(object 1, object 2) will return the square of the
 * distance between the two given objects.
 */

int distance(object *ob1,object *ob2) {
  int i;
  i= (ob1->x - ob2->x)*(ob1->x - ob2->x)+
         (ob1->y - ob2->y)*(ob1->y - ob2->y);
  return i;
}

/*
 * find_dir_2(delta-x,delta-y) will return a direction in which
 * an object which has subtracted the x and y coordinates of another
 * object, needs to travel toward it.
 */

int find_dir_2(int x, int y) {
  int q;
  if(!y)
    q= -300*x;
  else
    q=x*100/y;
  if(y>0) {
    if(q < -242)
      return 3 ;
    if (q < -41)
      return 2 ;
    if (q < 41)
      return 1 ;
    if (q < 242)
      return 8 ;
    return 7 ;
  }
  if (q < -242)
    return 7 ;
  if (q < -41)
    return 6 ;
  if (q < 41)
    return 5 ;
  if (q < 242)
    return 4 ;
  return 3 ;
}

/*
 * absdir(int): Returns a number between 1 and 8, which represent
 * the "absolute" direction of a number (it actually takes care of
 * "overflow" in previous calculations of a direction).
 */

int absdir(int d) {
  while(d<1) d+=8;
  while(d>8) d-=8;
  return d;
}

/*
 * dirdiff(dir1, dir2) returns how many 45-degrees differences there is
 * between two directions (which are expected to be absolute (see absdir())
 */

int dirdiff(int dir1, int dir2) {
  int d;
  d = abs(dir1 - dir2);
  if(d>4)
    d = 8 - d;
  return d;
}

/*
 * can_pick(picker, item): finds out if an object is possible to be
 * picked up by the picker.  Returnes 1 on success, otherwise 0.
 */

int can_pick(object *who,object *item) {
  return IS_WIZ(who)||
         (item->weight>=0&&!NO_PICK(item)&&!IS_ALIVE(item)&&
          (who->type==PLAYER||item->weight<who->weight/3));
}

/*
 * create clone from object to another
 */
object *ObjectCreateClone (object *asrc) {
    object *dst,*tmp,*src,*part,*prev,*item;

    if(!asrc) return NULL;
    src = asrc;
    if(src->head)
        src = src->head;

    prev = NULL;
    for(part = src; part; part = part->more) {
        tmp = get_object();
        copy_object(part,tmp);
        tmp->x -= src->x;
        tmp->y -= src->y;
        if(!part->head) {
            dst = tmp;
            tmp->head = NULL;
        } else {
            tmp->head = dst;
        }
        tmp->more = NULL;
        if(prev) 
            prev->more = tmp;
        prev = tmp;
    }
    /*** copy inventory ***/
    for(item = src->inv; item; item = item->below) {
	insert_ob_in_ob(ObjectCreateClone(item),dst);
    }
    return dst;
}

/*** end of object.c ***/
